#include <qlineedit.h>
#include <qcombobox.h>
#include <qtimer.h>
#include <qspinbox.h>
#include <qstatusbar.h>
#include <qpushbutton.h>

#include <helpers.h>
#include <iprovider.h>

#include "relatedplugin.h"
#include "relatedinput.h"
#include "relatedfeedbackwidget.h"

namespace NPlugin
{

RelatedPlugin::RelatedPlugin(const DebtagsPluginContainer& container)
	: _container(container)
{
	_pMainWindow = 0;
	_pProvider = 0;
	_pHandleMaker = 0;
	_pRelatedInput = 0;
	_pRelatedFeedbackWidget = 0;
	_isInactive = true;
}

RelatedPlugin::~RelatedPlugin()
{
	delete _pRelatedInput;
	delete _pRelatedFeedbackWidget;
}
 
/**
 * Methods
 */

/////////////////////////////////////////////////////
// Plugin Interface
/////////////////////////////////////////////////////

QString RelatedPlugin::title() const
{ 
	return tr("Related Plugin"); 
}

QString RelatedPlugin::briefDescription() const
{
	return tr("This searched for plugins related to a given package.");
}

QString RelatedPlugin::description() const
{
	return tr("This searched for plugins related to a given package.");
}

void RelatedPlugin::init(IProvider* pProvider)
{
	_pProvider = pProvider;
	_pMainWindow = pProvider->mainWindow();
	_pHandleMaker = &_pProvider->handleMaker();
	
	_pRelatedInput = new RelatedInput(_pMainWindow, "RelatedInput");
	const set<string>& packages = pProvider->packages();
	for (set<string>::const_iterator it = packages.begin(); it != packages.end(); ++it)
		_pRelatedInput->_pPackageInput->insertItem(toQString(*it));
	_pRelatedInput->_pPackageInput->setMinimumWidth(100);
	_pRelatedInput->_pPackageInput->setCurrentText("");
	connect(_pRelatedInput->_pPackageInput, SIGNAL(activated(const QString&)), SLOT(evaluateSearch()));
	connect(_pRelatedInput->_pClearButton, SIGNAL(clicked()), SLOT(onClearSearch()));
	
	_pRelatedFeedbackWidget = new RelatedFeedbackWidget(_pMainWindow, "RelatedFeedbackWidget");
	_pRelatedFeedbackWidget->setShown(false);
	connect(
		_pRelatedInput->_pMaximumDistanceInput, SIGNAL(valueChanged(int)), 
		SLOT(evaluateSearch())
	);
	connect(
		_pRelatedInput->_pPackageInput, SIGNAL(textChanged(const QString&)), 
		SLOT(onInputTextChanged(const QString&))
	);
	if (_container.collection()==0)
		setWidgetsEnabled(false);
}

/////////////////////////////////////////////////////
// Search Plugin Interface
/////////////////////////////////////////////////////

QString RelatedPlugin::inputWidgetTitle() const	
{
	return tr("Related"); 
}

QWidget* RelatedPlugin::inputWidget() const
{ 
	return _pRelatedInput;
}

QWidget* RelatedPlugin::shortInputAndFeedbackWidget() const
{
	return _pRelatedFeedbackWidget;
}

void RelatedPlugin::clearSearch()
{
	_pRelatedInput->_pPackageInput->setCurrentText("");
}

const Tagcoll::OpSet<int>& RelatedPlugin::searchResult() const
{
	return _searchResult;
}

/////////////////////////////////////////////////////
// helper functions
/////////////////////////////////////////////////////

void RelatedPlugin::onInputTextChanged(const QString& text)
{
	if (text == "")
		evaluateSearch();
}

void RelatedPlugin::evaluateSearch()
{
	// stop the delay timer in case that this evaluateSearch() was triggered by
	// another event
	_pProvider->reportBusy(this, tr("Searching Package Database for related packages"));
	_searchResult.clear();
	_pRelatedFeedbackWidget->_pRelatedSearchTextView->setText(
		_pRelatedInput->_pPackageInput->currentText()
	);
 	string package = toString(_pRelatedInput->_pPackageInput->currentText());
	int packageID = _pHandleMaker->getHandle(package);
	_isInactive = package.empty();	// if no input was given the search is inactive
	if ( !_isInactive )
	{
		///@todo check here if has item is neccesary
/*		if ( !_pPackageCollection->hasItem(packageID) )	// if the package entered is not in the collection
		{
			_isInactive = true;
			_pProvider->reportError(
				tr("No such package"),
				tr("There is no package with the given name available.")
			);
		}
		else */
		{
			int maxdist = _pRelatedInput->_pMaximumDistanceInput->value();
			Tagcoll::OpSet<string> ts = _container.collection()->getTagsetForItem( packageID );
			// the result holds all the items with the
			_searchResult = _container.collection()->getItemsForTagset(ts);	
				// same tagset as the currently selected package
			if (maxdist!=0)
			{
				// Get the related tagsets
				list< Tagcoll::OpSet<string> > rel = 
					_container.collection()->getRelatedTagsets(ts, maxdist);
				// add the packages which have tags like the related packages
				for (list< Tagcoll::OpSet<string> >::const_iterator i = rel.begin();
						i != rel.end(); i++)
				{
					_searchResult += _container.collection()->getItemsForTagset(*i);
				}
			}
		}
	}
	// show the feedback widget only if a search was entered
	_pRelatedFeedbackWidget->setShown(!_isInactive);
	_pProvider->reportReady(this);
	emit searchChanged(this);
}

void RelatedPlugin::debtagsDataChanged()
{
	if (_container.collection()==0)
		setWidgetsEnabled(false);
	else
		setWidgetsEnabled(true);
	clearSearch();
}

void RelatedPlugin::setWidgetsEnabled(bool enabled)
{
	if (_pRelatedInput)
		_pRelatedInput->setEnabled(enabled);
}
 
}	// namespace NPlugin
