
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: otk.c,v 1.103 2006/01/16 11:56:47 mschwerin Exp $
 *
 * The scrollbar code was backported from the xine-ui CVS and was originally
 * written by Miguel Freitas.
 *
 * We really have to worry about threads in this module! We have three threads:
 * the update-thread, the user-interaction-thread (motion, buttons, keys) and
 * the xine-lib-thread.
 *
 * The update-thread only interacts with otk in otk_update_job. The
 * user-interaction-thread always passes through otk_event_handler. So it
 * should be enough to enclose those two methods in mutexes.
 */
#include "config.h"

#include <errno.h>
#include <math.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "assert.h"
#include "heap.h"
#include "list.h"
#include "logger.h"
#include "odk.h"
#include "otk.h"
#include "oxine.h"
#include "scheduler.h"

#define DIRECTION_UP                    (1)
#define DIRECTION_DOWN                  (2)
#define DIRECTION_LEFT                  (3)
#define DIRECTION_RIGHT                 (4)

#define OTK_SELECTABLE_KEY              (1)
#define OTK_SELECTABLE_MOUSE            (2)

#define OTK_WIDGET_WINDOW               (1)
#define OTK_WIDGET_BUTTON               (2)
#define OTK_WIDGET_LABEL                (3)
#define OTK_WIDGET_LIST                 (4)
#define OTK_WIDGET_LISTENTRY            (5)
#define OTK_WIDGET_SLIDER               (6)
#define OTK_WIDGET_SCROLLBAR            (8)
#define OTK_WIDGET_CHECKBOX             (9)
#define OTK_WIDGET_BORDER               (10)

#define OSD_TEXT_PALETTE_TRANSPARENT    (0)
#define OSD_TEXT_PALETTE_BACKGROUND     (1)
#define OSD_TEXT_PALETTE_FOREGROUND     (10)

#define ABS(x) ((x)<0?-(x):(x))

typedef struct otk_button_s otk_button_t;
typedef struct otk_label_s otk_label_t;
typedef struct otk_window_s otk_window_t;
typedef struct otk_list_s otk_list_t;
typedef struct otk_listentry_s otk_listentry_t;
typedef struct otk_scrollbar_s otk_scrollbar_t;
typedef struct otk_slider_s otk_slider_t;
typedef struct otk_checkbox_s otk_checkbox_t;
typedef struct otk_selector_s otk_selector_t;
typedef struct otk_border_s otk_border_t;

struct otk_widget_s {
    int type;

    otk_t *otk;
    odk_t *odk;
    otk_window_t *window;

    int x;
    int y;
    int w;
    int h;

    int focus;
    int needupdate;
    int selectable;

    char *font;
    int fontsize;
    int alignment;

    void (*draw) (otk_widget_t * this);
    void (*destroy) (otk_widget_t * this);

    // we call this callback when this item has the focus
    void *focus_enter_cb_data;
    void (*focus_enter_cb) (void *user_data);

    // we call this callback when this item looses the focus
    void *focus_leave_cb_data;
    void (*focus_leave_cb) (void *user_data);
};

struct otk_slider_s {
    otk_widget_t widget;

    // horizontal or vertical?
    int direction;
    // just a backup
    int last_value;

    // we call this callback to when the slider is clicked
    void *cb_data;
    otk_slider_cb_t cb;

    // we call this callback to get the current value
    void *get_value_cb_data;
    otk_slider_data_cb_t get_value_cb;
};

struct otk_border_s {
    otk_widget_t widget;
};

struct otk_checkbox_s {
    otk_widget_t widget;

    int is_checked;

    // we call this callback when the checkbox is clicked
    void *cb_data;
    otk_button_cb_t cb;
};

struct otk_button_s {
    otk_widget_t widget;

    // there are two types of button
    // one with text and one with a pixmap being shown
    char *text;
    uint8_t *pixmap;

    // we call this callback when the button is clicked
    void *cb_data;
    otk_button_cb_t cb;
};

struct otk_listentry_s {
    otk_widget_t widget;

    otk_list_t *list;

    int pos_visible_list;
    int pos_complete_list;

    int is_first;
    int is_last;
    int is_selected;
    int is_visible;

    // text to show
    char *text;

    // time this item was last clicked
    time_t doubleclicktime;

    // we call this callback when this item is clicked
    void *activate_cb_data;
    otk_list_cb_t activate_cb;

    // we call this callback when the remove key is pressed
    void *remove_cb_data;
    otk_list_cb_t remove_cb;
};

struct otk_list_s {
    otk_widget_t widget;
    otk_widget_t *scrollbar;

    // the entries
    l_list_t *entries;
    // number of entries
    int num_entries;
    // number of selected entries
    int num_selected;
    // number of visible entries
    int num_visible;
    // the number of the entry at the top of the visible list
    int first_visible;

    // may the user select entries
    int allow_select;
    // may the user remove entries
    int allow_remove;

    // height of an entry
    int entry_height;
    // width of an entry
    int entry_width;

    // data that is sent in every callback
    void *cb_data;
};

struct otk_selector_s {
    otk_list_t list;

    // we call this callback when the visible entry changes
    void *cb_data;
    otk_selector_cb_t cb;
};

struct otk_label_s {
    otk_widget_t widget;

    char *text;
    int max_text_width;

    // upcall to get the new text
    otk_label_uc_t uc;
    void *uc_data;
};

struct otk_scrollbar_s {
    otk_widget_t widget;

    int position;
    int length;

    otk_button_cb_t cb_up;
    otk_button_cb_t cb_down;

    otk_scrollbar_cb_t cb_click;

    void *cb_data;
};

struct otk_window_s {
    otk_widget_t widget;

    int keep;
    int fill;
    int border;

    otk_widget_t *focus_ptr;

    l_list_t *subs;
};

struct otk_s {
    odk_t *odk;

    otk_window_t *window;
    l_list_t *windows;

    int textpalette_label;
    int textpalette_button[2];  // 1 is focused

    int update_job;

    void (*event_handler) (void *data, oxine_event_t * ev);
    void *event_handler_data;

    pthread_mutexattr_t draw_mutex_attr;
    pthread_mutex_t draw_mutex;
};

extern oxine_t *oxine;

/* 
 * ***************************************************************************
 * Name:            check_text_width
 * Access:          private
 *
 * Description:     Checks the width of text and truncates the text if
 *                  necessary.
 * ***************************************************************************
 */
static void
check_text_width (odk_t * odk, char *text, int width)
{
    int textwidth;
    int textheight;
    odk_get_text_size (odk, text, &textwidth, &textheight);

    if (textwidth > width) {
        int i = strlen (text) - 4;

        if (i < 0)
            return;

        text[i] = text[i + 1] = text[i + 2] = '.';
        text[i + 3] = 0;

        odk_get_text_size (odk, text, &textwidth, &textheight);
        while ((textwidth > width) && (i > 0)) {
            i--;
            text[i] = '.';
            text[i + 3] = 0;
            odk_get_text_size (odk, text, &textwidth, &textheight);
        }
    }
}


/* 
 * ***************************************************************************
 * Name:            is_correct_widget
 * Access:          private
 *
 * Description:     Checks if the expected type and the real type of a widget
 *                  are the same.
 * ***************************************************************************
 */
static int
is_correct_widget (otk_widget_t * widget, int expected)
{
    if (!widget)
        return 0;

    if (widget->type != expected)
        return 0;

    return 1;
}


/* 
 * ***************************************************************************
 * Name:            widget_remove
 * Access:          private
 *
 * Description:     Removes a widget from the window.
 * ***************************************************************************
 */
static void
widget_remove (otk_t * otk, otk_widget_t * widget)
{
    if (otk->window->focus_ptr == widget)
        otk->window->focus_ptr = NULL;
    l_list_remove (otk->window->subs, widget);
}


/* 
 * ***************************************************************************
 * Name:            widget_append
 * Access:          private
 *
 * Description:     Appends a widget to the window.
 * ***************************************************************************
 */
static void
widget_append (otk_t * otk, otk_widget_t * widget)
{
    l_list_append (otk->window->subs, widget);
}


/* 
 * ***************************************************************************
 * Name:            widget_destroy
 * Access:          private
 *
 * Description:     Destroys a widget.
 * ***************************************************************************
 */
static void
widget_destroy (void *data)
{
    otk_widget_t *widget = (otk_widget_t *) data;
    if (widget->font)
        ho_free (widget->font);
    widget->destroy (widget);
}


/* 
 * ***************************************************************************
 * Name:            find_widget_xy
 * Access:          private
 *
 * Description:     Returns the widget at (x/y).
 * ***************************************************************************
 */
static otk_widget_t *
find_widget_xy (otk_t * otk, int x, int y)
{
    if (!otk->window)
        return NULL;

    otk_widget_t *widget = l_list_first (otk->window->subs);
    while (widget) {
        if ((widget->x <= x) && (widget->y <= y)
            && ((widget->x + widget->w) >= x)
            && ((widget->y + widget->h) >= y)
            && (widget->selectable & OTK_SELECTABLE_MOUSE))
            return widget;

        widget = l_list_next (otk->window->subs, widget);
    }

    return NULL;
}


/* 
 * ***************************************************************************
 * Name:            get_distance
 * Access:          private
 *
 * Description:     Returns the distance between the widgets base and target.
 * ***************************************************************************
 */
static int
get_distance (otk_widget_t * base, otk_widget_t * target)
{

    int x1 = (base->x + base->w / 2) / 10;
    int y1 = (base->y + base->h / 2) / 10;
    int x2 = (target->x + target->w / 2) / 10;
    int y2 = (target->y + target->h / 2) / 10;
    int dist = (int) sqrt (pow ((x1 - x2), 2) + pow ((y1 - y2), 2));

    return dist;
}


/* 
 * ***************************************************************************
 * Name:            get_vertical_angle
 * Access:          private
 *
 * Description:     Returns the vertical angle between the widgets base and
 *                  target.
 * ***************************************************************************
 */
static double
get_vertical_angle (otk_widget_t * base, otk_widget_t * target)
{

    int x1 = base->x + base->w / 2;
    int y1 = base->y + base->h / 2;
    int x2 = target->x + target->w / 2;
    int y2 = target->y + target->h / 2;
    int x = ABS (x1 - x2);
    int y = ABS (y1 - y2);
    double a;

    if (x == 0)
        a = 0;
    else
        a = atan ((double) y / (double) x) + 1;

    return a;
}


/* 
 * ***************************************************************************
 * Name:            get_horizontal_angle
 * Access:          private
 *
 * Description:     Returns the horizonal angle between the widgets base and
 *                  target.
 * ***************************************************************************
 */
static double
get_horizontal_angle (otk_widget_t * base, otk_widget_t * target)
{

    int x1 = base->x + base->w / 2;
    int y1 = base->y + base->h / 2;
    int x2 = target->x + target->w / 2;
    int y2 = target->y + target->h / 2;
    int x = ABS (x1 - x2);
    int y = ABS (y1 - y2);
    double a;

    if (y == 0)
        a = 0;
    else
        a = atan ((double) x / (double) y) + 1;

    return a;
}


/* 
 * ***************************************************************************
 * Name:            find_neighbour
 * Access:          private
 *
 * Description:     Searches for the nearest neighbour in the given direction.
 * ***************************************************************************
 */
static otk_widget_t *
find_neighbour (otk_t * otk, int direction)
{
    if (!otk->window)
        return NULL;

    // if no widget has the focus, we grab the first selectable widget
    if (!otk->window->focus_ptr) {
        otk_widget_t *widget = l_list_first (otk->window->subs);
        while (widget) {
            if ((widget->selectable & OTK_SELECTABLE_KEY)) {
                return widget;
            }
            widget = l_list_next (otk->window->subs, widget);
        }
    }
    // if we were not able to find a selectable widget we quit
    if (!otk->window->focus_ptr)
        return NULL;

    int neighbour_ratio = 0;
    int x = otk->window->focus_ptr->x + otk->window->focus_ptr->w / 2;
    int y = otk->window->focus_ptr->y + otk->window->focus_ptr->h / 2;
    otk_widget_t *neighbour = NULL;

    otk_widget_t *widget = l_list_first (otk->window->subs);
    while (widget) {
        if ((widget != otk->window->focus_ptr)
            && (widget->selectable & OTK_SELECTABLE_KEY)) {

            int ratio = 0;
            int nx = widget->x + widget->w / 2;
            int ny = widget->y + widget->h / 2;

            switch (direction) {
                case DIRECTION_UP:
                    if ((y - ny) < 0)
                        break;
                    ratio = get_distance (otk->window->focus_ptr, widget)
                        * get_horizontal_angle (otk->window->focus_ptr,
                                                widget);
                    break;
                case DIRECTION_DOWN:
                    if ((ny - y) < 0)
                        break;
                    ratio = get_distance (otk->window->focus_ptr, widget)
                        * get_horizontal_angle (otk->window->focus_ptr,
                                                widget);
                    break;
                case DIRECTION_LEFT:
                    if ((x - nx) < 0)
                        break;
                    ratio = get_distance (otk->window->focus_ptr, widget)
                        * get_vertical_angle (otk->window->focus_ptr, widget);
                    break;
                case DIRECTION_RIGHT:
                    if ((nx - x) < 0)
                        break;
                    ratio = get_distance (otk->window->focus_ptr, widget)
                        * get_vertical_angle (otk->window->focus_ptr, widget);
                    break;
            }
            if (ratio > 0 && ((ratio < neighbour_ratio) || !neighbour_ratio)) {
                neighbour_ratio = ratio;
                neighbour = widget;
            }
        }
        widget = l_list_next (otk->window->subs, widget);
    }

    return neighbour;
}


/* 
 * ***************************************************************************
 * Name:            otk_widget_set_font
 * Access:          public
 *
 * Description:     Set font and fontsize of a widget.
 * ***************************************************************************
 */
void
otk_widget_set_font (otk_widget_t * widget, const char *font, int fontsize)
{
    if (widget->font)
        ho_free (widget->font);
    widget->font = ho_strdup (font);
    widget->fontsize = fontsize;
}


/* 
 * ***************************************************************************
 * Name:            otk_widget_set_alignment
 * Access:          public
 *
 * Description:     Sets the text alignment of a widget.
 * ***************************************************************************
 */
void
otk_widget_set_alignment (otk_widget_t * widget, int alignment)
{
    widget->alignment = alignment;
}


/* 
 * ***************************************************************************
 * Name:            otk_send_event
 * Access:          public
 *
 * Description:     Sends an event.
 * ***************************************************************************
 */
void
otk_send_event (otk_t * otk, oxine_event_t * ev)
{
    odk_oxine_event_send (otk->odk, ev);
}

/*
 * ***********************************************************************************
 * Button-Widget 
 * ***********************************************************************************
 */
static void
button_destroy (otk_widget_t * this)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_BUTTON))
        return;

    if (button->text)
        ho_free (button->text);
    if (button->pixmap)
        ho_free (button->pixmap);

    widget_remove (this->otk, this);

    ho_free (button);
}

static void
button_draw (otk_widget_t * this)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_BUTTON))
        return;

    odk_draw_rect (this->odk, this->x, this->y,
                   this->x + this->w, this->y + this->h,
                   this->otk->textpalette_button[this->focus] +
                   OSD_TEXT_PALETTE_BACKGROUND, 1);

    if (button->text) {
        odk_osd_set_font (this->odk, this->font, this->fontsize);
        check_text_width (this->odk, button->text, button->widget.w - 10);

        int x = this->x + this->w / 2;
        if (this->alignment & OTK_ALIGN_RIGHT) {
            x = this->x + this->w - 5;
        } else if (this->alignment & OTK_ALIGN_LEFT) {
            x = this->x + 5;
        }
        // we ignore any other alignment than VCENTER
        this->alignment &= ~OTK_ALIGN_TOP;
        this->alignment &= ~OTK_ALIGN_BOTTOM;
        this->alignment |= OTK_ALIGN_VCENTER;

        odk_draw_text (this->odk, x,
                       this->y + this->h / 2,
                       button->text,
                       this->alignment,
                       this->otk->textpalette_button[this->focus]);
    } else if (button->pixmap) {
        uint8_t palette_map[] =
            { (this->otk->textpalette_button[this->focus] +
               OSD_TEXT_PALETTE_BACKGROUND),
            (this->otk->textpalette_button[this->focus] +
             OSD_TEXT_PALETTE_FOREGROUND)
        };

        odk_draw_bitmap (this->odk, button->pixmap,
                         this->x + this->w / 2,
                         this->y + this->h / 2, 20, 20, palette_map);
    }
}

static otk_button_t *
button_new_basic (otk_t * otk, int x, int y,
                  int w, int h, otk_button_cb_t cb, void *cb_data)
{
    otk_button_t *button = ho_new (otk_button_t);

    button->widget.type = OTK_WIDGET_BUTTON;
    button->widget.x = otk->window->widget.x + x;
    button->widget.y = otk->window->widget.y + y;
    button->widget.w = w;
    button->widget.h = h;
    button->widget.otk = otk;
    button->widget.odk = otk->odk;
    button->widget.selectable = OTK_SELECTABLE_MOUSE | OTK_SELECTABLE_KEY;
    button->widget.draw = button_draw;
    button->widget.destroy = button_destroy;
    button->widget.needupdate = 0;
    button->widget.font = ho_strdup ("sans");
    button->widget.fontsize = 20;
    button->widget.alignment = ODK_ALIGN_LEFT | ODK_ALIGN_VCENTER;

    button->cb = cb;
    button->cb_data = cb_data;

    return button;
}

otk_widget_t *
otk_button_new (otk_t * otk, int x, int y,
                int w, int h, char *text, otk_button_cb_t cb, void *cb_data)
{
    otk_button_t *button = button_new_basic (otk, x, y, w, h, cb, cb_data);

    if (text)
        button->text = ho_strdup (text);
    else
        button->text = NULL;

    widget_append (otk, (otk_widget_t *) button);
    return (otk_widget_t *) button;
}

otk_widget_t *
otk_pixmap_button_new (otk_t * otk, int x, int y,
                       int w, int h, uint8_t * pixmap,
                       otk_button_cb_t cb, void *cb_data)
{
    otk_button_t *button = button_new_basic (otk, x, y, w, h, cb, cb_data);
    button->pixmap = pixmap;
    widget_append (otk, (otk_widget_t *) button);
    return (otk_widget_t *) button;
}

void
otk_button_set_text (otk_widget_t * this, const char *text)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_BUTTON))
        return;

    if (button->text)
        ho_free (button->text);

    if (text)
        button->text = ho_strdup (text);
    else
        button->text = NULL;
}


/*
 * ***********************************************************************************
 * Slider-Widget 
 * ***********************************************************************************
 */
static void
slider_destroy (otk_widget_t * this)
{
    otk_slider_t *slider = (otk_slider_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_SLIDER))
        return;

    widget_remove (this->otk, this);

    ho_free (slider);
}

static void
slider_draw (otk_widget_t * this)
{
    otk_slider_t *slider = (otk_slider_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_SLIDER))
        return;

    int value = 0;
    if (slider->get_value_cb)
        value = slider->get_value_cb (slider->get_value_cb_data);
    if (value < 0)
        value = slider->last_value;
    slider->last_value = value;

    odk_draw_rect (this->odk, this->x, this->y,
                   this->x + this->w, this->y + this->h,
                   this->otk->textpalette_button[this->focus] +
                   OSD_TEXT_PALETTE_BACKGROUND, 1);

    switch (slider->direction) {
        case OTK_SLIDER_HORIZONTAL:
            {
                int marker_middle_x = (value * this->w) / 100 + this->x + 5;
                if (marker_middle_x + 10 >= this->w + this->x)
                    marker_middle_x = this->w + this->x - 10;
                if (marker_middle_x - 10 <= this->x)
                    marker_middle_x = this->x + 10;

                odk_draw_rect (this->odk,
                               marker_middle_x - 5,
                               this->y + this->h / 2 - 15,
                               marker_middle_x + 5,
                               this->y + this->h / 2 + 15,
                               this->otk->textpalette_button[this->focus] +
                               OSD_TEXT_PALETTE_FOREGROUND, 1);
            }
            break;
        case OTK_SLIDER_VERTICAL:
            {
                int marker_middle_y = (value * this->h) / 100 + this->y + 5;
                if (marker_middle_y + 5 > this->h + this->y)
                    marker_middle_y = this->h + this->y - 5;
                if (marker_middle_y - 5 < this->y)
                    marker_middle_y = this->y + 5;
                odk_draw_rect (this->odk,
                               this->x + this->w / 2 - 15,
                               marker_middle_y - 5,
                               this->x + this->w / 2 + 15,
                               marker_middle_y + 5,
                               this->otk->textpalette_button[this->focus] +
                               OSD_TEXT_PALETTE_FOREGROUND, 1);
            }
    }
}

otk_widget_t *
otk_slider_new (otk_t * otk, int x, int y, int w, int h, int direction,
                otk_slider_data_cb_t get_value_cb, void *get_value_cb_data,
                otk_slider_cb_t cb, void *cb_data)
{
    otk_slider_t *slider = ho_new (otk_slider_t);

    slider->widget.type = OTK_WIDGET_SLIDER;
    slider->widget.x = otk->window->widget.x + x;
    slider->widget.y = otk->window->widget.y + y;
    slider->widget.w = w;
    slider->widget.h = h;
    slider->widget.otk = otk;
    slider->widget.odk = otk->odk;
    slider->widget.selectable = OTK_SELECTABLE_MOUSE;
    slider->widget.draw = slider_draw;
    slider->widget.destroy = slider_destroy;
    slider->widget.needupdate = 0;
    slider->direction = direction;
    slider->get_value_cb = get_value_cb;
    slider->get_value_cb_data = get_value_cb_data;
    slider->cb = cb;
    slider->cb_data = cb_data;

    widget_append (otk, (otk_widget_t *) slider);

    return (otk_widget_t *) slider;
}

/*
 * ***********************************************************************************
 * Checkbox-Widget 
 * ***********************************************************************************
 */
static void
checkbox_destroy (otk_widget_t * this)
{
    otk_checkbox_t *checkbox = (otk_checkbox_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_CHECKBOX))
        return;

    widget_remove (this->otk, this);

    ho_free (checkbox);
}

static void
checkbox_draw (otk_widget_t * this)
{
    otk_checkbox_t *checkbox = (otk_checkbox_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_CHECKBOX))
        return;

    odk_draw_rect (this->odk, this->x, this->y,
                   this->x + this->w, this->y + this->h,
                   this->otk->textpalette_button[this->focus] +
                   OSD_TEXT_PALETTE_BACKGROUND, 1);

    odk_draw_rect (this->odk, this->x, this->y, this->x + this->w - 1,
                   this->y + this->h - 1,
                   this->otk->textpalette_button[0] +
                   OSD_TEXT_PALETTE_FOREGROUND, 0);

    if (checkbox->is_checked) {
        odk_draw_text (this->odk, this->x + this->w / 2,
                       this->y + this->w / 2,
                       "X", ODK_ALIGN_CENTER | ODK_ALIGN_VCENTER,
                       this->otk->textpalette_button[1]);
    }
}

otk_widget_t *
otk_checkbox_new (otk_t * otk, int x, int y, int is_checked,
                  otk_button_cb_t cb, void *cb_data)
{
    otk_checkbox_t *checkbox = ho_new (otk_checkbox_t);

    checkbox->widget.type = OTK_WIDGET_CHECKBOX;
    checkbox->widget.x = otk->window->widget.x + x;
    checkbox->widget.y = otk->window->widget.y + y;
    checkbox->widget.w = 30;
    checkbox->widget.h = 30;
    checkbox->widget.otk = otk;
    checkbox->widget.odk = otk->odk;
    checkbox->widget.selectable = OTK_SELECTABLE_MOUSE | OTK_SELECTABLE_KEY;
    checkbox->widget.draw = checkbox_draw;
    checkbox->widget.destroy = checkbox_destroy;
    checkbox->widget.needupdate = 0;
    checkbox->is_checked = is_checked;
    checkbox->cb = cb;
    checkbox->cb_data = cb_data;

    widget_append (otk, (otk_widget_t *) checkbox);

    return (otk_widget_t *) checkbox;
}

/*
 * ***********************************************************************************
 * Scrollbar-Widget
 * ***********************************************************************************
 */
static void
scrollbar_destroy (otk_widget_t * this)
{
    otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_SCROLLBAR))
        return;

    widget_remove (this->otk, this);

    ho_free (scrollbar);
}

static void
scrollbar_draw (otk_widget_t * this)
{
    otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_SCROLLBAR))
        return;

    odk_draw_rect (this->odk, this->x, this->y,
                   this->x + this->w, this->y + this->h,
                   this->otk->textpalette_button[this->focus] +
                   OSD_TEXT_PALETTE_BACKGROUND, 1);

    int top_y = this->y + 5 + (scrollbar->position * (this->h - 10)) / 100;
    int bot_y =
        this->y + 5 +
        ((scrollbar->position + scrollbar->length) * (this->h - 10)) / 100;

    odk_draw_rect (this->odk, this->x + 5, top_y,
                   this->x + this->w - 5, bot_y,
                   this->otk->textpalette_button[this->focus] +
                   OSD_TEXT_PALETTE_FOREGROUND, 1);
}

otk_widget_t *
otk_scrollbar_new (otk_t * otk, int x, int y, int w, int h,
                   otk_scrollbar_cb_t cb_click, otk_button_cb_t cb_up,
                   otk_button_cb_t cb_down, void *cb_data)
{
    otk_scrollbar_t *scrollbar = ho_new (otk_scrollbar_t);

    scrollbar->widget.type = OTK_WIDGET_SCROLLBAR;
    scrollbar->widget.x = otk->window->widget.x + x;
    scrollbar->widget.y = otk->window->widget.y + y + w;
    scrollbar->widget.w = w;
    scrollbar->widget.h = h - (2 * w);
    scrollbar->widget.otk = otk;
    scrollbar->widget.odk = otk->odk;
    scrollbar->widget.selectable = OTK_SELECTABLE_MOUSE;
    scrollbar->widget.draw = scrollbar_draw;
    scrollbar->widget.destroy = scrollbar_destroy;
    scrollbar->widget.needupdate = 0;
    scrollbar->cb_click = cb_click;
    scrollbar->cb_up = cb_up;
    scrollbar->cb_down = cb_down;
    scrollbar->cb_data = cb_data;
    scrollbar->position = 0;
    scrollbar->length = 100;

    otk_pixmap_button_new (otk, x, y, w, w,
                           odk_get_bitmap (PIXMAP_SIMPLE_ARROW_UP), cb_up,
                           cb_data);
    otk_pixmap_button_new (otk, x, y + h - w, w, w,
                           odk_get_bitmap (PIXMAP_SIMPLE_ARROW_DOWN), cb_down,
                           cb_data);

    widget_append (otk, (otk_widget_t *) scrollbar);

    return (otk_widget_t *) scrollbar;
}

void
otk_scrollbar_set (otk_widget_t * this, int position, int length)
{
    otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_SCROLLBAR))
        return;

    if (position < 0)
        position = 0;
    if (position > 100)
        position = 100;
    scrollbar->position = position;

    if (length < 0)
        length = 0;
    if (length > 100)
        length = 100;
    scrollbar->length = length;
}

/*
 * ***********************************************************************************
 * List-Widget Callback Stuff
 * ***********************************************************************************
 */
static void listentries_adapt (otk_list_t * list);

static void
list_scroll_down (void *data)
{
    otk_list_t *list = (otk_list_t *) data;

    list->first_visible++;
    listentries_adapt (list);
    otk_draw (list->widget.otk);
}

static void
list_scroll_up (void *data)
{
    otk_list_t *list = (otk_list_t *) data;

    list->first_visible--;
    listentries_adapt (list);
    otk_draw (list->widget.otk);
}

static void
list_set_position (void *data, int position)
{
    otk_list_t *list = (otk_list_t *) data;

    list->first_visible = position * list->num_entries / 100;
    listentries_adapt (list);
    otk_draw (list->widget.otk);
}

/*
 * ***********************************************************************************
 * ListEntry-Widget
 * ***********************************************************************************
 */
static void
listentry_destroy (otk_widget_t * this)
{
    otk_listentry_t *listentry = (otk_listentry_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LISTENTRY))
        return;

    if (listentry->text)
        ho_free (listentry->text);

    widget_remove (this->otk, this);

    ho_free (listentry);
}

static void
listentry_draw (otk_widget_t * this)
{
    otk_listentry_t *listentry = (otk_listentry_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LISTENTRY))
        return;

    if (!listentry->is_visible)
        return;

    if (this->focus) {
        odk_draw_rect (this->odk, this->x, this->y,
                       this->x + this->w, this->y + this->h,
                       this->otk->textpalette_button[1] +
                       OSD_TEXT_PALETTE_BACKGROUND, 1);
    } else if (listentry->is_selected) {
        odk_draw_rect (this->odk, this->x, this->y,
                       this->x + this->w, this->y + this->h,
                       this->otk->textpalette_button[0] +
                       OSD_TEXT_PALETTE_BACKGROUND, 1);
    }

    if (!listentry->text)
        return;

    odk_osd_set_font (this->odk, this->font, this->fontsize);
    check_text_width (listentry->widget.odk, listentry->text,
                      listentry->widget.w - 10);

    if (this->focus) {
        odk_draw_text (this->odk, this->x + 5,
                       this->y + 3,
                       listentry->text, ODK_ALIGN_LEFT | ODK_ALIGN_TOP,
                       this->otk->textpalette_button[1]);
    } else {
        odk_draw_text (this->odk, this->x + 5,
                       this->y + 3,
                       listentry->text, ODK_ALIGN_LEFT | ODK_ALIGN_TOP,
                       this->otk->textpalette_button[0]);
    }
}

otk_widget_t *
otk_listentry_new (otk_widget_t * this, const char *text,
                   otk_list_cb_t activate_cb, void *activate_cb_data,
                   otk_list_cb_t remove_cb, void *remove_cb_data)
{
    otk_list_t *list = (otk_list_t *) this;
    otk_listentry_t *entry;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return NULL;

    entry = ho_new (otk_listentry_t);

    entry->widget.type = OTK_WIDGET_LISTENTRY;
    entry->widget.w = list->entry_width;
    entry->widget.h = list->entry_height;
    entry->widget.x = list->widget.x + 5;
    entry->widget.otk = list->widget.otk;
    entry->widget.odk = list->widget.odk;
    entry->widget.selectable = 0;
    entry->widget.draw = listentry_draw;
    entry->widget.destroy = listentry_destroy;
    entry->widget.needupdate = 0;
    entry->widget.font = ho_strdup (list->widget.font);
    entry->widget.fontsize = list->widget.fontsize;

    entry->list = list;
    entry->activate_cb_data = activate_cb_data;
    entry->activate_cb = activate_cb;
    entry->remove_cb_data = remove_cb_data;
    entry->remove_cb = remove_cb;

    if (text)
        entry->text = ho_strdup (text);
    else
        entry->text = NULL;

    widget_append (list->widget.otk, (otk_widget_t *) entry);
    l_list_append (list->entries, entry);

    list->num_entries++;
    listentries_adapt (list);

    return (otk_widget_t *) entry;
}


void
otk_list_clear_selection (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    otk_listentry_t *entry = l_list_first (list->entries);
    while (entry) {
        entry->is_selected = 0;
        entry = l_list_next (list->entries, entry);
    }

    list->num_selected = 0;
}

int
otk_list_get_selected_count (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return 0;

    return list->num_selected;
}

int *
otk_list_get_selected_pos (otk_widget_t * this, int *num_selected)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return NULL;

    *num_selected = list->num_selected;

    if (list->num_selected) {
        int counter = 0;
        int *selected_entries = ho_new (list->num_selected * sizeof (int));

        otk_listentry_t *entry = l_list_first (list->entries);
        int pos = 0;
        while (entry) {
            if (entry->is_selected) {
                selected_entries[counter++] = pos;
            }
            entry = l_list_next (list->entries, entry);
            pos++;
        }
        return selected_entries;
    }
    return NULL;
}

void **
otk_list_get_selected (otk_widget_t * this, int *num_selected)
{
    otk_list_t *list = (otk_list_t *) this;

    *num_selected = 0;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return NULL;

    *num_selected = list->num_selected;

    if (list->num_selected) {
        int counter = 0;
        void **selected_entries =
            ho_malloc (list->num_selected * sizeof (void *));

        otk_listentry_t *entry = l_list_first (list->entries);
        while (entry) {
            if (entry->is_selected) {
                selected_entries[counter++] = entry->activate_cb_data;
            }
            entry = l_list_next (list->entries, entry);
        }
        return selected_entries;
    }
    return NULL;
}


static void
otk_listentry_set_pos (otk_listentry_t * entry, int pos)
{
    if ((pos <= 0) || (pos > entry->list->num_visible)) {
        entry->is_visible = 0;
        entry->widget.selectable = 0;
        entry->is_first = 0;
        entry->is_last = 0;
        return;
    }
    entry->pos_visible_list = pos;
    entry->is_first = 0;
    entry->is_last = 0;

    entry->widget.y =
        entry->list->widget.y + 5 + entry->list->entry_height * (pos - 1);
    entry->is_visible = 1;
    entry->widget.selectable = OTK_SELECTABLE_MOUSE | OTK_SELECTABLE_KEY;

    /* we grab focus if nothing is selected */
    if (pos == 1) {
        entry->is_first = 1;
        if (!entry->widget.otk->window->focus_ptr) {
            otk_widget_set_focus ((otk_widget_t *) entry);
        }
    }
    if (pos == entry->list->num_visible || pos == entry->list->num_entries)
        entry->is_last = 1;
}


static void
listentries_adapt (otk_list_t * list)
{
    if ((list->num_entries - list->first_visible) < list->num_visible)
        list->first_visible = list->num_entries - list->num_visible;
    if (list->first_visible < 0)
        list->first_visible = 0;

    int i = 1;
    otk_listentry_t *entry = l_list_first (list->entries);
    while (entry) {
        entry->pos_complete_list = i;
        otk_listentry_set_pos (entry, i - list->first_visible);
        entry = l_list_next (list->entries, entry);
        i++;
    }

    if (list->num_entries && list->scrollbar) {
        otk_scrollbar_set (list->scrollbar,
                           list->first_visible * 100 / list->num_entries,
                           list->num_visible * 100 / list->num_entries);
    }
}

/*
 * ***********************************************************************************
 * List-Widget
 * ***********************************************************************************
 */
static void
list_destroy (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    l_list_free (list->entries, widget_destroy);

    widget_remove (this->otk, this);

    ho_free (list);
}

static void
list_draw (otk_widget_t * this)
{
    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    odk_draw_rect (this->odk, this->x, this->y, this->x + this->w,
                   this->y + this->h,
                   this->otk->textpalette_button[0] +
                   OSD_TEXT_PALETTE_TRANSPARENT, 0);

    odk_draw_rect (this->odk, this->x, this->y, this->x + this->w - 1,
                   this->y + this->h - 1,
                   this->otk->textpalette_button[0] +
                   OSD_TEXT_PALETTE_FOREGROUND, 0);
}

otk_widget_t *
otk_list_new (otk_t * otk, int x, int y, int w,
              int h, int allow_select, int allow_remove, void *list_cb_data)
{
    otk_list_t *list = ho_new (otk_list_t);

    list->widget.type = OTK_WIDGET_LIST;
    list->widget.x = otk->window->widget.x + x;
    list->widget.y = otk->window->widget.y + y;
    list->widget.w = w;
    list->widget.h = h;
    list->widget.otk = otk;
    list->widget.odk = otk->odk;
    list->widget.selectable = 0;
    list->widget.draw = list_draw;
    list->widget.destroy = list_destroy;
    list->widget.needupdate = 0;
    list->widget.font = ho_strdup ("sans");
    list->widget.fontsize = 20;

    list->allow_select = allow_select;
    list->allow_remove = allow_remove;

    list->entry_height = 30;

    list->entries = l_list_new ();

    list->num_entries = 0;
    list->num_selected = 0;
    list->num_visible = (list->widget.h - 10) / (list->entry_height);

    list->first_visible = 0;

    list->cb_data = list_cb_data;

    if (list->num_visible > 1) {
        list->entry_width = list->widget.w - 45;
        list->scrollbar =
            otk_scrollbar_new (otk, x + w - 35, y + 5, 30,
                               h - 10, list_set_position,
                               list_scroll_up, list_scroll_down, list);
    } else {
        list->entry_width = list->widget.w - 80;
        list->scrollbar = NULL;
        otk_pixmap_button_new (otk, x + w - 35,
                               y + 5, 30, 30,
                               odk_get_bitmap (PIXMAP_SIMPLE_ARROW_UP),
                               list_scroll_up, list);
        otk_pixmap_button_new (otk, x + w - 70,
                               y + 5, 30, 30,
                               odk_get_bitmap (PIXMAP_SIMPLE_ARROW_DOWN),
                               list_scroll_down, list);
    }

    widget_append (otk, (otk_widget_t *) list);

    return (otk_widget_t *) list;
}

int
otk_list_get_focus (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;
    otk_t *otk = this->otk;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return 0;

    if (!otk->window)
        return 0;

    if (!otk->window->focus_ptr)
        return 0;

    int i = 0;
    otk_listentry_t *current = l_list_first (list->entries);
    while (current) {
        if (((otk_widget_t *) current) == otk->window->focus_ptr)
            return i;
        current = l_list_next (list->entries, current);
        i++;
    }

    return 0;
}

void
otk_list_set_focus (otk_widget_t * this, int pos)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    if (pos > list->num_entries - 1)
        pos = list->num_entries - 1;

    int i = 0;
    otk_listentry_t *current = l_list_first (list->entries);
    while (current) {
        if (i == pos) {
            otk_widget_set_focus ((otk_widget_t *) current);
            return;
        }
        current = l_list_next (list->entries, current);
        i++;
    }
}

void
otk_list_set_pos (otk_widget_t * this, int newpos)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    list->first_visible = newpos;
    listentries_adapt (list);
}

int
otk_list_get_pos (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return -1;

    return list->first_visible;
}

void
otk_clear_list (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    list->num_entries = 0;
    list->num_selected = 0;
    l_list_clear (list->entries, widget_destroy);
}

void
otk_list_set_selected (otk_widget_t * this, int pos, int selected)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    int i = 0;
    otk_listentry_t *current = l_list_first (list->entries);
    while (current) {
        current->is_selected = 0;
        if (i == pos) {
            if (!current->is_selected && selected)
                list->num_selected++;

            if (current->is_selected && !selected)
                list->num_selected++;

            current->is_selected = selected;

            return;
        }
        current = l_list_next (list->entries, current);
        i++;
    }
}

/*
 * ***********************************************************************************
 * Selector-Widget
 * ***********************************************************************************
 */
static void
selector_scroll_up (void *selector_p)
{
    otk_selector_t *selector = (otk_selector_t *) selector_p;
    otk_list_t *list = (otk_list_t *) selector;

    list_scroll_up (list);

    if (selector->cb)
        selector->cb (selector->cb_data, list->first_visible);
}

static void
selector_scroll_down (void *selector_p)
{
    otk_selector_t *selector = (otk_selector_t *) selector_p;
    otk_list_t *list = (otk_list_t *) selector;

    list_scroll_down (list);

    if (selector->cb)
        selector->cb (selector->cb_data, list->first_visible);
}

otk_widget_t *
otk_selector_new (otk_t * otk, int x, int y, int w,
                  otk_selector_cb_t cb, void *cb_data)
{
    otk_selector_t *selector = ho_new (otk_selector_t);
    otk_list_t *list = (otk_list_t *) selector;

    list->widget.type = OTK_WIDGET_LIST;
    list->widget.x = otk->window->widget.x + x;
    list->widget.y = otk->window->widget.y + y;
    list->widget.w = w;
    list->widget.h = 40;
    list->widget.otk = otk;
    list->widget.odk = otk->odk;
    list->widget.selectable = 0;
    list->widget.draw = list_draw;
    list->widget.destroy = list_destroy;
    list->widget.needupdate = 0;
    list->widget.font = ho_strdup ("sans");
    list->widget.fontsize = 20;

    list->allow_select = 0;
    list->allow_remove = 0;

    list->entry_height = 30;

    list->entries = l_list_new ();

    list->num_entries = 0;
    list->num_selected = 0;
    list->num_visible = 1;

    list->first_visible = 0;

    list->cb_data = NULL;

    list->entry_width = list->widget.w - 80;
    list->scrollbar = NULL;
    otk_pixmap_button_new (otk, x + w - 35,
                           y + 5, 30, 30,
                           odk_get_bitmap (PIXMAP_SIMPLE_ARROW_UP),
                           selector_scroll_up, selector);
    otk_pixmap_button_new (otk, x + w - 70,
                           y + 5, 30, 30,
                           odk_get_bitmap (PIXMAP_SIMPLE_ARROW_DOWN),
                           selector_scroll_down, selector);

    selector->cb = cb;
    selector->cb_data = cb_data;

    widget_append (otk, (otk_widget_t *) selector);

    return (otk_widget_t *) selector;
}

/*
 * ***********************************************************************************
 * Label-Widget
 * ***********************************************************************************
 */
void
otk_label_set_upcall (otk_widget_t * this, otk_label_uc_t uc, void *uc_data)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LABEL))
        return;

    label->uc = uc;
    label->widget.needupdate = 1;
    label->uc_data = uc_data;
}

void
otk_label_set_text (otk_widget_t * this, const char *text)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LABEL))
        return;

    if (label->text)
        ho_free (label->text);

    if (text)
        label->text = ho_strdup (text);
    else
        label->text = NULL;
}

void
otk_label_set_max_width (otk_widget_t * this, int max_text_width)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LABEL))
        return;

    label->max_text_width = max_text_width;
}

static void
label_draw (otk_widget_t * this)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LABEL))
        return;

    if (!label->text)
        return;

    odk_osd_set_font (this->odk, this->font, this->fontsize);

    if (label->max_text_width)
        check_text_width (this->odk, label->text, label->max_text_width);

    odk_draw_text (this->odk, this->x, this->y, label->text,
                   this->alignment, this->otk->textpalette_label);
}

static void
label_destroy (otk_widget_t * this)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LABEL))
        return;

    widget_remove (this->otk, this);

    if (label->text)
        ho_free (label->text);

    ho_free (label);
}

otk_widget_t *
otk_label_new (otk_t * otk, int x, int y, int alignment, const char *text)
{
    otk_label_t *label = ho_new (otk_label_t);

    label->widget.type = OTK_WIDGET_LABEL;
    label->widget.x = otk->window->widget.x + x;
    label->widget.y = otk->window->widget.y + y;
    label->widget.otk = otk;
    label->widget.odk = otk->odk;
    label->widget.draw = label_draw;
    label->widget.destroy = label_destroy;
    label->widget.needupdate = 0;
    label->widget.font = ho_strdup ("sans");
    label->widget.fontsize = 30;
    label->widget.alignment = alignment;

    label->max_text_width = 0;
    if (text)
        label->text = ho_strdup (text);
    else
        label->text = NULL;

    widget_append (otk, (otk_widget_t *) label);

    return (otk_widget_t *) label;
}

/*
 * ***********************************************************************************
 * Border-Widget
 * ***********************************************************************************
 */
static void
border_draw (otk_widget_t * this)
{
    if (!is_correct_widget (this, OTK_WIDGET_BORDER))
        return;

    // TODO: this should work by drawing one rectangle only (bug in xine-lib?)
    odk_draw_rect (this->odk, this->x, this->y, this->x + this->w,
                   this->y + this->h,
                   this->otk->textpalette_label + OSD_TEXT_PALETTE_FOREGROUND,
                   0);
    odk_draw_rect (this->odk, this->x, this->y, this->x + this->w,
                   this->y + this->h + 1,
                   this->otk->textpalette_label + OSD_TEXT_PALETTE_FOREGROUND,
                   0);
}


static void
border_destroy (otk_widget_t * this)
{
    otk_border_t *border = (otk_border_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_BORDER))
        return;

    widget_remove (this->otk, this);

    ho_free (border);
}


otk_widget_t *
otk_border_new (otk_t * otk, int x, int y, int w, int h)
{
    otk_border_t *border = ho_new (otk_border_t);

    border->widget.type = OTK_WIDGET_BORDER;
    border->widget.x = otk->window->widget.x + x;
    border->widget.y = otk->window->widget.y + y;
    border->widget.w = w;
    border->widget.h = h;
    border->widget.otk = otk;
    border->widget.odk = otk->odk;
    border->widget.draw = border_draw;
    border->widget.destroy = border_destroy;

    widget_append (otk, (otk_widget_t *) border);

    return (otk_widget_t *) border;
}


/*
 * ***********************************************************************************
 * Window-Widget
 * ***********************************************************************************
 * There is always one current window. This current window is set when
 * creating a new window. It can also be set by calling otk_window_set_current.
 *
 * There are two possibilities of how windows are destroyed:
 *
 * 1. The user keeps a pointer to the window in a variable and marks the
 *    window not to be destroyed by calling otk_window_keep. The user can
 *    himself destroy the window by calling otk_window_destroy.
 *
 * 2. The window is automatically destroyed as soon as a new window is created.
 *
 * When calling otk_free all windows will be destroyed by otk.
 * ***********************************************************************************
 */
static void
window_draw (otk_widget_t * this)
{
    otk_window_t *window = (otk_window_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_WINDOW))
        return;

    if (window->fill) {
        odk_draw_rect (this->odk, window->widget.x, window->widget.y,
                       window->widget.x + window->widget.w,
                       window->widget.y + window->widget.h,
                       this->otk->textpalette_label +
                       OSD_TEXT_PALETTE_BACKGROUND, 1);
    } else if (!window->fill && window->border) {
        odk_draw_rect (this->odk, window->widget.x,
                       window->widget.y,
                       window->widget.x + window->widget.w,
                       window->widget.y + window->widget.h,
                       this->otk->textpalette_button[0] +
                       OSD_TEXT_PALETTE_TRANSPARENT, 0);
    }

    if (window->border) {
        odk_draw_rect (this->odk, window->widget.x, window->widget.y,
                       window->widget.x + window->widget.w - 1,
                       window->widget.y + window->widget.h - 1,
                       this->otk->textpalette_label +
                       OSD_TEXT_PALETTE_FOREGROUND, 0);
    }

    otk_widget_t *widget = l_list_first (window->subs);
    while (widget) {
        widget->draw (widget);
        widget = l_list_next (window->subs, widget);
    }
}

static void
window_destroy (otk_widget_t * this)
{
    otk_t *otk = this->otk;
    otk_window_t *window = (otk_window_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_WINDOW))
        return;

#ifdef DEBUG_THREADS
    debug ("[thread: %d] waiting for lock on draw_mutex",
           (int) pthread_self ());
#endif
    pthread_mutex_lock (&otk->draw_mutex);
#ifdef DEBUG_THREADS
    debug ("[thread: %d] aquired lock on draw_mutex", (int) pthread_self ());
#endif

    otk_window_t *current_window = this->otk->window;
    this->otk->window = window;

    // we free all widgets contained in this window
    l_list_free (window->subs, widget_destroy);
    // we remove this window from the list of windows
    l_list_remove (this->otk->windows, window);

    if (current_window == window)
        this->otk->window = NULL;
    else
        this->otk->window = current_window;

    ho_free (window);

    pthread_mutex_unlock (&otk->draw_mutex);
#ifdef DEBUG_THREADS
    debug ("[thread: %d] released lock on draw_mutex", (int) pthread_self ());
#endif
}

otk_widget_t *
otk_window_new (otk_t * otk, int x, int y, int w, int h, int border, int fill)
{
    if (otk->window && !otk->window->keep)
        window_destroy ((otk_widget_t *) otk->window);

    otk->window = ho_new (otk_window_t);

    otk->window->widget.type = OTK_WIDGET_WINDOW;
    otk->window->widget.x = x;
    otk->window->widget.y = y;
    otk->window->widget.w = w;
    otk->window->widget.h = h;
    otk->window->widget.otk = otk;
    otk->window->widget.odk = otk->odk;
    otk->window->widget.draw = window_draw;
    otk->window->widget.destroy = window_destroy;
    otk->window->widget.needupdate = 0;
    otk->window->keep = 0;
    otk->window->fill = fill;
    otk->window->border = border;
    otk->window->subs = l_list_new ();

    l_list_append (otk->windows, otk->window);

    return (otk_widget_t *) otk->window;
}

void
otk_window_keep (otk_widget_t * this, int keep)
{
    otk_window_t *window = (otk_window_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_WINDOW))
        return;

    window->keep = keep;
}

void
otk_window_destroy (otk_widget_t * this)
{
    if (!is_correct_widget (this, OTK_WIDGET_WINDOW))
        return;

    window_destroy (this);
}

void
otk_window_set_current (otk_t * otk, otk_widget_t * this)
{
    otk_window_t *window = (otk_window_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_WINDOW))
        return;

    otk->window = window;
}

/*
 * ***********************************************************************************
 * Eventhandlers
 * ***********************************************************************************
 */
static void
listentry_select (otk_listentry_t * listentry)
{
    listentry->doubleclicktime = time (NULL);
    switch (listentry->list->allow_select) {
        case OTK_LIST_SINGLE_SELECTION:
            {
                otk_listentry_t *current =
                    l_list_first (listentry->list->entries);
                while (current) {
                    if (current != listentry) {
                        current->is_selected = 0;
                    } else {
                        current->is_selected = (current->is_selected + 1) % 2;
                    }
                    current = l_list_next (listentry->list->entries, current);
                }
                listentry->list->num_selected = 1;
            }
            break;
        case OTK_LIST_MULTIPLE_SELECTION:
            {
                // kleiner Hack :-)
                listentry->list->num_selected -= listentry->is_selected;
                listentry->is_selected = (listentry->is_selected + 1) % 2;
                listentry->list->num_selected += listentry->is_selected;
            }
            break;
    }
    otk_draw (listentry->list->widget.otk);
}

static void
listentry_activate (otk_listentry_t * listentry)
{
    listentry->doubleclicktime = 0;

    otk_listentry_t *current = l_list_first (listentry->list->entries);
    while (current) {
        current->is_selected = 0;
        current = l_list_next (listentry->list->entries, current);
    }
    listentry->list->num_selected = 0;

    if (listentry->activate_cb)
        listentry->activate_cb (listentry->list->cb_data,
                                listentry->activate_cb_data);

}

static void
listentry_select_activate_handler (otk_listentry_t * listentry)
{
    time_t cur_time = time (NULL);
    if (cur_time - 1 < listentry->doubleclicktime) {
        listentry_activate (listentry);
    } else {
        listentry_select (listentry);
    }
}

static void
motion_handler (otk_t * otk, oxine_event_t * ev)
{
    if (!otk->window)
        return;

    otk_widget_t *widget =
        find_widget_xy (otk, ev->data.where.x, ev->data.where.y);

    if (widget && widget->selectable) {
        if (otk->window->focus_ptr != widget) {
            otk_widget_set_focus (widget);
            otk_draw (otk);
        }
    } else if (otk->window->focus_ptr) {
        if (otk->window->focus_ptr && otk->window->focus_ptr->focus_leave_cb)
            otk->window->focus_ptr->focus_leave_cb (otk->window->focus_ptr->
                                                    focus_leave_cb_data);

        otk->window->focus_ptr->focus = 0;
        otk->window->focus_ptr = NULL;
        otk_draw (otk);
    }
}

static void
button_handler (otk_t * otk, oxine_event_t * ev)
{
    if (!otk->window)
        return;

    if (ev->source.button == OXINE_BUTTON_NULL)
        return;

    otk_widget_t *widget =
        find_widget_xy (otk, ev->data.where.x, ev->data.where.y);

    if (!widget)
        return;

    switch (ev->source.button) {
        case OXINE_BUTTON1:    // left button
            if (is_correct_widget (widget, OTK_WIDGET_BUTTON)) {
                otk_button_t *button = (otk_button_t *) widget;

                if (button->cb)
                    button->cb (button->cb_data);
            }

            else if (is_correct_widget (widget, OTK_WIDGET_LISTENTRY)) {
                otk_listentry_t *listentry = (otk_listentry_t *) widget;
                listentry_select_activate_handler (listentry);
            }

            else if (is_correct_widget (widget, OTK_WIDGET_SLIDER)) {
                otk_slider_t *slider = (otk_slider_t *) widget;

                if (slider->cb) {
                    int position = 0;
                    if (slider->direction == OTK_SLIDER_HORIZONTAL) {
                        position =
                            (ev->data.where.x - widget->x) * 100 / widget->w;
                    } else if (slider->direction == OTK_SLIDER_VERTICAL) {
                        position =
                            (ev->data.where.y - widget->y) * 100 / widget->h;
                    }
                    slider->cb (slider->cb_data, position);
                }
            }

            else if (is_correct_widget (widget, OTK_WIDGET_CHECKBOX)) {
                otk_checkbox_t *checkbox = (otk_checkbox_t *) widget;

                checkbox->is_checked = (checkbox->is_checked + 1) % 2;

                if (checkbox->cb) {
                    checkbox->cb (checkbox->cb_data);
                }
            }

            else if (is_correct_widget (widget, OTK_WIDGET_SCROLLBAR)) {
                otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) widget;

                if (scrollbar->cb_click) {
                    int position =
                        (ev->data.where.y - widget->y) * 100 / widget->h;
                    scrollbar->cb_click (scrollbar->cb_data, position);
                }
            }
            break;

        case OXINE_BUTTON4:    // scrollwheel down
            if (is_correct_widget (widget, OTK_WIDGET_LISTENTRY)) {
                otk_listentry_t *entry =
                    (otk_listentry_t *) otk->window->focus_ptr;

                entry->list->first_visible--;
                listentries_adapt (entry->list);
                otk_draw (otk);
            }

            else if (is_correct_widget (widget, OTK_WIDGET_SLIDER)) {
                otk_slider_t *slider = (otk_slider_t *) widget;

                if (slider->cb && slider->get_value_cb) {
                    int position =
                        slider->get_value_cb (slider->get_value_cb_data);
                    if (slider->direction == OTK_SLIDER_HORIZONTAL) {
                        position -= widget->w / 100;
                    } else if (slider->direction == OTK_SLIDER_VERTICAL) {
                        position -= widget->h / 100;
                    }
                    slider->cb (slider->cb_data, position);
                }
            }

            else if (is_correct_widget (widget, OTK_WIDGET_SCROLLBAR)) {
                otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) widget;

                if (scrollbar->cb_click) {
                    int position = scrollbar->position - (widget->h / 100);
                    scrollbar->cb_click (scrollbar->cb_data, position);
                }
            }
            break;

        case OXINE_BUTTON5:    // scrollwheel up
            if (is_correct_widget (widget, OTK_WIDGET_LISTENTRY)) {
                otk_listentry_t *entry =
                    (otk_listentry_t *) otk->window->focus_ptr;
                entry->list->first_visible++;
                listentries_adapt (entry->list);
                otk_draw (otk);
            }

            else if (is_correct_widget (widget, OTK_WIDGET_SLIDER)) {
                otk_slider_t *slider = (otk_slider_t *) widget;

                if (slider->cb && slider->get_value_cb) {
                    int position =
                        slider->get_value_cb (slider->get_value_cb_data);
                    if (slider->direction == OTK_SLIDER_HORIZONTAL) {
                        position += widget->w / 100;
                    } else if (slider->direction == OTK_SLIDER_VERTICAL) {
                        position += widget->h / 100;
                    }
                    slider->cb (slider->cb_data, position);
                }
            }

            else if (is_correct_widget (widget, OTK_WIDGET_SCROLLBAR)) {
                otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) widget;

                if (scrollbar->cb_click) {
                    int position = scrollbar->position + (widget->h / 100);
                    scrollbar->cb_click (scrollbar->cb_data, position);
                }
            }
            break;
        default:
            break;
    }
}

static void
key_handler (otk_t * otk, oxine_event_t * ev)
{
    if (!otk->window)
        return;

    if (ev->source.key == OXINE_KEY_NULL)
        return;

    otk_widget_t *new_widget = NULL;

    switch (ev->source.key) {
        case OXINE_KEY_FIRST:
            if (is_correct_widget (otk->window->focus_ptr,
                                   OTK_WIDGET_LISTENTRY)) {
                otk_listentry_t *entry =
                    (otk_listentry_t *) otk->window->focus_ptr;

                entry->list->first_visible = 0;
                listentries_adapt (entry->list);

                new_widget = l_list_first (entry->list->entries);
            }
            break;
        case OXINE_KEY_LAST:
            if (is_correct_widget (otk->window->focus_ptr,
                                   OTK_WIDGET_LISTENTRY)) {
                otk_listentry_t *entry =
                    (otk_listentry_t *) otk->window->focus_ptr;

                entry->list->first_visible = entry->list->num_entries;
                listentries_adapt (entry->list);

                new_widget = l_list_last (entry->list->entries);
            }
            break;
        case OXINE_KEY_PREV:
            if (!odk_current_is_logo_mode (oxine->odk))
                break;
        case OXINE_KEY_PAGE_UP:
            if (is_correct_widget (otk->window->focus_ptr,
                                   OTK_WIDGET_LISTENTRY)) {
                otk_listentry_t *entry =
                    (otk_listentry_t *) otk->window->focus_ptr;

                entry->list->first_visible -= entry->list->num_visible;
                listentries_adapt (entry->list);

                new_widget = otk->window->focus_ptr;
                while (!((otk_listentry_t *) new_widget)->is_first) {
                    new_widget =
                        l_list_prev (entry->list->entries, new_widget);
                }
            }
            break;
        case OXINE_KEY_NEXT:
            if (!odk_current_is_logo_mode (oxine->odk))
                break;
        case OXINE_KEY_PAGE_DOWN:
            if (is_correct_widget (otk->window->focus_ptr,
                                   OTK_WIDGET_LISTENTRY)) {
                otk_listentry_t *entry =
                    (otk_listentry_t *) otk->window->focus_ptr;

                entry->list->first_visible += entry->list->num_visible;
                listentries_adapt (entry->list);

                new_widget = otk->window->focus_ptr;
                while (!((otk_listentry_t *) new_widget)->is_last) {
                    new_widget =
                        l_list_next (entry->list->entries, new_widget);
                }
            }
            break;
        case OXINE_KEY_SPEED_UP:
            if (!odk_current_is_logo_mode (oxine->odk))
                break;
        case OXINE_KEY_UP:
            if (is_correct_widget (otk->window->focus_ptr,
                                   OTK_WIDGET_LISTENTRY)) {
                otk_listentry_t *entry =
                    (otk_listentry_t *) otk->window->focus_ptr;

                if (entry->is_first) {
                    entry->list->first_visible--;
                    listentries_adapt (entry->list);
                }
                if (!entry->is_first) {
                    new_widget = l_list_prev (entry->list->entries, entry);
                }
            } else {
                new_widget = find_neighbour (otk, DIRECTION_UP);
            }
            break;
        case OXINE_KEY_SPEED_DOWN:
            if (!odk_current_is_logo_mode (oxine->odk))
                break;
        case OXINE_KEY_DOWN:
            if (is_correct_widget (otk->window->focus_ptr,
                                   OTK_WIDGET_LISTENTRY)) {
                otk_listentry_t *entry =
                    (otk_listentry_t *) otk->window->focus_ptr;

                if (entry->is_last) {
                    entry->list->first_visible++;
                    listentries_adapt (entry->list);
                }
                if (!entry->is_last) {
                    new_widget = l_list_next (entry->list->entries, entry);
                }
            } else {
                new_widget = find_neighbour (otk, DIRECTION_DOWN);
            }
            break;
        case OXINE_KEY_REWIND:
            if (!odk_current_is_logo_mode (oxine->odk))
                break;
        case OXINE_KEY_LEFT:
            new_widget = find_neighbour (otk, DIRECTION_LEFT);
            break;
        case OXINE_KEY_FFORWARD:
            if (!odk_current_is_logo_mode (oxine->odk))
                break;
        case OXINE_KEY_RIGHT:
            new_widget = find_neighbour (otk, DIRECTION_RIGHT);
            break;
        case OXINE_KEY_ACTIVATE:
            if (is_correct_widget (otk->window->focus_ptr, OTK_WIDGET_BUTTON)) {
                otk_button_t *button =
                    (otk_button_t *) otk->window->focus_ptr;
                button->cb (button->cb_data);
            }

            else if (is_correct_widget (otk->window->focus_ptr,
                                        OTK_WIDGET_LISTENTRY)) {
                otk_listentry_t *entry =
                    (otk_listentry_t *) otk->window->focus_ptr;
                listentry_activate (entry);
            }
            break;
        case OXINE_KEY_SELECT:
            if (is_correct_widget (otk->window->focus_ptr,
                                   OTK_WIDGET_LISTENTRY)) {
                otk_listentry_t *entry =
                    (otk_listentry_t *) otk->window->focus_ptr;
                listentry_select (entry);
            }
            break;
        case OXINE_KEY_REMOVE:
            if (is_correct_widget (otk->window->focus_ptr,
                                   OTK_WIDGET_LISTENTRY)) {
                otk_listentry_t *entry =
                    (otk_listentry_t *) otk->window->focus_ptr;
                otk_list_t *list = entry->list;

                if (list->allow_remove) {
                    // call remove-callback so the actual data can be removed
                    if (entry->remove_cb)
                        entry->remove_cb (list->cb_data,
                                          entry->remove_cb_data);

                    // get the next entry for the focus
                    new_widget = l_list_next (list->entries, entry);
                    if (!new_widget)
                        new_widget = l_list_prev (list->entries, entry);

                    // destroy and remove the widget
                    widget_destroy (entry);
                    l_list_remove (list->entries, entry);
                    list->num_entries--;

                    // adapt the list
                    listentries_adapt (list);
                }
            }
        default:
            break;
    }

    if (is_correct_widget (new_widget, OTK_WIDGET_LISTENTRY)
        || is_correct_widget (new_widget, OTK_WIDGET_BUTTON)) {
        otk_widget_set_focus (new_widget);
        otk_draw (otk);
    }
}

static void
otk_event_handler (void *this, oxine_event_t * ev)
{
    otk_t *otk = (otk_t *) this;

#ifdef DEBUG_THREADS
    debug ("[thread: %d] waiting for lock on draw_mutex",
           (int) pthread_self ());
#endif
    pthread_mutex_lock (&otk->draw_mutex);
#ifdef DEBUG_THREADS
    debug ("[thread: %d] aquired lock on draw_mutex", (int) pthread_self ());
#endif

    switch (ev->type) {
        case OXINE_EVENT_KEY:
            key_handler (otk, ev);
            break;
        case OXINE_EVENT_MOTION:
            motion_handler (otk, ev);
            break;
        case OXINE_EVENT_BUTTON:
            button_handler (otk, ev);
            break;
        case OXINE_EVENT_FRAME_FORMAT_CHANGED:
            otk_draw (otk);
            break;
        default:
            break;
    }

    if (otk->event_handler)
        otk->event_handler (otk->event_handler_data, ev);

    pthread_mutex_unlock (&otk->draw_mutex);
#ifdef DEBUG_THREADS
    debug ("[thread: %d] released lock on draw_mutex", (int) pthread_self ());
#endif
}


/*
 * ***********************************************************************************
 * Global Functions
 * ***********************************************************************************
 */
void
otk_draw (otk_t * otk)
{
    if (!otk->window)
        return;

#ifdef DEBUG_THREADS
    debug ("[thread: %d] waiting for lock on draw_mutex",
           (int) pthread_self ());
#endif
    pthread_mutex_lock (&otk->draw_mutex);
#ifdef DEBUG_THREADS
    debug ("[thread: %d] aquired lock on draw_mutex", (int) pthread_self ());
#endif

    odk_osd_clear (otk->odk);
    otk_widget_t *widget = (otk_widget_t *) otk->window;
    widget->draw (widget);
    odk_osd_show (otk->odk);

    pthread_mutex_unlock (&otk->draw_mutex);
#ifdef DEBUG_THREADS
    debug ("[thread: %d] released lock on draw_mutex", (int) pthread_self ());
#endif
}

void
otk_clear (otk_t * otk)
{
    if (otk->window && !otk->window->keep)
        window_destroy ((otk_widget_t *) otk->window);
    otk->window = NULL;

    odk_osd_clear (otk->odk);
    odk_osd_hide (otk->odk);
}

void
otk_widget_set_focus (otk_widget_t * widget)
{
    assert (widget);
    assert (widget->otk);
    assert (widget->otk->window);

    if (widget->selectable) {
        otk_window_t *window = widget->otk->window;
        otk_widget_t *has_focus = window->focus_ptr;

        if (has_focus) {
            has_focus->focus = 0;
            if (has_focus->focus_leave_cb)
                has_focus->focus_leave_cb (has_focus->focus_leave_cb_data);
        }

        has_focus = widget;
        window->focus_ptr = widget;

        if (has_focus) {
            has_focus->focus = 1;
            if (has_focus->focus_enter_cb)
                has_focus->focus_enter_cb (has_focus->focus_enter_cb_data);
        }
    }
}

void
otk_widget_set_focus_callbacks (otk_widget_t * widget, otk_cb_t enter_cb,
                                void *enter_cb_data, otk_cb_t leave_cb,
                                void *leave_cb_data)
{
    widget->focus_enter_cb_data = enter_cb_data;
    widget->focus_enter_cb = enter_cb;
    widget->focus_leave_cb_data = leave_cb_data;
    widget->focus_leave_cb = leave_cb;
}

void
otk_set_event_handler (otk_t * otk,
                       void (*cb) (void *data, oxine_event_t * ev),
                       void *data)
{
    otk->event_handler = cb;
    otk->event_handler_data = data;
}

void
otk_widget_set_update (otk_widget_t * this, int update)
{
    if (update)
        this->needupdate = 1;
    else
        this->needupdate = 0;
}

static void
otk_update_job (void *data)
{
    otk_t *otk = (otk_t *) data;
    int changed = 0;

#ifdef DEBUG_THREADS
    debug ("[thread: %d] waiting for lock on draw_mutex",
           (int) pthread_self ());
#endif
    pthread_mutex_lock (&otk->draw_mutex);
#ifdef DEBUG_THREADS
    debug ("[thread: %d] aquired lock on draw_mutex", (int) pthread_self ());
#endif

    if (otk->window) {
        otk_widget_t *widget = l_list_first (otk->window->subs);
        while (widget) {
            if (widget->needupdate) {
                changed = 1;
                switch (widget->type) {
                    case OTK_WIDGET_LABEL:
                        {
                            otk_label_t *label = (otk_label_t *) widget;
                            if (label->uc) {
                                label->uc (label->uc_data, widget);
                            }
                            break;
                        }
                }
            }
            widget = l_list_next (otk->window->subs, widget);
        }
    }

    otk_draw (otk);

    pthread_mutex_unlock (&otk->draw_mutex);
#ifdef DEBUG_THREADS
    debug ("[thread: %d] released lock on draw_mutex", (int) pthread_self ());
#endif

    otk->update_job = schedule_job (500, otk_update_job, otk);
}

otk_t *
otk_init (odk_t * odk)
{
    otk_t *otk;
    /* int err; */
    uint32_t c[3];
    uint8_t t[3];

    otk = ho_new (otk_t);
    otk->windows = l_list_new ();

    otk->odk = odk;

    odk_set_event_handler (odk, otk_event_handler, otk);

    // define the palette for a non-focused button
    c[0] = 0xffffff;            // foreground
    t[0] = 0xf;
    c[1] = 0x2a569d;            // background
    t[1] = 0xf;
    c[2] = 0xffffff;            // border
    t[2] = 0xf;
    odk_osd_get_user_color (odk, "button_foreground", &c[0], &t[0]);
    odk_osd_get_user_color (odk, "button_background", &c[1], &t[1]);
    odk_osd_get_user_color (odk, "button_border", &c[2], &t[2]);
    otk->textpalette_button[0] =
        odk_osd_alloc_text_palette (odk, c[0], t[0], c[1], t[1], c[2], t[2]);

    // define the palette for a focused button
    c[1] = 0x5175b0;
    t[1] = 0xf;
    odk_osd_get_user_color (odk, "focused_button_foreground", &c[0], &t[0]);
    odk_osd_get_user_color (odk, "focused_button_background", &c[1], &t[1]);
    odk_osd_get_user_color (odk, "focused_button_border", &c[2], &t[2]);
    otk->textpalette_button[1] =
        odk_osd_alloc_text_palette (odk, c[0], t[0], c[1], t[1], c[2], t[2]);

    // define the palette for a label
    c[0] = 0xffffff;
    t[0] = 0xf;
    c[1] = 0x5175b0;
    t[1] = 0xf;
    c[2] = 0xffffff;
    t[2] = 0xf;
    odk_osd_get_user_color (odk, "label_foreground", &c[0], &t[0]);
    odk_osd_get_user_color (odk, "label_background", &c[1], &t[1]);
    odk_osd_get_user_color (odk, "label_border", &c[2], &t[2]);
    otk->textpalette_label =
        odk_osd_alloc_text_palette (odk, c[0], t[0], c[1], t[1], c[2], t[2]);

    pthread_mutexattr_init (&otk->draw_mutex_attr);
    pthread_mutexattr_settype (&otk->draw_mutex_attr,
                               PTHREAD_MUTEX_RECURSIVE_NP);
    pthread_mutex_init (&otk->draw_mutex, &otk->draw_mutex_attr);
    otk->update_job = schedule_job (500, otk_update_job, otk);

    return otk;
}

void
otk_free (otk_t * otk)
{
    lock_job_mutex ();
    cancel_job (otk->update_job);
    unlock_job_mutex ();

    odk_set_event_handler (otk->odk, NULL, NULL);

    otk_clear (otk);

    l_list_free (otk->windows, widget_destroy);
    otk->windows = NULL;
    otk->window = NULL;

    pthread_mutex_destroy (&otk->draw_mutex);
    pthread_mutexattr_destroy (&otk->draw_mutex_attr);

    ho_free (otk);
}
