
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_osd.c,v 1.12 2006/02/07 07:34:52 mschwerin Exp $
 *
 * This file is part of the odk system.
 *
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>

#ifdef HAVE_IMAGEMAGICK
#include <magick/ImageMagick.h>
#endif

#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "odk_private.h"

/* 
 * ***************************************************************************
 * Name:            rgb2yuv
 * Access:          private
 *
 * Description:     Converts RGB color to YUV color.
 * ***************************************************************************
 */
static uint32_t
rgb2yuv (uint32_t rgb)
{
    uint8_t r = (rgb >> 16) & 0xff;
    uint8_t g = (rgb >> 8) & 0xff;
    uint8_t b = (rgb >> 0) & 0xff;

    //    printf("\nRGB 0x%06X r:%3X g:%3X b:%3X\n", rgb, r, g, b);
    //    printf("RGB 0x%06X r:%3d g:%3d b:%3d\n", rgb, r, g, b);

    uint8_t y = ((66 * r + 129 * g + 25 * b + 128) >> 8) + 16;
    uint8_t u = ((-38 * r - 74 * g + 112 * b + 128) >> 8) + 128;
    uint8_t v = ((112 * r - 94 * g - 18 * b + 128) >> 8) + 128;

    uint32_t yuv = ((y << 16) | (v << 8) | u);

    //    printf("YUV 0x%06X y:%3X u:%3X v:%3X\n", yuv, y, u, v);
    //    printf("YUV 0x%06X y:%3d u:%3d v:%3d\n", yuv, y, u, v);

    return yuv;
}


/* 
 * ***************************************************************************
 * Name:            get_color
 * Access:          private
 *
 * Description:     Returns the index of the color in the palette. If the
 *                  color is not in the palette yet, it is added. Input colors
 *                  should be YUV.
 * ***************************************************************************
 */
static int
get_color (odk_palette_t * palette, uint32_t color, uint8_t transparency)
{
    /* The first entry of the palette should always be transparent */
    if (palette->num_colors == 0) {
        palette->colors[0] = 0x00;
        palette->transparency[0] = 0x00;
        palette->num_colors++;
    }

    /* We ignore all entries that have no color */
    if (!color)
        return 0;

    int i = 0;

    int best_index = 0;
    uint8_t best_error = 0xff;

    /* What we do here is look for the color already in the palette, that best
     * matches the color we wish to add. The color must match. What is allowed
     * to be off a bit is the transparency. */
    for (; i < palette->num_colors; i++) {
        if (palette->colors[i] == color) {
            uint8_t current_error =
                abs (palette->transparency[i] - transparency);

            if (current_error == 0)
                return i;
            if (current_error < best_error) {
                best_error = current_error;
                best_index = i;
            }
        }
    }

    /* If the error in transparency is small enough we return the index of the
     * color we found. */
    if (best_error < 10)
        return best_index;

    /* If the error is to large and the palette is already full we return
     * index 0 which is completely transparent so that this color will not
     * appear in the output. */
    if (palette->num_colors == NUM_COLORS)
        return 0;

    /* If the error is to large and the palette still has space in it, we add
     * the new color. */
    palette->colors[palette->num_colors] = color;
    palette->transparency[palette->num_colors] = transparency;
    palette->num_colors++;

    return (palette->num_colors - 1);
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_get_color
 * Access:          public
 *
 * Description:     Returns index to given color. Input colors should be RGB.
 * ***************************************************************************
 */
int
odk_osd_get_color (odk_t * odk, uint32_t color, uint8_t transparency)
{
    int color_index = get_color (odk->palette, rgb2yuv (color), transparency);

    xine_osd_set_palette (odk->osd, odk->palette->colors,
                          odk->palette->transparency);

    return color_index;
}


/* 
 * ***************************************************************************
 * Name:            palette_transition
 * Access:          private
 *
 * Description:     Generates a gradient on palette out of given values. Input
 *                  colors should be YUV.
 * ***************************************************************************
 */
static void
palette_transition (odk_palette_t * palette, int num_colors,
                    uint32_t from_color, uint8_t from_transparency,
                    uint32_t to_color, uint8_t to_transparency)
{
    double transparency_step = to_transparency - from_transparency;
    transparency_step /= num_colors;

    uint8_t cb_from = from_color & 0xff;
    uint8_t cr_from = (from_color >> 8) & 0xff;
    uint8_t y_from = (from_color >> 16) & 0xff;

    uint8_t cb_to = to_color & 0xff;
    uint8_t cr_to = (to_color >> 8) & 0xff;
    uint8_t y_to = (to_color >> 16) & 0xff;

    double cb_step = (double) (cb_to - cb_from) / num_colors;
    double cr_step = (double) (cr_to - cr_from) / num_colors;
    double y_step = (double) (y_to - y_from) / num_colors;

    int first_index = palette->num_colors;
    int last_index = first_index + num_colors;

    int i = first_index + 1;
    for (; i < last_index; i++) {
        uint8_t cb_cur = cb_from + (int8_t) (cb_step * (i - first_index));
        uint8_t cr_cur = cr_from + (int8_t) (cr_step * (i - first_index));
        uint8_t y_cur = y_from + (int8_t) (y_step * (i - first_index));

        palette->colors[i] = cb_cur + (cr_cur << 8) + (y_cur << 16);
        palette->transparency[i] = from_transparency;
        palette->transparency[i] += transparency_step * (i - first_index);
    }

    palette->colors[first_index] = from_color;
    palette->transparency[first_index] = from_transparency;

    palette->colors[last_index] = to_color;
    palette->transparency[last_index] = to_transparency;

    palette->num_colors += num_colors;
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_alloc_text_palette
 * Access:          public
 *
 * Description:     Allocates 10 colors in the palette and returns the index
 *                  to the first color. Input colors should be RGB.
 * ***************************************************************************
 */
int
odk_osd_alloc_text_palette (odk_t * odk,
                            uint32_t fg_color, uint8_t fg_transparency,
                            uint32_t bg_color, uint8_t bg_transparency,
                            uint32_t bo_color, uint8_t bo_transparency)
{
    int first_color = odk->palette->num_colors;

    /* A xine text palette always consists of 11 colors of which color 0 is
     * transparent, color 1 is the background and color 10 is the foreground. */
    odk->palette->colors[first_color] = 0;
    odk->palette->transparency[first_color] = 0;
    odk->palette->num_colors += 1;

    palette_transition (odk->palette, 10,
                        rgb2yuv (bg_color), bg_transparency,
                        rgb2yuv (fg_color), fg_transparency);

    xine_osd_set_palette (odk->osd, odk->palette->colors,
                          odk->palette->transparency);

    return first_color;
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_get_user_color
 * Access:          public
 *
 * Description:     Returns the color with the given name.
 * ***************************************************************************
 */
void
odk_osd_get_user_color (odk_t * odk, char *name, uint32_t * color,
                        uint8_t * transparency)
{
    char id[512];
    snprintf (id, 511, "oxine.color_%s", name);
    char value[64];
    snprintf (value, 63, "%06x-%01x", *color, *transparency);
    const char *v;
    v = xine_config_register_string (odk->xine, id, value,
                                     "[RGB color]-[YUV opacity]",
                                     "color spec. format: [YUV color]-[opacity 0-f]",
                                     10, NULL, NULL);

    unsigned int c;
    unsigned int t;
    if (sscanf (v, "%06x-%01x", &c, &t) != 2) {
        error ("Badly formated color spec in entry using standard color!");
        return;
    }

    *color = c;
    *transparency = t;
}


/* 
 * ***************************************************************************
 * Name:            odk_get_bitmap
 * Access:          public
 *
 * Description:     Returns predefined pixmaps
 * ***************************************************************************
 */
uint8_t *
odk_get_bitmap (int type)
{
    uint8_t playlist[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    };
    uint8_t directory[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0
    };
    uint8_t home[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0
    };
    uint8_t poweroff[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 0,
        0, 0, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 0, 0,
        0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 0,
        0, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0,
        0, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0,
        1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
        0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0,
        0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0,
        0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0,
        0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0
    };

    uint8_t arrowup[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0,
        0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1
    };

    uint8_t *p = ho_malloc (sizeof (uint8_t) * 400);

    switch (type) {
        case PIXMAP_PLAYLIST:
            memcpy (p, playlist, 400);
            break;
        case PIXMAP_FOLDER:
            memcpy (p, directory, 400);
            break;
        case PIXMAP_HOME:
            memcpy (p, home, 400);
            break;
        case PIXMAP_POWEROFF:
            memcpy (p, poweroff, 400);
            break;
        case PIXMAP_SIMPLE_ARROW_UP:
            memcpy (p, arrowup, 400);
            break;
        case PIXMAP_SIMPLE_ARROW_DOWN:
            {
                int row = 19;
                for (; row >= 0; row--) {
                    memcpy (p + (19 - row) * 20, arrowup + row * 20, 20);
                }
            }
            break;
    }

    return p;
}


/* 
 * ***************************************************************************
 * Name:            odk_draw_bitmap
 * Access:          public
 *
 * Description:     Draw a bitmap. x and y define the center for the bitmap!
 * ***************************************************************************
 */
void
odk_draw_bitmap (odk_t * odk,
                 uint8_t * bitmap, int x, int y, int w, int h,
                 uint8_t * palette_map)
{
    if (!(odk->hscale && odk->vscale))
        return;

    //int pw = (int) ((double) w * odk->hscale);
    //int ph = (int) ((double) h * odk->vscale);
    int px = (int) ((double) x * odk->hscale) - w / 2;
    int py = (int) ((double) y * odk->vscale) - h / 2;

    // TODO: scale bitmap
    /*
       uint8_t pbmp[pw * ph];
       printf ("%d %d %d %d\n", w, h, pw, ph);
       for (y = 0; y < ph; y++) {
       for (x = 0; x < pw; x++) {
       int ox = x / odk->hscale;
       int oy = y / odk->vscale;
       pbmp[x + (y * pw)] = bitmap[ox + (oy * w)];
       }
       }
     */

    xine_osd_draw_bitmap (odk->osd, bitmap, px, py, w, h, palette_map);
}


/* 
 * ***************************************************************************
 * Name:            odk_draw_line
 * Access:          public
 *
 * Description:     Draw a line.
 * ***************************************************************************
 */
void
odk_draw_line (odk_t * odk, int x1, int y1, int x2, int y2, int color)
{
    if (!(odk->hscale && odk->vscale))
        return;

    int px1 = (int) ((double) x1 * odk->hscale);
    int py1 = (int) ((double) y1 * odk->vscale);
    int px2 = (int) ((double) x2 * odk->hscale);
    int py2 = (int) ((double) y2 * odk->vscale);

    xine_osd_draw_line (odk->osd, px1, py1, px2, py2, XINE_OSD_TEXT1 + color);
}


/* 
 * ***************************************************************************
 * Name:            odk_draw_rect
 * Access:          public
 *
 * Description:     Draw a rectangle.
 * ***************************************************************************
 */
void
odk_draw_rect (odk_t * odk,
               int x1, int y1, int x2, int y2, int color, int filled)
{
    if (!(odk->hscale && odk->vscale))
        return;

    int px1 = (int) ((double) x1 * odk->hscale);
    int py1 = (int) ((double) y1 * odk->vscale);
    int px2 = (int) ((double) x2 * odk->hscale);
    int py2 = (int) ((double) y2 * odk->vscale);

    xine_osd_draw_rect (odk->osd,
                        px1, py1, px2, py2, XINE_OSD_TEXT1 + color, filled);
}


/* 
 * ***************************************************************************
 * Name:            odk_draw_text
 * Access:          public
 *
 * Description:     Draw a text.
 * ***************************************************************************
 */
void
odk_draw_text (odk_t * odk,
               int x, int y, const char *text, int alignment, int color)
{
    if (!(odk->hscale && odk->vscale))
        return;

    {
        int w;
        int h;
        // We always want the same baseline for text.
        // It should not matter if the text contains 
        // characters like g or f.
        odk_get_text_size (odk, "DVB", &w, &h);

        if (odk->is_current_font_freetype) {
            if (alignment & ODK_ALIGN_VCENTER)
                y += h / 2;
            if (alignment & ODK_ALIGN_TOP)
                y += h;
        } else {
            if (alignment & ODK_ALIGN_VCENTER)
                y -= h / 2;
            if (alignment & ODK_ALIGN_BOTTOM)
                y -= h;
        }
    }
    {
        int w;
        int h;
        // The width has to be calculated using 
        // the real text.
        odk_get_text_size (odk, text, &w, &h);

        if (alignment & ODK_ALIGN_CENTER)
            x -= w / 2;
        if (alignment & ODK_ALIGN_RIGHT)
            x -= w;
    }

    int px = (int) ((double) x * odk->hscale);
    int py = (int) ((double) y * odk->vscale);

    xine_osd_draw_text (odk->osd, px, py, text, color);
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_show
 * Access:          public
 *
 * Description:     Shows the OSD.
 * ***************************************************************************
 */
void
odk_osd_show (odk_t * odk)
{
    if (odk->use_unscaled_osd && odk->is_unscaled_osd_available)
        xine_osd_show_unscaled (odk->osd, 0);
    else
        xine_osd_show (odk->osd, 0);

    odk->is_osd_visible = 1;
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_hide
 * Access:          public
 *
 * Description:     Hides the OSD.
 * ***************************************************************************
 */
void
odk_osd_hide (odk_t * odk)
{
    xine_osd_hide (odk->osd, 0);

    odk->is_osd_visible = 0;
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_clear
 * Access:          public
 *
 * Description:     Clears the OSD.
 * ***************************************************************************
 */
void
odk_osd_clear (odk_t * odk)
{
    xine_osd_clear (odk->osd);

    odk->is_osd_visible = 0;
}


/* 
 * ***************************************************************************
 * Name:            odk_get_text_size
 * Access:          public
 *
 * Description:     Returns width and height of the text.
 * ***************************************************************************
 */
void
odk_get_text_size (odk_t * odk, const char *text, int *width, int *height)
{
    int w;
    int h;

    if (!(odk->hscale && odk->vscale)) {
        *width = 0;
        *height = 0;
        return;
    }

    xine_osd_get_text_size (odk->osd, text, &w, &h);
    *width = (int) ((double) w / odk->hscale);
    *height = (int) ((double) h / odk->vscale);
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_set_font
 * Access:          public
 *
 * Description:     Sets the font and font-size xine-lib uses to paint text.
 * ***************************************************************************
 */
void
odk_osd_set_font (odk_t * odk, const char *font, int font_size)
{
    int psize;

    psize = (int) ((double) font_size * (odk->hscale + odk->vscale) / 2);

    /* smallest text size possible */
    if (psize < 16)
        psize = 16;

    if (strchr (font, '.') || strchr (font, '/'))
        odk->is_current_font_freetype = 1;
    else
        odk->is_current_font_freetype = 0;

    xine_osd_set_font (odk->osd, font, psize);
}


/* 
 * ***************************************************************************
 * Name:            odk_get_frame_size
 * Access:          private
 *
 * Description:     Returns width and height of the current frame.
 * ***************************************************************************
 */
void
odk_get_frame_size (odk_t * odk, int *w, int *h)
{
    int r;
    int f;

    if (!xine_get_current_frame (odk->win->stream, w, h, &r, &f, NULL)) {
        *w = xine_get_stream_info (odk->win->stream,
                                   XINE_STREAM_INFO_VIDEO_WIDTH);
        *h = xine_get_stream_info (odk->win->stream,
                                   XINE_STREAM_INFO_VIDEO_HEIGHT);
    }
}


/* 
 * ***************************************************************************
 * Name:            odk_get_output_size
 * Access:          private
 *
 * Description:     Returns width and height of the output window.
 * ***************************************************************************
 */
void
odk_get_output_size (odk_t * odk, int *width, int *height)
{
    *width = odk->win->output_width;
    *height = odk->win->output_height;
}


/* 
 * ***************************************************************************
 * Name:            adapt_osd_size
 * Access:          private
 *
 * Description:     Adapt the OSD to the current output size.
 * ***************************************************************************
 */
void
odk_osd_adapt_size (odk_t * odk)
{
    if (odk->osd)
        xine_osd_free (odk->osd);

    int width;
    int height;

    if (odk->use_unscaled_osd && odk->is_unscaled_osd_available)
        odk_get_output_size (odk, &width, &height);
    else
        odk_get_frame_size (odk, &width, &height);

    odk->vscale = (double) height / (double) V_HEIGHT;
    odk->hscale = (double) width / (double) V_WIDTH;

    odk->osd = xine_osd_new (odk->win->stream, 0, 0, width, height);
    xine_osd_set_palette (odk->osd, odk->palette->colors,
                          odk->palette->transparency);

    if (odk->image_cache) {
        odk_osd_free_image (odk->image_cache);
        odk->image_cache = NULL;
    }
}


/* 
 * ***************************************************************************
 * Name:            odk_is_osd_visible
 * Access:          public
 *
 * Description:     Is the OSD visible?
 * ***************************************************************************
 */
int
odk_is_osd_visible (odk_t * odk)
{
    return odk->is_osd_visible;
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_image_t
 * Access:          public
 *
 * Description:     Loads an OSD image.
 * ***************************************************************************
 */
#ifdef HAVE_IMAGEMAGICK
static odk_osd_image_t *
odk_osd_load_image (odk_t * odk, const char *mrl,
                    int x, int y, int width, int height)
{
    assert (odk->hscale && odk->vscale);
    /* They must be > 0 or -1 */
    assert (width > 0 || width == -1);
    assert (height > 0 || height == -1);
    /* Only one of them may be -1 */
    assert (!(width == -1 && height == -1));

    InitializeMagick (NULL);

    ExceptionInfo exception;
    GetExceptionInfo (&exception);

    ImageInfo *image_info = CloneImageInfo ((ImageInfo *) NULL);
    strcpy (image_info->filename, mrl);

    Image *image = ReadImage (image_info, &exception);
    CatchException (&exception);
    if (!image) {
        error (_("Could not read image '%s'!"), mrl);
        DestroyImageInfo (image_info);
        DestroyExceptionInfo (&exception);
        DestroyMagick ();

        return NULL;
    }

    /* Scale image */
    if ((width == -1) || (height == -1)) {
        if (width == -1)
            width =
                ((double) image->columns / (double) image->rows) *
                (double) height;
        if (height == -1)
            height =
                ((double) image->rows / (double) image->columns) *
                (double) width;
    }

    else {
        if (image->rows > image->columns)
            width =
                ((double) image->columns / (double) image->rows) *
                (double) height;
        if (image->rows < image->columns)
            height =
                ((double) image->rows / (double) image->columns) *
                (double) width;
    }

    width *= odk->vscale;
    height *= odk->hscale;

    Image *simage = ScaleImage (image, width, height, &exception);
    CatchException (&exception);
    if (!simage) {
        error (_("Could not scale image '%s'!"), mrl);
        DestroyImage (image);
        DestroyImageInfo (image_info);
        DestroyExceptionInfo (&exception);
        DestroyMagick ();

        return NULL;
    }

    /* Create OSD image data structure. */
    odk_osd_image_t *osd_image = ho_new (odk_osd_image_t);

    osd_image->osd = xine_osd_new (odk->win->stream, 0, 0,
                                   V_WIDTH * odk->hscale,
                                   V_HEIGHT * odk->vscale);
    osd_image->mrl = ho_strdup (mrl);

    /* Quantize the image (we only have 255 colors as the first color is
     * always transparent). */
    QuantizeInfo quantize_info;
    GetQuantizeInfo (&quantize_info);
    quantize_info.number_colors = NUM_COLORS - 1;
#ifdef USE_YUV
    quantize_info.colorspace = YUVColorspace;
#endif
    QuantizeImage (&quantize_info, simage);
    CatchException (&exception);

    /* Get image pixels. */
    const PixelPacket *pp = AcquireImagePixels (simage, 0, 0, width, height,
                                                &exception);
    CatchException (&exception);
    if (!pp) {
        error (_("Could not get pixel data!"));
        xine_osd_free (osd_image->osd);
        ho_free (osd_image->mrl);
        ho_free (osd_image);

        DestroyImage (image);
        DestroyImage (simage);
        DestroyImageInfo (image_info);
        DestroyExceptionInfo (&exception);
        DestroyMagick ();

        return NULL;
    }

    uint8_t image_data[width * height];

    odk_palette_t palette;
    palette.num_colors = 0;

    int p = 0;
    uint8_t palette_map[NUM_COLORS];
    for (; p < NUM_COLORS; p++)
        palette_map[p] = p;

    int col;
    int row;
    for (row = 0; row < height; row++) {
        for (col = 0; col < width; col++) {
#ifdef USE_YUV
            uint8_t y = ScaleQuantumToChar (pp->red);
            uint8_t u = ScaleQuantumToChar (pp->green);
            uint8_t v = ScaleQuantumToChar (pp->blue);
            uint32_t yuv = ((y << 16) | (v << 8) | u);
#else
            uint8_t r = ScaleQuantumToChar (pp->red);
            uint8_t g = ScaleQuantumToChar (pp->green);
            uint8_t b = ScaleQuantumToChar (pp->blue);
            uint32_t yuv = rgb2yuv ((r << 16) | (g << 8) | b);
#endif
            uint8_t t = 0xFF - ScaleQuantumToChar (pp->opacity);
            image_data[row * width + col] = get_color (&palette, yuv, t);
            pp++;
        }
    }

    /* Scale position */
    x *= odk->vscale;
    y *= odk->hscale;

    if (x < 0)
        x += V_WIDTH * odk->hscale - width;
    if (y < 0)
        y += V_HEIGHT * odk->vscale - height;

    xine_osd_set_palette (osd_image->osd, palette.colors,
                          palette.transparency);
    xine_osd_draw_bitmap (osd_image->osd, image_data,
                          x, y, width, height, palette_map);

    DestroyImage (simage);
    DestroyImage (image);

    DestroyImageInfo (image_info);
    DestroyExceptionInfo (&exception);
    DestroyMagick ();

    return osd_image;
}
#endif


/* 
 * ***************************************************************************
 * Name:            odk_osd_free_image
 * Access:          public
 *
 * Description:     Frees an OSD image.
 * ***************************************************************************
 */
void
odk_osd_free_image (odk_osd_image_t * osd_image)
{
#ifdef HAVE_IMAGEMAGICK
    xine_osd_free (osd_image->osd);
    ho_free (osd_image->mrl);
    ho_free (osd_image);
#endif
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_draw_image
 * Access:          public
 *
 * Description:     Loads and draws an image to OSD.
 * ***************************************************************************
 */
void
odk_osd_draw_image (odk_t * odk, const char *mrl,
                    int x, int y, int width, int height)
{
#ifdef HAVE_IMAGEMAGICK
    if (odk->image_cache) {
        if (strcmp (mrl, odk->image_cache->mrl) != 0) {
            odk_osd_free_image (odk->image_cache);
            odk->image_cache = NULL;
        }
    }

    if (!odk->image_cache)
        odk->image_cache = odk_osd_load_image (odk, mrl, x, y, width, height);
    if (!odk->image_cache)
        return;

    if (odk->use_unscaled_osd && odk->is_unscaled_osd_available)
        xine_osd_show_unscaled (odk->image_cache->osd, 0);
    else
        xine_osd_show (odk->image_cache->osd, 0);
#endif
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_hide_image
 * Access:          public
 *
 * Description:     Hides the image OSD.
 * ***************************************************************************
 */
void
odk_osd_hide_image (odk_t * odk)
{
#ifdef HAVE_IMAGEMAGICK
    if (odk->image_cache)
        xine_osd_hide (odk->image_cache->osd, 0);
#endif
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_init
 * Access:          private
 *
 * Description:     Init OSD stuff.
 * ***************************************************************************
 */
void
odk_osd_init (odk_t * odk)
{
    odk->is_osd_visible = 0;
    odk->is_current_font_freetype = 0;
    odk->is_unscaled_osd_available = 0;

    odk->hscale = 0;
    odk->vscale = 0;

    odk->palette = ho_new (odk_palette_t);
    odk->palette->num_colors = 0;

    /* Test support of unscaled OSD. */
    odk->osd = xine_osd_new (odk->win->stream, 0, 0, 10, 10);
    odk->is_unscaled_osd_available =
        (xine_osd_get_capabilities (odk->osd) & XINE_OSD_CAP_UNSCALED);
    xine_osd_free (odk->osd);
    odk->osd = NULL;
    if (odk->is_unscaled_osd_available) {
        info (_("The current video driver supports unscaled OSD!"));
    } else {
        warn (_("The current video driver does not support unscaled OSD!"));
    }

    /* Adapt size of OSD to current window/ output size. */
    odk_osd_adapt_size (odk);
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_free
 * Access:          private
 *
 * Description:     Free OSD stuff.
 * ***************************************************************************
 */
void
odk_osd_free (odk_t * odk)
{
    xine_osd_free (odk->osd);
    odk->osd = NULL;
    if (odk->image_cache) {
        odk_osd_free_image (odk->image_cache);
        odk->image_cache = NULL;
    }
    if (odk->palette)
        ho_free (odk->palette);
}
