/*
 *
 * Copyright (C) 2005, 2006 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * Authors:
 *  Cedric Pinson <cpinson@freesheep.org>
 *
 */

#include "osgcal.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif // HAVE_CONFIG_H

#ifdef USE_NPROFILE
#include <nprofile/profile.h>
#else  // USE_NPROFILE
#define NPROFILE_SAMPLE(a)
#endif

#include "osgcal.h"
#include <assert.h>
#include "osgviewer.h"
//#include <unistd.h>
#include <cstdio>

#include <string>
#include <vector>
#include <iostream>

#include <fcntl.h>

#include <glib.h>

#include <osgUtil/SceneView>
#include <osg/Timer>
#include <osg/Version>
#include <osgDB/ReadFile>
#include <osgDB/FileUtils>
#include <osgText/Text>
#include <osg/CoordinateSystemNode>
#include <osg/MatrixTransform>
#include <osg/Projection>

#include <osgCal/Model>

#ifndef DEFAULT_FILES
#define DEFAULT_FILES "franck/cal3d.xfg", "paladin/cal3d.cfg"
#endif // DEFAULT_FILES

#define DEFAULT_DATA_DIR "data"
#ifndef DATA_DIR
#define DATA_DIR "."
#endif // DATA_DIR

#ifdef WIN32
#undef main
#endif

int main(int argc, char** argv)
{
  {
    osgDB::FilePathList& path = osgDB::getDataFilePathList();
    path.push_front(DATA_DIR);
    path.push_front(DEFAULT_DATA_DIR);
  }

  ParseArguments arguments(argc,argv);
  if (!arguments.Parse())
    return 1;

  osg::Timer_t start_tick = osg::Timer::instance()->tick();

  osgViewer viewer;
  viewer.Create();

  osg::Group* root  = new osg::Group();
  osg::ref_ptr<osgCal::Model> model = new osgCal::Model();

  root->addChild(model.get());
  viewer.GetRoot()->addChild(root);

  osg::Object* object  = osgDB::readObjectFile(arguments.mPath);
  if (!object) {
    std::cerr << "osgDB::readObjectFile " << arguments.mPath << " failed, file not found or not a xfg,cfg file" << std::endl;
    return 1;
  }
  osg::ref_ptr<osgCal::CoreModel> coreModelRef = dynamic_cast<osgCal::CoreModel*>( object );
  osgCal::CoreModel* coreModel = coreModelRef.get();
  if(!model->setCoreModel(coreModel)) {
    std::cerr << "setCoreModel: " << CalError::getLastErrorDescription().c_str() << std::endl;
    return -1;
  }

  if(arguments.mSuffix == "cfg") {
    for(int coreMeshId = 0; coreMeshId < model->getCalCoreModel()->getCoreMeshCount(); coreMeshId++) {
      model->getCalModel()->attachMesh(coreMeshId);
    }

    // set the material set of the whole model
    model->getCalModel()->setMaterialSet(0);

    if (arguments.mVertexprogram_flag)
      model->setUseVertexProgram(true);
    // Creating a concrete model using the core template
    if(!model->create()) {
      std::cerr << "create: " << CalError::getLastErrorDescription().c_str() << std::endl;
      return -1;
    }

  } else if(arguments.mSuffix == "xfg") {
    osg::Group* fxgroup = new osg::Group();
    fxgroup->setName("fxGroup");
    fxgroup->getOrCreateStateSet()->setMode(GL_ALPHA_TEST, osg::StateAttribute::OFF);
    fxgroup->getOrCreateStateSet()->setMode(GL_DITHER, osg::StateAttribute::ON);
    fxgroup->getOrCreateStateSet()->setRenderBinDetails(100, "RenderBin");
    fxgroup->setCullingActive(false);

    root->addChild(fxgroup);

    model->setFXGroup(fxgroup);
    model->setFXState(viewer.GetSceneView()->getState());

    if(arguments.mPositional_arguments.size() <= 0) {
      std::string directory = g_dirname(arguments.mPath.c_str());
      std::string outfit = directory + "/outfit.xfg";

      std::string outfit_path = osgDB::findDataFile(outfit);
      if(outfit_path == "") {
        std::cerr << "unable to find file " << outfit << " in OSG path" << std::endl;
        return 1;
      }
      arguments.mPositional_arguments.push_back(outfit_path);
    }

    std::string outfit = read_file(arguments.mPositional_arguments.front());
    if(outfit == "") return 1;
      
    //
    // Apply outfit parameters
    //
    if(!model->initOutfitFromXMLString(outfit))
      return 1;

    if (arguments.mVertexprogram_flag)
      model->setUseVertexProgram(true);

    if(!model->create()) {
      std::cerr << "create failed " << CalError::getLastErrorDescription() << std::endl;
      return 1;
    }
  }

  // if no model has been successfully loaded report failure.
  if (!model) {
    std::cerr << argv[0] <<": No data loaded" << std::endl;
    return 1;
  }

  osg::Timer_t end_tick = osg::Timer::instance()->tick();
  std::cout << "Time to load = "<<osg::Timer::instance()->delta_s(start_tick,end_tick)<<std::endl;

  Manipulator* tm = new Manipulator(viewer, model.get(), arguments.mFixcam_flag, arguments.mVertexprogram_flag);

  if(arguments.mSuffix == "xfg")
    tm->setOutfits(arguments.mPositional_arguments);

  osg::StateSet* globalStateSet = viewer.GetSceneView()->getGlobalStateSet();
  if (!globalStateSet) {
    globalStateSet = new osg::StateSet;
    viewer.GetSceneView()->setGlobalStateSet(globalStateSet);
  }
  // disable back face culling else the both cape sides are not rendered 
  globalStateSet->setMode(GL_CULL_FACE, osg::StateAttribute::OFF  | osg::StateAttribute::OVERRIDE );

  if(arguments.mSuffix == "xfg") {
    //
    // Black magic that *must* disapear urgently (09/2005)
    //
		if(!model->applyParameterFromOutfitDescription()) {
			printf("applyParameterFromOutfitDescription failed, continuing...\n");
		}
  }

  osgText::Text* text = new osgText::Text;
  text->setPosition(osg::Vec3(5.f, 5.f, 0.f));
  text->setAlignment(osgText::Text::LEFT_BOTTOM);
  text->setCharacterSize(12.f);
  {
    osg::Projection* projection = new osg::Projection;
    projection->setMatrix(osg::Matrix::ortho2D(0,800,0,600));

    osg::MatrixTransform* modelview_abs = new osg::MatrixTransform;
    modelview_abs->setReferenceFrame(osg::Transform::ABSOLUTE_RF);
    modelview_abs->setMatrix(osg::Matrix::identity());

    osg::Geode* text_geode = new osg::Geode;
    text_geode->addDrawable(text);

    modelview_abs->addChild(text_geode);

    projection->addChild(modelview_abs);

    root->addChild(projection);
  }
  tm->SetComment(text);

  {
#ifdef USE_NPROFILE
    nprf::GetProfiler()->Reset();
#endif
    NPROFILE_SAMPLE("osgcal");

    tm->InitCamera();

    if(arguments.mBenchmark_flag) {

#ifndef USE_NPROFILE
      osg::notify(osg::FATAL) << "recompile with --enable-nprofile to enable --benchmark" << std::endl;
      exit(1);
#endif // USE_NPROFILE

      tm->launchAnim();

      for(int l = 0; l < arguments.mNumloops; l++) {

        int i=0;
        do {

          for(int f = 0; f < arguments.mNumframes; f++) {
            viewer.Update();
            viewer.Render();
            tm->RecomputeCamera();
          }

          tm->applyNextOutfit();
          i++;

        } while( i < tm->getNbOutfit() );
      }
    } else {
      if (arguments.mNumframes_flag) {
        for(int i = 0; i < arguments.mNumframes; i++) {
          viewer.Render();
          tm->RecomputeCamera();
        }
      } else {
        while( viewer.GetRunning() ) {
          // wait for all cull and draw threads to complete.
          viewer.Update();
          viewer.Render();
          tm->Handle(viewer.GetEventList());
          tm->Update();
          tm->RecomputeCamera();
        }
      }
    }
  }

#ifdef USE_NPROFILE
  nprf::GetProfiler()->EndProfile();
  nprf::GetProfiler()->GetRootNode()->DisplayFlatStats(std::cout);
  nprf::GetProfiler()->GetRootNode()->DisplayFlatStats(std::cout, nprf::MACHINE_READABLE);
  std::cout << std::endl;
#endif // USE_NPROFILE

  delete tm;
  viewer.Destroy();
  return 0;
}
