/*
 * Copyright (C) 1999-2001  Richard Hult 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <math.h>
#include <gnome.h>
#include <glade/glade.h>
#include "main.h"
#include "schematic.h"
#include "sim-engine.h"
#include "dialogs.h"
#include "plot.h"
#include "smallicon.h"
#include "pixmaps/mini_icon_plot.xpm"

#define PLOT_PADDING_X 50
#define PLOT_PADDING_Y 40
#define PLOT_WIDTH 400
#define PLOT_HEIGHT 300
#define DEBUG_PLOT 0

#define GET_Y_POINT(y_val,y_min,factor) (PLOT_HEIGHT-((y_val)-(y_min))*(factor))
#define GET_Y_VALUE(y_point,y_min,factor) ((y_min)+(PLOT_HEIGHT-(y_point))/(factor))
#define GET_X_POINT(x_val,x_min,factor) (((x_val)-(x_min))*(factor))
#define GET_X_VALUE(x_point,x_min,factor) ((x_min)+(x_point)/(factor))

const gchar *TITLE_FONT    = N_("-adobe-helvetica-bold-r-normal-*-12-*-*-*-*-*-*-*");
const gchar *AXIS_FONT     = N_("-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-*-*");
const gchar *UPSCRIPT_FONT = N_("-adobe-helvetica-medium-r-normal-*-8-*-*-*-*-*-*-*");
#define PLOT_AXIS_COLOR "black"
#define PLOT_CURVE_COLOR "medium sea green"
static int n_curve_colors = 5;
static char *plot_curve_colors[] = {
        "medium sea green",
        "royal blue",
        "firebrick",
        "gold",
        "turquoise",
        0
};

static gdouble x_min, x_max;

typedef struct {
	GtkWidget *window;
	GtkWidget *canvas;
        GtkWidget *coord;  /* shows the coordinates of the mose */
        GtkWidget *combo;

	GnomeCanvasGroup *plot_group;   /* holds the actual curve */
	GnomeCanvasGroup *pad_group;    /* contains axes, plot title etc */ 
	GnomeCanvasGroup *axis_group;   /* the x/y-axes */

	SimEngine *sim;
        SimulationData *current;

        gboolean logx;
        gboolean logy;

	gchar *title;
	gchar *xtitle;
	gchar *ytitle;

	gint width;           /* width of the plot, excluding */
	gint height;          /* axes, titles and padding etc */

        gdouble x_factor,x_scale;
        gdouble y_factor,y_scale;

	gdouble plot_min, plot_max;
	gdouble x_min,x_max;

	gdouble padding_x;     /* padding around the plot. Note that */
	gdouble padding_y;     /* titles, axes etc live here */

	gint selected;         /* the currently selected plot in the clist */
	gint prev_selected;
} Plot;

typedef enum {
        X_AXIS,
        Y_AXIS
} AxisType;

static GtkWidget *plot_window_create (Plot *plot);
static void make_plot (Plot *plot, gint plot_number);
static void destroy_window (GtkWidget *widget, Plot *plot);
static void destroy_plot (Plot *plot);
static gint delete_event_cb (GtkWidget *widget, GdkEvent *event, Plot *plot);
static void getbins (gdouble, gdouble, gint, gdouble *, gdouble *, gint *, gdouble *);
static void get_order_of_magnitude (double, double *,double *);
static void plot_canvas_movement(GtkWidget *, GdkEventMotion *, Plot *);
static void plot_draw_axis (Plot *, AxisType, double, double);

/* Define menus and toolbar */
static GnomeUIInfo plot_file_menu [] = {
	
	{ GNOME_APP_UI_ITEM, N_("_Close"), NULL, destroy_window, NULL, NULL,
	  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE, 'w', GDK_CONTROL_MASK 
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo plot_plot_menu [] = {
	{ GNOME_APP_UI_ITEM, N_("_Preferences..."), NULL, NULL, NULL, NULL,
	  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF, 0, 0 },
	GNOMEUIINFO_END
};

static GnomeUIInfo plot_help_menu[] =
{
/*	GNOMEUIINFO_HELP ("Schematic Plot"),*/
	GNOMEUIINFO_MENU_ABOUT_ITEM(NULL, NULL),
	GNOMEUIINFO_END
};

GnomeUIInfo plot_main_menu[] = {
	GNOMEUIINFO_SUBTREE(N_("_File"), plot_file_menu),
	GNOMEUIINFO_SUBTREE(N_("_Plot"), plot_plot_menu),
	GNOMEUIINFO_MENU_HELP_TREE (plot_help_menu),
	GNOMEUIINFO_END
};

static gchar *
get_variable_units (gchar *str) 
{
        gchar *tmp;
	
	if (str == NULL)
		return g_strdup ("####");

        if (!strcmp (str, "voltage")) {
                tmp = g_strdup ("Volt");

        } else if (!strcmp (str, "time") ) {
                tmp = g_strdup (_("s"));

        } else if (!strcmp (str, "frequency") ) {
                tmp = g_strdup (_("Hz"));

        } else if (!strcmp (str, "current") ) {
                tmp = g_strdup (_("A"));

        } else {
                tmp = g_strdup ("####");
        }

        return tmp;
}

static gint
delete_event_cb (GtkWidget *widget, GdkEvent *event, Plot *plot)
{
	plot->window = NULL;
	return FALSE; /* yes, please destroy me */
}

/* Call this to close the plot window */
static void
destroy_window (GtkWidget *widget, Plot *plot)
{
	gtk_widget_destroy (plot->window);
	plot->window = NULL;
}

static void 
show_plot (Plot *plot)
{
	if (plot) {
		destroy_plot (plot);
	}
	
	make_plot (plot, plot->selected);
}

static void
plot_selected (GtkCList *clist, gint row, gint col, GdkEvent *event, Plot *plot)
{
        
        plot->prev_selected = plot->selected;
        plot->selected = row + 1;
        if (plot->ytitle) {
		g_free (plot->ytitle);
	}
        plot->ytitle = get_variable_units (plot->current->var_units[row + 1]);
	show_plot (plot);
}

static void
analysis_selected (GtkEditable *editable, Plot *plot) 
{
        int        i;
        gchar     *ca;
        GtkWidget *clist, *entry;
        GList     *analysis;
        SimulationData *sdat;
	
        clist = gtk_object_get_data (GTK_OBJECT (plot->window), "clist");
        entry = GTK_COMBO (plot->combo)->entry;
        ca = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (plot->combo)->entry));
        plot->current = NULL;
        for (analysis = plot->sim->analysis; analysis; analysis = analysis->next) {
                sdat = SIM_DATA (analysis->data);
                if (!strcmp (ca, sim_engine_analysis_name (sdat))) {
                        plot->current = sdat;
                        break;
                }
        }

	if (plot->current == NULL) {
		/* Simulation failed? */
		return;
	}
	
        g_free (plot->title);
        plot->title = g_strdup_printf (
		_("Plot - %s"),
		sim_engine_analysis_name (plot->current));
	
        g_free (plot->xtitle);
        plot->xtitle = get_variable_units (plot->current->var_units[0]);

         /*  Set the variable names in the list  */
        gtk_clist_freeze (GTK_CLIST (clist));
        gtk_clist_clear(GTK_CLIST (clist));
        for (i = 1; i < plot->current->n_variables; i++) {
                if ( plot->current->type != DC_TRANSFER ) {
                        if (strchr (plot->current->var_names[i], '#') == NULL)
                                gtk_clist_append (GTK_CLIST (clist),
                                                  &plot->current->var_names[i]);
                }
                else {
                        gtk_clist_append (GTK_CLIST (clist),
                                          &plot->current->var_names[i]);
                }
        }
        gtk_clist_thaw (GTK_CLIST (clist));
        
        gtk_clist_select_row (GTK_CLIST (clist), 0, 0);
}

static void
destroy_plot (Plot *plot)
{
	if (plot->axis_group) gtk_object_destroy (GTK_OBJECT (plot->axis_group));
	if (plot->plot_group) gtk_object_destroy (GTK_OBJECT (plot->plot_group));

	plot->axis_group = NULL;
	plot->plot_group = NULL;
}

static void
plot_canvas_movement (GtkWidget *w, GdkEventMotion *event, Plot *plot)
{
        gchar *coord;
        gdouble x,y;

	x = event->x - plot->padding_x;
	y = event->y - plot->padding_y - 13; /* FIXME: why should I add this offset ? */
        x = GET_X_VALUE (x, plot->x_min, plot->x_factor);
        y = GET_Y_VALUE (y, plot->plot_min, plot->y_factor);

        if (plot->logx) {
		x = pow (10.0, x);
	}
        if (plot->logy) {
		y = pow (10.0, y);
	}

        coord = g_strdup_printf ("(%g, %g)", x / plot->x_scale, y / plot->y_scale);
	gtk_entry_set_text (GTK_ENTRY (plot->coord), coord);
	g_free (coord);
}

static void
make_plot (Plot *plot, gint plot_number)
{
	GnomeCanvas *canvas;
	GnomeCanvasGroup *plot_group, *axis_group;
        GnomeCanvasItem *plot_title, *name_curve;
	GnomeCanvasPoints *points;
        GtkWidget *clist;
        GList *list;

        gchar *xtitle, *ytitle, *title;
        double plot_title_width, plot_title_height;
        double y_factor, y;
        double x_factor;
	double y_min, y_max;
        gint   i, j, k, npts, n_curve;
        double data, data1;
        char     *ysc=0,*xsc=0;
        gdouble  sx,sy,ex,ey;
	
        clist = gtk_object_get_data (GTK_OBJECT (plot->window), "clist");
        y_min = 1.0e20;
        y_max =-1.0e20;
        y = 0.0;
        for (list = GTK_CLIST (clist)->selection; list; list = list->next) {
                i = (int) (list->data) + 1;
                y = plot->current->min_data[i];
                if (y_min>y)
			y_min=y;
                y = plot->current->max_data[i];
                if (y_max < y)
			y_max = y;
        }

        plot->logx = FALSE;
        if (plot->current->type == AC) {
                if (!strcmp (sim_settings_get_ac_type (plot->sim->sim_settings), "DEC"))
			plot->logx = TRUE;
	}
	
        if (y_max == y_min) {         
                plot->plot_max = (y_max==0.0 ? 1.0 : 1.1 * y_max);
                plot->plot_min = (y_max==0.0 ? -1.0 : 0.9 * y_max);
        } else {
                gdouble interval = (y_max - y_min);
                plot->plot_max = y_max + 0.1 * interval;
                plot->plot_min = y_min - 0.1 * interval;
	}

        if (fabs (plot->plot_max) > fabs (plot->plot_min)) {
                sx = plot->plot_max;
        } else {
                sx = plot->plot_min;
	}
	
        get_order_of_magnitude (sx, &sy, &ey);
        if (ey != 0.0) {
                sy = pow (10.0, ey);
                ysc = g_strdup_printf ("%.1e", sy);
	}
        else {
		sy = 1.0;
	}
	
        plot->y_scale = sy;

        x_min = plot->current->min_data[0];
        x_max = plot->current->max_data[0];

        get_order_of_magnitude (x_max, &sx, &ex);
        if (ex != 0.0) {
                sx = pow (10.0 ,ex);
                xsc = g_strdup_printf ("%.1e", sx);
        } else
		sx = 1.0;
        plot->x_scale = sx;
        if (plot->logx) {
                x_min = log10 (x_min);
                x_max = log10 (x_max);
                plot->x_scale = 1.0;
	}

	/* Used to go from actual data values to canvas coordinates. */
	y_factor = (PLOT_HEIGHT) / (plot->plot_max - plot->plot_min);
	x_factor = (PLOT_WIDTH) / (x_max - x_min);

	plot->x_factor = x_factor;
	plot->y_factor = y_factor;
	plot->x_min = x_min;
	plot->x_max = x_max;
		
	canvas = GNOME_CANVAS (plot->canvas);
	canvas->close_enough = 5;

	xtitle = plot->xtitle;
	ytitle = plot->ytitle;
	title = plot->title;

	/*
	 * 1. Make a group to hold the actual plot. Lets us keep track of the plot to
	 *    destroy it later. This lets us use relative coordinates too, which makes
	 *    the padding code transparent and simple.
	 */
        plot_group = GNOME_CANVAS_GROUP (
                gnome_canvas_item_new (gnome_canvas_root (canvas),
				       gnome_canvas_group_get_type (),
				       "x", plot->padding_x,
				       "y", plot->padding_y,
                                       NULL));
	plot->plot_group = plot_group;

	/*
	 * 2. Plot the curve, using the calculated scale factors. 
	 */
        n_curve = 0;
        for (list = GTK_CLIST (clist)->selection; list; list = list->next) {
                plot_number = (int) list->data + 1;
                
                for (i = 0; i < plot->current->got_points - 1;) {
                        /* Loop while the step is positive in the X axis. */
			data  = g_array_index (plot->current->data[0], double, i);
                        for (j = i + 1; j < plot->current->got_points - 1; j++) {
				data1 = g_array_index (plot->current->data[0], double, j);
				if ( data1 == data )
					break;                    
                        }
                        npts = j;

                        /*  Get this chunck of data. */                 
                        points = gnome_canvas_points_new (j - i);
                        for ( k = 0, j = i; j < npts; j++, k++) {
                                data = g_array_index (plot->current->data[plot_number], double, j);
                                points->coords[2 * k + 1] = GET_Y_POINT (data, plot->plot_min, y_factor);
              
                                data = g_array_index (plot->current->data[0], double, j);
				if (plot->logx) 
					data = log10 (data);
                                points->coords[2 * k] = GET_X_POINT (data, x_min, x_factor);
			}

                        if (k > 1)
				gnome_canvas_item_new (plot_group,
						       gnome_canvas_line_get_type (),
						       "points", points,
                                                       "fill_color", plot_curve_colors [n_curve % n_curve_colors],
						       NULL);
			gnome_canvas_points_free (points);
			i = npts;
                }
           
                /* Plot the name and color of the curve. */
                name_curve = 
                        gnome_canvas_item_new (
                                plot_group,
                                gnome_canvas_text_get_type (),
                                "text",plot->current->var_names[plot_number],
                                "x", plot->padding_x+50*n_curve,
                                "y", plot->padding_y,
                                "fill_color", plot_curve_colors[ n_curve % n_curve_colors],
                                "anchor", GTK_ANCHOR_NE,
                                "font", _(AXIS_FONT),
                                NULL);     

		{ 
			gdouble text_height;
			gtk_object_get (GTK_OBJECT(name_curve), "text_height", &text_height, NULL);
			gtk_object_set (GTK_OBJECT(name_curve), "y", -2.0 * text_height, NULL);  
		}

                n_curve++;
        }

	/*
	 * 3. Make a group to hold the the x- and y-axis. 
	 */
        axis_group = 
                GNOME_CANVAS_GROUP (
                        gnome_canvas_item_new (gnome_canvas_root (canvas),
					       gnome_canvas_group_get_type (),
					       "x", plot->padding_x,
					       "y", plot->padding_y,
                                               NULL));
	plot->axis_group = axis_group;
	
	/*
         * 4.1 Start with the the x-axis.
	 */
        plot_draw_axis (plot, X_AXIS, x_min, x_max);

	/*
	 * 4.2 y-axis, at x = 0.0.
	 */
        plot_draw_axis (plot,Y_AXIS,plot->plot_min,plot->plot_max);

        /* Main title    */
	plot_title = gnome_canvas_item_new (
		axis_group,
		gnome_canvas_text_get_type (),
		"text", title,
		"x", 0.0,
		"y", 0.0 - plot->padding_y,
		"fill_color", PLOT_AXIS_COLOR,
		"anchor", GTK_ANCHOR_W,
		"font", _(TITLE_FONT),
		NULL);
	
	/* Now find out the width of the plot title and make it centered. */
        gtk_object_get (GTK_OBJECT (plot_title), 
			"text_width", &plot_title_width,
			"text_height", &plot_title_height, NULL);

        gtk_object_set (GTK_OBJECT(plot_title), 
                        "x", (double) PLOT_WIDTH / 2 - plot_title_width / 2,
                        "y", (double) 0.0- plot->padding_y+plot_title_height / 2.0,
                        NULL);  

	g_free(ysc);
	g_free(xsc);
}

static GtkWidget *
plot_window_create (Plot *plot)
{  
	GtkWidget *clist;
	GtkWidget *canvas, *outer_table, *window, *button;
	GtkStyle *style;
	GdkColormap *colormap;
	GdkPixmap *icon;
	GdkBitmap *icon_mask;
	GladeXML *gui;
	gchar *msg;
	
	window = gnome_app_new ("plot", _("Oregano - Plot"));

	gtk_signal_connect (GTK_OBJECT (window), "delete_event",
			    GTK_SIGNAL_FUNC (delete_event_cb), plot);
	
	if (!g_file_exists (OREGANO_GLADEDIR "/plot-window.glade")) {
		msg = g_strdup_printf (_("Could not find the required file:\n%s"), 
				       OREGANO_GLADEDIR "/plot-window.glade");
		oregano_error (msg);
		g_free (msg);
		return NULL;
	}

	gui = glade_xml_new (OREGANO_GLADEDIR "/plot-window.glade", "plot-table");
	if (!gui) {
		oregano_error (_("Could not create plot window."));
		return NULL;
	}

	outer_table = glade_xml_get_widget (gui, "plot-table");
	canvas = plot->canvas = glade_xml_get_widget (gui, "plot-canvas");
        plot->coord  = glade_xml_get_widget(gui, "pos-label");
	
        gtk_signal_connect (GTK_OBJECT (canvas), "motion_notify_event",
                            GTK_SIGNAL_FUNC (plot_canvas_movement),
                            plot);

	gnome_app_set_contents (GNOME_APP (window), outer_table);

	button = glade_xml_get_widget (gui, "close-button");
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			    GTK_SIGNAL_FUNC (destroy_window), plot);

	gtk_widget_set_usize (canvas, PLOT_WIDTH + PLOT_PADDING_X*2, PLOT_HEIGHT + PLOT_PADDING_Y*2);
	gnome_canvas_set_scroll_region (GNOME_CANVAS (canvas), 0, 0, 
					PLOT_WIDTH + PLOT_PADDING_X*2, 
					PLOT_HEIGHT + PLOT_PADDING_Y*2);


	clist = glade_xml_get_widget (gui, "variable-list");
	gtk_clist_set_column_min_width (GTK_CLIST (clist), 0, 90);
	gtk_object_set_data (GTK_OBJECT (window), "clist", clist);

   	style =  gtk_style_new ();
	colormap = gtk_widget_get_colormap (GTK_WIDGET (canvas));
	gdk_color_white (colormap, &style->bg[GTK_STATE_NORMAL]);
	gtk_widget_set_style (GTK_WIDGET (canvas), style);

	gtk_widget_realize (GTK_WIDGET (window));
	icon = gdk_pixmap_create_from_xpm_d (window->window, &icon_mask, NULL, mini_icon_plot_xpm);
	set_small_icon (window->window, icon, icon_mask);
	

        plot->combo =  glade_xml_get_widget (gui, "cmbAnalysis");
        
	gtk_widget_show_all (window);  
	return window;
}

int
plot_show (SimEngine *engine)
{
        GtkWidget *clist, *entry;
        GList *analysis = NULL;
        GList *combo_items = NULL;
	Plot *plot;
        gchar *s_current = NULL;
        SimulationData *first = NULL;

	g_return_val_if_fail (engine != NULL, FALSE);
	g_return_val_if_fail (IS_SIM_ENGINE (engine), FALSE);

	plot = g_new0 (Plot, 1);

	plot->sim = engine;
	plot->window = plot_window_create (plot);

        plot->logx = FALSE;
        plot->logy = FALSE;

        plot->padding_x = PLOT_PADDING_X;
        plot->padding_y = PLOT_PADDING_Y;

	plot->axis_group = NULL;
	plot->plot_group = NULL;

        /*  Get the analysis we have */
        analysis = engine->analysis;
        for (; analysis ; analysis = analysis->next) {
                SimulationData *sdat = SIM_DATA (analysis->data);
                gchar *str = sim_engine_analysis_name (sdat);
                if (sdat->type == OP_POINT)
			continue;

                if (s_current == NULL) {
                        s_current = str;
                        first = sdat;
                }
                combo_items = g_list_append (combo_items, str);                
        }
        gtk_combo_set_popdown_strings (GTK_COMBO (plot->combo), combo_items);
        g_list_free (combo_items );
        entry = GTK_COMBO (plot->combo)->entry;
        gtk_signal_connect (GTK_OBJECT (entry), "changed",
                            GTK_SIGNAL_FUNC (analysis_selected), plot);

        gtk_entry_set_text (GTK_ENTRY (entry), s_current ? s_current : "?");

       	clist = gtk_object_get_data (GTK_OBJECT (plot->window), "clist");

        plot->title = g_strdup_printf (_("Plot - %s"), s_current);
        plot->xtitle = get_variable_units (first ? first->var_units[0] : "####");
        plot->ytitle = g_strdup (first->var_units[1]);

	gtk_signal_connect (GTK_OBJECT (clist), "select_row",
			    GTK_SIGNAL_FUNC (plot_selected), plot);
        gtk_signal_connect (GTK_OBJECT (clist), "unselect_row",
                            GTK_SIGNAL_FUNC (plot_selected), plot);


	gtk_clist_select_row (GTK_CLIST (clist), 0, 0);
	g_free (s_current);
	
	return TRUE;
}


static void
plot_draw_axis (Plot *plot, AxisType axtype, double x_min, double x_max)
{
        GnomeCanvasPoints *points;
        gdouble          x,x1,x0,y0,xtit,ytit,fAxis,fScale;
        gboolean         scale,x_axis;
        gchar           *title, *line;
        GnomeCanvasItem *plot_title,*dmy;
        GtkArg           arg[5];
        gint             Ndiv[2] = { 10, 5 }, nBins[2];
        gdouble          Wmn[2], Wmx[2], Bw[2];
        const static double eps = 0.00001;
        gboolean logx;
	gdouble text_width, text_height;

        /* Get some axis-dependent values  */

        points = gnome_canvas_points_new (2);
        x_axis = (axtype == X_AXIS);
        if (x_axis) {    
                y0 =  (plot->plot_min < 0.0 && plot->plot_max > 0.0 ? 0.0 : plot->plot_min); 
                x0 =  0.0;
                y0 = GET_Y_POINT (y0, plot->plot_min, plot->y_factor);
                scale =  plot->x_scale != 1.0;
                title = plot->xtitle;
                xtit  = (double) PLOT_WIDTH;
                ytit  = y0 + 5.0;
                fAxis = plot->x_factor;
                fScale = plot->x_scale;
                logx = plot->logx;
        } 
        else {
                x0 = 0.0;
                y0 = PLOT_HEIGHT;
                scale = plot->y_scale != 1.0;
                title  = plot->ytitle;
                xtit = -plot->padding_x / 2.0 + 5.0;
                ytit = -plot->padding_y / 2.0 -5.0;
                fAxis = plot->y_factor;
                fScale = plot->y_scale;
                logx = plot->logy;
        }

        /* Draw the axis arrow  */
        points->coords[0] = x0;
        points->coords[1] = y0;
        points->coords[2] = (x_axis ? PLOT_WIDTH + 10.0 : 0.0);    /* extra 10.0 !!! */
        points->coords[3] = (x_axis ? y0                : -10.0);
        gnome_canvas_item_new (plot->axis_group,
                               gnome_canvas_line_get_type (),
                               "points", points,
                               "fill_color", PLOT_AXIS_COLOR,
                               "last_arrowhead", TRUE,
                               "arrow_shape_a", 6.0,
                               "arrow_shape_b", 8.0,
                               "arrow_shape_c", 5.0,
                               NULL);
   
        /* Draw axis title */
        if (scale) {
                if (x_axis)
                        line = g_strdup_printf ("%s x10", title);
                else
                        line = g_strdup_printf ("%s\nx10", title);
        }
        else {
                line = g_strdup_printf ("%s", title);
	}
   
        plot_title = gnome_canvas_item_new (plot->axis_group,
                                            gnome_canvas_text_get_type (),
                                            "text", line,
                                            "x", xtit,
                                            "y", ytit,
                                            "fill_color", PLOT_AXIS_COLOR,
                                            "anchor", GTK_ANCHOR_NE,
                                            "font", _(AXIS_FONT),
                                            NULL);

	g_free (line);
	
        /*  Flush the X-axis to the right  */
        if (x_axis) {
		double text_height;

                gtk_object_get (GTK_OBJECT(plot_title), "text_height", &text_height, NULL);
                gtk_object_set (GTK_OBJECT(plot_title), 
                                "y", (double) PLOT_HEIGHT + 1.25 * text_height, 
				NULL); 
        }
	
        /*  Set the scale of the axis if needed  */
        if (scale) {
                float nw;
		double title_x, title_y;

                line = g_strdup_printf ("%.0f", log10 (fScale));

                gtk_object_get (GTK_OBJECT (plot_title), 
				"text_height", &text_height,
				"x", &title_x,
				"y", &title_y,
				NULL);

                nw =  GTK_VALUE_DOUBLE(arg[3])/strlen(title);
                dmy = gnome_canvas_item_new(plot->axis_group,
                                            gnome_canvas_text_get_type (),
                                            "text", line,
                                            "x",  x_axis ? title_x : (title_x - (strlen (title) - 3.5) * nw),
                                            "y",  title_y +
                                            (x_axis ? -0.25 : 0.4) * text_height,
                                            "fill_color", PLOT_AXIS_COLOR,
                                            "anchor", GTK_ANCHOR_NW,
                                            "font", _(UPSCRIPT_FONT),
                                            NULL);
		g_free (line);
        }

        /* Now put the ticks and labels on the axis */
        getbins (x_min, x_max, Ndiv[0], Wmn, Wmx, nBins, Bw);
        if ((x_min - Wmn[0]) > eps) {
		Wmn[0] += Bw[0]; nBins[0]--;
	}
        if ((Wmx[0] - x_max) > eps) {
		Wmx[0] -= Bw[0]; nBins[0]--;
	}
        getbins (0, Bw[0], Ndiv[1], Wmn + 1,Wmx + 1, nBins + 1, Bw + 1);

        for (x = Wmn[0]; x <= Wmx[0] + eps; x += Bw[0]) {
                gdouble val = x / fScale;

                points->coords[0] =  x_axis ? GET_X_POINT(x, x_min, fAxis) : x0 ;
                points->coords[1] =  x_axis ? y0 : GET_Y_POINT(x, x_min, fAxis);
                points->coords[2] =  x_axis ? points->coords[0] : x0 + 5.0;
                points->coords[3] =  x_axis ? y0 - 5.0 : points->coords[1];
                if (fabs(val) < 1.e-5) { 
			val = 0.0;
		}
		
                line = g_strdup_printf ("%.3g", logx ? pow (10.0, val) : val);
                gnome_canvas_item_new (plot->axis_group,
                                       gnome_canvas_line_get_type (),
                                       "points", points,
                                       "fill_color", PLOT_AXIS_COLOR,
                                       NULL);
                dmy = gnome_canvas_item_new (plot->axis_group,
                                             gnome_canvas_text_get_type (),
                                             "text", line,
                                             "x", points->coords[0],
                                             "y", points->coords[1],
                                             "fill_color", "red",
                                             "anchor", GTK_ANCHOR_NW,
                                             "font", _(AXIS_FONT),
                                             NULL);

		g_free (line);
      
                /* Align label */
                gtk_object_get (GTK_OBJECT(dmy), "text_width", &text_width, "text_height", &text_height, NULL);
                gtk_object_set (GTK_OBJECT(dmy), 
                                "x", (double) (x_axis 
                                       ? points->coords[0] - text_width/2.0
                                       : points->coords[0] - text_width - 4),
                                "y", (double) (x_axis
                                       ? points->coords[1] + text_height / 2.0
                                       : points->coords[1] - text_height / 2.0),
                                NULL);


                for (x1 = Wmn[1] + Bw[1]; x1 < Wmx[1] + eps && x + x1 < x_max; x1 += Bw[1]) {
                        points->coords[0] = x_axis ? GET_X_POINT (x + x1, plot->x_min, fAxis) : x0;
                        points->coords[1] = x_axis ? y0 : GET_Y_POINT (x + x1, x_min, fAxis);
                        points->coords[2] = x_axis ? points->coords[0] : 2.0;
                        points->coords[3] = x_axis ? y0 - 2.0 : points->coords[1];
                        gnome_canvas_item_new (plot->axis_group,
                                               gnome_canvas_line_get_type (),
                                               "points", points,
                                               "fill_color", PLOT_AXIS_COLOR,
                                               NULL);    
                }
        }
   
        gnome_canvas_points_free (points);
}


/*
 * GetBins
 *
 *   Author: Carlos
 *   Purpose: Finds out the binning for the scale of the control
 *            Wmin,Wmax       : range of the scale
 *            Nold            : wanted number of bins
 *            BinLow, BinHigh : Optimized low and hig bins for which
 *                              a label is to be drawn
 *            Nbins           : number of bins
 *            Bwidth          : bin width between ticks
 *
 */
static void
getbins (gdouble Wmin, gdouble Wmax, gint Nold,
	 gdouble *BinLow, gdouble *BinHigh, gint *Nbins, gdouble *Bwidth)
{
	gint lwid, kwid;
	gint ntemp = 0;
	gint jlog  = 0;
	gdouble siground = 0;
	gdouble alb, awidth, sigfig,atest;
	
	gdouble AL = MIN (Wmin, Wmax);
	gdouble AH = MAX (Wmin, Wmax);
	if (AL == AH) 
		AH = AL + 1;
	
/*
 * If Nold  ==  -1, program uses binwidth input from calling routine.
 */
	if (Nold == -1 && (*Bwidth) > 0) 
		goto L90;
	
	ntemp = MAX (Nold, 2);
	if (ntemp < 1) 
		ntemp = 1;
	
/*
 * Get nominal bin width in exponential form.
 */
	
 L20:
	awidth = (AH - AL) / ((gdouble)ntemp);
	if (awidth > 1.e30) 
		goto LOK;
	
	jlog   = (gint)(log10 (awidth));
	if (awidth <= 1) 
		jlog--;
	
	sigfig = awidth * ((gdouble)pow (10, -jlog));
	
/*
 * Round mantissa up to 1, 2, 2.5, 5, or 10. 
 */
	
	if (sigfig <= 1)
		siground = 1;
	else if (sigfig <= 2)
		siground = 2;
/*   else if (sigfig <= 2.5)  siground = 2.5;*/
	else if (sigfig <= 5)
		siground = 5;
	else {
		siground = 1;   
		jlog++; 
	}
	
	(*Bwidth) = siground * pow (10, jlog);
	
/*
 * Get new bounds from new width Bwidth.
 */
	
 L90:
	alb = AL / (*Bwidth);
	lwid = (gint) (alb);
	if (alb < 0)
		lwid--;
	(*BinLow) = siground*((gdouble)lwid)*pow(10,jlog);
	alb = AH / (*Bwidth) + 1.00001f;
	kwid = (gint) alb;
	if (alb < 0) 
		kwid--;
	(*BinHigh) = siground * ((gdouble) kwid) * pow (10, jlog);
	(*Nbins) = kwid - lwid;
	if (Nold == -1)
		goto LOK;
	if (Nold <= 5) {          /* Request for one bin is difficult case. */
		if (Nold > 1 || (*Nbins) == 1)
			goto LOK;
		
		(*Bwidth) = (*Bwidth) * 2;
		(*Nbins) = 1;
		goto LOK;
	}
	if (2 * (*Nbins) == Nold) {
		ntemp++;
		goto L20;
	}
	
 LOK:
	atest = (*Bwidth) * 0.0001f;
	if (fabs ((*BinLow) - Wmin)  >= atest) {
		(*BinLow) += (*Bwidth);
		(*Nbins)--;
	}
	
	if (fabs ((*BinHigh) - Wmax) >= atest) { 
		(*BinHigh) -= (*Bwidth);
		(*Nbins)--;
	}
}

static void
get_order_of_magnitude (double val, double *man, double *pw)
{
        gdouble b;
        gdouble sx;

        b = (val != 0.0 ? log10 (fabs (val)) / 3.0 : 0.0);
        b  = 3.0 * rint (b);
        sx = (b != 0.0 ? pow (10.0, b) : 1.0);
        *man = val / sx;
        *pw  = b;
}
