/*
 * node.c
 *
 * 
 * Author: 
 *  Richard Hult <rhult@hem.passagen.se>
 * 
 *  http://www.dtek.chalmers.se/~d4hult/oregano/ 
 * 
 * Copyright (C) 1999,2000  Richard Hult 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <math.h>
#include "node.h"
#include "part.h"

static void node_class_init (NodeClass *klass);
static void node_init (Node *node);

enum {
	DOT_ADDED,
	DOT_REMOVED,
	VOLTAGE_CHANGED,
	LAST_SIGNAL,
};

static guint node_signals [LAST_SIGNAL] = { 0 };
static GtkObject *parent_class = NULL;

guint
node_get_type (void)
{
	static guint node_type = 0;

	if (!node_type) {
		static const GtkTypeInfo node_info = {
			"Node",
			sizeof (Node),
			sizeof (NodeClass),
			(GtkClassInitFunc) node_class_init,
			(GtkObjectInitFunc) node_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		
		node_type = gtk_type_unique (GTK_TYPE_OBJECT, &node_info);
	}

	return node_type;
}

static void
node_destroy (GtkObject *object)
{
/*	Node *node = NODE (object);*/

	GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
node_class_init (NodeClass *klass)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *)klass;
	parent_class = gtk_type_class (GTK_TYPE_OBJECT);

	object_class->destroy = node_destroy;

	node_signals [DOT_ADDED] = 
 		gtk_signal_new ("dot_added",
 				GTK_RUN_FIRST,
 				object_class->type,
				0,
 				gtk_marshal_NONE__POINTER,
 				GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

	node_signals [DOT_REMOVED] = 
 		gtk_signal_new ("dot_removed",
 				GTK_RUN_FIRST,
 				object_class->type,
				0,
 				gtk_marshal_NONE__POINTER,
 				GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

	node_signals [VOLTAGE_CHANGED] = 
 		gtk_signal_new ("voltage_changed",
 				GTK_RUN_FIRST,
 				object_class->type,
				0,
 				gtk_marshal_NONE__NONE,
 				GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals (object_class, node_signals, LAST_SIGNAL);

}

static void
node_init (Node *node)
{
	node->pin_count = 0;
	node->wire_count = 0;
	node->pins = NULL;
	node->wires = NULL;
	node->visited = FALSE;
}

Node *
node_new (SheetPos pos)
{
	Node *node;

	node = gtk_type_new (node_get_type ());

	node->key = pos;

	return node;
}

#define SEP(p1,p2) (p1.x == p2.x && p1.y == p2.y)
#define ON_THE_WIRE(p1,start,end) ( fabs((end.y-start.y)*(p1.x-start.x)-(end.x-start.x)*(p1.y-start.y))<1.e-5 )
gboolean
node_needs_dot (Node *node)
{
	Wire *wire1, *wire2;
	SheetPos start_pos1, length1, end_pos1;
	SheetPos start_pos2, length2, end_pos2;

	if ((node->pin_count > 2) || (node->wire_count > 2))
		return TRUE;
	else if ((node->pin_count + node->wire_count) > 2)
		return TRUE;
	else if (node->wire_count == 2) {
		/*
		 * Check that we don't have two wire endpoints.
		 */

	   
		wire1 = node->wires->data;
		wire2 = node->wires->next->data;

		wire_get_pos_and_length (wire1, &start_pos1, &length1);		
		wire_get_pos_and_length (wire2, &start_pos2, &length2);

		end_pos1.x = start_pos1.x + length1.x;
		end_pos1.y = start_pos1.y + length1.y;
		end_pos2.x = start_pos2.x + length2.x;
		end_pos2.y = start_pos2.y + length2.y;

		if (!(SEP (start_pos1, start_pos2) ||
		      SEP (start_pos1, end_pos2) ||
		      SEP (end_pos1, end_pos2) ||
		      SEP (end_pos1, start_pos2))) {

		   /* 
		      The dot is only needed when the end/start-point of
		      one of the wires in on the other wire.
		   */
		   if ( ON_THE_WIRE(start_pos1,start_pos2,end_pos2) ||
			ON_THE_WIRE(  end_pos1,start_pos2,end_pos2) ||
			ON_THE_WIRE(start_pos2,start_pos1,end_pos1) ||
			ON_THE_WIRE(  end_pos2,start_pos1,end_pos1) 
			) {
			return TRUE;
		   }
		   else
		      return FALSE;
		}
		
		return FALSE;
	} else if (node->pin_count == 1 && node->wire_count == 1) {
		/*
		 * Check if we have one wire with a pin in the 'middle'.
		 */
		wire1 = node->wires->data;
		wire_get_pos_and_length (wire1, &start_pos1, &length1);		
		end_pos1.x = start_pos1.x + length1.x;
		end_pos1.y = start_pos1.y + length1.y;

		if (!SEP (node->key, start_pos1) && !SEP (node->key, end_pos1))
			return TRUE;
	}
	
	return FALSE;
}

gint
node_add_pin (Node *node, Pin *pin)
{
	gboolean dot;

	g_return_val_if_fail (node != NULL, FALSE);
	g_return_val_if_fail (IS_NODE (node), FALSE);
	g_return_val_if_fail (pin != NULL, FALSE);

	if (g_slist_find (node->pins, pin)) {
/*		g_print ("node_add_pin: pin already there.\n");*/
		return FALSE;
	}

	dot = node_needs_dot (node);
	
	node->pins = g_slist_prepend (node->pins, pin);
	node->pin_count++;

	if (!dot && node_needs_dot (node))
		gtk_signal_emit (GTK_OBJECT (node), node_signals[DOT_ADDED], &node->key);

	return TRUE;
}

gint
node_remove_pin (Node *node, Pin *pin)
{
	gboolean dot;

	g_return_val_if_fail (node != NULL, FALSE);
	g_return_val_if_fail (IS_NODE (node), FALSE);
	g_return_val_if_fail (pin != NULL, FALSE);

	if (node->pin_count == 0)
		return FALSE;

	dot = node_needs_dot (node);

	node->pins = g_slist_remove (node->pins, pin);
	node->pin_count--;

	if (dot && !node_needs_dot (node))
		gtk_signal_emit (GTK_OBJECT (node), node_signals[DOT_REMOVED], &node->key);

	return TRUE;
}

gint
node_add_wire (Node *node, Wire *wire)
{
	gboolean dot;

	g_return_val_if_fail (node != NULL, FALSE);
	g_return_val_if_fail (IS_NODE (node), FALSE);
	g_return_val_if_fail (wire != NULL, FALSE);
	g_return_val_if_fail (IS_WIRE (wire), FALSE);

	if (g_slist_find (node->wires, wire)) {
/*		g_print ("node_add_wire: wire already there.\n");*/
		return FALSE;
	}

	dot = node_needs_dot (node);

	node->wires = g_slist_prepend (node->wires, wire);
	node->wire_count++;

	if (!dot && node_needs_dot (node))
		gtk_signal_emit (GTK_OBJECT (node), node_signals[DOT_ADDED], &node->key);

	return TRUE;
}

gint
node_remove_wire (Node *node, Wire *wire)
{
	gboolean dot;

	g_return_val_if_fail (node != NULL, FALSE);
	g_return_val_if_fail (IS_NODE (node), FALSE);
	g_return_val_if_fail (wire != NULL, FALSE);
	g_return_val_if_fail (IS_WIRE (wire), FALSE);

	if (node->wire_count == 0)
		return FALSE;

	if (!g_slist_find (node->wires, wire)) {
		g_print ("node_remove_wire: not there.\n");
		return FALSE;
	}

	dot = node_needs_dot (node);

	node->wires = g_slist_remove (node->wires, wire);
	node->wire_count--;

	if (dot && !node_needs_dot (node))
		gtk_signal_emit (GTK_OBJECT (node), node_signals[DOT_REMOVED], &node->key);

	return TRUE;
}

gint
node_is_empty (Node *node)
{
	g_return_val_if_fail (node != NULL, FALSE);
	g_return_val_if_fail (IS_NODE (node), FALSE);

	if (node->wire_count == 0 && node->pin_count == 0)
	    return TRUE;

	return FALSE;
}

gint
node_is_visited (Node *node)
{
	g_return_val_if_fail (node != NULL, FALSE);
	g_return_val_if_fail (IS_NODE (node), FALSE);
	
	return node->visited;
}

void
node_set_visited (Node *node, gboolean is_visited)
{
	g_return_if_fail (node != NULL);
	g_return_if_fail (IS_NODE (node));
	
	node->visited = is_visited;
}
