/*
 * item-data.c
 *
 * ItemData object: part and wire model superclass.
 * 
 * Author: 
 *  Richard Hult <rhult@hem.passagen.se>
 * 
 *  http://www.dtek.chalmers.se/~d4hult/oregano/ 
 * 
 * Copyright (C) 1999,2000  Richard Hult 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <math.h>
#include <gnome.h>
#include "item-data.h"
#include "node-store.h"
#include "print.h"

static void item_data_class_init (ItemDataClass *klass);
static void item_data_init (ItemData *item_data);
static void item_data_set_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void item_data_get_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void item_data_copy (ItemData *dest, ItemData *src);
static void item_data_moved (ItemData *item_data, SheetPos *delta);

enum {
	ARG_0,
	ARG_STORE,
	ARG_POS
};

enum {
	MOVED,
	ROTATED,
	FLIPPED,
	HIGHLIGHT,
	LAST_SIGNAL
};

struct _ItemDataPriv { 
	NodeStore *store;
	SheetPos pos;

	/* 
	 * Bounding box.
	 */
	SheetPos b1, b2;
};

static guint item_data_signals [LAST_SIGNAL] = { 0 };
static GtkObjectClass *parent_class = NULL;

guint
item_data_get_type (void)
{
	static guint item_data_type = 0;

	if (!item_data_type) {
		static const GtkTypeInfo item_data_info = {
			"ItemData",
			sizeof (ItemData),
			sizeof (ItemDataClass),
			(GtkClassInitFunc) item_data_class_init,
			(GtkObjectInitFunc) item_data_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
	
		item_data_type = gtk_type_unique (gtk_object_get_type (), &item_data_info);
	}

	return item_data_type;
}

static void
item_data_shutdown (GtkObject *object)
{
	/*
	 * Remove the item from the sheet node store if there.
	 */
	if (ITEM_DATA (object)->priv->store) 
		item_data_unregister (ITEM_DATA (object));
	
	GTK_OBJECT_CLASS (parent_class)->shutdown (object);
}

static void
item_data_destroy (GtkObject *object)
{
	GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
item_data_class_init (ItemDataClass *klass)
{
	GtkObjectClass *object_class;

	parent_class = gtk_type_class (GTK_TYPE_OBJECT);

	object_class = (GtkObjectClass *) klass;

	gtk_object_add_arg_type ("ItemData::store", GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_STORE);
	gtk_object_add_arg_type ("ItemData::pos", GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_POS);

	object_class->destroy = item_data_destroy;
	object_class->set_arg = item_data_set_arg;
	object_class->get_arg = item_data_get_arg;
	object_class->shutdown = item_data_shutdown;

	item_data_signals [MOVED] = 
		gtk_signal_new ("moved",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (ItemDataClass, moved),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);

	item_data_signals [ROTATED] = 
		gtk_signal_new ("rotated",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__INT,
				GTK_TYPE_NONE, 1, 
				GTK_TYPE_INT);

	item_data_signals [FLIPPED] = 
		gtk_signal_new ("flipped",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__INT,
				GTK_TYPE_NONE, 1, 
				GTK_TYPE_INT);

	item_data_signals [HIGHLIGHT] = 
		gtk_signal_new ("highlight",
				GTK_RUN_FIRST,
				object_class->type,
				0, 
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0); 

	gtk_object_class_add_signals (object_class, item_data_signals, LAST_SIGNAL);

	/*
	 * Methods.
	 */
	klass->clone = NULL;
	klass->copy = item_data_copy;
	klass->rotate = NULL;
	klass->flip = NULL;
	klass->reg = NULL;
	klass->unreg = NULL;

	/*
	 * Signals.
	 */
	klass->moved = item_data_moved;  // FIXME: remove this field.
}

static void
item_data_init (ItemData *item_data)
{
	ItemDataPriv *priv = g_new0 (ItemDataPriv, 1);

	priv->pos.x = 0;
	priv->pos.y = 0;
	priv->b1.x = priv->b1.y = priv->b2.x = priv->b2.y = 0.0;

	item_data->priv = priv;
}

ItemData *
item_data_new (void)
{
	ItemData *item_data;

	item_data = gtk_type_new (item_data_get_type ());

	return item_data;
}

static void
item_data_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	ItemData *item_data = ITEM_DATA (object);

	switch (arg_id) {
	case ARG_STORE:
		item_data->priv->store = GTK_VALUE_POINTER (*arg);
		break;
	default:
		break;
	}
}

static void
item_data_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	ItemData *item_data = ITEM_DATA (object);

	switch (arg_id) {
	case ARG_STORE:
		GTK_VALUE_POINTER (*arg) = item_data->priv->store;
		break;
	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

void
item_data_get_pos (ItemData *item_data, SheetPos *pos)
{
	g_return_if_fail (item_data != NULL);
	g_return_if_fail (IS_ITEM_DATA (item_data));
	g_return_if_fail (pos != NULL);

	*pos = item_data->priv->pos;
}

void
item_data_set_pos (ItemData *item_data, SheetPos *pos)
{
	ItemDataPriv *priv;
	SheetPos delta;

	g_return_if_fail (item_data != NULL);
	g_return_if_fail (IS_ITEM_DATA (item_data));

	if (pos == NULL)
		return;

	priv = item_data->priv;

	delta.x = pos->x - priv->pos.x;
	delta.y = pos->y - priv->pos.y;

	priv->pos.x = pos->x;
	priv->pos.y = pos->y;

	gtk_signal_emit (GTK_OBJECT (item_data), item_data_signals[MOVED], &delta);
}

void
item_data_move (ItemData *item_data, SheetPos *delta)
{
	ItemDataPriv *priv;

	g_return_if_fail (item_data != NULL);
	g_return_if_fail (IS_ITEM_DATA (item_data));

	if (delta == NULL)
		return;

	priv = item_data->priv;
	priv->pos.x += delta->x;
	priv->pos.y += delta->y;

	gtk_signal_emit (GTK_OBJECT (item_data), item_data_signals[MOVED], delta);
}

static void
item_data_moved (ItemData *item_data, SheetPos *delta)
{
//	g_print ("mooooooved\n");
}

gpointer /*NodeStore * */
item_data_get_store (ItemData *item_data)
{
	g_return_val_if_fail (item_data != NULL, NULL);
	g_return_val_if_fail (IS_ITEM_DATA (item_data), NULL);

	return item_data->priv->store;
}

ItemData *
item_data_clone (ItemData *src)
{
	ItemDataClass *id_class;

	g_return_val_if_fail (src != NULL, NULL);
	g_return_val_if_fail (IS_ITEM_DATA (src), NULL);

	id_class = ITEM_DATA_CLASS (GTK_OBJECT (src)->klass);
	if (id_class->clone == NULL)
		return NULL;

	return id_class->clone (src);
}

static void
item_data_copy (ItemData *dest, ItemData *src)
{
	g_return_if_fail (dest != NULL);
	g_return_if_fail (IS_ITEM_DATA (dest));
	g_return_if_fail (src != NULL);
	g_return_if_fail (IS_ITEM_DATA (src));

	dest->priv->pos = src->priv->pos;
	dest->priv->store = NULL;
}

void
item_data_get_relative_bbox (ItemData *data, SheetPos *p1, SheetPos *p2)
{
	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_ITEM_DATA (data));

	if (p1) {
		p1->x = data->priv->b1.x;
		p1->y = data->priv->b1.y;
	}

	if (p2) {
		p2->x = data->priv->b2.x;
		p2->y = data->priv->b2.y;
	}
}

void
item_data_get_absolute_bbox (ItemData *data, SheetPos *p1, SheetPos *p2)
{
	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_ITEM_DATA (data));

	item_data_get_relative_bbox (data, p1, p2);
	
	if (p1) {
		p1->x += data->priv->pos.x;
		p1->y += data->priv->pos.y;
	}

	if (p2) {
		p2->x += data->priv->pos.x;
		p2->y += data->priv->pos.y;
	}
}

void
item_data_set_relative_bbox (ItemData *data, SheetPos *p1, SheetPos *p2)
{
	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_ITEM_DATA (data));

	if (p1) {
		data->priv->b1.x = p1->x;
		data->priv->b1.y = p1->y;
	}

	if (p2) {
		data->priv->b2.x = p2->x;
	        data->priv->b2.y = p2->y;
	}
}

void
item_data_list_get_absolute_bbox (GList *item_data_list, SheetPos *p1, SheetPos *p2)
{
	GList *list;
	SheetPos b1, b2;

	if (item_data_list == NULL)
		return;

	item_data_get_absolute_bbox (item_data_list->data, p1, p2);

	for (list = item_data_list; list; list = list->next) {
		item_data_get_absolute_bbox (list->data, &b1, &b2);		

		if (p1) {
			p1->x = MIN (p1->x, b1.x);
			p1->y = MIN (p1->y, b1.y);
		}

		if (p2) {
			p2->x = MAX (p2->x, b2.x);
			p2->y = MAX (p2->y, b2.y);
		}
	}
}

/*
void
item_data_update_bbox (ItemData *data)
{
	ItemDataClass *id_class;

	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_ITEM_DATA (data));
	
	id_class = ITEM_DATA_CLASS (GTK_OBJECT (data)->klass);
	if (id_class->update_bbox) {
		id_class->update_bbox (data);
	}
}
*/

void
item_data_rotate (ItemData *data, int angle, SheetPos *center)
{
	ItemDataClass *id_class;

	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_ITEM_DATA (data));
	
	id_class = ITEM_DATA_CLASS (GTK_OBJECT (data)->klass);
	if (id_class->rotate) {
		id_class->rotate (data, angle, center);
	}
}

void
item_data_flip (ItemData *data, gboolean horizontal, SheetPos *center)
{
	ItemDataClass *id_class;

	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_ITEM_DATA (data));
	
	id_class = ITEM_DATA_CLASS (GTK_OBJECT (data)->klass);
	if (id_class->flip) {
		id_class->flip (data, horizontal, center);
	}
}

void
item_data_unregister (ItemData *data)
{
	ItemDataClass *id_class;
	
	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_ITEM_DATA (data));
	
	id_class = ITEM_DATA_CLASS (GTK_OBJECT (data)->klass);
	if (id_class->unreg) {
		id_class->unreg (data);
	}
}

void
item_data_register (ItemData *data)
{
	ItemDataClass *id_class;
	
	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_ITEM_DATA (data));
	
	id_class = ITEM_DATA_CLASS (GTK_OBJECT (data)->klass);
	if (id_class->reg) {
		id_class->reg (data);
	}
}

char *
item_data_get_refdes_prefix (ItemData *data)
{
	ItemDataClass *id_class;

	g_return_val_if_fail (data != NULL, NULL);
	g_return_val_if_fail (IS_ITEM_DATA (data), NULL);

	id_class = ITEM_DATA_CLASS (GTK_OBJECT (data)->klass);
	if (id_class->get_refdes_prefix) {
		return id_class->get_refdes_prefix (data);
	}
	
	return NULL;
}

void
item_data_set_property (ItemData *data, char *property, char *value)
{
	ItemDataClass *id_class;

	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_ITEM_DATA (data));

	id_class = ITEM_DATA_CLASS (GTK_OBJECT (data)->klass);
	if (id_class->set_property) {
		return id_class->set_property (data, property, value);
	}
}

void
item_data_print (ItemData *data, OreganoPrintContext *ctxt)
{
	ItemDataClass *id_class;

	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_ITEM_DATA (data));
	g_return_if_fail (ctxt != NULL);

	id_class = ITEM_DATA_CLASS (GTK_OBJECT (data)->klass);
	if (id_class->print) {
		return id_class->print (data, ctxt);
	}
}

