//                                               -*- C++ -*-
/**
 *  @file  t_UserDefinedSpectralModel_std.cxx
 *  @brief The test file of class UserDefinedSpectralModel
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy$
 *  @date:   $LastChangedDate$
 *  Id:      $Id$
 */
#include <iostream>
#include "OT.hxx"
#include "OTtestcode.hxx"

using namespace OT;
using namespace OT::Test;

NumericalScalar clean(NumericalScalar in)
{
  // For -0.0 trouble
  if (fabs(in) < 1.e-6) return 0.0;
  return in;
  //return 1.e-6*round(1.e6*in);
}

HermitianMatrix clean(HermitianMatrix in)
{
  UnsignedLong dim(in.getDimension());
  for(UnsignedLong i = 0; i < dim; i++)
    for(UnsignedLong j = 0; j <=i; j++)
      {
        NumericalScalar realIJ = clean(real(in(i, j)));
        NumericalScalar imagIJ = clean(imag(in(i, j)));
        in(i, j) = NumericalComplex(realIJ, imagIJ);
      }
  return in;
}

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);

  try {

    /* Default constructor */
    UserDefinedSpectralModel myDefautModel;
    fullprint << "myDefautModel = " << myDefautModel << std::endl;

    /* Default dimension parameter to evaluate the model */
    const UnsignedLong dimension(2);

    /* Amplitude values */
    NumericalPoint amplitude(dimension);
    /* Scale values */
    NumericalPoint scale(dimension);
    /* Spatial correclation */
    CorrelationMatrix spatialCorrelation(dimension);
    for (UnsignedLong index = 0 ; index < dimension; ++index)
      {
        // constant amplitude
        amplitude[index] = 1.0 ;
        scale[index] = (index + 1.0) / dimension ;
        if (index > 0) spatialCorrelation(index, index - 1) = 1.0 / index;
      }

    /* Sample a CauchyModel */
    CauchyModel referenceModel(amplitude, scale, spatialCorrelation);

    UnsignedLong size(5);
    UserDefinedSpectralModel::HermitianMatrixCollection dspCollection(size);
    RegularGrid frequencyGrid(0.0, 2.0 / size, size);
    for (UnsignedLong i = 0; i < size; ++i)
      dspCollection[i] = referenceModel.computeSpectralDensity(frequencyGrid.getValue(i));

    /* Create a UserDefinedSpectralModel */
    UserDefinedSpectralModel myModel(frequencyGrid, dspCollection);
    fullprint << "myModel=" << myModel << std::endl;

    /* Sample the UserDefinedSpectralModel */
    RegularGrid samplingGrid(-0.4, 1.0 / (3.0 * size), 5 * size);
    for (UnsignedLong i = 0; i < samplingGrid.getN(); ++i)
      {
        NumericalScalar frequency(samplingGrid.getValue(i));
        fullprint << "frequency=" << clean(frequency) << ", myModel=\n" << clean(myModel.computeSpectralDensity(frequency)) << ", referenceModel=\n" << clean(referenceModel.computeSpectralDensity(frequency)) << std::endl;
      }

    /* Test save / load */
    Study myStudy;
    myStudy.setStorageManager(XMLStorageManager("testStudy.ot"));
    myStudy.add("myModel", myModel);
    myStudy.save();

    Study loadedStudy;
    loadedStudy.setStorageManager(XMLStorageManager("testStudy.ot"));
    loadedStudy.load();
    UserDefinedSpectralModel myModelLoaded;
    loadedStudy.fillObject("myModel", myModelLoaded);
    fullprint << "loaded model =" << myModelLoaded<< std::endl;

  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
