//                                               -*- C++ -*-
/**
 *  @file  TrendTransform.cxx
 *  @brief Abstract top-level class for all distributions
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: TrendTransform.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include "TrendTransform.hxx"
#include "PersistentObjectFactory.hxx"
#include "NumericalMathEvaluationImplementation.hxx"
#include "NumericalSample.hxx"
#include "Exception.hxx"

BEGIN_NAMESPACE_OPENTURNS

CLASSNAMEINIT(TrendTransform);

static Factory<TrendTransform> RegisteredFactory("TrendTransform");

/* Default constructor */
TrendTransform::TrendTransform()
  : TemporalFunction()
{
  // Nothing to do
}

/* Parameter constructor */
TrendTransform::TrendTransform(const NumericalMathFunction & function)
  : TemporalFunction()
{
  if (function.getInputDimension() != 1)
    throw InvalidDimensionException(HERE) << "NumericalMathFunction input dimension should be 1 for trend" ;
  p_evaluation_ = function.getEvaluationImplementation() ;
  // Set the descriptions
  setInputDescription(p_evaluation_->getInputDescription());
  setOutputDescription(p_evaluation_->getOutputDescription());
}

/* Parameter constructor */
TrendTransform::TrendTransform(const EvaluationImplementation & p_evaluation)
  : TemporalFunction()
{
  if (p_evaluation->getInputDimension() != 1)
    throw InvalidDimensionException(HERE) << "NumericalMathFunction input dimension should be 1 for trend" ;
  p_evaluation_ = p_evaluation;
  // Set the descriptions
  setInputDescription(p_evaluation_->getInputDescription());
  setOutputDescription(p_evaluation_->getOutputDescription());

}

/* Parameter constructor */
TrendTransform::TrendTransform(const NumericalMathEvaluationImplementation & evaluation)
  : TemporalFunction()
{
  if (evaluation.getInputDimension() != 1)
    throw InvalidDimensionException(HERE) << "NumericalMathFunction input dimension should be 1 for trend" ;
  p_evaluation_ = evaluation.clone();
}

/* Virtual constructor */
TrendTransform * TrendTransform::clone() const
{
  return new TrendTransform(*this);
}

/* Comparison operator */
Bool TrendTransform::operator ==(const TrendTransform & other) const
{
  return (getEvaluation() == other.getEvaluation());
}

/* String converter */
String TrendTransform::__repr__() const
{
  OSS oss;
  oss << "class=" << TrendTransform::GetClassName()
      << " evaluation=" << p_evaluation_->__repr__();
  return oss;
}

/* String converter */
String TrendTransform::__str__(const String & offset) const
{
  return OSS(false) << offset << p_evaluation_->__str__();
}

/* Operator () */
TimeSeries TrendTransform::operator() (const TimeSeries & inTS) const
{
  // the considered \phi function should from from R to R^n
  // the check will be done later
  // since the TimeSeriesImplementation is not available, we have to check some features carefully

  // 0 : Rework the timeSeries ==> for huge timeSeries, it is better to extract information from the ts
  // without using the asNumericalSample method
  // first we extract the timeGrid as a NumericalSample
  NumericalSample timeGridAsSample(inTS.getSize() ,1);

  for (UnsignedLong k = 0; k < timeGridAsSample.getSize(); ++k)
    {
      timeGridAsSample[k][0] = inTS[k][0];
    }

  // then we evaluation the function on the previous grid
  // result is a NumericalSample of dimension d = function.getOutputDimension()
  NumericalSample outputSample((*p_evaluation_)(timeGridAsSample));
  const UnsignedLong dimension(outputSample.getDimension());
  // finally as the function adds a trend, result
  for (UnsignedLong k = 0; k < timeGridAsSample.getSize(); ++k)
    {
      for (UnsignedLong d = 0; d < dimension; ++d)
        {
          // NumericalPoint should be compatible
          outputSample[k][d] += inTS[k][d+1];
        }
    }
  ++callsNumber_;
  return TimeSeries(inTS.getTimeGrid(), outputSample);
}

/* Method save() stores the object through the StorageManager */
void TrendTransform::save(Advocate & adv) const
{
  TemporalFunction::save(adv);
}

/* Method load() reloads the object from the StorageManager */
void TrendTransform::load(Advocate & adv)
{
  TemporalFunction::load(adv);
}

END_NAMESPACE_OPENTURNS
