//                                               -*- C++ -*-
/**
 * @file  EventRandomVectorImplementation.cxx
 * @brief The class that implements an event random vector, i.e. a
 *        composite random vector built upon a boolean function which
 *        is described by a threshold and a comparison operator. The
 *        antecedent of the event must be a "classical" composite random
 *        vector in the context of OpenTURNS.
 *
 * (C) Copyright 2005-2010 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: dutka $
 * \date   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 */

#include "EventRandomVectorImplementation.hxx"
#include "PersistentObjectFactory.hxx"
#include "ComparisonOperatorImplementation.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      CLASSNAMEINIT(EventRandomVectorImplementation);

      static Base::Common::Factory<EventRandomVectorImplementation> RegisteredFactory("EventRandomVectorImplementation");

      /* Constructor from RandomVector */
      EventRandomVectorImplementation::EventRandomVectorImplementation(const RandomVectorImplementation & antecedent,
								       const ComparisonOperator & op,
								       const NumericalScalar threshold,
								       const String & name)
	/* throw (InvalidArgumentException) */
	: CompositeRandomVector(name),
	  operator_(op),
	  threshold_(threshold)
      {
	// Event can only be constructed from composite random vectors
	if (!antecedent.isComposite())
	  throw InvalidArgumentException(HERE) << "Event can only be constructed from composite random vectors. The random vector ("
					       << antecedent << ") passed as first argument of event '"
					       << name << "' has incorrect type";
      	// EventRandomVectorImplementation can only be constructed from 1D random vectors
	if (antecedent.getDimension() != 1)
	  throw InvalidArgumentException(HERE) << "EventRandomVectorImplementation can only be constructed from 1D random vectors. The random vector ("
					       << antecedent << ") passed as first argument of eventRandomVectorImplementation '"
					       << name << "' has incorrect dimension";
	function_ = antecedent.getFunction();
	p_antecedent_ = antecedent.getAntecedent();
	setName(antecedent.getName());
      }
     
      EventRandomVectorImplementation * EventRandomVectorImplementation::clone() const
      {
        return new EventRandomVectorImplementation(*this);
      }

      /* String converter */
      String EventRandomVectorImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << EventRandomVectorImplementation::GetClassName()
	    << " antecedent=" << CompositeRandomVector::__repr__()
	    << " operator=" << operator_
	    << " threshold=" << threshold_;
	return oss;
      }
  
      /* Dimension accessor */
      UnsignedLong EventRandomVectorImplementation::getDimension() const
      {
	return 1;
      }

      /* Operator accessor */
      EventRandomVectorImplementation::ComparisonOperator EventRandomVectorImplementation::getOperator() const
      {
	return operator_;
      }
      
      /* Threshold accessor */
      NumericalScalar EventRandomVectorImplementation::getThreshold() const
      {
	return threshold_;
      }

      /* Realization accessor */
      EventRandomVectorImplementation::NumericalPoint EventRandomVectorImplementation::getRealization() const
      {
	return NumericalPoint(1, operator_(CompositeRandomVector::getRealization()[0], threshold_));
      }

      /* Numerical sample accessor */
      EventRandomVectorImplementation::NumericalSample EventRandomVectorImplementation::getNumericalSample(UnsignedLong size) const
      {
	// We don't build the return sample element by element because it doesn't
	// use the potential distribution of the computation. As the returned
	// sample can be huge, we use it twice in place
	// First, it stores a sample of its antecedent
	NumericalSample returnSample(CompositeRandomVector::getNumericalSample(size));
	// Then, we loop over the sample and substitute realizations of the eventRandomVectorImplementation
	// in place of the realizations of the antecedent
	for (UnsignedLong i = 0; i < size; i++)
	  {
	    returnSample[i][0] = operator_(returnSample[i][0], threshold_);
	  }
	returnSample.setName("EventRandomVectorImplementation sample");
	return returnSample;
      }

      /* Method save() stores the object through the StorageManager */
      void EventRandomVectorImplementation::save(StorageManager::Advocate & adv) const
      {
      	CompositeRandomVector::save(adv);
      	adv.saveAttribute( "operator_", operator_ );
      	adv.saveAttribute( "threshold_", threshold_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void EventRandomVectorImplementation::load(StorageManager::Advocate & adv)
      {
      	CompositeRandomVector::load(adv);
      	adv.loadAttribute( "operator_", operator_ );
      	adv.loadAttribute( "threshold_", threshold_ );
      }

    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
