//                                               -*- C++ -*-
/**
 *  @file  Compact.cxx
 *  @brief 
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $CompactChangedBy$
 *  @date:   $CompactChangedDate$
 *  Id:      $Id: Compact.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cstdlib>

#include "Compact.hxx"
#include "ResourceMap.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS
{

  namespace Base 
  {

    namespace Stat
    {

      /*
       * @class Compact
       */

      CLASSNAMEINIT(Compact);

      static Base::Common::Factory<Compact> RegisteredFactory("Compact");
      typedef Common::ResourceMap ResourceMap;

      const UnsignedLong Compact::DefaultHalfMaximumSize = 1024;//strtoul(ResourceMap::GetInstance().get("Compact-DefaultHalfMaximumSize").c_str(), NULL, 0);

      /* Constructor with parameters */
      Compact::Compact()
	: HistoryStrategyImplementation(),
	  halfMaximumSize_(DefaultHalfMaximumSize),
	  index_(0),
	  step_(1),
	  throwingCounter_(0)
      {
	// Nothing to do
      }

      /* Constructor with parameters */
      Compact::Compact(const UnsignedLong halfMaximumSize)
	: HistoryStrategyImplementation(),
	  halfMaximumSize_(halfMaximumSize),
	  index_(0),
	  step_(1),
	  throwingCounter_(0)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      Compact * Compact::clone() const
      {
	return new Compact(*this);
      }

      /* Store the point according to the strategy */
      void Compact::store(const NumericalPoint & point)
      {
	if (!isInitialized_)
	  {
	    sample_ = NumericalSample(2 * halfMaximumSize_, point.getDimension());
	    isInitialized_ = true;
	  }
	// If we don't throw this point
	if (throwingCounter_ == 0)
	  {
	    sample_[index_] = point;
	    ++index_;
	    // Reinitialize the counter
	    throwingCounter_ = step_;
	  }
	// Check if one needs compression
	if (index_ == 2 * halfMaximumSize_)
	  {
	    for (UnsignedLong i = 0; i < halfMaximumSize_; ++i)
	      {
		sample_[i] = sample_[2 * i + 1];
	      }
	    step_ *= 2;
	    throwingCounter_ = step_;
	    index_ = halfMaximumSize_;
	  }
	--throwingCounter_;
      }

      /* Sample accessor */
      NumericalSample Compact::getSample() const
      {
	// If nothing has been stored
	if (!isInitialized_) return sample_;
	NumericalSample outSample(index_, sample_.getDimension());
	for (UnsignedLong i = 0; i < index_; ++i)
	  {
	    // Fill the outSample component-wise in order to avoid memory comsumption due to the creation of Description objects
	    for (UnsignedLong j = 0; j < sample_.getDimension(); ++j)
	      {
		outSample[i][j] = sample_[i][j];
	      }
	  }
	return outSample;
      }

      /* HalfMaximumSize accessor */
      UnsignedLong Compact::getHalfMaximumSize() const
      {
	return halfMaximumSize_;
      }

      /* Index accessor */
      UnsignedLong Compact::getIndex() const
      {
	return index_;
      }

      /* String converter */
      String Compact::__repr__() const
      {
        OSS oss;
        oss << "class=" << Compact::GetClassName();
        oss << " halfMaximumSize_ "<<halfMaximumSize_;
        oss << " index_ "<<index_;
        oss << " step_ "<<step_;
        oss << " throwingCounter_ "<<throwingCounter_;
        return oss;
      }

      /* Method save() stores the object through the StorageManager */
      void Compact::save(StorageManager::Advocate & adv) const
      {
        HistoryStrategyImplementation::save(adv);
        adv.saveAttribute("halfMaximumSize_",halfMaximumSize_);
        adv.saveAttribute("index_",index_);
        adv.saveAttribute("step_",step_);
        adv.saveAttribute("throwingCounter_",throwingCounter_);
      }

      /* Method load() reloads the object from the StorageManager */
      void Compact::load(StorageManager::Advocate & adv)
      {
        HistoryStrategyImplementation::load(adv);
        adv.loadAttribute("halfMaximumSize_",halfMaximumSize_);
        adv.loadAttribute("index_",index_);
        adv.loadAttribute("step_",step_);
        adv.loadAttribute("throwingCounter_",throwingCounter_);

      }
      

    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */
