//                                               -*- C++ -*-
/**
 *  @file  WrapperFile.hxx
 *  @brief This class provides all the treatments for wrapper file manipulation
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: WrapperFile.hxx 1473 2010-02-04 15:44:49Z dutka $
 */
#ifndef OPENTURNS_WRAPPERFILE_HXX
#define OPENTURNS_WRAPPERFILE_HXX

#include <iostream>              // for std::ostream
#include "OTprivate.hxx"
#include "PersistentObject.hxx"
#include "WrapperData.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {


      /**
       * @class WrapperFile
       * @brief Provides all the treatments for wrapper file manipulation
       * @ingroup Wrapper
       *
       * All the methods needed for wrapper description file reading or writing
       * are located in this class. The class handles various file formats (plain text,
       * XML, etc.).
       */

      class WrapperFile
	: public Common::PersistentObject
      {
	CLASSNAME;
      public:

	typedef Common::NoWrapperFileFoundException NoWrapperFileFoundException;
	typedef Common::WrapperFileParsingException WrapperFileParsingException;

	/** @brief Find the path of a wrapper file from its name.
	 * @param name The name of the wrapper expurged of its extension
	 * @return The absolute path of the corresponding wrapper description file
	 * @throw NoWrapperFileFoundException
	 * The method uses Path class to implements its search algorithm.
	 * @see Path
	 */
	static FileName FindWrapperPathByName(const String & name)
	  /* throw(NoWrapperFileFoundException) */;

	/** @brief Build a wrapper from a name
	 * @param name The name of the wrapper expurged of its extension
	 * @return A structure describing the content of the wrapper description file
	 * @throw NoWrapperFileFoundException
	 */
	static WrapperFile FindWrapperByName(const String & name)
	  /* throw(NoWrapperFileFoundException) */;

	/** @brief Build a wrapper from a stream
	 * @param xmlStream A stream (in XML format) that is the content of a wrapper description file
	 * @return A structure describing the content of the wrapper description file
	 * @see toString()
	 */
	static WrapperFile BuildWrapperFromStream(const String & xmlStream);

      protected:

	/** The file name extension */
	static const String extension_;


      public:

	/** Default constructor */
	WrapperFile();

	/** Constructor from file */
	explicit WrapperFile(const FileName & pathToFile)
	  /* throw(WrapperFileParsingException) */;

	/* Virtual constructor */
	virtual WrapperFile * clone() const;

	/* String converter */
	virtual String __repr__() const;

	/** Description file path accessor
	 * @param path The path of the wrapper description file
	 */
	void setDescriptionFilePath(const FileName & path);
	/** Description file path accessor
	 * @return The path of the wrapper description file
	 */
	FileName getDescriptionFilePath() const;

	/** Wrapper data accessor
	 * @param data A structure describing the content of the wrapper description file
	 */
	void setWrapperData(const WrapperData & data);
	/** Wrapper data accessor
	 * @return A structure describing the content of the wrapper description file
	 */
	const WrapperData & getWrapperData() const;

	/** Write the internal data to a wrapper file
	 * @param pathToFile The path where the wrapper description file must be written
	 */
	void writeFile(const FileName & pathToFile);

	/** Stream out the internal data
	 * @return A string that can be later used by BuildWrapperFromStream()
	 */
	String toString() const;


      protected:

	/** Where the description file of the wrapper is located */
	FileName descriptionFilePath_;

	/** The data that the wrapper may exchange with the platform */
	WrapperData data_;

      private:

	/** Initialization */
	void init() const
	  /* throw(WrapperFileParsingException) */;

	/** Finalization */
	void done() const;

	/** File parsing function */
	void parseFile(const FileName & pathToFile)
	  /* throw(WrapperFileParsingException) */;

	/** Stream parsing function */
	void parseStream(const String & stream);

      } ; /* class WrapperFile */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_WRAPPERFILE_HXX */
