//                                               -*- C++ -*-
/**
 *  @file  ProductNumericalMathHessianImplementation.cxx
 *  @brief The class that implements the composition between numerical math hessians
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-05-29 16:13:02 +0200 (mar, 29 mai 2007) $
 *  Id:      $Id: ProductNumericalMathHessianImplementation.cxx 450 2007-05-29 14:13:02Z dutka $
 */
#include "ProductNumericalMathHessianImplementation.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(ProductNumericalMathHessianImplementation);

      static Common::Factory<ProductNumericalMathHessianImplementation> RegisteredFactory("ProductNumericalMathHessianImplementation");

      /* Default constructor */
      ProductNumericalMathHessianImplementation::ProductNumericalMathHessianImplementation(const EvaluationImplementation & p_leftEvaluation,
											   const GradientImplementation & p_leftGradient,
											   const HessianImplementation & p_leftHessian,
											   const EvaluationImplementation & p_rightEvaluation,
											   const GradientImplementation & p_rightGradient,
											   const HessianImplementation & p_rightHessian)
	/* throw(InvalidArgumentException) */
	: NumericalMathHessianImplementation(),
	  p_leftEvaluation_(p_leftEvaluation),
	  p_leftGradient_(p_leftGradient),
	  p_leftHessian_(p_leftHessian),
	  p_rightEvaluation_(p_rightEvaluation),
	  p_rightGradient_(p_rightGradient),
	  p_rightHessian_(p_rightHessian)
      {
	// Check the compatibility of the evaluations
	if ((p_leftEvaluation_->getOutputDimension() != 1) || (p_rightEvaluation_->getOutputDimension() != 1)) throw InvalidArgumentException(HERE) << "Error: the two functions must have an output dimension equals to 1.";
	if (p_leftEvaluation_->getInputDimension() != p_rightEvaluation_->getInputDimension()) throw InvalidArgumentException(HERE) << "Error: the two functions must have the same input dimension.";

	// Check the compatibility of the gradients
	if ((p_leftGradient_->getInputDimension()  != p_rightGradient_->getInputDimension()) || 
	    (p_leftGradient_->getOutputDimension() != p_rightGradient_->getOutputDimension()) ||
	    (p_leftGradient_->getInputDimension()  != p_leftEvaluation_->getInputDimension()) || 
	    (p_leftGradient_->getOutputDimension()  != 1)) throw InvalidArgumentException(HERE) << "Error: the gradients have incompatible dimensions.";
	// Check the compatibility of the hessians
	if ((p_leftHessian_->getInputDimension()  != p_rightHessian_->getInputDimension()) || 
	    (p_leftHessian_->getOutputDimension() != p_rightHessian_->getOutputDimension()) ||
	    (p_leftHessian_->getInputDimension()  != p_leftEvaluation_->getInputDimension()) || 
	    (p_leftHessian_->getOutputDimension()  != 1)) throw InvalidArgumentException(HERE) << "Error: the hessians have incompatible dimensions.";

      }

      /* Virtual constructor */
      ProductNumericalMathHessianImplementation * ProductNumericalMathHessianImplementation::clone() const
      {
	return new ProductNumericalMathHessianImplementation(*this);
      }

      /* Comparison operator */
      Bool ProductNumericalMathHessianImplementation::operator ==(const ProductNumericalMathHessianImplementation & other) const
      {
	return true;
      }
  
      /* String converter */
      String ProductNumericalMathHessianImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << ProductNumericalMathHessianImplementation::GetClassName()
	    << " name=" << getName()
	    << " leftEvaluation=" << p_leftEvaluation_->__repr__()
	    << " leftGradient=" << p_leftGradient_->__repr__()
	    << " leftHessian=" << p_leftHessian_->__repr__()
	    << " rightEvaluation=" << p_rightEvaluation_->__repr__()
	    << " rightGradient=" << p_rightGradient_->__repr__()
	    << " rightHessian=" << p_rightHessian_->__repr__();
	return oss;
      }
  
      /* Method hessian() returns the symmetric tensor of the second gradient of the function at point */
      ProductNumericalMathHessianImplementation::SymmetricTensor ProductNumericalMathHessianImplementation::hessian(const NumericalPoint & in) const
	/* throw(InvalidArgumentException,InternalException) */
      {
	callsNumber_++;
	UnsignedLong inputDimension(getInputDimension());
	SymmetricTensor leftHessian(p_leftHessian_->hessian(in));
	SymmetricTensor rightHessian(p_rightHessian_->hessian(in));
	Matrix leftGradient(p_leftGradient_->gradient(in));
	Matrix rightGradient(p_rightGradient_->gradient(in));
	NumericalScalar leftValue(p_leftEvaluation_->operator()(in)[0]);
	NumericalScalar rightValue(p_rightEvaluation_->operator()(in)[0]);
	// This line is quite strange. It is because the arithmetic is implemented only for matrices. As the different tensors are only 1-sheet tensors,
	// it is quite easy.
	Matrix sheet(leftHessian.getSheet(0) * rightValue + rightHessian.getSheet(0) * leftValue + leftGradient * rightGradient.transpose() + rightGradient * leftGradient.transpose());
	SymmetricTensor result(inputDimension, 1);
	for (UnsignedLong i = 0; i < inputDimension; ++i)
	  {
	    for (UnsignedLong j = i; j < inputDimension; ++j)
	      {
		result(i, j, 0) = sheet(i, j);
	      }
	  }
	return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong ProductNumericalMathHessianImplementation::getInputDimension() const
	/* throw(InternalException) */
      {
	return p_rightHessian_->getInputDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong ProductNumericalMathHessianImplementation::getOutputDimension() const
	/* throw(InternalException) */
      {
	return p_leftHessian_->getOutputDimension();
      }
     
      /* Method save() stores the object through the StorageManager */
      void ProductNumericalMathHessianImplementation::save(StorageManager::Advocate & adv) const
      {
	NumericalMathHessianImplementation::save(adv);
	adv.saveAttribute( "leftGradient_", *p_leftGradient_ );
	adv.saveAttribute( "leftHessian_", *p_leftHessian_ );
	adv.saveAttribute( "rightEvaluation_", *p_rightEvaluation_ );
	adv.saveAttribute( "rightGradient_", *p_rightGradient_ );
	adv.saveAttribute( "rightHessian_", *p_rightHessian_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void ProductNumericalMathHessianImplementation::load(StorageManager::Advocate & adv)
      {
	Common::TypedInterfaceObject<NumericalMathEvaluationImplementation> evaluationValue;
	Common::TypedInterfaceObject<NumericalMathGradientImplementation> gradientValue;
	Common::TypedInterfaceObject<NumericalMathHessianImplementation> hessianValue;
	NumericalMathHessianImplementation::load(adv);
	adv.loadAttribute( "leftGradient_", gradientValue );
	p_leftGradient_ = gradientValue.getImplementation();
	adv.loadAttribute( "leftHessian_", hessianValue );
	p_leftHessian_ = hessianValue.getImplementation();
	adv.loadAttribute( "rightEvaluation_", evaluationValue );
	p_rightEvaluation_ = evaluationValue.getImplementation();
	adv.loadAttribute( "rightGradient_", gradientValue );
	p_rightGradient_ = gradientValue.getImplementation();
	adv.loadAttribute( "rightHessian_", hessianValue );
	p_rightHessian_ = hessianValue.getImplementation();
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
