//                                               -*- C++ -*-
/**
 *  @file  NumericalMathFunctionImplementation.cxx
 *  @brief Abstract top-level class for all distributions
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: NumericalMathFunctionImplementation.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "NumericalMathFunction.hxx"
#include "NumericalMathFunctionImplementation.hxx"
#include "NoNumericalMathEvaluationImplementation.hxx"
#include "NoNumericalMathGradientImplementation.hxx"
#include "NoNumericalMathHessianImplementation.hxx"
#include "ComputedNumericalMathEvaluationImplementationFactory.hxx"
#include "ComputedNumericalMathGradientImplementationFactory.hxx"
#include "ComputedNumericalMathHessianImplementationFactory.hxx"
#include "AnalyticalNumericalMathEvaluationImplementation.hxx"
#include "ProductNumericalMathFunction.hxx"
#include "CenteredFiniteDifferenceGradient.hxx"
#include "CenteredFiniteDifferenceHessian.hxx"
#include "PersistentObjectFactory.hxx"
#include "Log.hxx"
#include "OSS.hxx"

#undef GetClassName

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      typedef Diff::CenteredFiniteDifferenceGradient  CenteredFiniteDifferenceGradient;
      typedef Diff::CenteredFiniteDifferenceHessian   CenteredFiniteDifferenceHessian;
      typedef Common::Log                             Log;

      CLASSNAMEINIT(NumericalMathFunctionImplementation);

      static Common::Factory<NumericalMathFunctionImplementation> RegisteredFactory("NumericalMathFunctionImplementation");

      // Inline documentation for analytical functions
      Bool NumericalMathFunctionImplementation::IsDocumentationInitialized_ = false;
      NumericalMathFunctionImplementation::Description NumericalMathFunctionImplementation::ValidConstants_;
      NumericalMathFunctionImplementation::Description NumericalMathFunctionImplementation::ValidFunctions_;
      NumericalMathFunctionImplementation::Description NumericalMathFunctionImplementation::ValidOperators_;

      /* Default constructor */
      NumericalMathFunctionImplementation::NumericalMathFunctionImplementation()
	: PersistentObject(),
	  p_evaluationImplementation_(new NoNumericalMathEvaluationImplementation),
	  p_gradientImplementation_(new NoNumericalMathGradientImplementation),
	  p_hessianImplementation_(new NoNumericalMathHessianImplementation),
	  p_initialEvaluationImplementation_(p_evaluationImplementation_),
	  p_initialGradientImplementation_(p_gradientImplementation_),
	  p_initialHessianImplementation_(p_hessianImplementation_),
	  useDefaultImplementationGradient_(false),
	  useDefaultImplementationHessian_(false)
      {
	// Nothing to do
      }

      /* Constructor from a wrapper name */
      NumericalMathFunctionImplementation::NumericalMathFunctionImplementation(const String & name)
	: PersistentObject(name),
	  p_evaluationImplementation_(new NoNumericalMathEvaluationImplementation),
	  p_gradientImplementation_(new NoNumericalMathGradientImplementation),
	  p_hessianImplementation_(new NoNumericalMathHessianImplementation),
	  p_initialEvaluationImplementation_(p_evaluationImplementation_),
	  p_initialGradientImplementation_(p_gradientImplementation_),
	  p_initialHessianImplementation_(p_hessianImplementation_),
	  useDefaultImplementationGradient_(false),
	  useDefaultImplementationHessian_(false)
      {
	// Read the description file of the wrapper
	WrapperFile wrapperFile = WrapperFile::FindWrapperByName( name );
	
	// We set the implementations
	initImplementations( wrapperFile );
      }

      /* Constructor from a wrapper file */
      NumericalMathFunctionImplementation::NumericalMathFunctionImplementation(const WrapperFile & wrapperFile)
	: PersistentObject(wrapperFile.getName()),
	  p_evaluationImplementation_(new NoNumericalMathEvaluationImplementation),
	  p_gradientImplementation_(new NoNumericalMathGradientImplementation),
	  p_hessianImplementation_(new NoNumericalMathHessianImplementation),
	  p_initialEvaluationImplementation_(p_evaluationImplementation_),
	  p_initialGradientImplementation_(p_gradientImplementation_),
	  p_initialHessianImplementation_(p_hessianImplementation_),
	  useDefaultImplementationGradient_(false),
	  useDefaultImplementationHessian_(false)
      {
	// We set the implementations
	initImplementations( wrapperFile );
      }


      /* This method set the implementations with the values listed in the wrapper file */
      void NumericalMathFunctionImplementation::initImplementations(const WrapperFile & wrapperFile)
      {
	const String name = wrapperFile.getName();

	ComputedNumericalMathEvaluationImplementation * implementation = new ComputedNumericalMathEvaluationImplementation( name, wrapperFile );
	p_evaluationImplementation_.reset(implementation);
	try {
	  p_gradientImplementation_.reset(new ComputedNumericalMathGradientImplementation( name, wrapperFile, implementation->getState() ));
	}
	catch (Common::Exception & ex) {
	  Log::Debug(OSS() << "" << ex.type() << " catched. Using CenteredFiniteDifferenceGradient for gradient");
	  p_gradientImplementation_.reset(new CenteredFiniteDifferenceGradient(sqrt(CenteredFiniteDifferenceGradient::DefaultEpsilon), p_evaluationImplementation_));
	  useDefaultImplementationGradient_ = true;

	}
	  
	try {
	  p_hessianImplementation_.reset(new ComputedNumericalMathHessianImplementation( name, wrapperFile, implementation->getState() ));
	}
	catch (Common::Exception & ex) {
	  Log::Debug(OSS() << "" << ex.type() << " catched. Using CenteredFiniteDifferenceHessian for hessian");
	  p_hessianImplementation_.reset(new CenteredFiniteDifferenceHessian(sqrt(CenteredFiniteDifferenceHessian::DefaultEpsilon), p_evaluationImplementation_));
	  useDefaultImplementationHessian_ = true;
	}
      }


      /* Analytical formula constructor */
      NumericalMathFunctionImplementation::NumericalMathFunctionImplementation(const Description & inputVariablesNames,
									       const Description & outputVariablesNames,
									       const Description & formulas)
	: PersistentObject(),
	  p_evaluationImplementation_(new AnalyticalNumericalMathEvaluationImplementation(inputVariablesNames, outputVariablesNames, formulas)),
	  p_gradientImplementation_(new CenteredFiniteDifferenceGradient(CenteredFiniteDifferenceGradient::DefaultEpsilon, p_evaluationImplementation_)),
	  p_hessianImplementation_(new CenteredFiniteDifferenceHessian(CenteredFiniteDifferenceHessian::DefaultEpsilon, p_evaluationImplementation_)),
	  p_initialEvaluationImplementation_(p_evaluationImplementation_),
	  p_initialGradientImplementation_(p_gradientImplementation_),
	  p_initialHessianImplementation_(p_hessianImplementation_)
      {
	// Nothing to do
      }

      /* Single function implementation constructor */
      NumericalMathFunctionImplementation::NumericalMathFunctionImplementation(const EvaluationImplementation & evaluationImplementation)
	: PersistentObject(),
	  p_evaluationImplementation_(evaluationImplementation),
	  p_gradientImplementation_(new CenteredFiniteDifferenceGradient(CenteredFiniteDifferenceGradient::DefaultEpsilon, p_evaluationImplementation_)),
	  p_hessianImplementation_(new CenteredFiniteDifferenceHessian(CenteredFiniteDifferenceHessian::DefaultEpsilon, p_evaluationImplementation_)),
	  p_initialEvaluationImplementation_(p_evaluationImplementation_),
	  p_initialGradientImplementation_(p_gradientImplementation_),
	  p_initialHessianImplementation_(p_hessianImplementation_),
	  useDefaultImplementationGradient_(true),
	  useDefaultImplementationHessian_(true)
      {
	// Nothing to do
      }

      /* Constructor from implementations */
      NumericalMathFunctionImplementation::NumericalMathFunctionImplementation(const EvaluationImplementation & evaluationImplementation,
									       const GradientImplementation & gradientImplementation,
									       const HessianImplementation  & hessianImplementation)
	: PersistentObject(),
	  p_evaluationImplementation_(evaluationImplementation),
	  p_gradientImplementation_(gradientImplementation),
	  p_hessianImplementation_(hessianImplementation),
	  p_initialEvaluationImplementation_(p_evaluationImplementation_),
	  p_initialGradientImplementation_(p_gradientImplementation_),
	  p_initialHessianImplementation_(p_hessianImplementation_),
	  useDefaultImplementationGradient_(false),
	  useDefaultImplementationHessian_(false)
      {
	// Nothing to do
      }


      /* Virtual constructor */
      NumericalMathFunctionImplementation * NumericalMathFunctionImplementation::clone() const
      {
	return new NumericalMathFunctionImplementation(*this);
      }

      /* Comparison operator */
      Bool NumericalMathFunctionImplementation::operator ==(const NumericalMathFunctionImplementation & other) const
      {
	return true;
      }
  
      /* String converter */
      String NumericalMathFunctionImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << NumericalMathFunctionImplementation::GetClassName()
	    << " name=" << getName()
            << " description=" << getDescription()
	    << " evaluationImplementation=" << p_evaluationImplementation_->__repr__()
	    << " gradientImplementation=" << p_gradientImplementation_->__repr__()
	    << " hessianImplementation=" << p_hessianImplementation_->__repr__();
	return oss;
      }
  
      /* Description Accessor */
      void NumericalMathFunctionImplementation::setDescription(const Description & description)
      {
	p_evaluationImplementation_->setDescription(description);
      }

      /* Description Accessor */
      OT::Base::Type::Description NumericalMathFunctionImplementation::getDescription() const
      {
	return p_evaluationImplementation_->getDescription();
      }

      /* Input description Accessor */
      OT::Base::Type::Description NumericalMathFunctionImplementation::getInputDescription() const
      {
	return p_evaluationImplementation_->getInputDescription();
      }

      /* Output description Accessor */
      OT::Base::Type::Description NumericalMathFunctionImplementation::getOutputDescription() const
      {
	return p_evaluationImplementation_->getOutputDescription();
      }

      /* Enable or disable the internal cache */
      void NumericalMathFunctionImplementation::enableCache() const
      {
	p_evaluationImplementation_->enableCache();
      }

      void NumericalMathFunctionImplementation::disableCache() const
      {
	p_evaluationImplementation_->disableCache();
      }
      
      Bool NumericalMathFunctionImplementation::isCacheEnabled() const
      {
	return p_evaluationImplementation_->isCacheEnabled();
      }

      /* Multiplication operator between two functions with the same input dimension and 1D output dimension */
      NumericalMathFunctionImplementation NumericalMathFunctionImplementation::operator * (const NumericalMathFunctionImplementation & right) const
      {
	return ProductNumericalMathFunction(clone(), right.clone());
      }

      /* Multiplication operator between two functions with the same input dimension and 1D output dimension */
      NumericalMathFunctionImplementation NumericalMathFunctionImplementation::operator * (const Implementation & p_right) const
      {
	return ProductNumericalMathFunction(clone(), p_right);
      }

      /* Function implementation accessors */
      void NumericalMathFunctionImplementation::setEvaluationImplementation(const EvaluationImplementation & evaluationImplementation)
      {
	p_evaluationImplementation_ = evaluationImplementation;
      }

      /* Function implementation accessors */
      const NumericalMathFunctionImplementation::EvaluationImplementation & NumericalMathFunctionImplementation::getEvaluationImplementation() const
      {
	return p_evaluationImplementation_;
      }

      
      /* Gradient implementation accessors */
      void NumericalMathFunctionImplementation::setGradientImplementation(const GradientImplementation & gradientImplementation)
      {
	p_gradientImplementation_ = gradientImplementation;
	useDefaultImplementationGradient_ = false;
      }

      /* Gradient implementation accessors */
      const NumericalMathFunctionImplementation::GradientImplementation & NumericalMathFunctionImplementation::getGradientImplementation() const
      {
	return p_gradientImplementation_;
      }

      
      /* Hessian implementation accessors */
      void NumericalMathFunctionImplementation::setHessianImplementation(const HessianImplementation & hessianImplementation)
      {
	p_hessianImplementation_ = hessianImplementation;
	useDefaultImplementationHessian_ = false;
      }

      /* Hessian implementation accessors */
      const NumericalMathFunctionImplementation::HessianImplementation & NumericalMathFunctionImplementation::getHessianImplementation() const
      {
	return p_hessianImplementation_;
      }
	
      /* Initial Function implementation accessors */
      const NumericalMathFunctionImplementation::EvaluationImplementation & NumericalMathFunctionImplementation::getInitialEvaluationImplementation() const
      {
	return p_initialEvaluationImplementation_;
      }
      void NumericalMathFunctionImplementation::setInitialEvaluationImplementation(const EvaluationImplementation & initialEvaluationImplementation)
      {
	p_evaluationImplementation_ = initialEvaluationImplementation;
      }

      /* Initial gradient implementation accessors */
      const NumericalMathFunctionImplementation::GradientImplementation & NumericalMathFunctionImplementation::getInitialGradientImplementation() const
      {
	return p_initialGradientImplementation_;
      }
      void NumericalMathFunctionImplementation::setInitialGradientImplementation(const GradientImplementation & initialGradientImplementation)
      {
	p_gradientImplementation_ = initialGradientImplementation;
      }

      /* Initial hessian implementation accessors */
      const NumericalMathFunctionImplementation::HessianImplementation & NumericalMathFunctionImplementation::getInitialHessianImplementation() const
      {
	return p_initialHessianImplementation_;
      }
      void NumericalMathFunctionImplementation::setInitialHessianImplementation(const HessianImplementation & initialHessianImplementation)
      {
	p_hessianImplementation_ = initialHessianImplementation;
      }



      /** Gradient according to the marginal parameters */
      NumericalMathFunctionImplementation::Matrix NumericalMathFunctionImplementation::parametersGradient(const NumericalPoint & in) const
      {
	return p_evaluationImplementation_->parametersGradient(in);
      }

      /** Parameters value and description accessor */
      NumericalMathFunctionImplementation::NumericalPointWithDescription NumericalMathFunctionImplementation::getParameters() const
      {
	return p_evaluationImplementation_->getParameters();
      }

      void NumericalMathFunctionImplementation::setParameters(const NumericalPointWithDescription & parameters)
      {
	p_evaluationImplementation_->setParameters(parameters);
      }

	
      /* Operator () */
      NumericalMathFunctionImplementation::NumericalPoint NumericalMathFunctionImplementation::operator() (const NumericalPoint & in) const
	/* throw(InvalidArgumentException,InternalException) */
      {
	return p_evaluationImplementation_->operator()(in);
      }

      /* Operator () */
      NumericalMathFunctionImplementation::NumericalSample NumericalMathFunctionImplementation::operator() (const NumericalSample & inSample) const
	/* throw(InvalidArgumentException,InternalException) */
      {
	return p_evaluationImplementation_->operator()(inSample);
      }

      /* Method gradient() returns the Jacobian transposed matrix of the function at point */
      NumericalMathFunctionImplementation::Matrix NumericalMathFunctionImplementation::gradient(const NumericalPoint & in) const
	/* throw(InvalidArgumentException,InternalException) */
      {
	if (useDefaultImplementationGradient_) Log::Warn(OSS() << "You are using a default implementation for the gradient. Be careful, your computation can be severely wrong!");
	return p_gradientImplementation_->gradient(in);
      }

      /* Method hessian() returns the symetric tensor of the function at point */
      NumericalMathFunctionImplementation::SymmetricTensor NumericalMathFunctionImplementation::hessian(const NumericalPoint & in) const
	/* throw(InvalidArgumentException,InternalException) */
      {
	if (useDefaultImplementationHessian_) Log::Warn(OSS() << "You are using a default implementation for the hessian. Be careful, your computation can be severely wrong!");
	return p_hessianImplementation_->hessian(in);
      }




      /* Accessor for input point dimension */
      UnsignedLong NumericalMathFunctionImplementation::getInputNumericalPointDimension() const
	/* throw(InternalException) */
      {
	return p_evaluationImplementation_->getInputDimension();
      }

      /* Accessor for output point dimension */
      UnsignedLong NumericalMathFunctionImplementation::getOutputNumericalPointDimension() const
	/* throw(InternalException) */
      {
	return p_evaluationImplementation_->getOutputDimension();

      }
     
      /* Accessor for input point dimension */
      UnsignedLong NumericalMathFunctionImplementation::getInputDimension() const
	/* throw(InternalException) */
      {
	return p_evaluationImplementation_->getInputDimension();
      }

      /* Accessor for output point dimension */
      UnsignedLong NumericalMathFunctionImplementation::getOutputDimension() const
	/* throw(InternalException) */
      {
	return p_evaluationImplementation_->getOutputDimension();

      }
     
      /* Get the i-th marginal function */
      NumericalMathFunctionImplementation::Implementation NumericalMathFunctionImplementation::getMarginal(const UnsignedLong i) const /* throw(InvalidArgumentException) */
      {
	if (i >= getOutputDimension()) throw InvalidArgumentException(HERE) << "The index of a marginal function must be in the range [0, dim-1]";
	// We build an analytical function that extract the needed component
	// If X1,...,XN are the descriptions of the input of this function, it is a function from R^n to R
	// with formula Yi = Xi
	// Build non-ambigous names for the inputs. We cannot simply use the output description, as it must be valid muParser identifiers
	UnsignedLong inputDimension(getOutputDimension());
	Description input(inputDimension);
	for (UnsignedLong index = 0; index < inputDimension; ++index)
	  {
	    input[index] = OSS() << "x" << index;
	  }
	// The description of the extracted component
	Description output(1);
	output[0] = getOutputDescription()[i];
	// Formula for the extraction
	Description formula(1);
	formula[0] = input[i];
	NumericalMathFunction left(input, output, formula);
	NumericalMathFunction right(*this);
	return new NumericalMathFunctionImplementation(*(NumericalMathFunction(left, right).getImplementation()));
      }

      /* Get the function corresponding to indices components */
      NumericalMathFunctionImplementation::Implementation NumericalMathFunctionImplementation::getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */
      {
	if (!indices.check(getOutputDimension() - 1)) throw InvalidArgumentException(HERE) << "The indices of a marginal function must be in the range [0, dim-1] and  must be different";
	// We build an analytical function that extract the needed component
	// If X1,...,XN are the descriptions of the input of this function, it is a function from R^n to R^p
	// with formula Yk = Xindices[k] for k=1,...,p
	// Build non-ambigous names for the inputs. We cannot simply use the output description, as it must be valid muParser identifiers
	UnsignedLong inputDimension(getOutputDimension());
	Description input(inputDimension);
	for (UnsignedLong index = 0; index < inputDimension; ++index)
	  {
	    input[index] = OSS() << "x" << index;
	  }
	// Extract the components
	UnsignedLong outputDimension(indices.getSize());
	Description output(outputDimension);
	Description formulas(outputDimension);
	Description currentOutputDescription(getOutputDescription());
	for (UnsignedLong index = 0; index < outputDimension; ++index)
	  {
	    output[index] = currentOutputDescription[indices[index]];
	    formulas[index] = input[indices[index]];
	  }
	NumericalMathFunction left(input, output, formulas);
	NumericalMathFunction right(*this);
	return new NumericalMathFunctionImplementation(*(NumericalMathFunction(left, right).getImplementation()));
      }

      /* Number of calls to the evaluation */
      UnsignedLong NumericalMathFunctionImplementation::getEvaluationCallsNumber() const
      {
	return p_evaluationImplementation_->getCallsNumber();
      }

      /* Number of calls to the gradient */
      UnsignedLong NumericalMathFunctionImplementation::getGradientCallsNumber() const
      {
	return p_gradientImplementation_->getCallsNumber();
      }

      /* Number of calls to the hessian */
      UnsignedLong NumericalMathFunctionImplementation::getHessianCallsNumber() const
      {
	return p_hessianImplementation_->getCallsNumber();
      }

      /* Initialization of the documentation */
      void NumericalMathFunctionImplementation::InitializeDocumentation()
      {
	if (IsDocumentationInitialized_) return;

	// First, the constants
	ValidConstants_.setName("Valid constants");
	ValidConstants_.add("_e -> Euler's constant (2.71828...)");
	ValidConstants_.add("_pi -> Pi constant (3.14159...)");

	// Second, the functions
	ValidFunctions_.setName("Valid functions");
	ValidFunctions_.add("sin(arg) -> sine function");
	ValidFunctions_.add("cos(arg) -> cosine function");
	ValidFunctions_.add("tan(arg) -> tangent function");
	ValidFunctions_.add("asin(arg) -> inverse sine function");
	ValidFunctions_.add("acos(arg) -> inverse cosine function");
	ValidFunctions_.add("atan(arg) -> inverse tangent function");
	ValidFunctions_.add("sinh(arg) -> hyperbolic sine function");
	ValidFunctions_.add("cosh(arg) -> hyperbolic cosine function");
	ValidFunctions_.add("tanh(arg) -> hyperbolic tangens function");
	ValidFunctions_.add("asinh(arg) -> inverse hyperbolic sine function");
	ValidFunctions_.add("acosh(arg) -> inverse hyperbolic cosine function");
	ValidFunctions_.add("atanh(arg) -> inverse hyperbolic tangent function");
	ValidFunctions_.add("log2(arg) -> logarithm in base 2");
	ValidFunctions_.add("log10(arg) -> logarithm in base 10");
	ValidFunctions_.add("log(arg) -> logarithm in base e (2.71828...)");
	ValidFunctions_.add("ln(arg) -> alias for log function");
	ValidFunctions_.add("lngamma(arg) -> log of the gamma function");
	ValidFunctions_.add("gamma(arg) -> gamma function");
	ValidFunctions_.add("exp(arg) -> exponential function");
	ValidFunctions_.add("erf(arg) -> error function");
	ValidFunctions_.add("erfc(arg) -> complementary error function");
	ValidFunctions_.add("sqrt(arg) -> square root function");
	ValidFunctions_.add("cbrt(arg) -> cubic root function");
	ValidFunctions_.add("besselJ0(arg) -> 1rst kind Bessel function with parameter 0");
	ValidFunctions_.add("besselJ1(arg) -> 1rst kind Bessel function with parameter 1");
	ValidFunctions_.add("besselY0(arg) -> 2nd kind Bessel function with parameter 0");
	ValidFunctions_.add("besselY1(arg) -> 2nd kind Bessel function with parameter 1");
	ValidFunctions_.add("sign(arg) -> sign function -1 if x<0; 1 if x>0");
	ValidFunctions_.add("rint(arg) -> round to nearest integer function");
	ValidFunctions_.add("abs(arg) -> absolute value function");
	ValidFunctions_.add("if(arg1, arg2, arg3) -> if arg1 then arg2 else arg3");
	ValidFunctions_.add("min(arg1, ..., argn) -> min of all arguments");
	ValidFunctions_.add("max(arg1, ..., argn) -> max of all arguments");
	ValidFunctions_.add("sum(arg1, ..., argn) -> sum of all arguments");
	ValidFunctions_.add("avg(arg1, ..., argn) -> mean value of all arguments");

	// Third, the operators
	ValidOperators_.setName("Valid operators");
	ValidOperators_.add("= -> assignement, can only be applied to variable names (priority -1)");
	ValidOperators_.add("and -> logical and (priority 1)");
	ValidOperators_.add("or -> logical or (priority 1)");
	ValidOperators_.add("xor -> logical xor (priority 1)");
	ValidOperators_.add("<= -> less or equal (priority 2)");
	ValidOperators_.add(">= -> greater or equal (priority 2)");
	ValidOperators_.add("!= -> not equal (priority 2)");
	ValidOperators_.add("== -> equal (priority 2)");
	ValidOperators_.add("> -> greater than (priority 2)");
	ValidOperators_.add("< -> less than (priority 2)");
	ValidOperators_.add("+ -> addition (priority 3)");
	ValidOperators_.add("- -> subtraction (priority 3)");
	ValidOperators_.add("* -> multiplication (priority 4)");
	ValidOperators_.add("/ -> division (priority 4)");
	ValidOperators_.add("~ -> logical negation (priority 4)");
	ValidOperators_.add("not -> alias for ~ (priority 4)");
	ValidOperators_.add("- -> sign change (priority 4)");
	ValidOperators_.add("^ -> raise x to the power of y (priority 5)");
      }

      /* Static methods for documentation of analytical fnctions */
      NumericalMathFunctionImplementation::Description NumericalMathFunctionImplementation::GetValidConstants()
      {
	if (!IsDocumentationInitialized_)
	  {
	    InitializeDocumentation();
	    IsDocumentationInitialized_ = true;
	  }
	return ValidConstants_;
      }

      NumericalMathFunctionImplementation::Description NumericalMathFunctionImplementation::GetValidFunctions()
      {
	if (!IsDocumentationInitialized_)
	  {
	    InitializeDocumentation();
	    IsDocumentationInitialized_ = true;
	  }
	return ValidFunctions_;
      }

      NumericalMathFunctionImplementation::Description NumericalMathFunctionImplementation::GetValidOperators()
      {
	if (!IsDocumentationInitialized_)
	  {
	    InitializeDocumentation();
	    IsDocumentationInitialized_ = true;
	  }
	return ValidOperators_;
      }

      /* Method save() stores the object through the StorageManager */
      void NumericalMathFunctionImplementation::save(StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
	adv.saveAttribute( "evaluationImplementation_", *p_evaluationImplementation_ );
	adv.saveAttribute( "gradientImplementation_", *p_gradientImplementation_ );
	adv.saveAttribute( "hessianImplementation_", *p_hessianImplementation_ );
	adv.saveAttribute( "initialEvaluationImplementation_", *p_initialEvaluationImplementation_ );
	adv.saveAttribute( "initialGradientImplementation_", *p_initialGradientImplementation_ );
	adv.saveAttribute( "initialHessianImplementation_", *p_initialHessianImplementation_ );
	adv.saveAttribute( "useDefaultImplementationGradient_", useDefaultImplementationGradient_ );
	adv.saveAttribute( "useDefaultImplementationHessian_", useDefaultImplementationHessian_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void NumericalMathFunctionImplementation::load(StorageManager::Advocate & adv)
      {
	Common::TypedInterfaceObject<NumericalMathEvaluationImplementation> evaluationValue;
	Common::TypedInterfaceObject<NumericalMathGradientImplementation> gradientValue;
	Common::TypedInterfaceObject<NumericalMathHessianImplementation> hessianValue;
	PersistentObject::load(adv);
	adv.loadAttribute( "evaluationImplementation_", evaluationValue );
	p_evaluationImplementation_ = evaluationValue.getImplementation();
	adv.loadAttribute( "gradientImplementation_", gradientValue );
	p_gradientImplementation_ = gradientValue.getImplementation();
	adv.loadAttribute( "hessianImplementation_", hessianValue );
	p_hessianImplementation_ = hessianValue.getImplementation();
	adv.loadAttribute( "initialEvaluationImplementation_", evaluationValue );
	p_initialEvaluationImplementation_ = evaluationValue.getImplementation();
	adv.loadAttribute( "initialGradientImplementation_", gradientValue );
	p_initialGradientImplementation_ = gradientValue.getImplementation();
	adv.loadAttribute( "initialHessianImplementation_", hessianValue );
	p_initialHessianImplementation_ = hessianValue.getImplementation();
	adv.loadAttribute( "useDefaultImplementationGradient_", useDefaultImplementationGradient_ );
	adv.loadAttribute( "useDefaultImplementationHessian_", useDefaultImplementationHessian_ );
      }
    



    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
