//                                               -*- C++ -*-
/**
 *  @file  t_RandomGenerator_std.cxx
 *  @brief The test file of class RandomGenerator for standard methods
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: t_RandomGenerator_std.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <iostream>
#include <sstream>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "RandomGenerator.hxx"
#include "NumericalPoint.hxx"
#include "DistFunc.hxx"
#include "Normal.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Distribution;

typedef RandomGenerator::UnsignedLongCollection UnsignedLongCollection;

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);

  try {
    // Test the initialization method
    RandomGenerator::SetSeed(0);
    // Test the state accessor for reading
    RandomGenerator::State initialState(RandomGenerator::GetState());
    fullprint << "state index=" << initialState.index_ << " state array=" << initialState.state_ << std::endl;
    // Test the generation of one realization
    NumericalScalar realization(RandomGenerator::Generate());
    fullprint << "one realization=" << realization << std::endl;
    RandomGenerator::State state(RandomGenerator::GetState());
    fullprint << "new state index=" << state.index_ << std::endl;
    // Test the generation of several realizations
    UnsignedLong size(10000);
    NumericalPoint several(RandomGenerator::Generate(size));
    fullprint << size << " realizations, first=" << several[0] << ", last=" << several[size  - 1] << std::endl;
    // Test the state accessor for writing
    RandomGenerator::SetState(initialState);
    fullprint << "restoring initial state, one realization=" << RandomGenerator::Generate() << std::endl;
    // Test the uniformity of the random generator
    size = 1000000;
    NumericalPoint manyPoints(RandomGenerator::Generate(size));
    UnsignedLong slice(10);
    NumericalPoint frequencies(slice, 0.0);
    NumericalScalar mean(0.0);
    for (UnsignedLong i = 0; i < size; i++)
      {
	frequencies[(UnsignedLong)(slice * manyPoints[i])]++;
	mean += manyPoints[i];
      }
    mean /= size;
    fullprint << "mean of " << size << " realizations=" << mean << std::endl;
    for (UnsignedLong i = 0; i < slice; i++)
      {
	frequencies[i] /= size;
	fullprint << "frequency in slice [" << NumericalScalar(i) / slice << ", " << NumericalScalar(i+1) / slice << "]=" << frequencies[i] << std::endl;
      }
    fullprint << "One integer generation in [[0, " << slice-1 << "]]=" << RandomGenerator::IntegerGenerate(slice) << std::endl;
    fullprint << "Another integer generation in [[0, " << slice-1 << "]]=" << RandomGenerator::IntegerGenerate(slice) << std::endl;
    UnsignedLongCollection manyUnsignedLong(RandomGenerator::IntegerGenerate(size, slice));
    frequencies = NumericalPoint(slice, 0.0);
    mean = 0.0;
    for (UnsignedLong i = 0; i < size; i++)
      {
	frequencies[manyUnsignedLong[i]]++;
	mean += manyUnsignedLong[i];
      }
    mean /= size;
    fullprint << "mean of " << size << " integer realizations=" << mean << std::endl;
    for (UnsignedLong i = 0; i < slice; i++)
      {
	frequencies[i] /= size;
	fullprint << "frequency for value " << i << "=" << frequencies[i] << std::endl;
      }
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
