//                                               -*- C++ -*-
/**
 *  @file  InverseNatafEllipticalCopulaGradient.cxx
 *  @brief Class for the InverseNataf transformation evaluation for elliptical
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: InverseNatafEllipticalCopulaGradient.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "InverseNatafEllipticalCopulaGradient.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      /*
       * @class InverseNatafEllipticalCopulaGradient
       *
       * This class offers an interface for the InverseNataf function for elliptical copula
       */

      CLASSNAMEINIT(InverseNatafEllipticalCopulaGradient);

      /* Parameter constructor */
      InverseNatafEllipticalCopulaGradient::InverseNatafEllipticalCopulaGradient(const Distribution & standardDistribution,
										 const SquareMatrix & cholesky):
	NumericalMathGradientImplementation(),
	standardDistribution_(standardDistribution),
	cholesky_(cholesky)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      InverseNatafEllipticalCopulaGradient * InverseNatafEllipticalCopulaGradient::clone() const
      {
	return new InverseNatafEllipticalCopulaGradient(*this);
      }

      /* String converter */
      String InverseNatafEllipticalCopulaGradient::__repr__() const
      {
	OSS oss;
	oss << "class=" << InverseNatafEllipticalCopulaGradient::GetClassName()
            << " standardDistribution=" << standardDistribution_
            << " cholesky=" << cholesky_;
                   
	return oss;
      }

      /*
       * Evaluation
       * The elliptical copula has a correlation matrix R. The inverse Nataf transform S reads:
       * Z(u) = L.u, where L is the Cholesky factor of R: L.L^t = R, L is lower triangular
       * Si(u) = F(Zi), where F is the CDF of the standard elliptical distribution
       * Its Jacobian J is:
       * Jij = dSi/duj = (dF/du)ij
       *               = (F'(Zi)dZ/du)ij
       *               = F'(Zi)Lij
       * thus, (DS)ij = Jji = LjiF'(Zj)
       */
      InverseNatafEllipticalCopulaGradient::Matrix InverseNatafEllipticalCopulaGradient::gradient(const NumericalPoint & in) const
	/* throw(InvalidArgumentException, InternalException) */
      {
	UnsignedLong dimension(getInputDimension());
	// First, correlate the components
	NumericalPoint point(cholesky_*in);
	Matrix result(dimension, dimension);
	Distribution standardMarginal(standardDistribution_.getMarginal(0));
	// Second, apply the commmon marginal distribution
	for (UnsignedLong j = 0; j < dimension; ++j)
	  {
	    NumericalScalar marginalPDF(standardMarginal.computePDF(NumericalPoint(1, point[j])));
	    for (UnsignedLong i = 0; i <= j; ++i)
	      {
		result(i, j) = marginalPDF * cholesky_(j, i);
	      }
	  }
	return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong InverseNatafEllipticalCopulaGradient::getInputDimension() const
	/* throw(InternalException) */
      {
	return cholesky_.getNbColumns();
      }

      /* Accessor for output point dimension */
      UnsignedLong InverseNatafEllipticalCopulaGradient::getOutputDimension() const
	/* throw(InternalException) */
      {
	return cholesky_.getNbRows();
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
