//                                               -*- C++ -*-
/**
 *  @file  NumericalPointWithDescription.cxx
 *  @brief NumericalPointWithDescription extends the classical mathematical point with a description of the coordinates
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: NumericalPointWithDescription.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "NumericalPointWithDescription.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      CLASSNAMEINIT(NumericalPointWithDescription);
      TEMPLATE_CLASSNAMEINIT(PersistentCollection<NumericalPointWithDescription>);

      static Common::Factory<NumericalPointWithDescription> RegisteredFactory1("NumericalPointWithDescription");
      static Common::Factory<PersistentCollection<NumericalPointWithDescription> > RegisteredFactory2("PersistentCollection<NumericalPointWithDescription>");

      /* Default constructor */
      NumericalPointWithDescription::NumericalPointWithDescription()
	: NumericalPoint(),
	  description_()
      {
        // Nothing to do
      }

      /* Constructor with size */
      NumericalPointWithDescription::NumericalPointWithDescription(const UnsignedLong size,
								   const NumericalScalar value)
	: NumericalPoint(size, value),
	  description_(size)
      {
        // Nothing to do
      }



      /* Constructor from a collection */
      NumericalPointWithDescription::NumericalPointWithDescription(const Type::Collection<NumericalScalar> & coll)
	: NumericalPoint(coll),
	  description_(coll.getSize())
      {
        // Nothing to do
      }

      /* Constructor from a base class */
      NumericalPointWithDescription::NumericalPointWithDescription(const Common::Pointer<NumericalPoint> & p_base)
	: NumericalPoint(*p_base),
	  description_(p_base->getSize())
      {
	// Nothing to do
      }



      /* Virtual constructor */
      NumericalPointWithDescription * NumericalPointWithDescription::clone() const
      {
	return new NumericalPointWithDescription(*this);
      }


      /* Destructor */
      NumericalPointWithDescription::~NumericalPointWithDescription() throw()
      {
	// Nothing to do
      }


      /* String converter */
      String NumericalPointWithDescription::__repr__() const
      {
	return OSS() << "class=" << NumericalPointWithDescription::GetClassName()
		     << " name=" << getName()
		     << " dimension=" << getDimension()
		     << " description=" << getDescription()
		     << " values=" << PersistentCollection<NumericalScalar>::__repr__();
      }

      class KeyValuePair : public Common::Object {
	String key_;
	NumericalScalar value_;
      public:
	KeyValuePair(const String & key = "", NumericalScalar value = 0.): key_(key), value_(value) {}
	String __repr__() const { return OSS() << key_ << " : " << value_; }
	String __str__()  const { return OSS() << key_ << " : " << value_; }
      };

      String NumericalPointWithDescription::__str__() const
      {
	const UnsignedLong size = getSize();
	const Description desc = getDescription();
	std::vector<KeyValuePair> assoc(size);
	for (UnsignedLong i=0; i<size; ++i) assoc[i] = KeyValuePair( desc[i], (*this)[i] );

	OSS oss;
	oss << "[";
	std::copy( assoc.begin(), assoc.end(), OSS_iterator<KeyValuePair>(oss, ", ") );
	oss << "]";
	return oss;
      }




      /* Description Accessor */
      void NumericalPointWithDescription::setDescription(const Description & description)
      {
	if (description.getSize() != getDimension())
	  throw Common::InvalidArgumentException(HERE) << "Description has incorrect dimension (" << description.getSize()
						       << "). Expected " << getDimension();
	description_ = description;
      }


      /* Description Accessor */
      Description NumericalPointWithDescription::getDescription() const
      {
	return description_;
      }


      

      /* Method save() stores the object through the StorageManager */
      void NumericalPointWithDescription::save(StorageManager::Advocate & adv) const
      {
	NumericalPoint::save(adv);
	adv.saveAttribute( "description_", description_ );
      }


      /* Method load() reloads the object from the StorageManager */
      void NumericalPointWithDescription::load(StorageManager::Advocate & adv)
      {
	NumericalPoint::load(adv);
	adv.loadAttribute( "description_", description_ );
      }


    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */
