//                                               -*- C++ -*-
/**
 *  @file  IdentityMatrix.cxx
 *  @brief The class IdentityMatrix implements identity matrices
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: IdentityMatrix.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "IdentityMatrix.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      CLASSNAMEINIT(IdentityMatrix);

      /* Default constructor */
      IdentityMatrix::IdentityMatrix()
	: CorrelationMatrix(0)
      {
        // Nothing to do
      }
      
      /* Constructor with size (dim, which is the same for nbRows_ and nbColumns_ )*/
      IdentityMatrix::IdentityMatrix(const UnsignedLong dim)
	: CorrelationMatrix(dim)
      {
	// Nothing to do
      }
      
      /* Constructor with implementation */
      IdentityMatrix::IdentityMatrix(const Implementation & i)
	: CorrelationMatrix(i) 
      {
	// Nothing to do
      }
      
      /* String converter */
      String IdentityMatrix::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " dimension=" << this->getDimension()
		     << " implementation=" << getImplementation()->__repr__();
      }

      /* Operator () : should do nothing, in order to prevent any modification on the matrix */
      NumericalScalar & IdentityMatrix::operator() (const UnsignedLong i,
						    const UnsignedLong j) /* throw(InvalidDimensionException) */
      {
	copyOnWrite();
        return (*getImplementation())(i, j);
      }
	
      /* Operator () gives access to the elements of the matrix (read only) */
      /* The element of the matrix is designated by its row number i and its column number j */
      const NumericalScalar & IdentityMatrix::operator() (const UnsignedLong i,
							  const UnsignedLong j)  const /* throw(InvalidDimensionException) */
      {
        return (*getImplementation())(i, j);
      }
      
      /* IdentityMatrix transpose */
      IdentityMatrix IdentityMatrix::transpose () const 
      { 
        return *this;
      }
      
      /* Multiplications */
      OT::Base::Type::Matrix IdentityMatrix::operator * (const Matrix & m) const /* throw(InvalidDimensionException) */
      {
        return m;
      }
      
      OT::Base::Type::SquareMatrix IdentityMatrix::operator * (const SquareMatrix & m) const /* throw(InvalidDimensionException) */
      {
        return m;
      }
      
      OT::Base::Type::SymmetricMatrix IdentityMatrix::operator * (const SymmetricMatrix & m) const /* throw(InvalidDimensionException) */
      {
        return m;
      }
      
      OT::Base::Stat::CovarianceMatrix IdentityMatrix::operator * (const CovarianceMatrix & m) const /* throw(InvalidDimensionException) */
      {
        return m;
      }
      
      OT::Base::Stat::CorrelationMatrix IdentityMatrix::operator * (const CorrelationMatrix & m) const /* throw(InvalidDimensionException) */
      {
        return m;
      }
      
      /* Resolution of a linear system */
      NumericalPoint IdentityMatrix::solveLinearSystem(const NumericalPoint & b) const /* throw(InvalidDimensionException) */
      {
        return b;
      }
  
      Matrix IdentityMatrix::solveLinearSystem(const Matrix & b) const /* throw(InvalidDimensionException) */
      {
        return b;
      }
  
      /* Compute determinant */
      NumericalScalar IdentityMatrix::computeDeterminant() const
      {
        return 1.0;
      }    
      
      /* Compute eigenvalues */
      NumericalPoint IdentityMatrix::computeEigenValues() const
      {
        return NumericalPoint(getNbRows(), 1.0);
      }

      /* Check if the matrix is SPD */
      Bool IdentityMatrix::isPositiveDefinite() const
      {
        return true;
      }
	
      /* Build the Cholesky factorization of the matrix */
      SquareMatrix IdentityMatrix::computeCholesky() const
      {
        return (*this);
      }
      
    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */
