//                                               -*- C++ -*-
/**
 *  @file  Log.cxx
 *  @brief Log records all user information to a file or tty
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: Log.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <iostream>
#include <cstdlib>
#include <cassert>
#include <errno.h>

#include "OTthread.hxx"
#include "OSS.hxx"
#include "Log.hxx"
#include "OTconfig.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      Log * Log::P_instance_ = 0;
      pthread_mutex_t instanceMutex = PTHREAD_MUTEX_INITIALIZER;


      const Log::Severity Log::NONE    =  0;
      const Log::Severity Log::ALL     = ~0;

      const Log::Severity Log::DBG     = 1 << 0;
      const Log::Severity Log::WRAPPER = 1 << 1;
      const Log::Severity Log::INFO    = 1 << 2;
      const Log::Severity Log::USER    = 1 << 3;
      const Log::Severity Log::WARN    = 1 << 4;
      const Log::Severity Log::ERROR   = 1 << 5;

      Log::Severity Log::Severity_ = Log::USER | Log::WARN | Log::ERROR;


      /* Constructor */
      Log::Log()
	: logName_(),
	  openturnsLogSeverityVariableName_("OPENTURNS_LOG_SEVERITY"),
	  p_file_(0),
	  previousMessage_(),
	  count_(0)
      {
	logName_[NONE]    = "   ";
	logName_[ALL]     = "ALL";

	logName_[DBG]     = "DBG";
	logName_[WRAPPER] = "WRP";
	logName_[INFO]    = "INF";
	logName_[USER]    = "USR";
	logName_[WARN]    = "WRN";
	logName_[ERROR]   = "ERR";

	initSeverityFromEnvironment();

	// Registration of destructor at exit
	std::atexit(Log::Release);
	push(Entry(INFO, "*** Log Beginning ***"));
      }


      /* Destructor */
      Log::~Log()
      {
	push(Entry(INFO, "*** Log End ***"));

	delete p_file_;
	p_file_ = 0;
      }


      /* Set Severity according to Openturns LogSeverity Variable */
      void Log::initSeverityFromEnvironment()
      {
        const char * logSeverityVariableContent = getenv(openturnsLogSeverityVariableName_);
        if (logSeverityVariableContent != NULL) {
          String severityVariableContent(logSeverityVariableContent);
          Severity_ = Log::NONE;

          const char delim = ',';
          IndexType begPos = 0, endPos;
          do {
            // search token
            endPos = severityVariableContent.find(delim, begPos);
            if (endPos == static_cast<IndexType>(String::npos))
              endPos = severityVariableContent.size();
            const String token(severityVariableContent.substr(begPos, endPos - begPos));

            // add severity
            std::map<Severity, String>::iterator iter;
            for (iter = logName_.begin(); iter != logName_.end(); ++iter)
              if (token == iter->second)
                Severity_ |= iter->first;

            // next token
            begPos = endPos + 1;
          } while (endPos != static_cast<IndexType>(severityVariableContent.size()));
        }
      }


      Log & Log::GetInstance()
      {
	if (!P_instance_) P_instance_ = new Log;
	// std::cerr << "Log::GetInstance() P_instance_ = " << P_instance_ << std::endl;
	return *P_instance_;
      }



      void Log::Release()
      {
	delete P_instance_;
	P_instance_ = 0;
      }



      /* Log messages according to its relative severity */
      void Log::Debug(const String & msg)
      {
	pthread_mutex_lock( &instanceMutex );
	Log::GetInstance().push(Entry(DBG, msg));
	pthread_mutex_unlock( &instanceMutex );
      }


      
      /* Log messages according to its relative severity */
      void Log::Wrapper(const String & msg)
      {
	pthread_mutex_lock( &instanceMutex );
	Log::GetInstance().push(Entry(WRAPPER, msg));
	pthread_mutex_unlock( &instanceMutex );
      }


      
      /*  Log messages according to its relative severity */
      void Log::Info(const String & msg)
      {
	pthread_mutex_lock( &instanceMutex );
	Log::GetInstance().push(Entry(INFO, msg));
	pthread_mutex_unlock( &instanceMutex );
      }



      /*  Log messages according to its relative severity */
      void Log::User(const String & msg)
      {
	pthread_mutex_lock( &instanceMutex );
	Log::GetInstance().push(Entry(USER, msg));
	pthread_mutex_unlock( &instanceMutex );
      }



      /* Log messages according to its relative severity */
      void Log::Warn(const String & msg)
      {
	pthread_mutex_lock( &instanceMutex );
	Log::GetInstance().push(Entry(WARN, msg));
	pthread_mutex_unlock( &instanceMutex );
      }



      /* Log messages according to its relative severity */
      void Log::Error(const String & msg)
      {
	pthread_mutex_lock( &instanceMutex );
	Log::GetInstance().push(Entry(ERROR, msg));
	pthread_mutex_unlock( &instanceMutex );
      }


      /* Get/Set the severity flags for the messages logged to the file */
      void Log::Show(Severity flags)
      {
	Log::Severity_ = flags;
      }

      Log::Severity Log::Flags()
      {
	return Log::Severity_;
      }

      /* Flush pending messages */
      void Log::Flush()
      {
	Log::GetInstance().flush();
      }


      void Log::flush()
      {
	printRepeatedMessage( previousMessage_ );
	previousMessage_ = Entry();
	count_ = 0;
      }


      /* Append an entry at the end of the list */
      void Log::push(const Entry & entry)
      {
	std::ostream & os = p_file_ ? *p_file_ : std::clog;
	if (entry.sev_ & Log::Severity_) {
	  if (entry == previousMessage_) ++count_;
	  else {
	    printRepeatedMessage( previousMessage_ );
	    previousMessage_ = entry ;
	    count_ = 0;
	    os << logName_[entry.sev_] << " - " << entry.msg_ << std::endl;
	  }
	}
      }

      void Log::printRepeatedMessage(const Entry & entry)
      {
	std::ostream & os = p_file_ ? *p_file_ : std::clog;
	if (count_ > 0)
	  os << logName_[entry.sev_] << " - (previous message repeated " << count_ << " time" << ((count_ == 1) ? "" : "s") << ")" << std::endl;
      }

      /* Set the name of the log file */
      void Log::SetFile(const FileName & file)
      {
	Log::GetInstance().setFile(file);
      }


      /* Set the name of the log file */
      void Log::setFile(const FileName & file)
      {
	push(Entry(INFO, OSS() << "Diverting log to file: " << file));
	push(Entry(INFO, "*** Log End ***"));
	delete p_file_;
	p_file_ = new std::ofstream(file.c_str());

	push(Entry(INFO, "*** Log Beginning ***"));
      }




    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */
