/* -*-c++-*- Producer - Copyright (C) 2001-2004  Don Burns
 *
 * This library is open source and may be redistributed and/or modified under
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * OpenSceneGraph Public License for more details.
 */


#include <Producer/Export>
#include <Producer/Types>
#include <Producer/Referenced>
#include <Producer/KeyboardMouse>
#include <Producer/RenderSurface>
#include <Producer/Events>
#include <Producer/Math>

#ifndef WIN32
#include <unistd.h>
#include <sys/time.h>
#include <signal.h>
#endif

#include <iostream>
#include <stdio.h>
#include <vector>
#include <map>
#include <algorithm>

#include <float.h>

#define TIMER_ID        555

namespace Producer {

class KeyboardMouseImplementationBase //: public Referenced
{
    public :
        KeyboardMouseImplementationBase(RenderSurface *rs): 
            _rs(rs), 
            _inputArea(NULL),
            _cb(NULL),
            _canceled(false)
        {
            _rs->setUseDefaultEsc(false);
        }

        KeyboardMouseImplementationBase(InputArea *inputArea): 
            _rs(NULL), 
            _inputArea(inputArea),
            _cb(NULL),
            _canceled(false)
        {}
        float mx() { return _mx; }
        float my() { return _my; }
        unsigned int mbutton() { return _mbutton; }

        virtual bool update(KeyboardMouseCallback &, bool ) = 0;
        virtual void startTimer() = 0;
        void setCallback(KeyboardMouseCallback *cb) { _cb = cb; }

#if 0
        void *run()
        {
            if( _cb == NULL ) return (void *)1L;
            /*
            bool use_timer = _cb->idle();
            if( use_timer )
                startTimer();
              */
            while( !isCanceled() )
            {
                update(*_cb,true);
            }
            return 0L;
        }
#endif

        void cancel(){
            _canceled = true;
        }

        virtual void positionPointer( float x, float y )
        {
            if( _inputArea != NULL )
            {
                float minL = FLT_MAX;
                int bestInputRectangle = 0;
                for(int i=0;i<_inputArea->getNumRenderSurfaces();++i)
                {
                    RenderSurface::InputRectangle ir = _inputArea->getRenderSurface(i)->getInputRectangle();
                    float dx = (x-(ir.left()+ir.width()*0.5f));
                    float dy = (y-(ir.bottom()+ir.height()*0.5f));
                    float dl = sqrt(dx*dx+dy*dy);
                    if (dl<minL)
                    {
                        minL = dl;
                        bestInputRectangle = i;
                    }
                }
                
                RenderSurface* rs = _inputArea->getRenderSurface(bestInputRectangle);
                RenderSurface::InputRectangle ir = rs->getInputRectangle();

                int wx, wy;
                unsigned int w, h;
                rs->getWindowRectangle( wx, wy, w, h );
                
                int ix = int( (float)w*(x - ir.left())/ir.width() );
                int iy = int( (float)h*(y - ir.bottom())/ir.height() );

                rs->positionPointer(ix, iy );

            }
            else
            {
                /*
                int wx, wy;
                unsigned int w, h;
                _rs->getWindowRectangle( wx, wy, w, h );
                int ix = int((float)w*(x*0.5f+0.5f)); // assume x is in 0.0->1.0 range
                int iy = int((float)h*(y*0.5f+0.5f)); // assume y is in 0.0->1.0 range
                */

                RenderSurface::InputRectangle ir = _rs->getInputRectangle();
                unsigned int ww = _rs->getWindowWidth();
                unsigned int wh = _rs->getWindowHeight();
                int ix = int(((x - ir.left())/ir.width()) * float(ww));
                int iy = int(((y - ir.bottom())/ir.height()) * float(wh));
                _rs->positionPointer(ix, iy );
            }
        }

        bool isCanceled(){
            return _canceled;
        }

    protected:
        virtual ~KeyboardMouseImplementationBase() {}

        RenderSurface *_rs;
        InputArea *_inputArea;
        KeyboardMouseCallback *_cb;

        float _mx, _my;
        unsigned int _mbutton;

        virtual void init() = 0;
        virtual void fini() = 0;

        void transformMouseMotion( Window win, int wx, int wy )
        {
            if( _inputArea == NULL )
            {
                RenderSurface::InputRectangle ir = _rs->getInputRectangle();
                unsigned int ww = _rs->getWindowWidth();
                unsigned int wh = _rs->getWindowHeight();
                _mx = ir.left() + (float(wx)/float(ww))*ir.width();
                _my = ir.bottom() + (float(int(wh) - wy)/float(wh))*ir.height();
            }
            else
                _inputArea->transformMouseMotion( win, wx, wy, _mx, _my );
        }

        bool _canceled;

};

}

#ifdef _X11_IMPLEMENTATION

#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/Xmu/WinUtil.h>


namespace Producer {

class KeyboardMouseImplementation : public KeyboardMouseImplementationBase
{
    public:
        KeyboardMouseImplementation(InputArea *ia):KeyboardMouseImplementationBase(ia) 
        { init(); }

        KeyboardMouseImplementation(RenderSurface *rs):KeyboardMouseImplementationBase(rs) 
        { init(); }


        bool update( KeyboardMouseCallback &cb, bool block=false)
        {
            if( block )
            {
#if 0
                XSync(dpy,0);
                if( !XPending(dpy) )
                {
                    fd_set rfds;
                    int fd = ConnectionNumber(dpy);
                    FD_ZERO(&rfds);
                    FD_SET( fd, &rfds );
                    if( select(fd+1,&rfds,0,0,0) < 0 )
                        return false;
                }
#endif
                do{ 
                    process_event(cb);
                }while( XPending(dpy));
            }
            else
            {
                while( XPending(dpy))
                    process_event(cb);
            }
            cb.idle();

            return true;
        }

        void startTimer()
        {
            struct sigaction act;
            sigset_t ss;
            sigemptyset(&ss);
            act.sa_handler = 0L;
            act.sa_sigaction = s_trap;
            act.sa_mask = ss;
            act.sa_flags = SA_SIGINFO;
            sigaction( SIGALRM, &act, 0L );

            struct itimerval itv;
            itv.it_interval.tv_sec  = 0;
            itv.it_interval.tv_usec = 10000;
            itv.it_value = itv.it_interval;
            setitimer(ITIMER_REAL, &itv, (struct itimerval *)0 );
        }

    private:
        Display *dpy;
        long _inputMask;

        virtual void init()    
        {
        // This implementation needs to be made more robust for 
        // use with multiple servers and multiple hosts.  Currently
        // assume multiple screens but only one server and host.
            char dpyname[128];
            if( _inputArea != NULL )
                _rs = _inputArea->getRenderSurface(0);

            std::string hostname = _rs->getHostName();
            if( hostname.empty() )
                sprintf( dpyname, ":%d.%d",  _rs->getDisplayNum(), _rs->getScreenNum() );
            else
                sprintf( dpyname, "%s:%d.%d",  
                        hostname.c_str(), 
                        _rs->getDisplayNum(), _rs->getScreenNum() );
        
            if((dpy = XOpenDisplay(dpyname)) == NULL)
            {
                std::cerr << 
                    "KeyboardMouse() unable to open display " << 
                    XDisplayName("") << std::endl;
                return;
            }
            
            _inputMask = StructureNotifyMask |
                         KeyPressMask | KeyReleaseMask |
                         PointerMotionMask  | ButtonPressMask | ButtonReleaseMask ;
                         //ButtonMotionMask  | ButtonPressMask | ButtonReleaseMask ;

            if( _inputArea != NULL )
            {
                 for( int i = 0; i < _inputArea->getNumWindows(); i++ )
                 {
                     Window win = _inputArea->getWindow(i);
                     XSelectInput( dpy, win,  _inputMask );
                }
            }
            else
            {
                Window win = _rs->getWindow();
                XSelectInput( dpy, win, _inputMask );
            }
            _mbutton = 0;
        }

        void process_event( KeyboardMouseCallback &cb)
        {
            XEvent ev;
            XNextEvent( dpy, &ev );
            switch( ev.type )
            {
                case DestroyNotify:
                    cb.shutdown();
                    break;

                case ConfigureNotify:
                    break;
    
                case MotionNotify :
                  {
                    int  wx, wy;
                    Window win = 0L;
                    if( ev.xmotion.same_screen )
                    {
                        wx = ev.xmotion.x;
                        wy = ev.xmotion.y;
                        win = ev.xmotion.window;
                    }
                    else
                    {
                        Window root;
                        int rx, ry;
                        unsigned int buttons;
                        Window parent = 0L;

                        for( int i= 0; i < ScreenCount(dpy); i++ )
                        {
                            if( XQueryPointer( dpy, RootWindow(dpy, i),
                                  &root, &win, &rx, &ry, &wx, &wy, &buttons) )
                            {
                                parent = win;
                                win = XmuClientWindow( dpy, parent );
                                break;
                            }
                        }
                        XWindowAttributes watt;
                        XGetWindowAttributes(dpy,parent,&watt);
                        wx -= watt.x;
                        wy -= watt.y;
                    }

                    transformMouseMotion(win, wx, wy );

                    if( _mbutton )
                        cb.mouseMotion(_mx, _my);
                    else
                        cb.passiveMouseMotion(_mx, _my);
                    }
                    break;
    
                case ButtonPress :
                    if( ev.xbutton.button == Button4 )
                        cb.mouseScroll(KeyboardMouseCallback::ScrollUp);
                    else if( ev.xbutton.button == Button5)
                        cb.mouseScroll(KeyboardMouseCallback::ScrollDown);
                    else
                    {
                        transformMouseMotion( ev.xmotion.window, ev.xbutton.x, ev.xmotion.y );
                        cb.buttonPress( _mx, _my, ev.xbutton.button );
                        _mbutton |= (1<<(ev.xbutton.button-1));
                    }
                    break;
    
                case ButtonRelease :
                    if( ev.xbutton.button != Button4 && ev.xbutton.button != Button5 )
                    {
                        transformMouseMotion( ev.xmotion.window, ev.xbutton.x, ev.xmotion.y);
                        cb.buttonRelease( _mx, _my, ev.xbutton.button );
                        _mbutton &= ~(1<<(ev.xbutton.button-1));
                    }
                    break;
    
                case KeyPress:
                    {
                        KeyCharacter keychar = Keyboard::getSingleton()->xkeyEventToKeyChar(dpy, ev.xkey);
                        if( keychar != KeyChar_Unknown )
                        {
                            if( keychar & 0xFF00 )
                                cb.specialKeyPress(keychar);
                            else
                                cb.keyPress(keychar);
                        }
                    }
                    break;
    
                case KeyRelease:
                    {
                        KeyCharacter keychar = Keyboard::getSingleton()->xkeyEventToKeyChar(dpy, ev.xkey);
                        if( keychar != KeyChar_Unknown )
                        {
                        if( keychar & 0xFF00 )
                            cb.specialKeyRelease(keychar);
                        else
                            cb.keyRelease(keychar);
                        }
                    }
                    break;
            }
        }

        virtual void fini()
        {
            XCloseDisplay(dpy);
        }
        static void s_trap( int, siginfo_t *, void *) {}
};

}
#endif

#ifdef _WIN32_IMPLEMENTATION

namespace Producer {

class KeyboardMouseImplementation : public KeyboardMouseImplementationBase
{
    public:
        KeyboardMouseImplementation(InputArea *ia):KeyboardMouseImplementationBase(ia)
        { init(); }
            
        KeyboardMouseImplementation(RenderSurface *rs):KeyboardMouseImplementationBase(rs) 
        { init(); }

        virtual ~KeyboardMouseImplementation() { fini(); }

        bool update( KeyboardMouseCallback &cb, bool block=false)
        {/* no blocking for now
            if( block ) 
            {
                q->waitWhileEmpty();
            }
            */
            std::vector< Producer::ref_ptr<EventQueue> >::iterator p;
            for( p = qs.begin(); p != qs.end(); p++ )
            {
                //HACK!!!
                Sleep(16);
                EventQueue *q = (*p).get();
                while( ! q->empty() )
                {
                    Event *ev = q->front().get();
                    switch( ev->type() )
                    {
                    case Event::WindowConfig :
                      {
                        WindowConfigEvent *wc = dynamic_cast<WindowConfigEvent *>(ev);
                        if( wc != NULL )
                        {
                            // NOt using these now...
                        }
                      }
                      break;
        
                    case Event::KeyPress :
                      {
                        KeyPressEvent *kp = dynamic_cast<KeyPressEvent *>(ev);
                        if( kp != NULL )
                        {
                            KeyCharacter kchar = KeyCharacter(kp->key());
                            if( kchar & 0xFF00 )
                                cb.specialKeyPress( kchar );
                            else
                                cb.keyPress( kchar );
                        }
                      }
                      break;

                    case Event::KeyRelease :
                      {
                        KeyReleaseEvent *kr = dynamic_cast<KeyReleaseEvent *>(ev);
                        if( kr != NULL )
                        {
                            KeyCharacter kchar = KeyCharacter(kr->key());
                            if( kchar & 0xFF00 )
                                cb.specialKeyRelease(kchar);
                            else
                                cb.keyRelease(kchar);
                        }
                      }
                      break;

                    case Event::ButtonPress :
                      {
                        ButtonPressEvent *bp = dynamic_cast<ButtonPressEvent *>(ev);
                        transformMouseMotion( bp->window(), bp->mx(), bp->my() );
                        _mbutton |= (1<<(bp->button()-1));

                        if( bp != NULL )
                            cb.buttonPress( _mx, _my, bp->button() );
                      }
                      break;

                    case Event::ButtonRelease :
                      {
                        ButtonReleaseEvent *br = dynamic_cast<ButtonReleaseEvent *>(ev);
                        if( br != NULL )
                        {
                            transformMouseMotion( br->window(), br->mx(), br->my() );
                            _mbutton &= ~(1<<(br->button()-1));

                            cb.buttonRelease( _mx, _my, br->button() );
                        }
                      }
                      break;

                    case Event::MouseScroll:
                        {
                            MouseScrollEvent *mse = dynamic_cast<MouseScrollEvent *>(ev);
                            if( mse != NULL )
                            {
                                cb.mouseScroll(mse->scrollingMotion() );
                            }
                        }
                        break;

                    case Event::MouseMotion :
                      {
                        MouseMotionEvent *mm = dynamic_cast<MouseMotionEvent *>(ev);
                        transformMouseMotion( mm->window(), mm->mx(), mm->my() );
                        if( mm != NULL )
                        {
                            if( _mbutton )
                                cb.mouseMotion(_mx, _my);
                            else
                                cb.passiveMouseMotion(_mx, _my);
                         }
                      }
                      break;
                      
                    case Event::Timer :
                          // Nothing to do, really.
                        break;

                    case Event::Shutdown:
                        cb.shutdown();
                        break;

                    case Event::DoubleButtonPress :
                    case Event::ButtonMotion :
                        break;
                        
                    }
                    q->pop_front();
                }
                cb.idle();
            }    
            return true;
        }

        virtual void startTimer() 
        {
            ::SetTimer(_rs->getWindow(),TIMER_ID, 10, NULL );
        }


    private:

        virtual void init()    
        {
            unsigned int mask = 
                        Event::KeyPressMask   |
                        Event::KeyReleaseMask |
                        Event::ButtonPressMask |
                        Event::ButtonReleaseMask |
                        Event::ButtonMotionMask |
                        Event::MouseMotionMask|
                        Event::MouseScrollMask |
                        Event::WindowConfigMask|
                        Event::ShutdownMask;

            if( _inputArea != NULL )
            {
            
                for( int i = 0; i < _inputArea->getNumRenderSurfaces(); i++ )
                {
                    RenderSurface *rs = _inputArea->getRenderSurface(i);
                    EventQueue *q = rs->selectInput(mask);
                    qs.push_back(q);
                }
            }
            else
            {
                EventQueue *q = _rs->selectInput(mask);
                qs.push_back(q);
            }
               
            _mbutton = 0;
        }

        virtual void fini()
        {
            cancel();
        }

        std::vector< Producer::ref_ptr<EventQueue> > qs;
};

}

#endif


#ifdef _OSX_CGL_IMPLEMENTATION

namespace Producer {

class KeyboardMouseImplementation : public KeyboardMouseImplementationBase
{
    public:
        KeyboardMouseImplementation(InputArea *ia):KeyboardMouseImplementationBase(ia)
        { init(); }
            
        KeyboardMouseImplementation(RenderSurface *rs):KeyboardMouseImplementationBase(rs) 
        { init(); }

        virtual ~KeyboardMouseImplementation() { fini(); }

        bool update( KeyboardMouseCallback &cb, bool block=false)
		{
    		EventRecord event;
			bool res;
			while( EventAvail( everyEvent, &event ) )
			{
    		if( (res = WaitNextEvent( everyEvent, &event, 0, NULL )))                                                 
    		{   
        		switch( event.what )
        		{
            		case osEvt : 
                		break;                                                                                
        
            		case updateEvt:
                		break;
    
            		case kHighLevelEvent:                                                                     
                		AEProcessAppleEvent(&event);
                		break;
    
            		case keyDown:
						{
							KeyCharacter keychar = Keyboard::getSingleton()->akeyEventToKeyChar(event.message & charCodeMask);
                            if( keychar & 0xFF00 )
                                cb.specialKeyPress(keychar);
                            else
                                cb.keyPress(keychar);
						}
						break;

            		case keyUp:                                                                               
						{
							KeyCharacter keychar = Keyboard::getSingleton()->akeyEventToKeyChar(event.message & charCodeMask);
                            if( keychar & 0xFF00 )
                                cb.specialKeyRelease(keychar);
                            else
                                cb.keyRelease(keychar);
						}
                		break;                                                                                
                                                                                                      
            		case mouseDown:                                                                           
						{
							unsigned int ui = *(unsigned int *)&event.where;
							short x = (ui&0x0000FFFF);
							short y = (ui&0xFFFF0000) >> 16;
							transformMouseMotion( 0L, x,y);
							unsigned int b = 1;
							if( event.modifiers & 0xF000 )
								b = event.modifiers & 0x1000 ? 3 : 1;

							if( event.modifiers & 0xF00 )
								b = event.modifiers & 0x100 ? 2 :
									event.modifiers & 0x800 ? 3 : 1;

//printf( "Mouse Down: button %d (modifiers = 0x%x)\n", b, event.modifiers );

							_lastButtonPressed = b;
							_mbutton |= (1<<(b-1));
							cb.buttonPress(_mx,_my, b);
						}
                		break;                                                                                

            		case mouseUp:                                                                           
						{
							unsigned int ui = *(unsigned int *)&event.where;
							short x = (ui&0x0000FFFF);
							short y = (ui&0xFFFF0000) >> 16;
							transformMouseMotion( 0L, x,y);

							/*
							unsigned int b = 1;
							if( event.modifiers & 0xF000 )
								b = event.modifiers & 0x1000 ? 3 : 1;

							if( event.modifiers & 0xF00 )
								b = event.modifiers & 0x100 ? 2 :
									event.modifiers & 0x800 ? 3 : 1;
							
printf( "Mouse Up: button %d (modifiers = 0x%x)\n", b, event.modifiers );

							_mbutton &= ~(1<<(b-1));
							cb.buttonRelease(_mx,_my,b);
							*/
							// Ugh.... CGL does not provide a hint about
							// which button was released.  So... we hope
							// it was the last button pressed
							cb.buttonRelease(_mx,_my,_lastButtonPressed);
						}
						break;
                                                                                                      
            		default :                                                                                 
                		break;                                                                                
        		}                                                                                             
    		}                                                                                                 
			}
			{
				Point pt;
				GetMouse(&pt);
				unsigned int ui = *(unsigned int *)&event.where;
				short x = (ui&0x0000FFFF);
				short y = (ui&0xFFFF0000) >> 16;
				if( _ox != x || _oy != y )
				{
					_ox = x;
					_oy = y;
					transformMouseMotion(0, x, y);
                    if( _mbutton )
                          cb.mouseMotion(_mx, _my);
                      else
                          cb.passiveMouseMotion(_mx, _my);
				}
			}

//usleep(16000);

			return true;
		}

        virtual void startTimer() {}


    private:

		short _ox, _oy;
		int _lastButtonPressed;

        virtual void init()    
        {
			_lastButtonPressed = 0;
        }

        virtual void fini()
        {
        }

};

}

#endif


using namespace Producer;

KeyboardMouse::KeyboardMouse(Producer::RenderSurface *rs) : 
    _implementation(NULL),
    _initialized(false)
{
    _rs = rs;
    //std::cout << "Constructing Threaded KeyboardMouse "<<this<<std::endl;
}

KeyboardMouse::KeyboardMouse(Producer::InputArea *inputArea) : 
    _implementation(NULL),
    _initialized(false) 
{
    _inputArea = inputArea;
    //std::cout << "Constructing Threaded KeyboardMouse "<<this<<std::endl;
} 

KeyboardMouse::~KeyboardMouse()
{
    if( /*threadI*/isRunning() )
    {
        if( _implementation != NULL )
            _implementation->cancel();
    }
    
    cancel();

    while (isRunning())
    {
        //std::cout << "waiting for keyboard mouse to cancel"<<std::endl;
        OpenThreads::Thread::YieldCurrentThread();
    }

    if( _implementation != NULL )
        delete _implementation;
}

bool KeyboardMouse::init( void )
{
    if( _initialized ) return _initialized;

    if( _inputArea != NULL )
    {
        _inputArea->getRenderSurface(0)->waitForRealize();
        if( _inputArea->isRealized() == false ) return false;
            _implementation = new KeyboardMouseImplementation(_inputArea.get() );
    }
    else
    {
        _rs->waitForRealize();
        if( _rs->isRealized() == false ) return false;
            _implementation = new KeyboardMouseImplementation(_rs.get() );
    }

    _initialized = true;
    return _initialized;
}

void KeyboardMouse::update(KeyboardMouseCallback &cb, bool block)
{
    if( !_initialized && !init() ) 
        return;
    _implementation->update(cb, block);
}

void KeyboardMouse::setCallback( KeyboardMouseCallback *cb )
{
    _cb = cb;
}

void KeyboardMouse::positionPointer( float x, float y )
{
    if( !_initialized && !init() ) 
            return;
    _implementation->positionPointer( x, y );
}

bool KeyboardMouse::computePixelCoords(float x, float y, RenderSurface* rs, float& pixel_x, float& pixel_y)
{
    if( _inputArea != NULL )
    {
        // first locate which InputRectange is appropriate for specified RenderSurface.
        int i;
        for(i=0;i<_inputArea->getNumRenderSurfaces();++i)
        {
            if (_inputArea->getRenderSurface(i)==rs) break;
        }
        
        // the RenderSurface isn't in the InputArea list.. therefore cannot make mouse coords to it.
        if (i==_inputArea->getNumRenderSurfaces()) return false;

        // we have a valid InputRectangle
        RenderSurface::InputRectangle ir = _inputArea->getRenderSurface(i)->getInputRectangle();

        // clip out coords which arn't in this InputRectangle.
        if (x<ir.left()) return false;
        if (x>(ir.left()+ir.width())) return false;
        if (y<ir.bottom()) return false;
        if (y>(ir.bottom()+ir.height())) return false;

        float rx = (x-ir.left())/ir.width();
        float ry = (y-ir.bottom())/ir.height();
        
        int wx, wy;
        unsigned int w, h;
        rs->getWindowRectangle( wx, wy, w, h );

        pixel_x = (float)wx + ((float)w)* rx;
        pixel_y = (float)wy + ((float)h)* ry;

        return true;

    }
    else
    {
        // clip out coords that arn't on RenderSurface.
        if (x<-1.0f) return false;
        if (x>1.0f) return false;

        if (y<-1.0f) return false;
        if (y>1.0f) return false;

        float rx = (x+1.0f)*0.5f;
        float ry = (y+1.0f)*0.5f;

        int wx, wy;
        unsigned int w, h;
        _rs->getWindowRectangle( wx, wy, w, h );

        pixel_x = (float)wx + ((float)w)* rx;
        pixel_y = (float)wy + ((float)h)* ry;

        return true;
        
    }
}

void KeyboardMouse::run()
{
    if( _cb == NULL )
    {
        std::cerr << "KeyboardMouse: internal thread cannot start because\n"
                     "no callback has been specified.  Please specify a \n"
                     "callback with KeyboardMouse::setCallback() first.\n";
        //return (void *)1L;
        return;
    }

    if( !_initialized  )
    {
        // Init() will fail if the render surface
        // or input Area is not realized.
        // Check every 5th of a second until it
        // is realized.
            
        // April 1, 2003 DB - This loop should no 
        // longer happen.  Init should succeed the
            // first time since the mplementation of
            // waitForRealize() in RenderSurface
        while( !init() )
#ifdef WIN32 
            Sleep(200);
#else
            usleep(200000);
#endif
    }

    _implementation->setCallback( _cb.get() );

    //_implementation->run();


     /*
     bool use_timer = _cb->idle();
     if( use_timer )
             startTimer();
    */
    while( !_implementation->isCanceled() )
    {
        _implementation->update(*_cb,true);
        testCancel();
    }
}


