# Copyright (C) 2000-2001 The OpenRPG Project
#
#     openrpg-dev@lists.sourceforge.net
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# --
#
# File: chatutils.py
# Author: Chris Davis
# Maintainer:
# Version:
#   $Id: chatwnd.py,v 1.101 2005/11/07 03:20:44 mduo13 Exp $
#
# Description: This file contains some of the basic definitions for the chat
# utilities in the orpg project.
#
# History
# 2002-01-20 HeroMan
#   + Added 4 dialog items on toolbar in support of Alias Library Functionallity
#   + Shrunk the text view button to an image
# 2005-04-25 Snowdog
#   + Added simple_html_repair() to post() to fix malformed html in the chat window
#   + Added strip_script_tags() to post() to remove crash point. See chat_util.py
# 2005-04-25 Snowdog
#   + Added simple_html_repair() to post() to fix malformed html in the chat window
#

__version__ = "$Id: chatwnd.py,v 1.101 2005/11/07 03:20:44 mduo13 Exp $"


##
## Module Loading
##
from orpg.orpg_windows import *
import orpg.dirpath
import orpg.tools.rgbhex
import orpg.tools.inputValidator
import orpg.tools.orpg_sound
import webbrowser
from string import *
from orpg.orpg_version import VERSION
import orpg.tools.orpg_update
import commands
import chat_msg
import time   #  needed for detecting when the user stops typing.
#  imports the predicting text control to use in place of wxTextCtrl in chattxt
import orpg.tools.predTextCtrl
from orpg.networking.mplay_client import MPLAY_CONNECTED    #  needed to only send typing/not_typing messages while connected
import os
import time
import re
import sys
import cStringIO # for reading inline imagedata as a stream
from HTMLParser import HTMLParser
import orpg.plugins#mDuo13 added
import chat_util
import traceback

# Global parser for stripping HTML tags:
# The 'tag stripping' is implicit, because this parser echoes every
# type of html data *except* the tags.
class HTMLStripper(HTMLParser):
  def __init__(self):
    self.accum = ""
  def handle_data(self, data):  # quote cdata literally
    self.accum += data
  def handle_entityref(self, name): # entities must be preserved exactly
    self.accum += "&" + name
  def handle_charref(self, name):  # charrefs too
    self.accum += "&#" + name + ";"

htmlstripper = HTMLStripper()

# utility function;  see Post().
def strip_html(string):
  "Return string tripped of html tags."
  htmlstripper.reset()
  htmlstripper.accum = ""
  htmlstripper.feed(string)
  htmlstripper.close()
  return htmlstripper.accum





def log( settings, text ):
    filename = settings.get_setting('GameLogPrefix')
    if filename > '' and filename[0] != commands.ANTI_LOG_CHAR:
        filename = filename + time.strftime( '-%Y-%m-%d.html', time.localtime( time.time() ) )
        #filename = time.strftime( filename, time.localtime( time.time() ) )
        timestamp = time.ctime(time.time())
        header = '[%s] : ' % ( timestamp );
        if settings.get_setting('TimeStampGameLog') != '1':
          header = ''
        f = open( orpg.dirpath.dir_struct["user"]+filename, 'a' )
        f.write( '%s%s<br>\n' % ( header, text ) )
        f.close()

########################
## timer
##
##  Stolen from main.py to use for typing pause detection.
########################

class mytimer(wxTimer):
    def __init__(self,func):
        self.func = func
        wxTimer.__init__(self)

    def Notify(self):
        self.func()

# This class displayes the chat information in html?
#
# Defines:
#   __init__(self, parent, id)
#   OnLinkClicked(self, linkinfo)
#   CalculateAllFonts(self, defaultsize)
#   SetDefaultFontAndSize(self, fontname)
#
class chat_html_window(wxHtmlWindow):
    """ a wxHTMLwindow that will load links  """

    # initialization subroutine
    #
    # !self : instance of self
    # !parent :
    # !id :
    def __init__(self, parent, id):
        wxHtmlWindow.__init__(self, parent, id, wxDefaultPosition,wxDefaultSize,
                                wxSUNKEN_BORDER | wxHW_SCROLLBAR_AUTO)

        self.parent = parent
        # Hook up event handler for right-mouse-click
        EVT_RIGHT_UP(self, self.on_rclick)

    # def __init__ - end

    def on_rclick(self, event):
      "Create a popup menu when right-clicked."

      if not hasattr(self, "popup_close"):
        self.popup_close = wxNewId()
        EVT_MENU(self, self.popup_close, self.OnPopupClose)

      popup_menu = wxMenu()
      item = wxMenuItem(popup_menu, self.popup_close, "Close window")
      popup_menu.AppendItem(item)
      self.PopupMenu(popup_menu, event.GetPosition())
      popup_menu.Destroy()

    def OnPopupClose(self, event):
      self.parent.parent.destroy_private_tab(self.parent)



    # This subroutine fires up the webbrowser when a link is clicked.
    #
    # !self : instance of self
    # !linkinfo : instance of a class that contains the link information
    def OnLinkClicked(self, linkinfo):
        href = linkinfo.GetHref()
        webbrowser.open(href)
    # def OnLinkClicked - end

    def CalculateAllFonts(self, defaultsize):
        return [int(defaultsize * 0.4),
                int(defaultsize * 0.7),
                int(defaultsize),
                int(defaultsize * 1.3),
                int(defaultsize * 1.7),
                int(defaultsize * 2),
                int(defaultsize * 2.5)]

    def SetDefaultFontAndSize(self, fontname, fontsize):
        """Set 'fontname' to the default chat font.
           Returns current font settings in a (fontname, fontsize) tuple."""
        self.SetFonts(fontname, "", self.CalculateAllFonts(fontsize))
        return (self.GetFont().GetFaceName(), self.GetFont().GetPointSize())


# class chat_html_window - end



#########################
#chat frame window
#########################
# These are kinda global...and static..and should be located somewhere else
# then the middle of a file between two classes.
IDC_TREE = wxNewId()
IDC_SEND = wxNewId()
IDC_BOLD = wxNewId()
IDC_ITALIC = wxNewId()
IDC_UNDER = wxNewId()
IDC_TEXT_COLOR = wxNewId()
CHAT_SAVE = wxNewId()
IDC_D4 = wxNewId()
IDC_D6 = wxNewId()
IDC_D8 = wxNewId()
IDC_D10 = wxNewId()
IDC_D12 = wxNewId()
IDC_D20 = wxNewId()
IDC_D100 = wxNewId()
IDC_NUMDICE = wxNewId()
IDC_MODS = wxNewId()
TOGM = wxNewId()
SLOCK = wxNewId()
TEXTPOP = wxNewId()
SCROL_TOOL = wxNewId()
SAVE_LOG = wxNewId()
TEXTPOP_MENU = wxNewId()

# Heroman - Alias & Filter toolbar buttons
IDC_ALIAS = wxNewId()
IDC_ALIAS_REFRESH=wxNewId()
IDC_FILTER = wxNewId()
IDC_FILTER_REFRESH=wxNewId()
DEFAULT_ALIAS_NAME='Use Real Name'
DEFAULT_FILTER_NAME='No Filter'

BACKGROUND_COLOR = 1050
TEXT_COLOR = 1060
SYSTEM_COLOR = 1070
INFO_COLOR = 1080
EMOTE_COLOR = 1090

BUF_SIZE = wxNewId()

#DUMMY MENUS
SET_CHAT_FOCUS = wxNewId()


# This class defines the tabbed 'notebook' that holds multiple chatpanels.
# It's the widget attached to the main application frame.
#
# Inherits:  wxNotebook
#
# Defines:
#   create_private_tab(self, playerid)
#   get_tab_index(self, chatpanel)
#   destroy_private_tab(self, chatpanel)
#   OnPageChanged(self, event)
#   set_default_font(self, font, fontsize)

class chat_notebook(wxNotebook):

  def __init__(self, parent, id, openrpg):
    wxNotebook.__init__(self, parent, id)
    self.openrpg = openrpg
    self.panel_list = []  # list of 'private' chatpanels holding whispered chats

    # build list of images to put in chatpanel tabs
    imagelist = wxImageList(16, 16)
    self.chatNormalIdx = imagelist.Add(getNormalBitmap())
    self.chatAttentionIdx = imagelist.Add(getAttentionBitmap())
    self.AssignImageList(imagelist)

    # Create "main" chatpanel tab, undeletable, connected to 'public' room.
    self.MainChatPanel = chat_panel(self, -1, openrpg, "all")
    self.AddPage(self.MainChatPanel, "Main Room")

    # Hook up event handler for flipping tabs
    EVT_NOTEBOOK_PAGE_CHANGED(self, self.GetId(), self.OnPageChanged)

    # html font/fontsize is global to all the notebook tabs.
    self.font = self.MainChatPanel.chatwnd.GetFont().GetFaceName()
    self.fontsize = self.MainChatPanel.chatwnd.GetFont().GetPointSize()


  def get_tab_index(self, chatpanel):
    "Return the index of a chatpanel in the wxNotebook."

    for i in range(self.GetPageCount()):
      if (self.GetPage(i) == chatpanel):
        return i


  def create_private_tab(self, playerid):
    "Add a new chatpanel directly connected to integer 'playerid' via whispering."

    private_tab = chat_panel(self, -1, self.openrpg, playerid)
    playername = strip_html(self.MainChatPanel.session.get_player_by_player_id(playerid)[0])
    self.AddPage(private_tab, playername)
    private_tab.chatwnd.SetDefaultFontAndSize(self.font, self.fontsize)

    self.panel_list.append(private_tab)

    idx = self.get_tab_index(private_tab)
    self.SetPageImage(idx, self.chatAttentionIdx)

    return private_tab

#------------------------------------------------------------------------------------------------------
# [START]Group Tab Added by Digitalxero
# This will allow plugins to creat custom whisper tabs that are tied to something other then player_id
#------------------------------------------------------------------------------------------------------
  def create_group_tab(self, tab_name, group_name):
    "Add a new chatpanel directly connected to integer 'playerid' via whispering."

    private_tab = chat_panel(self, -1, self.openrpg, group_name)
    self.AddPage(private_tab, tab_name)
    private_tab.chatwnd.SetDefaultFontAndSize(self.font, self.fontsize)

    self.panel_list.append(private_tab)

    idx = self.get_tab_index(private_tab)
    self.SetPageImage(idx, self.chatAttentionIdx)

    return private_tab
#------------------------------------------------------------------------------------------------------
# [END]Group Tab Added by Digitalxero
# This will allow plugins to creat custom whisper tabs that are tied to something other then player_id
#------------------------------------------------------------------------------------------------------

  def destroy_private_tab(self, chatpanel):
    "Destroy a chatpanel tab attached to notebook."

    if (chatpanel != self.MainChatPanel):
      self.panel_list.remove(chatpanel)
      chatpanel.chat_timer.Stop()
      self.DeletePage(self.get_tab_index(chatpanel))


  def OnPageChanged(self, event):
    "When private chattabs are selected, set the bitmap back to 'normal'."

    selected_idx = event.GetSelection()
    selected_chatpanel = self.GetPage(selected_idx)
    if (selected_chatpanel.sendtarget != "all"):
      self.SetPageImage(selected_idx, self.chatNormalIdx)
    event.Skip()



# This class defines and builds the Chat Frame for OpenRPG
#
# Inherits: wxPanel
#
# Defines:
#   __init__((self, parent, id, openrpg, sendtarget)
#   build_ctrls(self)
#   on_buffer_size(self,evt)
#   set_colors(self)
#   set_buffersize(self)
#   set_chat_text(self,txt)
#   OnChar(self,event)
#   on_chat_save(self,evt)
#   on_text_color(self,event)
#   colorize(self, color, text)
#   on_text_format(self,event)
#   change_color(self,event)
#   OnSize(self,event)
#   scroll_down(self)
#   InfoPost(self,s)
#   Post(self,s="",send=false,myself=false)
#   ParsePost(self,s,send=false,myself=false)
#   ParseDice(self,s)
#   ParseNodes(self,s)
#   get_sha_checksum(self)
#   get_color(self)
#

class chat_panel(wxPanel):

    # This is the initialization subroutine
    #
    # !self : instance of self
    # !parent : parent that defines the chatframe
    # !id :
    # !openrpg :
        # !sendtarget:  who gets outbound messages: either 'all' or a playerid
    def __init__(self, parent, id, openrpg, sendtarget):
        wxPanel.__init__(self, parent, id)
        self.session = openrpg.get_component('session')
        self.settings = openrpg.get_component('settings')
        self.myopenrpg = openrpg
        self.parent = parent
        # who receives outbound messages, either "all" or "playerid" string
        self.sendtarget = sendtarget

        # create die roller manager
        self.chatbuffer = []
        self.roller_manager = openrpg.get_component('roller_manager')
        # create rpghex tool
        self.r_h = orpg.tools.rgbhex.RGBHex()
        self.set_colors()
        self.version = VERSION
        self.histidx = -1
        self.temptext = ""
        self.history = []
        #self.lasthistevt = None
        self.buffersize = int(self.settings.get_setting('buffersize'))
        self.bufferpointer = self.lowbuffer = self.hibuffer = 0
        #chat commands
        self.lockscroll = 0      # set the default to scrolling on.
        self.chat_cmds = commands.chat_commands(self)

        self.lastSend = 0         #  this is used to help implement the player typing indicator
        self.lastPress = 0        #  this is used to help implement the player typing indicator
        self.chat_timer = mytimer(self.typingTimerFunc)
        self.chat_timer.Start(1000)
        EVT_SIZE(self, self.OnSize)

        self.build_ctrls()
        #openrpg dir
        self.root_dir = orpg.dirpath.dir_struct["home"]


        # html font/fontsize is global to all the notebook tabs.
        self.font = self.chatwnd.GetFont().GetFaceName()
        self.fontsize = self.chatwnd.GetFont().GetPointSize()
        StartupFont = self.settings.get_setting("defaultfont")
        StartupFontSize = self.settings.get_setting("defaultfontsize")
        if(StartupFont != "") and (StartupFontSize != ""):
            try:
                self.set_default_font(StartupFont, int(StartupFontSize))
            except:
                pass

    def set_default_font(self, fontname=None, fontsize=None):
        """Set all chatpanels to new default fontname/fontsize.
           Returns current font settings in a (fontname, fontsize) tuple."""

        if (fontname is not None):
            newfont = fontname
        else:
            newfont = self.font

        if (fontsize is not None):
            newfontsize = int(fontsize)
        else:
            newfontsize = int(self.fontsize)

        self.chatwnd.SetDefaultFontAndSize(newfont, newfontsize)
        self.InfoPost("Font is now " + newfont + " point size " + `newfontsize`)

        self.font = newfont
        self.fontsize = newfontsize
        return (self.font, self.fontsize)



    def build_menu(self):
        top_frame = self.myopenrpg.get_component('frame')

        menu = wxMenu()
        menu.Append(BACKGROUND_COLOR,"&Background color")
        menu.Append(TEXT_COLOR,"&Text color")
        menu.AppendSeparator()
        menu.Append(SET_CHAT_FOCUS,"&Chat Focus\tCtrl-H")
        menu.AppendSeparator()
        menu.Append( SCROL_TOOL, "Toggle &Scroll Lock")
        menu.Append( SAVE_LOG, "Save Chat &Log")
        menu.Append( TEXTPOP_MENU, "Text &View")

        top_frame.mainmenu.Insert(2, menu, '&Chat')

        EVT_MENU(top_frame, TEXTPOP_MENU, self.pop_textpop)
        EVT_MENU(top_frame, SAVE_LOG, self.on_chat_save)
        EVT_MENU(top_frame, SCROL_TOOL, self.togglescroll)

        EVT_MENU(top_frame, BACKGROUND_COLOR, self.change_color)
        EVT_MENU(top_frame, TEXT_COLOR, self.change_color)
        EVT_MENU(top_frame, SET_CHAT_FOCUS, self.set_chat_text_focus)


    def get_hot_keys(self):
        # dummy menus for hotkeys
        #top_frame = self.myopenrpg.get_component('frame')
        #EVT_MENU(top_frame, SET_CHAT_FOCUS, self.set_chat_text_focus)
        self.build_menu()
        entries = []
        entries.append((wxACCEL_CTRL,ord('H'),SET_CHAT_FOCUS))
        return entries

    # This subroutine builds the controls for the chat frame
    #
    # !self : instance of self
    def build_ctrls(self):
        #chat log
        self.chatwnd = chat_html_window(self,-1)
        if (self.sendtarget == "all"):
            self.Post(self.colorize(self.syscolor,
                                  "<b>Welcome to <a href='http://www.openrpg.com'>OpenRPG</a> version " \
                                  + self.version + "...  </b>"))
            self.chat_cmds.on_help()
#        self.toolbar_sizer = wxBoxSizer(wxHORIZONTAL)
        self.chattxt = orpg.tools.predTextCtrl.predTextCtrl(self, -1, "",
            style=wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB,keyHook = self.myKeyHook, validator=None )
        self.build_bar()
        self.basesizer = wxBoxSizer(wxVERTICAL)
        self.basesizer.Add( self.chatwnd,1,wxEXPAND )
        self.basesizer.Add( self.toolbar_sizer, 0, wxEXPAND )
        self.basesizer.Add( self.chattxt, 0, wxEXPAND )
        self.SetSizer(self.basesizer)
        #events
        EVT_TEXT(self, BUF_SIZE, self.on_buffer_size)
        EVT_BUTTON(self, IDC_BOLD, self.on_text_format)
        EVT_BUTTON(self, IDC_ITALIC, self.on_text_format)
        EVT_BUTTON(self, IDC_UNDER, self.on_text_format)
        EVT_BUTTON(self, IDC_TEXT_COLOR, self.on_text_color)
        EVT_BUTTON(self, CHAT_SAVE, self.on_chat_save)

        EVT_BUTTON( self, IDC_D4, self.onDieRoll )
        EVT_BUTTON( self, IDC_D6, self.onDieRoll )
        EVT_BUTTON( self, IDC_D8, self.onDieRoll )
        EVT_BUTTON( self, IDC_D10, self.onDieRoll )
        EVT_BUTTON( self, IDC_D12, self.onDieRoll )
        EVT_BUTTON( self, IDC_D20, self.onDieRoll )
        EVT_BUTTON( self, IDC_D100, self.onDieRoll )
        EVT_BUTTON(self, TEXTPOP, self.pop_textpop)
        try:
            EVT_TOGGLEBUTTON(self, SLOCK,self.lock_scroll)
        except:
            EVT_BUTTON(self, SLOCK,self.lock_scroll)

        EVT_RIGHT_DCLICK(self.chatwnd,self.set_chat_text_focus)
        EVT_MOTION(self.chatwnd,self.OnMotion)
        EVT_LEFT_UP(self.chatwnd,self.chatwin_lclick)
        # Maybe nuke this for the prefs dialog instead


#        if self.settings.get_setting('SuppressChatAutoComplete') == '1':
#            EVT_CHAR(self.chattxt, self.OnChar)
#        else:
        EVT_CHAR(self.chattxt, self.chattxt.OnChar)

        #EVT_TEXT_ENTER(self.chattxt,1, self.OnChar)
    # def build_ctrls - end

    def chatwin_lclick(self,event):
        try:
            #if this variable exists, then we're using a version that
            #supports text selection in the HTML win -- so we don't want
            #to screw with it
            wxHW_NO_SELECTION
            event.Skip()
        except:
            #otherwise, we might as well set chat focus
            self.set_chat_text_focus(event)

    def set_sizer(self):
        self.SetSizer(self.basesizer)

    def build_bar(self):
        self.toolbar_sizer = wxBoxSizer(wxHORIZONTAL)
        self.scroll_lock = None
        self.numDieText = None
        self.dieModText = None


        if self.settings.get_setting('Toolbar_On') == "1":
            self.build_alias()
            self.build_dice()
            self.build_gm()
            self.build_scroll()
            self.build_text()
            self.toolbar_sizer.Add( self.textpop_lock, 0, wxEXPAND )
            self.toolbar_sizer.Add(self.scroll_lock,0,wxEXPAND)
            self.build_formating()
            self.build_colorbutton()


    def build_scroll(self):
        self.scroll_lock = None
        if sys.platform != "darwin":
            self.scroll_lock = wxToggleButton( self, SLOCK, "Scroll ON",size=wxSize(70,25))
        else:
            self.scroll_lock = wxButton( self, SLOCK, "*Disabled*",size=wxSize(80,25))

    def build_alias(self):
        sampleList=[DEFAULT_ALIAS_NAME,'','','','','']
        self.aliasList=wxChoice(self,IDC_ALIAS,size=wxSize(100, 25),choices=sampleList)
        self.aliasButton = createMaskedButton( self, orpg.dirpath.dir_struct["icon"]+'player.gif', 'Refresh list of aliases from Game Tree', IDC_ALIAS_REFRESH, '#bdbdbd' )

        sampleList=[DEFAULT_FILTER_NAME,'','','','','']
        self.filterList=wxChoice(self,IDC_FILTER,size=wxSize(100, 25),choices=sampleList)
        self.filterButton = createMaskedButton( self, orpg.dirpath.dir_struct["icon"]+'add_filter.gif', 'Refresh list of filters from Game Tree', IDC_FILTER_REFRESH, '#bdbdbd' )

        self.toolbar_sizer.Add( self.aliasButton, 0, wxEXPAND )
        self.toolbar_sizer.Add( self.aliasList,0,wxEXPAND)
        self.toolbar_sizer.Add( self.filterButton, 0, wxEXPAND )
        self.toolbar_sizer.Add( self.filterList,0,wxEXPAND)
        # Heroman - Refresh alias, filter lists
        EVT_BUTTON(self,IDC_ALIAS_REFRESH,self.onRefreshAliasList)
        EVT_BUTTON(self,IDC_FILTER_REFRESH,self.onRefreshFilterList)
        if self.settings.get_setting('AliasTool_On') == '0':
            self.toggle_alias('0')
        else:
            self.toggle_alias('1')

    def toggle_alias(self,act):
        if act == '0':
            self.toolbar_sizer.Show(self.aliasList,false)
            self.toolbar_sizer.Show(self.filterList,false)
            self.toolbar_sizer.Show(self.aliasButton,false)
            self.toolbar_sizer.Show(self.filterButton,false)
            self.toolbar_sizer.Layout()
        else:
            self.toolbar_sizer.Show(self.aliasList,true)
            self.toolbar_sizer.Show(self.filterList,true)
            self.toolbar_sizer.Show(self.aliasButton,true)
            self.toolbar_sizer.Show(self.filterButton,true)
            self.toolbar_sizer.Layout()

    def build_gm(self):
        if sys.platform != "darwin":
           self.to_gm = wxToggleButton( self, TOGM, "To GM(s)", size= wxSize(70,25))
        else:
           self.to_gm = wxButton( self, TOGM, "*Disabled*", size=wxSize(80,25))
        self.toolbar_sizer.Add(self.to_gm,0,wxEXPAND)

        if self.settings.get_setting('ToGMsButton_On') == '0':
            self.toggle_gm('0')
        else:
           self.toggle_gm('1')

    def toggle_gm(self, act):
        if act == '0':
            self.toolbar_sizer.Show(self.to_gm,false)
            self.toolbar_sizer.Layout()
        else:
            self.toolbar_sizer.Show(self.to_gm,true)
            self.toolbar_sizer.Layout()


    def build_text(self):
        self.textpop_lock = createMaskedButton(self, orpg.dirpath.dir_struct["icon"]+'note.gif', 'Open Text View Of Chat Session', TEXTPOP, '#bdbdbd')

    def build_dice(self):
        self.numDieText = wxTextCtrl( self, IDC_NUMDICE, "1", size=wxSize(25, 25), validator=orpg.tools.inputValidator.MathOnlyValidator() )
        self.dieModText = wxTextCtrl( self, IDC_MODS, "", size=wxSize(50, 25), validator=orpg.tools.inputValidator.MathOnlyValidator() )
        self.d4Button = createMaskedButton(self, orpg.dirpath.dir_struct["icon"]+'b_d4.gif', 'Roll d4', IDC_D4, '#bdbdbd')
        self.d6Button = createMaskedButton(self, orpg.dirpath.dir_struct["icon"]+'b_d6.gif', 'Roll d6', IDC_D6, '#bdbdbd')
        self.d8Button = createMaskedButton(self, orpg.dirpath.dir_struct["icon"]+'b_d8.gif', 'Roll d8', IDC_D8, '#bdbdbd')
        self.d10Button = createMaskedButton(self, orpg.dirpath.dir_struct["icon"]+'b_d10.gif', 'Roll d10', IDC_D10, '#bdbdbd')
        self.d12Button = createMaskedButton(self, orpg.dirpath.dir_struct["icon"]+'b_d12.gif', 'Roll d12', IDC_D12, '#bdbdbd')
        self.d20Button = createMaskedButton(self, orpg.dirpath.dir_struct["icon"]+'b_d20.gif', 'Roll d20', IDC_D20, '#bdbdbd')
        self.d100Button = createMaskedButton(self, orpg.dirpath.dir_struct["icon"]+'b_d100.gif', 'Roll d100', IDC_D100, '#bdbdbd')

        self.toolbar_sizer.Add( self.numDieText, 0, wxALIGN_CENTER | wxEXPAND)
        self.toolbar_sizer.Add( self.d4Button, 0 ,wxEXPAND)
        self.toolbar_sizer.Add( self.d6Button, 0 ,wxEXPAND)
        self.toolbar_sizer.Add( self.d8Button, 0 ,wxEXPAND)
        self.toolbar_sizer.Add( self.d10Button, 0 ,wxEXPAND)
        self.toolbar_sizer.Add( self.d12Button, 0 ,wxEXPAND)
        self.toolbar_sizer.Add( self.d20Button, 0 ,wxEXPAND)
        self.toolbar_sizer.Add( self.d100Button, 0 ,wxEXPAND)
        self.toolbar_sizer.Add( self.dieModText, 0, wxALIGN_CENTER, 5 )
        if self.settings.get_setting('DiceButtons_On') == '0':
            self.toggle_dice('0')
        else:
            self.toggle_dice('1')

    def toggle_dice(self, act):
        if act == '0':
            self.toolbar_sizer.Show(self.numDieText, false)
            self.toolbar_sizer.Show(self.d4Button, false)
            self.toolbar_sizer.Show(self.d6Button, false)
            self.toolbar_sizer.Show(self.d8Button, false)
            self.toolbar_sizer.Show(self.d10Button, false)
            self.toolbar_sizer.Show(self.d12Button, false)
            self.toolbar_sizer.Show(self.d20Button, false)
            self.toolbar_sizer.Show(self.d100Button, false)
            self.toolbar_sizer.Show(self.dieModText, false)
            self.toolbar_sizer.Layout()
        else:
            self.toolbar_sizer.Show(self.numDieText, true)
            self.toolbar_sizer.Show(self.d4Button, true)
            self.toolbar_sizer.Show(self.d6Button, true)
            self.toolbar_sizer.Show(self.d8Button, true)
            self.toolbar_sizer.Show(self.d10Button, true)
            self.toolbar_sizer.Show(self.d12Button, true)
            self.toolbar_sizer.Show(self.d20Button, true)
            self.toolbar_sizer.Show(self.d100Button, true)
            self.toolbar_sizer.Show(self.dieModText, true)
            self.toolbar_sizer.Layout()


    def build_formating(self):
        self.boldButton = createMaskedButton( self, orpg.dirpath.dir_struct["icon"]+'bold.gif', 'Make the selected text Bold', IDC_BOLD, '#bdbdbd')
        self.italicButton = createMaskedButton( self, orpg.dirpath.dir_struct["icon"]+'italic.gif', 'Italicize the selected text', IDC_ITALIC, '#bdbdbd' )
        self.underlineButton = createMaskedButton( self, orpg.dirpath.dir_struct["icon"]+'underlined.gif', 'Underline the selected text', IDC_UNDER, '#bdbdbd' )
        self.toolbar_sizer.Add( self.boldButton, 0, wxEXPAND )
        self.toolbar_sizer.Add( self.italicButton, 0, wxEXPAND )
        self.toolbar_sizer.Add( self.underlineButton, 0, wxEXPAND )
        if self.settings.get_setting('FormattingButtons_On') == '0':
            self.toggle_formating('0')
        else:
            self.toggle_formating('1')

    def toggle_formating(self, act):
        if act == '0':
            self.toolbar_sizer.Show(self.boldButton, false)
            self.toolbar_sizer.Show(self.italicButton, false)
            self.toolbar_sizer.Show(self.underlineButton, false)
            self.toolbar_sizer.Layout()
        else:
            self.toolbar_sizer.Show(self.boldButton, true)
            self.toolbar_sizer.Show(self.italicButton, true)
            self.toolbar_sizer.Show(self.underlineButton, true)
            self.toolbar_sizer.Layout()

    # Heroman - Ideally, we would use static labels...
    def build_colorbutton(self):
        self.color_button = wxButton(self, IDC_TEXT_COLOR, "C",wxPoint(0,0), wxSize(22,0))
        self.saveButton = createMaskedButton( self, orpg.dirpath.dir_struct["icon"]+'save.bmp', 'Save the chatbuffer', CHAT_SAVE, '#c0c0c0', wxBITMAP_TYPE_BMP )
        self.color_button.SetBackgroundColour(wxBLACK)
        self.color_button.SetForegroundColour(wxWHITE)
        self.toolbar_sizer.Add(self.color_button, 0, wxEXPAND)
        self.toolbar_sizer.Add( self.saveButton, 0, wxEXPAND )

    def AddAliasHandle(self,dom,names):
        for n in dom._get_childNodes():
            try:
                if n.getAttribute('class') == "voxchat_handler":
                    list = n.getElementsByTagName( 'voxchat.alias' )
                    for entry in list:
                        name = entry.getAttribute( 'name' )
                        names.append( name )
                self.AddAliasHandle(n,names)
            except:
                continue


    def AddFilterHandle(self,dom,names):
        for n in dom._get_childNodes():
            try:
                if n.getAttribute('class') == "voxchat_handler":
                    list = n.getElementsByTagName( 'voxchat.filter' )
                    for entry in list:
                        name = entry.getAttribute( 'name' )
                        names.append( name )
                self.AddFilterHandle(n,names)
            except:
                continue

    def RefreshAliasList(self):
        self.aliasList.Clear()
        self.aliasList.Append(DEFAULT_ALIAS_NAME)
        gametree=self.myopenrpg.get_component('tree')
        names = []
        self.AddAliasHandle(gametree.master_dom,names)
        names.sort()
        if names:
            last = names[-1]
            for i in range(len(names)-2, -1, -1):
                if last==names[i]: del names[i]
                else: last=names[i]

        for sortName in names:
            self.aliasList.Append(sortName)
#        self.InfoPost("Found " + str(aliasCount) + " aliases from top level game tree.")

    def RefreshFilterList(self):
        self.filterList.Clear()
        self.filterList.Append(DEFAULT_FILTER_NAME)
        gametree=self.myopenrpg.get_component('tree')
        aliasFilters = []
        self.AddFilterHandle(gametree.master_dom,aliasFilters)
        aliasFilters.sort()
        if aliasFilters:
            last = aliasFilters[-1]
            for i in range(len(aliasFilters)-2, -1, -1):
                if last==aliasFilters[i]: del aliasFilters[i]
                else: last=aliasFilters[i]
        for sortName in aliasFilters:
            self.filterList.Append(sortName)


    def applyFilter(self,text,filterName):
        gametree=self.myopenrpg.get_component('tree')
        list=gametree.master_dom.getElementsByTagName( 'voxchat.filter' )
        for node in list:
            if (node.getAttribute('name')==filterName):
                list = node.getElementsByTagName( 'rule' )
                for rule in list:
                    match = rule.getAttribute( 'match' )
                    sub = rule.getAttribute( 'sub' )
                    text = re.sub( match, sub, text )
                return text
        return text

    def onRefreshFilterList(self,evt):
        self.RefreshFilterList()

    def onRefreshAliasList(self,evt):
        self.RefreshAliasList()

    def OnMotion(self,evt):
        contain = self.chatwnd.GetInternalRepresentation()
        if contain:
            sx = sy = 0
            x = y = 0
            (sx,sy) = self.chatwnd.GetViewStart()
            (sx1,sy1) = self.chatwnd.GetScrollPixelsPerUnit()
            sx = sx*sx1
            sy = sy*sy1
            (x,y) = evt.GetPosition()
            lnk = contain.GetLink(sx+x,sy+y)

            if lnk:
                try:
                    link = lnk.GetHref()
                    self.session.set_status_url(link)
                except:
                    pass

        else:
            print "Error, self.chatwnd.GetInternalRepresentation() return None"

        evt.Skip()


    #  This subroutine is registered with predTextCtrl to be run for every OnChar event
    #  It checks if we need to send a typing message

    #
    #  self:  duh
    #  event:  raw KeyEvent from OnChar()
    def myKeyHook(self,event):

        if self.session.get_status() == MPLAY_CONNECTED:   #  only do if we're connected
            thisPress = time.time()                #  thisPress is local temp variable

            if (thisPress - self.lastSend) > 4:    #  Check to see if it's been 5 seconds since our last notice
                                                   #    If we're not already typing, then self.lastSend will be 0
                self.sendTyping(1)                 #  send a not typing event here (1 for true)

            self.lastPress = thisPress             #  either way, record the time of this keystroke for use in
                                                   #  self.typingTimerFunc()

        if self.settings.get_setting('SuppressChatAutoComplete') == '1':
            return 1
        else:
            return 0



    #  This subroutine gets called once a second by the typing Timer
    #  It checks if we need to send a not_typing message
    #
    #  self:  duh
    def typingTimerFunc(self):
        #following added by mDuo13
        ##############refresh_counter()##############
        for plugin_fname in orpg.plugins.frame.enabled_plugins.keys():
            plugin = orpg.plugins.frame.enabled_plugins[plugin_fname]
            try:
                plugin.refresh_counter(self)
            except Exception, e:
                if str(e) != "'module' object has no attribute 'refresh_counter'":
                    #print e
                    traceback.print_exc()
        #end mDuo13 added code


        if self.lastSend:                          #  This will be zero when not typing, so equiv to if is_typing
            thisTime = time.time()                 #  thisTime is a local temp variable
            if (thisTime - self.lastPress) > 4:    #  Check to see if it's been 5 seconds since our last keystroke
                                               #    If we're not already typing, then self.lastSend will be 0

                self.sendTyping(0)                 #  send a typing event here (0 for false)


    #  This subroutine actually takes care of sending the messages for typing/not_typing events
    #
    #  self:  duh
    #  typing:  boolean
    def sendTyping(self,typing):

        if typing:
            self.lastSend = time.time()  #  remember our send time for use in myKeyHook()
            #I think this is cleaner
            status_text = self.settings.get_setting('TypingStatusAlias')
            if status_text == "" or status_text == None:
                status_text = "Typing"
            self.session.set_text_status(status_text)

            #theMsg = self.session.toxml("typing")        #  build the xml message to send
            #self.session.pretranslate(theMsg)            #  since the server won't return our message to us,
                                                         #    pretend it does by calling the pretranlation routine
                                                         #    to simulate this behavior locally.

            #self.session.outbox.put(theMsg)              #  code to send xml

        else:
            self.lastSend = 0                            #  set lastSend to zero to indicate we're not typing
            #I think this is cleaner
            status_text = self.settings.get_setting('IdleStatusAlias')
            if status_text == "" or status_text == None:
                status_text = "Idle"
            self.session.set_text_status(status_text)
            #theMsg = self.session.toxml("not_typing")    #  build the xml message to send
            #self.session.pretranslate(theMsg)            #  since the server won't return our message to us,
                                                         #    to simulate this behavior locally.

            #self.session.outbox.put(theMsg)              #  code to send xml



    # This subroutine checks the buffersize of the chat window to make sure it
    # is good.
    #
    # !self : instance of self
    # !evt :
    def on_buffer_size(self,evt):
#        txt = self.buftxt.GetValue()
        try:
            num = int(evt)
            if num > 0:
                self.buffersize = num
#                self.settings.set_setting('buffersize',txt)
        except:
            print "Invalid chat buffer size!"
    # def in_buffer_size - end


    # This subroutine sets the colors of the chat based on the settings in the
    # self instance.
    #
    # !self : instance of self
    def set_colors(self):
        # chat window backround color
        self.bgcolor = self.settings.get_setting('bgcolor')
         # chat window normal text color
        self.textcolor = self.settings.get_setting('textcolor')
        # color of text player types
        self.mytextcolor = self.settings.get_setting('mytextcolor')
        # color of system warnings
        self.syscolor = self.settings.get_setting('syscolor')
        # color of system info messages
        self.infocolor = self.settings.get_setting('infocolor')
        # color of emotes
        self.emotecolor = self.settings.get_setting('emotecolor')
        # color of whispers
        self.whispercolor = self.settings.get_setting('whispercolor')
    # def set_colors - end


    # This subroutine will set the size of the buffer on the chat window
    #
    # !self : instance of self
    def set_buffersize(self):
        self.buffersize = int(self.settings.get_setting('buffersize'))
#        self.buftxt.SetValue(self.settings.get_setting('buffersize'))
    # def set_buffersize - end


    # This subroutine will insert text into the chat window
    #
    # !self : instance of self
    # !txt : text to be inserted into the chat window
    def set_chat_text(self,txt):
        self.chattxt.SetValue(txt)
        self.chattxt.SetFocus()
        self.chattxt.SetInsertionPointEnd()
    # def set_chat_text - end

    def get_chat_text(self):
        return self.chattxt.GetValue()

    # This subroutine sets the focus to the chat window
    def set_chat_text_focus(self,event):
        self.chattxt.SetFocus()
    # def set_chat_text_focus - end

    # This subrtouine grabs the user input and make the special keys and
    # modifiers work.
    #
    # !self : instance of self
    # !event :
    #
    # Note:  self.chattxt now handles it's own Key events.  It does, however still
    #        call it's parent's (self) OnChar to handle "default" behavior.
    def OnChar(self,event):
        s = self.chattxt.GetValue()
        #self.histlen = len(self.history) - 1

        ## RETURN KEY (no matter if there is text in chattxt)
        #  This section is run even if there is nothing in the chattxt (as opposed to the next WXK_RETURN handler
        if event.KeyCode() == WXK_RETURN:
            if self.session.get_status() == MPLAY_CONNECTED:          #  only do if we're connected
                self.sendTyping(0)                                    #  Send a "not_typing" event on enter key press

        macroText=""
        if event.KeyCode() == WXK_F1:
            macroText=self.settings.get_setting('F1')
        elif event.KeyCode() == WXK_F2:
            macroText=self.settings.get_setting('F2')
        elif event.KeyCode() == WXK_F3:
            macroText=self.settings.get_setting('F3')
        elif event.KeyCode() == WXK_F4:
            macroText=self.settings.get_setting('F4')
        elif event.KeyCode() == WXK_F5:
            macroText=self.settings.get_setting('F5')
        elif event.KeyCode() == WXK_F6:
            macroText=self.settings.get_setting('F6')
        elif event.KeyCode() == WXK_F7:
            macroText=self.settings.get_setting('F7')
        elif event.KeyCode() == WXK_F8:
            macroText=self.settings.get_setting('F8')
        elif event.KeyCode() == WXK_F9:
            macroText=self.settings.get_setting('F9')
        elif event.KeyCode() == WXK_F10:
            macroText=self.settings.get_setting('F10')
        elif event.KeyCode() == WXK_F11:
            macroText=self.settings.get_setting('F11')
        elif event.KeyCode() == WXK_F12:
            macroText=self.settings.get_setting('F12')

        # Append to the existing typed text as needed and make sure the status doesn't change back.
        if len(macroText):
            self.sendTyping(0)
            s = macroText

        ## RETURN KEY (and not text in control)
        if (event.KeyCode() == WXK_RETURN and len(s)) or len(macroText):
#            if self.settings.get_setting( "SuppressChatAutoComplete" ) != "0":
            self.histidx = -1
            self.temptext = ""
            self.history = [s] + self.history#prepended instead of appended now, so higher index = greater age
            #self.lasthistevt=None
            #following added by mDuo13
            ################message()################
            for plugin_fname in orpg.plugins.frame.enabled_plugins.keys():
                plugin = orpg.plugins.frame.enabled_plugins[plugin_fname]
                try:
                    s = plugin.message(self, s)
                except Exception, e:
                    if str(e) != "'module' object has no attribute 'message'":
                        #print e
                        traceback.print_exc()
            #end mDuo13 added code


            try:
                if self.to_gm.GetValue()==1 and s[0]!="/":
                    the_gms = self.get_gms()
                    if len(the_gms):
                        gmstring = ""
                        for each_gm in the_gms:
                            gmstring +=each_gm+","
                        s = "/w "+gmstring[:-1]+"="+s
            except Exception, e:
                print e


            if s[0] != "/": ## it's not a slash command
                s = self.ParsePost( s, true, true )
            else:
                #s = self.ParseDice(s)
                self.chat_cmds.docmd(s) # emote is in chatutils.py

            if not len(macroText):
                self.chattxt.SetValue("")

            # play sound
            sound_file = self.settings.get_setting("SendSound")
            sound_player = orpg.tools.orpg_sound.orpg_sound(self.settings.get_setting("UnixSoundPlayer"));
            sound_player.play(sound_file)


        ## UP KEY
        elif event.KeyCode() == WXK_UP:
            if self.histidx < len(self.history)-1:
                #text that's not in history but also hasn't been sent to chat gets stored in self.temptext
                #this way if someone presses the up key, they don't lose their current message permanently
                #(unless they also press enter at the time)
                if self.histidx is -1:
                    self.temptext = self.chattxt.GetValue()
                self.histidx += 1
                self.chattxt.SetValue(self.history[self.histidx])
                self.chattxt.SetInsertionPointEnd()
            else:
                self.histidx = len(self.history) -1#in case it got too high somehow, this should fix it
                #self.InfoPost("**Going up? I don't think so.**")
            #print self.histidx, "in",self.history

        ## DOWN KEY
        elif event.KeyCode() == WXK_DOWN:
            #histidx of -1 indicates currently viewing text that's not in self.history
            if self.histidx > -1:
                self.histidx -= 1
                if self.histidx is -1:#remember, it just decreased
                    self.chattxt.SetValue(self.temptext)
                else:
                    self.chattxt.SetValue(self.history[self.histidx])
                self.chattxt.SetInsertionPointEnd()
            else:
                self.histidx = -1#just in case it somehow got below -1, this should fix it
                #self.InfoPost("**Going down? I don't think so.**")
            #print self.histidx, "in",self.history

        ## TAB KEY
        elif  event.KeyCode() == WXK_TAB:

            if s !="":
                found = 0
                nicks = []
                testnick = ""
                inlength = len(s)
                for getnames in self.session.players.keys():
                    striphtmltag = re.compile ('<[^>]+>*')
                    testnick = striphtmltag.sub ("", self.session.players[getnames][0])
                    if string.lower(s) == string.lower(testnick[:inlength]):
                        found = found + 1
                        nicks[len(nicks):]=[testnick]
                if found == 0: ## no nick match
                    self.Post(self.colorize(self.syscolor," ** No match found"))
                elif found > 1: ## matched more than 1, tell user what matched
                    nickstring = ""
                    nicklist = []
                    for foundnicks in nicks:
                        nickstring = nickstring + foundnicks + ", "
                        nicklist.append(foundnicks)
                    nickstring = nickstring[:-2]
                    self.Post(self.colorize(self.syscolor, " ** Multiple matches found: " + nickstring))
                    # set text to the prefix match between first two matches
                    settext = re.match(''.join(map(lambda x: '(%s?)' % x, string.lower(nicklist[0]))), string.lower(nicklist[1])).group()
                    # run through the rest of the nicks
                    for i in nicklist:
                        settext = re.match(''.join(map(lambda x: '(%s?)' % x, string.lower(i))), string.lower(settext)).group()
                    if settext:
                        self.chattxt.SetValue(settext)
                        self.chattxt.SetInsertionPointEnd()
                else: ## put the matched name in the chattxt box
                    settext = nicks[0] + ": "
                    self.chattxt.SetValue(settext)
                    self.chattxt.SetInsertionPointEnd()
            else: ## not online, and no text in chattxt box
                self.Post(self.colorize(self.syscolor, " ** That's the Tab key, Dave"))
        ## PAGE UP
        elif event.KeyCode() in (WXK_PRIOR,WXK_PAGEUP):
            if self.bufferpointer < len(self.chatbuffer)-self.buffersize and len(self.chatbuffer) > self.buffersize:
                self.bufferpointer = self.bufferpointer + self.buffersize
                self.Post()
            else:
                event.Skip()
        ## PAGE DOWN
        elif event.KeyCode() in (WXK_NEXT, WXK_PAGEDOWN):
            if self.bufferpointer > 0:
                self.bufferpointer = self.bufferpointer - self.buffersize
                self.Post()
            else:
                event.Skip()

        ## END
        elif event.KeyCode() == WXK_END:
            self.bufferpointer = 0
            self.Post()
            event.Skip()

        ## NOTHING
        else:
            event.Skip()
    # def OnChar - end




    def onDieRoll( self, evt ):
        """Roll the dice based on the button pressed and the die modifiers entered, if any."""

        # Get any die modifiers if they have been entered
        numDie = self.numDieText.GetValue()
        dieMod = self.dieModText.GetValue()

        dieText = numDie

        # Now, apply and roll die mods based on the button that was pressed
        id = evt.GetId()
        if id == IDC_D4:
            dieText += "d4"
        elif id == IDC_D6:
            dieText += "d6"
        elif id == IDC_D8:
            dieText += "d8"
        elif id == IDC_D10:
            dieText += "d10"
        elif id == IDC_D12:
            dieText += "d12"
        elif id == IDC_D20:
            dieText += "d20"
        elif id == IDC_D100:
            dieText += "d100"

        if len(dieMod) and dieMod[0] not in "*/-+":
            dieMod = "+" + dieMod

        dieText += dieMod
        dieText = "["+dieText+"]"
        self.ParsePost(dieText, 1, 1)
        self.chattxt.SetFocus()


    # This subroutine saves a chat buffer as html to a file chosen via a
    # FileDialog.
    #
    # !self : instance of self
    # !evt :
    def on_chat_save(self,evt):
        f =wxFileDialog(self,"Save Chat Buffer",".","","HTM* (*.htm*)|*.htm*|HTML (*.html)|*.html|HTM (*.htm)|*.htm",wxSAVE)
        if f.ShowModal() == wxID_OK:
            file = open(f.GetPath(),"w")
            buffertext = "<html><body bgcolor='"+self.bgcolor+"' text='"+self.textcolor+"'>"
            for bufferslice in self.chatbuffer:
                buffertext = buffertext + bufferslice
            file.write(buffertext+"</body></html>")
            file.close()
        f.Destroy()
        os.chdir(self.root_dir)
        auto_purge_after_save = (self.settings.get_setting('AutoPurgeAfterSave')).lower()
        if auto_purge_after_save == "auto":
           self.Purge_buffer()
        elif auto_purge_after_save == "ask":
            f = wxMessageDialog(self,"Purge Buffer","",wxYES_NO)
            if f.ShowModal() == wxID_OK:
                self.Purge_buffer()
            f.Destroy()
    # def on_chat_save - end


    # This subroutine sets the color of selected text, or base text color if
    # nothing is selected
    def on_text_color(self,event):
        hexcolor = self.r_h.do_hex_color_dlg(self)
        if hexcolor != None:
            (beg,end) = self.chattxt.GetSelection()
            if beg != end:
                txt = self.chattxt.GetValue()
                txt = txt[:beg]+self.colorize(hexcolor,txt[beg:end]) +txt[end:]
                self.chattxt.SetValue(txt)
                self.chattxt.SetInsertionPointEnd()
                self.chattxt.SetFocus()
            else:
                self.color_button.SetBackgroundColour(hexcolor)
                self.mytextcolor = hexcolor
                self.settings.set_setting('mytextcolor',hexcolor)
                self.Post()
    # def on_text_color - end


    # This subroutine take a color and a text string and formats it into html.
    #
    # !self : instance of self
    # !color : color for the text to be set
    # !text : text string to be included in the html.
    def colorize(self, color, text):
        "Puts font tags of 'color' around 'text' value, and returns the string"
        return "<font color='" + color + "'>" + text + "</font>"
    # def colorize - end


    # This subroutine takes and event and inserts text with the basic format
    # tags included.
    #
    # !self : instance of self
    # !event :
    def on_text_format(self,event):
        id = event.GetId()
        txt = self.chattxt.GetValue()
        (beg,end) = self.chattxt.GetSelection()
        if beg != end:
            sel_txt = txt[beg:end]
        else:
            sel_txt = txt

        if id == IDC_BOLD:
            sel_txt = "<b>" + sel_txt + "</b>"
        elif id == IDC_ITALIC:
            sel_txt = "<i>" + sel_txt + "</i>"
        elif id == IDC_UNDER:
            sel_txt = "<u>" + sel_txt + "</u>"

        if beg != end:
            txt = txt[:beg] + sel_txt + txt[end:]
        else:
            txt = sel_txt

        self.chattxt.SetValue(txt)
        self.chattxt.SetInsertionPointEnd()
        self.chattxt.SetFocus()
    # def on_text_format - end


    # This subroutine will change the color of the text and the background
    # color.
    #
    # !self : instance of self
    # !event :
    def change_color(self,event):
        event_id = event.GetId()
        hexcolor = self.get_color()
        if hexcolor != None:
            if event_id == TEXT_COLOR:
                self.textcolor = hexcolor
                self.settings.set_setting('textcolor',hexcolor)
            elif event_id == BACKGROUND_COLOR:
                self.bgcolor = hexcolor
                self.settings.set_setting('bgcolor',hexcolor)
            self.Post()
    # def change_color - end

    def togglescroll(self, event):
        if self.lockscroll == 1:
            self.lockscroll = 0
            onoff = "On"
        elif self.lockscroll == 0:
            self.lockscroll = 1
            onoff = "Off"
        if self.settings.get_setting('Toolbar_On') == "1":
            try:
                self.scroll_lock.SetValue(self.lockscroll)
            except:
                pass #if the Toolbar was turned on this session there will be no instance of scroll_lock
        self.Post("Set scrolling to <b>"+onoff+"</b>")
        self.lock_scroll(1)

    def lock_scroll(self,event):
        if self.scroll_lock == None:
            return
        self.lockscroll = self.scroll_lock.GetValue()
        if self.lockscroll == 0:
            self.scroll_lock.SetLabel("Scroll ON")
            if (self.storedata <> 0):
                self.chatwnd.SetPage(self.storedata)
            self.scroll_down()
        else:
            self.scroll_lock.SetLabel("Scroll OFF")

    # This subroutine will popup a text window with the chatbuffer contents
    #
    # !self : instance of self
    # !event :
    def pop_textpop(self, event):
        "searchable popup text view of chatbuffer"
        h_buffertext = ""
        for h_bufferslice in self.chatbuffer:
            h_buffertext = h_buffertext + string.replace(h_bufferslice, "<br>", "<br>\n")
        h_dlg = wxScrolledMessageFrameEditor(self, h_buffertext, "Text View of Chat Window", None, (500,300))
        h_dlg.Show(true)

    # This subroutine will change the dimension of the window
    #
    # !self : instance of self
    # !event :
    def OnSize(self, event = None):
        s = self.GetClientSizeTuple()
        if self.settings.get_setting('Toolbar_On') == "1":
            self.chatwnd.SetDimensions(0,0,s[0],s[1]-50)
            self.chattxt.SetDimensions(0,s[1]-25,s[0],25)
            self.toolbar_sizer.SetDimension(0,s[1]-50,s[0],25)
        else:
            self.chatwnd.SetDimensions(0,0,s[0],s[1]-25)
            self.chattxt.SetDimensions(0,s[1]-25,s[0],25)
        self.scroll_down()
    # def OnSize - end


    # This subroutine provides the ability to scroll up and down in the chat
    # window.
    #
    # !self : instance of self
##    def scroll_window(self,scroll_function = SCROLL_DOWN_ONE_LINE):
##        sy = self.chattextpointer
##        print "scroll_window:  sy = ", sy
##        (dummy,vh) = self.chatwnd.GetVirtualSize()
##        print "virt size = ", dummy, vh
##        (pw,ph) = self.chatwnd.GetScrollPixelsPerUnit()
##        print "scroll pixel units = ", pw, ph
##        (dummy,h) = self.chatwnd.GetClientSizeTuple()
##        print "client size = ", dummy, h


    def scroll_down(self):
        (vw,vh) = self.chatwnd.GetVirtualSize()
        (w,h) = self.chatwnd.GetClientSizeTuple()
        h_dif = vh-(h)
        (pw,ph) = self.chatwnd.GetScrollPixelsPerUnit()
        self.chatwnd.Scroll(-1,h_dif/ph)


    ###### message helpers ######

    def system_message(self,text):
        self.send_chat_message(text,chat_msg.SYSTEM_MESSAGE)
        self.SystemPost(text)

    def info_message(self,text):
        self.send_chat_message(text,chat_msg.INFO_MESSAGE)
        self.InfoPost(text)

    def get_gms(self):
        the_gms = []
        for playerid in self.session.players:
            if len(self.session.players[playerid])>7:
                if self.session.players[playerid][7]=="GM" and self.session.group_id != '0':
                    the_gms += [playerid]
        return the_gms

    def emote_message(self,text):
        text = self.ParseDice(text)
        # Heroman - apply any filtering selected
        text=self.ParseFilter(text)
        try:
            if self.to_gm.GetValue():
                the_gms = self.get_gms()
                for each_gm in the_gms:
                    prfx += self.session.players[each_gm][0]+" "
                    self.send_chat_message(text,chat_msg.EMOTE_MESSAGE, each_gm)
            else:
                self.send_chat_message(text,chat_msg.EMOTE_MESSAGE)
        except:
            self.send_chat_message(text,chat_msg.EMOTE_MESSAGE)
        if self.settings.get_setting('Toolbar_On') == "1":
            try:
                alias=self.aliasList.GetStringSelection();
            except:
                alias=""
            if alias!="" and alias!=DEFAULT_ALIAS_NAME:
                name = alias
            else:
                name = self.chat_display_name(self.session.get_my_info())
        else:
            name = self.chat_display_name(self.session.get_my_info())
        text = "** " + name + " " + text + " **"
        try:
            if self.to_gm.GetValue():
                 #text = prfx+text
                pass
        except:
            pass
        self.EmotePost(text)


    def whisper_to_players(self,text,player_ids, usealias=false):
        tabbed_whispers_p = self.settings.get_setting("tabbedwhispers")
        try:
            activeidx = self.parent.GetSelection()
        except:
            activeidx = 0
        if activeidx != 0:
            pass
        else:
            text = self.ParseDice(text)

        # Heroman - apply any filtering selected
        text=self.ParseFilter(text)
        player_names = ""
        # post to our chat before we colorize
        for m in player_ids:
            id = m.strip()
            if self.session.is_valid_id(id):
                returned_name = self.session.get_player_by_player_id(id)[0]
                player_names += returned_name
                player_names += ", "
            else:
                player_names += " Unknown!"
                player_names += ", "
        comma = ","
        comma.join(player_ids)
        if (self.sendtarget == "all"):
          self.InfoPost("<i>whispering to "+ player_names + " " + text + "</i> ")
        # colorize and loop, sending whisper messages to all valid clients
        text = self.colorize(self.mytextcolor, text)
        for id in player_ids:
            id = id.strip()
            if self.session.is_valid_id(id):
                self.send_chat_message(text,chat_msg.WHISPER_MESSAGE,id,usealias)
            else:
                self.InfoPost(id + " Unknown!")

    def send_chat_message(self,text,type=chat_msg.CHAT_MESSAGE,player_id="all",usealias=false):
        msg = chat_msg.chat_msg()
        msg.set_text(text)
        msg.set_type(type)
        if self.settings.get_setting('Toolbar_On') == "1" and usealias==false:
            try:
                alias=self.aliasList.GetStringSelection()
            except:
                alias=""
        elif usealias != false:
            alias = usealias
        else:
            alias = ""
        if alias!="" and alias!=DEFAULT_ALIAS_NAME:
            playername = self.settings.get_setting( 'player' )
            msg.set_alias(alias)
            self.chat_cmds.docmd( "/name "+ alias)
            self.session.send(msg.toxml(),player_id)
            self.chat_cmds.docmd( "/name "+ playername)
        else:
            self.session.send(msg.toxml(),player_id)
        del msg

    #### incoming chat message handler #####

    def post_incoming_msg(self,msg,player):
        # pull data
        type = msg.get_type()
        text = msg.get_text()
        alias = msg.get_alias()
#----------------------------------------
# Posted check added by Digitalxero
# This is for the plugin ability it
# lets the system know wether it needs
# to post the message or not
#----------------------------------------
        posted = 0

        # who sent us the message?
        if alias:
            display_name = alias
        elif player:
            display_name = self.chat_display_name(player)
        else:
            display_name = "Server Administrator"

                #following added by mDuo13
        #########receive_msg()###########
        for plugin_fname in orpg.plugins.frame.enabled_plugins.keys():
            plugin = orpg.plugins.frame.enabled_plugins[plugin_fname]
            try:
                text, type, display_name, posted = plugin.receive_msg(self, text, type, display_name, player)
            except Exception, e:
                if str(e) != "'module' object has no attribute 'receive_msg'":
                    #print e
                    traceback.print_exc()
        #end mDuo13 added code

        #image stripping for players' names
        strip_img = self.settings.get_setting("Show_Images_In_Chat")
        if (strip_img == "0"):
            #text = chat_util.strip_img_tags(text)#moved back to original location
            display_name = chat_util.strip_img_tags(display_name)
        #end image stripping. --mDuo13, July 11th, 2005

        # default sound
        recvSound = "RecvSound"

        # act on the type of messsage
        if (type == chat_msg.CHAT_MESSAGE) and (posted == 0):
            text = "<B>"+display_name+"</B>: " + text
            self.Post(text)

        elif (type == chat_msg.WHISPER_MESSAGE) and (posted == 0):
            tabbed_whispers_p = self.settings.get_setting("tabbedwhispers")
            displaypanel = self
            whisperingstring = " (whispering): "
            panelexists = 0
            if (tabbed_whispers_p == "1"):
                try:
                  # Check to see if parent notebook already has a private tab for player
                    for panel in self.parent.panel_list:
                       if (panel.sendtarget == player[2]):
                            displaypanel = panel
                            idx = self.parent.get_tab_index(displaypanel)
                            activeidx = self.parent.GetSelection()
                            if(activeidx != idx):
                                self.parent.SetPageImage(idx, self.parent.chatAttentionIdx)
                            panelexists = 1
                            break
                       # if not, have parent notebook create a new tab
                    if (panelexists == 0):
                        displaypanel = self.parent.create_private_tab(player[2])
                        cidx = self.parent.GetSelection()
                        nidx = self.parent.get_tab_index(displaypanel)
                        self.parent.SetSelection(nidx)
                        self.parent.SetSelection(cidx)
                        self.parent.SetPageImage(nidx, self.parent.chatAttentionIdx)

                    if (len(self.parent.panel_list)>0):
                        text = "<B>"+display_name+"</B>: " + text
                        displaypanel.Post(text)
                        recvSound = "WhisperSound"
                except:
                    # no tabbed whispering feature
                    text = "<i><B>" + display_name + "</B>" + whisperingstring + text + "</i>"
                    self.Post(text)
                    recvSound = "WhisperSound"
            else:
                # no tabbed whispering feature
                text = "<i><B>" + display_name + "</B>" + whisperingstring + text + "</i>"
                self.Post(text)
                recvSound = "WhisperSound"

        elif (type == chat_msg.EMOTE_MESSAGE) and (posted == 0):
            text = "** " + display_name + " " + text + " **"
            self.EmotePost(text)
        elif (type == chat_msg.INFO_MESSAGE) and (posted == 0):
            text = "<B>"+display_name+"</B>: " + text
            self.InfoPost(text)
        elif (type == chat_msg.SYSTEM_MESSAGE) and (posted == 0):
            text = "<B>"+display_name+"</B>: " + text
            self.SystemPost(text)

        # playe sound
        sound_file = self.settings.get_setting(recvSound)
        sound_player = orpg.tools.orpg_sound.orpg_sound(self.settings.get_setting("UnixSoundPlayer"));
        sound_player.play(sound_file)

    #### Posting helpers #####

    def InfoPost(self,s):
        self.Post(self.colorize(self.infocolor, s))

    def SystemPost(self,s):
        self.Post(self.colorize(self.syscolor, s))

    def EmotePost(self,s):
        self.Post(self.colorize(self.emotecolor, s))

    def Purge_buffer(self):
        x = self.chatbuffer[len(self.chatbuffer)-self.buffersize:]
        self.chatbuffer=x
        self.InfoPost("Buffer purged")

    #### Standard Post method #####

    def Post(self,s="",send=false,myself=false, alias=false):
        strip_p = self.settings.get_setting("striphtml")
        strip_img = self.settings.get_setting("Show_Images_In_Chat")#moved back 7-11-05. --mDuo13
        if (strip_p == "1"):
            s = strip_html(s)
        if (strip_img == "0"):
            s = chat_util.strip_img_tags(s)
        s = chat_util.simple_html_repair(s)
        s = chat_util.strip_script_tags(s)
        s = chat_util.strip_li_tags(s)
        s = chat_util.strip_body_tags(s)#7-27-05 mDuo13
        s = chat_util.strip_misalignment_tags(s)#7-27-05 mDuo13
        self.storedata = 0

        #following added by mDuo13
        if myself:
            #########send_msg()#############
            for plugin_fname in orpg.plugins.frame.enabled_plugins.keys():
                plugin = orpg.plugins.frame.enabled_plugins[plugin_fname]
                try:
                    s, send = plugin.send_msg(self, s, send)
                except Exception, e:
                    if str(e) != "'module' object has no attribute 'send_msg'":
                        #print e
                        traceback.print_exc()
        #########post_msg() - other##########
        if not myself and not send:
            for plugin_fname in orpg.plugins.frame.enabled_plugins.keys():
                plugin = orpg.plugins.frame.enabled_plugins[plugin_fname]
                try:
                    s = plugin.post_msg(self, s, myself)
                except Exception, e:
                    if str(e) != "'module' object has no attribute 'post_msg'":
                        #print e
                        traceback.print_exc()
        #end mDuo13 added code
        if myself:
            name = ""
            if self.settings.get_setting('Toolbar_On') == "1" and alias == false:
                try:
                    alias=self.aliasList.GetStringSelection();
                except:
                    alias="" # toolbar status could have changed this session
                if alias!="" and alias!=DEFAULT_ALIAS_NAME:
                    alt=self.session.get_my_info()
                    name = "<B>"+self.chat_display_name([alias,alt[1],alt[2]])+"</B>: "
                else:
                    name = "<B>"+self.chat_display_name(self.session.get_my_info())+"</B>: "
            elif alias != false:
                alt=self.session.get_my_info()
                name = "<B>"+self.chat_display_name([alias,alt[1],alt[2]])+"</B>: "
            else:
                name = "<B>"+self.chat_display_name(self.session.get_my_info())+"</B>: "
            s = self.colorize(self.mytextcolor, s)
        else:
            name = ""
        #following line based on sourceforge patch #880403 from mDuo
        # EDIT: Had to rework blank line check to handle malformed HTML throwing error.
        #       this limits the effectiveness of this check -SD
        lineHasText = 1
        try:
            lineHasText = strip_html(s).replace("&nbsp;","").replace(" ","").strip()!=""
        except:
            # HTML parser has errored out (most likely). Being as all we are doing is
            # scanning for empty/blank lines anyway there is no harm in letting a
            # troublesome message though. Worst case is a blank line to chat.
            lineHasText = 1
        if lineHasText:
            #following added by mDuo13
            if myself:
                ########post_msg() - self #######
                s2 = s#this exists to prevent the post_msg() function from modifying outgoing messages
                for plugin_fname in orpg.plugins.frame.enabled_plugins.keys():
                    plugin = orpg.plugins.frame.enabled_plugins[plugin_fname]
                    try:
                        s2 = plugin.post_msg(self, s2, myself)
                    except Exception, e:
                        if str(e) != "'module' object has no attribute 'post_msg'":
                            #print e
                            traceback.print_exc()
                if s2!="":
                    self.chatbuffer.append(self.TimeIndexString() + name +  s2 + "<br>")
                    log( self.settings, name + s2 )
            else:
                #end mDuo13 added code
                #following two lines tabbed in by mDuo13
                self.chatbuffer.append( self.TimeIndexString() + name +  s + "<br>")
                log( self.settings, name + s )

        purge_at = (self.settings.get_setting('PurgeAtBuffersizeTimesLines'))
        if purge_at == 0:
            purge_at = 10000
            if (len(self.chatbuffer) > self.buffersize*purge_at):
                self.Purge_buffer()

        buffertext = "<body bgcolor='" + self.bgcolor + "'text='" + self.textcolor + "'>"

        if self.bufferpointer == 0: ## we're not in the chat history
            hipointer = self.hibuffer = len(self.chatbuffer)
            lowpointer = self.lowbuffer = self.hibuffer - self.buffersize
            if self.lowbuffer < 0:
                lowpointer = self.lowbuffer = 0
        else: ## we are in the chat history
            hipointer = self.hibuffer - self.bufferpointer
            lowpointer = hipointer - self.buffersize
            if lowpointer < 0:
                lowpointer = 0
            buffertext = buffertext + self.colorize(self.infocolor, " ** You are viewing lines " + str(lowpointer) + "-" + str(hipointer) + " of your chat buffer.<BR>")
        for bufferslice in self.chatbuffer[lowpointer:hipointer]:
            buffertext = buffertext + bufferslice + "\n"

        #buffertext = string.replace(buffertext,'"',"'")
        buffertext = self.replace_quotes(buffertext)
        #print buffertext
##        self.chatwnd.SetPage(buffertext)

        if self.lockscroll == 0:
             self.chatwnd.SetPage(buffertext)
             self.scroll_down()
        else:
             self.storedata = buffertext
        if send:
             #self.session.send(s)
             if (self.sendtarget == "all"):
                 self.send_chat_message(s, usealias=alias)
             else:
                 self.whisper_to_players(s, [self.sendtarget], usealias=alias)

    #
    # TimeIndexString()
    #
    # time indexing for chat display only (don't log time indexing)
    # added by Snowdog 4/04
    def TimeIndexString(self):
        try:
            mtime = ""
            if self.settings.get_setting('Chat_Time_Indexing') == "0":
                pass
            elif self.settings.get_setting('Chat_Time_Indexing') == "1":
                mtime = time.strftime("%I:%M ", time.localtime())
            elif self.settings.get_setting('Chat_Time_Indexing') == "2":
                mtime = time.strftime("%I:%M:%S ", time.localtime())
            else:
                # this allows users to fully customize their time strings if desired.
                userformat = self.settings.get_setting('Chat_Time_Indexing')+" "
                mtime = time.strftime( userformat, time.localtime())
            return mtime
        except:
            return "[ERROR]"



    ####  Post with parsing dice ####
    def ParsePost(self,s,send=false,myself=false,alias=false):
        s = self.NormalizeParse(s)
        self.Post(s,send,myself,alias)

    def NormalizeParse(self,s):
        s = self.ParseNode(s)
        s = self.ParseDice(s)
        s = self.ParseFilter(s)
        return s

    def ParseFilter(self,s):
        if self.settings.get_setting('Toolbar_On') == "1":
            try:
                filter=self.filterList.GetStringSelection();
                if filter!="" and filter!=DEFAULT_FILTER_NAME:
                    s=self.applyFilter(s,filter)
            except:
                pass  # this try except catches if the user hasn't rebooted openrpg since they turned on the Toolbar.
        return s

    def ParseNode(self,s):
        cur_loc = 0
        "Parses player input for embedded nodes rolls"
        while 1:
            loc = s.find("!@",cur_loc)
            if loc == -1: break
            cur_loc = loc+1
            nex = s.find("!@",cur_loc)
            myend = s.find("@!",cur_loc)
            if (nex > myend) or (nex == -1):
                old = s[loc:myend+2]
                if len(old) >0:
                    newstr = self.resolve_nodes(old)
                    s = s.replace(old,newstr,1)
                    cur_loc = 0
            else:
                cur_loc = loc+1
        return s


    def ParseDice(self,s):

        "Parses player input for embedded dice rolls"
        cur_pos = 0
        while 1:
            start = s.find('[',cur_pos)
            end = s.find(']',cur_pos)
            #break if no []
            if not start >= 0 or not end >= 0 :
                break
            old = s[start:end+1]

            try:
                #newstr = self.dice.roll(s[start+1:end])

                newstr = self.PraseUnknowns(s[start+1:end])

                if newstr[0].upper() == 'Q':
                    quiet = 1
                    newstr= newstr[1:]
                else: quiet = 0

                if self.settings.get_setting("dievars") == "1":
                    roll = "[" + newstr + "]"
                else:
                    roll = old

                newstr = self.roller_manager.resolveDieStr(newstr)
                if not quiet:
                    newstr = roll + " -> " + newstr

            except Exception, e:
                print e
                newstr = "[Bad dice format] - [" + newstr +"]"

            s = s[:start] + s[start:].replace(old,newstr,1)
            cur_pos = start + len(newstr)
        return s


    def PraseUnknowns(self,s):
        newstr = "0"
        dlg = wxTextEntryDialog(self,"Missing Value?",newstr)
        while 1:
            pos = s.find('?')
            #break if no []
            if pos < 0:
                break
            oldstr = s[pos:pos+1]

            if dlg.ShowModal() == wxID_OK:
                newstr = dlg.GetValue()
            s = s.replace(oldstr,newstr,1)
        dlg.Destroy()
        return s



    # This subroutine builds a chat display name.
    #

    def chat_display_name(self,player):
        if self.settings.get_setting("ShowIDInChat") == "0":
            display_name = player[0]
        else:
            display_name = "("+player[2]+") "+player[0]
        return display_name


    # This subroutine will get a hex color and return it, or return nothing
    #
    def get_color(self):
        data = wxColourData()
        data.SetChooseFull(true)
        dlg = wxColourDialog(self, data)
        if dlg.ShowModal() == wxID_OK:
            data = dlg.GetColourData()
            (red,green,blue) = data.GetColour().Get()
            hexcolor = self.r_h.hexstring(red, green, blue)
            dlg.Destroy()
            return hexcolor
        else:
            dlg.Destroy()
            return None
    # def get_color - end


    def replace_quotes(self,s):

        in_tag = 0
        i = 0
        rs = s[:]

        for c in s:
            if c == "<":
                in_tag += 1
            elif c == ">":
                if in_tag:
                    in_tag -= 1
            elif c == '"':
                if in_tag:
                    rs = rs[:i] + "'" + rs[i+1:]
            i += 1

        return rs

    def resolve_nodes(self,s):
        value = ""
        s = s[2:len(s)-2]
        node_path_list = s.split("::")
        node = None
        gametree=self.myopenrpg.get_component('tree')
        nodes = gametree.master_dom.getElementsByTagName('nodehandler')
        for nnodes in node_path_list:
            for entry in nodes:
                t_node = entry
                atts = t_node.getAttributeKeys()
                for attributess in range(0,len(atts)):
                    Value = t_node.getAttribute('name')
                    if Value == nnodes:
                        break
                if Value == nnodes:
                    break
            nodes = entry.getElementsByTagName('nodehandler')
        if Value != nnodes:
            return s
        nodetype = entry.getAttribute('class')
        if nodetype == "textctrl_handler":
            x = t_node.getElementsByTagName('text')
            t_node = x[0]._get_firstChild()
            value = t_node._get_nodeValue()
#        elif nodetype == "rpg_grid_handler":
#            x = t_node.getElementsByTagName('grid')
#            for grids in x:
#                t_node = x[0]._get_firstChild()
#                gn =x.getAttribute('name')
#        if gn == s1[1]:
#                    value = t_
#        print entry.getAttribute('class')
        if value =="":
            value = s
        return value


# ---------------------------------------------------------------------
# Some inline image data for the chat-tabs.
# This data was produced by running:  print repr(file('image.png').read())

def getAttentionData():
    return \
'\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x10\x00\x00\x00\x10\x08\x02\x00\x00\x00\x90\x91h6\x00\x00\x00\tpHYs\x00\x00\x0b\x12\x00\x00\x0b\x12\x01\xd2\xdd~\xfc\x00\x00\x01\x9dIDATx\x9c}\x8f]K\x94a\x14E\xd7>\xef\xe3h4\x83\x13\x89\x8a\x91\x1fEbQ]\xd4]\x7f\xbd_\x10\x84]\x14dZ\x84F9:\x96\xa3\x0ej\xd3<gwaB\x90v\xae\xd7\xda{\x1f\xd9il\x8f\xe4c\xb3g\xef;{\xc9\t\xd8L6tC\xf3\xd2\x82\xa3+uD\x14\xa80\xc6\xfb\x99\x9f2\xdf9\xb7\x93\x1d\xfbP\xb6\xe3\x06\xccT-D<\x14\x8f"\xeeI\xedb\xdb\xeeg\xbe\xadu=\xfd*s+\xf9.*8\xab\xc5\xcd\x88\xb9\xc2\xae8\x12#\xc7j1G\xce\x8f5\xd7\xc7\xf9\xd2\xde\xc2\xa7\x92A\xe0P\xc0\x89=\xfcU\x07\xc5\xe7Ac\xda\xc5\xd9\xaf~\x9f\xf9\xda\xfe\x00C\xd4\x82\x04.\x1c\x08H|0\xae\xeb\xa8[X\x08\xb3g\x7f\xae\xde\x80S(\x97\xf4\xdf\')P/s#\xd9\t\xbb\x8f{p \xf4\x0f\xfaG0\xd8i\xef\xa6\xbf\x06\xfai\x0f\xc9\x91\xaf\xc1/[\xaa\x19\xc3(.3\nn\xfek\\h\x84\xdc\x11]4\x85\xea\xb5\xb0!\x1b\xd1\x11\xd3!\xcd\x87V\x1a\xe6q\x00W\x0f\x93\xa0\x15Z\x0c-F0\xa7\xb8\x1f\xcd\x13i\xda\xaeW=\x9e\xd6\xd8\xb1\xa8X\x13+\x85\xe86^s\xf4\x93\xa1\xea\x1bq\x88m]\x94\t,Z\xf6\xf2DyQ\xf4,b\xa9\xa06\xb1\x14>+\x9e\xa8\xcd-{\xb3\xd6/\xf2\xb9e4\x15t\xa2\xb9+=mx\x1e\xf1\x18\xcd\xcaNS\xc9\x83\xcc^\xf5\xa6\xd9\xc6\xdf\x9c\xc7\xc6\x8a\x89\xf0,\xcd\x9d\xd0\x83\xd0\xb24\'M\xca6`\x8f\xf1\x19\x0c\xec\x1f\xf6\xc0\x9c\xd8\x19\x9a\x14m\xe26\x9a\t\xb5MK\xe87\xb49\xfd!Vt\xf5f\x00\x00\x00\x00IEND\xaeB`\x82'

def getAttentionBitmap():
    return wxBitmapFromImage(getAttentionImage())

def getAttentionImage():
    stream = cStringIO.StringIO(getAttentionData())
    return wxImageFromStream(stream)


def getNormalData():
    return \
'\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x10\x00\x00\x00\x10\x08\x02\x00\x00\x00\x90\x91h6\x00\x00\x00\tpHYs\x00\x00\x0b\x12\x00\x00\x0b\x12\x01\xd2\xdd~\xfc\x00\x00\x01\x9eIDATx\x9c\x8d\xcb\xcfK\xd3q\x1c\xc7\xf1\xd7\xeb\xfd\xf9\xda\x886\x1c$\xb9\x8c\x99V\xb3\x15\x16\xe8\x8eI\x87\x0e\xb1\xf5\x0fx\xebo\x88N\xfd\x03\xd1\xbf\xe1\xcds\x04\x99\x04\x15\x11A\x87\x141b?\x88pPN\x17\xe5\\\x10\xad\xef\xe7\xfd\xea\xd0\xa5\xc3\x98>\xcf\x8f\'%\x17$\r\xa8CaO\xda\x97w\x1c}@B& o,\x90S\xb2<\x99#,\x01"\x90B\xfb\xee\x9f\xdc?\xc8w\x1cm\xe9\x07%\xd9I`"r\xca\xec\nq\xd5\xec\x02\x99M$I]\xf7\xed\x187\\\xef\xdc[\xd5\xb9\x06\xfe\xebYk!\xc1.\xd1#\x06\xb29F\xef*n\xa7\xfe&\x8d\xcf\xab\xa5\xd7\x18Vmy\xf5\xfe\xc3\'I\xb8i\xb6\xc44\xfd\x18\xfde\x8c\xeb\xb7/=\x1e\xaa\xff\xb5\xd6(\'\xc9\xad\xc4\xaa&\xecI\x9f\xa3\xea#4\x80\xda\xe5\xba{\xdd\xd16\xa9\x0buj\xa5\xc6\xe8\x01\x80\xb4\xeb\xfab\xe0o\xe9\xe7\x91\x1a@\xbfw\n\x18\x18\x00\x80\xc7\x19\x00\x900*G\xe4\x8f\xa3s\xe3N\x8c\x1bY0\xce\xae7\x17\x8e\x1c\x8c\xd3\xc6i3L\xd2.Z\xb8v\xb6\xd8\x1e\xa1\x9f\xb6\x16ieb\xd6`\xf9\xc0r\xb0\xca\xca\xab{\xb5\xe5\xd5\xa1z\xadY\x19\x0b7\x12[4;O\xd7\x1fx\xcfU\x8fq+\xea\xbd\xd4|p\xf7\xd1\xe6\xdb%\x00\x85b{\xe5\xc5\x1d\x0bE\xf2z@%\xd8<m\x86\x92\x0b\x11\xfe\xcd\xbd\x13\xd5\x14v\xa0\xaf\xf2CA\xb41\xd3\x19\x84s\xc6\x92q\x86\x9c$3\x94\x04@J\xa1_\xc0\x81\xf4]:\x10\xfa\x92\x1b3D\x16v\x1a\x9c0f\x85\x13\x04\xff\x02\xf1\xb8\xd1\xec\x08\xfa\x9c\x89\x00\x00\x00\x00IEND\xaeB`\x82'


def getNormalBitmap():
    return wxBitmapFromImage(getNormalImage())

def getNormalImage():
    stream = cStringIO.StringIO(getNormalData())
    return wxImageFromStream(stream)

