;;;-*-Mode: LISP; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   Portions copyright (c) 2001 Clozure Associates.
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;



;; L1-pathnames.lisp
;-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_
;ANSI CL logical pathnames

(in-package :ccl)


;These are used by make-pathname
(defun %verify-logical-component (name type)
  (when (and name (neq name :unspecific))
    (setq name (ensure-simple-string name))
    (when (or (eql 0 (length name))
              (%str-member *pathname-escape-character* name) ;; Hmm, why?
              (%path-mem "/;" name))
      (error "Illegal logical pathname ~A component ~S" type name)))
  name)


(defun verify-logical-host-name (host)
  (or (and host
	   (%verify-logical-component host "host")
	   (%str-assoc host %logical-host-translations%)
	   host)
      (host-error host)))

(defun %logical-version-component (version)
  (if (or (fixnump version)
          (stringp version)
          (memq version '(nil :wild :newest :unspecific)))
    version
    (require-type version '(or fixnum string (member nil :wild :newest :unspecific)))))

(defun logical-pathname-translations (host)
  (setq host (verify-logical-host-name host))
  (let ((translations (%str-assoc host %logical-host-translations%)))
    (unless translations (host-error host))
    (%cdr translations)))

(defun logical-host-p (host)
  (%str-assoc host %logical-host-translations%))

(defun host-error (host) ; supposed to be a type-error
  (signal-type-error host  '(satisfies logical-host-p) "~S is not a defined logical host"))

(defun set-logical-pathname-translations (host list)
  (setq host (%verify-logical-component  host "host"))
  (let ((old (%str-assoc host %logical-host-translations%))
	(new (let ((%logical-host-translations% (cons (list host) %logical-host-translations%)))
	       ;; Do this in the context when host is defined, so no errors.
	       (mapcar #'(lambda (trans)
			   (destructuring-bind (from to &rest ignored) trans
			     (declare (ignore ignored))
			     (let ((from-path (parse-namestring from host))
				   (to-path (pathname to)))
			       (list (require-type from-path 'logical-pathname) to-path))))
		       list))))
    (if old
      (progn (%rplaca old host) (%rplacd old new))
      (push (cons host new) %logical-host-translations%)))
  list)

(defsetf logical-pathname-translations set-logical-pathname-translations)

; doesnt check if already there - adds at front 
(defun add-logical-pathname-translation (host translation)
  (let ((trans (%str-assoc host  %logical-host-translations%)))
    (if (not trans)
      (set-logical-pathname-translations host (list translation))
      (let ((new (debind (from to &rest ignored) translation
                        (declare (ignore ignored))
                        (list (parse-namestring from host) (pathname to)))))
        (rplacd trans (cons new (cdr trans)))
        (cdr trans)))))

#|
(defun maybe-diddle-case (thing diddle-p)
  (if (and diddle-p (not (or (symbolp thing) (integerp thing))))
      (labels ((check-for (pred in)
		 (typecase in
		   (pattern
		    (dolist (piece (pattern-pieces in))
		      (when (typecase piece
			      (simple-string
			       (check-for pred piece))
			      (cons
			       (case (car in)
				 (:character-set
				  (check-for pred (cdr in))))))
			(return t))))
		   (list
		    (dolist (x in)
		      (when (check-for pred x)
			(return t))))
		   (simple-base-string
		    (dotimes (i (length in))
		      (when (funcall pred (schar in i))
			(return t))))
		   (t nil)))
	       (diddle-with (fun thing)
		 (typecase thing
		   (pattern
		    (make-pattern
		     (mapcar #'(lambda (piece)
				 (typecase piece
				   (simple-base-string
				    (funcall fun piece))
				   (cons
				    (case (car piece)
				      (:character-set
				       (cons :character-set
					     (funcall fun (cdr piece))))
				      (t
				       piece)))
				   (t
				    piece)))
			     (pattern-pieces thing))))
		   (list
		    (mapcar fun thing))
		   (simple-base-string 
		    (funcall fun thing))
		   (t
		    thing))))
	(let ((any-uppers (check-for #'upper-case-p thing))
	      (any-lowers (check-for #'lower-case-p thing)))
	  (cond ((and any-uppers any-lowers)
		 ;; Mixed case, stays the same.
		 thing)
		(any-uppers
		 ;; All uppercase, becomes all lower case.
		 (diddle-with #'(lambda (x) (if (stringp x)
						(string-downcase x)
						x)) thing))
		(any-lowers
		 ;; All lowercase, becomes all upper case.
		 (diddle-with #'(lambda (x) (if (stringp x)
						(string-upcase x)
						x)) thing))
		(t
		 ;; No letters?  I guess just leave it.
		 thing))))
      thing))

(defun substitute-into (pattern subs diddle-case)
  (declare (type pattern pattern)
	   (type list subs)
	   (values (or simple-base-string pattern)))
  (let ((in-wildcard nil)
	(pieces nil)
	(strings nil))
    (dolist (piece (pattern-pieces pattern))
      (cond ((simple-string-p piece)
	     (push piece strings)
	     (setf in-wildcard nil))
	    (in-wildcard)
	    (t
	     (setf in-wildcard t)
	     (unless subs
	       (error "Not enough wildcards in FROM pattern to match ~
		       TO pattern:~%  ~S"
		      pattern))
	     (let ((sub (pop subs)))
	       (typecase sub
		 (pattern
		  (when strings
		    (push (apply #'concatenate 'simple-string
				 (nreverse strings))
			  pieces))
		  (dolist (piece (pattern-pieces sub))
		    (push piece pieces)))
		 (simple-string
		  (push sub strings))
		 (t
		  (error "Can't substitute this into the middle of a word:~
			  ~%  ~S"
			 sub)))))))

    (when strings
      (push (apply #'concatenate 'simple-string (nreverse strings))
	    pieces))
    (values
     (maybe-diddle-case
      (if (and pieces (simple-string-p (car pieces)) (null (cdr pieces)))
	  (car pieces)
	  (make-pattern (nreverse pieces)))
      diddle-case)
     subs)))

(defun didnt-match-error (source from)
  (error "Pathname components from Source and From args to TRANSLATE-PATHNAME~@
	  did not match:~%  ~S ~S"
	 source from))

(defun compute-directory-substitutions (orig-source orig-from)
  (let ((source orig-source)
	(from orig-from))
    (let* ((subs ()))
      (loop
	(unless source
	  (unless (every #'(lambda (x) (eq x :wild-inferiors)) from)
	    (didnt-match-error orig-source orig-from))
	  (push () subs)
	  (return))
	(unless from (didnt-match-error orig-source orig-from))
	(let ((from-part (pop from))
	      (source-part (pop source)))
	  (typecase from-part
	    (pattern
	     (typecase source-part
	       (pattern
		(if (pattern= from-part source-part)
		    (push source-part subs)
		    (didnt-match-error orig-source orig-from)))
	       (simple-string
		(multiple-value-bind
		    (won new-subs)
		    (pattern-matches from-part source-part)
		  (if won
		      (dolist (sub new-subs)
			(push sub subs))
		      (didnt-match-error orig-source orig-from))))
	       (t
		(didnt-match-error orig-source orig-from))))
	    ((member :wild)
	     (push source-part subs))
	    ((member :wild-inferiors)
	     (let ((remaining-source (cons source-part source)))
	       (let* ((res ()))
		 (loop
		   (when (directory-components-match remaining-source from)
		     (return))
		   (unless remaining-source
		     (didnt-match-error orig-source orig-from))
		   (push (pop remaining-source) res))
		 (push (nreverse res) subs)
		 (setq source remaining-source))))
	    (simple-string
	     (unless (and (simple-string-p source-part)
			  (string= from-part source-part))
	       (didnt-match-error orig-source orig-from)))
	    (t
	     (didnt-match-error orig-source orig-from)))))
      (nreverse subs))))

(defun translate-component (source from to diddle-case) 
  (typecase to
    (pattern
     (typecase from
       (pattern
	(typecase source
	  (pattern
	   (if (pattern= from source)
	       source
	       (didnt-match-error source from)))
	  (simple-string
	   (multiple-value-bind
	       (won subs)
	       (pattern-matches from source)
	     (if won
	       (values (substitute-into to subs diddle-case))
	       (didnt-match-error source from))))
	  (t
	   (maybe-diddle-case source diddle-case))))
       ((member :wild)
	(values (substitute-into to (list source) diddle-case)))
       (t
	(if (components-match source from)
	  (maybe-diddle-case source diddle-case)
	  (didnt-match-error source from)))))
    ((member nil :wild)
     (maybe-diddle-case source diddle-case))
    (t
     (if (components-match source from)
       to
       (didnt-match-error source from)))))

(defun translate-directories (source from to diddle-case)
  (if (not (and ;;source
	        to
		from))
    (or to
	(mapcar #'(lambda (x) (maybe-diddle-case x diddle-case)) source))
    (let* ((res ()))
      (push (first (or source from)) res)
      (let ((subs-left (compute-directory-substitutions (rest source)
							(rest from))))
	(dolist (to-part (rest to))
	  (typecase to-part
	    ((member :wild)
	     (assert subs-left)
	     (let ((match (pop subs-left)))
	       (when (listp match)
		 (error ":WILD-INFERIORS not paired in from and to ~
			   patterns:~%  ~S ~S" from to))
	       (maybe-diddle-case match diddle-case)))
	    ((member :wild-inferiors)
	     (let ((match (pop subs-left)))
	       (unless (listp match)
		 (error ":WILD-INFERIORS not paired in from and to ~
			   patterns:~%  ~S ~S" from to))
	       (dolist (x match)
		 (push (maybe-diddle-case x diddle-case) res))))
	    (pattern
	     (multiple-value-bind
		 (new new-subs-left)
		 (substitute-into to-part subs-left diddle-case)
	       (setf subs-left new-subs-left)
	       new))
	    (t (push to-part res)))))
      (nreverse res))))

(defun translate-pathname (source from-wildname to-wildname &key)
  "Use the source pathname to translate the from-wildname's wild and
   unspecified elements into a completed to-pathname based on the to-wildname."
  (declare (type path-designator source from-wildname to-wildname))
  (let* ((source (pathname source)))
    (let* ((from (pathname from-wildname)))
      (let* ((to (pathname to-wildname)))
	  (let* ((source-host (pathname-host source))
		 (to-host (pathname-host to))
		 (diddle-case nil))
	    (macrolet ((frob (field &optional (op 'translate-component))
			 `(let ((result (,op (,field source)
					     (,field from)
					     (,field to)
					     diddle-case)))
			    (if (eq result :error)
				(error "~S doesn't match ~S" source from)
				result))))
	      (cons-pathname
	       (frob pathname-directory translate-directories)
	       (frob %pathname-name)
	       (frob %pathname-type)
	       (or to-host source-host)
	       (frob pathname-version))))))))

(defun pattern= (pattern1 pattern2)
  (declare (type pattern pattern1 pattern2))
  (let ((pieces1 (pattern-pieces pattern1))
	(pieces2 (pattern-pieces pattern2)))
    (and (= (length pieces1) (length pieces2))
	 (every #'(lambda (piece1 piece2)
		    (typecase piece1
		      (simple-string
		       (and (simple-string-p piece2)
			    (string= piece1 piece2)))
		      (cons
		       (and (consp piece2)
			    (eq (car piece1) (car piece2))
			    (string= (cdr piece1) (cdr piece2))))
		      (t
		       (eq piece1 piece2))))
		pieces1
		pieces2))))

(defun pattern-matches (pattern string)
  (declare (type pattern pattern)
	   (type simple-string string))
  (let ((len (length string)))
    (labels ((maybe-prepend (subs cur-sub chars)
	       (if cur-sub
		   (let* ((len (length chars))
			  (new (make-string len :element-type 'base-char))
			  (index len))
		     (dolist (char chars)
		       (setf (schar new (decf index)) char))
		     (cons new subs))
		   subs))
	     (matches (pieces start subs cur-sub chars)
	       (if (null pieces)
		   (if (= start len)
		       (values t (maybe-prepend subs cur-sub chars))
		       (values nil nil))
		   (let ((piece (car pieces)))
		     (etypecase piece
		       (simple-string
			(let ((end (+ start (length piece))))
			  (and (<= end len)
			       (string= piece string
					:start2 start :end2 end)
			       (matches (cdr pieces) end
					(maybe-prepend subs cur-sub chars)
					nil nil))))
		       (list
			(ecase (car piece)
			  (:character-set
			   (and (< start len)
				(let ((char (schar string start)))
				  (if (find char (cdr piece) :test #'char=)
				      (matches (cdr pieces) (1+ start) subs t
					       (cons char chars))))))))
		       ((member :single-char-wild)
			(and (< start len)
			     (matches (cdr pieces) (1+ start) subs t
				      (cons (schar string start) chars))))
		       ((member :multi-char-wild)
			(multiple-value-bind
			    (won new-subs)
			    (matches (cdr pieces) start subs t chars)
			  (if won
			      (values t new-subs)
			      (and (< start len)
				   (matches pieces (1+ start) subs t
					    (cons (schar string start)
						  chars)))))))))))
      (multiple-value-bind
	  (won subs)
	  (matches (pattern-pieces pattern) 0 nil nil nil)
	(values won (reverse subs))))))

(defun components-match (thing wild)
  (declare (type (or pattern symbol simple-string integer) thing wild))
  (or (eq thing wild)
      (eq wild :wild)
      (typecase thing
	(simple-base-string
	 ;; String is matched by itself, a matching pattern or :WILD.
	 (typecase wild
	   (pattern 
	    (values (pattern-matches wild thing)))
	   (simple-base-string
	    (string= thing wild))))
	(pattern
	 ;; A pattern is only matched by an identical pattern.
	 (and (pattern-p wild) (pattern= thing wild)))
	(integer
	 ;; an integer (version number) is matched by :WILD or the same
	 ;; integer.  This branch will actually always be NIL as long is the
	 ;; version is a fixnum.
	 (eql thing wild)))))

(defun directory-components-match (thing wild)
  (or (eq thing wild)
      (eq wild :wild)
      (and (consp wild)
	   (or (and (= (length wild) 2)
		    (eq (car wild) :absolute)
		    (null thing)
		    (eq (cadr wild) :wild-inferiors))
	       (let ((wild1 (first wild)))
		 (if (eq wild1 :wild-inferiors)
		     (let ((wild-subdirs (rest wild)))
		       (or (null wild-subdirs)
			   (loop
			       (when (directory-components-match thing wild-subdirs)
				 (return t))
			       (pop thing)
			     (unless thing (return nil)))))
		     (and (consp thing)
			  (components-match (first thing) wild1)
			  (directory-components-match (rest thing)
						      (rest wild)))))))))


(defun pathname-match-p (in-pathname in-wildname)
  "Pathname matches the wildname template?"
  (declare (type path-designator in-pathname))
  (let* ((pathname (pathname in-pathname)))
    (let* ((wildname (pathname in-wildname)))
      (macrolet ((frob (field &optional (op 'components-match ))
		   `(or (null (,field wildname))
			(,op (,field pathname) (,field wildname)))))
	(and (or (null (pathname-host wildname))
		 (equal (pathname-host wildname) (pathname-host pathname)))
	     (frob pathname-directory directory-components-match)
	     (frob pathname-name)
	     (frob pathname-type)
	     (frob pathname-version))))))

|#

(defun %component-match-p (name wild) 
  (if (or (eq name :unspecific)(and (stringp name) (or  (string= name "*")(string= name "**"))))
    (setq name nil))  
  (if (or (eq wild :unspecific)(and (stringp wild) (or (string= wild "*")(string= wild "**"))))
    (setq wild nil))
  (cond ((null name) 
         (null wild))
        ((null wild)
         t)
        (t (%path-str*= name wild))))

(defun translate-directory (source from to reversible &optional thost)
  (declare (ignore thost)) ;; leftover from a mac kludge.
  (flet ((string-test (x y)(and (stringp x) (stringp y)(string= x y))))
    (let* ((result (translate-directory2 (cdr source)(cdr from)(cdr to) reversible))
           (relative-p (eq (car source) :relative)))
      (cond ((and (not relative-p)(eq result (cdr source))) source)
            ((and (not relative-p)(eq result (cdr to))) to)
            (t (cons (car (or to source from)) result))))))



(defun translate-directory2 (source from to reversible)
  ; we already know it matches
  (let (result srest match tfirst trest twild)
    (multiple-value-setq (tfirst trest twild)
			 (%split-ccdirectory to))
    (when (and to (not twild))
      (return-from translate-directory2 to))
    (multiple-value-bind (ffirst frest fwild)
			 (%split-ccdirectory from)
      (setq srest (nthcdr (length ffirst) source))
      (cond ((eq fwild '**)
	     (setq match (nth-value 1 (%pathname-match-dir1 srest frest t)))               
	     (cond ((eq twild '**)
		    (setq result (nconc tfirst match))
		    (setq srest (nthcdr (length match) srest)))
		   (t (return-from translate-directory2
			(translate-directory2 source (nconc ffirst match frest)
					      to reversible)))))
	    ((eq twild '**)
	     (let ((length (length tfirst)))
	       (setq srest (nthcdr length source))
	       (setq frest (nthcdr length from))
	       (setq  match (nth-value 1 (%pathname-match-dir1 srest trest t)))
	       (cond ((null  match)
		      (setq result tfirst))
		     (t (setq srest (nthcdr (setq length (length match)) srest))
			(setq frest (nthcdr length frest))
			(setq result (nconc tfirst match))))))
	    (t
	     (cond ((null fwild)
		    ; to has a wild component e.g. *abc, from is not wild
		    ; by defintion source is also not wild
		    ; which random source component gets plugged in here??
		    (setq srest (nthcdr (length tfirst) source))
		    (setq frest (nthcdr (length tfirst) source))))
	     (let ((part (translate-component
				(car srest) (car frest)(car trest) reversible)))
	       (if (null part)(setq result tfirst)
		   (progn
		     (setq part (list part))
		     (setq result (nconc tfirst part)))))
	     (setq srest (cdr srest) frest (cdr frest) trest (cdr trest))))
      (when trest 
	(let ((foo (translate-directory2 srest frest trest reversible)))
	  (when foo (setq result (nconc result foo))))))
    result))

; cc stands for cdr canonical
; ("abc" "**" "def" => ("abc") ("def")
; ("abc" "*de") => ("abc") ("*de")
(defun %split-ccdirectory (dir)
  (let ((pos 0) (wildp nil)(rest dir))
    (dolist (e dir)
      (case e
        (:wild (setq wildp '*))
        (:wild-inferiors 
         (setq wildp '**)
         (setq rest (cdr rest)))
	(:up nil)
        (t 
         (when (%path-mem "*" e)
           (cond ((string= e "**")
                  (setq rest (cdr rest))
                  (setq wildp '**))
                 ((eql 1 (length (the string e)))
                  (setq wildp '*))
                 (t (setq wildp t))))))
      (when wildp (return))
      (setq rest (cdr rest))
      (setq pos (%i+ 1 pos)))
    (cond ((not wildp)
           (values dir))
          (t (let (first)
               (when rest (setq rest (copy-list rest)))
               (dotimes (i pos)
                 (declare (fixnum i))
                 (push (car dir) first)
                 (setq dir (cdr dir)))
               (values (nreverse first) rest wildp))))))

; could avoid calling component-match-p by checking here maybe
; if "gazonk" "gaz*" "h*" => "honk"
; then "gazonk" "gaz*" "*" => "onk" or is it "gazonk" (per pg 625)
; I believe in symbolics land "gazonk" is a regular translation
; and "onk" is a reversible translation (achieved by not doing pg 625) AHH
; similarly is "a:" "a:**:" "**"  Nil or "a:" 
(defun translate-component (source from to &optional reversible)                   
  (let ((orig-to to))
    (cond 
     ((and (consp source)(consp from)) ; source and from both logical 
      (setq source (cadr source) from (cadr from)))
     ((or (consp source)(consp from)) ;  or neither
      #-bccl (error "Something non-kosher in translate pathname")
      ))
    (when (memq from '(:wild :wild-inferiors)) (setq from "*"))
    (when (memq source '(:wild :wild-inferiors))(setq source "*"))
    (when (memq to '(:wild :wild-inferiors))(setq to "*"))
    (cond ((consp to)(setq to (cadr to))))  ;??
    (cond ((and (stringp to)(not (%path-mem "*" to)))
           to)
          ((and (or (not reversible)(not (stringp source))) ; <<
                (or (null to)
                    (and (stringp to)(or (string= to "**")(string= to "*")))))
           source)
          ((eq to :unspecific) to)  ; here we interpret :unspecific to mean don't want it
          ((not (stringp source)) to)
          (t 
           (let ((slen (length source)) srest match spos result (f2 nil) snextpos)
             (multiple-value-bind (tfirst trest twild)
                                  (%split-component to)
               (cond ((and to (not twild))(return-from translate-component to)))
               (multiple-value-bind (ffirst frest fwild)
                                    (%split-component from)          
                 (cond (fwild
                        (setq spos (if ffirst (length ffirst) 0))       ; start of source hunk
                        (if frest (setq f2 (%split-component frest)))
                        (setq snextpos (if f2 (%path-member f2 source spos) slen))
                        (setq match (%substr source spos snextpos))
                        (if frest (setq srest (%substr source snextpos slen)))
                        (setq result (if tfirst (%str-cat tfirst match) match))
                        (when frest 
                          (let ((foo (translate-component srest frest trest reversible)))
                            (when foo (setq result (%str-cat result foo))))))
                       (t  ; to is wild, from and source are not
                        (setq result (if tfirst (%str-cat tfirst source) source))
                        (when trest (setq result (%str-cat result trest))))))
               (if (consp orig-to)(progn (error "shouldnt")(list :logical result)) result) ; 7/96
               ))))))


(defun %path-member (small big &optional (start 0))
  (let* ((end (length big))
         (s-end (length small))
         (s-start 1)
         (c1 (%schar small 0))
         (pstart start))
    (if (%i> s-end end)(return-from %path-member nil))
    (when (eql c1 *pathname-escape-character*)
      (setq c1 (%schar small 1))
      (setq s-start 2))      
    (while (and (progn (if (eql (%schar big pstart) *pathname-escape-character*)
                         (setq pstart (%i+ pstart 1)))
                       T)
                (%i< pstart end)
                (neq (%schar big pstart) c1))
      (setq pstart (%i+ pstart 1)))
    (if (neq c1 (%schar big pstart))(return-from %path-member nil))
    (setq start (%i+ pstart 1))
    (while (and (progn (if (eql (%schar big start) *pathname-escape-character*)
                         (setq start (%i+ 1 start)))
                       (if (eql (%schar small s-start) *pathname-escape-character*)
                         (setq s-start (%i+ 1 s-start)))
                       T)
                (%i< start end)
                (%i< s-start s-end)
                (eql (%schar big start)(%schar small s-start)))
      (setq start (%i+ start 1) s-start (%i+ s-start 1)))
    (cond ((= (the fixnum s-start) (the fixnum s-end))
            pstart)
          ((%i< start end)
            (%path-member small big (%i+ 1 pstart)))
          (T nil))))

(defun %split-component (thing &aux pos)
  ;"ab*cd*"  ->  "ab" "cd*"  
  (if (or (null thing)(eq thing :unspecific)(null (setq pos (%path-mem "*" thing))))
    (values thing nil nil)
    (let* ((len (length thing)))
      (declare (fixnum len))
      (values (if (%izerop pos) nil (%substr thing 0 pos))
              (cond ((eql len (%i+ pos 1)) nil)
                    (t 
                     (when (eq (%schar thing (+ pos 1)) #\*)
                       (setq pos (+ pos 1)))
                     (cond ((eql len (%i+ pos 1)) nil)
                           (t (%substr thing (%i+ pos 1) len)))))
              T))))

(defun translate-pathname (source from-wildname to-wildname &key reversible)
  (when (not (pathnamep source)) (setq source (pathname source)))
  (flet ((foo-error (source from)
	   (error "Source ~S and from-wildname ~S do not match" source from)))
    (let (r-host r-device r-directory r-name r-type r-version s-host f-host t-host)
      (setq s-host (pathname-host source))
      (setq f-host (pathname-host from-wildname))
      (setq t-host (pathname-host to-wildname))
      (if (not (%host-component-match-p s-host f-host)) (foo-error source from-wildname))
      (setq r-host (translate-component s-host f-host t-host reversible))
      (let ((s-dir (%std-directory-component (pathname-directory source) s-host))
            (f-dir (%std-directory-component (pathname-directory from-wildname) f-host))
            (t-dir (%std-directory-component (pathname-directory to-wildname) t-host)))
        (let ((match (%pathname-match-directory s-dir f-dir)))
          (if (not match)(foo-error source from-wildname))
          (setq r-directory  (translate-directory s-dir f-dir t-dir reversible t-host))))
      (let ((s-name (pathname-name source))
            (f-name (pathname-name from-wildname))
            (t-name (pathname-name to-wildname)))
        (if (not (%component-match-p s-name f-name))(foo-error source from-wildname))        
        (setq r-name (translate-component s-name f-name t-name reversible)))
      (let ((s-type (pathname-type source))
            (f-type (pathname-type from-wildname))
            (t-type (pathname-type to-wildname)))
        (if (not (%component-match-p s-type f-type))(foo-error source from-wildname))
        (setq r-type (translate-component s-type f-type t-type reversible)))
      (let ((s-version (pathname-version source))
            (f-version (pathname-version from-wildname))
            (t-version (pathname-version to-wildname)))
        (if (not (%component-match-p s-version f-version))(foo-error source from-wildname))
        (setq r-version (translate-component s-version f-version t-version reversible))
        ;(if (eq r-version :unspecific)(setq r-version nil))
        )
      (make-pathname :device r-device :host r-host :directory r-directory
                     :name r-name :type r-type :version r-version :defaults nil)
      )))



(defvar %empty-logical-pathname% (%cons-logical-pathname nil nil nil nil nil))

;; This extends CL in that it allows a host-less pathname, like "foo;bar;baz".
(defun logical-pathname (thing &aux (path thing))
  (when (typep path 'stream) (setq path (%path-from-stream path)))
  (etypecase path
    (logical-pathname path)
    (pathname (report-bad-arg thing 'logical-pathname))
    (string
     (multiple-value-bind (sstr start end) (get-sstring path)
       ;; Prescan the host, to avoid unknown host errors.
       (let ((host (pathname-host-sstr sstr start end t)))
	 (let ((%logical-host-translations% (cons (list host) %logical-host-translations%)))
	   (declare (special %logical-host-translations%))
	   ;; By calling string-to-pathname with a logical pathname as default, we force
	   ;; parsing as a logical pathname.
	   (string-to-pathname sstr start end nil %empty-logical-pathname%)))))))

(defun %host-component-match-p (path-host wild-host)
  ;; Note that %component-match-p is case sensitive.  Need a
  ;; case-insensitive version for hosts. 
  (or (string-equal path-host wild-host)
      (%component-match-p path-host wild-host)))

(defun pathname-match-p (pathname wildname)
  (let ((path-host (pathname-host pathname))
        (wild-host (pathname-host wildname)))
    (and
     (%host-component-match-p path-host wild-host)
     (%component-match-p (pathname-device pathname)(pathname-device wildname))
     (%pathname-match-directory
      (%std-directory-component (pathname-directory pathname) path-host)
      (%std-directory-component (pathname-directory wildname) wild-host))
     (%component-match-p (pathname-name pathname)(pathname-name wildname))
     (%component-match-p (pathname-type pathname)(pathname-type wildname))
     (%component-match-p (pathname-version pathname)(pathname-version wildname)))))


; expects canonicalized directory - how bout absolute vs. relative?
(defun %pathname-match-directory (path wild)
  (cond ((equal path wild) t)
	 ; Don't allow matching absolute and relative, so that can have distinct
	 ; absolute and wild translations in logical-pathname-translations for
	 ; a host, and have them match separately.
	((and (consp path)(consp wild)(neq (car path) (car wild)))
	 nil)  ; one absolute & one relative ??
        ((or ;(and (null wild)
             ;     (let ((dir (cadr path)))
             ;       (if (stringp dir)(string= dir "**")(eq dir :wild-inferiors))))
             (and (null (cddr wild))
                  (let ((dir (cadr wild)))
                    (if (stringp dir)(string= dir "**")(eq dir :wild-inferiors))))))
	((null path)
	 ;; Make missing dir match (:absolute) or (:relative) - is that right?
	 (null (cdr wild)))
	((null wild)
	 nil)
        (t (%pathname-match-dir0 (cdr path)(cdr wild)))))

; munch on tails path and wild 
(defun %pathname-match-dir0 (path wild)
  (flet ((only-wild (dir)
                    (when (null (cdr dir))
                      (setq dir (car dir))
                      (if (stringp dir)(string= dir "**")(eq dir :wild-inferiors)))))
    (cond ((eq path wild) t)
          ((only-wild wild)
           t)
          (t (let ((result t))
               (block nil 
                 (while (and path wild)
                   (let ((pathstr (car path))
                         (wildstr (car wild)))
                     (case wildstr
                       (:wild (setq wildstr "*"))
                       (:wild-inferiors (setq wildstr "**")))
                     (case pathstr
                       (:wild (setq pathstr "*"))
                       (:wild-inferiors (setq pathstr "**")))
                     (when (not 
                            (cond ((string= wildstr "**")
                                   (setq result (%pathname-match-dir1 path (cdr wild)))
                                   (return-from nil))
                                  ((%path-str*= pathstr wildstr))))
                       (setq result nil)
                       (return-from nil))
                     (setq wild (cdr wild) path (cdr path))))
                 (when (and (or path wild)(not (only-wild wild)))
                   (setq result nil)))
               result)))))

(defun %pathname-match-dir0 (path wild)
  (flet ((only-wild (dir)
                    (when (null (cdr dir))
                      (setq dir (car dir))
                      (when (consp dir) (setq dir (cadr dir)))
                      (if (stringp dir)(string= dir "**")(eq dir :wild-inferiors)))))
    (cond ((eq path wild) t)
          ((only-wild wild)
           t)
          (t (let ((result t))
               (block nil 
                 (while (and path wild)
                   (let ((pathstr (car path))
                         (wildstr (car wild)))                     
                     ; allow logical to match physical today
                     ; because one of these days these logical things will disappear!
                     (when (consp pathstr)(setq pathstr (cadr pathstr)))
                     (when (consp wildstr)(setq wildstr (cadr wildstr)))
                     (case wildstr
                       (:wild (setq wildstr "*"))
                       (:wild-inferiors (setq wildstr "**")))
                     (case pathstr
                       (:wild (setq pathstr "*"))
                       (:wild-inferiors (setq pathstr "**")))
                     (if (or (memq wildstr '(:up :back))(memq pathstr '(:up :back))) ;; ????? <<<<
                       (when (neq pathstr wildstr)(setq result nil) (return-from nil))
                       (when (not 
                              (cond ((string= wildstr "**")
                                     (setq result (%pathname-match-dir1 path (cdr wild)))
                                     (return-from nil))
                                    ((%path-str*= pathstr wildstr))))
                         (setq result nil)
                         (return-from nil)))
                     (setq wild (cdr wild) path (cdr path))))
                 (when (and (or path wild)(not (only-wild wild)))
                   (setq result nil)))
               result)))))



; wild is stuff after a "**" - looking for what matches the **  in (path)
(defun %pathname-match-dir1 (path wild &optional cons-result)
  (let ((match nil) pathstr wildstr)
    (cond ((null wild)
           (values T (if cons-result (mapcar #'(lambda (e)
                                            (if (consp e)(cadr e) e))
                                        path))))
          ((%pathname-match-dir0 path wild))   ; ie ** matches nothing
          (t 
           (prog nil
             AGN
               (setq pathstr (car path) wildstr (car wild))
               (when (consp pathstr)(setq pathstr (cadr pathstr)))
               (when (consp wildstr)(setq wildstr (cadr wildstr)))
               (case wildstr
                 (:wild (setq wildstr "*"))
                 (:wild-inferiors (setq wildstr "**")))
               (case pathstr
                 (:wild (setq pathstr "*"))
                 (:wild-inferiors (setq pathstr "**")))
               (until (or (not (consp path))
                          (%path-str*= pathstr wildstr))
                 (when cons-result (push pathstr match))
                 (setq path (cdr path))
                 (setq pathstr (car path))
                 (when (consp pathstr)(setq pathstr (cadr pathstr))))
               ;; either got a match - w and path both have the thing we looked for
               ;; or path is empty
               (when (null path)(return nil))
               (let ((path1 (cdr path))(wild (cdr wild)))
                 (when (and (null path1)(null wild))
                   (return (values t (when match (nreverse match)))))
                 (cond ((%pathname-match-dir0 path1 wild)  ; is the rest happy too?
                        (return (values t (nreverse match))))
                       (t (when cons-result (push pathstr match)) ; nope, let ** eat more
                          (setq path (cdr path))
                          (go AGN)))))))))

; three times bigger and 3 times slower - does it matter?
(defun %path-str*= (string pattern)
  (multiple-value-bind (string s-start s-end) (get-sstring string)
    (multiple-value-bind (pattern p-start p-end) (get-sstring pattern)
      (path-str-sub pattern string p-start s-start p-end s-end))))

(defun path-str-sub (pattern str p-start s-start p-end s-end)
  (declare (fixnum p-start s-start p-end s-end)
	   (type simple-base-string pattern str))
  (declare (optimize (speed 3)(safety 0)))
  (let ((p (%scharcode pattern p-start))
        (esc (char-code *pathname-escape-character*)))
    (cond 
     ((eq p (char-code #\*))
      ; starts with a * find what we looking for unless * is last in which case done
      (loop ; lots of *'s same as one
        (when (eq (%i+ 1 p-start)  p-end)
          (return-from path-str-sub t))
        (if (eq (%schar pattern (%i+ 1 p-start)) #\*)
          (setq p-start (1+ p-start))
          (return)))
      (let* ((next* (%path-mem "*" pattern (%i+ 1 p-start)))
             (len (- (or next* p-end) (%i+ 1 p-start))))
        (loop
          (when (> (+ s-start len) s-end)(return nil))
          (let ((res (find-str-pattern pattern str (%i+ 1 p-start) s-start (or next* p-end) s-end))) 
            (if (null res)
              (return nil)
              (if (null next*)
                (if (eq res s-end)
                  (return t))                  
                (return (path-str-sub pattern str next* (+ s-start len) p-end s-end)))))
          (setq s-start (1+ s-start)))))
     (t (when (eq p esc)
          (setq p-start (1+ p-start))
          (setq p (%scharcode pattern p-start)))
        (let* ((next* (%path-mem "*" pattern (if (eq p (char-code #\*))(%i+ 1 p-start) p-start)))
               (this-s-end (if next* (+ s-start (- next* p-start)) s-end)))
          (if (> this-s-end s-end)
            nil
            (if  (path-str-match-p pattern str p-start s-start (or next* p-end) this-s-end)
              (if (null next*)
                t                  
                (path-str-sub pattern str next* this-s-end p-end s-end)))))))))

; find match of pattern between start and end in str 
; rets one past end of pattern in str or nil
(defun find-str-pattern (pattern str p-start s-start p-end s-end)
  (declare (fixnum p-start s-start p-end s-end)
	   (type simple-base-string pattern str))
  (declare (optimize (speed 3)(safety 0)))
  (let* ((first-p (%scharcode pattern p-start))
         (esc (char-code *pathname-escape-character*)))
    (when (and (eq first-p esc) (not (eq (setq p-start (1+ p-start)) p-end)))
      (setq first-p (%scharcode pattern p-start)))
    (do* ((i s-start (1+ i))
          (last-i (%i- s-end (%i- p-end p-start))))
         ((> i last-i) nil)
      (declare (fixnum i last-i))
      (let ((s (%scharcode str i)))
        (when (eq first-p s)
          (do* ((j (1+ i) (1+ j))
                (k (1+ p-start)(1+ k)))
               ((>= k p-end) (return-from find-str-pattern j))
            (declare (fixnum j k))
            (let* ((p (%scharcode pattern k))
                   (s (%scharcode str j)))
              (when (and (eq p esc) (< (setq k (1+ k)) p-end))
                (setq p (%scharcode pattern k)))
              (when (not (eq p s))
                (return)))))))))

(defun path-str-match-p (pattern str p-start s-start p-end s-end)
  (declare (fixnum p-start s-start p-end s-end)
	   (type simple-base-string pattern str))
  (declare (optimize (speed 3)(safety 0)))
  ;; does pattern match str between p-start p-end
  (let ((esc (char-code *pathname-escape-character*)))
    (loop      
      (when (eq p-start p-end)
        (return (eq s-start s-end)))
      (when (eq s-start s-end)
	(return nil))
      (let ((p (%scharcode pattern p-start)))
        (when (eq p esc)
	  (when (eq (setq p-start (1+ p-start)) p-end)
	    (return nil))
          (setq p (%scharcode pattern p-start)))
	(unless (eq p (%scharcode str s-start))
	  (return nil))
	(setq p-start (1+ p-start))
	(setq s-start (1+ s-start))))))
      
             
(defun ccl-directory ()
  (let* ((dirpath (getenv "CCL_DEFAULT_DIRECTORY")))
    (if dirpath
      (make-pathname :defaults nil :directory dirpath)      
      (mac-default-directory))))


(defun user-homedir-pathname (&optional host)
  (declare (ignore host))  
  (let* ((dir (get-user-home-dir (getuid))))
    (if dir
	(make-pathname :defaults nil :directory dir))))

(progn
(setf (logical-pathname-translations "home")
          `(("**;*.*" ,(merge-pathnames "**/*.*" (user-homedir-pathname)))))

(setf (logical-pathname-translations "ccl")
          `(("lib;**;*.fasl" "ccl:bin;*.fasl")
	    ("l1;**;*.fasl" "ccl:l1f;*.fasl")
            ("l1;**;*.pfsl" "ccl:l1pf;*.pfsl")
            ("l1;**;*.sfsl" "ccl:l1sf;*.sfsl")
	    ("l1;**;*.dfsl" "ccl:l1df;*.dfsl")
            ("l1;**;*.*" "ccl:level-1;**;*.*")
            ("l1f;**;*.pfsl" "ccl:l1pf;**;*.pfsl")
            ("l1f;**;*.sfsl" "ccl:l1sf;**;*.sfsl")
	    ("l1f;**;*.dfsl" "ccl:l1df;**;*.dfsl")
            ("bin;**;*.pfsl" "ccl:binppc;**;*.pfsl")
            ("bin;**;*.sfsl" "ccl:binsparc;**;*.sfsl")
	    ("bin;**;*.dfsl" "ccl:bindarwin;**.*.dfsl")
            ("l1pf;**;*.*" "ccl:l1-pfsls;**;*.*")
            ("l1sf;**;*.*" "ccl:l1-sfsls;**;*.*")
	    ("l1df;**;*.*" "ccl:l1-dfsls;**;*.*")
            ("l1f;**;*.*" "ccl:l1-fasls;**;*.*")
            ("ccl;*.*" ,(merge-pathnames "*.*" (ccl-directory)))
            ("**;*.*" ,(merge-pathnames "**/*.*" (ccl-directory)))))

)

(defloadvar *user-homedir-pathname* (user-homedir-pathname))

(defun translate-logical-pathname (pathname &key)
  (setq pathname (pathname pathname))
  (let ((host (pathname-host pathname)))
    (cond ((eq host :unspecific) pathname)
	  ((null host) (%cons-pathname (pathname-directory pathname)
				       (pathname-name pathname)
				       (pathname-type pathname)))
	  (t
	   (let ((rule (assoc pathname (logical-pathname-translations host)
			      :test #'pathname-match-p)))  ; how can they match if hosts neq??
	     (if rule
	       (translate-logical-pathname
		(translate-pathname pathname (car rule) (cadr rule)))
	       (signal-file-error $xnotranslation pathname)))))))

;This function should be changed to standardize the name more than it does.
;It should eliminate non-leading instances of "::" etc at least.  We might also
;want it to always return an absolute pathname (i.e. fill in the default mac
;directory), so as to make it a sort of harmless truename (which is how I think
;it's mainly used).  Unfortunately that would make it go to the file system,
;but it might be worth it.
;This function used to also remove quoting so as to make the name suitable for
;passing to rom.  It doesn't anymore. Use mac-namestring for that.
; does anybody use this??
; DO - merge in default if relative, and do the :: stuff
; perhaps call it expand-pathname or expanded-pathname

(defun full-pathname (path &key (no-error t))
  (let ((orig-path path))
    (cond (no-error
           ; note that ignore-errors wont work until var %handlers% is defined (in l1-init)
           (setq path (ignore-errors
                       (translate-logical-pathname (merge-pathnames path))))
           (when (null path) (return-from full-pathname nil)))
          (t (setq path (translate-logical-pathname (merge-pathnames path)))))
    (let* ((ihost (pathname-host orig-path))
           (dir (%pathname-directory path)))
      (when (and no-error (not dir) (%pathname-directory path)) ; WHAT is  that noop - since 3.0??
        (return-from full-pathname nil))
      (when (and ihost (neq ihost :unspecific))  ; << this is new. is it right?
        (if (eq (car dir) :relative)  ; don't make relative to mac-default-dir if had a host???
          (setq dir (cons :absolute (cdr dir)))))
      (setq dir (absolute-directory-list dir))      
      (unless (eq dir (%pathname-directory path))
        (setq path (cons-pathname dir (%pathname-name path) (%pathname-type path)
                                  (pathname-host path) (pathname-version path))))
      path)))





(defparameter *module-search-path* (list
                                    (cons-pathname '(:absolute "bin") nil nil "ccl")
                                    *user-homedir-pathname*
                                    (cons-pathname  nil nil nil "ccl")
                                    (cons-pathname '(:absolute "lib") nil nil "ccl")
				    (cons-pathname '(:absolute "library") nil nil "ccl")
				    (cons-pathname '(:absolute "examples") nil nil "ccl"))
  "Holds a list of pathnames to search for the file that has same name
   as a module somebody is looking for.")

