;;;-*-Mode: LISP; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;

(eval-when (:compile-toplevel :execute)
  (require "NXENV")
  (require "SPARCENV"))

(eval-when (:load-toplevel :execute :compile-toplevel)
  (require "SPARC-BACKEND"))


(defmacro with-sparc-p2-declarations (declsform &body body)
  `(let* ((*sparc2-noforcestk* *sparc2-noforcestk*)
          (*sparc2-tail-allow* *sparc2-tail-allow*)
          (*sparc2-reckless* *sparc2-reckless*)
          (*sparc2-inhibit-eventchecks* *sparc2-inhibit-eventchecks*)
          (*sparc2-open-code-inline* *sparc2-open-code-inline*)
          (*sparc2-trust-declarations* *sparc2-trust-declarations*))
     (sparc2-decls ,declsform)
     ,@body))

(defmacro with-sparc-local-vinsn-macros ((segvar &optional vreg-var xfer-var) &body body)
  (declare (ignorable xfer-var))
  (let* ((template-name-var (gensym))
         (template-temp (gensym))
         (args-var (gensym))
         (labelnum-var (gensym))
         (retvreg-var (gensym))
         (label-var (gensym)))
    `(macrolet ((! (,template-name-var &rest ,args-var)
                  (let* ((,template-temp (get-vinsn-template-cell ,template-name-var *sparc-vinsn-templates*)))
                    (unless ,template-temp
                      (warn "VINSN \"~A\" not defined" ,template-name-var))
                    `(%emit-vinsn ,',segvar (load-time-value (get-vinsn-template-cell ',,template-name-var *sparc-vinsn-templates*)) *sparc-vinsn-templates* ,@,args-var))))
       (macrolet ((<- (,retvreg-var)
                    `(sparc2-copy-register ,',segvar ,',vreg-var ,,retvreg-var))
                  (@  (,labelnum-var)
                    `(backend-gen-label ,',segvar ,,labelnum-var))
                  (-> (,label-var)
                    `(! jump (aref *backend-labels* ,,label-var)))
                  (^ (&rest branch-args)
                    `(sparc2-branch ,',segvar ,',xfer-var ,',vreg-var ,@branch-args)))
         ,@body))))





(defvar *sparc-current-context-annotation* nil)
(defvar *sparc2-inhibit-eventchecks* nil)
(defvar *sparc2-woi* nil)
(defvar *sparc2-open-code-inline* nil)
(defvar *sparc2-register-restore-count* 0)
(defvar *sparc2-register-restore-ea* nil)
(defvar *sparc2-compiler-register-save-label* nil)
(defvar *sparc2-valid-register-annotations* 0)
(defvar *sparc2-register-annotation-types* nil)
(defvar *sparc2-register-ea-annotations* nil)

(defparameter *sparc2-tail-call-aliases* 
  '((%call-next-method . (%tail-call-next-method . 1))))

(defvar *sparc2-popreg-labels* nil)
(defvar *sparc2-popj-labels* nil)
(defvar *sparc2-valret-labels* nil)
(defvar *sparc2-nilret-labels* nil)

(defvar *sparc2-icode* nil)
(defvar *sparc2-undo-stack* nil)
(defvar *sparc2-undo-because* nil)


(defvar *sparc2-cur-afunc* nil)
(defvar *sparc2-vstack* 0)
(defvar *sparc2-cstack* 0)
(defvar *sparc2-undo-count* 0)
(defvar *sparc2-returning-values* nil)
(defvar *sparc2-vcells* nil)
(defvar *sparc2-fcells* nil)
(defvar *sparc2-entry-vsp-saved-p* nil)

(defvar *sparc2-entry-label* nil)
(defvar *sparc2-tail-label* nil)
(defvar *sparc2-tail-allow* t)
(defvar *sparc2-reckless* nil)
(defvar *sparc2-trust-declarations* nil)
(defvar *sparc2-entry-vstack* nil)
(defvar *sparc2-fixed-nargs* nil)
(defvar *sparc2-need-nargs* t)

(defparameter *sparc2-inhibit-register-allocation* nil)
(defvar *sparc2-record-symbols* nil)
(defvar *sparc2-recorded-symbols* nil)
(defvar *sparc2-noforcestk* nil)

(defvar *sparc2-result-reg* sparc::%arg_z)





(declaim (fixnum *sparc2-vstack* *sparc2-cstack*))

 
(defparameter *sparc2-debug-mask* 0)
(defconstant sparc2-debug-verbose-bit 0)
(defconstant sparc2-debug-vinsns-bit 1)
(defconstant sparc2-debug-lcells-bit 1)


; Before any defsparc2's, make the *sparc2-specials* vector.

(defvar *sparc2-all-lcells* ())




     
(defun sparc2-free-lcells ()
  (without-interrupts 
   (let* ((prev (pool.data *lcell-freelist*)))
     (dolist (r *sparc2-all-lcells*)
       (setf (lcell-kind r) prev
             prev r))
     (setf (pool.data *lcell-freelist*) prev)
     (setq *sparc2-all-lcells* nil))))

(defun sparc2-note-lcell (c)
  (push c *sparc2-all-lcells*)
  c)

(defvar *sparc2-top-vstack-lcell* ())
(defvar *sparc2-bottom-vstack-lcell* ())

(defun sparc2-new-lcell (kind parent width attributes info)
  (sparc2-note-lcell (make-lcell kind parent width attributes info)))

(defun sparc2-new-vstack-lcell (kind width attributes info)
  (setq *sparc2-top-vstack-lcell* (sparc2-new-lcell kind *sparc2-top-vstack-lcell* width attributes info)))

(defun sparc2-reserve-vstack-lcells (n)
  (dotimes (i n) (sparc2-new-vstack-lcell :reserved 4 0 nil)))

(defun sparc2-vstack-mark-top ()
  (sparc2-new-lcell :tos *sparc2-top-vstack-lcell* 0 0 nil))

; Alist mapping VARs to lcells/lregs
(defvar *sparc2-var-cells* ())

(defun sparc2-note-var-cell (var cell)
  ;(format t "~& ~s -> ~s" (var-name var) cell)
  (push (cons var cell) *sparc2-var-cells*))

(defun sparc2-note-top-cell (var)
  (sparc2-note-var-cell var *sparc2-top-vstack-lcell*))

(defun sparc2-lookup-var-cell (var)
  (or (cdr (assq var *sparc2-var-cells*))
      (and nil (warn "Cell not found for ~s" (var-name var)))))

(defun sparc2-collect-lcells (kind &optional (bottom *sparc2-bottom-vstack-lcell*) (top *sparc2-top-vstack-lcell*))
  (do* ((res ())
        (cell top (lcell-parent cell)))
       ((eq cell bottom) res)
    (if (null cell)
      (error "Horrible compiler bug.")
      (if (eq (lcell-kind cell) kind)
        (push cell res)))))

  
; ensure that lcell's offset matches what we expect it to.
; For bootstrapping.

(defun sparc2-ensure-lcell-offset (c expected)
  (if c (= (calc-lcell-offset c) expected) (zerop expected)))



(defun sparc2-check-lcell-depth (&optional (context "wherever"))
  (when (logbitp sparc2-debug-verbose-bit *sparc2-debug-mask*)
    (let* ((depth (calc-lcell-depth *sparc2-top-vstack-lcell*)))
      (or (= depth *sparc2-vstack*)
          (warn "~a: lcell depth = ~d, vstack = ~d" context depth *sparc2-vstack*)))))

(defun sparc2-do-lexical-reference (seg vreg ea)
  (declare (fixnum ea))
  (when vreg
    (with-sparc-local-vinsn-macros (seg vreg) 
      (if (memory-spec-p ea)
        (ensuring-node-target (target vreg)
          (progn
            (sparc2-stack-to-register seg ea target)
            (if (addrspec-vcell-p ea)
              (! vcell-ref target target))))
        (<- ea)))))

(defun sparc2-do-lexical-setq (seg vreg ea valreg)
  (with-sparc-local-vinsn-macros (seg vreg)
    (cond ((addrspec-vcell-p ea)     ; closed-over vcell
           (with-node-temps (valreg) (vcell)
             (sparc2-stack-to-register seg ea vcell)
             (! vcell-set& vcell valreg)))      ; Memoize it
          ((memory-spec-p ea)    ; vstack slot
           (sparc2-register-to-stack seg valreg ea))
          (t
           (sparc2-copy-register seg ea valreg)))
    (when vreg
      (<- valreg))))

; ensure that next-method-var is heap-consed (if it's closed over.)
; it isn't ever setqed, is it ?
(defun sparc2-heap-cons-next-method-var (seg var)
  (with-sparc-local-vinsn-macros (seg)
    (when (eq (ash 1 $vbitclosed)
              (logand (logior (ash 1 $vbitclosed)
                              (ash 1 $vbitcloseddownward))
                      (the fixnum (nx-var-bits var))))
      (let ((ea (var-ea var)))
        (sparc2-do-lexical-reference seg sparc::%arg_z ea)
        (sparc2-set-nargs seg 1)
        (! ref-constant sparc::%fname (backend-immediate-index (sparc2-symbol-entry-locative '%cons-magic-next-method-arg)))
        (! call-known-symbol)
        (sparc2-do-lexical-setq seg nil ea sparc::%arg_z)))))


(defun sparc2-reverse-cc (cond)
  (cdr (assq cond '((#.sparc::condne . #.sparc::condne)
                    (#.sparc::conde . #.sparc::conde)
                    (#.sparc::condle . #.sparc::condge)
                    (#.sparc::condl . #.sparc::condg)
                    (#.sparc::condge . #.sparc::condle)
                    (#.sparc::condg . #.sparc::condl)))))
             
(defun acode-condition-to-sparc-icc (cond)
  (condition-to-sparc-icc (cadr cond)))

(defun condition-to-sparc-icc (cond)
  (ecase cond
    (:EQ (values sparc::conde t))
    (:NE (values sparc::conde nil))
    (:GT (values sparc::condg t))
    (:LE (values sparc::condg nil))
    (:LT (values sparc::condl t))
    (:GE (values sparc::condl nil))))

(defun acode-condition-to-sparc-fcc (cond)
  (condition-to-sparc-fcc (cadr cond)))

(defun condition-to-sparc-fcc (cond)
  (ecase cond
    (:EQ (values (logior 16 sparc::fconde) t))
    (:NE (values (logior 16 sparc::fconde) nil))
    (:GT (values (logior 16 sparc::fcondg) t))
    (:LE (values (logior 16 sparc::fcondg) nil))
    (:LT (values (logior 16 sparc::fcondl) t))
    (:GE (values (logior 16 sparc::fcondl) nil))))

; is CC an FPU condition code ?
(defun fcc-p (cc)
  (logbitp 4 cc))

(defun sparc-invert-fcc (cc)
  (cdr (assq (logand #xf cc)
	     '((#.sparc::fconde . #.sparc::fcondne)
	       (#.sparc::fcondne . #.sparc::fconde)
	       (#.sparc::fcondl . #.sparc::fcondge)
	       (#.sparc::fcondge . #.sparc::fcondl)
	       (#.sparc::fcondg . #.sparc::fcondle)
	       (#.sparc::fcondle . #.sparc::fcondg)))))

       
(defun sparc2-compile (afunc &optional lfun-maker lambda-form *sparc2-record-symbols*)
  (progn
    (when (eq lfun-maker t) (setq lfun-maker #'sparc2-xmake-function))
    (dolist (a  (afunc-inner-functions afunc))
      (unless (afunc-lfun a)
        (sparc2-compile a 
                      (or (null lfun-maker) lfun-maker) 
                      (if lambda-form 
                        (afunc-lambdaform a)) 
                      *sparc2-record-symbols*)))  ; always compile inner guys
    (let* ((*sparc2-cur-afunc* afunc)
           (*sparc2-returning-values* nil)
           (*sparc2-inhibit-eventchecks* nil)
           (*sparc-current-context-annotation* nil)
           (*sparc2-woi* nil)
           (*next-lcell-id* -1)
           (*sparc2-open-code-inline* nil)
           (*sparc2-register-restore-count* nil)
           (*sparc2-compiler-register-save-label* nil)
           (*sparc2-valid-register-annotations* 0)
           (*sparc2-register-ea-annotations* (sparc2-make-stack 16))
           (*sparc2-register-restore-ea* nil)
           (*sparc2-vstack* 0)
           (*sparc2-cstack* 0)
           (*sparc2-all-lcells* ())
           (*sparc2-top-vstack-lcell* nil)
           (*sparc2-bottom-vstack-lcell* (sparc2-new-vstack-lcell :bottom 0 0 nil))
           (*sparc2-var-cells* nil)
	   (*backend-vinsns* *sparc-vinsn-templates*)
	   (*backend-node-regs* sparc-node-regs)
	   (*backend-node-temps* sparc-temp-node-regs)
	   (*available-backend-node-temps* sparc-temp-node-regs)
           (*backend-allocate-high-node-temps* t)
	   (*backend-imm-temps* sparc-imm-regs)
	   (*available-backend-imm-temps* sparc-imm-regs)
	   (*backend-crf-temps* (ash 1 0))
	   (*available-backend-crf-temps* (ash 1 0))
	   (*backend-fp-temps* sparc-temp-fp-regs)
	   (*available-backend-fp-temps* sparc-temp-fp-regs)
           (bits 0)
           (*logical-register-counter* -1)
           (*backend-all-lregs* ())
	   
           (*sparc2-popj-labels* nil)
           (*sparc2-popreg-labels* nil)
           (*sparc2-valret-labels* nil)
           (*sparc2-nilret-labels* nil)
           (*sparc2-undo-count* 0)
	   (*backend-labels* (sparc2-make-stack 64 arch::subtag-simple-vector))
           (*sparc2-undo-stack* (sparc2-make-stack 64  arch::subtag-simple-vector))
           (*sparc2-undo-because* (sparc2-make-stack 64))
	   (*backend-immediates* (sparc2-make-stack 64  arch::subtag-simple-vector))
           (*sparc2-entry-label* nil)
           (*sparc2-tail-label* nil)
           (*sparc2-inhibit-register-allocation* nil)
           (*sparc2-tail-allow* t)
           (*sparc2-reckless* nil)
           (*sparc2-trust-declarations* t)
           (*sparc2-entry-vstack* nil)
           (*sparc2-fixed-nargs* nil)
           (*sparc2-need-nargs* t)
           (*sparc2-noforcestk* nil)
           (fname (afunc-name afunc))
           (*sparc2-entry-vsp-saved-p* nil)
           (*sparc2-vcells* (afunc-vcells afunc))
           (*sparc2-fcells* (afunc-fcells afunc))
           *sparc2-recorded-symbols*)
      (set-fill-pointer
       *backend-labels*
       (set-fill-pointer
        *sparc2-undo-stack*
        (set-fill-pointer 
         *sparc2-undo-because*
         (set-fill-pointer
          *backend-immediates* 0))))
      (backend-get-next-label) ; start @ label 1, 0 is confused with NIL in compound cd
      (with-dll-node-freelist (vinsns *vinsn-freelist*)
        (unwind-protect
          (progn
            (setq bits (sparc2-form vinsns (make-wired-lreg *sparc2-result-reg*) $backend-return (afunc-acode afunc)))
            (dotimes (i (length *backend-immediates*))
              (let ((imm (aref *backend-immediates* i)))
                (when (sparc2-symbol-locative-p imm) (aset *backend-immediates* i (car imm)))))
            (when (logbitp sparc2-debug-vinsns-bit *sparc2-debug-mask*)
              (format t "~% vinsns for ~s (after generation)" (afunc-name afunc))
              (do-dll-nodes (v vinsns) (format t "~&~s" v))
              (format t "~%~%"))
            (with-dll-node-freelist (*lap-instructions* *lap-instruction-freelist*)
              (let* ((*lap-labels* nil))
                (sparc2-expand-vinsns vinsns) 
	    (if (logbitp $fbitnonnullenv (the fixnum (afunc-bits afunc)))
                  (setq bits (+ bits (ash 1 $lfbits-nonnullenv-bit))))
                (let* ((function-debugging-info (afunc-lfun-info afunc)))
                  (when (or function-debugging-info lambda-form *sparc2-record-symbols*)
                    (if lambda-form (setq function-debugging-info 
                                          (list* 'function-lambda-expression lambda-form function-debugging-info)))
                    (if *sparc2-record-symbols*
                      (setq function-debugging-info (nconc (list 'function-symbol-map *sparc2-recorded-symbols*)
                                                           function-debugging-info)))
                    (setq bits (logior (ash 1 $lfbits-symmap-bit) bits))
                    (backend-new-immediate function-debugging-info)))
                (if (or fname lambda-form *sparc2-recorded-symbols*)
                  (backend-new-immediate fname)
                  (setq bits (logior (ash -1 $lfbits-noname-bit) bits)))                                     
                (unless (afunc-parent afunc)
                  (sparc2-fixup-fwd-refs afunc))
                (setf (afunc-all-vars afunc) nil)
                (setf (afunc-argsword afunc) bits)
                (let* ((regsave-label (if (typep *sparc2-compiler-register-save-label* 'vinsn-note)
                                          (vinsn-label-info (vinsn-note-label *sparc2-compiler-register-save-label*))))
                       (regsave-reg (if regsave-label (- 32 *sparc2-register-restore-count*)))
                       (regsave-addr (if regsave-label (- *sparc2-register-restore-ea*))))
                  (when lfun-maker
                    (setf (afunc-lfun afunc)
                          (funcall lfun-maker
                                   *lap-instructions*
                                   *lap-labels*
                                   *backend-immediates*
                                   bits
                                   regsave-label
                                   regsave-reg
                                   regsave-addr))))
                (sparc2-digest-symbols))))
          (backend-remove-labels))))
    afunc))

(defun sparc2-xmake-function (codeheader labels imms bits *sparc-lap-regsave-label* *sparc-lap-regsave-reg* *sparc-lap-regsave-addr*)
  (let* ((*lap-instructions* codeheader)
         (*lap-labels* labels)
         (numimms (length imms))
         (function (%alloc-misc (+ numimms 2) #+sparc-target arch:subtag-function #-sparc-target arch::subtag-xfunction)))
    (dotimes (i numimms)
      (let* ((imm (aref imms i)))
	#-sparc-target
	(when (typep imm 'function)
	  (let* ((p (position imm *struct-ref-vector*)))
	    (if p
		(setq imm (aref *sparc-struct-ref-vector* p))
		(warn "Unknown native functional constant: ~s" imm))))
	(setf (uvref function (1+ i)) imm)))
    (setf (uvref function (+ numimms 1)) bits)
    (let* ((maxpc (sparc-lap-encode-regsave-info (sparc-lap-do-labels)))
           (code-vector (%alloc-misc (ash maxpc -2) #+sparc-target arch::subtag-code-vector #-sparc-target arch::subtag-xcode-vector))
           (i 0))
      (sparc-lap-resolve-labels)
      (do-dll-nodes (insn *lap-instructions*)
        (sparc-lap-generate-instruction code-vector i insn t)
        (incf i))
      (setf (uvref function 0) code-vector)
      (%make-code-executable code-vector)
      function)))
      
    
(defun sparc2-make-stack (size &optional (subtype arch::subtag-s16-vector))
  (make-uarray-1 subtype size t 0 nil nil nil nil t nil))

(defun sparc2-fixup-fwd-refs (afunc)
  (dolist (f (afunc-inner-functions afunc))
    (sparc2-fixup-fwd-refs f))
  (let ((fwd-refs (afunc-fwd-refs afunc)))
    (when fwd-refs
      (let* ((v (afunc-lfun afunc))
             (vlen (uvsize v)))
        (declare (fixnum vlen))
        (dolist (ref fwd-refs)
          (let* ((ref-fun (afunc-lfun ref)))
            (do* ((i 1 (1+ i)))
                 ((= i vlen))
              (declare (fixnum i))
              (if (eq (%svref v i) ref)
                (setf (%svref v i) ref-fun)))))))))

(defun sparc2-digest-symbols ()
  (if *sparc2-recorded-symbols*
    (let* ((symlist *sparc2-recorded-symbols*)
           (len (length symlist))
           (syms (make-array len))
           (ptrs (make-array (%i+  (%i+ len len) len)))
           (i -1)
           (j -1))
      (declare (fixnum i j))
      (dolist (info symlist (progn (%rplaca symlist syms)
                                   (%rplacd symlist ptrs)))
        (flet ((label-address (note start-p sym)
                 (let* ((label (vinsn-note-label note))
                        (lap-label (if label (vinsn-label-info label))))
                   (if lap-label
                     (lap-label-address lap-label)
                     (error "Missing or bad ~s label: ~s" 
                       (if start-p 'start 'end) sym)))))
          (destructuring-bind (var sym startlab endlab) info
            (let* ((ea (var-ea var))
                   (ea-val (ldb (byte 16 0) ea)))
              (setf (aref ptrs (incf i)) (if (memory-spec-p ea)
                                           (logior (ash ea-val 6) #o77)
                                           ea-val)))
            (setf (aref syms (incf j)) sym)
            (setf (aref ptrs (incf i)) (label-address startlab t sym))
            (setf (aref ptrs (incf i)) (label-address endlab nil sym))))))))

(defun sparc2-decls (decls)
  (if (fixnump decls)
    (locally (declare (fixnum decls))
      (setq *sparc2-noforcestk* (neq 0 (%ilogand2 $decl_noforcestk decls))
            *sparc2-tail-allow* (neq 0 (%ilogand2 $decl_tailcalls decls))
            *sparc2-inhibit-eventchecks* (or *sparc2-woi* (neq 0 (%ilogand2 $decl_eventchk decls)))
            *sparc2-open-code-inline* (neq 0 (%ilogand2 $decl_opencodeinline decls))
            *sparc2-reckless* (neq 0 (%ilogand2 $decl_unsafe decls))
            *sparc2-trust-declarations* (neq 0 (%ilogand2 $decl_trustdecls decls))))))


(defun %sparc2-bigger-cdr-than (x y)
  (declare (cons x y))
  (> (the fixnum (cdr x)) (the fixnum (cdr y))))

; Return an unordered list of "varsets": each var in a varset can be assigned a register
; and all vars in a varset can be assigned the same register (e.g., no scope conflicts.)

(defun sparc2-partition-vars (vars)
  (labels ((var-weight (var)
             (let* ((bits (nx-var-bits var)))
               (declare (fixnum bits))
               (if (eql 0 (logand bits (logior
                                        (ash 1 $vbitpuntable)
                                        (ash -1 $vbitspecial)
                                        (ash 1 $vbitnoreg))))
                 (if (eql (logior (ash 1 $vbitclosed) (ash 1 $vbitsetq))
                          (logand bits (logior (ash 1 $vbitclosed) (ash 1 $vbitsetq))))
                   0
                   (%i+ (%ilogand $vrefmask bits) (%ilsr 8 (%ilogand $vsetqmask bits))))
                 0)))
           (sum-weights (varlist) 
             (let ((sum 0))
               (dolist (v varlist sum) (incf sum (var-weight v)))))
           (vars-disjoint-p (v1 v2)
             (if (eq v1 v2)
               nil
               (if (memq v1 (var-binding-info v2))
                 nil
                 (if (memq v2 (var-binding-info v1))
                   nil
                   t)))))
    (setq vars (%sort-list-no-key
                ;(delete-if #'(lambda (v) (eql (var-weight v) 0)) vars) 
                (do* ((handle (cons nil vars))
                      (splice handle))
                     ((null (cdr splice)) (cdr handle))                  
                  (declare (dynamic-extent handle) (type cons handle splice))
                  (if (eql 0 (var-weight (%car (cdr splice))))
                    (rplacd splice (%cdr (cdr splice)))
                    (setq splice (cdr splice))))
                #'(lambda (v1 v2) (%i> (var-weight v1) (var-weight v2)))))
    ; This isn't optimal.  It partitions all register-allocatable variables into sets such that
    ; 1) no variable is a member of more than one set and
    ; 2) all variables in a given set are disjoint from each other
    ; A set might have exactly one member.
    ; If a register is allocated for any member of a set, it's allocated for all members of that
    ; set.
    (let* ((varsets nil))
      (do* ((all vars (cdr all)))
           ((null all))
        (let* ((var (car all)))
          (when (dolist (already varsets t)
                  (when (memq var (car already)) (return)))
            (let* ((varset (cons var nil)))
              (dolist (v (cdr all))
                (when (dolist (already varsets t)
                        (when (memq v (car already)) (return)))
                  (when (dolist (d varset t)
                          (unless (vars-disjoint-p v d) (return)))
                    (push v varset))))
              (let* ((weight (sum-weights varset)))
                (declare (fixnum weight))
                (if (>= weight 3)
                  (push (cons (nreverse varset) weight) varsets)))))))
      varsets)))

; Maybe globally allocate registers to symbols naming functions & variables,
; and to simple lexical variables.
(defun sparc2-allocate-global-registers (fcells vcells all-vars no-regs)
  (if no-regs
    (progn
      (dolist (c fcells) (%rplacd c nil))
      (dolist (c vcells) (%rplacd c nil))
      (values 0 nil))
    (let* ((maybe (sparc2-partition-vars all-vars)))
      (dolist (c fcells) 
        (if (>= (the fixnum (cdr c)) 3) (push c maybe)))
      (dolist (c vcells) 
        (if (>= (the fixnum (cdr c)) 3) (push c maybe)))
      (do* ((things (%sort-list-no-key maybe #'%sparc2-bigger-cdr-than) (cdr things))
            (n 0 (1+ n))
            (regno sparc::%save0 (1+ regno))
            (constant-alist ()))
           ((or (null things) (= n $numsparcsaveregs))
            (dolist (cell fcells) (%rplacd cell nil))
            (dolist (cell vcells) (%rplacd cell nil))
            (values n constant-alist))
        (declare (list things)
                 (fixnum n regno))
        (let* ((thing (car things)))
          (if (or (memq thing fcells)
                  (memq thing vcells))
            (push (cons thing regno) constant-alist)
            (dolist (var (car thing))
              (nx-set-var-bits var 
                               (%ilogior (%ilogand (%ilognot $vrefmask) (nx-var-bits var))
                                 regno
                                 (%ilsl $vbitreg 1))))))))))
          
    
;; Vpush the last N non-volatile-registers.
;; Could use a STM here, especially if N is largish or optimizing for space.
(defun sparc2-save-nvrs (seg n)
  (declare (fixnum n))
  (when (> n 0)
    (setq *sparc2-compiler-register-save-label* (sparc2-emit-note seg :regsave))
    (with-sparc-local-vinsn-macros (seg)
      (! save-nvrs (+ (1- sparc::%save0) n)))
    (dotimes (i n)
      (sparc2-new-vstack-lcell :regsave 4 0 (+ sparc::%save0 i)))
    (incf *sparc2-vstack* (the fixnum (* n 4)))
    (setq *sparc2-register-restore-ea* *sparc2-vstack*
          *sparc2-register-restore-count* n)))


; If there are an indefinite number of args/values on the vstack,
; we have to restore from a register that matches the compiler's
; notion of the vstack depth.  This can be computed by the caller 
; (sum of vsp & nargs, or copy of vsp  before indefinite number of 
; args pushed, etc.)
; We DON'T try to compute this from the saved context, since the
; saved vsp may belong to a different stack segment.  (It's cheaper
; to compute/copy than to load it, anyway.)

(defun sparc2-restore-nvrs (seg ea nregs &optional from-fp)
  (when (null from-fp)
    (setq from-fp sparc::%vsp))
  (when (and ea nregs)
    (with-sparc-local-vinsn-macros (seg)
      (let* ((first (+ (1- sparc::%save0) nregs)))
        (declare (fixnum cell first n))
        (! restore-nvrs first from-fp (- *sparc2-vstack* ea))))))


;; The change is to ask for a stack-consed rest var if the rest var is ignored.
;; And also to pop the rest var immediately if it's ignored, rather than at the end
;; of the function.  That will allow calling the final function tail-recursively.  



(defun sparc2-bind-lambda (seg lcells req opt rest keys auxen optsupvloc passed-in-regs lexpr &optional inherited
                             &aux (vloc 0) (numopt (%ilsl 2 (list-length (%car opt))))
                             (nkeys (list-length (%cadr keys))) 
                             reg)
  (declare (fixnum vloc))
  (sparc2-check-lcell-depth)
  (dolist (arg inherited)
    (if (memq arg passed-in-regs)
      (sparc2-set-var-ea seg arg (var-ea arg))
      (let* ((lcell (pop lcells)))
        (if (setq reg (sparc2-assign-register-var arg))
          (sparc2-init-regvar seg arg reg (sparc2-vloc-ea vloc))
          (sparc2-bind-var seg arg vloc lcell))
        (setq vloc (%i+ vloc 4)))))
  (dolist (arg req)
    (if (memq arg passed-in-regs)
      (sparc2-set-var-ea seg arg (var-ea arg))
      (let* ((lcell (pop lcells)))
        (if (setq reg (sparc2-assign-register-var arg))
          (sparc2-init-regvar seg arg reg (sparc2-vloc-ea vloc))
          (sparc2-bind-var seg arg vloc lcell))
        (setq vloc (%i+ vloc 4)))))
  (when opt
    (if (sparc2-hard-opt-p opt)
      (setq vloc (apply #'sparc2-initopt seg vloc optsupvloc lcells (nthcdr (- (length lcells) (ash numopt -2)) lcells) opt)
              lcells (nthcdr (ash numopt -2) lcells))
      (dolist (var (%car opt))
        (if (memq var passed-in-regs)
          (sparc2-set-var-ea seg var (var-ea var))
          (let* ((lcell (pop lcells)))
            (if (setq reg (sparc2-assign-register-var var))
              (sparc2-init-regvar seg var reg (sparc2-vloc-ea vloc))
              (sparc2-bind-var seg var vloc lcell))
            (setq vloc (+ vloc 4)))))))
  (when keys
    (apply #'sparc2-init-keys seg vloc lcells keys)
    (setq vloc (+ vloc (%ilsl 3 nkeys))
          lcells (nthcdr (+ nkeys nkeys) lcells)))
  (when rest
    (if lexpr
      (progn
        (if (setq reg (sparc2-assign-register-var rest))
          (progn
            (sparc2-load-lexpr-address seg reg)
            (sparc2-set-var-ea seg rest reg))
          (with-imm-temps () ((nargs-cell :u32))
            (sparc2-load-lexpr-address seg nargs-cell)
            (let* ((loc *sparc2-vstack*))
              (sparc2-vpush-register seg nargs-cell :reserved)
              (sparc2-note-top-cell rest)
              (sparc2-bind-var seg rest loc *sparc2-top-vstack-lcell*)))))
      (progn
        (if (setq reg (sparc2-assign-register-var rest))
          (sparc2-init-regvar seg rest reg (sparc2-vloc-ea vloc))
          (sparc2-bind-var seg rest vloc (pop lcells)))
        (setq vloc (+ vloc 4)))))
  (sparc2-seq-bind seg (%car auxen) (%cadr auxen)))

(defun sparc2-initopt (seg vloc spvloc lcells splcells vars inits spvars)
  (with-sparc-local-vinsn-macros (seg)
    (dolist (var vars vloc)
      (let* ((initform (pop inits))
             (spvar (pop spvars))
             (lcell (pop lcells))
             (splcell (pop splcells))
             (reg (sparc2-assign-register-var var))
             (regloadedlabel (if reg (backend-get-next-label))))
        (unless (nx-null initform)
          (sparc2-stack-to-register seg (sparc2-vloc-ea spvloc) sparc::%arg_z)
          (let ((skipinitlabel (backend-get-next-label)))
            (with-crf-target () crf
              (sparc2-compare-registers seg crf (sparc2-make-compound-cd 0 skipinitlabel) sparc::%arg_z sparc::%rnil sparc::conde t))
            (if reg
              (sparc2-form seg reg regloadedlabel initform)
              (sparc2-register-to-stack seg (sparc2-one-untargeted-reg-form seg initform sparc::%arg_z) (sparc2-vloc-ea vloc)))
            (@ skipinitlabel)))
        (if reg
          (progn
            (sparc2-init-regvar seg var reg (sparc2-vloc-ea vloc))
            (@ regloadedlabel))
          (sparc2-bind-var seg var vloc lcell))
        (when spvar
          (if (setq reg (sparc2-assign-register-var spvar))
            (sparc2-init-regvar seg spvar reg (sparc2-vloc-ea spvloc))
            (sparc2-bind-var seg spvar spvloc splcell))))
      (setq vloc (%i+ vloc 4))
      (if spvloc (setq spvloc (%i+ spvloc 4))))))

(defun sparc2-init-keys (seg vloc lcells allow-others keyvars keysupp keyinits keykeys)
  (declare (ignore keykeys allow-others))
  (with-sparc-local-vinsn-macros (seg)
    (dolist (var keyvars)
      (let* ((spvar (pop keysupp))
             (initform (pop keyinits))
             (reg (sparc2-assign-register-var var))
             (regloadedlabel (if reg (backend-get-next-label)))
             (var-lcell (pop lcells))
             (sp-lcell (pop lcells))
             (sploc (%i+ vloc 4)))
        (unless (nx-null initform)
          (sparc2-stack-to-register seg (sparc2-vloc-ea sploc) sparc::%arg_z)
          (let ((skipinitlabel (backend-get-next-label)))
            (with-crf-target () crf
              (sparc2-compare-registers seg crf (sparc2-make-compound-cd 0 skipinitlabel) sparc::%arg_z sparc::%rnil sparc::conde t))
            (if reg
              (sparc2-form seg reg regloadedlabel initform)
              (sparc2-register-to-stack seg (sparc2-one-untargeted-reg-form seg initform sparc::%arg_z) (sparc2-vloc-ea vloc)))
            (@ skipinitlabel)))
        (if reg
          (progn
            (sparc2-init-regvar seg var reg (sparc2-vloc-ea vloc))
            (@ regloadedlabel))
          (sparc2-bind-var seg var vloc var-lcell))
        (when spvar
          (if (setq reg (sparc2-assign-register-var spvar))
            (sparc2-init-regvar seg spvar reg (sparc2-vloc-ea sploc))
            (sparc2-bind-var seg spvar sploc sp-lcell))))
      (setq vloc (%i+ vloc 8)))))

; Vpush register r, unless var gets a globally-assigned register.
; Return NIL if register was vpushed, else var.
(defun sparc2-vpush-arg-register (seg reg var)
  (when var
    (let* ((bits (nx-var-bits var)))
      (declare (fixnum bits))
      (if (logbitp $vbitreg bits)
        var
        (progn 
          (sparc2-vpush-register seg reg :reserved)
          nil)))))


; nargs has been validated, arguments defaulted and canonicalized.
; Save caller's context, then vpush any argument registers that
; didn't get global registers assigned to their variables.
; Return a list of vars/nils for each argument register 
;  (nil if vpushed, var if still in arg_reg).
(defun sparc2-argregs-entry (seg revargs)
  (with-sparc-local-vinsn-macros (seg)
    (let* ((nargs (length revargs))
           (reg-vars ()))
      (declare (type (unsigned-byte 16) nargs))
      (if (<= nargs $numsparcargregs)       ; caller didn't vpush anything
        (if *sparc2-open-code-inline*
          (! save-lisp-context-vsp)
          (! save-lisp-context-vsp-ool))
        (let* ((offset (ash (the fixnum (- nargs $numsparcargregs)) 2)))
          (declare (fixnum offset))
          (if *sparc2-open-code-inline*
            (! save-lisp-context-offset offset)
            (! save-lisp-context-offset-ool offset))))
      (destructuring-bind (&optional zvar yvar xvar &rest stack-args) revargs
        (let* ((nstackargs (length stack-args)))
          (sparc2-set-vstack (ash nstackargs 2))
          (dotimes (i nstackargs)
            (sparc2-new-vstack-lcell :reserved 4 0 nil))
          (if (>= nargs 3)
            (push (sparc2-vpush-arg-register seg sparc::%arg_x xvar) reg-vars))
          (if (>= nargs 2)
            (push (sparc2-vpush-arg-register seg sparc::%arg_y yvar) reg-vars))
          (if (>= nargs 1)
            (push (sparc2-vpush-arg-register seg sparc::%arg_z zvar) reg-vars))))
      reg-vars)))

; Just required args.
; Since this is just a stupid bootstrapping port, always save 
; lisp context.
(defun sparc2-req-nargs-entry (seg rev-fixed-args)
  (let* ((nargs (length rev-fixed-args)))
    (declare (type (unsigned-byte 16) nargs))
    (with-sparc-local-vinsn-macros (seg)
      (unless *sparc2-reckless*
        (! check-exact-nargs nargs))
      (sparc2-argregs-entry seg rev-fixed-args))))

; No more than three &optional args; all default to NIL and none have
; supplied-p vars.  No &key/&rest.
(defun sparc2-simple-opt-entry (seg rev-opt-args rev-req-args)
  (let* ((min (length rev-req-args))
         (nopt (length rev-opt-args))
         (max (+ min nopt)))
    (declare (type (unsigned-byte 16) min nopt max))
    (with-sparc-local-vinsn-macros (seg)
      (unless *sparc2-reckless*
        (when rev-req-args
          (! check-min-nargs min))
        (! check-max-nargs max))
      (if (= nopt 1)
        (! default-1-arg min)
        (if (= nopt 2)
          (! default-2-args min)
          (! default-3-args min)))
      (sparc2-argregs-entry seg (append rev-opt-args rev-req-args)))))

; if "num-fixed" is > 0, we've already ensured that at least that many args
; were provided; that may enable us to generate better code for saving the
; argument registers.
; We're responsible for computing the caller's VSP and saving
; caller's state.
(defun sparc2-lexpr-entry (seg num-fixed)
  (with-sparc-local-vinsn-macros (seg)
    (! save-lexpr-argregs num-fixed)
    (dotimes (i num-fixed)
      (! copy-lexpr-argument))
    (! save-lisp-context-lexpr)))

(defun sparc2-load-lexpr-address (seg dest)
  (with-sparc-local-vinsn-macros (seg)
    (! load-vframe-address dest *sparc2-vstack*)))


(defun sparc2-structured-initopt (seg lcells vloc context vars inits spvars)
  (with-sparc-local-vinsn-macros (seg)
    (dolist (var vars vloc)
      (let* ((initform (pop inits))
             (spvar (pop spvars))
             (spvloc (%i+ vloc 4))
             (var-lcell (pop lcells))
             (sp-lcell (pop lcells)))
        (unless (nx-null initform)
          (sparc2-stack-to-register seg (sparc2-vloc-ea spvloc) sparc::%arg_z)
          (let ((skipinitlabel (backend-get-next-label)))
            (with-crf-target () crf
              (sparc2-compare-registers seg crf (sparc2-make-compound-cd 0 skipinitlabel) sparc::%arg_z sparc::%rnil sparc::conde t))
            (sparc2-register-to-stack seg (sparc2-one-untargeted-reg-form seg initform sparc::%arg_z) (sparc2-vloc-ea vloc))
            (@ skipinitlabel)))
        (sparc2-bind-structured-var seg var vloc var-lcell context)
        (when spvar
          (sparc2-bind-var seg spvar spvloc sp-lcell)))
      (setq vloc (%i+ vloc 8)))))



(defun sparc2-structured-init-keys (seg lcells vloc context allow-others keyvars keysupp keyinits keykeys)
  (declare (ignore keykeys allow-others))
  (with-sparc-local-vinsn-macros (seg)
    (dolist (var keyvars)
      (let* ((spvar (pop keysupp))
             (initform (pop keyinits))
             (sploc (%i+ vloc 4))
             (var-lcell (pop lcells))
             (sp-lcell (pop lcells)))
        (unless (nx-null initform)
          (sparc2-stack-to-register seg (sparc2-vloc-ea sploc) sparc::%arg_z)
          (let ((skipinitlabel (backend-get-next-label)))
            (with-crf-target () crf
              (sparc2-compare-registers seg crf (sparc2-make-compound-cd 0 skipinitlabel) sparc::%arg_z sparc::%rnil sparc::conde t))
            (sparc2-register-to-stack seg (sparc2-one-untargeted-reg-form seg initform sparc::%arg_z) (sparc2-vloc-ea vloc))
            (@ skipinitlabel)))
        (sparc2-bind-structured-var seg var vloc var-lcell context)
        (when spvar
          (sparc2-bind-var seg spvar sploc sp-lcell)))
      (setq vloc (%i+ vloc 8)))))

(defun sparc2-vloc-ea (n &optional vcell-p)
  (setq n (make-memory-spec (dpb memspec-frame-address memspec-type-byte n)))
  (if vcell-p
    (make-vcell-memory-spec n)
    n))


(defun sparc2-form (seg vreg xfer form)
  (if (nx-null form)
    (sparc2-nil seg vreg xfer)
    (if (nx-t form)
      (sparc2-t seg vreg xfer)
      (let* ((op nil)
             (fn nil))
        (if (and (consp form)
                 (setq fn (svref *sparc2-specials* (%ilogand #.operator-id-mask (setq op (acode-operator form))))))
          (if (and (null vreg)
                   (%ilogbitp operator-acode-subforms-bit op)
                   (%ilogbitp operator-assignment-free-bit op))
            (dolist (f (%cdr form) (sparc2-branch seg xfer nil))
              (sparc2-form seg nil nil f ))
            (apply fn seg vreg xfer (%cdr form)))
          (error "sparc2-form ? ~s" form))))))

; dest is a float reg - form is acode
(defun sparc2-form-float (seg freg xfer form)
  (declare (ignore xfer))
  (when (or (nx-null form)(nx-t form))(error "sparc2-form to freg ~s" form))
  (when (and (= (get-regspec-mode freg) hard-reg-class-fpr-mode-double)
             (sparc2-form-typep form 'double-float))
    ; kind of screwy - encoding the source type in the dest register spec
    (set-node-regspec-type-modes freg hard-reg-class-fpr-type-double))
  (let* ((fn nil))
    (if (and (consp form)
             (setq fn (svref *sparc2-specials* (%ilogand #.operator-id-mask (acode-operator form)))))      
      (apply fn seg freg nil (%cdr form))
      (error "sparc2-form ? ~s" form))))



(defun sparc2-form-typep (form type)
  (acode-form-typep form type *sparc2-trust-declarations*)
)

(defun sparc2-form-type (form)
  (acode-form-type form *sparc2-trust-declarations*))
  
(defun sparc2-use-operator (op seg vreg xfer &rest forms)
  (declare (dynamic-extent forms))
  (apply (svref *sparc2-specials* (%ilogand operator-id-mask op)) seg vreg xfer forms))

; Returns true iff lexical variable VAR isn't setq'ed in FORM.
; Punts a lot ...
(defun sparc2-var-not-set-by-form-p (var form)
  (or (not (%ilogbitp $vbitsetq (nx-var-bits var)))
      (sparc2-setqed-var-not-set-by-form-p var form)))

(defun sparc2-setqed-var-not-set-by-form-p (var form)
  (setq form (acode-unwrapped-form form))
  (or (atom form)
      (sparc-constant-form-p form)
      (sparc2-lexical-reference-p form)
      (let ((op (acode-operator form))
            (subforms nil))
        (if (eq op (%nx1-operator setq-lexical))
          (and (neq var (cadr form))
               (sparc2-setqed-var-not-set-by-form-p var (caddr form)))
          (and (%ilogbitp operator-side-effect-free-bit op)
               (flet ((not-set-in-formlist (formlist)
                        (dolist (subform formlist t)
                          (unless (sparc2-setqed-var-not-set-by-form-p var subform) (return)))))
                 (if
                   (cond ((%ilogbitp operator-acode-subforms-bit op) (setq subforms (%cdr form)))
                         ((%ilogbitp operator-acode-list-bit op) (setq subforms (cadr form))))
                   (not-set-in-formlist subforms)
                   (and (or (eq op (%nx1-operator call))
                            (eq op (%nx1-operator lexical-function-call)))
                        (sparc2-setqed-var-not-set-by-form-p var (cadr form))
                        (setq subforms (caddr form))
                        (not-set-in-formlist (car subforms))
                        (not-set-in-formlist (cadr subforms))))))))))
  
(defun sparc2-nil (seg vreg xfer)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (if (sparc2-for-value-p vreg)
      (ensuring-node-target (target vreg)
        (! load-nil target)))
    (sparc2-branch seg (sparc2-cd-false xfer) vreg)))

(defun sparc2-t (seg vreg xfer)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (if (sparc2-for-value-p vreg)
      (ensuring-node-target (target vreg)
        (! load-t target)))
    (sparc2-branch seg (sparc2-cd-true xfer) vreg)))


(defun sparc2-for-value-p (vreg)
  (and vreg (not (backend-crf-p vreg))))

(defun sparc2-mvpass (seg form &optional xfer)
  (sparc2-form seg  sparc::%arg_z (%ilogior2 (or xfer 0) $backend-mvpass-mask) form))

(defun sparc2-adjust-vstack (delta)
  (sparc2-set-vstack (%i+ *sparc2-vstack* delta)))

(defun sparc2-set-vstack (new)
  (setq *sparc2-vstack* new))


; Emit a note at the end of the segment.
(defun sparc2-emit-note (seg class &rest info)
  (declare (dynamic-extent info))
  (let* ((note (make-vinsn-note class info)))
    (append-dll-node (vinsn-note-label note) seg)
    note))

; Emit a note immediately before the target vinsn.
(defun sparc-prepend-note (vinsn class &rest info)
  (declare (dynamic-extent info))
  (let* ((note (make-vinsn-note class info)))
    (insert-dll-node-before (vinsn-note-label note) vinsn)
    note))

(defun sparc2-close-note (seg note)
  (let* ((end (close-vinsn-note note)))
    (append-dll-node (vinsn-note-label end) seg)
    end))






(defun sparc2-stack-to-register (seg memspec reg)
  (with-sparc-local-vinsn-macros (seg)
    (! vframe-load reg (memspec-frame-address-offset memspec) *sparc2-vstack*)))

(defun sparc2-lcell-to-register (seg lcell reg)
  (with-sparc-local-vinsn-macros (seg)
    (! lcell-load reg lcell (sparc2-vstack-mark-top))))


(defun sparc2-register-to-stack (seg reg memspec)
  (with-sparc-local-vinsn-macros (seg)
    (! vframe-store reg (memspec-frame-address-offset memspec) *sparc2-vstack*)))


(defun sparc2-ea-open (ea)
  (if (and ea (addrspec-vcell-p ea))
    (make-memory-spec (memspec-frame-address-offset ea))
    ea))

(defun sparc2-set-NARGS (seg n)
  (if (> n call-arguments-limit)
    (error "~s exceeded." call-arguments-limit)
    (with-sparc-local-vinsn-macros (seg)
      (! set-nargs n))))

(defun sparc2-assign-register-var (v)
  (let ((bits (nx-var-bits v)))
    (when (%ilogbitp $vbitreg bits)
      (%ilogand bits $vrefmask))))

; Can't cross-compile this.  Too bad.
(defun sparc2-single-float-bits (the-sf)
  (uvref the-sf arch::single-float.value-cell))

(defun sparc2-double-float-bits (the-df)
  (values (uvref the-df arch::double-float.value-cell)
          (uvref the-df arch::double-float.val-low-cell)))

(defun sparc2-immediate (seg vreg xfer form)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (if vreg
      (if (and (= (hard-regspec-class vreg) hard-reg-class-fpr)
               (or (and (typep form 'double-float) (= (get-regspec-mode vreg) hard-reg-class-fpr-mode-double))
                   (and (typep form 'short-float)(= (get-regspec-mode vreg) hard-reg-class-fpr-mode-single))))
        (if (zerop form)
          (if (eql form 0.0d0)
            (! zero-double-float-register vreg)
            (! zero-single-float-register vreg))
          (if (typep form 'short-float)
            (let* ((bits (sparc2-single-float-bits form)))
              (with-imm-temps () ((bitsreg :u32))
                (! lwi bitsreg bits)
                (! load-single-float-constant vreg bitsreg)))
            (multiple-value-bind (high low) (sparc2-double-float-bits form)
              (declare (integer high low))
              (with-imm-temps () ((highreg :u32) (lowreg :u32))
                (if (zerop high)
                  (setq highreg sparc::%rzero)
                  (! lwi highreg high))
                (if (zerop low)
                  (setq lowreg sparc::%rzero)
                  (! lwi lowreg low))
                (! load-double-float-constant vreg highreg lowreg)))))            
        (ensuring-node-target (target vreg)
          (if (characterp form)
            (sparc2-lwi seg target (logior (ash (char-code form) arch::charcode-shift) arch::subtag-character))
            (sparc2-store-immediate seg form target))))
      (if (and (listp form) *load-time-eval-token* (eq (car form) *load-time-eval-token*))
        (sparc2-store-immediate seg form sparc::%temp0)))
    (^)))

(defun sparc2-register-constant-p (form)
  (and (consp form)
           (or (memq form *sparc2-vcells*)
               (memq form *sparc2-fcells*))
           (%cdr form)))

(defun sparc2-store-immediate (seg imm dest)
  (with-sparc-local-vinsn-macros (seg)
    (let* ((reg (sparc2-register-constant-p imm)))
      (if reg
        (sparc2-copy-register seg dest reg)
        (let* ((idx (backend-immediate-index imm)))
          (if (< idx 8192)
            (! ref-constant dest idx)
            (with-imm-target () (idxreg :s32)
              (sparc2-lwi seg idxreg (+ sparc::misc-data-offset (ash (1+ idx) 2)))
              (! ref-indexed-constant dest idxreg)))))
      dest)))


; Returns label iff form is (local-go <tag>) and can go without adjusting stack.
(defun sparc2-go-label (form)
  (let ((current-stack (sparc2-encode-stack)))
    (while (and (acode-p form) (or (eq (acode-operator form) (%nx1-operator progn))
                                   (eq (acode-operator form) (%nx1-operator local-tagbody))))
      (setq form (caadr form)))
    (when (acode-p form)
      (let ((op (acode-operator form)))
        (if (and (eq op (%nx1-operator local-go))
                 (sparc2-equal-encodings-p (%caddr (%cadr form)) current-stack))
          (%cadr (%cadr form))
          (if (and (eq op (%nx1-operator local-return-from))
                   (nx-null (caddr form)))
            (let ((tagdata (car (cadr form))))
              (and (sparc2-equal-encodings-p (cdr tagdata) current-stack)
                   (null (caar tagdata))
                   (< 0 (cdar tagdata) $backend-mvpass)
                   (cdar tagdata)))))))))

(defun sparc2-single-valued-form-p (form)
  (setq form (acode-unwrapped-form form))
  (or (nx-null form)
      (nx-t form)
      (if (acode-p form)
        (let ((op (acode-operator form)))
          (or (%ilogbitp operator-single-valued-bit op)
              (and (eql op (%nx1-operator values))
                   (let ((values (cadr form)))
                     (and values (null (cdr values)))))
              nil                       ; Learn about functions someday
              )))))





(defun sparc2-fix-stackreg (seg regnum amt)
  (with-sparc-local-vinsn-macros (seg)
    (! adjust-stack-register regnum amt)))

; safe = T means assume "vector" is miscobj, do bounds check.
; safe = fixnum means check that subtag of vector = "safe" and do bounds check.
; safe = nil means crash&burn.
; This mostly knows how to reference the elements of an immediate miscobj.
; This returns a boxed object.  It'd be nice if there was some way not to.
; Need dfloat vreg scheme.
(defun sparc2-vref (seg vreg xfer subtag vector index safe)
  (declare (fixnum subtag))
  (if (= (the fixnum (logand subtag sparc::fulltagmask)) sparc::fulltag-nodeheader)
    (sparc2-misc-node-ref seg vreg xfer vector index safe)
    (with-sparc-local-vinsn-macros (seg vreg xfer)
      (unless (= (the fixnum (logand subtag sparc::fulltagmask)) sparc::fulltag-immheader)
        (error "Bug: not a SPARC subtag: ~s" subtag))
      (if (null vreg)
        (progn
          (sparc2-form seg nil nil vector)
          (sparc2-form seg nil xfer index))
        (if (and (= (hard-regspec-class vreg) hard-reg-class-fpr)
                 (= subtag arch::subtag-double-float-vector))
          (sparc2-df-vref seg vreg xfer vector index safe)
          (if (and (= (hard-regspec-class vreg) hard-reg-class-fpr)
                   (= subtag arch::subtag-single-float-vector))
            (sparc2-sf-vref seg vreg xfer vector index safe)
            (let* ((index-known-fixnum (acode-fixnum-form-p index))            
                   (unscaled-idx nil)
                   (src nil))
              (ensuring-node-target (target vreg)
                (if (or safe (not index-known-fixnum))
                  (multiple-value-setq (src unscaled-idx)
                    (sparc2-two-untargeted-reg-forms seg vector sparc::%arg_y index sparc::%arg_z))
                  (setq src (sparc2-one-untargeted-reg-form seg vector sparc::%arg_z)))
                (when safe
                  (if (typep safe 'fixnum)
                    (! trap-unless-typecode= src safe))
                  (unless index-known-fixnum
                    (! trap-unless-tag= unscaled-idx arch::tag-fixnum))
                  (! check-misc-bound unscaled-idx src))
                (if (<= subtag arch::max-32-bit-ivector-subtag)
                  (if (and index-known-fixnum (<= index-known-fixnum sparc::max-32-bit-constant-index))
                    (cond ((= subtag arch::subtag-single-float-vector)
                           (! misc-ref-c-single-float 0 src index-known-fixnum)
                           (! single->heap target 0))
                          (t
                           (with-imm-temps () (temp)
                             (! misc-ref-c-u32 temp src index-known-fixnum)
                             (if (= subtag arch::subtag-s32-vector)
                               (! s32->integer target temp)
                               (! u32->integer target temp)))))
                    (with-imm-temps () (idx-reg)
                      (if index-known-fixnum
                        (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-data-offset (ash index-known-fixnum 2)))
                        (! scale-32bit-misc-index idx-reg unscaled-idx))
                      (cond ((= subtag arch::subtag-single-float-vector)
                             (! misc-ref-single-float 0 src idx-reg)
                             (! single->heap target 0))
                            (t (with-imm-temps (idx-reg) (temp)
                                 (! misc-ref-u32 temp src idx-reg)
                                 (if (= subtag arch::subtag-s32-vector)
                                   (! s32->integer target temp)
                                   (! u32->integer target temp)))))))
                  (if (<= subtag arch::max-8-bit-ivector-subtag)
                    (with-imm-temps () (temp)
                      (if (and index-known-fixnum (<= index-known-fixnum sparc::max-8-bit-constant-index))
                        (! misc-ref-c-u8 temp src index-known-fixnum)
                        (with-imm-temps () (idx-reg)
                          (if index-known-fixnum
                            (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-data-offset index-known-fixnum))
                            (! scale-8bit-misc-index idx-reg unscaled-idx))
                          (! misc-ref-u8 temp src idx-reg)))
                      (if (= subtag arch::subtag-u8-vector)
                        (! u8->fixnum target temp)
                        (if (= subtag arch::subtag-s8-vector)
                          (! s8->fixnum target temp)
                          (! u8->char target temp))))
                    (if (<= subtag arch::max-16-bit-ivector-subtag)
                      (with-imm-temps () (temp)
                        (if (and index-known-fixnum (<= index-known-fixnum sparc::max-16-bit-constant-index))
                          (! misc-ref-c-u16 temp src index-known-fixnum)
                          (with-imm-temps () (idx-reg)
                            (if index-known-fixnum
                              (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-data-offset (ash index-known-fixnum 1)))
                              (! scale-16bit-misc-index idx-reg unscaled-idx))
                            (! misc-ref-u16 temp src idx-reg)))
                        (if (= subtag arch::subtag-u16-vector)
                          (! u16->fixnum target temp)
                          (if (= subtag arch::subtag-s16-vector)
                            (! s16->fixnum target temp)
                            (! u8->char target temp))))
                      ; Down to the dregs.
                      (if (= subtag arch::subtag-double-float-vector)
                        (progn
                          (if (and index-known-fixnum (<= index-known-fixnum sparc::max-64-bit-constant-index))
                            (! misc-ref-c-double-float 0 src index-known-fixnum)
                            (with-imm-temps () (idx-reg)
                              (if index-known-fixnum
                                (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-dfloat-offset (ash index-known-fixnum 3)))
                                (! scale-64bit-misc-index idx-reg unscaled-idx))
                              (! misc-ref-double-float 0 src idx-reg)))
                          (! double->heap target 0))
                        (if (and index-known-fixnum (<= index-known-fixnum sparc::max-1-bit-constant-index))
                          (! misc-ref-c-bit[fixnum] target src index-known-fixnum)
                          (with-imm-temps () (word-index bitnum dest)
                            (if index-known-fixnum
                              (progn
                                (sparc2-lwi seg word-index (+ sparc::misc-data-offset (ash index-known-fixnum -5)))
                                (sparc2-lwi seg bitnum (logand index-known-fixnum #x1f)))
                              (! scale-1bit-misc-index word-index bitnum unscaled-idx))
                            (! misc-ref-u32 dest src word-index)
                            (! extract-variable-bit[fixnum] target dest bitnum))))))))
              (^))))))))

; In this case, the target register is an fp reg and the vector is declared
; do be a double-float vector.  Avoid boxing the result!
(defun sparc2-df-vref (seg vreg xfer vector index safe)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((index-known-fixnum (acode-fixnum-form-p index))
           (src)
           (unscaled-idx))
      (if (or safe (not index-known-fixnum))
        (multiple-value-setq (src unscaled-idx)
          (sparc2-two-untargeted-reg-forms seg vector sparc::%arg_y index sparc::%arg_z))
        (setq src (sparc2-one-untargeted-reg-form seg vector sparc::%arg_z)))
      (when safe
        (if (typep safe 'fixnum)
          (! trap-unless-typecode= src safe))
        (unless index-known-fixnum
          (! trap-unless-tag= unscaled-idx arch::tag-fixnum))
        (! check-misc-bound unscaled-idx src))
      (if (and index-known-fixnum (<= index-known-fixnum sparc::max-64-bit-constant-index))
        (! misc-ref-c-double-float vreg src index-known-fixnum)
        (with-imm-temps () (idx-reg)
          (if index-known-fixnum
            (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-dfloat-offset (ash index-known-fixnum 3)))
            (! scale-64bit-misc-index idx-reg unscaled-idx))
          (! misc-ref-double-float vreg src idx-reg)))
      (^))))

(defun sparc2-df-aset2 (seg vreg xfer array i j new safe &optional dim0 dim1)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((i-known-fixnum (acode-fixnum-form-p i))
           (j-known-fixnum (acode-fixnum-form-p j))
           (src)
           (unscaled-i)
           (unscaled-j)
           (constidx
            (and dim0 dim1 i-known-fixnum j-known-fixnum
                 (>= i-known-fixnum 0)
                 (>= j-known-fixnum 0)
                 (< i-known-fixnum dim0)
                 (< j-known-fixnum dim1)
                 (+ (* i-known-fixnum dim1) j-known-fixnum))))
      (with-fp-target () fp-val
        (if constidx
          (multiple-value-setq (src fp-val)
            (sparc2-two-targeted-reg-forms seg array sparc::%arg_z new fp-val))
          (multiple-value-setq (src unscaled-i unscaled-j fp-val)
            (sparc2-four-untargeted-reg-forms seg
                                            array sparc::%arg_x
                                            i sparc::%arg_y
                                            j sparc::%arg_z
                                            new fp-val)))
        (when safe      
          (when (typep safe 'fixnum)
            (! trap-unless-typecode= src arch::subtag-arrayH)
            (! check-arrayH-rank src 2)
            (! check-arrayH-flags src
               (dpb safe sparc::arrayH.flags-cell-subtag-byte
                    (ash 1 $arh_simple_bit))))
          (unless i-known-fixnum
            (! trap-unless-tag= unscaled-i arch::tag-fixnum))
          (unless j-known-fixnum
            (! trap-unless-tag= unscaled-j arch::tag-fixnum)))
        (with-imm-temps () (dim1 idx-reg)
          (unless constidx
            (if safe                  
                (! check-2d-bound dim1 unscaled-i unscaled-j src)
                (! 2d-dim1 dim1 src))
            (! 2d-df-scaled-index idx-reg src unscaled-i unscaled-j dim1))
          (with-node-temps () (v)
            (! array-data-vector-ref v src)
            (when (and constidx (>= constidx sparc::max-64-bit-constant-index))
              (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-dfloat-offset
                                                     (ash constidx 3)))
              (setq constidx nil))
            (if constidx
                (! misc-set-c-double-float fp-val v constidx)
                (! misc-set-double-float fp-val v idx-reg))))
      (^)))))

(defun sparc2-df-aref2 (seg vreg xfer array i j safe &optional dim0 dim1)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((i-known-fixnum (acode-fixnum-form-p i))
           (j-known-fixnum (acode-fixnum-form-p j))
           (src)
           (unscaled-i)
           (unscaled-j)
           (constidx
            (and dim0 dim1 i-known-fixnum j-known-fixnum
                 (>= i-known-fixnum 0)
                 (>= j-known-fixnum 0)
                 (< i-known-fixnum dim0)
                 (< j-known-fixnum dim1)
                 (+ (* i-known-fixnum dim1) j-known-fixnum))))
      (if constidx
        (setq src (sparc2-one-targeted-reg-form seg array sparc::%arg_z))
        (multiple-value-setq (src unscaled-i unscaled-j)
          (sparc2-three-untargeted-reg-forms seg
                                           array sparc::%arg_x
                                           i sparc::%arg_y
                                           j sparc::%arg_z)))
      (when safe        
        (when (typep safe 'fixnum)
          (! trap-unless-typecode= src arch::subtag-arrayH)
          (! check-arrayH-rank src 2)
          (! check-arrayH-flags src
             (dpb safe sparc::arrayH.flags-cell-subtag-byte
                  (ash 1 $arh_simple_bit))))
        (unless i-known-fixnum
          (! trap-unless-tag= unscaled-i arch::tag-fixnum))
        (unless j-known-fixnum
          (! trap-unless-tag= unscaled-j arch::tag-fixnum)))
      (with-imm-temps () (dim1 idx-reg)
        (unless constidx
          (if safe                    
              (! check-2d-bound dim1 unscaled-i unscaled-j src)
              (! 2d-dim1 dim1 src))
          (! 2d-df-scaled-index idx-reg src unscaled-i unscaled-j dim1))
        (with-node-temps () (v)
          (! array-data-vector-ref v src)
          (when (and constidx (>= constidx sparc::max-64-bit-constant-index))
            (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-dfloat-offset
                                                   (ash constidx 3)))
            (setq constidx nil))
          (if constidx
            (! misc-ref-c-double-float vreg v constidx)
            (! misc-ref-double-float vreg v idx-reg))))
      (^))))

(defun sparc2-sf-vref (seg vreg xfer vector index safe)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((index-known-fixnum (acode-fixnum-form-p index))
           (src)
           (unscaled-idx))
      (if (or safe (not index-known-fixnum))
        (multiple-value-setq (src unscaled-idx)
          (sparc2-two-untargeted-reg-forms seg vector sparc::%arg_y index sparc::%arg_z))
        (setq src (sparc2-one-untargeted-reg-form seg vector sparc::%arg_z)))
      (when safe
        (if (typep safe 'fixnum)
          (! trap-unless-typecode= src safe))
        (unless index-known-fixnum
          (! trap-unless-tag= unscaled-idx arch::tag-fixnum))
        (! check-misc-bound unscaled-idx src))
      (if (and index-known-fixnum (<= index-known-fixnum sparc::max-32-bit-constant-index))
        (! misc-ref-c-single-float vreg src index-known-fixnum)
        (with-imm-temps () (idx-reg)
          (if index-known-fixnum
            (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-dfloat-offset (ash index-known-fixnum 3)))
            (! scale-32bit-misc-index idx-reg unscaled-idx))
          (! misc-ref-single-float vreg src idx-reg)))
      (^))))

          


(defun sparc2-constant-value-ok-for-subtag (subtag form)
  (declare (fixnum subtag))
  (if (and (acode-p form)
           (or (eq (acode-operator form) (%nx1-operator immediate))
               (eq (acode-operator form) (%nx1-operator fixnum))))
    (let* ((val (%cadr form))
           (typep (cond ((= subtag arch::subtag-s32-vector)
                         (typep val '(signed-byte 32)))
                        ((= subtag arch::subtag-single-float-vector)
                         (typep val 'short-float))
                        ((= subtag arch::subtag-double-float-vector)
                         (typep val 'double-float))
                        ((<= subtag arch::max-32-bit-ivector-subtag)
                         (typep val '(unsigned-byte 32)))
                        ((= subtag arch::subtag-simple-base-string)
                         (typep val 'base-char))
                        ((= subtag arch::subtag-s8-vector)
                         (typep val '(signed-byte 8)))
                        ((= subtag arch::subtag-u8-vector)
                         (typep val '(unsigned-byte 8)))
                        ((= subtag arch::subtag-simple-general-string)
                         (typep val 'character))
                        ((= subtag arch::subtag-s16-vector) 
                         (typep val '(signed-byte 16)))
                        ((= subtag arch::subtag-u16-vector)
                         (typep val '(unsigned-byte 16)))
                        ((= subtag arch::subtag-bit-vector)
                         (typep val 'bit)))))
      (if typep val))))

(defun sparc2-vset (seg vreg xfer subtag vector index value safe)
  (declare (fixnum subtag))
  (if (= (the fixnum (logand subtag sparc::fulltagmask)) sparc::fulltag-nodeheader)
    (sparc2-misc-node-set seg vreg xfer vector index value safe)
    (if (and (= subtag arch::subtag-double-float-vector)
             (or (null vreg) (= (hard-regspec-class vreg) hard-reg-class-fpr)))
      (sparc2-df-vset seg vreg xfer vector index value safe)
      (with-sparc-local-vinsn-macros (seg vreg xfer)
        (unless (= (the fixnum (logand subtag sparc::fulltagmask)) sparc::fulltag-immheader)
          (error "Bug: not a SPARC subtag: ~s" subtag))
        (let* ((index-known-fixnum (acode-fixnum-form-p index))
               (constval (sparc2-constant-value-ok-for-subtag subtag value))
               (need-val-reg (or vreg (not constval)))
               (unscaled-idx nil)
               (idx-reg nil)
               (val-reg)
               (src nil))
          (if (or safe (not index-known-fixnum))
            (if need-val-reg
              (multiple-value-setq (src unscaled-idx val-reg)
                (sparc2-three-untargeted-reg-forms seg vector sparc::%arg_x index sparc::%arg_y value sparc::%arg_z))
              (multiple-value-setq (src unscaled-idx)
                (sparc2-two-untargeted-reg-forms seg vector sparc::%arg_y index sparc::%arg_z)))
            (if need-val-reg
              (multiple-value-setq (src val-reg)
                (sparc2-two-untargeted-reg-forms seg vector sparc::%arg_y value sparc::%arg_z))
              (setq src (sparc2-one-untargeted-reg-form seg vector sparc::%arg_z))))
          (when safe
            (if (typep safe 'fixnum)
              (! trap-unless-typecode= src safe))
            (unless index-known-fixnum
              (! trap-unless-tag= unscaled-idx arch::tag-fixnum))
            (! check-misc-bound unscaled-idx src))
          (with-imm-temps () (temp)
            (cond ((<= subtag arch::max-32-bit-ivector-subtag)
                   (if constval
                     (sparc2-lwi seg temp constval)
                     (cond ((= subtag arch::subtag-single-float-vector)
                            (when safe
                              (! trap-unless-typecode= val-reg arch::subtag-single-float))
                            (! misc-ref-c-u32 temp val-reg sparc::single-float.value-cell))
                           ((= subtag arch::subtag-s32-vector)
                            (! unbox-s32 temp val-reg))
                           (t
                            (! unbox-u32 temp val-reg))))
                   (if (and index-known-fixnum 
                            (<= index-known-fixnum sparc::max-32-bit-constant-index))
                     (! misc-set-c-u32 temp src index-known-fixnum)
                     (progn
                       (setq idx-reg (make-unwired-lreg (select-imm-temp :u32)))
                       (if index-known-fixnum
                         (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-data-offset (ash index-known-fixnum 2)))
                         (! scale-32bit-misc-index idx-reg unscaled-idx))
                       (! misc-set-u32 temp src idx-reg))))                   
                  ((<= subtag arch::max-8-bit-ivector-subtag)
                   (if constval
                     (sparc2-lwi seg temp (if (characterp constval) (char-code constval) constval))
                     (if safe
                       (cond ((= subtag arch::subtag-simple-base-string)
                              (! unbox-base-char temp val-reg))
                             ((= subtag arch::subtag-s8-vector)
                              (! unbox-s8 temp val-reg))
                             (t
                              (! unbox-u8 temp val-reg)))
                       (if (= subtag arch::subtag-simple-base-string)
                         (! character->code temp val-reg)
                         (! fixnum->u32 temp val-reg))))
                   (if (and index-known-fixnum 
                            (<= index-known-fixnum sparc::max-8-bit-constant-index))
                     (! misc-set-c-u8 temp src index-known-fixnum)
                     (progn
                       (setq idx-reg (make-unwired-lreg (select-imm-temp :u32)))
                       (if index-known-fixnum
                         (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-data-offset index-known-fixnum))
                         (! scale-8bit-misc-index idx-reg unscaled-idx))
                       (! misc-set-u8 temp src idx-reg))))
                  ((<= subtag arch::max-16-bit-ivector-subtag)
                   (if constval
                     (sparc2-lwi seg temp (if (characterp constval) (char-code constval) constval))
                     (if safe
                       (cond ((= subtag arch::subtag-simple-general-string)
                              (! unbox-character temp val-reg))
                             ((= subtag arch::subtag-s16-vector)
                              (! unbox-s16 temp val-reg))
                             (t
                              (! unbox-u16 temp val-reg)))
                       (if (= subtag arch::subtag-simple-general-string)
                         (! character->code temp val-reg)
                         (! fixnum->u32 temp val-reg))))
                   (if (and index-known-fixnum 
                            (<= index-known-fixnum sparc::max-16-bit-constant-index))
                     (! misc-set-c-u16 temp src index-known-fixnum)
                     (progn
                       (setq idx-reg (make-unwired-lreg (select-imm-temp :u32)))
                       (if index-known-fixnum
                         (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-data-offset (ash index-known-fixnum 1)))
                         (! scale-16bit-misc-index idx-reg unscaled-idx))
                       (! misc-set-u16 temp src idx-reg))))
                  ((= subtag arch::subtag-double-float-vector)
                   (if safe
                     (! get-double? 0 val-reg)
                     (! get-double 0 val-reg))
                   (if (and index-known-fixnum 
                            (<= index-known-fixnum sparc::max-64-bit-constant-index))
                     (! misc-set-c-double-float 0 src index-known-fixnum)
                     (progn
                       (setq idx-reg temp)
                       (if index-known-fixnum
                         (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-dfloat-offset (ash index-known-fixnum 3)))
                         (! scale-64bit-misc-index idx-reg unscaled-idx))
                       (! misc-set-double-float 0 src idx-reg))))
                  (t
                   ; bit-vector case.
                   ; It's easiest to do this when the bitnumber is known (and a little easier still
                   ; if the value's known.)
                   (if (and index-known-fixnum (<= index-known-fixnum sparc::max-1-bit-constant-index))
                     (let* ((word-index (ash index-known-fixnum -5))
                            (bit-number (logand index-known-fixnum #x1f)))
                       (! misc-ref-c-u32 temp src word-index)
                       (if constval                         
                         (if (zerop constval)
                           (! set-constant-ppc-bit-to-0 temp temp bit-number)
                           (! set-constant-ppc-bit-to-1 temp temp bit-number))
                         (with-imm-temps () (bitval)
                           (! unbox-bit bitval val-reg)
                           (! set-constant-ppc-bit-to-variable-value temp temp bitval bit-number)))
                       (! misc-set-c-u32 temp src word-index))
                     ; When the bit-number isn't known, we have to do one of the following:
                     ; A) If the value's known:
                     ;   1) generate a mask with a 1 in the "bitnum" bit and 0s elsewhere.
                     ;   2) Grab the word out of the vector.
                     ;   3) If the value's 0, do an ANDC with the mask and word, else an OR.
                     ; B) When the value's not known:
                     ;   1) Extract the value into SPARC bit 0 of some register, trapping if value not a bit.
                     ;   2) Shift the value right "bitnum" bits.
                     ;   3) Generate a mask with a 1 in the "bitnum" bit and 0s elsewhere.
                     ;   4) Reference the word, ANDC it with the mask, OR the shifted value in.
                     (with-imm-temps () (word-index bit-number)
                       (! scale-1bit-misc-index word-index bit-number unscaled-idx)
                       (if constval
                         (progn
                           (! lwi temp #x80000000)
                           (! shift-right-variable-word bit-number temp bit-number)       ; (A1)
                           (! misc-ref-u32 temp src word-index)   ; (A2)
                           (if (zerop constval)   ; (A3)
                             (! u32logandc2 temp temp bit-number)
                             (! u32logior temp temp bit-number)))
                         (with-imm-temps () (bitval)
                           (! unbox-bit-bit0 bitval val-reg)      ; (B1)
                           (! shift-right-variable-word bitval bitval bit-number)         ; (B2)
                           (! lwi temp #x80000000)
                           (! shift-right-variable-word bit-number temp bit-number)       ; (B3)
                           (! misc-ref-u32 temp src word-index)
                           (! u32logandc2 temp temp bit-number)      ; clear bit-number'th bit
                           (! u32logior temp temp bitval)))       ; (B4)                     
                       (! misc-set-u32 temp src word-index))))))
          (when vreg (<- val-reg))
          (^))))))

;; In this case, the destination (vreg) is either an FPR or null, so
;; we can maybe avoid boxing the value.
(defun sparc2-df-vset (seg vreg xfer vector index value safe)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((index-known-fixnum (acode-fixnum-form-p index))
           (src nil)
           (unscaled-idx nil))
      (with-fp-target () fp-val
        (if (or safe (not index-known-fixnum))
          (multiple-value-setq (src unscaled-idx fp-val)
            (sparc2-three-untargeted-reg-forms seg vector sparc::%arg_y index sparc::%arg_z value (or vreg fp-val)))
          (multiple-value-setq (src fp-val)
            (sparc2-two-untargeted-reg-forms seg vector sparc::%arg_z value (or vreg fp-val))))
        (when safe
            (if (typep safe 'fixnum)
              (! trap-unless-typecode= src safe))
            (unless index-known-fixnum
              (! trap-unless-tag= unscaled-idx arch::tag-fixnum))
            (! check-misc-bound unscaled-idx src))
        (if (and index-known-fixnum
                 (<= index-known-fixnum sparc::max-64-bit-constant-index))
          (! misc-set-c-double-float fp-val src index-known-fixnum)
          (with-imm-temps () (idx-reg)
            (if index-known-fixnum
              (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-dfloat-offset (ash index-known-fixnum 3)))
              (! scale-64bit-misc-index idx-reg unscaled-idx))
            (! misc-set-double-float fp-val src idx-reg)))
        (<- fp-val)                     ; should be a no-op in this case
        (^)))))

            
          
        


(defun sparc2-tail-call-alias (immref sym &optional arglist)
  (let ((alias (cdr (assq sym *sparc2-tail-call-aliases*))))
    (if (and alias (or (null arglist) (eq (+ (length (car arglist)) (length (cadr arglist))) (cdr alias))))
      (make-acode (%nx1-operator immediate) (car alias))
      immref)))

; If BODY is essentially an APPLY involving an &rest arg, try to avoid
; consing it.
(defun sparc2-eliminate-&rest (body rest key-p auxen rest-values)
  (when (and rest (not key-p) (not (cadr auxen)) rest-values)
    (when (eq (logand (the fixnum (nx-var-bits rest))
                      (logior $vsetqmask (ash -1 $vbitspecial)
                              (ash 1 $vbitclosed) (ash 1 $vbitsetq) (ash 1 $vbitcloseddownward)))
              0)               ; Nothing but simple references
      (do* ()
           ((not (acode-p body)))
        (let* ((op (acode-operator body)))
          (if (or (eq op (%nx1-operator lexical-function-call))
                  (eq op (%nx1-operator call)))
            (destructuring-bind (fn-form (stack-args reg-args) &optional spread-p) (%cdr body)
               (unless (and (eq spread-p t)
                           (eq (sparc2-lexical-reference-p (%car reg-args)) rest))
                (return nil))
              (flet ((independent-of-all-values (form)        
                       (setq form (acode-unwrapped-form form))
                       (or (sparc-constant-form-p form)
                           (let* ((lexref (sparc2-lexical-reference-p form)))
                             (and lexref 
                                  (neq lexref rest)
                                  (dolist (val rest-values t)
                                    (unless (sparc2-var-not-set-by-form-p lexref val)
                                      (return))))))))
                (unless (or (eq op (%nx1-operator lexical-function-call))
                            (independent-of-all-values fn-form))
                  (return nil))
                (if (dolist (s stack-args t)
                          (unless (independent-of-all-values s)
                            (return nil)))
                  (let* ((arglist (append stack-args rest-values)))
                    (return
                     (make-acode op 
                                 fn-form 
                                 (if (<= (length arglist) $numsparcargregs)
                                   (list nil (reverse arglist))
                                   (list (butlast arglist $numsparcargregs)
                                         (reverse (last arglist $numsparcargregs))))
                                 nil)))
                  (return nil))))
            (if (eq op (%nx1-operator local-block))
              (setq body (%cadr body))
              (if (and (eq op (%nx1-operator if))
                       (eq (sparc2-lexical-reference-p (%cadr body)) rest))
                (setq body (%caddr body))
                (return nil)))))))))

(defun sparc2-call-fn (seg vreg xfer fn arglist spread-p)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (when spread-p
      (destructuring-bind (stack-args reg-args) arglist
        (when (and (null (cdr reg-args))
                   (nx-null (acode-unwrapped-form (car reg-args))))
          (setq spread-p nil)
          (let* ((nargs (length stack-args)))
            (declare (fixnum nargs))
            (if (<= nargs $numsparcargregs)
              (setq arglist (list nil (reverse stack-args)))
              (setq arglist (list (butlast stack-args $numsparcargregs) (reverse (last stack-args $numsparcargregs)))))))))
    (let* ((lexref (sparc2-lexical-reference-p fn))
           (simple-case (or (fixnump fn) 
                            (sparc2-immediate-function-p fn)
                            (and 
                             lexref
                             (not spread-p)
                             (flet ((all-simple (args)
                                      (dolist (arg args t)
                                        (when (and arg (not (sparc2-var-not-set-by-form-p lexref arg)))
                                          (return)))))
                               (and (all-simple (car arglist))
                                    (all-simple (cadr arglist))
                                    (setq fn (var-ea lexref)))))))
           (cstack *sparc2-cstack*)
           (top *sparc2-top-vstack-lcell*)
           (vstack *sparc2-vstack*))
      (setq xfer (or xfer 0))
      (when (and (eq xfer $backend-return)
                 (eq 0 *sparc2-undo-count*)
                 (acode-p fn)
                 (eq (acode-operator fn) (%nx1-operator immediate))
                 (symbolp (cadr fn)))
        (setq fn (sparc2-tail-call-alias fn (%cadr fn) arglist)))
      
      (if (and (eq xfer $backend-return) (not (sparc2-tailcallok xfer)))
        (progn
          (sparc2-call-fn seg vreg $backend-mvpass fn arglist spread-p)
          (sparc2-set-vstack (%i+ (if simple-case 0 4) vstack))
          (setq  *sparc2-cstack* cstack)
          (let ((*sparc2-returning-values* t)) (sparc2-do-return seg)))
        (let* ((mv-p (sparc2-mv-p xfer)))
          (unless simple-case
            (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg fn sparc::%arg_z))
            (setq fn (sparc2-vloc-ea vstack)))
          (sparc2-invoke-fn seg fn (sparc2-arglist seg arglist) spread-p xfer)
          (if (and (%ilogbitp $backend-mvpass-bit xfer)
                   (not simple-case))
            (progn
              (! call-subprim .SPsave-values)
              (! vstack-discard 1)
              (sparc2-set-nargs seg 0)
              (! call-subprim .SPrecover-values))
            (unless (or mv-p simple-case)
              (! vstack-discard 1)))
          (sparc2-set-vstack vstack)
          (setq *sparc2-top-vstack-lcell* top)
          (setq *sparc2-cstack* cstack)
          (when (or (%ilogbitp $backend-mvpass-bit xfer) (not mv-p))
            (<- sparc::%arg_z)
            (sparc2-branch seg (%ilogand2 (%ilognot $backend-mvpass-mask) xfer) vreg))))
      nil)))

(defun sparc2-restore-full-lisp-context (seg)
  (with-sparc-local-vinsn-macros (seg)
    (if *sparc2-open-code-inline*
      (! restore-full-lisp-context)
      (! restore-full-lisp-context-ool))))

(defun sparc2-call-symbol (seg jump-p)
  ; fname contains a symbol; we can either call it via
  ; a call to .SPjmpsym or expand the instructions inline.
  ; Since the branches are unconditional, the call doesn't
  ; cost much, but doing the instructions inline would give
  ; an instruction scheduler some opportunities to improve
  ; performance, so this isn't a strict time/speed tradeoff.
  ; This should probably dispatch on something other than
  ; *sparc2-open-code-inline*, since that does imply a time/speed
  ; tradeoff.
  (with-sparc-local-vinsn-macros (seg)
    (if *sparc2-open-code-inline*
      (if jump-p
        (! jump-known-symbol)
        (! call-known-symbol))
      (if jump-p
        (! jump-subprim .SPjmpsym)
        (! call-subprim .SPjmpsym)))))

; Nargs = nil -> multiple-value case.
(defun sparc2-invoke-fn (seg fn nargs spread-p xfer)
  (with-sparc-local-vinsn-macros (seg)
    (let* ((f-op (acode-unwrapped-form fn))
           (immp (and (consp f-op)
                      (eq (%car f-op) (%nx1-operator immediate))))
           (symp (and immp (symbolp (%cadr f-op))))
           (label-p (and (fixnump fn) 
                         (locally (declare (fixnum fn))
                           (and (= fn -1) (- fn)))))
           (tail-p (eq xfer $backend-return))
           (func (if (consp f-op) (%cadr f-op)))
           (a-reg nil)
           (lfunp (and (acode-p f-op) 
                       (eq (acode-operator f-op) (%nx1-operator simple-function))))
           (expression-p (and (fixnump fn) (not label-p)))
           (callable (or symp lfunp label-p))
           (destreg (if symp sparc::%fname (if lfunp sparc::%nfn (unless label-p sparc::%temp0)))))
      (when expression-p
        ;Have to do this before spread args, since might be vsp-relative.
        (if nargs
          (sparc2-do-lexical-reference seg destreg fn)
          (sparc2-copy-register seg destreg fn)))
      (if (or symp lfunp)
        (setq func (if symp (sparc2-symbol-entry-locative func)
                       (sparc2-afunc-lfun-ref func))
              a-reg (sparc2-register-constant-p func)))
      (when tail-p
        #-no-compiler-bugs
        (unless (or immp symp lfunp (fixnump fn)) (error "Well, well, well.  How could this have happened ?"))
        (when a-reg
          (sparc2-copy-register seg destreg a-reg))
        (unless spread-p
          (if nargs
            (sparc2-restore-nvrs seg *sparc2-register-restore-ea* *sparc2-register-restore-count*)
            (when *sparc2-register-restore-count*
              (with-imm-temps () (vsp0)
                (! fixnum-add vsp0 sparc::%vsp sparc::%nargs)
                (sparc2-restore-nvrs seg *sparc2-register-restore-ea* *sparc2-register-restore-count* vsp0))))))
       (if spread-p
         (progn
           (sparc2-set-nargs seg (%i- nargs 1))
           (when (and tail-p *sparc2-register-restore-count*)
             (! copy-gpr sparc::%temp1 sparc::%vsp))          ; .SPspread-lexpr-z & .SPspreadargz preserve temp1
           (! call-subprim (if (eq spread-p 0) .SPspread-lexpr-z .SPspreadargz))
           (when (and tail-p *sparc2-register-restore-count*)
             (sparc2-restore-nvrs seg *sparc2-register-restore-ea* *sparc2-register-restore-count* sparc::%temp1)))
        (if nargs
          (sparc2-set-nargs seg nargs) 
          (! call-subprim .SPvpopargregs)))
      (if callable
        (if (not tail-p)
          (if (sparc2-mvpass-p xfer)
            (progn
              (if label-p
                (sparc2-copy-register seg sparc::%temp0 sparc::%fn)
                (if a-reg
                  (sparc2-copy-register seg sparc::%temp0 a-reg)
                  (sparc2-store-immediate seg func sparc::%temp0)))
              (! call-subprim .SPmvpass))
            (progn 
              (if label-p
                (progn
                  (sparc2-copy-register seg sparc::%nfn sparc::%fn)
                  (! call-label (aref *backend-labels* 1)))
                (progn
                  (if a-reg
                    (sparc2-copy-register seg destreg a-reg)
                    (sparc2-store-immediate seg func destreg))
                  (if symp
                    (sparc2-call-symbol seg nil)
                    (! call-known-function))))))
          (progn
            (sparc2-unwind-stack seg xfer 0 0 #x7ff)
            (if (and (not spread-p) nargs (%i<= nargs $numsparcargregs))
              (progn
                (if label-p
                  (sparc2-copy-register seg sparc::%nfn sparc::%fn))
                (unless (or label-p a-reg) (sparc2-store-immediate seg func destreg))
                (sparc2-restore-full-lisp-context seg)
                (if label-p
                  (! jump (aref *backend-labels* 1))
                  (progn
                    (if symp
                      (sparc2-call-symbol seg t)
                      (! jump-known-function)))))
              (progn
                (if label-p
                  (sparc2-copy-register seg sparc::%nfn sparc::%fn)
                  (unless a-reg (sparc2-store-immediate seg func destreg)))
                (! jump-subprim 
                   (cond ((or spread-p (null nargs)) (if symp .SPtcallsymgen .SPtcallnfngen))
                         ((%i> nargs $numsparcargregs) (if symp .SPtcallsymslide .SPtcallnfnslide))
                         (t (if symp .SPtcallsymvsp .SPtcallnfnvsp))))))))
        ; The general (funcall) case.
        (progn
          (unless (fixnump fn) 
            (sparc2-one-targeted-reg-form seg fn destreg))
          (if (not tail-p)
            (if (sparc2-mvpass-p xfer)
              (! call-subprim .SPmvpass)
              (! call-subprim .SPfuncall))                  
            (! jump-subprim 
               (cond ((or (null nargs) spread-p) .SPtfuncallgen)
                     ((%i> nargs $numsparcargregs) .SPtfuncallslide)
                     (t .SPtfuncallvsp)))))))
    nil))

(defun sparc2-seq-fbind (seg vreg xfer vars afuncs body p2decls)
  (let* ((old-stack (sparc2-encode-stack))
         (copy afuncs)
         (func nil))
    (with-sparc-p2-declarations p2decls 
      (dolist (var vars) 
        (when (neq 0 (afunc-fn-refcount (setq func (pop afuncs))))
          (sparc2-seq-bind-var seg var (nx1-afunc-ref func))))
      (sparc2-undo-body seg vreg xfer body old-stack)
      (dolist (var vars)
        (when (neq 0 (afunc-fn-refcount (setq func (pop copy))))
          (sparc2-close-var seg var))))))

(defun sparc2-make-closure (seg afunc downward-p)
  (with-sparc-local-vinsn-macros (seg)
    (flet ((var-to-reg (var target)
             (let* ((ea (var-ea (var-bits var))))
               (if ea
                 (progn
                   (sparc2-addrspec-to-reg seg (sparc2-ea-open ea) target)
                   target)
                 sparc::%rnil)))
           (set-some-cells (dest cellno c0 c1 c2 c3)
             (declare (fixnum cellno))
             (! misc-set-c-node c0 dest cellno)
             (incf cellno)
             (when c1
               (! misc-set-c-node c1 dest cellno)
               (incf cellno)
               (when c2
                 (! misc-set-c-node c2 dest cellno)
                 (incf cellno)
                 (when c3
                   (! misc-set-c-node c3 dest cellno)
                   (incf cellno))))
             cellno))
      (let* ((inherited-vars (afunc-inherited-vars afunc))
             (dest (make-wired-lreg sparc::%arg_z))
             (vsize (+ (length inherited-vars) 
                       2                  ; %closure-code%, afunc
                       2)))               ; name, lfun-bits
        (declare (list inherited-vars))
        (if downward-p
          (progn
            (let* ((*sparc2-vstack* *sparc2-vstack*)
                   (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*))
              (sparc2-lwi seg sparc::%arg_x (ash arch::subtag-function sparc::fixnumshift))
              (! %closure-code% sparc::%arg_y)
              (sparc2-store-immediate seg (sparc2-afunc-lfun-ref afunc) sparc::%arg_z)
              (sparc2-vpush-register-arg seg sparc::%arg_x)
              (sparc2-vpush-register-arg seg sparc::%arg_y)
              (sparc2-vpush-register-arg seg sparc::%arg_z)
              ; Could be smarter about memory traffic here.
              (dolist (v inherited-vars)
                (sparc2-vpush-register-arg seg (var-to-reg v sparc::%arg_z)))       
              (sparc2-vpush-register-arg seg sparc::%rnil)
              (sparc2-lwi seg sparc::%arg_z (ash (ash 1 $lfbits-trampoline-bit) sparc::fixnumshift))
              (sparc2-vpush-register-arg seg sparc::%arg_z)
              (sparc2-set-nargs seg (1+ vsize))     ; account for subtag
              (! call-subprim-0 dest .SPstkgvector))
            (sparc2-open-undo $undostkblk))
          (let* ((cell 0))
            (declare (fixnum cell))
            (sparc2-lwi seg sparc::%imm0 (logior (ash vsize sparc::num-subtag-bits) arch::subtag-function))
            (! %alloc-misc-fixed dest sparc::%imm0 (ash vsize sparc::word-shift))
            (! %closure-code% sparc::%arg_x)
            (sparc2-store-immediate seg (sparc2-afunc-lfun-ref afunc) sparc::%arg_y)
            (with-node-temps (sparc::%arg_z) (t0 t1 t2 t3)
              (do* ((ccode sparc::%arg_x nil)
                    (func sparc::%arg_y nil))
                   ((null inherited-vars))
                (let* ((t0r (or ccode (if inherited-vars (var-to-reg (pop inherited-vars) t0))))
                       (t1r (or func (if inherited-vars (var-to-reg (pop inherited-vars) t1))))
                       (t2r (if inherited-vars (var-to-reg (pop inherited-vars) t2)))
                       (t3r (if inherited-vars (var-to-reg (pop inherited-vars) t3))))
                  (setq cell (set-some-cells dest cell t0r t1r t2r t3r)))))
            (sparc2-lwi seg sparc::%arg_y (ash (ash 1 $lfbits-trampoline-bit) sparc::fixnumshift))
            (! misc-set-c-node sparc::%rnil dest cell)
            (! misc-set-c-node sparc::%arg_y dest (1+ cell))))
        dest))))
        
(defun sparc2-symbol-entry-locative (sym)
  (setq sym (require-type sym 'symbol))
  (when (eq sym '%call-next-method-with-args)
    (setf (afunc-bits *sparc2-cur-afunc*)
          (%ilogior (%ilsl $fbitnextmethargsp 1) (afunc-bits *sparc2-cur-afunc*))))
  (or (assq sym *sparc2-fcells*)
      (let ((new (list sym)))
        (push new *sparc2-fcells*)
        new)))

(defun sparc2-symbol-value-locative (sym)
  (setq sym (require-type sym 'symbol))
  (or (assq sym *sparc2-vcells*)
      (let ((new (list sym)))
        (push new *sparc2-vcells*)
        new)))

(defun sparc2-symbol-locative-p (imm)
  (and (consp imm)
       (or (memq imm *sparc2-vcells*)
           (memq imm *sparc2-fcells*))))




(defun sparc2-immediate-function-p (f)
  (setq f (acode-unwrapped-form f))
  (and (acode-p f)
       (or (eq (%car f) (%nx1-operator immediate))
           (eq (%car f) (%nx1-operator simple-function)))))

(defun sparc-constant-form-p (form)
  (setq form (nx-untyped-form form))
  (if form
    (or (nx-null form)
        (nx-t form)
        (and (consp form)
             (or (eq (acode-operator form) (%nx1-operator immediate))
                 (eq (acode-operator form) (%nx1-operator fixnum))
                 (eq (acode-operator form) (%nx1-operator simple-function)))))))

(defun sparc2-long-constant-p (form)
  (setq form (acode-unwrapped-form form))
  (or (acode-fixnum-form-p form)
      (and (acode-p form)
           (eq (acode-operator form) (%nx1-operator immediate))
           (setq form (%cadr form))
           (if (integerp form) 
             form
             (progn
               (if (symbolp form) (setq form (symbol-name form)))
               (if (and (stringp form) (eql (length form) 4))
                 (%stack-block ((buf 4))
                   (%put-ostype buf form)
                   (%get-unsigned-long buf))
                 (if (characterp form) (%char-code form))))))))


(defun sparc-side-effect-free-form-p (form)
  (when (consp (setq form (acode-unwrapped-form form)))
    (or (sparc-constant-form-p form)
        ;(eq (acode-operator form) (%nx1-operator bound-special-ref))
        (if (eq (acode-operator form) (%nx1-operator lexical-reference))
          (not (%ilogbitp $vbitsetq (nx-var-bits (%cadr form))))))))

(defun sparc2-formlist (seg stkargs &optional revregargs)
  (let* ((nregs (length revregargs))
         (n nregs))
    (declare (fixnum n))
    (dolist (arg stkargs)
      (let* ((reg (sparc2-one-untargeted-reg-form seg arg sparc::%arg_z)))
        (sparc2-vpush-register-arg seg reg)
        (incf n)))
    (when revregargs
      (let* ((zform (%car revregargs))
             (yform (%cadr revregargs))
             (xform (%caddr revregargs)))
        (if (eq 3 nregs)
          (sparc2-three-targeted-reg-forms seg xform sparc::%arg_x yform sparc::%arg_y zform sparc::%arg_z)
          (if (eq 2 nregs)
            (sparc2-two-targeted-reg-forms seg yform sparc::%arg_y zform sparc::%arg_z)
            (sparc2-one-targeted-reg-form seg zform sparc::%arg_z)))))
    n))

(defun sparc2-arglist (seg args)
  (sparc2-formlist seg (car args) (cadr args)))



; treat form as a 32-bit immediate value and load it into immreg.
; This is the "lenient" version of 32-bit-ness; OSTYPEs and chars
; count, and we don't care about the integer's sign.

(defun sparc2-unboxed-integer-arg-to-reg (seg form immreg)
  (with-sparc-local-vinsn-macros (seg)
    (let* ((value (sparc2-long-constant-p form)))
      (if value
        (if (eql value 0)
          (make-wired-lreg sparc::%rzero :mode (gpr-mode-name-value :u32))
          (progn
            (unless (typep immreg 'lreg)
              (setq immreg (make-unwired-lreg immreg :mode (gpr-mode-name-value :u32))))
            (sparc2-lwi seg immreg value)
            immreg))
        (progn 
          (sparc2-one-targeted-reg-form seg form sparc::%arg_z)
          (! call-subprim .SPgetXlong)
          (make-wired-lreg sparc::%imm0 :mode (gpr-mode-name-value :u32)))))))


(defun sparc2-macptr-arg-to-reg (seg form address-reg)  
  (sparc2-one-targeted-reg-form seg
                              form 
                              (set-regspec-mode address-reg (gpr-mode-name-value :address))))


(defun sparc2-one-lreg-form (seg form lreg)
  (let ((is-float (= (hard-regspec-class lreg) hard-reg-class-fpr)))
    (if is-float
      (sparc2-form-float seg lreg nil form)
      (sparc2-form seg lreg nil form))
    lreg))

(defun sparc2-one-targeted-reg-form (seg form reg)
  (sparc2-one-lreg-form seg form (if (typep reg 'lreg) reg (make-wired-lreg reg))))

(defun sparc2-one-untargeted-lreg-form (seg form reg)
  (sparc2-one-lreg-form seg form (if (typep reg 'lreg) reg (make-unwired-lreg reg))))

; Evaluated form into lisp register "suggested", unless it's already sitting somewhere else.
(defun sparc2-one-untargeted-reg-form (seg form suggested)
  ; If the suggested reg isn't a gpr, just evaluate the form to the register
  (let* ((gpr-p (= (hard-regspec-class suggested) hard-reg-class-gpr))
         (node-p (if gpr-p (= (get-regspec-mode suggested) hard-reg-class-gpr-mode-node))))
    (if node-p
      (let* ((reg (backend-ea-physical-reg (sparc2-lexical-reference-ea form) hard-reg-class-gpr)))
        (if reg
          reg
          (if (nx-null form)
            sparc::%rnil
            (if (eql 0 (acode-fixnum-form-p form))
              sparc::%rzero
              (if (and (acode-p form) 
                       (eq (acode-operator form) (%nx1-operator immediate)) 
                       (setq reg (sparc2-register-constant-p (cadr form))))
                reg
                
                (sparc2-one-untargeted-lreg-form seg form suggested))))))
      (sparc2-one-untargeted-lreg-form seg form suggested))))
             

(defun sparc2-push-register (seg areg)
  (let* ((a-float (= (hard-regspec-class areg) hard-reg-class-fpr))
         (a-double (if a-float (= (get-regspec-mode areg) hard-reg-class-fpr-mode-double)))
         (a-node (unless a-float (= (get-regspec-mode areg) hard-reg-class-gpr-mode-node)))
         vinsn)
    (with-sparc-local-vinsn-macros (seg)
      (if a-node
        (setq vinsn (sparc2-vpush-register seg areg :node-temp))
        (progn
          (setq vinsn
                (if a-float
                  (if a-double
                    (! temp-push-double-float areg)
                    (! temp-push-single-float areg))
                  (! temp-push-unboxed-word areg)))
          (sparc2-open-undo $undostkblk)))
      vinsn)))

(defun sparc2-pop-register (seg areg)
  (let* ((a-float (= (hard-regspec-class areg) hard-reg-class-fpr))
         (a-double (if a-float (= (get-regspec-mode areg) hard-reg-class-fpr-mode-double)))
         (a-node (unless a-float (= (get-regspec-mode areg) hard-reg-class-gpr-mode-node)))
         vinsn)
    (with-sparc-local-vinsn-macros (seg)
      (if a-node
        (setq vinsn (sparc2-vpop-register seg areg))
        (progn
          (setq vinsn
                (if a-float
                  (if a-double
                    (! temp-pop-double-float areg)
                    (! temp-pop-single-float areg))
                  (! temp-pop-unboxed-word areg)))
          (sparc2-close-undo)))
      vinsn)))

(defun sparc2-acc-reg-for (reg)
  (let* ((class (hard-regspec-class reg))
         (mode (get-regspec-mode reg)))
    (declare (fixnum class mode))
    (cond ((= class hard-reg-class-fpr)
           (make-wired-lreg sparc::%f4 :class class :mode mode))
          ((= class hard-reg-class-gpr)
           (if (= mode hard-reg-class-gpr-mode-node)
             sparc::%arg_z
             (make-wired-lreg sparc::%imm0 :mode mode)))
          (t (error "Unknown register class for reg ~s" reg)))))
  

; we never leave the first form pushed (the 68K compiler had some subprims that
; would vpop the first argument out of line.)
(defun sparc2-two-targeted-reg-forms (seg aform areg bform breg)
  (let* ((avar (sparc2-lexical-reference-p aform))
         (atriv (sparc2-trivial-p bform))
         (aconst (and (not atriv) (or (sparc-side-effect-free-form-p aform)
                                      (if avar (sparc2-var-not-set-by-form-p avar bform)))))
         (apushed (not (or atriv aconst))))
    (progn
      (unless aconst
        (if atriv
          (sparc2-one-targeted-reg-form seg aform areg)
          (sparc2-push-register seg (sparc2-one-untargeted-reg-form seg aform (sparc2-acc-reg-for areg)))))
      (sparc2-one-targeted-reg-form seg bform breg)
      (if aconst
        (sparc2-one-targeted-reg-form seg aform areg)
        (if apushed
          (sparc2-pop-register seg areg))))
    (values areg breg)))


(defun sparc2-two-untargeted-reg-forms (seg aform areg bform breg)
  (let* ((avar (sparc2-lexical-reference-p aform))
         (adest areg)
         (bdest breg)
         (atriv (sparc2-trivial-p bform))
         (aconst (and (not atriv) (or (sparc-side-effect-free-form-p aform)
                                      (if avar (sparc2-var-not-set-by-form-p avar bform)))))
         (apushed (not (or atriv aconst))))
    (progn
      (unless aconst
        (if atriv
          (setq adest (sparc2-one-untargeted-reg-form seg aform areg))
          (sparc2-push-register seg (sparc2-one-untargeted-reg-form seg aform (sparc2-acc-reg-for areg)))))
      (setq bdest (sparc2-one-untargeted-reg-form seg bform breg))
      (if aconst
        (setq adest (sparc2-one-untargeted-reg-form seg aform areg))
        (if apushed
          (sparc2-pop-register seg areg))))
    (values adest bdest)))


(defun sparc2-three-targeted-reg-forms (seg aform areg bform breg cform creg)
  (let* ((atriv (or (null aform) 
                    (and (sparc2-trivial-p bform)
                         (sparc2-trivial-p cform))))
         (btriv (or (null bform)
                    (sparc2-trivial-p cform)))
         (aconst (and (not atriv) 
                      (or (sparc-side-effect-free-form-p aform)
                          (let ((avar (sparc2-lexical-reference-p aform)))
                            (and avar 
                                 (sparc2-var-not-set-by-form-p avar bform)
                                 (sparc2-var-not-set-by-form-p avar cform))))))
         (bconst (and (not btriv)
                      (or
                       (sparc-side-effect-free-form-p bform)
                       (let ((bvar (sparc2-lexical-reference-p bform)))
                         (and bvar (sparc2-var-not-set-by-form-p bvar cform)))))))
    (if (and aform (not aconst))
      (if atriv
        (sparc2-one-targeted-reg-form seg aform areg)
        (sparc2-push-register seg (sparc2-one-untargeted-reg-form seg aform (sparc2-acc-reg-for areg)))))
    (if (and bform (not bconst))
      (if btriv
        (sparc2-one-targeted-reg-form seg bform breg)
        (sparc2-push-register seg (sparc2-one-untargeted-reg-form seg bform (sparc2-acc-reg-for breg)))))
    (sparc2-one-targeted-reg-form seg cform creg)
    (unless btriv 
      (if bconst
        (sparc2-one-targeted-reg-form seg bform breg)
        (sparc2-pop-register seg breg)))
    (unless atriv
      (if aconst
        (sparc2-one-targeted-reg-form seg aform areg)
        (sparc2-pop-register seg areg)))
    (values areg breg creg)))

(defun sparc2-three-untargeted-reg-forms (seg aform areg bform breg cform creg)
  (let* ((atriv (or (null aform) 
                    (and (sparc2-trivial-p bform)
                         (sparc2-trivial-p cform))))
         (btriv (or (null bform)
                    (sparc2-trivial-p cform)))
         (aconst (and (not atriv) 
                      (or (sparc-side-effect-free-form-p aform)
                          (let ((avar (sparc2-lexical-reference-p aform)))
                            (and avar 
                                 (sparc2-var-not-set-by-form-p avar bform)
                                 (sparc2-var-not-set-by-form-p avar cform))))))
         (bconst (and (not btriv)
                      (or
                       (sparc-side-effect-free-form-p bform)
                       (let ((bvar (sparc2-lexical-reference-p bform)))
                         (and bvar (sparc2-var-not-set-by-form-p bvar cform))))))
         (adest areg)
         (bdest breg)
         (cdest creg))
    (if (and aform (not aconst))
      (if atriv
        (setq adest (sparc2-one-targeted-reg-form seg aform areg))
        (sparc2-push-register seg (sparc2-one-untargeted-reg-form seg aform (sparc2-acc-reg-for areg)))))
    (if (and bform (not bconst))
      (if btriv
        (setq bdest (sparc2-one-untargeted-reg-form seg bform breg))
        (sparc2-push-register seg (sparc2-one-untargeted-reg-form seg bform (sparc2-acc-reg-for breg)))))
    (setq cdest (sparc2-one-untargeted-reg-form seg cform creg))
    (unless btriv 
      (if bconst
        (setq bdest (sparc2-one-untargeted-reg-form seg bform breg))
        (sparc2-pop-register seg breg)))
    (unless atriv
      (if aconst
        (setq adest (sparc2-one-untargeted-reg-form seg aform areg))
        (sparc2-pop-register seg areg)))
    (values adest bdest cdest)))

(defun sparc2-four-untargeted-reg-forms (seg aform areg bform breg cform creg dform dreg)
  (let* ((atriv (or (null aform) 
                    (and (sparc2-trivial-p bform)
                         (sparc2-trivial-p cform)
                         (sparc2-trivial-p dform))))
         (btriv (or (null bform)
                    (and (sparc2-trivial-p cform)
                         (sparc2-trivial-p dform))))
         (ctriv (or (null cform)
                    (sparc2-trivial-p dform)))
         (aconst (and (not atriv) 
                      (or (sparc-side-effect-free-form-p aform)
                          (let ((avar (sparc2-lexical-reference-p aform)))
                            (and avar 
                                 (sparc2-var-not-set-by-form-p avar bform)
                                 (sparc2-var-not-set-by-form-p avar cform)
                                 (sparc2-var-not-set-by-form-p avar dform))))))
         (bconst (and (not btriv)
                      (or
                       (sparc-side-effect-free-form-p bform)
                       (let ((bvar (sparc2-lexical-reference-p bform)))
                         (and bvar
                              (sparc2-var-not-set-by-form-p bvar cform)
                              (sparc2-var-not-set-by-form-p bvar dform))))))
         (cconst (and (not ctriv)
                      (or
                       (sparc-side-effect-free-form-p cform)
                       (let ((cvar (sparc2-lexical-reference-p cform)))
                         (and cvar
                              (sparc2-var-not-set-by-form-p cvar dform))))))
         (adest areg)
         (bdest breg)
         (cdest creg)
         (ddest dreg))
    (if (and aform (not aconst))
      (if atriv
        (setq adest (sparc2-one-targeted-reg-form seg aform areg))
        (sparc2-push-register seg (sparc2-one-untargeted-reg-form seg aform (sparc2-acc-reg-for areg)))))
    (if (and bform (not bconst))
      (if btriv
        (setq bdest (sparc2-one-untargeted-reg-form seg bform breg))
        (sparc2-push-register seg (sparc2-one-untargeted-reg-form seg bform (sparc2-acc-reg-for breg)))))
    (if (and cform (not cconst))
      (if ctriv
        (setq cdest (sparc2-one-untargeted-reg-form seg cform creg))
        (sparc2-push-register seg (sparc2-one-untargeted-reg-form seg cform (sparc2-acc-reg-for creg)))))
    (setq ddest (sparc2-one-untargeted-reg-form seg dform dreg))
    (unless ctriv 
      (if cconst
        (setq cdest (sparc2-one-untargeted-reg-form seg cform creg))
        (sparc2-pop-register seg creg)))
    (unless btriv 
      (if bconst
        (setq bdest (sparc2-one-untargeted-reg-form seg bform breg))
        (sparc2-pop-register seg breg)))
    (unless atriv
      (if aconst
        (setq adest (sparc2-one-untargeted-reg-form seg aform areg))
        (sparc2-pop-register seg areg)))
    (values adest bdest cdest ddest)))

(defun sparc2-lwi (seg reg value)
  (with-sparc-local-vinsn-macros (seg)
    (if (typep value '(signed-byte 13))
       (! load-s13 reg value)
       (progn
	 (setq value (logand value #xffffffff))
	 (! lwi reg value)))))

(defun sparc2-multiple-value-body (seg form)
  (let* ((lab (backend-get-next-label))
         (*sparc2-vstack* *sparc2-vstack*)
         (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*)
         (old-stack (sparc2-encode-stack)))
    (with-sparc-local-vinsn-macros (seg)
      (sparc2-open-undo $undomvexpect)
      (sparc2-undo-body seg nil (%ilogior2 $backend-mvpass-mask lab) form old-stack)
      (@ lab))))

(defun sparc2-afunc-lfun-ref (afunc)
  (or
   (afunc-lfun afunc)
   (progn (pushnew afunc (afunc-fwd-refs *sparc2-cur-afunc*) :test #'eq)
          afunc)))

(defun sparc2-augment-arglist (afunc arglist &optional (maxregs $numsparcargregs))
  (let ((inherited-args (afunc-inherited-vars afunc)))
    (when inherited-args
      (let* ((current-afunc *sparc2-cur-afunc*)
             (stkargs (car arglist))
             (regargs (cadr arglist))
             (inhforms nil)
             (numregs (length regargs))
             (own-inhvars (afunc-inherited-vars current-afunc)))
        (dolist (var inherited-args)
          (let* ((root-var (nx-root-var var))
                 (other-guy 
                  (dolist (v own-inhvars #|(error "other guy not found")|# root-var)
                    (when (eq root-var (nx-root-var v)) (return v)))))
            (push (make-acode (%nx1-operator inherited-arg) other-guy) inhforms)))
        (dolist (form inhforms)
          (if (%i< numregs maxregs)
            (progn
              (setq regargs (nconc regargs (list form)))
              (setq numregs (%i+ numregs 1)))
            (push form stkargs)))
        (%rplaca (%cdr arglist) regargs) ; might have started out NIL.
        (%rplaca arglist stkargs)))) 
  arglist)




; There are other cases involving constants that are worth exploiting.
(defun sparc2-compare (seg vreg xfer i j cr-bit true-p)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((jconstant (acode-fixnum-form-p j))
           (js13 (typep jconstant '(signed-byte  #.(- 13 sparc::fixnumshift))))
           (iconstant (acode-fixnum-form-p i))
           (is13 (typep iconstant '(signed-byte  #.(- 13 sparc::fixnumshift))))                      
           (boolean (backend-crf-p vreg)))
      (if (and boolean (or js13 is13))
        (let* ((reg (sparc2-one-untargeted-reg-form seg (if js13 i j) sparc::%arg_z)))
          (! compare-signed-s13const  reg (ash (if js13 jconstant iconstant) sparc::fixnumshift))
          (unless (or js13 (eq cr-bit sparc::conde)) 
            (setq cr-bit (sparc2-reverse-cc cr-bit)))
          (^ cr-bit true-p))
        (if (and (eq cr-bit sparc::conde) 
                 (or js13 is13))
          (sparc2-test-reg-%izerop 
           seg 
           vreg 
           xfer 
           (sparc2-one-untargeted-reg-form 
            seg 
            (if js13 i j) 
            sparc::%arg_z) 
           cr-bit 
           true-p 
           (ash (if js13 jconstant iconstant) sparc::fixnumshift))
          (multiple-value-bind (ireg jreg) (sparc2-two-untargeted-reg-forms seg i sparc::%arg_y j sparc::%arg_z)
            (sparc2-compare-registers seg vreg xfer ireg jreg cr-bit true-p)))))))

(defun sparc2-compare-registers (seg vreg xfer ireg jreg cr-bit true-p)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (if vreg
      (regspec-crf-gpr-case 
       (vreg dest)
       (progn
         (! compare  ireg jreg)
         (^ cr-bit true-p))
       (ensuring-node-target (target dest)
         (unless true-p (setq cr-bit (logxor 8 cr-bit)))
         (ecase cr-bit
           (#. sparc::conde  (! eq->boolean target ireg jreg))
	   (#. sparc::condne (! ne->boolean target ireg jreg))
           (#. sparc::condl (! lt->boolean target ireg jreg))
	   (#. sparc::condge  (! ge->boolean target ireg jreg))
	   (#. sparc::condg (! gt->boolean target ireg jreg))
           (#. sparc::condle (! le->boolean target ireg jreg)))
         (^)))
      (^))))


(defun sparc2-compare-single-float-registers (seg vreg xfer ireg jreg cr-bit true-p)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (if vreg
      (regspec-crf-gpr-case 
       (vreg dest)
       (progn
         (! single-float-compare ireg jreg)
         (^ cr-bit true-p))
       (progn
	 (ensuring-node-target (target dest)
           (unless true-p
	     (setq cr-bit (sparc-invert-fcc cr-bit)))
	   (! single-float-compare->boolean target (logand cr-bit 15) ireg jreg))
       (^)))  
      (^))))

(defun sparc2-compare-double-float-registers (seg vreg xfer ireg jreg cr-bit true-p)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (if vreg
      (regspec-crf-gpr-case 
       (vreg dest)
       (progn
         (! double-float-compare  ireg jreg)
         (^ cr-bit true-p))
       (progn
	 (ensuring-node-target (target dest)
          (unless true-p
	     (setq cr-bit (sparc-invert-fcc cr-bit)))
	   (! double-float-compare->boolean target (logand cr-bit 15) ireg jreg))
       (^)))  
      (^))))


(defun sparc2-immediate-form-p (form)
  (if (and (consp form)
           (or (eq (%car form) (%nx1-operator immediate))
               (eq (%car form) (%nx1-operator simple-function))))
    t))

(defun sparc2-test-%izerop (seg vreg xfer form cr-bit true-p)
  (sparc2-test-reg-%izerop seg vreg xfer (sparc2-one-untargeted-reg-form seg form sparc::%arg_z) cr-bit true-p 0))

(defun sparc2-test-reg-%izerop (seg vreg xfer reg cr-bit true-p  zero)
  (declare (fixnum reg zero))
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (regspec-crf-gpr-case 
     (vreg dest)
     (progn
       (! compare-signed-s13const reg zero)
       (^ cr-bit true-p))
     (with-imm-temps (reg) (b31-reg scaled)
       (if (zerop zero)
         (setq scaled reg)
         (! subtract-constant scaled reg zero))
       (ecase cr-bit
         (#. sparc::conde
          (if true-p
            (! eq0->bit31 b31-reg scaled)
            (! ne0->bit31 b31-reg scaled)))
         (#. sparc::condl
          (if true-p
            (! lt0->bit31 b31-reg scaled)
            (! ge0->bit31 b31-reg scaled)))
         (#. sparc::condg
          (if true-p
            (! gt0->bit31 b31-reg scaled)
            (! le0->bit31 b31-reg scaled))))
          (ensuring-node-target (target dest)
            (! bit31->truth target b31-reg))
       (^)))))

(defun sparc2-lexical-reference-ea (form &optional (no-closed-p t))
  (let* ((addr nil))
    (when (acode-p (setq form (acode-unwrapped-form form)))
      (if (eq (acode-operator form) (%nx1-operator lexical-reference))
        (unless (and no-closed-p (addrspec-vcell-p (setq addr (var-ea (%cadr form)))))
          addr)))))


(defun sparc2-vpush-register (seg src &optional why info attr)
  (with-sparc-local-vinsn-macros (seg)
    (prog1
      (! vpush-register src)
      (sparc2-new-vstack-lcell (or why :node) 4 (or attr 0) info)
      (sparc2-adjust-vstack +4))))

(defun sparc2-vpush-register-arg (seg src)
  (sparc2-vpush-register seg src :outgoing-argument))


(defun sparc2-vpop-register (seg dest)
  (with-sparc-local-vinsn-macros (seg)
    (prog1
      (! vpop-register dest)
      (setq *sparc2-top-vstack-lcell* (lcell-parent *sparc2-top-vstack-lcell*))
      (sparc2-adjust-vstack -4))))

(defun sparc2-copy-register (seg dest src)
  (with-sparc-local-vinsn-macros (seg)
    (when dest
      (unless src (error "Bug: no source register to copy."))
      (let* ((dest-gpr (backend-ea-physical-reg dest hard-reg-class-gpr))
             (src-gpr (backend-ea-physical-reg src hard-reg-class-gpr))
             (dest-fpr (backend-ea-physical-reg dest hard-reg-class-fpr))
             (src-fpr (backend-ea-physical-reg src hard-reg-class-fpr))
             (src-mode (get-regspec-mode src))
             (dest-mode (get-regspec-mode dest))
             (dest-crf (backend-ea-physical-reg dest hard-reg-class-crf)))
        (if (and dest-gpr (or (eql dest-gpr sparc::%rnil) (eql dest-gpr sparc::%rzero)))
          (break "Bad destination register: ~s" dest-gpr))
        (if (and dest-crf src-gpr)
          ; "Copying" a GPR to a CR field means comparing it to rnil
          (! compare  src sparc::%rnil)
          (if (and dest-gpr src-gpr)
            (if (eql src-gpr sparc::%rzero)        
              ; Rzero always contains 0, so we can
              ; save ourselves some trouble.
              ; This assumes that (LI dest-gpr 0) is easier
              ; on the register-renaming pipeline nonsense than
              ; (MR dest-gpr rzero) would be.
              (! lwi dest-gpr 0)
              (case dest-mode
                (#.hard-reg-class-gpr-mode-node      ; boxed result.
                 (case src-mode
                   (#.hard-reg-class-gpr-mode-node
                    (unless (eql  dest-gpr src-gpr)
                      (! copy-gpr dest src)))
                   (#.hard-reg-class-gpr-mode-u32
                    (! u32->integer dest src))
                   (#.hard-reg-class-gpr-mode-s32
                    (! s32->integer dest src))
                   (#.hard-reg-class-gpr-mode-u16
                    (! u16->fixnum dest src))
                   (#.hard-reg-class-gpr-mode-s16
                    (! s16->fixnum dest src))
                   (#.hard-reg-class-gpr-mode-u8
                    (! u8->fixnum dest src))
                   (#.hard-reg-class-gpr-mode-s8
                    (! s8->fixnum dest src))
                   (#.hard-reg-class-gpr-mode-address
                    (! macptr->heap dest src))))
                ((#.hard-reg-class-gpr-mode-u32
                  #.hard-reg-class-gpr-mode-address)
                 (case src-mode
                   (#.hard-reg-class-gpr-mode-node
                    (let* ((src-type (get-node-regspec-type-modes src)))
                      (declare (fixnum src-type))
                      (case dest-mode
                        (#.hard-reg-class-gpr-mode-u32
                         (! unbox-u32 dest src))
                        (#.hard-reg-class-gpr-mode-address
                         (unless (logbitp #.hard-reg-class-gpr-mode-address src-type)
                           (! trap-unless-typecode= src arch::subtag-macptr))
                         (! deref-macptr dest src)))))
                   ((#.hard-reg-class-gpr-mode-u32
                     #.hard-reg-class-gpr-mode-s32
                     #.hard-reg-class-gpr-mode-address)
                    (unless (eql  dest-gpr src-gpr)
                      (! copy-gpr dest src)))
                   ((#.hard-reg-class-gpr-mode-u16
                     #.hard-reg-class-gpr-mode-s16)
                    (! u16->u32 dest src))
                   ((#.hard-reg-class-gpr-mode-u8
                     #.hard-reg-class-gpr-mode-s8)
                    (! u8->u32 dest src))))
                (#.hard-reg-class-gpr-mode-s32
                 (case src-mode
                   (#.hard-reg-class-gpr-mode-node
                    (! unbox-s32 dest src))
                   ((#.hard-reg-class-gpr-mode-u32
                     #.hard-reg-class-gpr-mode-s32
                     #.hard-reg-class-gpr-mode-address)
                    (unless (eql  dest-gpr src-gpr)
                      (! copy-gpr dest src)))
                   (#.hard-reg-class-gpr-mode-u16
                    (! u16->u32 dest src))                 
                   (#.hard-reg-class-gpr-mode-s16
                    (! s16->s32 dest src))
                   (#.hard-reg-class-gpr-mode-u8
                    (! u8->u32 dest src))
                   (#.hard-reg-class-gpr-mode-s8
                    (! s8->s32 dest src))))
                (#.hard-reg-class-gpr-mode-u16
                 (case src-mode
                   (#.hard-reg-class-gpr-mode-node
                    (! unbox-u16 dest src))
                   ((#.hard-reg-class-gpr-mode-u8
                     #.hard-reg-class-gpr-mode-s8)
                    (! u8->u32 dest src))
                   (t
                    (unless (eql dest-gpr src-gpr)
                      (! copy-gpr dest src)))))
                (#.hard-reg-class-gpr-mode-s16
                 (case src-mode
                   (#.hard-reg-class-gpr-mode-node
                    (! unbox-s16 dest src))
                   (#.hard-reg-class-gpr-mode-s8
                    (! s8->s32 dest src))
                   (#.hard-reg-class-gpr-mode-u8
                    (! u8->u32 dest src))
                   (t
                    (unless (eql dest-gpr src-gpr)
                      (! copy-gpr dest src)))))
                (#.hard-reg-class-gpr-mode-u8
                 (case src-mode
                   (#.hard-reg-class-gpr-mode-node
                    (! unbox-u8 dest src))
                   (t
                    (unless (eql dest-gpr src-gpr)
                      (! copy-gpr dest src)))))
                (#.hard-reg-class-gpr-mode-s8
                 (case src-mode
                   (#.hard-reg-class-gpr-mode-node
                    (! unbox-s8 dest src))
                   (t
                    (unless (eql dest-gpr src-gpr)
                      (! copy-gpr dest src)))))))
            (if src-gpr
              (if dest-fpr
                (progn
                  (case src-mode
                    (#.hard-reg-class-gpr-mode-node
                     (case dest-mode
                       (#.hard-reg-class-fpr-mode-double
                        ; if we knew the source was double, we set a  bit in the dest reg spec (weird huh)
                        (unless (logbitp hard-reg-class-fpr-type-double 
                                         (get-node-regspec-type-modes dest))
                          (! trap-unless-typecode= src arch::subtag-double-float))
                        (! get-double dest src))
                       (#.hard-reg-class-fpr-mode-single
                        (! trap-unless-typecode= src arch::subtag-single-float)
                        (! get-single dest src)))))))
              (if dest-gpr
                (case dest-mode
                  (#.hard-reg-class-gpr-mode-node
                   (case src-mode
                     (#.hard-reg-class-fpr-mode-double
                      (! double->heap dest src))
                     (#.hard-reg-class-fpr-mode-single
                      (! single->heap dest src)))))
                (if (and src-fpr dest-fpr)
                  (unless (eql dest-fpr src-fpr)
                    (! copy-fpr dest src)))))))))))
  
(defun sparc2-unreachable-store (&optional vreg)
  ; I don't think that anything needs to be done here,
  ; but leave this guy around until we're sure.
  ; (SPARC2-VPUSH-REGISTER will always vpush something, even
  ; if code to -load- that "something" never gets generated.
  ; If I'm right about this, that means that the compile-time
  ; stack-discipline problem that this is supposed to deal
  ; with can't happen.)
  (declare (ignore vreg))
  nil)

; bind vars to initforms, as per let*, &aux.
(defun sparc2-seq-bind (seg vars initforms)
  (dolist (var vars)
    (sparc2-seq-bind-var seg var (pop initforms))))

(defun sparc2-dynamic-extent-form (seg curstack val)
  (when (acode-p val)
    (with-sparc-local-vinsn-macros (seg)
      (let* ((op (acode-operator val)))
        (cond ((eq op (%nx1-operator list))
               (let* ((*sparc2-vstack* *sparc2-vstack*)
                      (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*))
                 (sparc2-set-nargs seg (sparc2-formlist seg (%cadr val) nil))
                 (sparc2-open-undo $undostkblk curstack)
                 (! call-subprim .SPstkconslist))
               (setq val sparc::%arg_z))
              ((eq op (%nx1-operator list*))
               (let* ((arglist (%cadr val)))                   
                 (let* ((*sparc2-vstack* *sparc2-vstack*)
                        (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*))
                   (sparc2-arglist seg arglist))
                 (when (car arglist)
                   (sparc2-set-nargs seg (length (%car arglist)))
                   (! call-subprim .SPstkconslist-star)
                   (sparc2-open-undo $undostkblk curstack))
                 (setq val sparc::%arg_z)))
              ((eq op (%nx1-operator multiple-value-list))
               (sparc2-multiple-value-body seg (%cadr val))
               (sparc2-open-undo $undostkblk curstack)
               (! call-subprim .SPstkconslist)
               (setq val sparc::%arg_z))
              ((eq op (%nx1-operator cons))
               (sparc2-two-targeted-reg-forms seg (%cadr val) sparc::%arg_y (%caddr val) sparc::%arg_z)
               (sparc2-open-undo $undostkblk )
               (! make-tsp-cons sparc::%arg_z sparc::%arg_y sparc::%arg_z) 
               (setq val sparc::%arg_z))
              ((eq op (%nx1-operator %consmacptr%))
               (with-imm-target () (address :address)
                 (sparc2-one-targeted-reg-form seg val address)
                 (with-node-temps () (node)
                   (! macptr->stack node address)
                   (sparc2-open-undo $undostkblk)
                   (setq val node))))
              ((eq op (%nx1-operator %new-ptr))
               (let ((clear-form (caddr val)))
                 (if (nx-constant-form-p clear-form)
                   (progn 
                     (sparc2-one-targeted-reg-form seg (%cadr val) sparc::%arg_z)
                     (sparc2-open-undo $undostkblk)
                     (if (nx-null clear-form)
                       (! call-subprim .SPmakestackblock)
                       (! call-subprim .SPmakestackblock0)))
                   (with-crf-target () crf
                     (let ((stack-block-0-label (backend-get-next-label))
                           (done-label (backend-get-next-label)))
                       (sparc2-two-targeted-reg-forms seg (%cadr val) sparc::%arg_z clear-form sparc::%arg_y)
                       (! compare sparc::%arg_y sparc::%rnil)
                       (! bne crf (aref *backend-labels* stack-block-0-label))
                       (! call-subprim .SPmakestackblock)
                       (-> done-label)
                       (@ stack-block-0-label)
                       (! call-subprim .SPmakestackblock0)
                       (@ done-label)))))
               (setq val sparc::%arg_z))
              ((eq op (%nx1-operator make-list))
               (sparc2-two-targeted-reg-forms seg (%cadr val) sparc::%arg_y (%caddr val) sparc::%arg_z)
               (sparc2-open-undo $undostkblk curstack)
               (! call-subprim .SPmakestacklist)
               (setq val sparc::%arg_z))       
              ((eq (%car val) (%nx1-operator vector))
               (let* ((*sparc2-vstack* *sparc2-vstack*)
                      (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*))
                 (sparc2-set-nargs seg (sparc2-formlist seg (%cadr val) nil))
                 (! call-subprim .SPmkstackv))
               (sparc2-open-undo $undostkblk)
               (setq val sparc::%arg_z))
              ((eq op (%nx1-operator %ppc-gvector))
               (let* ((*sparc2-vstack* *sparc2-vstack*)
                      (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*)
                      (arglist (%cadr val)))
                 (sparc2-set-nargs seg (sparc2-formlist seg (append (car arglist) (reverse (cadr arglist))) nil))
                 (! call-subprim .SPstkgvector))
               (sparc2-open-undo $undostkblk)
               (setq val sparc::%arg_z)) 
              ((eq op (%nx1-operator closed-function)) 
               (setq val (sparc2-make-closure seg (cadr val) t))) ; can't error
              ((eq op (%nx1-operator %make-uvector))
               (destructuring-bind (element-count subtag &optional (init 0 init-p)) (%cdr val)
                 (if init-p
                   (progn
                     (sparc2-three-targeted-reg-forms seg element-count sparc::%arg_x subtag sparc::%arg_y init sparc::%arg_z)
                     (! call-subprim .SPstack-misc-alloc-init))
                   (progn
                     (sparc2-two-targeted-reg-forms seg element-count sparc::%arg_y subtag sparc::%arg_z)
                     (! call-subprim .SPstack-misc-alloc)))
                 (sparc2-open-undo $undostkblk)
                 (setq val sparc::%arg_z)))))))
  val)

(defun sparc2-addrspec-to-reg (seg addrspec reg)
  (if (memory-spec-p addrspec)
    (sparc2-stack-to-register seg addrspec reg)
    (sparc2-copy-register seg reg addrspec)))
  
(defun sparc2-seq-bind-var (seg var val)
  (with-sparc-local-vinsn-macros (seg)
    (let* ((sym (var-name var))
           (bits (nx-var-bits var))
           (closed-p (and (%ilogbitp $vbitclosed bits)
                          (%ilogbitp $vbitsetq bits)))
           (curstack (sparc2-encode-stack))
           (make-vcell (and closed-p (eq bits (var-bits var))))
           (closed-downward (and closed-p (%ilogbitp $vbitcloseddownward bits))))
      (unless (fixnump val)
        (setq val (nx-untyped-form val))
        (when (and (%ilogbitp $vbitdynamicextent bits) (acode-p val))
          (setq val (sparc2-dynamic-extent-form seg curstack val))))
      (if (%ilogbitp $vbitspecial bits)
        (sparc2-dbind seg val sym)
        (let ((puntval nil))
          (flet ((sparc2-puntable-binding-p (var initform)
                   ; The value returned is acode.
                   (let* ((bits (nx-var-bits var)))
                     (if (%ilogbitp $vbitpuntable bits)
                       (nx-untyped-form initform)))))
            (declare (inline sparc2-puntable-binding-p))
            (if (and (not (sparc2-load-ea-p val))
                     (setq puntval (sparc2-puntable-binding-p var val)))
              (progn
                (nx-set-var-bits var (%ilogior (%ilsl $vbitpunted 1) bits))
                (sparc2-set-var-ea seg var puntval))
              (progn
                (let* ((vloc *sparc2-vstack*)
                       (reg (sparc2-assign-register-var var)))
                  (if (sparc2-load-ea-p val)
                    (if reg 
                      (sparc2-addrspec-to-reg seg val reg)
                      (if (memory-spec-p val)
                        (with-node-temps () (temp)
                          (sparc2-addrspec-to-reg seg val temp)
                          (sparc2-vpush-register seg temp :node var bits))
                        (sparc2-vpush-register seg val :node var bits)))
                    (if reg
                      (sparc2-one-targeted-reg-form seg val reg)
                      (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg val sparc::%arg_z) :node var bits)))
                  (sparc2-set-var-ea seg var (or reg (sparc2-vloc-ea vloc closed-p)))
                  (if reg
                    (sparc2-note-var-cell var reg)
                    (sparc2-note-top-cell var))
                  (when make-vcell
                    (with-node-temps () (temp)
                        (sparc2-stack-to-register seg vloc temp)
                        (if closed-downward
                          (progn
                            (! make-tsp-vcell temp temp)
                            (sparc2-open-undo $undostkblk))
                          (! make-vcell temp temp))
                        (sparc2-register-to-stack seg temp vloc))))))))))))



; Never make a vcell if this is an inherited var.
; If the var's inherited, its bits won't be a fixnum (and will
; therefore be different from what NX-VAR-BITS returns.)
(defun sparc2-bind-var (seg var vloc &optional lcell &aux 
                          (bits (nx-var-bits var)) 
                          (closed-p (and (%ilogbitp $vbitclosed bits) (%ilogbitp $vbitsetq bits)))
                          (closed-downward (if closed-p (%ilogbitp $vbitcloseddownward bits)))
                          (make-vcell (and closed-p (eq bits (var-bits var))))
                          (addr (sparc2-vloc-ea vloc)))
  (with-sparc-local-vinsn-macros (seg)
    (if (%ilogbitp $vbitspecial bits)
      (progn
        (sparc2-dbind seg addr (var-name var))
        t)
      (progn
        (when (%ilogbitp $vbitpunted bits)
          (error "bind-var: var ~s was punted" var))
        (when make-vcell
          (with-node-temps () (temp)
            (sparc2-stack-to-register seg vloc temp)
            (if closed-downward
              (progn
                (! make-tsp-vcell temp temp)
                (sparc2-open-undo $undostkblk))
              (! make-vcell temp temp))
            (sparc2-register-to-stack seg temp vloc)))
        (when lcell
          (setf (lcell-kind lcell) :node
                (lcell-attributes lcell) bits
                (lcell-info lcell) var)
          (sparc2-note-var-cell var lcell))          
        (sparc2-set-var-ea seg var (sparc2-vloc-ea vloc closed-p))        
        closed-downward))))

(defun sparc2-set-var-ea (seg var ea)
  (setf (var-ea var) ea)
  (when (and (typep ea 'fixnum) *sparc2-record-symbols*)
    (let* ((start (sparc2-emit-note seg :begin-variable-scope)))
      (push (list var (var-name var) start (close-vinsn-note start))
          *sparc2-recorded-symbols*)))
  ea)

(defun sparc2-close-var (seg var)
  (let ((bits (nx-var-bits var)))
    (when (and *sparc2-record-symbols* 
         (%izerop (%ilogand (%ilogior (ash -1 $vbitspecial)
                                      (%ilsl $vbitpunted 1)) bits)))
      (let ((endnote (%car (%cdddr (assq var *sparc2-recorded-symbols*)))))
        (unless endnote (error "sparc2-close-var ?"))
        (setf (vinsn-note-class endnote) :end-variable-scope)
        (append-dll-node (vinsn-note-label endnote) seg)))))

(defun sparc2-load-ea-p (ea)
  (or (typep ea 'fixnum)
      (typep ea 'lreg)
      (typep ea 'lcell)))

(defun sparc2-dbind (seg value sym)
  (with-sparc-local-vinsn-macros (seg)
    (let* ((ea-p (sparc2-load-ea-p value))
           (nil-p (unless ea-p (eq (setq value (nx-untyped-form value)) *nx-nil*)))
           (self-p (unless ea-p (and (or
                                      (eq (acode-operator value) (%nx1-operator bound-special-ref))
                                      (eq (acode-operator value) (%nx1-operator special-ref)))
                                     (eq (cadr value) sym)))))
      (if (or nil-p self-p)
        (progn
          (sparc2-store-immediate seg (sparc2-symbol-value-locative sym) sparc::%arg_z)
          (if nil-p
            (! call-subprim .SPbind-nil)
            (if (or *sparc2-reckless* (eq (acode-operator value) (%nx1-operator special-ref)))
              (! call-subprim .SPbind-self)
              (! call-subprim .SPbind-self-boundp-check))))
        (progn
          (if ea-p 
            (sparc2-store-ea seg value sparc::%arg_z)
            (sparc2-one-targeted-reg-form seg value sparc::%arg_z))
          (sparc2-store-immediate seg (sparc2-symbol-value-locative sym) sparc::%arg_y)
          (! call-subprim .SPbind)))
      (sparc2-open-undo $undospecial)
      (sparc2-new-vstack-lcell :special 12 (ash 1 $vbitspecial) sym)
      (sparc2-adjust-vstack 12))))

; Store the contents of EA - which denotes either a vframe location
; or a hard register - in reg.

(defun sparc2-store-ea (seg ea reg)
  (if (typep ea 'fixnum)
    (if (memory-spec-p ea)
      (sparc2-stack-to-register seg ea reg)
      (sparc2-copy-register seg reg ea))
    (if (typep ea 'lreg)
      (sparc2-copy-register seg reg ea)
      (if (typep ea 'lcell)
        (sparc2-lcell-to-register seg ea reg)))))


      

; Callers should really be sure that this is what they want to use.
(defun sparc2-absolute-long (seg vreg xfer value)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (when vreg
      (sparc2-lwi seg vreg value))
    (^)))



(defun sparc2-store-macptr (seg vreg address-reg)
  (with-sparc-local-vinsn-macros (seg vreg)
    (when (sparc2-for-value-p vreg)
      (if (logbitp vreg sparc-imm-regs)
        (<- address-reg)
        (! macptr->heap vreg address-reg)))))

(defun sparc2-store-signed-longword (seg vreg imm-reg)
  (with-sparc-local-vinsn-macros (seg vreg)
    (when (sparc2-for-value-p vreg)
      (if (logbitp vreg sparc-imm-regs)
        (<- imm-reg)
        (! s32->integer vreg imm-reg)))))

(defun sparc2-store-signed-halfword (seg vreg imm-reg)
  (with-sparc-local-vinsn-macros (seg vreg)
    (when (sparc2-for-value-p vreg)
      (if (logbitp vreg sparc-imm-regs)
        (<- imm-reg)
        (! s16->fixnum vreg imm-reg)))))


(defun sparc2-store-unsigned-halfword (seg vreg imm-reg)
  (with-sparc-local-vinsn-macros (seg vreg)
    (when (sparc2-for-value-p vreg)
      (if (logbitp vreg sparc-imm-regs)
        (<- imm-reg)
        (! u16->fixnum vreg imm-reg)))))




; If "value-first-p" is true and both "offset" and "val" need to be 
; evaluated, evaluate "val" before evaluating "offset".
(defun sparc2-%immediate-set-ptr (seg vreg xfer deref ptr offset val value-first-p)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((intval (acode-absolute-ptr-p val))
           (offval (acode-fixnum-form-p offset))
           (absptr (and offval (acode-absolute-ptr-p ptr)))
           (for-value (sparc2-for-value-p vreg)))
      (flet ((address-and-node-regs ()
               (if for-value
                 (progn
                   (sparc2-one-targeted-reg-form seg val sparc::%arg_z)
                   (if (eq intval 0)
                     (values sparc::%rzero sparc::%arg_z)
                     (progn
                       (if intval
                         (sparc2-lwi seg sparc::%imm0 intval)                         
                         (! deref-macptr sparc::%imm0 sparc::%arg_z))
                       (values sparc::%imm0 sparc::%arg_z))))
                 (if (eq intval 0)
                   (values sparc::%rzero nil)
                   (values (sparc2-macptr-arg-to-reg seg val sparc::%imm0) nil)))))
        (if (and absptr offval)
          (setq absptr (+ absptr offval) offval 0)
          (setq absptr nil))
        (and offval (%i> (integer-length offval) 15) (setq offval nil))
        (and absptr (%i> (integer-length absptr) 15) (setq absptr nil))
        (if absptr
          (multiple-value-bind (address node) (address-and-node-regs)
            (! mem-set-c-fullword address sparc::%rzero absptr)
            (if for-value
              (<- node)))
          ; No absolute ptr (which is presumably a rare case anyway.)
          (if offval
            ; Easier: need one less register than in the general case.
            (with-imm-target () (ptr-reg :address)
              (sparc2-one-targeted-reg-form seg ptr ptr-reg)
              (if intval
                (with-imm-target (ptr-reg) (val-target :address)                    
                  (if (eql intval 0)
                    (setq val-target sparc::%rzero)
                    (sparc2-lwi seg val-target intval))
                  (if deref
                    (! mem-ref-c-fullword ptr-reg ptr-reg 0))
                  (! mem-set-c-fullword val-target ptr-reg offval)
                  (if for-value
                    (<- (set-regspec-mode val-target (gpr-mode-name-value :address)))))
                (progn
                  (! temp-push-unboxed-word ptr-reg)
                  (sparc2-open-undo $undostkblk)
                  (multiple-value-bind (address node) (address-and-node-regs)
                    (with-imm-target (address) (ptr-reg :address)
                      (! temp-pop-unboxed-word ptr-reg)
                      (sparc2-close-undo)
                      (if deref
                        (! mem-ref-c-fullword ptr-reg ptr-reg 0))
                      (! mem-set-c-fullword address ptr-reg offval)
                      (if for-value
                        (<- node)))))))
            ;; No (16-bit) constant offset.  Might still have a 32-bit constant offset;
            ;; might have a constant value.  Might not.  Might not.
            ;; Easiest to special-case the constant-value case first ...
            (let* ((xptr-reg nil)
                   (xoff-reg nil)
                   (xval-reg nil)
                   (node-arg_z nil)
                   (constant-offset (acode-fixnum-form-p offset)))
              (if intval
                (if constant-offset
                  (with-imm-target () (ptr-reg :address)
                    (sparc2-one-targeted-reg-form seg ptr ptr-reg)
                    (with-imm-target (ptr-reg) (off-reg :s32)
                      (sparc2-lwi seg off-reg constant-offset)
                      (with-imm-target (ptr-reg off-reg) (val-reg :address)
                        (if (eql intval 0)
                          (setq val-reg sparc::%rzero)
                          (sparc2-lwi seg val-reg intval))
                        (setq xptr-reg ptr-reg
                              xoff-reg off-reg
                              xval-reg val-reg))))
                  ; Offset's non-constant.  Temp-push the pointer, evaluate
                  ; and unbox the offset, load the value, pop the pointer.
                  (progn
                    (with-imm-target () (ptr-reg :address)
                      (sparc2-one-targeted-reg-form seg ptr ptr-reg)
                      (! temp-push-unboxed-word ptr-reg)
                      (sparc2-open-undo $undostkblk))
                    (with-imm-target () (off-reg :s32)
                      (! fixnum->s32 off-reg (sparc2-one-targeted-reg-form seg offset sparc::%arg_z))
                      (with-imm-target (off-reg) (val-reg :s32)
                        (if (eql intval 0)
                          (setq val-reg sparc::%rzero)
                          (sparc2-lwi seg val-reg intval))
                        (with-imm-target (off-reg val-reg) (ptr-reg :address)
                          (! temp-pop-unboxed-word ptr-reg)
                          (sparc2-close-undo)
                          (setq xptr-reg ptr-reg
                                xoff-reg off-reg
                                xval-reg val-reg))))))
                ;; No intval; maybe constant-offset.
                (with-imm-target () (ptr-reg :address)
                  (sparc2-one-targeted-reg-form seg ptr ptr-reg)
                  (! temp-push-unboxed-word ptr-reg)
                  (sparc2-open-undo $undostkblk)
                  (if (or constant-offset (not value-first-p))
                    (progn
                      (if (not constant-offset)
                        (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg offset sparc::%arg_z)))
                      (multiple-value-bind (address node) (address-and-node-regs)
                        (with-imm-target (address) (off-reg :s32)
                          (if constant-offset
                            (sparc2-lwi seg off-reg constant-offset)
                            (with-node-temps (sparc::%arg_z) (temp)
                              (sparc2-vpop-register seg temp)
                              (! fixnum->s32 off-reg temp)))
                          (with-imm-target (sparc::%imm0 off-reg) (ptr-reg :address)
                            (! temp-pop-unboxed-word ptr-reg)
                            (sparc2-close-undo)
                            (setq xptr-reg ptr-reg
                                  xoff-reg off-reg
                                  xval-reg address
                                  node-arg_z node)))))
                    (progn
                      ; The "for-value" case can't happen here.
                      (with-imm-target (ptr-reg) (address :address)
                        (sparc2-two-targeted-reg-forms seg val address offset sparc::%arg_z)
                        (with-imm-target (address ptr-reg) (off-reg :s32)
                          (! fixnum->s32 off-reg sparc::%arg_z)
                          (! temp-pop-unboxed-word ptr-reg)
                          (sparc2-close-undo)
                          (setq xptr-reg ptr-reg
                                  xoff-reg off-reg
                                  xval-reg address
                                  node-arg_z nil)))))))
              (if deref
                (! mem-ref-c-fullword xptr-reg xptr-reg 0))
              (! mem-set-fullword xval-reg xptr-reg xoff-reg)
              (when for-value
                (if node-arg_z
                  (<- node-arg_z)
                  (<- (set-regspec-mode 
                       xval-reg
                       (gpr-mode-name-value :address))))))))
        (^)))))
  
(defun sparc2-memory-store-displaced (seg valreg basereg displacement size deref)
  (with-sparc-local-vinsn-macros (seg)
    (if deref
      (! mem-ref-c-fullword basereg basereg 0))
    (case size
      (4 (! mem-set-c-fullword valreg basereg displacement))
      (2 (! mem-set-c-halfword valreg basereg displacement))
      (1 (! mem-set-c-byte valreg basereg displacement)))))

(defun sparc2-memory-store-indexed (seg valreg basereg idxreg size deref)
  (with-sparc-local-vinsn-macros (seg)
    (if deref
      (! mem-ref-c-fullword basereg basereg 0))
    (case size
      (4 (! mem-set-fullword valreg basereg idxreg))
      (2 (! mem-set-halfword valreg basereg idxreg))
      (1 (! mem-set-byte valreg basereg idxreg)))))
      
(defun sparc2-%immediate-store  (seg vreg xfer bits ptr offset val value-first-p)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (if (eql 0 (%ilogand #xf bits))
      (sparc2-%immediate-set-ptr seg vreg xfer (%ilogbitp 4 bits) ptr offset val value-first-p)
      (let* ((deref (%ilogbitp 4 bits))
             (size
              (if (eq (setq bits (%ilogand2 #xf bits)) 3) 
                1
                (if (eq bits 2) 
                  2 
                  4)))
             (long-p (eq bits 1))
             (intval (if long-p (sparc2-long-constant-p val) (acode-fixnum-form-p val)))
             (offval (acode-fixnum-form-p offset))
             (absptr (unless deref (and offval (acode-absolute-ptr-p ptr))))
             (for-value (sparc2-for-value-p vreg)))
        (declare (fixnum size))
        (flet ((val-to-argz-and-imm0 ()
                 (sparc2-one-targeted-reg-form seg val sparc::%arg_z)
                 (if (eq size 4)
                   (! call-subprim .SPgetXlong)
                   (! fixnum->s32 sparc::%imm0 sparc::%arg_z))))
          (if (and absptr offval)
            (setq absptr (+ absptr offval) offval 0)
            (setq absptr nil))
          (and offval (%i> (integer-length offval) 15) (setq offval nil))
          (and absptr (%i> (integer-length absptr) 15) (setq absptr nil))
          (if absptr
            (if intval
              (with-imm-target () (val-target :s32)
                (if (eql intval 0)
                  (setq val-target sparc::%rzero)
                  (sparc2-lwi seg val-target intval))
                (sparc2-memory-store-displaced seg val-target sparc::%rzero absptr size nil)
                (if for-value
                  (<- (set-regspec-mode 
                       val-target 
                       (gpr-mode-name-value
                        (if (eq size 4)
                          :s32
                          (if (eq size 2)
                            :s16
                            :s8)))))))
              (progn
                (val-to-argz-and-imm0)
                (sparc2-memory-store-displaced seg sparc::%imm0 sparc::%rzero absptr size nil)
                (<- sparc::%arg_z)))
            ; No absolute ptr (which is presumably a rare case anyway.)
            (if offval
              ; Easier: need one less register than in the general case.
              (with-imm-target () (ptr-reg :address)
                (sparc2-one-targeted-reg-form seg ptr ptr-reg)
                (if intval
                  (with-imm-target (ptr-reg) (val-target :s32)                    
                    (if (eql intval 0)
                      (setq val-target sparc::%rzero)
                      (sparc2-lwi seg val-target intval))
                    (sparc2-memory-store-displaced seg val-target ptr-reg offval size deref)
                    (if for-value
                      (<- (set-regspec-mode 
                           val-target 
                           (gpr-mode-name-value
                            (if (eq size 4)
                              :s32
                              (if (eq size 2)
                                :s16
                                :s8)))))))
                  (progn
                    (! temp-push-unboxed-word ptr-reg)
                    (sparc2-open-undo $undostkblk)
                    (val-to-argz-and-imm0)                  
                    (with-imm-target (sparc::%imm0) (ptr-reg :address)
                      (! temp-pop-unboxed-word ptr-reg)
                      (sparc2-close-undo)
                      (sparc2-memory-store-displaced seg sparc::%imm0 ptr-reg offval size deref)                    
                      (if for-value
                        (<- sparc::%arg_z))))))
              ;; No (16-bit) constant offset.  Might still have a 32-bit constant offset;
              ;; might have a constant value.  Might not.  Might not.
              ;; Easiest to special-case the constant-value case first ...
              (let* ((xptr-reg nil)
                     (xoff-reg nil)
                     (xval-reg nil)
                     (node-arg_z nil)
                     (constant-offset (acode-fixnum-form-p offset)))
                (if intval
                  (if constant-offset
                    (with-imm-target () (ptr-reg :address)
                      (sparc2-one-targeted-reg-form seg ptr ptr-reg)
                      (with-imm-target (ptr-reg) (off-reg :s32)
                        (sparc2-lwi seg off-reg constant-offset)
                        (with-imm-target (ptr-reg off-reg) (val-reg :s32)
                          (if (eql intval 0)
                            (setq val-reg sparc::%rzero)
                            (sparc2-lwi seg val-reg intval))
                          (setq xptr-reg ptr-reg
                                xoff-reg off-reg
                                xval-reg val-reg))))
                    ; Offset's non-constant.  Temp-push the pointer, evaluate
                    ; and unbox the offset, load the value, pop the pointer.
                    (progn
                      (with-imm-target () (ptr-reg :address)
                        (sparc2-one-targeted-reg-form seg ptr ptr-reg)
                        (! temp-push-unboxed-word ptr-reg)
                        (sparc2-open-undo $undostkblk))
                      (with-imm-target () (off-reg :s32)
                        (! fixnum->s32 off-reg (sparc2-one-targeted-reg-form seg offset sparc::%arg_z))
                        (with-imm-target (off-reg) (val-reg :s32)
                          (if (eql intval 0)
                            (setq val-reg sparc::%rzero)
                            (sparc2-lwi seg val-reg intval))
                          (with-imm-target (off-reg val-reg) (ptr-reg :address)
                            (! temp-pop-unboxed-word ptr-reg)
                            (sparc2-close-undo)
                            (setq xptr-reg ptr-reg
                                  xoff-reg off-reg
                                  xval-reg val-reg))))))
                  ;; No intval; maybe constant-offset.
                  (with-imm-target () (ptr-reg :address)
                    (sparc2-one-targeted-reg-form seg ptr ptr-reg)
                    (! temp-push-unboxed-word ptr-reg)
                    (sparc2-open-undo $undostkblk)
                    (if (or constant-offset (not value-first-p))
                      (progn
                        (if (not constant-offset)
                          (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg offset sparc::%arg_z)))
                        (val-to-argz-and-imm0)
                        (with-imm-target (sparc::%imm0) (off-reg :s32)
                          (if constant-offset
                            (sparc2-lwi seg off-reg constant-offset)
                            (with-node-temps (sparc::%arg_z) (temp)
                              (sparc2-vpop-register seg temp)
                              (! fixnum->s32 off-reg temp)))
                          (with-imm-target (sparc::%imm0 off-reg) (ptr-reg :address)
                            (! temp-pop-unboxed-word ptr-reg)
                            (sparc2-close-undo)
                            (setq xptr-reg ptr-reg
                                  xoff-reg off-reg
                                  xval-reg sparc::%imm0
                                  node-arg_z t))))
                      (progn
                        (sparc2-two-targeted-reg-forms seg val sparc::%arg_z offset sparc::%arg_y)
                        (if (eq size 4)
                          (! call-subprim .SPgetXlong)
                          (! fixnum->s32 sparc::%imm0 sparc::%arg_z))
                        (with-imm-target (sparc::%imm0) (off-reg :s32)
                          (! fixnum->s32 off-reg sparc::%arg_y)
                          (with-imm-target (sparc::%imm0 off-reg) (ptr-reg :address)
                            (! temp-pop-unboxed-word ptr-reg)
                            (sparc2-close-undo)
                            (setq xptr-reg ptr-reg
                                    xoff-reg off-reg
                                    xval-reg sparc::%imm0
                                    node-arg_z nil)))))))
                (sparc2-memory-store-indexed seg xval-reg xptr-reg xoff-reg size deref)              
                (when for-value
                  (if node-arg_z
                    (<- sparc::%arg_z)
                    (<- (set-regspec-mode 
                         xval-reg
                         (gpr-mode-name-value
                          (if (eq size 4)
                            :s32
                            (if (eq size 2)
                              :s16
                              :s8))))))))))
          (^))))))





(defun sparc2-encoding-undo-count (encoding)
 (svref encoding 0))

(defun sparc2-encoding-cstack-depth (encoding)    ; hardly ever interesting
  (svref encoding 1))

(defun sparc2-encoding-vstack-depth (encoding)
  (svref encoding 2))

(defun sparc2-encoding-vstack-top (encoding)
  (svref encoding 3))

(defun sparc2-encode-stack ()
  (vector *sparc2-undo-count* *sparc2-cstack* *sparc2-vstack* *sparc2-top-vstack-lcell*))

(defun sparc2-decode-stack (encoding)
  (values (sparc2-encoding-undo-count encoding)
          (sparc2-encoding-cstack-depth encoding)
          (sparc2-encoding-vstack-depth encoding)
          (sparc2-encoding-vstack-top encoding)))

(defun sparc2-equal-encodings-p (a b)
  (dotimes (i 3 t)
    (unless (eq (svref a i) (svref b i)) (return))))

(defun sparc2-open-undo (&optional (reason $undocatch) (curstack (sparc2-encode-stack)))
  (set-fill-pointer 
   *sparc2-undo-stack*
   (set-fill-pointer *sparc2-undo-because* *sparc2-undo-count*))
  (vector-push-extend curstack *sparc2-undo-stack*)
  (vector-push-extend reason *sparc2-undo-because*)
  (setq *sparc2-undo-count* (%i+ *sparc2-undo-count* 1)))

(defun sparc2-close-undo (&aux
                        (new-count (%i- *sparc2-undo-count* 1))
                        (i (aref *sparc2-undo-stack* new-count)))
  (multiple-value-setq (*sparc2-undo-count* *sparc2-cstack* *sparc2-vstack* *sparc2-top-vstack-lcell*)
    (sparc2-decode-stack i))
  (set-fill-pointer 
   *sparc2-undo-stack*
   (set-fill-pointer *sparc2-undo-because* new-count)))





; "Trivial" means can be evaluated without allocating or modifying registers.
; Interim definition, which will probably stay here forever.
(defun sparc2-trivial-p (form &aux op bits)
  (setq form (nx-untyped-form form))
  (and
   (consp form)
   (not (eq (setq op (%car form)) (%nx1-operator call)))
   (or
    (nx-null form)
    (nx-t form)
    (eq op (%nx1-operator simple-function))
    (eq op (%nx1-operator fixnum))
    (eq op (%nx1-operator immediate))
    (eq op (%nx1-operator bound-special-ref))
    (and (or (eq op (%nx1-operator inherited-arg)) 
             (eq op (%nx1-operator lexical-reference)))
         (or (%ilogbitp $vbitpunted (setq bits (nx-var-bits (cadr form))))
             (neq (%ilogior (%ilsl $vbitclosed 1) (%ilsl $vbitsetq 1))
                  (%ilogand (%ilogior (%ilsl $vbitclosed 1) (%ilsl $vbitsetq 1)) bits)))))))





(defun sparc2-lexical-reference-p (form)
  (when (acode-p form)
    (let ((op (acode-operator (setq form (acode-unwrapped-form form)))))
      (when (or (eq op (%nx1-operator lexical-reference))
                (eq op (%nx1-operator inherited-arg)))
        (%cadr form)))))

(defun sparc2-ref-symbol-value (seg vreg xfer sym check-boundp)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (when vreg
      (setq sym (sparc2-symbol-value-locative sym))
      (let* ((symreg (or (sparc2-register-constant-p sym)
                         (sparc2-store-immediate seg sym (if (and check-boundp 
                                                                (eq sparc::%arg_z (hard-regspec-value vreg)))
                                                         sparc::%arg_y 
                                                         sparc::%arg_z)))))
        (ensuring-node-target (target vreg)
          (if check-boundp
            (if (eq (hard-regspec-value vreg) (hard-regspec-value symreg))
              (with-node-temps (target symreg) (other)
                (! copy-node-gpr other symreg)
                (! symbol-value target other))
              (! symbol-value target symreg))
            (! node-slot-ref target symreg sparc::symbol.vcell-cell)))))
    (^)))

;; Should be less eager to box result
(defun sparc2-extract-charcode (seg vreg xfer char safe)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((src (sparc2-one-untargeted-reg-form seg char sparc::%arg_z)))
      (when safe
        (! trap-unless-lowbyte= src arch::subtag-character))
      (if vreg
        (ensuring-node-target (target vreg)
          (! character->fixnum target src)))
      (^))))
  
(defun sparc2-reference-list (seg vreg xfer listform refcdr)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((src (sparc2-one-untargeted-reg-form seg listform sparc::%arg_z)))
      (if vreg
        (ensuring-node-target (target vreg)
          (if refcdr
            (! %cdr target src)
            (! %car target src))))
      (^))))

; If safe, ensure that index is a fixnum (if non-constant)
; and check vector bound.
; If we're going to have to evaluate the index into a register (to do
; the bounds check), but know that the index could be a constant 13-bit
; displacement, this'll look pretty silly ..
(defun sparc2-misc-node-ref (seg vreg xfer miscobj index safe)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((index-known-fixnum (acode-fixnum-form-p index))
           (unscaled-idx nil)
           (src nil))
      (if (or safe (not index-known-fixnum))
        (multiple-value-setq (src unscaled-idx)
          (sparc2-two-untargeted-reg-forms seg miscobj sparc::%arg_y index sparc::%arg_z))
        (setq src (sparc2-one-untargeted-reg-form seg miscobj sparc::%arg_z)))
      (when safe
        (if (typep safe 'fixnum)
          (! trap-unless-typecode= src safe))
        (unless index-known-fixnum
          (! trap-unless-tag= unscaled-idx arch::tag-fixnum))
        (! check-misc-bound unscaled-idx src))
      (when vreg
        (ensuring-node-target (target vreg)
          (if (and index-known-fixnum (<= index-known-fixnum sparc::max-32-bit-constant-index))
            (progn
              (! misc-ref-c-node target src index-known-fixnum))
            (let* ((idx-reg sparc::%imm0))
              (if index-known-fixnum
                (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-data-offset (ash index-known-fixnum 2)))
                (! scale-32bit-misc-index idx-reg unscaled-idx))
              (! misc-ref-node target src idx-reg)))))
      (^))))

(defun sparc2-misc-node-set (seg vreg xfer miscobj index value safe)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((index-known-fixnum (acode-fixnum-form-p index))
           (src )
           (unscaled-idx )
           (val-reg )
           (memoize (sparc2-acode-needs-memoization value)))
      (if (or safe (not index-known-fixnum))
        (multiple-value-setq (src unscaled-idx val-reg)
          (sparc2-three-untargeted-reg-forms seg miscobj sparc::%arg_x index sparc::%arg_y value sparc::%arg_z))
        (multiple-value-setq (src val-reg)
          (sparc2-two-untargeted-reg-forms seg miscobj sparc::%arg_y value sparc::%arg_z)))
      (when safe
        (if (typep safe 'fixnum)
          (! trap-unless-typecode= src safe))
        (unless index-known-fixnum
          (! trap-unless-tag= unscaled-idx arch::tag-fixnum))
        (! check-misc-bound unscaled-idx src))
      (if (and index-known-fixnum (<= index-known-fixnum sparc::max-32-bit-constant-index))
        (progn
          (if memoize
            (! misc-set-c-node& val-reg src index-known-fixnum)
            (! misc-set-c-node val-reg src index-known-fixnum)))
        (let* ((idx-reg sparc::%imm0))
          (if index-known-fixnum
            (sparc2-absolute-long seg idx-reg nil (+ sparc::misc-data-offset (ash index-known-fixnum 2)))
            (! scale-32bit-misc-index idx-reg unscaled-idx))
          (if memoize
            (! misc-set-node& val-reg src idx-reg)
            (! misc-set-node val-reg src idx-reg))))
      (<- val-reg)
      (^))))



(defun sparc2-misc-byte-count (subtag element-count)
  (declare (fixnum subtag))
  (if (or (= sparc::fulltag-nodeheader (logand subtag sparc::fulltagmask))
          (<= subtag arch::max-32-bit-ivector-subtag))
    (ash element-count 2)
    (if (<= subtag arch::max-8-bit-ivector-subtag)
      element-count
      (if (<= subtag arch::max-16-bit-ivector-subtag)
        (ash element-count 1)
        (if (= subtag arch::subtag-bit-vector)
          (ash (+ element-count 7) -3)
          (+ 4 (ash element-count 3)))))))

; The naive approach is to vpush all of the initforms, allocate the miscobj,
; then sit in a loop vpopping the values into the vector.
; That's "naive" when most of the initforms in question are "side-effect-free"
; (constant references or references to un-SETQed lexicals), in which case
; it makes more sense to just store the things into the vector cells, vpushing/
; vpopping only those things that aren't side-effect-free.  (It's necessary
; to evaluate any non-trivial forms before allocating the miscobj, since that
; ensures that the initforms are older (in the EGC sense) than it is.)
; The break-even point space-wise is when there are around 3 non-trivial initforms
; to worry about.


(defun sparc2-allocate-initialized-gvector (seg vreg xfer subtag initforms)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (if (null vreg)
      (dolist (f initforms) (sparc2-form seg nil nil f))
      (let* ((*sparc2-vstack* *sparc2-vstack*)
             (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*)
             (n (length initforms))
             (nntriv (let* ((count 0)) 
                       (declare (fixnum count))
                       (dolist (f initforms count) 
                         (unless (sparc-side-effect-free-form-p f)
                           (incf count)))))                                             
             (header (logior (ash n sparc::num-subtag-bits) subtag)))
        (declare (fixnum n nntriv))
        (cond ((or *sparc2-open-code-inline* (> nntriv 3))
               (sparc2-formlist seg initforms nil)
               (sparc2-lwi seg sparc::%imm0 header)
               (! %ppc-gvector vreg sparc::%imm0 (ash n sparc::word-shift)))
              (t
               (let* ((pending ())
                      (vstack *sparc2-vstack*))
                 (declare (fixnum vstack))
                 (dolist (form initforms)
                   (if (sparc-side-effect-free-form-p form)
                     (push form pending)
                     (progn
                       (push nil pending)
                       (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg form sparc::%arg_z)))))
                 (sparc2-lwi seg sparc::%imm0 header)
                 (ensuring-node-target (target vreg)
                   (! %alloc-misc-fixed target sparc::%imm0 (ash n sparc::word-shift))
                   (with-node-temps (target) (nodetemp)
                     (do* ((forms pending (cdr forms))
                           (index (1- n) (1- index))
                           (pushed-cell (+ vstack (the fixnum (ash nntriv sparc::word-shift)))))
                          ((null forms))
                       (declare (list forms) (fixnum pushed-cell))
                       (let* ((form (car forms))
                              (reg nodetemp))
                         (if form
                           (setq reg (sparc2-one-untargeted-reg-form seg form nodetemp))
                           (progn
                             (decf pushed-cell 4)
                             (sparc2-stack-to-register seg (sparc2-vloc-ea pushed-cell) nodetemp)))
                         (! misc-set-c-node reg target index)))))
                 (! vstack-discard nntriv))))))
     (^)))

;; Heap-allocated constants -might- need memoization: they might be newly-created,
;; as in the case of synthesized toplevel functions in .pfsl files.
(defun sparc2-acode-needs-memoization (valform)
  (if (sparc2-form-typep valform 'fixnum)
    nil
    (let* ((val (acode-unwrapped-form valform)))
      (if (or (eq val *nx-t*)
              (eq val *nx-nil*)
              (and (acode-p val)
                   (let* ((op (acode-operator val)))
                     (or (eq op (%nx1-operator fixnum)) #|(eq op (%nx1-operator immediate))|#))))
        nil
        t))))

(defun sparc2-modify-cons (seg vreg xfer ptrform valform safe setcdr returnptr)
  (if (sparc2-form-typep ptrform 'cons)
    (setq safe nil))                    ; May also have been passed as NIL.
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (multiple-value-bind (ptr-vreg val-vreg) (sparc2-two-untargeted-reg-forms seg ptrform sparc::%arg_y valform sparc::%arg_z)
    (let* ((memoize (sparc2-acode-needs-memoization valform)))
      (when safe
        (! TRAP-UNLESS-FULLTAG= ptr-vreg sparc::fulltag-cons))
      (if setcdr
        (if memoize
          (! %set-cdr& ptr-vreg val-vreg)
          (! %set-cdr ptr-vreg val-vreg))
        (if memoize
          (! %set-car& ptr-vreg val-vreg)
          (! %set-car ptr-vreg val-vreg)))
      (if returnptr
        (<- ptr-vreg)
        (<- val-vreg))
      (^)))))



(defun sparc2-find-nilret-label ()
  (dolist (l *sparc2-nilret-labels*)
    (destructuring-bind (label vsp csp register-restore-count register-restore-ea &rest agenda) l
      (and (or (and (eql 0 register-restore-count)
                    (or (not (eql 0 vsp))
                        (eq vsp *sparc2-vstack*)))
                (and 
                 (eq register-restore-count *sparc2-register-restore-count*)
                 (eq vsp *sparc2-vstack*)))
           (or agenda (eq csp *sparc2-cstack*))
           (eq register-restore-ea *sparc2-register-restore-ea*)
           (eq (%ilsr 1 (length agenda)) *sparc2-undo-count*)
           (dotimes (i (the fixnum *sparc2-undo-count*) t) 
             (unless (and (eq (pop agenda) (aref *sparc2-undo-because* i))
                          (eq (pop agenda) (aref *sparc2-undo-stack* i)))
               (return)))
           (return label)))))

(defun sparc2-record-nilret-label ()
  (let* ((lab (backend-get-next-label))
         (info nil))
    (dotimes (i (the fixnum *sparc2-undo-count*))
      (push (aref *sparc2-undo-because* i) info)
      (push (aref *sparc2-undo-stack* i) info))
    (push (cons
                 lab 
                 (cons
                  *sparc2-vstack*
                  (cons 
                   *sparc2-cstack*
                   (cons
                    *sparc2-register-restore-count*
                    (cons
                     *sparc2-register-restore-ea*
                     (nreverse info))))))
          *sparc2-nilret-labels*)
    lab))

; If we know that the form is something that sets a CR bit,
; allocate a CR field and evaluate the form in such a way
; as to set that bit.
; If it's a compile-time constant, branch accordingly and
; let the dead code die.
; Otherwise, evaluate it to some handy register and compare
; that register to RNIL.
; "XFER" is a compound destination.
(defun sparc2-conditional-form (seg xfer form)
  (let* ((uwf (acode-unwrapped-form form)))
    (if (nx-null uwf)
      (sparc2-branch seg (sparc2-cd-false xfer) nil)
      (if (sparc-constant-form-p uwf)
        (sparc2-branch seg (sparc2-cd-true xfer) nil)
        (with-crf-target () crf
          (sparc2-form seg crf xfer form))))))

      
(defun sparc2-branch (seg xfer crf &optional cr-bit true-p)
  (declare (ignore crf))
  (let* ((*sparc2-vstack* *sparc2-vstack*)
         (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*))
    (with-sparc-local-vinsn-macros (seg)
      (setq xfer (or xfer 0))
      (when (%ilogbitp $backend-mvpass-bit xfer) ;(sparc2-mvpass-p cd)
        (setq xfer (%ilogand (%ilognot $backend-mvpass-mask) xfer))
        (unless *sparc2-returning-values*
          (sparc2-vpush-register seg sparc::%arg_z)
          (sparc2-set-nargs seg 1)))
      (if (neq 0 xfer)
        (if (eq xfer $backend-return)    ;; xfer : RETURN ==> popj
          (sparc2-do-return seg)
          (if (not (sparc2-cd-compound-p xfer))
            (-> xfer)  ;; xfer : label# ==> BRA label#
            ;; cd is compound : (<true> / <false>)
            (let* ((fcc-p nil)
		   (truebranch (sparc2-cd-true xfer))
                   (falsebranch (sparc2-cd-false xfer))
                   (tbranch (if true-p truebranch falsebranch))
                   (nbranch (if true-p falsebranch truebranch))
                   (tn0 (neq 0 tbranch))
                   (tnret (neq $backend-return tbranch))
                   (nn0 (neq 0 nbranch))
                   (nnret (neq $backend-return nbranch))
                   (tlabel (if (and tnret tn0) (aref *backend-labels* tbranch)))
                   (nlabel (if (and nnret nn0) (aref *backend-labels* nbranch))))
              (unless cr-bit (setq cr-bit sparc::conde))
	      (if (fcc-p cr-bit)
		(setq fcc-p t cr-bit (logand #xf cr-bit)))
              (if (and tn0 tnret nn0 nnret)
                (progn
		  (if fcc-p
		   (! float-branch-true cr-bit tlabel)
                   (! branch-true cr-bit tlabel))    ;; (label# /  label#)
                  (-> nbranch)))
                (if (and nnret tnret)
                  (if nn0
		    (if fcc-p
		      (! float-branch-true (sparc-invert-fcc cr-bit) nlabel)
                      (! branch-false  cr-bit nlabel))
		    (if fcc-p
		      (! float-branch-true cr-bit tlabel)
                      (! branch-true  cr-bit tlabel)))
                  (let* ((aux-label (backend-get-next-label))
                         (auxl (aref *backend-labels* aux-label)))
                    (if tn0
		      (if fcc-p
			(! float-branch-true cr-bit auxl)
                        (! branch-true  cr-bit auxl))
		      (if fcc-p
			(! float-branch-true (sparc-invert-fcc cr-bit) auxl)
                        (! branch-false cr-bit auxl)))
                    (sparc2-do-return seg)
                    (@ aux-label))))))))))

(defun sparc2-cd-merge (cd label)
  (setq cd (or cd 0))
  (let ((mvpass (%ilogbitp $backend-mvpass-bit cd)))
    (if (neq 0 (%ilogand2 (%ilognot $backend-mvpass-mask) cd))
      (if (sparc2-cd-compound-p cd)
        (sparc2-make-compound-cd
         (sparc2-cd-merge (sparc2-cd-true cd) label)
         (sparc2-cd-merge (sparc2-cd-false cd) label)
         mvpass)
        cd)
      (if mvpass 
        (%ilogior2 $backend-mvpass-mask label)
        label))))

(defun sparc2-mvpass-p (xfer)
  (if xfer (or (%ilogbitp $backend-mvpass-bit xfer) (eq xfer $backend-mvpass))))

(defun sparc2-cd-compound-p (xfer)
  (if xfer (%ilogbitp $backend-compound-branch-target-bit xfer)))

(defun sparc2-cd-true (xfer)
 (if (sparc2-cd-compound-p xfer)
   (ldb  $backend-compound-branch-true-byte xfer)
  xfer))

(defun sparc2-cd-false (xfer)
 (if (sparc2-cd-compound-p xfer)
   (ldb  $backend-compound-branch-false-byte xfer)
   xfer))

(defun sparc2-make-compound-cd (tpart npart &optional mvpass-p)
  (dpb (or npart 0) $backend-compound-branch-false-byte
       (dpb (or tpart 0) $backend-compound-branch-true-byte
            (logior (if mvpass-p $backend-mvpass-mask 0) $backend-compound-branch-target-mask))))

(defun sparc2-invert-cd (cd)
  (if (sparc2-cd-compound-p cd)
    (sparc2-make-compound-cd (sparc2-cd-false cd) (sparc2-cd-true cd) (logbitp $backend-mvpass-bit cd))
    cd))

(defun sparc2-long-constant-p (form)
  (setq form (acode-unwrapped-form form))
  (or (acode-fixnum-form-p form)
      (and (acode-p form)
           (eq (acode-operator form) (%nx1-operator immediate))
           (setq form (%cadr form))
           (if (integerp form) 
             form
             (progn
               (if (symbolp form) (setq form (symbol-name form)))
               (if (and (stringp form) (eql (length form) 4))
                 (logior (ash (%char-code (char form 0)) 24)
                         (ash (%char-code (char form 1)) 16)
                         (ash (%char-code (char form 2)) 8)
                         (%char-code (char form 3)))
                 (if (characterp form) (%char-code form))))))))

; execute body, cleanup afterwards (if need to)
(defun sparc2-undo-body (seg vreg xfer body old-stack)
  (let* ((current-stack (sparc2-encode-stack))
         (numundo (%i- *sparc2-undo-count* (sparc2-encoding-undo-count old-stack))))
    (declare (fixnum numundo))
    (with-sparc-local-vinsn-macros (seg vreg xfer)
      (if (eq current-stack old-stack)
        (sparc2-form seg vreg xfer body)
        (if (eq xfer $backend-return)
          (progn
            (sparc2-form seg vreg xfer body)
            (dotimes (i numundo) (sparc2-close-undo)))
          (if (sparc2-mvpass-p xfer)
            (progn
              (sparc2-mvpass seg body) ; presumed to be ok
              (let* ((*sparc2-returning-values* :pass))
                (sparc2-nlexit seg xfer numundo)
                (^))
              (dotimes (i numundo) (sparc2-close-undo)))
            (progn
              ; There are some cases where storing thru sparc::%arg_z can be avoided (stores to vlocs, specials,
              ; etc.) and some other case where it can't ($test, $vpush.)  The case of a null vd can
              ; certainly avoid it; the check of numundo is to keep $acc boxed in case of nthrow.
              (sparc2-form  seg (if (or vreg (not (%izerop numundo))) sparc::%arg_z) nil body)
              (sparc2-unwind-set seg xfer old-stack)
              (when vreg (<- sparc::%arg_z))
              (^))))))))


(defun sparc2-unwind-set (seg xfer encoding)
  (multiple-value-bind (target-catch target-cstack target-vstack target-vstack-lcell)
                       (sparc2-decode-stack encoding)
    (sparc2-unwind-stack seg xfer target-catch target-cstack target-vstack)
    (setq *sparc2-undo-count* target-catch 
          *sparc2-cstack* target-cstack
          *sparc2-vstack* target-vstack
          *sparc2-top-vstack-lcell* target-vstack-lcell)))

(defun sparc2-unwind-stack (seg xfer target-catch target-cstack target-vstack)
  (let* ((current-catch *sparc2-undo-count*)
         (current-cstack *sparc2-cstack*)
         (current-vstack *sparc2-vstack*)
         (diff (%i- current-catch target-catch))
         target
         (exit-vstack current-vstack))
    (declare (ignore-if-unused target))
    (when (neq 0 diff)
      (setq exit-vstack (sparc2-nlexit seg xfer diff))
      (multiple-value-setq (target current-cstack current-vstack)
                           (sparc2-decode-stack (aref *sparc2-undo-stack* target-catch))))
    (if (%i< 0 (setq diff (%i- current-cstack target-cstack)))
      (sparc2-fix-stackreg seg sparc::%lsp diff))
    (if (%i< 0 (setq diff (%i- current-vstack target-vstack)))
      (with-sparc-local-vinsn-macros (seg)
        (! vstack-discard (ash diff -2))))
    exit-vstack))

; We can sometimes combine unwinding the catch stack with returning from the function
; by jumping to a subprim that knows how to do this.  If catch frames were distinguished
; from unwind-protect frames, we might be able to do this even when saved registers
; are involved (but the subprims restore them from the last catch frame.)
; *** there are currently only subprims to handle the "1 frame" case; add more ***
(defun sparc2-do-return (seg)
  (let* ((*sparc2-vstack* *sparc2-vstack*)
         (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*)
         (mask *sparc2-register-restore-count*)
         (ea *sparc2-register-restore-ea*)
         (label nil)
         (vstack nil)
         (foldp (not *sparc2-open-code-inline*)))
    (if (%izerop mask) (setq mask nil))
    (with-sparc-local-vinsn-macros (seg)
      (progn
        (setq vstack (sparc2-set-vstack (sparc2-unwind-stack seg $backend-return 0 0 #x7ff)))
        (if *sparc2-returning-values*
          (cond ((and mask foldp (setq label (%cdr (assq vstack *sparc2-valret-labels*))))
                 (-> label))
                (t
                 (@ (setq label (backend-get-next-label)))
                 (push (cons vstack label) *sparc2-valret-labels*)
                 (when mask
                   (with-imm-temps () (vsp0)
                     (! fixnum-add vsp0 sparc::%vsp sparc::%nargs)
                     (sparc2-restore-nvrs seg ea mask vsp0)))
                 (! jump-subprim .SPnvalret)))
          (if (null mask)
            (if *sparc2-open-code-inline*
              (progn
                (! restore-full-lisp-context)
                (! jump-return-pc))
              (! popj))
            (if (and foldp (setq label (assq *sparc2-vstack* *sparc2-popreg-labels*)))
              (-> (cdr label))
              (let* ((new-label (backend-get-next-label)))
                (@ new-label)
                (push (cons *sparc2-vstack* new-label) *sparc2-popreg-labels*)
                (sparc2-set-vstack (sparc2-restore-nvrs seg ea mask))
                (if *sparc2-open-code-inline*
                  (progn
                    (! restore-full-lisp-context)
                    (! jump-return-pc))
                  (! popj))))))))
    nil))



(defun sparc2-mvcall (seg vreg xfer fn arglist &optional recursive-p)
  (let* ((cstack *sparc2-cstack*)
         (vstack *sparc2-vstack*))
    (with-sparc-local-vinsn-macros (seg vreg xfer)
      (if (and (eq xfer $backend-return) (not (sparc2-tailcallok xfer)))
        (progn
          (sparc2-mvcall seg vreg $backend-mvpass fn arglist t)
          (sparc2-set-vstack (%i+ (if arglist 4 0) vstack))
          (setq *sparc2-cstack* cstack)
          (let* ((*sparc2-returning-values* t)) (^)))
        (let* ((mv-p (sparc2-mv-p xfer)))
          (if (null arglist)
            (sparc2-call-fn seg vreg xfer fn arglist nil)
            (progn
              (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg fn sparc::%arg_z))
              (sparc2-multiple-value-body seg (pop arglist))
              (when arglist
                (sparc2-open-undo $undostkblk)
                (! call-subprim .SPsave-values)
                (dolist (form arglist)
                  (sparc2-multiple-value-body seg form)
                  (! call-subprim .SPadd-values))
                (sparc2-set-nargs seg 0)
                (! call-subprim .SPrecover-values)
                (sparc2-close-undo))
              (! lisp-word-ref sparc::%temp0 sparc::%vsp sparc::%nargs)
              (sparc2-invoke-fn seg sparc::%temp0 nil nil xfer)))
          (unless recursive-p
            (if mv-p
              (unless (eq xfer $backend-return)
                (let* ((*sparc2-returning-values* t))
                  (^)))
              (progn 
                (sparc2-adjust-vstack -4)          ; discard function
                (! vstack-discard 1)
                (<- sparc::%arg_z)
                (^)))))))))


(defun sparc2-hard-opt-p (opts)
  (or
   (dolist (x (%cadr opts))
     (unless (nx-null x) (return t)))
   (dolist (x (%caddr opts))
     (when x (return t)))))

(defun sparc2-close-lambda (seg req opt rest keys auxen)
  (dolist (var req)
    (sparc2-close-var seg var))
  (dolist (var (%car opt))
    (sparc2-close-var seg var))
  (dolist (var (%caddr opt))
    (when var
      (sparc2-close-var seg var)))
  (if rest
    (sparc2-close-var seg rest))
  (dolist (var (%cadr keys))
    (sparc2-close-var seg var))
  (dolist (var (%caddr keys))
    (if var (sparc2-close-var seg var)))
  (dolist (var (%car auxen))
    (sparc2-close-var seg var)))

(defun sparc2-close-structured-var (seg var)
  (if (sparc2-structured-var-p var)
    (apply #'sparc2-close-structured-lambda seg (cdr var))
    (sparc2-close-var seg var)))

(defun sparc2-close-structured-lambda (seg whole req opt rest keys auxen)
  (if whole
    (sparc2-close-var seg whole))
  (dolist (var req)
    (sparc2-close-structured-var seg var))
  (dolist (var (%car opt))
    (sparc2-close-structured-var seg var))
  (dolist (var (%caddr opt))
    (when var
      (sparc2-close-var seg var)))
  (if rest
    (sparc2-close-structured-var seg rest))
  (dolist (var (%cadr keys))
    (sparc2-close-structured-var seg var))
  (dolist (var (%caddr keys))
    (if var (sparc2-close-var seg var)))
  (dolist (var (%car auxen))
    (sparc2-close-var seg var)))


(defun sparc2-init-regvar (seg var reg addr)
  (sparc2-stack-to-register seg addr reg)
  (sparc2-set-var-ea seg var reg))

(defun sparc2-bind-structured-var (seg var vloc lcell &optional context)
  (if (not (sparc2-structured-var-p var))
    (let* ((reg (sparc2-assign-register-var var)))
      (if reg
        (sparc2-init-regvar seg var reg (sparc2-vloc-ea vloc))
        (sparc2-bind-var seg var vloc lcell)))
    (let* ((v2 (%cdr var))
           (v v2)
           (vstack *sparc2-vstack*)
           (whole (pop v))
           (req (pop v))
           (opt (pop v))
           (rest (pop v))
           (keys (pop v)))
      
      (apply #'sparc2-bind-structured-lambda seg 
             (sparc2-spread-lambda-list seg (sparc2-vloc-ea vloc) whole req opt rest keys context)
             vstack context v2))))

(defun sparc2-bind-structured-lambda (seg lcells vloc context whole req opt rest keys auxen
                        &aux (nkeys (list-length (%cadr keys))))
  (declare (fixnum vloc))
  (when whole
    (sparc2-bind-structured-var seg whole vloc (pop lcells))
    (incf vloc 4))
  (dolist (arg req)
    (sparc2-bind-structured-var seg arg vloc (pop lcells) context)
    (incf vloc 4))
  (when opt
   (if (sparc2-hard-opt-p opt)
     (setq vloc (apply #'sparc2-structured-initopt seg lcells vloc context opt)
           lcells (nthcdr (ash (length (car opt)) 1) lcells))
     (dolist (var (%car opt))
       (sparc2-bind-structured-var seg var vloc (pop lcells) context)
       (incf vloc 4))))
  (when rest
    (sparc2-bind-structured-var seg rest vloc (pop lcells) context)
    (incf vloc 4))
  (when keys
    (apply #'sparc2-structured-init-keys seg lcells vloc context keys)
    (setq vloc (%i+ vloc (%ilsl 3 nkeys))))
  (sparc2-seq-bind seg (%car auxen) (%cadr auxen)))

(defun sparc2-structured-var-p (var)
  (and (consp var) (or (eq (%car var) *nx-lambdalist*)
                       (eq (%car var) (%nx1-operator lambda-list)))))

(defun sparc2-simple-var (var &aux (bits (cadr var)))
  (if (or (%ilogbitp $vbitclosed bits)
          (%ilogbitp $vbitspecial bits))
    (nx-error "Non-simple-variable ~S" (%car var))
    var))

(defun sparc2-nlexit (seg xfer &optional (nlevels 0))
  (let* ((numnthrow 0)
         (n *sparc2-undo-count*)
         (cstack *sparc2-cstack*)
         (vstack *sparc2-vstack*)
         (target-cstack)
         (target-vstack)
         (lastcatch n)
         (i nil)
         (returning (eq xfer $backend-return))
         (junk1 nil)
         (nspecs 0)
         (dest (%i- n nlevels))
         (retval *sparc2-returning-values*)
         reason)
    (declare (ignorable junk1))
    (declare (fixnum nspecs))
    (with-sparc-local-vinsn-macros (seg)
      (when (neq 0 nlevels)
        (let* ((numnlispareas 0))
          (declare (fixnum numnlispareas))
          (flet ((popnlispareas ()
                   (dotimes (i numnlispareas)
                     (! discard-temp-frame)))
                 (throw-through-numnthrow-catch-frames ()
                   (when (neq 0 numnthrow)
                     (sparc2-lwi seg sparc::%imm0 (ash numnthrow sparc::fixnum-shift))
                     (! call-subprim (if retval .SPnthrowvalues .SPnthrow1value))
                     (setq numnthrow 0)
                     (multiple-value-setq (junk1 cstack vstack)
                       (sparc2-decode-stack (aref *sparc2-undo-stack* lastcatch))))))
            (while (%i> n dest)
              (cond ((eql $undocatch (setq reason (aref *sparc2-undo-because* (setq n (%i- n 1)))))
                     (popnlispareas)
                     (setq numnthrow (%i+ numnthrow 1) lastcatch n))
                    ((eql $undostkblk reason)
                     (throw-through-numnthrow-catch-frames)
                     (incf numnlispareas))))
            (throw-through-numnthrow-catch-frames)
            (setq i lastcatch)
            (while (%i> i dest)
              (let ((reason (aref *sparc2-undo-because* (setq i (%i- i 1)))))
                (if (eq reason $undospecial)
                  (setq nspecs (%i+ nspecs 1)))))
            (if (> nspecs 0)
              (sparc2-dpayback seg nspecs))
            (when (and (neq lastcatch dest)
                       (%i>
                        vstack
                        (setq target-vstack 
                              (nth-value 2 (sparc2-decode-stack (aref *sparc2-undo-stack* dest)))))
                       (neq retval t))
              (unless returning
                (let ((vdiff (%i- vstack target-vstack)))
                  (if retval
                    (progn
                      (sparc2-lwi seg sparc::%imm0 vdiff)
                      (! call-subprim .SPmvslide))
                    (sparc2-fix-stackreg seg sparc::%vsp vdiff)))))
            (setq numnlispareas 0)
            (while (%i> lastcatch dest)
              (let ((reason (aref *sparc2-undo-because* (setq lastcatch (%i- lastcatch 1)))))
                (setq target-cstack (nth-value 1
                                               (sparc2-decode-stack (aref *sparc2-undo-stack* lastcatch))))
                (if (eq reason $undostkblk)
                  (incf numnlispareas))
                (if (%i> cstack target-cstack)
                  (sparc2-fix-stackreg seg sparc::%lsp (%i- cstack target-cstack)))
                ; else what's going on? $sp-stkcons, for one thing
                (setq cstack target-cstack)))
            (popnlispareas)))
        vstack))))


; Restore the N most recent dynamic bindings.
(defun sparc2-dpayback (seg n)
  (declare (fixnum n))
  (with-sparc-local-vinsn-macros (seg)
    (! dpayback n)))

(defun sparc2-spread-lambda-list (seg listform whole req opt rest keys 
                                    &optional enclosing-ea cdr-p)
  (with-sparc-local-vinsn-macros (seg)
    (let* ((numopt (length (%car opt)))
           (nkeys (length (%cadr keys)))
           (numreq (length req))
           (vtotal numreq)
           (old-top *sparc2-top-vstack-lcell*)
           (doadlword (dpb nkeys (byte 8 16) (dpb numopt (byte 8 8) (dpb numreq (byte 8 0) 0 )))))
      (declare (fixnum numopt nkeys numreq vtotal doadlword))
      (when (or (> numreq 255) (> numopt 255) (> nkeys 255))
        (error "A lambda list can contain a maximum of 255 required, 255 optional, and 255 keywords args"))
      (if (fixnump listform)
        (sparc2-store-ea seg listform sparc::%temp3)
        (sparc2-one-targeted-reg-form seg listform sparc::%temp3))
      (when whole
        (sparc2-vpush-register seg sparc::%temp3 :reserved))
      (when keys
        (setq doadlword (%ilogior2 (ash #x80000000 -6) doadlword))
        (incf  vtotal (%ilsl 1 nkeys))
        (if (%car keys)                 ; &allow-other-keys
          (setq doadlword (%ilogior doadlword (ash #x80000000 -5))))
        (sparc2-store-immediate seg (%car (%cdr (%cdr (%cdr (%cdr keys))))) sparc::%temp2))
      (when opt
        (setq vtotal (%i+ vtotal numopt))
        (when (sparc2-hard-opt-p opt)
          (setq doadlword (%ilogior2 doadlword (ash #x80000000 -7)))
          (setq vtotal (%i+ vtotal numopt))))
      (when rest
        (setq doadlword (%ilogior2 (ash #x80000000 -4) doadlword) vtotal (%i+ vtotal 1)))
      (sparc2-reserve-vstack-lcells vtotal)
      (sparc2-lwi seg sparc::%nargs doadlword)
      (if cdr-p
        (! call-subprim .SPmacro-bind)
        (if enclosing-ea
          (progn
            (sparc2-store-ea seg enclosing-ea sparc::%arg_z)
            (! call-subprim .SPdestructuring-bind-inner))
          (! call-subprim .SPdestructuring-bind)))
      (sparc2-set-vstack (%i+ *sparc2-vstack* (%ilsl 2 vtotal)))
      (sparc2-collect-lcells :reserved old-top))))


(defun sparc2-tailcallok (xfer)
  (and (eq xfer $backend-return)
       *sparc2-tail-allow*
       (eq 0 *sparc2-undo-count*)))

(defun sparc2-mv-p (cd)
  (or (eq cd $backend-return) (sparc2-mvpass-p cd)))

(defun sparc2-vinsn-optimize (header)
  ;; There are several little peephole optimizations that're worth
  ;; considering.  Becase VPUSHing is so expensive on the SPARC,
  ;; replacing sequences of 2 or more VPUSHes with a specialized
  ;; VPUSHn would save space (and a little time.)
  ;; This should actually be one of the last things we do, since
  ;; other things might add/remove vpushes & vpops.
  (sparc2-pack-vpush-sequences header))


; Vinsn is known to vpush a single register.  Return
; that register.

(defun sparc2-register-vpushed-by (vinsn)
  (svref (vinsn-variable-parts vinsn) 0))

; If vinsn vpushes a single register, return that register.
(defun sparc2-vpush-p (vinsn)
  (and (vinsn-attribute-= vinsn :push :node :vsp)
       (not (vinsn-attribute-= vinsn :multiple))
       (sparc2-register-vpushed-by vinsn)))

; "start" is a vpush.  Return the (possibly null) end of a chain
; of up to "max" succeeding vpush instructions.
(defun sparc2-vpush-chain (start max)
  (let* ((last nil)
	 (count 0))
    (declare (fixnum count))
    (do* ((next (dll-node-succ start) (dll-node-succ next)))
	 ((or (= count max) (not (sparc2-vpush-p next))))
      (setq last next)
      (incf count))
    (values last count)))

; The last operand to the VPUSHn vinsn is the first register vpushed.
; Easiest to just go through the chain backwards
(defun sparc2-replace-vpush-chain (header last first count)
  (declare (fixnum count))
  (let* ((regs (make-array count)))
    (declare (dynamic-extent regs) (simple-vector regs))
    (do* ((i 0 (1+ i))
	  (vinsn last (dll-node-pred vinsn)))
	 ((= i count))
      (declare (fixnum i))
      (setf (svref regs i) (sparc2-register-vpushed-by vinsn)))
    (with-sparc-local-vinsn-macros (header)
      (let* ((multi
	      (case count
		(2 (! vpush2 (svref regs 0) (svref regs 1)))
		(3 (! vpush3 (svref regs 0) (svref regs 1) (svref regs 2)))
		(4 (! vpush4 (svref regs 0) (svref regs 1) (svref regs 2)
		      (svref regs 3)))
		(5 (! vpush5 (svref regs 0) (svref regs 1) (svref regs 2)
		      (svref regs 3) (svref regs 4)))
		(6 (! vpush6 (svref regs 0) (svref regs 1) (svref regs 2)
		      (svref regs 3) (svref regs 4) (svref regs 5)))
		(7 (! vpush7 (svref regs 0) (svref regs 1) (svref regs 2)
		      (svref regs 3) (svref regs 4) (svref regs 5)
		      (svref regs 6)))
		(8 (! vpush8 (svref regs 0) (svref regs 1) (svref regs 2)
		      (svref regs 3) (svref regs 4) (svref regs 5)
		      (svref regs 6) (svref regs 7)))
		(9 (! vpush9 (svref regs 0) (svref regs 1) (svref regs 2)
		      (svref regs 3) (svref regs 4) (svref regs 5)
		      (svref regs 6) (svref regs 7) (svref regs 8)))))
	     (pred (dll-node-pred first)))
	(move-dll-nodes multi pred 1)
	(remove-dll-node first count)
	multi))))

(defun sparc2-pack-vpush-sequences (header)
  (do* ((current (dll-header-first header) (dll-node-succ current)))
       ((eq current header))
    (when (sparc2-vpush-p current)
      (multiple-value-bind (end count) (sparc2-vpush-chain current 8)
	(when end
	  (setq current
		(sparc2-replace-vpush-chain header end current (1+ count))))))))
  
(defun sparc2-expand-note (note)
  (let* ((lab (vinsn-note-label note)))
    (case (vinsn-note-class note)
      ((:regsave :begin-variable-scope :end-variable-scope)
       (setf (vinsn-label-info lab) (emit-lap-label lab))))))

; We have to pay attention here to whether the :OP2 operand
; value is a register number or an immediate.
(defun sparc2-expand-vinsns (header &optional (optimize t))
  (when optimize
    (optimize-vinsns header)
    (sparc2-vinsn-optimize header))
  (do-dll-nodes (v header)
    (if (%vinsn-label-p v)
      (let* ((id (vinsn-label-id v)))
        (if (typep id 'fixnum)
          (when (or t (vinsn-label-refs v))
            (setf (vinsn-label-info v) (emit-lap-label v)))
          (sparc2-expand-note id)))
      (sparc2-expand-vinsn v)))
  (free-logical-registers)
  (sparc2-free-lcells))

; It's not clear whether or not predicates, etc. want to look
; at an lreg or just at its value slot.
; It's clear that the assembler just wants the value, and that
; the value had better be assigned by the time we start generating
; machine code.
; For now, we replace lregs in the operand vector with their values
; on entry, but it might be reasonable to make PARSE-OPERAND-FORM
; deal with lregs ...
(defun sparc2-expand-vinsn (vinsn)
;  (format t "~& ~a" vinsn)
  (let* ((template (vinsn-template vinsn))
         (note-stack ())
         (vp (vinsn-variable-parts vinsn))
         (nvp (vinsn-template-nvp template)))
    (declare (fixnum nvp))
    (dotimes (i nvp)
      (let* ((val (svref vp i)))
        (when (typep val 'lreg)
          (setf (svref vp i) (lreg-value val)))))                       
    (dolist (name (vinsn-template-local-labels template))
      (make-lap-label name))
    (labels ((parse-operand-form (valform)
               (cond ((atom valform) valform)
                     ((atom (cdr valform)) (svref vp (car valform)))
                     (t (let* ((op-vals (cdr valform))
                               (parsed-ops (make-list (length op-vals)))
                               (tail parsed-ops))
                          (declare (dynamic-extent parsed-ops)
                                   (cons parsed-ops tail))
                          (dolist (op op-vals (apply (car valform) parsed-ops))
                            (setq tail (cdr (rplaca tail (parse-operand-form op)))))))))
             (expand-insn-form (f)
               (let* ((operands (cdr f))
                      (head (make-list (length operands)))
                      (tail head))
                 (declare (dynamic-extent head)
                          (cons (head tail)))
                 (dolist (op operands)
                   (rplaca tail (parse-operand-form op))
                   (setq tail (cdr tail)))
;		 (format t "~&-> ~a ~s" (arch::opcode-name (svref sparc::*sparc-opcodes* (car f)))  head)
                 (sparc-emit-lap-instruction (svref sparc::*sparc-opcodes* (car f)) 
                                           head t)))
             (eval-predicate (f)
               (case (car f)
                 (:pred (let* ((op-vals (cddr f))
                               (parsed-ops (make-list (length op-vals)))
                               (tail parsed-ops))
                          (declare (dynamic-extent parsed-ops)
                                   (cons parsed-ops tail))
                          (dolist (op op-vals (apply (cadr f) parsed-ops))
                            (setq tail (cdr (rplaca tail (parse-operand-form op)))))))
                 (:not (not (eval-predicate (cadr f))))
                 (:or (some #'eval-predicate (cadr f)))
                 (:and (every #'eval-predicate (cadr f)))
                 (t (error "Unknown predicate: ~s" f))))
             (expand-form (f)
               (if (keywordp f)
                 (emit-lap-label f)
                 (if (atom f)
                   (case f
                     (< (let* ((id (length note-stack))
                               (start (make-lap-note-begin :id id))
                               (end (make-lap-note-end :id id :peer start)))
                          (push end note-stack)
                          (setf (lap-note-begin-peer start) end)
                          (emit-lap-note start)))
                     (> (emit-lap-note (pop note-stack)))
                     (t
                      (error "Invalid form in vinsn body: ~s" f)))
                   (if (atom (car f))
                     (expand-insn-form f)
                     (if (eval-predicate (car f))
                       (dolist (subform (cdr f))
                         (expand-form subform))))))))
      (declare (dynamic-extent #'expand-form #'parse-operand-form #'expand-insn-form #'eval-predicate))
      ;(format t "~& vinsn = ~s" vinsn)
      (dolist (form (vinsn-template-body template))
        (expand-form form ))
      (setf (vinsn-variable-parts vinsn) nil)
      (when vp
        (free-varparts-vector vp)))))


(defun sparc2-emit-event-poll (seg)
  (with-sparc-local-vinsn-macros (seg)
    (unless *sparc2-inhibit-eventchecks*
      (let* ((event-poll-template (need-vinsn-template (load-time-value (get-vinsn-template-cell 'event-poll *sparc-vinsn-templates*))*sparc-vinsn-templates*)))
        (or (do* ((prev (dll-header-last seg) (dll-node-pred prev)))
                 ((eq prev seg) nil)
              (when (typep prev 'vinsn)
                (if (eq (vinsn-template prev) event-poll-template)
                  (progn
                    (remove-dll-node prev)
                    (append-dll-node prev seg)
                    (return t))
                  (return nil))))            
            (append-dll-node (make-vinsn event-poll-template) seg))))))

(defparameter *sparc2-builtin-subprims* '(((0 . 23) . #..SPbuiltin-plus)))

(defun sparc2-builtin-index-subprim (idx)
  (dolist (cell *sparc2-builtin-subprims*)
    (destructuring-bind ((low . high) . base) cell
      (if (and (>= idx low)
               (< idx high))
        (return (+ base (ash (- idx low) 2)))))))

(defun sparc2-fixed-call-builtin (seg vreg xfer name subprim)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((index (arch::builtin-function-name-offset name))
           (idx-subprim (sparc2-builtin-index-subprim index))
           (tail-p (sparc2-tailcallok xfer)))
      (when tail-p
        (sparc2-restore-nvrs seg *sparc2-register-restore-ea* *sparc2-register-restore-count*)
        (sparc2-restore-full-lisp-context seg))
      (if idx-subprim
        (setq subprim idx-subprim)
        (! lwi sparc::%imm0 (ash index sparc::fixnumshift)))
      (if tail-p
        (! jump-subprim subprim)
        (progn
          (! call-subprim subprim)
          (<- sparc::%arg_z)
          (^))))))

(defun sparc2-unary-builtin (seg vreg xfer name form)
  (sparc2-one-targeted-reg-form seg form sparc::%arg_z)
  (sparc2-fixed-call-builtin seg vreg xfer name .SPcallbuiltin1))

(defun sparc2-binary-builtin (seg vreg xfer name form1 form2)
  (sparc2-two-targeted-reg-forms seg form1 sparc::%arg_y form2 sparc::%arg_z)
  (sparc2-fixed-call-builtin seg vreg xfer name .SPcallbuiltin2))

(defun sparc2-ternary-builtin (seg vreg xfer name form1 form2 form3)
  (sparc2-three-targeted-reg-forms seg form1 sparc::%arg_x form2 sparc::%arg_y form3 sparc::%arg_z)
  (sparc2-fixed-call-builtin seg vreg xfer name .SPcallbuiltin3))


(eval-when (:compile-toplevel :execute :load-toplevel)


(defmacro defsparc2 (name locative arglist &body forms)
  (multiple-value-bind (body decls)
                       (parse-body forms nil t)
    (destructuring-bind (vcode-block dest control &rest other-args) arglist
      (let* ((fun `(nfunction ,name 
                              (lambda (,vcode-block ,dest ,control ,@other-args) ,@decls 
                                      (block ,name (with-sparc-local-vinsn-macros (,vcode-block ,dest ,control) ,@body))))))
        `(progn
           (record-source-file ',name 'function)
           (svset *sparc2-specials* (%ilogand #.operator-id-mask (%nx1-operator ,locative)) ,fun))))))
)
  
(defsparc2 sparc2-lambda lambda-list (seg vreg xfer req opt rest keys auxen body p2decls)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((stack-consed-rest nil)
           (lexprp (if (consp rest) (progn (setq rest (car rest)) t)))
           (rest-var-bits (and rest (nx-var-bits rest)))
           (rest-ignored-p (and rest (not lexprp) (%ilogbitp $vbitignore rest-var-bits)))
           (want-stack-consed-rest (or rest-ignored-p
                                       (and rest (not lexprp) (%ilogbitp $vbitdynamicextent rest-var-bits))))
           (afunc *sparc2-cur-afunc*)
           (inherited-vars (afunc-inherited-vars afunc))
           (fbits (afunc-bits afunc))
           (methodp (%ilogbitp $fbitmethodp fbits))
           (method-var (if methodp (pop req)))
           (next-method-p (%ilogbitp $fbitnextmethp fbits))
           (allow-other-keys-p (%car keys))
           (hardopt (sparc2-hard-opt-p opt))
           (lap-p (when (and (consp (%car req)) (eq (%caar req) '&lap))
                    (prog1 (%cdar req) (setq req nil))))
           (num-inh (length inherited-vars))
           (num-req (length req))
           (num-opt (length (%car opt)))
           (no-regs nil)
           (arg-regs nil)
           optsupvloc
           reglocatives
           pregs
           (reserved-lcells nil)
           (*sparc2-vstack* 0))
      (declare (type (unsigned-byte 16) num-req num-opt num-inh reqvloc))
      (with-sparc-p2-declarations p2decls
        (setq *sparc2-inhibit-register-allocation*
              (setq no-regs (%ilogbitp $fbitnoregs fbits)))
        (multiple-value-setq (pregs reglocatives) 
          (sparc2-allocate-global-registers *sparc2-fcells* *sparc2-vcells* (afunc-all-vars afunc) no-regs))
        (@ (backend-get-next-label)) ; generic self-reference label, should be label #1
        (unless next-method-p
          (setq method-var nil))
        
        (let* ((rev-req (reverse req))
               (rev-fixed (if inherited-vars (reverse (append inherited-vars req)) rev-req))
               (num-fixed (length rev-fixed))
               (rev-opt (reverse (car opt))))

          (if (not (or opt rest keys))
            (setq arg-regs (sparc2-req-nargs-entry seg rev-fixed))
            (if (and (not (or hardopt rest keys))
                     (<= num-opt $numsparcargregs))
              (setq arg-regs (sparc2-simple-opt-entry seg rev-opt rev-fixed))
              (progn
                ; If the minumum acceptable number of args is non-zero, ensure
                ; that at least that many were received.  If there's an upper bound,
                ; enforce it.
                
                (when rev-fixed
                  (sparc2-reserve-vstack-lcells num-fixed)                    
                  (! check-min-nargs num-fixed))
                (unless (or rest keys)
                  (! check-max-nargs (+ num-fixed num-opt)))
                ;; Going to have to call one or more subprims.  First save
                ;; the LR in LOC-PC.
                ;; If there were &optional args, initialize their values
                ;; to NIL.  All of the argregs get vpushed as a result of this.
                (when opt
                  (sparc2-reserve-vstack-lcells num-opt)
                  (! default-optionals (+ num-fixed num-opt)))
                (when keys
                  (let* ((keyvect (%car (%cdr (%cdr (%cdr (%cdr keys))))))
                         (flags (the fixnum (logior (the fixnum (if rest 4 0)) 
                                                    (the fixnum (if (or methodp allow-other-keys-p) 1 0)))))
                         (nkeys (length keyvect))
                         (nprev (+ num-fixed num-opt))
                         (spno (if (= 0 nprev)
                                 .SPsimple-keywords
                                 (if (= 0 num-opt)
                                   .SPkeyword-args
                                   .SPkeyword-bind))))
                    (declare (fixnum flags nkeys nprev spno))
                    (dotimes (i (the fixnum (+ nkeys nkeys)))
                      (sparc2-new-vstack-lcell :reserved 4 0 nil))
                    (! misc-ref-c-node sparc::%temp3 sparc::%nfn (1+ (backend-immediate-index keyvect)))
                    (sparc2-lwi seg sparc::%imm2 (ash flags arch::fixnumshift))
                    (sparc2-lwi seg sparc::%imm3 (ash nkeys arch::fixnumshift))
                    (unless (= nprev 0)
                      (sparc2-lwi seg sparc::%imm0 (ash nprev sparc::fixnumshift)))
                    (! call-subprim* spno)))
                (when rest
                  ;; If any keyword-binding's happened, the key/value pairs have been slid to the top-of-stack
                  ;; for us.  There'll be an even number of them (nargs - the "previous" (required/&optional)
                  ;; count.)
                  (if lexprp
                    (sparc2-lexpr-entry seg num-fixed)
                    (progn
                      (if want-stack-consed-rest
                        (setq stack-consed-rest t))
                      (let* ((nprev (+ num-fixed num-opt))
                             (simple (and (not keys) (= 0 nprev)))
                             (spno (if stack-consed-rest
                                     (if simple
                                       .SPstack-rest-arg
                                       (if (and (not keys) (= 0 num-opt))
                                         .SPreq-stack-rest-arg
                                         .SPstack-cons-rest-arg))
                                     (if simple
                                       .SPheap-rest-arg
                                       (if (and (not keys) (= 0 num-opt))
                                         .SPreq-heap-rest-arg
                                         .SPheap-cons-rest-arg)))))
                        (declare (fixnum nprev))
                        (unless simple
                          (sparc2-lwi seg sparc::%imm0 (ash nprev sparc::fixnumshift)))
                        (! call-subprim* spno))
                      ; Make an lcell for the &rest arg
                      (sparc2-reserve-vstack-lcells 1))))
                (when hardopt
                  (sparc2-reserve-vstack-lcells num-opt)
                  (sparc2-lwi seg sparc::%imm0 (ash num-opt sparc::fixnumshift))
                  ; .SPopt-supplied-p wants nargs to contain the actual arg-count
                  ; minus the number of "fixed" (required, inherited) args.
                  (unless (= 0 num-fixed)
                    (! scale-nargs num-fixed))
                  (! call-subprim* .SPopt-supplied-p))
                (let* ((nwords-vpushed (+ num-fixed 
                                          num-opt 
                                          (if hardopt num-opt 0) 
                                          (if lexprp 0 (if rest 1 0))
                                          (ash (length (%cadr keys)) 1)))
                       (nbytes-vpushed (ash nwords-vpushed 2)))
                  (declare (fixnum nwords-vpushed nbytes-vpushed))
                  (unless (or lexprp keys) 
                    (if *sparc2-open-code-inline*
                      (! save-lisp-context-offset nbytes-vpushed)
                      (! save-lisp-context-offset-ool nbytes-vpushed)))
                  (sparc2-set-vstack nbytes-vpushed)
                  (setq optsupvloc (- *sparc2-vstack* (ash num-opt 2)))))))
          ;; Caller's context is saved; *sparc2-vstack* is valid.  Might still have method-var
          ;; to worry about.
          (unless (= 0 pregs)
            ;; Save NVRs; load constants into any that get constants.
            (sparc2-save-nvrs seg pregs)
            (dolist (pair reglocatives)
              (declare (cons pair))
              (let* ((constant (car pair))
                     (reg (cdr pair)))
                (declare (cons constant))
                (rplacd constant reg)
                (! ref-constant reg (backend-immediate-index (car constant))))))
          (when method-var
            (sparc2-seq-bind-var seg method-var sparc::%next-method-context))
          ; If any arguments are still in arg_x, arg_y, arg_z, that's because they weren't vpushed
          ; in a "simple" entry case and belong in some NVR.  Put them in their NVRs, so that we can
          ; handle arbitrary expression evaluation (special binding, value-cell consing, etc.) without
          ; clobbering the argument registers.
          (when arg-regs
            (do* ((vars arg-regs (cdr vars))
                  (arg-reg-num sparc::%arg_z (1- arg-reg-num)))
                 ((null vars))
              (declare (list vars) (fixnum arg-reg-num))
              (let* ((var (car vars)))
                (when var
                  (let* ((reg (sparc2-assign-register-var var)))
                    (sparc2-copy-register seg reg arg-reg-num)
                    (setf (var-ea var) reg))))))
          (setq *sparc2-entry-vsp-saved-p* t)
#|
          (when stack-consed-rest
            (if rest-ignored-p
              (if nil (sparc2-jsrA5 $sp-popnlisparea))
              (progn
                (sparc2-open-undo $undostkblk))))
|#
          (when stack-consed-rest
            (sparc2-open-undo $undostkblk))
          (setq *sparc2-entry-vstack* *sparc2-vstack*)
          (setq reserved-lcells (sparc2-collect-lcells :reserved))
          (sparc2-emit-event-poll seg)
          (sparc2-bind-lambda seg reserved-lcells req opt rest keys auxen optsupvloc arg-regs lexprp inherited-vars))
        (when method-var (sparc2-heap-cons-next-method-var seg method-var))
        (sparc2-form seg vreg xfer body)
        (sparc2-close-lambda seg req opt rest keys auxen)
        (dolist (v inherited-vars)
          (sparc2-close-var seg v))
        (when method-var
          (sparc2-close-var seg method-var))
        (let* ((bits 0))
          (when (%i> num-inh (ldb $lfbits-numinh -1))
            (setq num-inh (ldb $lfbits-numinh -1)))
          (setq bits (dpb num-inh $lfbits-numinh bits))
          (unless lap-p
            (when (%i> num-req (ldb $lfbits-numreq -1))
              (setq num-req (ldb $lfbits-numreq -1)))
            (setq bits (dpb num-req $lfbits-numreq bits))
            (when (%i> num-opt (ldb $lfbits-numopt -1))
              (setq num-opt (ldb $lfbits-numopt -1)))
            (setq bits (dpb num-opt $lfbits-numopt bits))
            (when hardopt (setq bits (%ilogior (%ilsl $lfbits-optinit-bit 1) bits)))
            (when rest (setq bits (%ilogior (if lexprp (%ilsl $lfbits-restv-bit 1) (%ilsl $lfbits-rest-bit 1)) bits)))
            (when keys (setq bits (%ilogior (%ilsl $lfbits-keys-bit 1) bits)))
            (when allow-other-keys-p (setq bits (%ilogior (%ilsl $lfbits-aok-bit 1) bits)))
            (when (%ilogbitp $fbitnextmethargsp (afunc-bits afunc))
              (if methodp
                (setq bits (%ilogior (%ilsl $lfbits-nextmeth-with-args-bit 1) bits))
                (let ((parent (afunc-parent afunc)))
                  (when parent
                    (setf (afunc-bits parent) (bitset $fbitnextmethargsp (afunc-bits parent)))))))
            (when methodp
              (setq bits (logior (ash 1 $lfbits-method-bit) bits))
              (when next-method-p
                (setq bits (logior (%ilsl $lfbits-nextmeth-bit 1) bits))))) 
          bits)))))


(defsparc2 sparc2-progn progn (seg vreg xfer forms)
  (declare (list forms))
  (if (null forms)
    (sparc2-nil seg vreg xfer)
    (loop
      (let* ((form (pop forms)))
        (if forms
          (sparc2-form seg nil nil form)
          (return (sparc2-form seg vreg xfer form)))))))

#|
(defsparc2 sparc2-prog1 prog1 (seg vreg xfer forms)
  (if (eq (list-length forms) 1)
    (sparc2-use-operator (%nx1-operator values) seg vreg xfer forms)
    (progn
      (if vreg
        (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg (pop forms) sparc::%arg_z))
        (sparc2-form seg nil nil (pop forms)))
      (dolist (form forms)
        (sparc2-form seg nil nil form))
      (if vreg (sparc2-vpop-register seg vreg))
      (^))))
|#


(defsparc2 sparc2-prog1 prog1 (seg vreg xfer forms)
  (if (eq (list-length forms) 1)
    (sparc2-use-operator (%nx1-operator values) seg vreg xfer forms)
    (if (null vreg)
      (sparc2-use-operator (%nx1-operator progn) seg vreg xfer forms)
      (let* ((float-p (= (hard-regspec-class vreg) hard-reg-class-fpr))
             (crf-p (= (hard-regspec-class vreg) hard-reg-class-crf))
             (node-p (unless (or float-p crf-p)
                       (= (get-regspec-mode vreg) hard-reg-class-gpr-mode-node)))
             (first (pop forms)))
        (sparc2-push-register seg 
                            (if (or node-p crf-p)
                              (sparc2-one-untargeted-reg-form seg first sparc::%arg_z)
                              (sparc2-one-targeted-reg-form seg first vreg)))
        (dolist (form forms)
          (sparc2-form seg nil nil form))
        (if crf-p
          (progn
            (sparc2-vpop-register seg sparc::%arg_z)
            (<- sparc::%arg_z))
          (sparc2-pop-register seg vreg))
        (^)))))

(defsparc2 sparc2-free-reference free-reference (seg vreg xfer sym)
  (sparc2-ref-symbol-value seg vreg xfer sym t))

(defsparc2 sparc2-special-ref special-ref (seg vreg xfer sym)
  (sparc2-ref-symbol-value seg vreg xfer sym t))

(defsparc2 sparc2-bound-special-ref bound-special-ref (seg vreg xfer sym)
  (sparc2-ref-symbol-value seg vreg xfer sym nil))

(defsparc2 sparc2-%svref %svref (seg vreg xfer vector index)
  (sparc2-misc-node-ref seg vreg xfer vector index nil))

(defsparc2 sparc2-svref svref (seg vreg xfer vector index)
  (sparc2-misc-node-ref seg vreg xfer vector index (unless *sparc2-reckless* arch::subtag-simple-vector)))

;; It'd be nice if this didn't box the result.  Worse things happen ...
;;  Once there's a robust mechanism, adding a CHARCODE storage class shouldn't be hard.
(defsparc2 sparc2-%sbchar %sbchar (seg vreg xfer string index)
  (sparc2-vref seg vreg xfer arch::subtag-simple-base-string string index (unless *sparc2-reckless* arch::subtag-simple-base-string)))

(defsparc2 sparc2-%sechar %sechar (seg vreg xfer string index)
  (sparc2-vref seg vreg xfer arch::subtag-simple-general-string string index (unless *sparc2-reckless* arch::subtag-simple-general-string)))


(defsparc2 sparc2-%svset %svset (seg vreg xfer vector index value)
  (sparc2-misc-node-set seg vreg xfer vector index value nil))

(defsparc2 sparc2-svset svset (seg vreg xfer vector index value)
   (sparc2-misc-node-set seg vreg xfer vector index value (unless *sparc2-reckless* arch::subtag-simple-vector)))

(defsparc2 sparc2-typed-form typed-form (seg vreg xfer typespec form)
  (declare (ignore typespec)) ; Boy, do we ever !
  (sparc2-form seg vreg xfer form))

(defsparc2 sparc2-%primitive %primitive (seg vreg xfer &rest ignore)
  (declare (ignore seg vreg xfer ignore))
  (error "You're probably losing big: using %primitive ..."))

(defsparc2 sparc2-consp consp (seg vreg xfer cc form)
  (if (null vreg)
    (sparc2-form seg vreg xfer form)
    (let* ((tagreg sparc::%imm0))
      (multiple-value-bind (cr-bit true-p) (acode-condition-to-sparc-icc cc)
        (! extract-fulltag tagreg (sparc2-one-untargeted-reg-form seg form sparc::%arg_z))
        (sparc2-test-reg-%izerop seg vreg xfer tagreg cr-bit true-p sparc::fulltag-cons)))))
      

(defsparc2 sparc2-cons cons (seg vreg xfer y z)
  (if (null vreg)
    (progn
      (sparc2-form seg nil nil y)
      (sparc2-form seg nil xfer z))
    (multiple-value-bind (yreg zreg) (sparc2-two-untargeted-reg-forms seg y sparc::%arg_y z sparc::%arg_z)
      (ensuring-node-target (target vreg)
        (! cons target yreg zreg))
      (^))))

(defsparc2 sparc2-%rplaca %rplaca (seg vreg xfer ptr val)
  (sparc2-modify-cons seg vreg xfer ptr val nil nil t))

(defsparc2 sparc2-%rplacd %rplacd (seg vreg xfer ptr val)
  (sparc2-modify-cons seg vreg xfer ptr val nil t t))

(defsparc2 sparc2-rplaca rplaca (seg vreg xfer ptr val)
  (sparc2-modify-cons seg vreg xfer ptr val t nil t))

(defsparc2 sparc2-set-car set-car (seg vreg xfer ptr val)
  (sparc2-modify-cons seg vreg xfer ptr val t nil nil))

(defsparc2 sparc2-rplacd rplacd (seg vreg xfer ptr val)
  (sparc2-modify-cons seg vreg xfer ptr val t t t))

(defsparc2 sparc2-set-cdr set-cdr (seg vreg xfer ptr val)
  (sparc2-modify-cons seg vreg xfer ptr val t t nil))

(defsparc2 sparc2-%car %car (seg vreg xfer form)
  (sparc2-reference-list seg vreg xfer form nil))

(defsparc2 sparc2-%cdr %cdr (seg vreg xfer form)
  (sparc2-reference-list seg vreg xfer form  t))

(defsparc2 sparc2-car car (seg vreg xfer form)
  (sparc2-reference-list seg vreg xfer form nil))

(defsparc2 sparc2-cdr cdr (seg vreg xfer form)
  (sparc2-reference-list seg vreg xfer form t))

(defsparc2 sparc2-vector vector (seg vreg xfer arglist)
  (sparc2-allocate-initialized-gvector seg vreg xfer arch::subtag-simple-vector arglist))

;; If we can pick a constant subtype out of the arglist, do so and allocate it inline.
;; If it looks like a 68k subtag (or anything other than a sparc::fulltag-nodeheader
;; tag), warn and see if we can come up with a better guess.
;; Of course, it'd be nice if we could catch this at runtime as well.  I'd guess
;; that most calls to %gvector have constant subtype/subtag arguments.  We'll see.
(defsparc2 sparc2-%sparc-gvector %ppc-gvector (seg vreg xfer arglist)
  (let* ((all-on-stack (append (car arglist) (reverse (cadr arglist))))
         (subtag-form (car all-on-stack))
         (subtag (acode-fixnum-form-p subtag-form)))
    (if (null vreg)
      (dolist (form all-on-stack (^)) (sparc2-form seg nil nil form))
      (if (null subtag)
        (progn                            ; Vpush everything and call subprim
          (let* ((*sparc2-vstack* *sparc2-vstack*)
                 (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*))
            (sparc2-set-nargs seg (sparc2-formlist seg all-on-stack nil))
            (! call-subprim .SPgvector))
          (<- sparc::%arg_z)
          (^))
        (let* ((subtag-tag (logand subtag sparc::full-tag-mask)))
          (declare (fixnum subtag-tag))
          (unless (= subtag-tag sparc::fulltag-nodeheader)
            (let* ((newtag arch::subtag-simple-vector))
              (warn "%gvector: subtag was ~d, using ~d instead" subtag newtag)
              (setq subtag newtag)))
          (sparc2-allocate-initialized-gvector seg vreg xfer subtag (cdr all-on-stack)))))))

;; Should be less eager to box result
(defsparc2 sparc2-%char-code %char-code (seg vreg xfer c)
  (sparc2-extract-charcode seg vreg xfer c nil))

(defsparc2 sparc2-char-code char-code (seg vreg xfer c)
  (sparc2-extract-charcode seg vreg xfer c (not (sparc2-form-typep c 'character))))

(defsparc2 sparc2-%ilogior2 %ilogior2 (seg vreg xfer form1 form2)
  (let* ((fix1 (acode-fixnum-form-p form1))
         (fix2 (acode-fixnum-form-p form2)))
    (if (and fix1 fix2)
      (sparc2-use-operator (%nx1-operator fixnum) seg vreg xfer (logior fix1 fix2)))
    (let* ((fixval (or fix1 fix2))
           (unboxed-fixval (if fixval (ash fixval sparc::fixnumshift)))
           (low (if (typep unboxed-fixval '(signed-byte 13)) unboxed-fixval))
           (otherform (if low (if fix1 form2 form1))))
      (if otherform
        (let* ((other-reg (sparc2-one-untargeted-reg-form seg otherform sparc::%arg_z)))
          (when vreg
            (ensuring-node-target (target vreg) 
              (! logior-low target other-reg low))))
        (multiple-value-bind (r1 r2) (sparc2-two-untargeted-reg-forms seg form1 sparc::%arg_y form2 sparc::%arg_z)
          (if vreg (ensuring-node-target (target vreg) (! %logior2 target r1 r2)))))   
      (^))))


(defsparc2 sparc2-%ilogand2 %ilogand2 (seg vreg xfer form1 form2)
  (let* ((fix1 (acode-fixnum-form-p form1))
         (fix2 (acode-fixnum-form-p form2)))
    (if (and fix1 fix2)
      (sparc2-use-operator (%nx1-operator fixnum) seg vreg xfer (logand fix1 fix2)))
    (let* ((fixval (or fix1 fix2))
           (unboxed-fixval (if fixval (ash fixval sparc::fixnumshift)))
           (low (and (typep unboxed-fixval '(signed-byte 13)) unboxed-fixval))
           (otherform (if low (if fix1 form2 form1))))
      (if otherform
        (let* ((other-reg (sparc2-one-untargeted-reg-form seg otherform sparc::%arg_z)))
          (when vreg
            (ensuring-node-target (target vreg) 
              (! logand-low target other-reg low))))
        (multiple-value-bind (r1 r2) (sparc2-two-untargeted-reg-forms seg form1 sparc::%arg_y form2 sparc::%arg_z)
          (if vreg (ensuring-node-target (target vreg) (! %logand2 target r1 r2)))))
      (^))))

(defsparc2 sparc2-%ilogxor2 %ilogxor2 (seg vreg xfer form1 form2)
  (let* ((fix1 (acode-fixnum-form-p form1))
         (fix2 (acode-fixnum-form-p form2)))
    (if (and fix1 fix2)
      (sparc2-use-operator (%nx1-operator fixnum) seg vreg xfer (logxor fix1 fix2)))
    (let* ((fixval (or fix1 fix2))
           (unboxed-fixval (if fixval (ash fixval sparc::fixnumshift)))
           (low (and (typep unboxed-fixval '(signed-byte 13)) unboxed-fixval))
           (otherform (if low (if fix1 form2 form1))))
      (if otherform
        (let* ((other-reg (sparc2-one-untargeted-reg-form seg otherform sparc::%arg_z)))
          (when vreg
            (ensuring-node-target (target vreg) 
	     (! logxor-low target other-reg low))))
        (multiple-value-bind (r1 r2) (sparc2-two-untargeted-reg-forms seg form1 sparc::%arg_y form2 sparc::%arg_z)
          (if vreg (ensuring-node-target (target vreg) (! %logxor2 vreg r1 r2)))))
      (^))))

(defsparc2 sparc2-%ineg %ineg (seg vreg xfer n)
  (let* ((src (sparc2-one-untargeted-reg-form seg n sparc::%arg_z)))
    (when vreg
      (ensuring-node-target (target vreg) 
        (! negate-fixnum target src)))
    (^)))

(defsparc2 sparc2-%%ineg %%ineg (seg vreg xfer n)
  (let* ((src (sparc2-one-untargeted-reg-form seg n sparc::%arg_z)))
    (when vreg
      (ensuring-node-target (target vreg) 
        (! negate-fixnum-no-ovf target src)))
    (^)))

(defsparc2 sparc2-characterp characterp (seg vreg xfer cc form)
  (sparc2-char-p seg vreg xfer cc form))

(defsparc2 sparc2-struct-ref struct-ref (seg vreg xfer struct offset)
  (sparc2-misc-node-ref seg vreg xfer struct offset arch::subtag-struct))

(defsparc2 sparc2-struct-set struct-set (seg vreg xfer struct offset value)
  (sparc2-misc-node-set seg vreg xfer struct offset value arch::subtag-struct))

(defsparc2 sparc2-sparc-lisptag ppc-lisptag (seg vreg xfer node)
  (if (null vreg)
    (sparc2-form seg vreg xfer node)
    (progn
      (ensuring-node-target (target vreg) 
        (! extract-tag[fixnum] target (sparc2-one-untargeted-reg-form seg node sparc::%arg_z)))
      (^))))

(defsparc2 sparc2-sparc-fulltag ppc-fulltag (seg vreg xfer node)
  (if (null vreg)
    (sparc2-form seg vreg xfer node)
    (progn
      (ensuring-node-target (target vreg) 
        (! extract-fulltag[fixnum] target (sparc2-one-untargeted-reg-form seg node sparc::%arg_z)))
      (^))))

(defsparc2 sparc2-sparc-typecode ppc-typecode (seg vreg xfer node)
  (if (null vreg)
    (sparc2-form seg vreg xfer node)
    (let* ((reg (sparc2-one-untargeted-reg-form seg node (if (eq (hard-regspec-value vreg) sparc::%arg_z) 
                                                         sparc::%arg_y sparc::%arg_z))))
      (ensuring-node-target (target vreg) 
        (! extract-typecode[fixnum] target reg ))
      (^))))

; Note that this goes through the whole MISC-NODE-SET mechanism.
; Among other things, that helps to ensure memoization.
(defsparc2 sparc2-setq-special setq-special (seg vreg xfer sym val)
  (sparc2-misc-node-set seg 
                      vreg 
                      xfer 
                      (make-acode (%nx1-operator immediate) (sparc2-symbol-value-locative sym) )
                      (make-acode (%nx1-operator fixnum) sparc::symbol.vcell-cell) 
                      val
                      nil))

(defsparc2 sparc2-local-go local-go (seg vreg xfer tag)
  (let* ((curstack (sparc2-encode-stack))
         (label (cadr tag))
         (deststack (caddr tag)))
    (if (not (sparc2-equal-encodings-p curstack deststack))
      (multiple-value-bind (catch cstack vstack)
                           (sparc2-decode-stack deststack)
        (sparc2-unwind-stack seg xfer catch cstack vstack)))
    (-> label)
    (sparc2-unreachable-store vreg)))

(defsparc2 sparc2-local-block local-block (seg vreg xfer blocktag body)
  (let* ((curstack (sparc2-encode-stack))
         (compound (sparc2-cd-compound-p xfer))
         (mvpass-p (sparc2-mvpass-p xfer))
         (need-label (if xfer (or compound mvpass-p) t))
         end-of-block
         last-cd
         (dest (if (backend-crf-p vreg) sparc::%arg_z vreg)))
    (if need-label
      (setq end-of-block (backend-get-next-label)))
    (setq last-cd (if need-label (%ilogior2 (if mvpass-p $backend-mvpass-mask 0) end-of-block) xfer))
    (%rplaca blocktag (cons (cons dest last-cd) curstack))
    (if mvpass-p
      (sparc2-multiple-value-body seg body)
      (sparc2-form seg dest (if xfer last-cd) body))
    (when need-label
      (@ end-of-block)
      (if compound
        (<- dest))
      (sparc2-branch seg (%ilogand (%ilognot $backend-mvpass-mask) (or xfer 0)) vreg))))

(defsparc2 sparc2-%izerop %izerop (seg vreg xfer cc form)
  (multiple-value-bind (cr-bit true-p) (acode-condition-to-sparc-icc cc)
    (sparc2-test-%izerop seg vreg xfer form cr-bit true-p)))


(defsparc2 sparc2-uvsize uvsize (seg vreg xfer v)
  (let* ((misc-reg (sparc2-one-untargeted-reg-form seg v sparc::%arg_z)))
    (unless *sparc2-reckless* (! trap-unless-tag= misc-reg arch::tag-misc))
    (if vreg 
      (ensuring-node-target (target vreg)
        (! misc-element-count[fixnum] target misc-reg)))
    (^)))

(defsparc2 sparc2-%ilsl %ilsl (seg vreg xfer form1 form2)
  (if (null vreg)
    (progn
      (sparc2-form seg nil nil form1)
      (sparc2-form seg nil xfer form2))
    (let* ((const (acode-fixnum-form-p form1)))
      (ensuring-node-target (target vreg)
        (if const
          (let* ((src (sparc2-one-untargeted-reg-form seg form2 sparc::%arg_z)))
            (if (<= const 31)
              (! %ilsl-c target const src)
              (!  lwi target 0)))
          (multiple-value-bind (count src) (sparc2-two-untargeted-reg-forms seg form1 sparc::%arg_y form2 sparc::%arg_z)
            (! %ilsl target count src))))
      (^))))

(defsparc2 sparc2-endp endp (seg vreg xfer cc form)
  (let* ((formreg (sparc2-one-untargeted-reg-form seg form sparc::%arg_z)))
    (! trap-unless-tag= formreg arch::tag-list)
    (multiple-value-bind (cr-bit true-p) (acode-condition-to-sparc-icc cc)
      (sparc2-compare-registers seg vreg xfer formreg sparc::%rnil cr-bit true-p))))


(defsparc2 sparc2-%code-char %code-char (seg vreg xfer c)
  (if (null vreg)
    (sparc2-form seg nil xfer c)
    (progn
      (ensuring-node-target (target vreg)
        (! fixnum->char target (sparc2-one-untargeted-reg-form seg c sparc::%arg_z)))
      (^))))

(defsparc2 sparc2-%schar %schar (seg vreg xfer str idx)
  (multiple-value-bind (src unscaled-idx)
                       (sparc2-two-untargeted-reg-forms seg str sparc::%arg_y idx sparc::%arg_z)
    (if vreg
      (ensuring-node-target (target vreg)
        (! %schar target src unscaled-idx)))
    (^)))

(defsparc2 sparc2-%set-schar %set-schar (seg vreg xfer str idx char)
  (multiple-value-bind (src unscaled-idx char)
                       (sparc2-three-untargeted-reg-forms seg
                                                        str sparc::%arg_x
                                                        idx sparc::%arg_y
                                                        char sparc::%arg_z)
    (! %set-schar  src unscaled-idx char)
    (when vreg (<- char)) 
    (^)))

(defsparc2 sparc2-%set-scharcode %set-scharcode (seg vreg xfer str idx char)
  (multiple-value-bind (src unscaled-idx char)
                       (sparc2-three-untargeted-reg-forms seg str sparc::%arg_x idx sparc::%arg_y
                                                        char sparc::%arg_z)
    (! %set-scharcode  src unscaled-idx char)
    (when vreg (<- char)) 
    (^)))

(defsparc2 sparc2-%scharcode %scharcode (seg vreg xfer str idx)
  (multiple-value-bind (src unscaled-idx)
                       (sparc2-two-untargeted-reg-forms seg str sparc::%arg_y idx sparc::%arg_z)
    (if vreg
      (ensuring-node-target (target vreg)
        (! %scharcode target src unscaled-idx)))
    (^)))

      

(defsparc2 sparc2-code-char code-char (seg vreg xfer c)
  (let* ((reg (sparc2-one-untargeted-reg-form seg c sparc::%arg_z)))
    (! require-u8 reg)                 ; Typecheck even if result unused.
    (if vreg
      (ensuring-node-target (target vreg)
        (! fixnum->char target reg)))
    (^)))

(defsparc2 sparc2-eq eq (seg vreg xfer cc form1 form2)
  (multiple-value-bind (cr-bit true-p) (acode-condition-to-sparc-icc cc)
    (sparc2-compare seg vreg xfer form1 form2 cr-bit true-p)))

(defsparc2 sparc2-neq neq (seg vreg xfer cc form1 form2)
  (multiple-value-bind (cr-bit true-p) (acode-condition-to-sparc-icc cc)
    (sparc2-compare seg vreg xfer form1 form2 cr-bit true-p)))

(defsparc2 sparc2-embedded-non-local-lexit embedded-nlexit (seg vreg xfer form)
  (sparc2-form seg vreg xfer form))

(defsparc2 sparc2-embedded-call embedded-call (seg vreg xfer form)
  (sparc2-form seg vreg xfer form))

(defsparc2 sparc2-%word-to-int %word-to-int (seg vreg xfer form)
  (if (null vreg)
    (sparc2-form seg nil xfer form)
    (progn
      (ensuring-node-target (target vreg)
        (! sign-extend-halfword target (sparc2-one-untargeted-reg-form seg form sparc::%arg_z)))
      (^))))

(defsparc2 sparc2-multiple-value-list multiple-value-list (seg vreg xfer form)
  (sparc2-multiple-value-body seg form)
  (! call-subprim .SPconslist)
  (when vreg
    (<- sparc::%arg_z))
  (^))

(defsparc2 sparc2-immform immediate (seg vreg xfer form)
  (sparc2-immediate seg vreg xfer form))

(defsparc2 sparc2-lexical-reference lexical-reference (seg vreg xfer varnode)
  (let* ((ea-or-form (var-ea varnode)))
    (if (and (acode-punted-var-p varnode) (not (fixnump ea-or-form)))
      (sparc2-form seg vreg xfer ea-or-form)
      (let* ((cell (sparc2-lookup-var-cell varnode)))
        (if (and cell (typep cell 'lcell))
          (if (sparc2-ensure-lcell-offset cell (logand ea-or-form #xffff))
            (and nil (format t "~& could use cell ~s for var ~s" cell (var-name varnode)))
            (if (logbitp sparc2-debug-verbose-bit *sparc2-debug-mask*)
              (break "wrong ea for lcell for var ~s: got ~d, expected ~d" 
                     (var-name varnode) (calc-lcell-offset cell) (logand ea-or-form #xffff))))
          (if (not cell)
            (when (memory-spec-p ea-or-form)
              (if (logbitp sparc2-debug-verbose-bit *sparc2-debug-mask*)
                (format t "~& no lcell for ~s." (var-name varnode))))))
        
        (unless (fixnump ea-or-form)
          (break "bogus ref to var ~s (~s) : ~s " varnode (var-name varnode) ea-or-form))
        (sparc2-do-lexical-reference seg vreg ea-or-form)
        (^)))))

(defsparc2 sparc2-setq-lexical setq-lexical (seg vreg xfer varspec form)
  (let* ((ea (var-ea varspec)))
    (unless (fixnump ea) (break "setq lexical is losing BIG"))
    (let* ((valreg (sparc2-one-untargeted-reg-form seg form (if (and (register-spec-p ea) 
                                                                   (or (null vreg) (eq ea vreg)))
                                                            ea
                                                            sparc::%arg_z))))
      (sparc2-do-lexical-setq seg vreg ea valreg))
    (^)))

(defsparc2 sparc2-fixnum fixnum (seg vreg xfer value)
  (if (null vreg)
    (^)
    (let* ((class (hard-regspec-class vreg))
           (unboxed (if (= class hard-reg-class-gpr)
                      (not (= hard-reg-class-gpr-mode-node (get-regspec-mode vreg))))))
      (if unboxed
        (sparc2-absolute-long seg vreg xfer value)
        (if (= class hard-reg-class-crf)
          (progn
            ;(break "Would have clobbered a GPR!")
            (sparc2-branch seg (sparc2-cd-true xfer) nil))
          (progn
            (ensuring-node-target (target vreg)
              (sparc2-absolute-long seg target nil (ash value sparc::fixnumshift)))
            (^)))))))

(defsparc2 sparc2-%ilogbitp %ilogbitp (seg vreg xfer cc bitnum form)
  (if (null vreg)
    (progn
      (sparc2-form seg nil nil bitnum)
      (sparc2-form seg vreg xfer form))
    (multiple-value-bind (cr-bit true-p) (acode-condition-to-sparc-icc cc)
      (let* ((fixbit (acode-fixnum-form-p bitnum)))
        (if fixbit
          (let* ((reg (sparc2-one-untargeted-reg-form seg form sparc::%arg_z))
                 (ppc-bit (- 31 (max (min (+ fixbit sparc::fixnumshift) 31) sparc::fixnumshift))))
            (with-imm-temps () (bitreg)
              (! extract-constant-ppc-bit bitreg reg ppc-bit)
              (regspec-crf-gpr-case 
               (vreg dest)
               (progn
                 (! compare-signed-s13const bitreg 0)
                 (^ cr-bit true-p))
               (progn
                 (if true-p
                   (! invert-bit31 bitreg))
                 (ensuring-node-target (target dest)
                   (! bit31->truth target bitreg))
                 (^)))))
          (multiple-value-bind (rbit rform) (sparc2-two-untargeted-reg-forms seg bitnum sparc::%arg_y form sparc::%arg_z)
             (with-imm-temps () (bitreg)
               (! extract-variable-non-insane-bit bitreg rform rbit)
               (regspec-crf-gpr-case 
               (vreg dest)
               (progn
                 (! compare-signed-s13const bitreg 0)
                 (^ cr-bit true-p))
               (progn
                 (if true-p
                   (! invert-bit31 bitreg))
                 (ensuring-node-target (target dest)
                   (! bit31->truth target bitreg))
                 (^))))))))))

(defsparc2 sparc2-uvref uvref (seg vreg xfer vector index)
  (sparc2-two-targeted-reg-forms seg vector sparc::%arg_y index sparc::%arg_z)
  (! call-subprim .SPmisc-ref)
  (<- sparc::%arg_z)
  (^))

(defsparc2 sparc2-uvset uvset (seg vreg xfer vector index value)
  (sparc2-three-targeted-reg-forms seg vector sparc::%arg_x index sparc::%arg_y value sparc::%arg_z)
  (! call-subprim .SPmisc-set)
  (<- sparc::%arg_z)
  (^))

(defsparc2 sparc2-%decls-body %decls-body (seg vreg xfer form p2decls)
  (with-sparc-p2-declarations p2decls
    (sparc2-form seg vreg xfer form)))



(defsparc2 sparc2-%err-disp %err-disp (seg vreg xfer arglist)
  (sparc2-set-nargs seg (sparc2-arglist seg arglist))
  (! call-subprim .SPksignalerr)
  (sparc2-nil seg vreg xfer))

(defsparc2 sparc2-newblocktag newblocktag (seg vreg xfer)
  (when vreg
    (! call-subprim .SPnewblocktag)
    (<- sparc::%arg_z))
  (^))

(defsparc2 sparc2-newgotag newgotag (seg vreg xfer)
  (when vreg
    (! call-subprim .SPnewgotag)
    (<- sparc::%arg_z))
  (^))

(defsparc2 sparc2-symbol-name symbol-name (seg vreg xfer sym)
  (let* ((reg (sparc2-one-targeted-reg-form seg sym sparc::%arg_z)))
    (unless *sparc2-reckless*
      (! trap-unless-typecode= reg arch::subtag-symbol))
    (when vreg
      (ensuring-node-target (target vreg)
        (! misc-ref-c-node target reg sparc::symbol.pname-cell)))
    (^)))

(defsparc2 sparc2-local-tagbody local-tagbody (seg vreg xfer taglist body)
  (let* ((encstack (sparc2-encode-stack))
         (tagop (%nx1-operator tag-label)))
    (dolist (tag taglist)
      (rplacd tag (cons (backend-get-next-label) (cons encstack (cadr (cddr (cddr tag)))))))
    ; Check to see if tags referenced by backward branches will
    ; "obviously" do an eventcheck (via function call or otherwise)
    ; after they're branched to.
    ; The annotations produced by pass 1 are conservative; if we want to
    ; try walking the forms, we might do a better job.
    ; This can't work:
    ; A) we don't know for sure that calls will event-poll for us
    ; B) we don't know for sure that things that look like calls
    ;    won't be optimized into something else.
    #+bad-idea
    (let* ((forms body)
           (tag nil)
           (form nil))
      (while forms
        (setq form (pop forms))
        (if (and (eq (acode-operator form) tagop)
                 (cddr (setq tag (cddr form)))
                 (dolist (after forms)
                   (let ((op (acode-operator after)))
                     (when (or (eq op (%nx1-operator embedded-call))
                               ;; Do branches back to our
                               ;; tag branch back to another tag which is branched back to?
                               (and (eq op tagop)
                                    (cddr (cddr after))))
                       ;(format t "~&win one,")
                       (return t))  ;No gratuitous eventcheck here.
                     (when (eq op (%nx1-operator embedded-nlexit))
                       ;; Punt for now. 
                       ;(format t "~&lose one.")
                       (return nil)))))
          (rplacd (cdr tag) nil))))
    (dolist (form body)
      (if (eq (acode-operator form) tagop)
        (let ((tag (cddr form)))
          (@ (car tag))
          (when (cddr tag) (sparc2-emit-event-poll seg)))
        (sparc2-form seg nil nil form)))
    (sparc2-nil seg vreg xfer)))

(defsparc2 sparc2-call call (seg vreg xfer fn arglist &optional spread-p)
  (sparc2-call-fn seg vreg xfer fn arglist spread-p))

(defsparc2 sparc2-self-call self-call (seg vreg xfer arglist &optional spread-p)
  (setq arglist (sparc2-augment-arglist *sparc2-cur-afunc* arglist (if spread-p 1 $numsparcargregs)))
  (sparc2-call-fn seg vreg xfer -1 arglist spread-p))


(defsparc2 sparc2-lexical-function-call lexical-function-call (seg vreg xfer afunc arglist &optional spread-p)
  (sparc2-call-fn seg vreg xfer (list (%nx1-operator simple-function) afunc)
                (sparc2-augment-arglist afunc arglist (if spread-p 1 $numsparcargregs))
                spread-p))

(defsparc2 sparc2-builtin-call builtin-call (seg vreg xfer index arglist)
  (let* ((nargs (sparc2-arglist seg arglist))
         (tail-p (and (sparc2-tailcallok xfer) (<= nargs $numsparcargregs)))
         (idx (acode-fixnum-form-p index))
         (idx-subprim (sparc2-builtin-index-subprim idx))
         (subprim
          (or idx-subprim
              (case nargs
                (0 .SPcallbuiltin0)
                (1 .SPcallbuiltin1)
                (2 .SPcallbuiltin2)
                (3 .SPcallbuiltin3)
                (t .SPcallbuiltin)))))
    (when tail-p
      (sparc2-restore-nvrs seg *sparc2-register-restore-ea* *sparc2-register-restore-count*)
      (sparc2-restore-full-lisp-context seg))
    (unless idx-subprim
      (! lwi sparc::%imm0 (ash idx sparc::fixnumshift))
      (when (eql subprim .SPcallbuiltin)
        (sparc2-set-nargs seg nargs)))
    (if tail-p
      (! jump-subprim subprim)
      (progn
        (! call-subprim subprim)
        (<- sparc::%arg_z)
        (^)))))
      

(defsparc2 sparc2-if if (seg vreg xfer testform true false)
  (if (nx-constant-form-p (acode-unwrapped-form testform))
    (sparc2-form seg vreg xfer (if (nx-null (acode-unwrapped-form testform)) false true))
    (let* ((cstack *sparc2-cstack*)
           (vstack *sparc2-vstack*)
           (top-lcell *sparc2-top-vstack-lcell*)
           (entry-stack (sparc2-encode-stack))
           (true-stack nil)
           (false-stack nil)
           (true-cleanup-label nil)
           (same-stack-effects nil)
           (true-is-goto (sparc2-go-label true))
           (false-is-goto (and (not true-is-goto) (sparc2-go-label false)))
           (endlabel (backend-get-next-label))
           (falselabel (backend-get-next-label))
           (need-else (unless false-is-goto (or (not (nx-null false)) (sparc2-for-value-p vreg))))
           (both-single-valued (and (not *sparc2-open-code-inline*)
                                    (eq xfer $backend-return)
                                    (sparc2-for-value-p vreg)
                                    need-else
                                    (sparc2-single-valued-form-p true) 
                                    (sparc2-single-valued-form-p false))))
      (if (eq 0 xfer) 
        (setq xfer nil))
      (if both-single-valued            ; it's implied that we're returning
        (progn
          (let ((merge-else-branch-label (if (nx-null false) (sparc2-find-nilret-label))))
            (sparc2-conditional-form seg (sparc2-make-compound-cd 0 falselabel) testform)
            (sparc2-form seg sparc::%arg_z endlabel true)
            (if (and merge-else-branch-label (neq -1 (aref *backend-labels* merge-else-branch-label)))
              (backend-copy-label merge-else-branch-label falselabel)
              (progn
                (@ falselabel)
                (if (nx-null false) (@ (sparc2-record-nilret-label)))
                (sparc2-form seg sparc::%arg_z nil false)))
            (@ endlabel)
            (<- sparc::%arg_z)
            (^)))
        (progn
          (if (and need-else (sparc2-mvpass-p xfer))
            (setq true-cleanup-label (backend-get-next-label)))
          (sparc2-conditional-form 
           seg
           (sparc2-make-compound-cd 
            (or true-is-goto 0)
            (or false-is-goto 
                (if need-else 
                  (if true-is-goto 0 falselabel) 
                  (if true-is-goto xfer (sparc2-cd-merge xfer falselabel))))) 
           testform)  
          (if true-is-goto
            (sparc2-unreachable-store)
            (if true-cleanup-label
              (progn
                (sparc2-open-undo $undomvexpect)
                (sparc2-form seg vreg (%ilogior2 $backend-mvpass-mask true-cleanup-label) true))
              (sparc2-form seg vreg (if need-else (sparc2-cd-merge xfer endlabel) xfer) true)))
          (setq true-stack (sparc2-encode-stack))
          (setq *sparc2-cstack* cstack)
          (sparc2-set-vstack vstack)
          (setq *sparc2-top-vstack-lcell* top-lcell)
          (if false-is-goto (sparc2-unreachable-store))
          (let ((merge-else-branch-label (if (and (nx-null false) (eq xfer $backend-return)) (sparc2-find-nilret-label))))
            (if (and merge-else-branch-label (neq -1 (aref *backend-labels* merge-else-branch-label)))
              (backend-copy-label merge-else-branch-label falselabel)
              (progn
                (@ falselabel)
                (when need-else
                  (if true-cleanup-label
                    (sparc2-mvpass seg false)
                    (sparc2-form seg vreg xfer false))
                  (setq false-stack (sparc2-encode-stack))))))
          (when true-cleanup-label
            (if (setq same-stack-effects (sparc2-equal-encodings-p true-stack false-stack)) ; can share cleanup code
              (@ true-cleanup-label))
            (let* ((*sparc2-returning-values* :pass))
              (sparc2-nlexit seg xfer 1)
              (sparc2-branch seg (if (and xfer (neq xfer $backend-mvpass-mask)) xfer (if (not same-stack-effects) endlabel)) vreg))
            (unless same-stack-effects
              (@ true-cleanup-label)
              (multiple-value-setq (true *sparc2-cstack* *sparc2-vstack* *sparc2-top-vstack-lcell*)
                (sparc2-decode-stack true-stack))
              (let* ((*sparc2-returning-values* :pass))
                (sparc2-nlexit seg xfer 1)
                (^)))
            (sparc2-close-undo)
            (multiple-value-setq (*sparc2-undo-count* *sparc2-cstack* *sparc2-vstack* *sparc2-top-vstack-lcell*) 
              (sparc2-decode-stack entry-stack)))
          (@ endlabel))))))

(defsparc2 sparc2-or or (seg vreg xfer forms)
  (let* ((mvpass (sparc2-mvpass-p xfer))
         (tag1 (backend-get-next-label))
         (tag2 (backend-get-next-label))
         (vstack *sparc2-vstack*)
         (cstack *sparc2-cstack*)
         (dest (if (backend-crf-p vreg) vreg (if vreg sparc::%arg_z (available-crf-temp *available-backend-crf-temps*))))
         (cd1 (sparc2-make-compound-cd 
               (if (eq dest sparc::%arg_z) tag1 (sparc2-cd-merge (sparc2-cd-true xfer) tag1)) 0)))
    (while (cdr forms)
      (sparc2-form seg dest (if (eq dest sparc::%arg_z) nil cd1) (car forms))
      (when (eq dest sparc::%arg_z)
        (with-crf-target () val-crf
          (sparc2-copy-register seg val-crf dest)
          (sparc2-branch seg cd1 val-crf)))
      (setq forms (%cdr forms)))
    (if mvpass
      (progn (sparc2-multiple-value-body seg (car forms)) 
             (let* ((*sparc2-returning-values* t)) (sparc2-branch seg (sparc2-cd-merge xfer tag2) vreg)))
      (sparc2-form seg  vreg (if (eq dest sparc::%arg_z) (sparc2-cd-merge xfer tag2) xfer) (car forms)))
    (setq *sparc2-vstack* vstack *sparc2-cstack* cstack)
    (@ tag1)
    (when (eq dest sparc::%arg_z)
      (<- sparc::%arg_z)
      (^))
    (@ tag2)))

(defsparc2 sparc2-simple-function simple-function (seg vreg xfer afunc)
  (sparc2-immediate seg vreg xfer (sparc2-afunc-lfun-ref afunc)))

(defsparc2 sparc2-list list (seg vreg xfer arglist)
  (if (null vreg)
    (dolist (form arglist)
      (sparc2-form seg vreg nil form)) 
    (let* ((*sparc2-vstack* *sparc2-vstack*)
           (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*)
           (nargs (sparc2-formlist seg arglist nil)))
      (sparc2-set-nargs seg nargs)
      (! call-subprim .SPconslist)
      (<- sparc::%arg_z)))
  (^))

(defsparc2 sparc2-list* list* (seg vreg xfer arglist)
  (if (null vreg)
    (dolist (arg (apply #'append arglist))
      (sparc2-form seg nil nil arg))
    (let* ((*sparc2-vstack* *sparc2-vstack*)
           (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*)
           (nargs (sparc2-arglist seg arglist)))
      (declare (fixnum args))
      (when (> nargs 1)
        (sparc2-set-nargs seg (1- nargs))
        (! call-subprim .SPconslist-star))
      (<- sparc::%arg_z)))
  (^))

(defsparc2 sparc2-minus1 minus1 (seg vreg xfer form)
  (sparc2-unary-builtin seg vreg xfer '%negate form))

(defsparc2 sparc2-add2 add2 (seg vreg xfer form1 form2)
  (sparc2-binary-builtin seg vreg xfer '+-2 form1 form2))

(defsparc2 sparc2-logbitp logbitp (seg vreg xfer bitnum int)
  (sparc2-binary-builtin seg vreg xfer 'logbitp bitnum int))

(defsparc2 sparc2-logior2 logior2 (seg vreg xfer form1 form2)
  (sparc2-binary-builtin seg vreg xfer 'logior-2 form1 form2))

(defsparc2 sparc2-logxor2 logxor2 (seg vreg xfer form1 form2)
  (sparc2-binary-builtin seg vreg xfer 'logxor-2 form1 form2))

(defsparc2 sparc2-logand2 logand2 (seg vreg xfer form1 form2)
  (sparc2-binary-builtin seg vreg xfer 'logand-2 form1 form2))

(defsparc2 sparc2-%quo2 %quo2 (seg vreg xfer form1 form2)
  (sparc2-binary-builtin seg vreg xfer '/-2 form1 form2))

(defsparc2 sparc2-%aref1 %aref1 (seg vreg xfer v i)
  (sparc2-binary-builtin seg vreg xfer '%aref1 v i))

(defsparc2 sparc2-%aset1 aset1 (seg vreg xfer v i n)
  (sparc2-ternary-builtin seg vreg xfer '%aset1 v i n))

(defsparc2 sparc2-%i+ %i+ (seg vreg xfer form1 form2 &optional overflow)
  (let* ((v1 (acode-fixnum-form-p form1))
	 (v2 (acode-fixnum-form-p form2))
	 (const (let* ((v (or v1 v2)))
		  (and (typep v '(signed-byte #.(- 13 arch::fixnumshift))) v)))
	 (other (and const (if (eq const v1) form2 form1))))
    (cond ((null vreg) 
         (sparc2-form seg nil nil form1) 
         (sparc2-form seg nil nil form2))
	((and v1 v2 (typep (+ v1 v2) 'fixnum))
	 (sparc2-lwi seg vreg (ash (+ v1 v2) sparc::fixnumshift)))	
	(const
	 (let* ((src (sparc2-one-untargeted-reg-form seg other sparc::%arg_z)))
	   (ensuring-node-target (target vreg)
             (if overflow
	       (! fixnum-add-constant-overflow target src (ash const arch::fixnumshift))
	       (! fixnum-add-constant target src (ash const arch::fixnumshift))))))
	(t
	 (multiple-value-bind (x y) (sparc2-two-untargeted-reg-forms seg form1 sparc::%arg_y form2 sparc::%arg_z)
	   (ensuring-node-target (target vreg)
	    (if overflow
	      (! fixnum-add-overflow target x y)
	      (! fixnum-add target x y))))))
    (^)))


(defsparc2 sparc2-%i- %i- (seg vreg xfer num1 num2 &optional overflow)
  (let* ((v1 (acode-fixnum-form-p num1))
	 (v2 (acode-fixnum-form-p num2)))
    (cond ((null vreg)
	   (sparc2-form seg nil nil num1)
	   (sparc2-form seg nil xfer num2))
	  ((and v1 v2)
	   (sparc2-use-operator (%nx1-operator fixnum) seg vreg xfer (%i- v1 v2)))
	  ((and v2 (typep v2 '(signed-byte #.(- 13 arch::fixnumshift))))
	   (let* ((src (sparc2-one-untargeted-reg-form seg num1 sparc::%arg_z)))
	     (ensuring-node-target (target vreg)
              (if overflow
	       (! fixnum-sub-constant-overflow target src (ash v2 arch::fixnumshift))
	       (! fixnum-sub-constant target src (ash v2 arch::fixnumshift)))
	      (^))))
	  (t
	   (multiple-value-bind (x y) (sparc2-two-untargeted-reg-forms seg num1 sparc::%arg_y num2 sparc::%arg_z)
	     (ensuring-node-target (target vreg)
	      (if overflow
	       (! fixnum-sub-overflow target x y)
	       (! fixnum-sub target x y)))
	     (^))))))

(defsparc2 sparc2-%i* %i* (seg vreg xfer num1 num2)
  (if (null vreg)
    (progn
      (sparc2-form seg nil nil num1)
      (sparc2-form seg nil xfer num2))  
    (let* ((fix1 (acode-fixnum-form-p num1))
           (fix2 (acode-fixnum-form-p num2))
           (other (if (typep fix1 '(signed-byte 13)) num2 (if (typep fix2 '(signed-byte 13)) num1))))
      (if (and fix1 fix2)
        (sparc2-lwi seg vreg (ash (* fix1 fix2) sparc::fixnumshift))
        (if other
          (! multiply-immediate vreg (sparc2-one-untargeted-reg-form seg other sparc::%arg_z) (or fix1 fix2))
          (multiple-value-bind (rx ry) (sparc2-two-untargeted-reg-forms seg num1 sparc::%arg_y num2 sparc::%arg_z)
            (ensuring-node-target (target vreg)
              (! multiply-fixnums target rx ry)))))
      (^))))

(defsparc2 sparc2-nth-value nth-value (seg vreg xfer n form)
  (let* ((*sparc2-vstack* *sparc2-vstack*)
         (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*))
    (let* ((nreg (sparc2-one-untargeted-reg-form seg n sparc::%arg_z)))
      (unless (acode-fixnum-form-p n)
        (! trap-unless-tag= nreg arch::tag-fixnum))
      (sparc2-vpush-register seg nreg))
     (sparc2-multiple-value-body seg form) ; sets nargs
    (! call-subprim .SPnthvalue))
  (<- sparc::%arg_z)
  (^))

(defsparc2 sparc2-values values (seg vreg xfer forms)
  (if (eq (list-length forms) 1)
    (if (sparc2-cd-compound-p xfer)
      (sparc2-form seg vreg xfer (%car forms))
      (progn
        (sparc2-form seg vreg nil (%car forms))
        (^)))
    (if (not (sparc2-mv-p xfer))
      (if forms
        (sparc2-use-operator (%nx1-operator prog1) seg vreg xfer forms)
        (sparc2-nil seg vreg xfer))
      (progn
        (let* ((*sparc2-vstack* *sparc2-vstack*)
               (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*))
          (sparc2-set-nargs seg (sparc2-formlist seg forms nil)))
        (let* ((*sparc2-returning-values* t))
          (^))))))

(defsparc2 sparc2-base-char-p base-char-p (seg vreg xfer cc form)
  (sparc2-char-p seg vreg xfer cc form))

(defun sparc2-char-p (seg vreg xfer cc form)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (multiple-value-bind (cr-bit true-p) (acode-condition-to-sparc-icc cc)
      (! mask-base-char sparc::%imm0 (sparc2-one-untargeted-reg-form seg form sparc::%arg_z))
      (sparc2-test-reg-%izerop seg vreg xfer sparc::%imm0 cr-bit true-p arch::subtag-character))))


(defsparc2 sparc2-let* let* (seg vreg xfer vars vals body p2decls &aux
                             (old-stack (sparc2-encode-stack)))
  (sparc2-check-lcell-depth)
  (with-sparc-p2-declarations p2decls
    (sparc2-seq-bind seg vars vals)
    (sparc2-undo-body seg vreg xfer body old-stack))
  (dolist (v vars) (sparc2-close-var seg v)))

(defsparc2 sparc2-multiple-value-bind multiple-value-bind (seg vreg xfer vars valform body p2decls)
  (let* ((n (list-length vars))
         (vloc *sparc2-vstack*)
         (nbytes (%ilsl 2 n))
         (old-stack (sparc2-encode-stack)))
    (with-sparc-p2-declarations p2decls
      (sparc2-multiple-value-body seg valform)
      (sparc2-lwi seg sparc::%imm0 nbytes)
      (! call-subprim .SPfitvals)
      (sparc2-set-vstack (%i+ vloc nbytes))
      (let* ((old-top *sparc2-top-vstack-lcell*)
             (lcells (progn (sparc2-reserve-vstack-lcells n) (sparc2-collect-lcells :reserved old-top))))
        (dolist (var vars)
          (sparc2-bind-var seg var vloc (pop lcells))
          (setq vloc (%i+ vloc 4))))
      (sparc2-undo-body seg vreg xfer body old-stack)
      (dolist (var vars)
        (sparc2-close-var seg var)))))

(defsparc2 sparc2-debind debind (seg vreg xfer lambda-list bindform req opt rest keys auxen whole body p2decls cdr-p)
  (declare (ignore lambda-list))
  (let* ((old-stack (sparc2-encode-stack))
         (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*)
         (vloc *sparc2-vstack*))
    (with-sparc-p2-declarations p2decls      
      (sparc2-bind-structured-lambda seg 
                                   (sparc2-spread-lambda-list seg bindform whole req opt rest keys nil cdr-p)
                                   vloc (sparc2-vloc-ea vloc) whole req opt rest keys auxen)
      (sparc2-undo-body seg vreg xfer body old-stack)
      (sparc2-close-structured-lambda seg whole req opt rest keys auxen))))

(defsparc2 sparc2-multiple-value-prog1 multiple-value-prog1 (seg vreg xfer forms)
  (if (or (not (sparc2-mv-p xfer)) (sparc2-single-valued-form-p (%car forms)))
    (sparc2-use-operator (%nx1-operator prog1) seg vreg xfer forms)
    (progn
      (let* ((*sparc2-vstack* *sparc2-vstack*)
             (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*))
        (sparc2-multiple-value-body seg (%car forms))
        (sparc2-open-undo $undostkblk)
        (! call-subprim .SPsave-values))
      (dolist (form (cdr forms))
        (sparc2-form seg nil nil form))
      (sparc2-set-nargs seg 0)
      (! call-subprim .SPrecover-values)
      (sparc2-close-undo)
      (let* ((*sparc2-returning-values* t))
        (^)))))

(defsparc2 sparc2-not not (seg vreg xfer cc form)
 (multiple-value-bind (icc true-p) (acode-condition-to-sparc-icc cc)
   (sparc2-compare-registers
    seg
    vreg 
    xfer
    (sparc2-one-untargeted-reg-form seg form sparc::%arg_z) 
    sparc::%rnil 
    icc
    true-p)))


(defsparc2 sparc2-%alloc-misc %make-uvector (seg vreg xfer element-count st &optional initval)
  (if (null vreg)
    (progn
      (sparc2-form seg nil nil element-count)
      (sparc2-form seg nil xfer st))
    (let* ((subtag (acode-fixnum-form-p st))
           (nelements (acode-fixnum-form-p element-count))
           (nbytes (if (and subtag nelements) (sparc2-misc-byte-count subtag nelements))))
      (if (and nbytes (null initval) (< (logand (lognot 7) (+ nbytes 4 7)) #x8000))
        (with-imm-temps () (header)
          (sparc2-lwi seg header (sparc::make-vheader nelements subtag))
          (ensuring-node-target (target vreg)
            (! %alloc-misc-fixed target header nbytes)))
        (progn
          (if initval
            (progn
              (sparc2-three-targeted-reg-forms seg element-count sparc::%arg_x st sparc::%arg_y initval sparc::%arg_z)
              (! call-subprim .SPmisc-alloc-init)
              (<- sparc::%arg_z))
            (progn
              (sparc2-two-targeted-reg-forms seg element-count sparc::%arg_y st sparc::%arg_z)
              (! call-subprim .SPmisc-alloc)
              (<- sparc::%arg_z)))))
      (^))))

(defsparc2 sparc2-%iasr %iasr (seg vreg xfer form1 form2)
  (if (null vreg)
    (progn
      (sparc2-form seg nil nil form1)
      (sparc2-form seg vreg xfer form2))
    (let* ((count (acode-fixnum-form-p form1)))
      (ensuring-node-target (target vreg)
        (if count
          (! %iasr-c target (if (> count 31) 31 count)
             (sparc2-one-untargeted-reg-form seg form2 sparc::%arg_z))
          (multiple-value-bind (cnt src) (sparc2-two-targeted-reg-forms seg form1 sparc::%arg_y form2 sparc::%arg_z)
            (! %iasr target cnt src))))
      (^))))

(defsparc2 sparc2-%ilsr %ilsr (seg vreg xfer form1 form2)
  (if (null vreg)
    (progn
      (sparc2-form seg nil nil form1)
      (sparc2-form seg vreg xfer form2))
    (let* ((count (acode-fixnum-form-p form1)))
      (ensuring-node-target (target vreg)
        (if count
          (let ((src (sparc2-one-untargeted-reg-form seg form2 sparc::%arg_z)))
            (if (<= count 31)
              (! %ilsr-c target count src)
              (!  lwi target 0)))
          (multiple-value-bind (cnt src) (sparc2-two-targeted-reg-forms seg form1 sparc::%arg_y form2 sparc::%arg_z)
            (! %ilsr target cnt src))))
      (^))))


(defsparc2 sparc2-%i<> %i<> (seg vreg xfer cc form1 form2)
  (multiple-value-bind (cr-bit true-p) (acode-condition-to-sparc-icc cc)
    (sparc2-compare seg vreg xfer form1 form2 cr-bit true-p)))

(defsparc2 sparc2-double-float-compare double-float-compare (seg vreg xfer cc form1 form2)
   (multiple-value-bind (cr-bit true-p) (acode-condition-to-sparc-fcc cc)
    (with-fp-target () (r1 :double-float)
      (with-fp-target (r1) (r2 :double-float)
        (multiple-value-bind (r1 r2) (sparc2-two-untargeted-reg-forms seg form1 r1 form2 r2)
          (sparc2-compare-double-float-registers seg vreg xfer r1 r2 cr-bit true-p))))))

(defsparc2 sparc2-short-float-compare short-float-compare (seg vreg xfer cc form1 form2)
  (multiple-value-bind (cr-bit true-p) (acode-condition-to-sparc-fcc cc)
    (with-fp-target () (r1 :single-float)
      (with-fp-target (r1) (r2 :single-float)
        (multiple-value-bind (r1 r2) (sparc2-two-untargeted-reg-forms seg form1 r1 form2 r2)
          (sparc2-compare-single-float-registers seg vreg xfer r1 r2 cr-bit true-p))))))
 
(eval-when (:compile-toplevel :execute)
  (defmacro defsparc2-df-op (fname opname vinsn)
    `(defsparc2 ,fname ,opname (seg vreg xfer f0 f1)
       (if (null vreg)
         (progn
           (sparc2-form seg nil nil f0)
           (sparc2-form seg vreg xfer f1))
         (with-fp-target () (r1 :double-float)
           (with-fp-target (r1) (r2 :double-float)
             (multiple-value-bind (r1 r2) (sparc2-two-untargeted-reg-forms seg f0 r1 f1 r2)
               (! CLEAR-FPU-EXCEPTIONS sparc::%fp-zero)
               (if (= (hard-regspec-class vreg) hard-reg-class-fpr)
                 (let* ((vreg-val (hard-regspec-value vreg)))
                   (declare (fixnum vreg-val))
                   (if (or (= vreg-val (hard-regspec-value r1))
                           (= vreg-val (hard-regspec-value r2)))
                     (with-fp-target (r1 r2) (result :double-float)
                       (! ,vinsn result r1 r2)
                       (<- result))
                     (! ,vinsn vreg r1 r2)))
                 (with-fp-target (r1 r2) (result :double-float)
                   (! ,vinsn result r1 r2)
                   (ensuring-node-target (target vreg)
                     (<- result))))
               (^)))))))
  
  (defmacro defsparc2-sf-op (fname opname vinsn)
    `(defsparc2 ,fname ,opname (seg vreg xfer f0 f1)
       (if (null vreg)
         (progn
           (sparc2-form seg nil nil f0)
           (sparc2-form seg vreg xfer f1))
         (with-fp-target () (r1 :single-float)
           (with-fp-target (r1) (r2 :single-float)
             (multiple-value-bind (r1 r2) (sparc2-two-untargeted-reg-forms seg f0 r1 f1 r2)
               (! CLEAR-FPU-EXCEPTIONS sparc::%fp-zero)
               (if (= (hard-regspec-class vreg) hard-reg-class-fpr)
                 (let* ((vreg-val (hard-regspec-value vreg)))
                   (declare (fixnum vreg-val))
                   (if (or (= vreg-val (hard-regspec-value r1))
                           (= vreg-val (hard-regspec-value r2)))
                     (with-fp-target (r1 r2) (result :single-float)
                       (! ,vinsn result r1 r2)
                       (<- result))
                     (! ,vinsn vreg r1 r2)))
                 (with-fp-target (r1 r2) (result :single-float)
                   (! ,vinsn result r1 r2)
                   (ensuring-node-target (target vreg)
                     (<- result))))
               (^)))))))
)

(defsparc2-df-op sparc2-%double-float+-2 %double-float+-2 double-float+-2)
(defsparc2-df-op sparc2-%double-float--2 %double-float--2 double-float--2)
(defsparc2-df-op sparc2-%double-float*-2 %double-float*-2 double-float*-2)
(defsparc2-df-op sparc2-%double-float/-2 %double-float/-2 double-float/-2)

(defsparc2-sf-op sparc2-%short-float+-2 %short-float+-2 single-float+-2)
(defsparc2-sf-op sparc2-%short-float--2 %short-float--2 single-float--2)
(defsparc2-sf-op sparc2-%short-float*-2 %short-float*-2 single-float*-2)
(defsparc2-sf-op sparc2-%short-float/-2 %short-float/-2 single-float/-2)
               


(defsparc2 sparc2-immediate-get-ptr immediate-get-ptr (seg vreg xfer bits ptr offset)
  (let* ((deref (%ilogbitp 4 bits))
         (absptr (unless deref (acode-absolute-ptr-p ptr)))
         (triv-p (sparc2-trivial-p offset))
         (dest vreg)
         (offval (acode-fixnum-form-p offset)))
    (cond ((not vreg)
           (sparc2-form seg nil nil ptr)
           (sparc2-form seg nil xfer offset))
          (t
           (if (and absptr offval) 
             (setq absptr (+ absptr offval) offval 0)
             (setq absptr nil))
           (and offval (%i> (integer-length offval) 15) (setq offval nil))
           (and absptr (%i> (integer-length absptr) 15) (setq absptr nil))
           (if absptr
             (! mem-ref-c-fullword dest sparc::%rzero absptr)
             (if offval
               (let* ((src (sparc2-macptr-arg-to-reg seg ptr sparc::%imm0)))
                 (if deref 
                   (! mem-ref-c-fullword src src 0))
                 (! mem-ref-c-fullword dest src offval))
               (let* ((src (sparc2-macptr-arg-to-reg seg ptr sparc::%imm0)))
                 (if triv-p
                   (with-imm-temps (src) (x)
                     (if (acode-fixnum-form-p offset)
                       (sparc2-lwi seg x (acode-fixnum-form-p offset))
                       (! fixnum->s32 x (sparc2-one-untargeted-reg-form seg offset sparc::%arg_z)))
                     (if deref 
                       (! mem-ref-c-fullword src src 0))
                     (! mem-ref-fullword dest src x))
                   (progn
                     (! temp-push-unboxed-word src)
                     (sparc2-open-undo $undostkblk)
                     (let* ((oreg (sparc2-one-untargeted-reg-form seg offset sparc::%arg_z)))
                       (with-imm-temps () (src x)
                         (! temp-pop-unboxed-word src)
                         (sparc2-close-undo)
                         (if deref 
                           (! mem-ref-c-fullword src src 0))
                         (! fixnum->s32 x oreg)
                         (! mem-ref-fullword dest src x)))))))) 
           (^)))))

; This returns an unboxed object, unless the caller wants to box it.
(defsparc2 sparc2-immediate-get-xxx immediate-get-xxx (seg vreg xfer bits ptr offset)
  (let* ((lowbits (%ilogand2 3 bits))
         (deref (%ilogbitp 4 bits))
         (size 
          (if (eq lowbits 3) 
            1 
            (if (eq lowbits 2) 
              2 
              4)))
         (extend-p (%ilogbitp 2 bits)) ;(setq bits (%ilogand2 #xf bits))))
         (absptr (unless deref (acode-absolute-ptr-p ptr)))
         (triv-p (sparc2-trivial-p offset))
         (offval (acode-fixnum-form-p offset)))
    (declare (fixnum size))
    (cond ((null vreg)
           (sparc2-form seg nil nil ptr)
           (sparc2-form seg nil xfer offset))
          (t 
           (if (and absptr offval) 
             (setq absptr (+ absptr offval) offval 0)
             (setq absptr nil))
           (and offval (%i> (integer-length offval) 15) (setq offval nil))
           (and absptr (%i> (integer-length absptr) 15) (setq absptr nil))
           (with-imm-target () (dest :u32)
             (if absptr
               (if (eq size 4)
                 (! mem-ref-c-fullword dest sparc::%rzero absptr)
                 (if (eq size 2)
                   (! mem-ref-c-u16 dest sparc::%rzero absptr)
                   (! mem-ref-c-u8 dest sparc::%rzero absptr)))
               (if offval
                 (with-imm-target () (src-reg :address)
                   (sparc2-one-targeted-reg-form seg ptr src-reg)
                   (if deref
                     (! mem-ref-c-fullword src-reg src-reg 0))
                   (if (eq size 4)
                     (! mem-ref-c-fullword dest src-reg offval)
                     (if (eq size 2)
                       (! mem-ref-c-u16 dest src-reg offval)
                       (! mem-ref-c-u8 dest src-reg offval))))
                 (with-imm-target () (src-reg :address)
                   (with-imm-target (src-reg) (offset-reg :s32)                                   
                     (sparc2-one-targeted-reg-form seg ptr src-reg)
                     (if triv-p
                       (if (acode-fixnum-form-p offset)
                         (sparc2-lwi seg offset-reg (acode-fixnum-form-p offset))
                         (! fixnum->s32 offset-reg (sparc2-one-untargeted-reg-form seg offset sparc::%arg_z)))
                       (progn
                         (! temp-push-unboxed-word src-reg)
                         (sparc2-open-undo $undostkblk)
                         (! fixnum->s32 offset-reg (sparc2-one-untargeted-reg-form seg offset sparc::%arg_z))
                         (! temp-pop-unboxed-word src-reg)
                         (sparc2-close-undo)))
                     (if deref
                       (! mem-ref-c-fullword src-reg src-reg 0))
                     (if (eq size 4)
                       (! mem-ref-fullword dest src-reg offset-reg)
                       (if (eq size 2)
                         (! mem-ref-u16 dest src-reg offset-reg)
                         (! mem-ref-u8 dest src-reg offset-reg)))))))
             ; %get-fixnum: if storing to a node vreg, ignore any overflow.
             (if (and (eq size 4) 
                      (%ilogbitp 5 bits)
		      (node-reg-p vreg))
               (! box-fixnum vreg (set-regspec-mode dest  (gpr-mode-name-value :s32)))
               (<- (set-regspec-mode 
                    dest 
                    (gpr-mode-name-value
                     (if (eq size 4)
                       (if (%ilogbitp 5 bits)
                         :s32           ; %get-fixnum to "raw" target.
                         (if (%ilogbitp 3 bits)
                           :u32
                           :s32))
                       (if (eq size 2)
                         (if extend-p
                           :s16
                           :u16)
                         (if extend-p
                           :s8
                           :u8))))))))
           (^)))))

(defsparc2 sparc2-let let (seg vreg xfer vars vals body p2decls)
  (let* ((old-stack (sparc2-encode-stack))
         (val nil)
         (bits nil)
         (valcopy vals))
    (with-sparc-p2-declarations p2decls
      (dolist (var vars)
        (setq val (%car valcopy))
        (cond ((or (%ilogbitp $vbitspecial (setq bits (nx-var-bits var)))
                   (and (%ilogbitp $vbitreg bits)
                        (dolist (val (%cdr valcopy))
                          (unless (sparc2-trivial-p val) (return t)))))
               (let* ((pair (cons (sparc2-vloc-ea *sparc2-vstack*) nil)))
                 (%rplaca valcopy pair)
                 (if (and (%ilogbitp $vbitdynamicextent bits)
                          (progn
                            (setq val 
                                  (sparc2-dynamic-extent-form seg (sparc2-encode-stack) val))
                            (sparc2-load-ea-p val)))
                   (progn
                     (%rplaca pair (sparc2-vloc-ea *sparc2-vstack*))
                     (sparc2-vpush-register seg val :reserved))
                 (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg val sparc::%arg_z) :reserved))
                 (%rplacd pair *sparc2-top-vstack-lcell*)))
              (t (sparc2-seq-bind-var seg var val)
                 (%rplaca valcopy nil)))
        (setq valcopy (%cdr valcopy)))
      (dolist (var vars)
        (declare (list val))
        (when (setq val (pop vals))
          (if (%ilogbitp $vbitspecial (nx-var-bits var))
            (sparc2-dbind seg (car val) (var-name var))
            (sparc2-seq-bind-var seg var (car val)))))
      (sparc2-undo-body seg vreg xfer body old-stack)
      (dolist (var vars)
        (sparc2-close-var seg var)))))

(defsparc2 sparc2-closed-function closed-function (seg vreg xfer afunc)
  (sparc2-make-closure seg afunc nil)
  (when vreg (<- sparc::%arg_z))
  (^))

(defsparc2 sparc2-flet flet (seg vreg xfer vars afuncs body p2decls)
  (sparc2-seq-fbind seg vreg xfer vars afuncs body p2decls))

(defsparc2 sparc2-labels labels (seg vreg xfer vars afuncs body p2decls)
  (let* ((fwd-refs nil)
         (func nil)
         (togo vars)
         (real-vars ())
         (real-funcs ())
         (funs afuncs))
    (dolist (v vars)
      (when (neq 0 (afunc-fn-refcount (setq func (pop funs))))
        (push v real-vars)
        (push func real-funcs)
        (let* ((i 2)
               (our-var nil)
               (item nil))
          (declare (fixnum i))
          (dolist (ref (afunc-inherited-vars func))
            (when (memq (setq our-var (var-bits ref)) togo)
              (setq item (cons i our-var))
              (let* ((refs (assq v fwd-refs)))
                (if refs
                  (push item (cdr refs))
                  (push (list v item) fwd-refs))))
            (incf i)))
        (setq togo (%cdr togo))))       
    (if (null fwd-refs)
      (sparc2-seq-fbind seg vreg xfer (nreverse real-vars) (nreverse real-funcs) body p2decls)
      (let* ((old-stack (sparc2-encode-stack)))
        (setq real-vars (nreverse real-vars) real-funcs (nreverse real-funcs))
        (with-sparc-p2-declarations p2decls
          (dolist (var real-vars)
            (sparc2-seq-bind-var seg var (nx1-afunc-ref (pop real-funcs))))
          (dolist (ref fwd-refs)
            (let ((ea (var-ea (pop ref))))
              (sparc2-addrspec-to-reg seg ea sparc::%temp0)
              (dolist (r ref)
                (let* ((v-ea (var-ea (cdr r))))
                  (let* ((val-reg (if (eq v-ea ea)
                                    sparc::%temp0
                                    (progn
                                      (sparc2-addrspec-to-reg seg v-ea sparc::%temp1)
                                      sparc::%temp1))))
                    ; can't be sure of relative ages of function vectors, so have to memoize.
                    ; the concept of "forward" references implies that the memoization may
                    ; actually be needed.
                    (! misc-set-c-node& val-reg sparc::%temp0 (car r)))))))
          (sparc2-undo-body seg vreg xfer body old-stack)
          (dolist (var real-vars)
            (sparc2-close-var seg var)))))))

; Make a function call (e.g., to mapcar) with some of the toplevel arguments
; stack-consed (downward) closures.  Bind temporaries to these closures so
; that tail-recursion/non-local exits work right.
; (all of the closures are distinct: FLET and LABELS establish dynamic extent themselves.)
(defsparc2 sparc2-with-downward-closures with-downward-closures (seg vreg xfer tempvars closures callform)
  (let* ((old-stack (sparc2-encode-stack)))
    (sparc2-seq-bind seg tempvars closures)
    (sparc2-undo-body seg vreg xfer callform old-stack)
    (dolist (v tempvars) (sparc2-close-var seg v))))


(defsparc2 sparc2-local-return-from local-return-from (seg vreg xfer blocktag value)
  (declare (ignorable vreg xfer))
  (let* ((*sparc2-undo-count* *sparc2-undo-count*)
         (tagdata (car blocktag))
         (cur-stack (sparc2-encode-stack))
         (dest-vd (caar tagdata))
         (dest-cd (cdar tagdata))
         (mv-p (sparc2-mvpass-p dest-cd))
         (dest-stack  (cdr tagdata))
         (need-break (neq cur-stack dest-stack)))
    (let* ((*sparc2-vstack* *sparc2-vstack*)
           (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*)
           (*sparc2-cstack* *sparc2-cstack*))
      (if 
        (or
         (eq dest-cd $backend-return)
         (and mv-p 
              (eq (sparc2-encoding-undo-count cur-stack)
                  (sparc2-encoding-undo-count dest-stack)) 
              (eq (sparc2-encoding-cstack-depth cur-stack)
                  (sparc2-encoding-cstack-depth dest-stack))))
        (sparc2-form seg dest-vd dest-cd value)
        (if mv-p
          (progn
            (sparc2-multiple-value-body seg value)
            (let* ((*sparc2-returning-values* :pass))
              (sparc2-nlexit seg dest-cd (%i- *sparc2-undo-count* (sparc2-encoding-undo-count dest-stack)))
              (sparc2-branch seg dest-cd vreg)))
          (progn
            (sparc2-form 
             seg
             (if need-break (if dest-vd sparc::%arg_z) dest-vd) 
             (if need-break nil dest-cd)
             value)
            (when need-break
              (sparc2-unwind-set seg dest-cd dest-stack)
              (when dest-vd (sparc2-copy-register seg dest-vd sparc::%arg_z))
              (sparc2-branch seg dest-cd dest-vd))))))
    (sparc2-unreachable-store)))

(defsparc2 sparc2-inherited-arg inherited-arg (seg vreg xfer arg)
  (when vreg
    (sparc2-addrspec-to-reg seg (sparc2-ea-open (var-ea arg)) vreg))
  (^))


(defsparc2 sparc2-%lisp-word-ref %lisp-word-ref (seg vreg xfer base offset)
  (let* ((fixoffset (acode-fixnum-form-p offset)))
    (cond ((null vreg)
           (sparc2-form seg nil nil base)
           (sparc2-form seg nil xfer offset))
          ((typep fixoffset '(signed-byte 14))
           (ensuring-node-target (target vreg)
             (! lisp-word-ref-c target 
                (sparc2-one-untargeted-reg-form seg base sparc::%arg_z) 
                (ash fixoffset sparc::word-shift)))
           (^))
          (t (multiple-value-bind (breg oreg)
                                  (sparc2-two-untargeted-reg-forms seg base sparc::%arg_y offset sparc::%arg_z)
               (ensuring-node-target (target vreg)
                 (! lisp-word-ref target breg oreg))
               (^))))))

(defsparc2 sparc2-int>0-p int>0-p (seg vreg xfer cc form)
  (multiple-value-bind (cr-bit true-p) (acode-condition-to-sparc-icc cc)
    (sparc2-one-targeted-reg-form seg form sparc::%arg_z)
    (! call-subprim .SPinteger-sign)
    (sparc2-test-reg-%izerop seg vreg xfer sparc::%imm0 cr-bit true-p 0)))


(defsparc2 sparc2-throw throw (seg vreg xfer tag valform )
  (declare (ignorable vreg xfer))
  (let* ((*sparc2-vstack* *sparc2-vstack*)
         (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*))
    (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg tag sparc::%arg_z))
    (if (sparc2-trivial-p valform)
      (progn
        (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg valform sparc::%arg_z))
        (sparc2-set-nargs seg 1))
      (sparc2-multiple-value-body seg valform))
    (! call-subprim .SPthrow)))

; This (and unwind-protect and things like that) are a little funky in that
; they have no good way of specifying the exit-point.  The bad way is to
; follow the call to the catch-frame-creating subprim with a branch to that
; exit-point; the subprim returns to the following instruction.
; If the compiler ever gets smart about eliminating dead code, it has to
; be careful not to consider the block following the jump to be dead.
; Use a vinsn other than JUMP to reference the label.
(defsparc2 sparc2-catch catch (seg vreg xfer tag valform)
  (let* ((tag-label (backend-get-next-label))
         (mv-pass (sparc2-mv-p xfer)))
    (sparc2-one-targeted-reg-form seg tag sparc::%arg_z)
    (! build-catch-frame (if mv-pass .SPmkcatchmv .SPmkcatch1v) (aref *backend-labels* tag-label))
    (sparc2-open-undo)
    (if mv-pass
      (sparc2-multiple-value-body seg valform)  
      (sparc2-one-targeted-reg-form seg valform sparc::%arg_z))
    (sparc2-lwi seg sparc::%imm0 (ash 1 sparc::fixnumshift))
    (! call-subprim (if mv-pass .SPnthrowvalues .SPnthrow1value))
    (sparc2-close-undo)
    (@ tag-label)
    (unless mv-pass (if vreg (<- sparc::%arg_z)))
    (let* ((*sparc2-returning-values* mv-pass)) ; nlexit keeps values on stack
      (^))))


(defsparc2 sparc2-fixnum-overflow fixnum-overflow (seg vreg xfer form)
  (destructuring-bind (op n0 n1) (acode-unwrapped-form form)
    (sparc2-use-operator op seg vreg xfer n0 n1 *nx-t*)))

(defsparc2 sparc2-%aref2 aref2 (seg vreg xfer subtag arr i j &optional dim0 dim1)
  (if (null vreg)
    (progn
      (sparc2-form seg nil nil arr)
      (sparc2-form seg nil nil i)
      (sparc2-form seg nil xfer j)))
  (let* ((fixtype (acode-fixnum-form-p subtag))
         (safe (unless *sparc2-reckless* fixtype))
         (dim0 (acode-fixnum-form-p dim0))
         (dim1 (acode-fixnum-form-p dim1)))
    (case fixtype
      (#.arch::subtag-double-float-vector
       (if (= (hard-regspec-class vreg) hard-reg-class-fpr)
           (sparc2-df-aref2 seg vreg xfer arr i j safe dim0 dim1)
           (with-fp-target () (target :double-float)
             (sparc2-df-aref2 seg target nil arr i j safe dim0 dim1)
             (<- target)
             (^))))
      (t (error "Bug: shouldn't have tried to open-code %AREF2 call.")))))

(defsparc2 sparc2-%aset2 aset2 (seg vreg xfer subtag arr i j new &optional dim0 dim1)
  (if (null vreg)
    (progn
      (sparc2-form seg nil nil arr)
      (sparc2-form seg nil nil i)
      (sparc2-form seg nil nil j)
      (sparc2-form seg nil xfer new)))
  (let* ((fixtype (acode-fixnum-form-p subtag))
         (safe (unless *sparc2-reckless* fixtype))
         (dim0 (acode-fixnum-form-p dim0))
         (dim1 (acode-fixnum-form-p dim1)))
    (cond ((and (eq fixtype arch::subtag-double-float-vector)
                (or (null vreg)
                    (= (hard-regspec-class vreg) hard-reg-class-fpr)))
           (sparc2-df-aset2 seg vreg xfer arr i j new safe dim0 dim1))
          (t
           (sparc2-form seg vreg xfer
                      (make-acode (%nx1-operator call)
                                  (make-acode (%nx1-operator immediate) '%aset2)
                                  (list (list arr)
                                        (list new j i))))))))

(defsparc2 sparc2-%typed-uvref %typed-uvref (seg vreg xfer subtag uvector index)
  (let* ((fixtype (acode-fixnum-form-p subtag)))
    (if fixtype
      (sparc2-vref seg vreg xfer fixtype uvector index (unless *sparc2-reckless* fixtype))
      (progn
        (sparc2-three-targeted-reg-forms seg subtag sparc::%arg_x uvector sparc::%arg_y index sparc::%arg_z)
        (! call-subprim .SPsubtag-misc-ref)
        (when vreg (<- sparc::%arg_z))
        (^)))))

(defsparc2 sparc2-%typed-uvset %typed-uvset (seg vreg xfer subtag uvector index newval)
  (let* ((fixtype (acode-fixnum-form-p subtag)))
    (if fixtype
      (sparc2-vset seg vreg xfer fixtype uvector index newval (unless *sparc2-reckless* fixtype))
      (progn                            ; Could always do a four-targeted-reg-forms ...
        (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg subtag sparc::%arg_z))
        (sparc2-three-targeted-reg-forms seg uvector sparc::%arg_x index sparc::%arg_y newval sparc::%arg_z)
        (sparc2-vpop-register seg sparc::%temp0)
        (! call-subprim .SPsubtag-misc-set)
        (when vreg (<- sparc::%arg_z))
        (^)))))

(defsparc2 sparc2-%macptrptr% %macptrptr% (seg vreg xfer form)
  (with-imm-target () (target :address)
    (sparc2-one-targeted-reg-form seg form (or vreg target)))
  (^))
           

; cons a macptr, unless "vreg" is an immediate register of mode :address.
(defsparc2 sparc2-%consmacptr% %consmacptr% (seg vreg xfer form)
  (cond ((null vreg) (sparc2-form seg nil xfer form))
        ((eql (get-regspec-mode vreg) hard-reg-class-gpr-mode-address)
         (sparc2-form seg vreg xfer form))
        (t         
         (with-imm-target () (temp :address)
           (<- (sparc2-one-targeted-reg-form seg form temp))
           (^)))))

(defsparc2 sparc2-%immediate-ptr-to-int %immediate-ptr-to-int (seg vreg xfer form)
  (if (null vreg)
    (sparc2-form seg nil xfer form)
    (with-imm-target () (address-reg :address)
      (sparc2-form seg address-reg nil form)
      (<- (set-regspec-mode address-reg (gpr-mode-name-value :u32)))
      (^))))

(defsparc2 sparc2-%immediate-int-to-ptr %immediate-int-to-ptr (seg vreg xfer form)
  (if (null vreg)
    (sparc2-form seg nil xfer form)
    (progn
      (unless (logbitp (hard-regspec-value vreg) sparc-imm-regs)
        (error "I give up.  When will I get this right ?"))
      (let* ((u32-reg (sparc2-one-targeted-reg-form seg 
                                                    form
                                                    (make-wired-lreg vreg :mode hard-reg-class-gpr-mode-u32))))
        (<- u32-reg)
        (^)))))


(defsparc2 sparc2-%function %function (seg vreg xfer sym)
  (when vreg
    (let* ((symreg (sparc2-one-untargeted-reg-form seg (make-acode (%nx1-operator immediate)
                                                                 (sparc2-symbol-entry-locative sym)) sparc::%arg_z)))
      (with-node-temps (vreg symreg) (val)
        (! symbol-function val symreg)
        (<- val))))
  (^))

(defsparc2 sparc2-%unbound-marker %unbound-marker (seg vreg xfer)
  (when vreg       
    (ensuring-node-target (target vreg)
      (sparc2-lwi seg target sparc::unbound-marker)))
  (^))

(defsparc2 sparc2-%illegal-marker %illegal-marker (seg vreg xfer)
  (when vreg    
    (ensuring-node-target (target vreg)
      (sparc2-lwi seg target sparc::illegal-marker)))
  (^))

(defsparc2 sparc2-lambda-bind lambda-bind (seg vreg xfer vals req rest keys-p auxen body p2decls)
  (let* ((old-stack (sparc2-encode-stack))
         (nreq (list-length req))
         (rest-arg (nthcdr nreq vals))
         (apply-body (sparc2-eliminate-&rest body rest keys-p auxen rest-arg)))
    (sparc2-seq-bind seg req vals)
    (when apply-body (setq rest nil body apply-body))
    (let*
      ((vloc *sparc2-vstack*)
       (restloc vloc)
       (nvloc (progn (if (or rest keys-p) (sparc2-formlist seg rest-arg)) *sparc2-vstack*)))
      (with-sparc-p2-declarations p2decls
        (when rest
          (when keys-p
            (until (eq restloc nvloc)
              (with-node-temps () (temp)
                (sparc2-stack-to-register seg (sparc2-vloc-ea restloc) temp)
                (sparc2-vpush-register seg temp))
              (setq restloc (%i+ restloc 4))))
          (sparc2-set-nargs seg (length rest-arg))
          (if (%ilogbitp $vbitdynamicextent (nx-var-bits rest))
            (progn
              (! call-subprim .SPstkconslist)
              (sparc2-open-undo $undostkblk))
            (! call-subprim .SPconslist))
          (sparc2-vpush-register seg sparc::%arg_z)
          (sparc2-set-vstack (%i+ restloc 4)))
        (when rest (sparc2-bind-var seg rest restloc))
        (destructuring-bind (vars inits) auxen
          (while vars
            (let ((val (%car inits))) 
              (if (fixnump val)
                (progn
                  (when rest (setq val (%i+ (%i+ val val) 1)))
                  (sparc2-bind-var seg (%car vars) (%i+ vloc (%ilsl 2 val))))
                (sparc2-seq-bind-var seg (%car vars) val)))
            (setq vars (%cdr vars) inits (%cdr inits))))
        (sparc2-undo-body seg vreg xfer body old-stack)
        (dolist (var req) (sparc2-close-var seg var))
        (when rest (sparc2-close-var seg rest))
        (dolist (var (%car auxen)) (sparc2-close-var seg var))))))

(macrolet 
  ((def-sparc2-require (function op &optional (vinsn op))
     `(defsparc2 ,function ,op (seg vreg xfer val)
        (let* ((val-reg (sparc2-one-untargeted-reg-form 
                         seg 
                         val 
                         (if (eq vreg sparc::%arg_z) sparc::%arg_y sparc::%arg_z))))
          (! ,vinsn val-reg)
          (when vreg (<- val-reg))
          (^)))))
  (def-sparc2-require sparc2-require-simple-vector require-simple-vector)
  (def-sparc2-require sparc2-require-simple-string require-simple-string)
  (def-sparc2-require sparc2-require-integer require-integer)
  (def-sparc2-require sparc2-require-fixnum require-fixnum)
  (def-sparc2-require sparc2-require-real require-real)
  (def-sparc2-require sparc2-require-list require-list)
  (def-sparc2-require sparc2-require-character require-character)
  (def-sparc2-require sparc2-require-number require-number)
  (def-sparc2-require sparc2-require-symbol require-symbol))

(defsparc2 sparc2-%badarg2 %badarg2 (seg vreg xfer badthing goodthing)
  (sparc2-two-targeted-reg-forms seg badthing sparc::%arg_y goodthing sparc::%arg_z)
  (sparc2-lwi seg sparc::%arg_x (ash $XWRONGTYPE sparc::fixnumshift))
  (sparc2-set-nargs seg 3)
  (! call-subprim .SPksignalerr)
  (<- sparc::%rnil)
  (^))  
          
(defsparc2 sparc2-%set-sbchar %set-sbchar (seg vreg xfer string index value)
  (sparc2-vset 
   seg 
   vreg 
   xfer 
   arch::subtag-simple-base-string 
   string 
   index
   value 
   (unless *sparc2-reckless* arch::subtag-simple-base-string)))


; If we didn't use this for stack consing, turn it into a call.  Ugh.

(defsparc2 sparc2-make-list make-list (seg vreg xfer size initial-element)
  (sparc2-form seg vreg xfer (make-acode (%nx1-operator call)
                                       (make-acode (%nx1-operator immediate) 'make-list)
                                       (list nil
                                             (list initial-element 
                                                   (make-acode (%nx1-operator immediate)
                                                               :initial-element)
                                                   size)))))


(defsparc2 sparc2-setq-free setq-free (seg vreg xfer sym val)
  (sparc2-one-targeted-reg-form seg val sparc::%arg_z)
  (sparc2-immediate seg sparc::%arg_y nil (sparc2-symbol-value-locative sym))
  (! call-subprim .SPsetqsym)
  (<- sparc::%arg_z)
  (^))

(defsparc2 sparc2-%setf-macptr %setf-macptr (seg vreg xfer x y)
  (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg x sparc::%arg_z))
  (with-imm-target () (src-reg :address)
    (sparc2-one-targeted-reg-form seg y src-reg)
    (sparc2-vpop-register seg sparc::%arg_z)
    (unless (or *sparc2-reckless* (sparc2-form-typep x 'macptr))
      (with-imm-temps (src-reg) ()
        (! trap-unless-typecode= sparc::%arg_z arch::subtag-macptr)))
    (! set-macptr-address src-reg sparc::%arg_z)
    (<- sparc::%arg_z)
    (^)))

(defsparc2 sparc2-%setf-double-float %setf-double-float (seg vref xfer fnode fval)
  (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg fnode sparc::%arg_z))
  (let* ((target (make-hard-fp-reg sparc::%f4 hard-reg-class-fpr-mode-double)))
    (sparc2-one-targeted-reg-form seg fval target)
    (sparc2-vpop-register seg sparc::%arg_z)
    (unless (or *sparc2-reckless* (sparc2-form-typep fnode 'double-float))
      (! trap-unless-typecode= sparc::%arg_z arch::subtag-double-float))
    (! store-double sparc::%arg_z target)
    (<- sparc::%arg_z)
    (^)))

(defsparc2 sparc2-%setf-short-float %setf-short-float (seg vreg xfer fnode fval)
  (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg fnode sparc::%arg_z))
  (let* ((target (make-hard-fp-reg sparc::%f4 hard-reg-class-fpr-mode-single)))
    (sparc2-one-targeted-reg-form seg fval target)
    (sparc2-vpop-register seg sparc::%arg_z)
    (unless (or *sparc2-reckless* (sparc2-form-typep fnode 'short-float))
      (! trap-unless-typecode= sparc::%arg_z arch::subtag-single-float))
    (! store-single sparc::%arg_z target)
    (<- sparc::%arg_z)
    (^)))

(defsparc2 sparc2-unwind-protect unwind-protect (seg vreg xfer protected-form cleanup-form)
  (let* ((cleanup-label (backend-get-next-label))
         (protform-label (backend-get-next-label))
         (old-stack (sparc2-encode-stack)))
    (! build-catch-frame .SPmkunwind (aref *backend-labels* cleanup-label))
    (-> protform-label)
    (@ cleanup-label)
    (let* ((*sparc2-vstack* *sparc2-vstack*)
           (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*)
           (*sparc2-cstack* (%i+ *sparc2-cstack* 16)))
      (sparc2-open-undo $undostkblk)      ; tsp frame created by nthrow.
      (! save-cleanup-context)
      (setq *sparc2-cstack* (%i+ *sparc2-cstack* 16))       ; the frame we just pushed
      (sparc2-form seg nil nil cleanup-form)
      (sparc2-close-undo)
      (! restore-cleanup-context)
      (! jump-return-pc)) ; rts
    (sparc2-open-undo)
    (@ protform-label)
    (sparc2-undo-body seg vreg xfer protected-form old-stack)))

(defsparc2 sparc2-progv progv (seg vreg xfer symbols values body)
  (let* ((cleanup-label (backend-get-next-label))
         (protform-label (backend-get-next-label))
         (old-stack (sparc2-encode-stack)))
    (sparc2-two-targeted-reg-forms seg symbols sparc::%arg_y values sparc::%arg_z)
    (! call-subprim .SPprogvsave)
    (sparc2-open-undo $undostkblk)
    (! build-catch-frame .SPmkunwind (aref *backend-labels* cleanup-label))
    (-> protform-label)
    (@ cleanup-label)
    (! jump-subprim .SPprogvrestore)
    (sparc2-open-undo)
    (@ protform-label)
    (sparc2-undo-body seg vreg xfer body old-stack)))

(defsparc2 sparc2-%ptr-eql %ptr-eql (seg vreg xfer cc x y )
  (if (null vreg)
    (progn
      (sparc2-form seg nil nil x)
      (sparc2-form seg nil xfer y))
    (let* ((x-abs (acode-absolute-ptr-p x t))
           (y-abs (acode-absolute-ptr-p y t))
           (abs (or x-abs y-abs))
           (other (if abs (if x-abs y x))))
      (multiple-value-bind (cr-bit true-p) (acode-condition-to-sparc-icc cc)
        (if other
          (with-imm-target () (other-target :address)
            (sparc2-one-targeted-reg-form seg other other-target)
            (if (typep abs '(signed-byte 13))              
              (sparc2-test-reg-%izerop seg vreg xfer other-target cr-bit true-p abs)
              (with-imm-temps (other-target) ((abs-target :address))
                (use-imm-temp other-target)
                (sparc2-lwi seg abs-target abs)
                (sparc2-compare-registers seg vreg xfer other-target abs-target cr-bit true-p))))
          ; Neither expression is obviously a constant-valued macptr.
          (with-imm-target () (target-a :address)
            (sparc2-one-targeted-reg-form seg x target-a)
            (! temp-push-unboxed-word target-a)
            (sparc2-open-undo $undostkblk)
            (sparc2-one-targeted-reg-form seg y target-a)
            (with-imm-target (target-a) (target-b :address)
              (! temp-pop-unboxed-word target-b)
              (sparc2-close-undo)
              (sparc2-compare-registers seg vreg xfer target-b target-a cr-bit true-p))))))))


(defsparc2 sparc2-%immediate-set-xxx %immediate-set-xxx (seg vreg xfer bits ptr offset val)
  (sparc2-%immediate-store seg vreg xfer bits ptr offset val nil))

(defsparc2 sparc2-immediate-put-xxx immediate-put-xxx (seg vreg xfer bits ptr val offset)
  (sparc2-%immediate-store seg nil nil bits ptr offset val t)
  (sparc2-nil seg vreg xfer))

(defsparc2 sparc2-%immediate-inc-ptr %immediate-inc-ptr (seg vreg xfer ptr by)
  (let* ((triv-by (sparc2-trivial-p by))
         (fixnum-by (acode-fixnum-form-p by)))
    (if (and fixnum-by (eql 0 fixnum-by))
      (sparc2-form seg vreg xfer ptr)
      (with-imm-target (vreg) (ptr-reg :address)
        (sparc2-one-targeted-reg-form seg ptr ptr-reg)
        (if fixnum-by
          (with-imm-target (vreg ptr-reg) (result :address)
            (! add-unboxed-constant result ptr-reg fixnum-by)
            (<- result))
          (progn
            (unless triv-by
              (! temp-push-unboxed-word ptr-reg)
              (sparc2-open-undo $undostkblk))
            (with-imm-target (vreg ptr-reg) (by-reg :s32)
              (sparc2-one-targeted-reg-form seg by by-reg)
              (unless triv-by
                (! temp-pop-unboxed-word ptr-reg)
                (sparc2-close-undo))
              (with-imm-target (vreg ptr-reg by-reg) (result :address)
                (! add-unboxed result ptr-reg by-reg)
                (<- result)))))
        (^)))))

; The form in question just binds *interrupt-level* to -1, but we want
; to disable the generation of event-polling sequences that would otherwise
; appear lexically within the body.
(defsparc2 sparc2-without-interrupts without-interrupts (seg vreg xfer form)
  (let* ((*sparc2-inhibit-eventchecks* t))
    (sparc2-form seg vreg xfer form)))


(defsparc2 sparc2-multiple-value-call multiple-value-call (seg vreg xfer fn arglist)
  (sparc2-mvcall seg vreg xfer fn arglist))

(defsparc2 sparc2-ff-call eabi-ff-call (seg vreg xfer address argspecs argvals resultspec)
  (let* ((*sparc2-vstack* *sparc2-vstack*)
         (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*)
         (*sparc2-cstack* *sparc2-cstack*)
         (nextarg 0)
         (num-c-arg-words (the fixnum
                            (+ (the fixnum (length argvals)) 
                               (the fixnum (count-if #'(lambda (x)
                                                         (eq x :double-float))
                                                     argspecs))))))
      (declare (fixnum nextarg))
      (! alloc-c-frame num-c-arg-words )
      (sparc2-open-undo $undo-sparc-c-frame)
      (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg address sparc::%arg_z))
      ; Evaluate each form into the C frame, according to the matching argspec.
      (do* ((specs argspecs (cdr specs))
            (vals argvals (cdr vals)))
           ((null specs))
        (declare (list specs vals))
        (let* ((valform (car vals))
               (spec (car specs))
               (absptr (acode-absolute-ptr-p valform)))
          (case spec
            (:double-float
             (let* ((target (make-hard-fp-reg sparc::%f4 hard-reg-class-fpr-mode-double)))
               (sparc2-one-targeted-reg-form seg valform target)
               (with-imm-temps () (high low)
                 (! double-float-to-register-pair high low target)
                 (! store-c-arg high nextarg)
                 (incf nextarg)
                 (! store-c-arg low nextarg))))
            (:single-float
             (let* ((target (make-hard-fp-reg sparc::%f4 hard-reg-class-fpr-mode-single)))
               (sparc2-one-targeted-reg-form seg valform target )
               (! store-single-c-arg target nextarg)))
             (:address
              (with-imm-target () (ptr :address)
                 (if absptr
                   (sparc2-lwi seg ptr absptr)
                   (sparc2-one-targeted-reg-form seg valform ptr))
               (! store-c-arg ptr nextarg)))
            (t
             (! store-c-arg
                (with-imm-target () (valreg :u32)
                  (sparc2-unboxed-integer-arg-to-reg seg valform valreg))
                nextarg)))
          (incf nextarg)))
      (sparc2-vpop-register seg sparc::%arg_z)
      (! call-subprim .SPeabi-ff-call) 
      (sparc2-close-undo)
      (when vreg
        (if (eq resultspec :void)
          (<- sparc::%rnil)
          (if (eq resultspec :double-float)
            (<- (make-hard-fp-reg sparc::%f4 hard-reg-class-fpr-mode-double))
            (if (eq resultspec :single-float)
              (<- (make-hard-fp-reg sparc::%f4 hard-reg-class-fpr-mode-single ))
              (<- (set-regspec-mode sparc::%imm0 (gpr-mode-name-value
                                                    (case resultspec
                                                      (:address :address)
                                                      (:signed-byte :s8)
                                                      (:unsigned-byte :u8)
                                                      (:signed-halfword :s16)
                                                      (:unsigned-halfword :u16)
                                                      (:signed-fullword :s32)
                                                      (t :u32)))))))))
      (^)))

(defun sparc2-ff-call-internal (seg vreg xfer address argspecs argvals resultspec subprim)
  (with-sparc-local-vinsn-macros (seg vreg xfer)
    (let* ((*sparc2-vstack* *sparc2-vstack*)
           (*sparc2-top-vstack-lcell* *sparc2-top-vstack-lcell*)
           (*sparc2-cstack* *sparc2-cstack*)
           (nextarg 0)
           (fp-loads ()))
      (declare (fixnum nextarg))
      (! alloc-c-frame (the fixnum
                         (+ (the fixnum (length argvals)) 
                            (the fixnum (count-if #'(lambda (x) (eq x :double-float)) argspecs)))))
      (sparc2-open-undo $undo-sparc-c-frame)
      (sparc2-vpush-register seg (sparc2-one-untargeted-reg-form seg address sparc::%arg_z))
      ; Evaluate each form into the C frame, according to the matching argspec.
      ; Remember type and arg offset of any FP args, since FP regs will have to be loaded
      ; later.
      (do* ((specs argspecs (cdr specs))
            (vals argvals (cdr vals)))
           ((null specs))
        (declare (list specs vals))
        (let* ((valform (car vals))
               (spec (car specs))
               (longval (sparc2-long-constant-p valform))
               (absptr (acode-absolute-ptr-p valform)))
          (case spec
            (:double-float
             (sparc2-one-targeted-reg-form seg valform (make-hard-fp-reg sparc::%f4 hard-reg-class-fpr-mode-double))
             (! set-double-c-arg sparc::%f4 nextarg)            
             (push (cons :double nextarg) fp-loads)
             (incf nextarg))
            (:single-float
             (sparc2-one-targeted-reg-form
              seg valform (make-hard-fp-reg
                           sparc::%f4 hard-reg-class-fpr-mode-single))
             (! set-single-c-arg sparc::%f4 nextarg)
             (push (cons :single nextarg) fp-loads))
            (:address
             (with-imm-target () (ptr :address)
               (if absptr
                 (sparc2-lwi seg ptr absptr)
                 (sparc2-one-targeted-reg-form seg valform ptr))
               (! set-c-arg ptr nextarg)))
            (t
             (with-imm-target () (valreg :u32)
               (if longval
                 (sparc2-lwi seg valreg longval)
                 (sparc2-unboxed-integer-arg-to-reg seg valform valreg))
               (! set-c-arg valreg nextarg))))
          (incf nextarg)))
      (do* ((fpreg sparc::%f4 (1+ fpreg))
            (reloads (nreverse fp-loads) (cdr reloads)))
           ((or (null reloads) (= fpreg sparc::%f4)))
        (declare (list reloads) (fixnum fpreg))
        (let* ((reload (car reloads))
               (size (car reload))
               (from (cdr reload)))
          (if (eq size :double)
            (! reload-double-c-arg fpreg from)
            (! reload-single-c-arg fpreg from))))
      (sparc2-vpop-register seg sparc::%arg_z)
      (! call-subprim subprim)            ; subprim is .SPffcalladdress or .SPffcallslep
      (sparc2-close-undo)
      (when vreg
        (if (eq resultspec :void)
          (<- sparc::%rnil)
          (if (eq resultspec :double-float)
            (<- (make-hard-fp-reg sparc::%f4 hard-reg-class-fpr-mode-double))
            (if (eq resultspec :single-float)
              (<- (make-hard-fp-reg 
                   sparc::%f4 
                    hard-reg-class-fpr-mode-single))
              (<- (set-regspec-mode sparc::%imm0 (gpr-mode-name-value
                                                    (case resultspec
                                                      (:address :address)
                                                      (:signed-byte :s8)
                                                      (:unsigned-byte :u8)
                                                      (:signed-halfword :s16)
                                                      (:unsigned-halfword :u16)
                                                      (:signed-fullword :s32)
                                                      (t :u32)))))))))
      (^))))

               
             
(defsparc2 sparc2-%temp-list %temp-list (seg vreg xfer arglist)
  (sparc2-use-operator (%nx1-operator list) seg vreg xfer arglist))

(defsparc2 sparc2-%temp-cons %temp-cons (seg vreg xfer car cdr)
  (sparc2-use-operator (%nx1-operator cons) seg vreg xfer car cdr))

(defsparc2 sparc2-point-h integer-point-h (seg vreg xfer form)
  (if (null vreg)
    (sparc2-form seg nil xfer form)
    (with-imm-target () (s32 :s32)
      (let* ((immval (sparc2-long-constant-p form)))
        (if immval
          (sparc2-lwi seg s32 immval)
          (sparc2-one-targeted-reg-form seg form s32))
        (ensuring-node-target (target vreg)
          (! s16->fixnum target (set-regspec-mode s32 (gpr-mode-name-value :s16)))))
      (^))))

 
(defsparc2 sparc2-point-v integer-point-v (seg vreg xfer form)
  (if (null vreg)
    (sparc2-form seg nil xfer form)
    (with-imm-target () (s32 :s32)
      (let* ((immval (sparc2-long-constant-p form)))
        (if immval
          (sparc2-lwi seg s32 immval)
          (sparc2-one-targeted-reg-form seg form s32))
        (ensuring-node-target (target vreg)
          (! s32-highword target s32))
      (^)))))

         
(defsparc2 sparc2-%vreflet %vreflet (seg vreg xfer vars vals body p2decls)
  (let* ((old-stack (sparc2-encode-stack)))
    (with-sparc-p2-declarations p2decls
      (dolist (var vars)
        (sparc2-vpush-register 
         seg 
         (sparc2-one-untargeted-reg-form seg (pop vals) sparc::%arg_z))
        (with-node-temps () (ptr)
          (! macptr->stack ptr sparc::%vsp)
          (sparc2-open-undo $undostkblk)
          (sparc2-seq-bind-var seg var ptr)))
      (sparc2-undo-body seg vreg xfer body old-stack)
      (dolist (var vars) (sparc2-close-var seg var)))))


;; Under MacsBug 5.3 (and some others ?), this'll do a low-level user
;; break.  If the debugger doesn't recognize the trap instruction,
;; you'll have to manually advance the PC past it.  "arg" winds up in the
;; arg_z register; whatever's in arg_z on return is returned by
;; the %debug-trap construct.

(defsparc2 sparc2-%debug-trap %debug-trap (seg vreg xfer arg)
  (sparc2-one-targeted-reg-form seg arg sparc::%arg_z)
  (! %debug-trap)
  (<- sparc::%arg_z)
  (^))

;------

#+not-yet
(progn




  


(defsparc2 sparc2-short-float short-float (val)
  (sparc2-absolute-long (%address-of val)))










;Make a gcable macptr.
(defsparc2 sparc2-%new-ptr %new-ptr (b vreg xfer size clear-p )
  (declare (ignore b vreg xfer size clear-p))
  (error "%New-ptr is a waste of precious silicon."))





       





  






)
