#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) Enrique J. Hernández 2014

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""
Helper methods to set the Package and Dependencies fields, if missing, from Apport crashes.

This is specific to Zentyal.
"""
from datetime import datetime


def map_package(report):
    """
    Given a report, it will return the package and the version depending on the
    DistroRelease and the ExecutableTimestamp fields specific from Zentyal repositories.

    :param apport.report.Report report: the crash report
    :returns: a tuple containing the package and the version of the package.
    :rtype tuple:
    """
    if 'DistroRelease' not in report or 'ExecutableTimestamp' not in report:
        raise SystemError('No DistroRelease or ExecutableTimestamp to map the package')

    distro_release = report['DistroRelease']
    crash_date = datetime.fromtimestamp(int(report['ExecutableTimestamp']))
    if distro_release == 'Ubuntu 14.04':
        if crash_date >= datetime(2014, 5, 24, 1, 31):  # Release date
            return ('samba', '3:4.1.7+dfsg-2~zentyal2~64')
        return ('samba', '3:4.1.7+dfsg-2~zentyal1~32')
    elif distro_release == 'Ubuntu 13.10':
        return ('samba', '2:4.1.6+dfsg-1~zentyal1~106')
    elif distro_release == 'Ubuntu 12.04':
        if crash_date < datetime(2013, 10, 2):
            return ('samba4', '4.1.0rc3-zentyal3')
        elif crash_date < datetime(2013, 12, 10, 13, 03):
            return ('samba4', '4.1.0rc4-zentyal1')
        elif crash_date < datetime(2013, 12, 17, 11, 34):
            return ('samba4', '4.1.2-zentyal2')
        elif crash_date < datetime(2014, 3, 5, 20, 16):
            return ('samba4', '4.1.3-zentyal2')
        elif crash_date < datetime(2014, 5, 30, 8, 41):
            return ('samba4', '4.1.5-zentyal1')
        else:
            return ('samba4', '4.1.7-zentyal1')
    else:
        raise SystemError('Invalid Distro Release %s' % distro_release)


def map_dependencies(report):
    """
    Given a report, it will return the dependencies from the package depending on the
    DistroRelease fields specific from Zentyal repositories.

    :param apport.report.Report report: the crash report
    :returns: a list of the current dependencies packages
    :rtype tuple:
    """
    if 'DistroRelease' not in report:
        raise SystemError('No DistroRelease to get the dependencies packages')

    distro_release = report['DistroRelease']
    if distro_release == 'Ubuntu 14.04':
        return (
            'adduser',
            'apt-utils',
            'attr',
            'base-passwd',
            'busybox-initramfs',
            'ca-certificates',
            'ckeditor',
            'coreutils',
            'cpio',
            'cron',
            'dbus',
            'debconf',
            'debconf-i18n',
            'debianutils',
            'dpkg',
            'e2fslibs',
            'e2fsprogs',
            'file',
            'findutils',
            'gcc-4.8-base',
            'gcc-4.9-base',
            'gnustep-base-common',
            'gnustep-base-runtime',
            'gnustep-common',
            'ifupdown',
            'initramfs-tools',
            'initramfs-tools-bin',
            'initscripts',
            'insserv',
            'iproute2',
            'isc-dhcp-client',
            'isc-dhcp-common',
            'javascript-common',
            'klibc-utils',
            'kmod',
            'krb5-locales',
            'libacl1',
            'libaio1',
            'libapache2-mod-wsgi',
            'libapparmor1',
            'libapt-inst1.5',
            'libapt-pkg4.12',
            'libarchive-extract-perl',
            'libasn1-8-heimdal',
            'libattr1',
            'libaudit-common',
            'libaudit1',
            'libavahi-client3',
            'libavahi-common-data',
            'libavahi-common3',
            'libblkid1',
            'libbsd0',
            'libbz2-1.0',
            'libc6',
            'libcap2',
            'libcgmanager0',
            'libcomerr2',
            'libcups2',
            'libcurl3-gnutls',
            'libdb5.3',
            'libdbus-1-3',
            'libdebconfclient0',
            'libdrm2',
            'libevent-2.0-5',
            'libexpat1',
            'libffi6',
            'libfile-copy-recursive-perl',
            'libgcc1',
            'libgcrypt11',
            'libgdbm3',
            'libglib2.0-0',
            'libglib2.0-data',
            'libgmp10',
            'libgnustep-base1.24',
            'libgnutls26',
            'libgpg-error0',
            'libgpm2',
            'libgssapi-krb5-2',
            'libgssapi3-heimdal',
            'libhcrypto4-heimdal',
            'libhdb9-heimdal',
            'libheimbase1-heimdal',
            'libheimntlm0-heimdal',
            'libhx509-5-heimdal',
            'libicu52',
            'libidn11',
            'libjs-jquery',
            'libjs-jquery-ui',
            'libjs-prototype',
            'libjs-scriptaculous',
            'libjs-sphinxdoc',
            'libjs-swfobject',
            'libjs-underscore',
            'libjson-c2',
            'libjson0',
            'libk5crypto3',
            'libkdc2-heimdal',
            'libkeyutils1',
            'libklibc',
            'libkmod2',
            'libkrb5-26-heimdal',
            'libkrb5-3',
            'libkrb5support0',
            'liblasso3',
            'libldap-2.4-2',
            'libldb1',
            'liblocale-gettext-perl',
            'liblog-message-simple-perl',
            'liblzma5',
            'libmagic1',
            'libmapi0',
            'libmapiproxy0',
            'libmapistore0',
            'libmemcached10',
            'libmodule-pluggable-perl',
            'libmount1',
            'libmysqlclient18',
            'libncurses5',
            'libncursesw5',
            'libnih-dbus1',
            'libnih1',
            'libntdb1',
            'libobjc4',
            'libp11-kit0',
            'libpam-modules',
            'libpam-modules-bin',
            'libpam-runtime',
            'libpam-systemd',
            'libpam0g',
            'libpcre3',
            'libplymouth2',
            'libpng12-0',
            'libpod-latex-perl',
            'libpopt0',
            'libpq5',
            'libprocps3',
            'libpython-stdlib',
            'libpython2.7',
            'libpython2.7-minimal',
            'libpython2.7-stdlib',
            'libreadline6',
            'libroken18-heimdal',
            'librtmp0',
            'libsasl2-2',
            'libsasl2-modules',
            'libsasl2-modules-db',
            'libsbjson2.3',
            'libselinux1',
            'libsemanage-common',
            'libsemanage1',
            'libsepol1',
            'libslang2',
            'libsope1',
            'libsqlite3-0',
            'libss2',
            'libssl1.0.0',
            'libstdc++6',
            'libsystemd-daemon0',
            'libsystemd-login0',
            'libtalloc2',
            'libtasn1-6',
            'libtdb1',
            'libterm-ui-perl',
            'libtevent0',
            'libtext-charwidth-perl',
            'libtext-iconv-perl',
            'libtext-soundex-perl',
            'libtext-wrapi18n-perl',
            'libtinfo5',
            'libudev1',
            'libustr-1.0-1',
            'libuuid1',
            'libwbclient0',
            'libwind0-heimdal',
            'libxml2',
            'libxmlsec1',
            'libxmlsec1-openssl',
            'libxslt1.1',
            'libxtables10',
            'logrotate',
            'lsb-base',
            'makedev',
            'memcached',
            'mime-support',
            'module-init-tools',
            'mount',
            'mountall',
            'multiarch-support',
            'mysql-common',
            'netbase',
            'openchange-ocsmanager',
            'openchange-rpcproxy',
            'openchangeproxy',
            'openchangeserver',
            'openssl',
            'passwd',
            'perl',
            'perl-base',
            'perl-modules',
            'plymouth',
            'plymouth-theme-ubuntu-text',
            'procps',
            'psmisc',
            'python',
            'python-beaker',
            'python-bs4',
            'python-chardet',
            'python-crypto',
            'python-decorator',
            'python-dns',
            'python-dnspython',
            'python-formencode',
            'python-ldb',
            'python-lxml',
            'python-mako',
            'python-markupsafe',
            'python-minimal',
            'python-mysqldb',
            'python-nose',
            'python-ntdb',
            'python-ocsmanager',
            'python-openid',
            'python-openssl',
            'python-paste',
            'python-pastedeploy',
            'python-pastedeploy-tpl',
            'python-pastescript',
            'python-pkg-resources',
            'python-pygments',
            'python-pylons',
            'python-repoze.lru',
            'python-routes',
            'python-rpclib',
            'python-samba',
            'python-scgi',
            'python-setuptools',
            'python-simplejson',
            'python-six',
            'python-spyne',
            'python-sqlalchemy',
            'python-sqlalchemy-ext',
            'python-support',
            'python-talloc',
            'python-tdb',
            'python-tempita',
            'python-tz',
            'python-waitress',
            'python-weberror',
            'python-webhelpers',
            'python-webob',
            'python-webtest',
            'python2.7',
            'python2.7-minimal',
            'readline-common',
            'samba',
            'samba-common',
            'samba-common-bin',
            'samba-dsdb-modules',
            'samba-libs',
            'samba-vfs-modules',
            'sed',
            'sensible-utils',
            'sgml-base',
            'shared-mime-info',
            'sogo',
            'sogo-common',
            'sogo-openchange',
            'systemd-services',
            'sysv-rc',
            'sysvinit-utils',
            'tar',
            'tdb-tools',
            'tmpreaper',
            'tzdata',
            'ucf',
            'udev',
            'unzip',
            'update-inetd',
            'upstart',
            'util-linux',
            'uuid-runtime',
            'xml-core',
            'zip',
            'zlib1g'
        )
    elif distro_release == 'Ubuntu 13.10':
        return (
            'adduser',
            'apt-utils',
            'base-passwd',
            'busybox-initramfs',
            'ca-certificates',
            'ckeditor',
            'coreutils',
            'cpio',
            'cron',
            'dbus',
            'debconf',
            'debconf-i18n',
            'debianutils',
            'dpkg',
            'e2fslibs',
            'e2fsprogs',
            'file',
            'findutils',
            'gcc-4.8-base',
            'gnustep-base-common',
            'gnustep-base-runtime',
            'gnustep-common',
            'ifupdown',
            'initramfs-tools',
            'initramfs-tools-bin',
            'initscripts',
            'insserv',
            'iproute2',
            'isc-dhcp-client',
            'isc-dhcp-common',
            'klibc-utils',
            'kmod',
            'libacl1',
            'libaio1',
            'libapache2-mod-wsgi',
            'libapparmor1',
            'libapt-inst1.5',
            'libapt-pkg4.12',
            'libasn1-8-heimdal',
            'libattr1',
            'libaudit-common',
            'libaudit1',
            'libavahi-client3',
            'libavahi-common-data',
            'libavahi-common3',
            'libblkid1',
            'libbsd0',
            'libbz2-1.0',
            'libc6',
            'libcap2',
            'libclass-isa-perl',
            'libcomerr2',
            'libcups2',
            'libcurl3-gnutls',
            'libdb5.1',
            'libdbus-1-3',
            'libdrm2',
            'libevent-2.0-5',
            'libexpat1',
            'libffi6',
            'libfile-copy-recursive-perl',
            'libgcc1',
            'libgcrypt11',
            'libgdbm3',
            'libglib2.0-0',
            'libgmp10',
            'libgnustep-base1.24',
            'libgnutls26',
            'libgpg-error0',
            'libgssapi-krb5-2',
            'libgssapi3-heimdal',
            'libhcrypto4-heimdal',
            'libhdb9-heimdal',
            'libheimbase1-heimdal',
            'libheimntlm0-heimdal',
            'libhx509-5-heimdal',
            'libicu48',
            'libidn11',
            'libjs-jquery',
            'libjs-jquery-ui',
            'libjs-prototype',
            'libjs-scriptaculous',
            'libjs-sphinxdoc',
            'libjs-underscore',
            'libjson-c2',
            'libjson0',
            'libk5crypto3',
            'libkdc2-heimdal',
            'libkeyutils1',
            'libklibc',
            'libkmod2',
            'libkrb5-26-heimdal',
            'libkrb5-3',
            'libkrb5support0',
            'liblasso3',
            'libldap-2.4-2',
            'libldb1',
            'liblocale-gettext-perl',
            'liblzma5',
            'libmagic1',
            'libmapi0',
            'libmapiproxy0',
            'libmapistore0',
            'libmemcached10',
            'libmount1',
            'libmysqlclient18',
            'libncurses5',
            'libncursesw5',
            'libnih-dbus1',
            'libnih1',
            'libntdb1',
            'libobjc4',
            'libp11-kit0',
            'libpam-modules',
            'libpam-modules-bin',
            'libpam-runtime',
            'libpam-systemd',
            'libpam0g',
            'libpci3',
            'libpcre3',
            'libplymouth2',
            'libpng12-0',
            'libpopt0',
            'libpq5',
            'libprocps0',
            'libpython-stdlib',
            'libpython2.7',
            'libpython2.7-minimal',
            'libpython2.7-stdlib',
            'libreadline6',
            'libroken18-heimdal',
            'librtmp0',
            'libsasl2-2',
            'libsasl2-modules',
            'libsasl2-modules-db',
            'libsbjson2.3',
            'libselinux1',
            'libsemanage-common',
            'libsemanage1',
            'libsepol1',
            'libslang2',
            'libsope1',
            'libsqlite3-0',
            'libss2',
            'libssl1.0.0',
            'libstdc++6',
            'libswitch-perl',
            'libsystemd-daemon0',
            'libsystemd-login0',
            'libtalloc2',
            'libtasn1-3',
            'libtdb1',
            'libtevent0',
            'libtext-charwidth-perl',
            'libtext-iconv-perl',
            'libtext-wrapi18n-perl',
            'libtinfo5',
            'libudev1',
            'libusb-1.0-0',
            'libustr-1.0-1',
            'libuuid1',
            'libwbclient0',
            'libwind0-heimdal',
            'libxml2',
            'libxmlsec1',
            'libxmlsec1-openssl',
            'libxslt1.1',
            'libxtables10',
            'logrotate',
            'lsb-base',
            'makedev',
            'memcached',
            'mime-support',
            'module-init-tools',
            'mount',
            'mountall',
            'multiarch-support',
            'mysql-common',
            'netbase',
            'openchange-ocsmanager',
            'openchange-rpcproxy',
            'openchangeproxy',
            'openchangeserver',
            'openssl',
            'passwd',
            'pciutils',
            'perl',
            'perl-base',
            'perl-modules',
            'plymouth',
            'plymouth-theme-ubuntu-text',
            'procps',
            'psmisc',
            'python',
            'python-beaker',
            'python-chardet',
            'python-crypto',
            'python-decorator',
            'python-dnspython',
            'python-formencode',
            'python-ldb',
            'python-lxml',
            'python-mako',
            'python-mapistore',
            'python-markupsafe',
            'python-minimal',
            'python-mysqldb',
            'python-nose',
            'python-ntdb',
            'python-ocsmanager',
            'python-openssl',
            'python-paste',
            'python-pastedeploy',
            'python-pastescript',
            'python-pkg-resources',
            'python-pygments',
            'python-pylons',
            'python-repoze.lru',
            'python-routes',
            'python-rpclib',
            'python-samba',
            'python-setuptools',
            'python-simplejson',
            'python-spyne',
            'python-support',
            'python-talloc',
            'python-tdb',
            'python-tempita',
            'python-tz',
            'python-weberror',
            'python-webhelpers',
            'python-webob',
            'python-webtest',
            'python2.7',
            'python2.7-minimal',
            'readline-common',
            'samba',
            'samba-common',
            'samba-common-bin',
            'samba-dsdb-modules',
            'samba-libs',
            'samba-vfs-modules',
            'sed',
            'sensible-utils',
            'sgml-base',
            'shared-mime-info',
            'sogo',
            'sogo-common',
            'sogo-openchange',
            'systemd-services',
            'sysv-rc',
            'sysvinit-utils',
            'tar',
            'tdb-tools',
            'tmpreaper',
            'tzdata',
            'ucf',
            'udev',
            'update-inetd',
            'upstart',
            'usbutils',
            'util-linux',
            'xml-core',
            'zip',
            'zlib1g'
        )
    elif distro_release == 'Ubuntu 12.04':
        return (
            'adduser',
            'apache2',
            'apache2-utils',
            'apache2.2-bin',
            'apache2.2-common',
            'autotools-dev',
            'base-passwd',
            'bind9-host',
            'binutils',
            'busybox-initramfs',
            'ca-certificates',
            'coreutils',
            'cpio',
            'cpp-4.6',
            'debconf',
            'debianutils',
            'dnsutils',
            'dpkg',
            'findutils',
            'gcc-4.6',
            'gcc-4.6-base',
            'gnustep-base-common',
            'gnustep-base-runtime',
            'gnustep-common',
            'gnustep-make',
            'gobjc-4.6',
            'ifupdown',
            'initramfs-tools',
            'initramfs-tools-bin',
            'initscripts',
            'insserv',
            'iproute',
            'klibc-utils',
            'libacl1',
            'libapache2-mod-wsgi',
            'libapr1',
            'libaprutil1',
            'libaprutil1-dbd-sqlite3',
            'libaprutil1-ldap',
            'libasn1-8-heimdal',
            'libattr1',
            'libavahi-client3',
            'libavahi-common-data',
            'libavahi-common3',
            'libbind9-80',
            'libblkid1',
            'libbsd0',
            'libbz2-1.0',
            'libc-bin',
            'libc-dev-bin',
            'libc6',
            'libc6-dev',
            'libcap2',
            'libclass-isa-perl',
            'libcomerr2',
            'libcups2',
            'libcurl3',
            'libdb5.1',
            'libdbus-1-3',
            'libdm0',
            'libdns81',
            'libdrm-intel1',
            'libdrm-nouveau1a',
            'libdrm-radeon1',
            'libdrm2',
            'libevent-2.0-5',
            'libexpat1',
            'libffi6',
            'libgcc1',
            'libgcrypt11',
            'libgdbm3',
            'libgeoip1',
            'libglib2.0-0',
            'libgmp10',
            'libgnustep-base1.22',
            'libgnutls26',
            'libgomp1',
            'libgpg-error0',
            'libgssapi-krb5-2',
            'libgssapi3-heimdal',
            'libhcrypto4-heimdal',
            'libheimbase1-heimdal',
            'libheimntlm0-heimdal',
            'libhx509-5-heimdal',
            'libicu48',
            'libidn11',
            'libisc83',
            'libisccc80',
            'libisccfg82',
            'libjs-prototype',
            'libjs-scriptaculous',
            'libk5crypto3',
            'libkeyutils1',
            'libklibc',
            'libkrb5-26-heimdal',
            'libkrb5-3',
            'libkrb5support0',
            'libldap-2.4-2',
            'liblwres80',
            'liblzma5',
            'libmapi0',
            'libmapiproxy0',
            'libmapistore0',
            'libmemcached6',
            'libmount1',
            'libmpc2',
            'libmpfr4',
            'libmysqlclient18',
            'libncurses5',
            'libncursesw5',
            'libnih-dbus1',
            'libnih1',
            'libobjc3',
            'libp11-kit0',
            'libpam-modules',
            'libpam-modules-bin',
            'libpam0g',
            'libpciaccess0',
            'libpcre3',
            'libplymouth2',
            'libpng12-0',
            'libpython2.7',
            'libquadmath0',
            'libreadline6',
            'libroken18-heimdal',
            'librtmp0',
            'libsasl2-2',
            'libsbjson2.3',
            'libselinux1',
            'libslang2',
            'libsope-appserver4.9',
            'libsope-core4.9',
            'libsope-gdl1-4.9',
            'libsope-ldap4.9',
            'libsope-mime4.9',
            'libsope-xml4.9',
            'libsqlite3-0',
            'libssl1.0.0',
            'libstdc++6',
            'libswitch-perl',
            'libtasn1-3',
            'libtinfo5',
            'libudev0',
            'libuuid1',
            'libwind0-heimdal',
            'libxml2',
            'libxslt1.1',
            'linux-libc-dev',
            'lsb-base',
            'makedev',
            'memcached',
            'mime-support',
            'module-init-tools',
            'mount',
            'mountall',
            'multiarch-support',
            'mysql-common',
            'ncurses-bin',
            'openchange-ocsmanager',
            'openchange-rpcproxy',
            'openchangeproxy',
            'openchangeserver',
            'openssl',
            'passwd',
            'perl',
            'perl-base',
            'perl-modules',
            'plymouth',
            'procps',
            'python',
            'python-beaker',
            'python-decorator',
            'python-dnspython',
            'python-formencode',
            'python-lxml',
            'python-mako',
            'python-mapistore',
            'python-markupsafe',
            'python-minimal',
            'python-mysqldb',
            'python-nose',
            'python-ocsmanager',
            'python-paste',
            'python-pastedeploy',
            'python-pastescript',
            'python-pkg-resources',
            'python-pygments',
            'python-pylons',
            'python-routes',
            'python-rpclib',
            'python-setuptools',
            'python-simplejson',
            'python-spyne',
            'python-support',
            'python-tempita',
            'python-tz',
            'python-weberror',
            'python-webhelpers',
            'python-webob',
            'python-webtest',
            'python2.7',
            'python2.7-minimal',
            'readline-common',
            'samba4',
            'sed',
            'sensible-utils',
            'sgml-base',
            'sogo',
            'sogo-openchange',
            'sope4.9-libxmlsaxdriver',
            'sysv-rc',
            'sysvinit-utils',
            'tar',
            'tmpreaper',
            'tzdata',
            'udev',
            'upstart',
            'util-linux',
            'xml-core',
            'xz-utils',
            'zlib1g'
        )
    else:
        raise SystemError('Invalid Distro Release %s' % distro_release)
