/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef PATCH_H
#define PATCH_H

#include <cstdlib>
#include <ladspa.h>
#include "config.h"
#include "Array.h"
#include "NodeTree.h"
#include "NodeBase.h"
#include "List.h"
#include "CrossThreadMutex.h"
#include "Plugin.h"

#ifdef HAVE_DSSI
#include <alsa/asoundlib.h>
#endif

using std::string;

namespace Om {

class Connection;
class InputNode;
class OutputNode;
class MidiInNode;


/** A group of nodes in a graph, possibly polyphonic.
 *
 * Note that this is also a Node, just one which contains Nodes.
 * Therefore infinite subpatching is possible, of polyphonic
 * patches of polyphonic nodes etc. etc.
 *
 * \ingroup engine
 */
class Patch : public NodeBase
{
public:
	Patch(const string& name, uint poly, Patch* parent, samplerate srate, size_t buffer_size, uint local_poly);
	virtual ~Patch();

	Patch* as_patch() { return static_cast<Patch*>(this); }

	void activate();
	void deactivate();

	void run(size_t nframes);
	
	virtual Port* const port(const string& port_name) const;
	
	uint num_ports() const;

	void send_creation_messages(lo_address addr) const;
	void send_deletion_messages(lo_address addr) const;
	
	// Patch specific stuff not inherited from Node
	
	void add_node(TreeNode* tn);
	TreeNode* remove_node(TreeNode* tn);

	const List<Connection*>& connections() const { return m_connections; }
	
	void                         add_connection(ListNode<Connection*>* const c) { m_connections.push_back(c); }
	ListNode<Connection*>* const remove_connection(const string& src_port_path, const string& dst_port_path);
	
	List<MidiInNode*>& midi_in_nodes()                                  { return m_midi_in_nodes; }
	void               add_midi_in_node(ListNode<MidiInNode*>* const n) { m_midi_in_nodes.push_back(n); }
	
	List<InputNode*>&        input_nodes()        { return m_input_nodes; }
	const List<InputNode*>&  input_nodes()  const { return m_input_nodes; }
	List<OutputNode*>&       output_nodes()       { return m_output_nodes; }
	const List<OutputNode*>& output_nodes() const { return m_output_nodes; }
	
	List<Node*>* const process_order()                { return m_process_order; }
	void               process_order(List<Node*>* po) { m_process_order = po; }
	
	/** Whether to run this patch's DSP in the audio thread */
	bool process() const { return m_process; }
	void process(bool b) { m_process = b; }

	/** Used to protect access to the node tree while removing (not thread safe) */
	CrossThreadMutex&  node_remove_mutex()            { return m_node_remove_mutex; }

	uint num_in_ports() const; 
	uint num_out_ports() const;

	uint internal_poly() const { return m_internal_poly; }
	
	const NodeTree&    nodes() const { return m_nodes; }

	const Plugin* const plugin() const                 { return &m_plugin; }
	void                plugin(const Plugin* const pi) { exit(EXIT_FAILURE); }

#ifdef HAVE_ALSA
#ifdef HAVE_DSSI
	snd_seq_event_t* dssi_events_array()        { return m_dssi_events_array; }
	size_t           dssi_events_size()         { return m_dssi_events_size; }
	void             dssi_events_size(size_t s) { m_dssi_events_size = s; }
#endif
#endif

private:
	// Prevent copies
	Patch(const Patch& copy) : NodeBase(*this) { exit(EXIT_FAILURE); }
	Patch& operator=(const Patch& copy)        { exit(EXIT_FAILURE); }

	uint              m_internal_poly;
	List<Node*>*      m_process_order;
	List<Connection*> m_connections;
	CrossThreadMutex  m_node_remove_mutex;
	List<MidiInNode*> m_midi_in_nodes;
	List<InputNode*>  m_input_nodes;
	List<OutputNode*> m_output_nodes;
	NodeTree          m_nodes;
	bool              m_process;

#ifdef HAVE_DSSI
	snd_seq_event_t*  m_dssi_events_array;
	size_t            m_dssi_events_size;
#endif
	
	Plugin m_plugin;
};


} // namespace Om

#endif // PATCH_H
