/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef INPUTPORT_H
#define INPUTPORT_H

#include <string>
#include <cstdlib>
#include <cassert>
#include <ladspa.h>
#include "Port.h"
#include "List.h"

using std::string;

namespace Om {

class Connection;
class OutputPort;
class Node;


/** An input port on a Node or Patch.
 *
 * Input ports always have an allocated buffer (all ports do, m_local_buffer)
 * but if the port has a single connection this buffer may not be used (instead
 * using the buffer directly from the Connection object so no copying takes
 * place)
 *
 * If a port has multiple connections, they will be mixed down into the local
 * buffer and it will be used.
 *
 * \ingroup engine
 */
class InputPort : public Port
{
public:
	InputPort(Node* node, const string& name, uint index, uint poly, PortInfo* port_info, size_t buffer_size);
	~InputPort();
	
	void                         add_connection(ListNode<Connection*>* const c);
	ListNode<Connection*>* const remove_connection(const OutputPort* const src_port);

	void join(const OutputPort* const port);
	bool is_connected() const { return (m_connections.size() > 0); }

	bool is_connected_to(const OutputPort* const port) const;

	void prepare_buffers();
	
private:
	// Prevent copies
	InputPort(const InputPort& copy) : Port(copy) { exit(EXIT_FAILURE); }
	InputPort& operator=(const InputPort& copy)   { exit(EXIT_FAILURE); }

	List<Connection*> m_connections;
	bool              m_is_joined;
};


} // namespace Om

#endif // INPUTPORT_H
