/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "InputPort.h"
#include <iostream>
#include <cstdlib>
#include <cassert>
#include "Connection.h"
#include "OutputPort.h"
#include "Node.h"
#include "Om.h"
#include "util.h"

namespace Om {


InputPort::InputPort(Node* node, const string& name, uint index, uint poly, PortInfo* port_info, size_t buffer_size)
: Port(node, name, index, poly, port_info, buffer_size),
  m_is_joined(false)
{
	assert(is_input() && !is_output());
}


InputPort::~InputPort()
{
}


/** Add a connection.  Realtime safe.
 *
 * The buffer of this port will be set directly to the connection's buffer
 * if there is only one connection, since no mixing needs to take place.
 */
void
InputPort::add_connection(ListNode<Connection*>* const c)
{
	m_connections.push_back(c);

	if (!m_is_joined) {
		if (m_connections.size() == 1) {
			// use buffer directly to avoid copying
			for (uint i=0; i < m_poly; ++i) {
				m_buffers.at(i) = const_cast<sample*>(c->elem()->buffer(i));
				parent_node()->set_port_buffer(i, m_index, m_buffers.at(i));
			}
		} else if (m_connections.size() == 2) {
			// used to have one buffer, directly used, now we have two
			// so we have to fix that and use local ones again
			for (uint i=0; i < m_poly; ++i) {
				m_buffers.at(i) = m_local_buffers.at(i);
				parent_node()->set_port_buffer(i, m_index, m_buffers.at(i));
			}
		}
	}
}


/** Remove a connection.  Realtime safe.
 */
ListNode<Connection*>* const
InputPort::remove_connection(const OutputPort* const src_port)
{
	bool found = false;
	ListNode<Connection*>* connection = NULL;
	for (List<Connection*>::iterator i = m_connections.begin(); i != m_connections.end(); ++i) {
		if ((*i)->src_port()->path() == src_port->path()) {
			connection = m_connections.remove(i);
			found = true;
		}
	}

	if ( ! found) {
		cerr << "WARNING:  [InputPort::remove_connection] Connection not found !" << endl;
		exit(EXIT_FAILURE);
	} else {
		if (m_connections.size() == 0) {
			// Write silence
			for (uint i=0; i < m_poly; ++i) {
				if (!m_is_joined) {
					m_buffers.at(i) = m_local_buffers.at(i);
					parent_node()->set_port_buffer(i, m_index, m_buffers.at(i));
					m_is_local_buffer = true;
				}
				Om::write_buffer(m_buffers.at(i), 0.0f, 0, m_buffer_size-1);
			}
		} else if (!m_is_joined && m_connections.size() == 1) {
			// Share a buffer if possible
			for (uint i=0; i < m_poly; ++i) {
				m_buffers.at(i) = const_cast<sample*>((*m_connections.begin())->buffer(i));
				parent_node()->set_port_buffer(i, m_index, m_buffers.at(i));
			}
		}	
	}

	return connection;
}


/** Returns whether this port is connected to the passed port.
 */
bool
InputPort::is_connected_to(const OutputPort* const port) const
{
	for (List<Connection*>::const_iterator i = m_connections.begin(); i != m_connections.end(); ++i)
		if ((*i)->src_port() == port)
			return true;
	
	return false;
}


/** "Joins" this port to an OutputPort, so they share the same buffer.
 *
 * This is used by OutputNode and InputNode to provide two different ports
 * (internal and external) that share a buffer.
 */
void
InputPort::join(const OutputPort* const port)
{
	if (parent_node() != NULL) {
		assert(m_poly == port->poly());
	
		for (uint i=0; i < m_poly; ++i) {
			m_buffers.at(i) = port->buffer(i);
			parent_node()->set_port_buffer(i, m_index, m_buffers.at(i));
		}
	}
	m_is_joined = true;
}
	

/** Prepare buffer for access, mixing if necessary.  Realtime safe.
 */
void
InputPort::prepare_buffers()
{
	// FIXME: this function is entirely too slow

	const uint num_ins = m_connections.size();
	if (num_ins == 0) return;

	Connection*   connection;
	const sample* src;
	sample*       dst;
	int           buffer_size;

	for (List<Connection*>::iterator c = m_connections.begin(); c != m_connections.end(); ++c)
		(*c)->prepare_buffers();

	// If only one connection, buffer is used directly (no copying)
	if (!m_is_joined && num_ins == 1)
		return;

	for (uint voice=0; voice < m_poly; ++voice) {
		// Size problem with zero_buffer and this buffer?
		//memcpy(m_buffers.at(voice), om->zero_buffer(), sizeof(sample) * m_buffer_size);
		Om::write_buffer(m_buffers.at(voice), 0.0f, 0, m_buffer_size-1);

		for (List<Connection*>::iterator c = m_connections.begin(); c != m_connections.end(); ++c) {
			connection = *c;
			src = connection->buffer(voice);
			dst = m_buffers.at(voice);
			buffer_size = m_buffer_size;
			
			assert(dst != NULL);
			assert(src != NULL);

			while (buffer_size--)
				*dst++ += *src++;
		}
		
		// Scale the buffer down.
		// I have decided this is a bad idea..
		/*if (num_ins > 1) {
			float mult = 1.0f/((float)num_ins); // Avoid dividing
			dst = m_buffers.at(voice);
			buffer_size = m_buffer_size;
			while (buffer_size--)
				*dst++ *= mult;
		}*/
	}
}


} // namespace Om

