/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "AddNodeEvent.h"
#include "Patch.h"
#include "Node.h"
#include "NodeTree.h"
#include "Plugin.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "Om.h"
#include "OmApp.h"
#include "Patch.h"
#include "NodeFactory.h"
#include "JackDriver.h"
#include "OSCSender.h"
#include "SlowEventQueue.h"
#include "OmPath.h"
#include "util.h"
#include "ObjectStore.h"
#include "OmPath.h"

namespace Om {


AddNodeEvent::AddNodeEvent(Request* request, const string& path, Plugin* pi, bool poly)
: SlowEvent(request),
  m_path(path),
  m_plugin(pi),
  m_poly(poly),
  m_patch(NULL),
  m_node(NULL),
  m_tree_node(NULL),
  m_process_order(NULL),
  m_node_already_exists(false),
  m_acquired_mutex(false)
{
	/* Notice that this is NOT a blocking event.  It doesn't need to be, because the
	 * node is inserted to the tree in the prepare() method, so an immediately following
	 * metadata (or whatever) event will be able to find it without waiting for the
	 * entire event to finish.
	 * 
	 * There is an exception to this for input/output nodes (that define a port on a
	 * patch).  These ports are inserted in the patch's list in the execute method (to
	 * prevent a race condition), so what happens in the prepare() method is not enough
	 * for the next event to notice this node's existance.  So, adding input/output
	 * nodes is a blocking event */

	// Update:  removed this hack in favour of hack in Patch::port that doesn't
	// block the event queue
	/*
	if (m_plugin->type() == Plugin::Internal
		&& (   m_plugin->plug_label() == "control_input"
			|| m_plugin->plug_label() == "control_output"
		    || m_plugin->plug_label() == "audio_input"
			|| m_plugin->plug_label() == "audio_output"))
		m_blocking = true;
	*/
}


AddNodeEvent::~AddNodeEvent()
{
	delete m_plugin;
}


void
AddNodeEvent::prepare()
{
	Node* existing = om->object_store()->find_node(m_path);
	if (existing != NULL) {
		m_node_already_exists = true;
		SlowEvent::prepare();
		return;
	}

	m_patch = om->object_store()->find_patch(OmPath::parent(m_path));

	if (m_patch != NULL) {
		if (m_poly)
			m_node = om->node_factory()->load_plugin(m_plugin, OmPath::name(m_path), m_patch->internal_poly(), m_patch);
		else
			m_node = om->node_factory()->load_plugin(m_plugin, OmPath::name(m_path), 1, m_patch);
		
		if (m_node != NULL) {
			m_node->activate();
		
			m_tree_node = new TreeNode(m_node);

			m_patch->node_remove_mutex().soft_lock();
			m_acquired_mutex = true;
			
			// This can be done here because the audio thread doesn't touch the
			// node tree - just the process order array
			m_patch->add_node(m_tree_node);
	
			if (m_patch->process())
				m_process_order = Om::find_process_order(m_patch);
		}
	}
	SlowEvent::prepare();
}


void
AddNodeEvent::execute(uint sample_offset)
{
	if (m_patch == NULL || m_node == NULL) return;

	SlowEvent::execute(sample_offset);

	if (m_tree_node != NULL) {
		m_tree_node->node()->add_to_patch();
		
		if (m_patch->process_order() != NULL)
			om->maid()->push(m_patch->process_order());
		m_patch->process_order(m_process_order);
	}
}


void
AddNodeEvent::post_process()
{
	if (m_acquired_mutex)
		m_patch->node_remove_mutex().soft_unlock();

	string msg;
	if (m_node_already_exists) {
		msg = "Could not create node - ";
		msg += m_path + " already exists.";
		m_request->respond_error(msg);
	} else if (m_patch == NULL) {
		m_request->respond_error("Could not find patch in /patch/add_node");
	} else if (m_node == NULL) {
		msg = "Unable to load node ";
		msg.append(m_path).append(" (you're missing the plugin \"").append(
			m_plugin->lib_name()).append(":").append(m_plugin->plug_label()).append("\")");;
		m_request->respond_error(msg);
	} else {
		m_request->respond_ok();
		om->osc_sender()->send_node_creation_messages(m_node);
	}

	if (m_blocking)
		om->jack_driver()->slow_event_queue()->signal_blocking_event_finished();
}


} // namespace Om

