/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "PatchWindow.h"
#include <iostream>
#include <cassert>
#include <fstream>
#include "OmGtk.h"
#include "OmGtkApp.h"
#include "OmPatchBayArea.h"
#include "PatchController.h"
#include "LoadPluginWindow.h"
#include "PatchModel.h"
#include "NewSubpatchWindow.h"
#include "LoadSubpatchWindow.h"
#include "NodeControlWindow.h"
#include "PatchDescriptionWindow.h"
#include "Controller.h"

namespace OmGtk {


PatchWindow::PatchWindow(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& xml)
: Gtk::Window(cobject),
  m_patch_controller(NULL),
  m_patch_bay(NULL),
  m_load_plugin_window(NULL),
  m_new_subpatch_window(NULL),
  m_enable_signal(true),
  m_is_visible(false),
  m_position_stored(false),
  m_x(0),
  m_y(0)
{
	property_visible() = false;

	xml->get_widget("patch_file_save_patch_menuitem", m_menu_file_save_patch);
	xml->get_widget("patch_file_save_patch_as_menuitem", m_menu_file_save_patch_as);
	xml->get_widget("patch_file_close_menuitem", m_menu_file_close);
	//xml->get_widget("patch_view_refresh_menuitem", m_menu_view_refresh);
	xml->get_widget("patch_view_control_window_menuitem", m_menu_view_control_window);
	xml->get_widget("patch_description_menuitem", m_menu_patch_description);
	xml->get_widget("patch_destroy_menuitem", m_menu_destroy_patch);
	xml->get_widget("patch_add_plugin_menuitem", m_menu_add_plugin);
	xml->get_widget("patch_add_new_subpatch_menuitem", m_menu_new_subpatch);
	xml->get_widget("patch_add_subpatch_from_file_menuitem", m_menu_load_subpatch);
	xml->get_widget("patch_canvas_scrolledwindow", m_canvas_scrolledwindow);
	xml->get_widget("patch_zoom_scale", m_zoom_slider);
	//xml->get_widget("patch_destroy_button", m_destroy_button);
	xml->get_widget("patch_polyphony_label", m_polyphony_label);
	xml->get_widget("patch_process_checkbutton", m_process_checkbutton);
	
	xml->get_widget_derived("load_plugin_win", m_load_plugin_window);
	xml->get_widget_derived("new_subpatch_win", m_new_subpatch_window);
	xml->get_widget_derived("load_subpatch_win", m_load_subpatch_window);
	xml->get_widget_derived("patch_description_win", m_description_window);

	m_load_plugin_window->set_transient_for(*this);
	m_new_subpatch_window->set_transient_for(*this);
	m_load_subpatch_window->set_transient_for(*this);
	m_description_window->set_transient_for(*this);
	
	m_process_checkbutton->set_active(true);
	m_menu_view_control_window->property_sensitive() = false;

	m_menu_file_save_patch->signal_activate().connect(    sigc::mem_fun(this, &PatchWindow::event_save_patch));
	m_menu_file_save_patch_as->signal_activate().connect( sigc::mem_fun(this, &PatchWindow::event_save_patch_as));
	m_menu_file_close->signal_activate().connect(         sigc::mem_fun(this, &PatchWindow::event_close));
	m_menu_view_control_window->signal_activate().connect(sigc::mem_fun(this, &PatchWindow::show_control_window));
	m_menu_patch_description->signal_activate().connect(  sigc::mem_fun(this, &PatchWindow::show_description_window));
	m_menu_destroy_patch->signal_activate().connect(      sigc::mem_fun(this, &PatchWindow::event_destroy));
	m_menu_add_plugin->signal_activate().connect(         sigc::mem_fun(this, &PatchWindow::event_show_add_plugin_win));
	m_menu_new_subpatch->signal_activate().connect(       sigc::mem_fun(this, &PatchWindow::event_show_new_subpatch_win));
	m_menu_load_subpatch->signal_activate().connect(      sigc::mem_fun(this, &PatchWindow::event_show_load_subpatch_win));
	m_zoom_slider->signal_value_changed().connect(        sigc::mem_fun(this, &PatchWindow::zoom_changed));
	//m_destroy_button->signal_clicked().connect(           sigc::mem_fun(this, &PatchWindow::destroy_clicked));
	m_process_checkbutton->signal_toggled().connect(      sigc::mem_fun(this, &PatchWindow::process_toggled));
}


PatchWindow::~PatchWindow()
{
	hide();
	
	delete m_load_plugin_window;
	delete m_new_subpatch_window;
	delete m_load_subpatch_window;
}


/** Sets the patch controller for this window and initializes everything.
 *
 * This function MUST be called before using the window in any way!
 */
void
PatchWindow::patch_controller(PatchController* pc)
{
	//m_patch_controller = new PatchController(pm, controller);
	m_patch_controller = pc;
	
	m_patch_bay = pc->patch_bay();
	
	assert(m_load_plugin_window != NULL);
	assert(m_new_subpatch_window != NULL);
	assert(m_load_subpatch_window != NULL);

	m_load_plugin_window->patch_controller(m_patch_controller);
	m_new_subpatch_window->patch_controller(m_patch_controller);
	m_load_subpatch_window->patch_controller(m_patch_controller);

	m_canvas_scrolledwindow->add(*m_patch_bay);
	int width, height;
	get_size(width, height);
	m_patch_bay->scroll_to(
		((int)m_patch_bay->width() - width)/2,
		((int)m_patch_bay->height() - height)/2);

	m_patch_bay->show();
	m_canvas_scrolledwindow->show();

	m_patch_controller->patch_bay(m_patch_bay);
	assert(m_patch_controller->model() != NULL);

	/*m_patch_controller->model()->width(1600*2);
	m_patch_controller->model()->height(1200*2);*/

	Glib::ustring title = "Om - ";
	title += m_patch_controller->model()->path();
	property_title() = title;

	char txt[8];
	snprintf(txt, 8, "%d", pc->model()->poly());
	m_polyphony_label->set_text(txt);

	m_description_window->patch_model(pc->model());
}


void
PatchWindow::show_control_window()
{
	if (m_patch_controller != NULL)
		m_patch_controller->show_control_window();
}


void
PatchWindow::show_description_window()
{
	m_description_window->show();
}


void
PatchWindow::zoom_changed()
{
	float z = m_zoom_slider->get_value();
	m_patch_bay->zoom(z);
}

/*
void
PatchWindow::destroy_clicked()
{
	controller->destroy_patch(m_patch_controller->model()->path());
}
*/

void
PatchWindow::process_toggled()
{
	if (!m_enable_signal)
		return;

	if (m_process_checkbutton->get_active()) {
		controller->enable_patch(m_patch_controller->model()->path());
		app->patch_enabled(m_patch_controller->model()->path());
	} else {
		controller->disable_patch(m_patch_controller->model()->path());
		app->patch_disabled(m_patch_controller->model()->path());
	}
}


void
PatchWindow::enabled(bool e)
{
	m_enable_signal = false;
	m_process_checkbutton->set_active(e);
	m_enable_signal = true;
}


void
PatchWindow::event_save_patch()
{
	if (m_patch_controller->model()->filename() == ""
		|| m_patch_controller->model()->filename().substr(0, 1) != "/") {
		event_save_patch_as();
	} else {
		controller->save_patch(m_patch_controller->model(), m_patch_controller->model()->filename(), false);
	}
}


void
PatchWindow::event_save_patch_as()
{
	Gtk::FileChooserDialog dialog(*this, "Save Patch", Gtk::FILE_CHOOSER_ACTION_SAVE);
	
	Gtk::VBox* box = dialog.get_vbox();
	Gtk::Label warning("Warning:  Recursively saving will overwrite any subpatch files \
		without confirmation.");
	box->pack_start(warning, false, false, 2);
	Gtk::CheckButton recursive_checkbutton("Recursively save all subpatches");
	box->pack_start(recursive_checkbutton, false, false, 0);
	recursive_checkbutton.show();
			
	dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	dialog.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_OK);	
	
	int result = dialog.run();
	bool recursive = recursive_checkbutton.get_active();
	
	assert(result == Gtk::RESPONSE_OK || result == Gtk::RESPONSE_CANCEL || result == Gtk::RESPONSE_NONE);
	
	if (result == Gtk::RESPONSE_OK) {	
		string filename = dialog.get_filename();
		if (filename.length() < 4 || filename.substr(filename.length()-3) != ".om")
			filename += ".om";
			
		bool confirm = false;
		std::fstream fin;
		fin.open(filename.c_str(), std::ios::in);
		if (fin.is_open()) {  // File exists
			string msg = "File already exists!  Are you sure you want to overwrite ";
			msg += filename + "?";
			Gtk::MessageDialog confirm_dialog(*this,
				msg, false, Gtk::MESSAGE_WARNING, Gtk::BUTTONS_YES_NO, true);
			if (confirm_dialog.run() == Gtk::RESPONSE_YES)
				confirm = true;
			else
				confirm = false;
		} else {  // File doesn't exist
			confirm = true;
		}
		fin.close();
		
		if (confirm) {
			controller->save_patch(m_patch_controller->model(), filename, recursive);
			m_patch_controller->model()->filename(filename);
		}
	}
}


void
PatchWindow::show()
{
	if (m_position_stored)
		move(m_x, m_y);
	Gtk::Window::show();
	m_is_visible = true;
}


void
PatchWindow::hide()
{
	m_position_stored = true;
	get_position(m_x, m_y);
	Gtk::Window::hide();
	m_is_visible = false;
}


bool
PatchWindow::on_delete_event(GdkEventAny* ev)
{
	m_is_visible = false;
	app->patch_window_hidden(m_patch_controller->model()->path());
	return false; // destroy window
}


void
PatchWindow::event_close()
{
	hide();
	app->patch_window_hidden(m_patch_controller->model()->path());
}


void
PatchWindow::event_destroy()
{
	controller->destroy_patch(m_patch_controller->model()->path());
}


void
PatchWindow::event_show_add_plugin_win()
{
	m_load_plugin_window->show();
	m_load_plugin_window->raise();
}


void
PatchWindow::event_show_new_subpatch_win()
{
	m_new_subpatch_window->show();
	m_new_subpatch_window->raise();
}


void
PatchWindow::event_show_load_subpatch_win()
{
	m_load_subpatch_window->show();
	m_load_subpatch_window->raise();
}


} // namespace OmGtk
