/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "OmGtkApp.h"
#include <cassert>
#include <string>
#include <fstream>
#include <libgnomecanvasmm.h>
#include "config.h"
#include "OmGtk.h"
#include "OmGtkObject.h"
#include "OmModule.h"
#include "ControlPanel.h"
#include "SubpatchModule.h"
#include "OmPatchBayArea.h"
#include "PatchController.h"
#include "LoadPluginWindow.h"
#include "PatchWindow.h"
#include "PatchController.h"
#include "NewPatchWindow.h"
#include "LoadPatchWindow.h"
#include "MessagesWindow.h"
#include "ConfigWindow.h"
#include "Controller.h"
#include "GtkClientHooks.h"
#include "GladeFactory.h"
#include "OmPath.h"
#include "PatchModel.h"
#ifdef HAVE_LASH
#include "LashController.h"
#endif
using std::cerr; using std::cout; using std::endl;
using std::string;

namespace OmGtk {

class OmPort;

OmGtkApp::OmGtkApp()
: m_enable_signal(true),
  m_current_shown_patch(NULL),
  m_root_patch_controller(new PatchController(new PatchModel("", 1), controller, NULL, false))
{
	Glib::RefPtr<Gnome::Glade::Xml> glade_xml = glade_factory->new_glade_reference();

	glade_xml->get_widget("main_win", m_main_window);
	glade_xml->get_widget_derived("new_patch_win", m_new_patch_window);
	glade_xml->get_widget_derived("load_patch_win", m_load_patch_window);
	glade_xml->get_widget_derived("config_win", m_config_window);
	glade_xml->get_widget_derived("main_patches_treeview", m_patches_treeview);
	glade_xml->get_widget("about_win", m_about_window);
	
	m_config_window->state_manager(state_manager);

	glade_xml->get_widget_derived("messages_win", m_messages_window);
	
	glade_xml->get_widget("main_file_open_patch_menuitem", m_menu_file_open_patch);
	glade_xml->get_widget("main_file_new_patch_menuitem", m_menu_file_new_patch);
	glade_xml->get_widget("main_file_load_session_menuitem", m_menu_file_load_session);
	glade_xml->get_widget("main_file_save_session_as_menuitem", m_menu_file_save_session_as);
	glade_xml->get_widget("main_file_configuration_menuitem", m_menu_file_configuration);
	glade_xml->get_widget("main_file_quit_nokill_menuitem", m_menu_file_quit);
	glade_xml->get_widget("main_file_quit_and_kill_menuitem", m_menu_file_quit_and_kill);
	glade_xml->get_widget("help_about_menuitem", m_menu_help_about);
	
	glade_xml->get_widget("main_process_checkbutton", m_process_checkbutton);
	glade_xml->get_widget("about_close_button", m_about_close_button);
	glade_xml->get_widget("main_patches_controls_hpane", m_patches_controls_hpane);
	glade_xml->get_widget("main_controls_scrollwin", m_controls_scrollwin);
	glade_xml->get_widget("engine_error_dialog", m_engine_error_dialog);
	glade_xml->get_widget("engine_error_dialog_close_button", m_engine_error_close_button);
	glade_xml->get_widget("main_controls_viewport", m_controls_viewport);
	glade_xml->get_widget("main_statusbar", m_status_bar);

	m_main_window->show();

	m_patch_treestore = Gtk::TreeStore::create(m_patch_tree_columns);
	m_patches_treeview->set_app(this);
	m_patches_treeview->set_model(m_patch_treestore);
	Gtk::TreeViewColumn* name_col = Gtk::manage(new Gtk::TreeViewColumn("Patch", m_patch_tree_columns.name_col));
	Gtk::TreeViewColumn* enabled_col = Gtk::manage(new Gtk::TreeViewColumn("Run", m_patch_tree_columns.enabled_col));
	Gtk::TreeViewColumn* visible_col = Gtk::manage(new Gtk::TreeViewColumn("Visible", m_patch_tree_columns.visible_col));
	name_col->set_resizable(true);
	name_col->set_expand(true);

	m_patches_treeview->append_column(*name_col);
	m_patches_treeview->append_column(*enabled_col);
	m_patches_treeview->append_column(*visible_col);
	Gtk::CellRendererToggle* enabled_renderer = dynamic_cast<Gtk::CellRendererToggle*>(
		m_patches_treeview->get_column_cell_renderer(1));
	Gtk::CellRendererToggle* visible_renderer = dynamic_cast<Gtk::CellRendererToggle*>(
		m_patches_treeview->get_column_cell_renderer(2));
	enabled_renderer->property_activatable() = true;
	visible_renderer->property_activatable() = true;

	// Idle callback, process events and whatnot
	// Gtk refreshes at priority G_PRIORITY_HIGH_IDLE+20
	Glib::signal_timeout().connect(
		sigc::mem_fun(this, &OmGtkApp::idle_callback), 20, G_PRIORITY_HIGH_IDLE+10);
	
	m_patch_tree_selection = m_patches_treeview->get_selection();
	
	m_menu_file_new_patch->signal_activate().connect(      sigc::mem_fun(this, &OmGtkApp::event_new_patch));
	m_menu_file_open_patch->signal_activate().connect(     sigc::mem_fun(this, &OmGtkApp::event_open_patch));
	m_menu_file_load_session->signal_activate().connect(   sigc::mem_fun(this, &OmGtkApp::event_load_session));
	m_menu_file_save_session_as->signal_activate().connect(sigc::mem_fun(this, &OmGtkApp::event_save_session_as));
	m_menu_file_configuration->signal_activate().connect(  sigc::mem_fun(this, &OmGtkApp::event_configuration));
	m_menu_file_quit->signal_activate().connect(           sigc::mem_fun(this, &OmGtkApp::quit));
	m_menu_file_quit_and_kill->signal_activate().connect(  sigc::mem_fun(this, &OmGtkApp::event_quit_and_kill));
	m_process_checkbutton->signal_toggled().connect(       sigc::mem_fun(this, &OmGtkApp::event_process_toggled));
	m_menu_help_about->signal_activate().connect(          sigc::mem_fun(this, &OmGtkApp::event_show_about));
	m_patch_tree_selection->signal_changed().connect(      sigc::mem_fun(this, &OmGtkApp::event_patch_selected));   
	m_patches_treeview->signal_row_activated().connect(    sigc::mem_fun(this, &OmGtkApp::event_patch_activated));
	m_about_close_button->signal_clicked().connect(        sigc::mem_fun(m_about_window, &Gtk::Window::hide));
	m_engine_error_close_button->signal_clicked().connect( sigc::mem_fun(m_engine_error_dialog, &Gtk::Dialog::hide));
	
	enabled_renderer->signal_toggled().connect(sigc::mem_fun(this, &OmGtkApp::event_patch_enabled_toggled));
	visible_renderer->signal_toggled().connect(sigc::mem_fun(this, &OmGtkApp::event_patch_visible_toggled));
	
	m_patches_treeview->columns_autosize();
	Gtk::Requisition box_size;
	m_patches_treeview->size_request(box_size);
	int width = box_size.width;
	m_patches_controls_hpane->property_position() = width+30;
}


OmGtkApp::~OmGtkApp()
{
	//delete m_load_plugin_window;
	delete m_root_patch_controller;
}


void
OmGtkApp::error_message(const string& str)
{
	m_messages_window->post(str);
	m_messages_window->show();
	m_messages_window->raise();
}


void
OmGtkApp::engine_enabled(bool e)
{
	m_enable_signal = false;
	m_process_checkbutton->set_active(e);
	m_enable_signal = true;
}


void
OmGtkApp::patch_enabled(const string& path)
{
	m_enable_signal = false;
	
	PatchController* pc = patch(path);
	
	if (pc != NULL) {
		pc->enabled(true);
		Gtk::TreeModel::iterator i = find_patch(m_patch_treestore->children(), path);
		if (i != m_patch_treestore->children().end())
			(*i)[m_patch_tree_columns.enabled_col] = true;
	} else {
		cerr << "[OmGtkApp::patch_enabled] Can not find patch " << path << "." << endl;
	}

	m_enable_signal = true;
}


void
OmGtkApp::patch_disabled(const string& path)
{
	m_enable_signal = false;
	
	PatchController* pc = patch(path);
	
	if (pc != NULL) {
		pc->enabled(false);
		Gtk::TreeModel::iterator i = find_patch(m_patch_treestore->children(), path);
		if (i != m_patch_treestore->children().end())
			(*i)[m_patch_tree_columns.enabled_col] = false;
	} else {
		cerr << "[OmGtkApp::patch_disabled] Can not find patch " << path << "." << endl;
	}

	m_enable_signal = true;
}

void
OmGtkApp::patch_renamed(const string& old_path, const string& new_path)
{
	// FIXME: This is gross, PatchControllers really shouldn't be stored in two places
	
	PatchController* patch_controller = NULL;
	
	// Update all paths (inc subpatches) in the patch map
	for (map<string,PatchController*>::iterator mi = m_patches.begin();
			mi != m_patches.end(); ) {
		map<string,PatchController*>::iterator next = mi;
		++next;
		PatchController* pc = NULL;
		pc = (*mi).second;
		if (pc->model()->path() == old_path) { // the patch itself
			patch_controller = pc;
			m_patches.erase(mi);
			m_patches[new_path] = pc;
		} else if (pc->model()->path().substr(0, old_path.length()) == old_path) {
			string new_sub_path = new_path +"/"+ pc->model()->path().substr(old_path.length()+1);
			m_patches.erase(mi);
			m_patches[new_sub_path] = pc;
			pc->path(new_sub_path);
		}
		mi = next;
	}
	
	if (patch_controller == NULL) {
		cerr << "[OmGtkApp::patch_renamed] Can not find patch " << old_path << endl;
		return;
	}	
	
	Gtk::TreeModel::iterator ti = find_patch(m_patch_treestore->children(), old_path);
	if (ti != m_patch_treestore->children().end()) {
		(*ti)[m_patch_tree_columns.name_col] = OmPath::name(new_path);
	} else {
		cerr << "[OmGtkApp::patch_renamed] Can not find patch " << old_path << " in treeview." << endl;
	}

	patch_controller->path(new_path);
	assert(patch_controller->model()->path() == new_path);
	assert(patch_controller->module()->node_model()->path() == new_path);
	assert(patch_controller->module()->patch_model()->path() == new_path);
}


void
OmGtkApp::patch_window_hidden(const string& path)
{
	PatchController* pc = patch(path);
	
	if (pc != NULL) {
		Gtk::TreeModel::iterator i = find_patch(m_patch_treestore->children(), path);
		if (i != m_patch_treestore->children().end())
			(*i)[m_patch_tree_columns.visible_col] = false;
	} else {
		cerr << "[OmGtkApp::patch_window_hidden] Can not find patch " << path << "." << endl;
	}
}


Gtk::TreeModel::iterator
OmGtkApp::find_patch(Gtk::TreeModel::Children root, const string& path)
{
	PatchController* pc = NULL;
	
	for (Gtk::TreeModel::iterator c = root.begin(); c != root.end(); ++c) {
		pc = (*c)[m_patch_tree_columns.patch_controller_col];
		if (pc->model()->path() == path) {
			return c;
		} else if ((*c)->children().size() > 0) {
			Gtk::TreeModel::iterator ret = find_patch(c->children(), path);
			if (ret != c->children().end())
				return ret;
		}
	}
	return root.end();
}


OmModule*
OmGtkApp::module(const string& path) const
{
	// Is a normal module in a patch?
	PatchController* parent = patch(OmPath::parent(path));
	OmModule* module = NULL;
	if (parent != NULL) {
		module = parent->patch_bay()->find_module(OmPath::name(path));
		if (module != NULL)
			return module;
	}

	// Is a top level patch's (invisible) subpatch module?
	parent = patch(path);
	if (parent != NULL)
		return parent->module();

	return NULL;
}


OmPort*
OmGtkApp::port(const string& path) const
{
	OmModule* parent = module(OmPath::parent(path));
	if (parent != NULL) {
		OmPort* port = parent->port(OmPath::name(path));
		return port;
	}
	return NULL;
}


void
OmGtkApp::add_patch(PatchModel* pm, bool show_window)
{
	PatchController* parent_pc = NULL;
	string parent_path = OmPath::parent(pm->path());
	if (parent_path != "") {
		parent_pc = patch(parent_path);
		if (parent_pc != NULL)
			pm->parent(parent_pc->model());
	} else {
		parent_pc = m_root_patch_controller;
	}
	if (parent_pc == NULL) {
		cerr << "[NewPatchEvent] Did not find parent of " << pm->path()
		<< ".  Patch will not appear.  This is not good.." << endl;
		return;
	}
	
	PatchController* pc = new PatchController(pm, controller, parent_pc);
	m_patches[pm->path()] = pc;
	
	if (pm->x() == 0 && pm->y() == 0) {
		int x, y;
		parent_pc->get_new_module_location(x, y);
		pm->x(x);
		pm->y(y);
	}
	SubpatchModule* spm = new SubpatchModule(parent_pc->patch_bay(), pc);
	parent_pc->new_subpatch(spm);
	pc->module(spm);

	// Add to patch selector
	if (pm->parent() == NULL) {
		Gtk::TreeModel::iterator iter = m_patch_treestore->append();
		Gtk::TreeModel::Row row = *iter;
		row[m_patch_tree_columns.name_col] = OmPath::name(pm->path());
		row[m_patch_tree_columns.enabled_col] = true;
		row[m_patch_tree_columns.visible_col] = show_window;
		row[m_patch_tree_columns.patch_controller_col] = pc;
	} else {
		Gtk::TreeModel::Children children = m_patch_treestore->children();
		Gtk::TreeModel::iterator c = find_patch(children, pm->parent()->path());
		
		if (c != children.end()) {
			Gtk::TreeModel::iterator iter = m_patch_treestore->append(c->children());
			Gtk::TreeModel::Row row = *iter;
			row[m_patch_tree_columns.name_col] = OmPath::name(pm->path());
			row[m_patch_tree_columns.enabled_col] = true;
			row[m_patch_tree_columns.visible_col] = show_window;
			row[m_patch_tree_columns.patch_controller_col] = pc;
		}
	}

	if (show_window)
		pc->window()->show();
}


void
OmGtkApp::destroy_patch(const string& path)
{
	map<string, PatchController*>::iterator w = m_patches.find(path);
	
	// Remove from patch tree
	Gtk::TreeModel::iterator i = find_patch(m_patch_treestore->children(), path);
	if (i != m_patch_treestore->children().end())
		m_patch_treestore->erase(i);

	// Remove from patches list and destroy
	if (w != m_patches.end()) {
		if (m_current_shown_patch == (*w).second) {
			m_current_shown_patch->claim_control_panel();
			m_current_shown_patch = NULL;
		}
		m_patches.erase(w);
		delete (*w).second;
	} else {
		cerr << "[OmGtkApp::destroy_patch_window] Unable to find patch window " << path << endl;
	}
}


bool
OmGtkApp::idle_callback()
{
	assert(controller != NULL);
	assert(controller->client_hooks() != NULL);
	
	controller->client_hooks()->process_events();

#ifdef HAVE_LASH
	if (lash_controller->enabled())
		lash_controller->process_events();
#endif
	
	return true;
}


void
OmGtkApp::new_plugin(const PluginModel* pi)
{
	LoadPluginWindow* lpw = NULL;
	
	// I don't like this, it's too slow (this function gets called a LOT of
	// times VERY fast on initial connection), but it's needed to avoid
	// duplicate plugins appearing :/
	// Update: fixed engine to not load a plugin twice, still possible to send them to this
	// client multiple times if something screws up though...
	/*for (list<const PluginModel*>::iterator i = m_plugin_models.begin(); i != m_plugin_models.end(); ++i)
		if ((*i)->lib_name() == pi->lib_name() && (*i)->plug_label() == pi->plug_label()) {
			cerr << "Duplicate plugin received (" << pi->lib_name() << ":" << pi->plug_label() << endl;
			return;
		}
		*/
		
	m_plugin_models.push_back(pi);
	for (map<string, PatchController*>::iterator i = m_patches.begin(); i != m_patches.end(); ++i) {
		lpw = (*i).second->window()->load_plugin_window();
		if (lpw != NULL && lpw->has_shown())
			lpw->add_plugin(pi);
	}

	//cout << "Adding plugin " << pi->plug_label() << " (" << m_plugin_models.size() << " plugins)" << endl;
}


/******** Event Handlers ************/



void
OmGtkApp::event_new_patch()
{
	m_new_patch_window->show();
}


void
OmGtkApp::event_open_patch()
{
	m_load_patch_window->show();
}


void
OmGtkApp::event_load_session()
{
	Gtk::FileChooserDialog* dialog
		= new Gtk::FileChooserDialog(*m_main_window, "Load Session", Gtk::FILE_CHOOSER_ACTION_OPEN);
	
	dialog->add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	dialog->add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_OK);	
	int result = dialog->run();
	string filename = dialog->get_filename();
	delete dialog;

	cout << result << endl;
	
	assert(result == Gtk::RESPONSE_OK || result == Gtk::RESPONSE_CANCEL || result == Gtk::RESPONSE_NONE);
	
	if (result == Gtk::RESPONSE_OK)
		//state_manager->load_session(filename);
		controller->load_session(filename);
}


void
OmGtkApp::event_save_session_as()
{
	Gtk::FileChooserDialog dialog(*m_main_window, "Save Session", Gtk::FILE_CHOOSER_ACTION_SAVE);
	
	/*
	Gtk::VBox* box = dialog.get_vbox();
	Gtk::Label warning("Warning:  Recursively saving will overwrite any subpatch files \
		without confirmation.");
	box->pack_start(warning, false, false, 2);
	Gtk::CheckButton recursive_checkbutton("Recursively save all subpatches");
	box->pack_start(recursive_checkbutton, false, false, 0);
	recursive_checkbutton.show();
	*/		
	dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	dialog.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_OK);	
	
	int result = dialog.run();
	//bool recursive = recursive_checkbutton.get_active();
	
	assert(result == Gtk::RESPONSE_OK || result == Gtk::RESPONSE_CANCEL || result == Gtk::RESPONSE_NONE);
	
	if (result == Gtk::RESPONSE_OK) {	
		string filename = dialog.get_filename();
		if (filename.length() < 11 || filename.substr(filename.length()-10) != ".omsession")
			filename += ".omsession";
			
		bool confirm = false;
		std::fstream fin;
		fin.open(filename.c_str(), std::ios::in);
		if (fin.is_open()) {  // File exists
			string msg = "File already exists!  Are you sure you want to overwrite ";
			msg += filename + "?";
			Gtk::MessageDialog confirm_dialog(*m_main_window,
				msg, false, Gtk::MESSAGE_WARNING, Gtk::BUTTONS_YES_NO, true);
			if (confirm_dialog.run() == Gtk::RESPONSE_YES)
				confirm = true;
			else
				confirm = false;
		} else {  // File doesn't exist
			confirm = true;
		}
		fin.close();
		
		if (confirm) {
			controller->save_session(filename);
		}
	}
}


void
OmGtkApp::event_configuration()
{
	m_config_window->show();
}


void
OmGtkApp::quit()
{
	m_main_window->hide();
}

void
OmGtkApp::event_quit_and_kill()
{
	controller->quit();
	m_main_window->hide();
}


void
OmGtkApp::event_process_toggled() 
{
	if (m_enable_signal) {
		if (m_process_checkbutton->get_active())
			controller->enable_engine();
		else
			controller->disable_engine();
	}
}


void
OmGtkApp::event_patch_selected()
{
	Gtk::TreeModel::iterator active = m_patch_tree_selection->get_selected();
	Gtk::TreeModel::Row row = *active;
	PatchController* pc = row[m_patch_tree_columns.patch_controller_col];
	
	if (pc != NULL) {
		if (m_current_shown_patch != NULL)
			m_current_shown_patch->claim_control_panel();
		
		m_controls_viewport->remove();
		
		pc->control_panel()->reparent(*m_controls_viewport);
	
		m_current_shown_patch = pc;
	}
}


/** Show the context menu for the selected patch in the patches treeview.
 */
void
OmGtkApp::show_patch_menu(GdkEventButton* ev)
{
	Gtk::TreeModel::iterator active = m_patch_tree_selection->get_selected();
	if (active) {
		Gtk::TreeModel::Row row = *active;
		PatchController* pc = row[m_patch_tree_columns.patch_controller_col];
		assert (pc != NULL);
		assert(pc->module() != NULL);
		pc->module()->show_menu(ev);
	}
}


void
OmGtkApp::event_patch_activated(const Gtk::TreeModel::Path& path, Gtk::TreeView::Column* col)
{
	Gtk::TreeModel::iterator active = m_patch_treestore->get_iter(path);
	Gtk::TreeModel::Row row = *active;
	PatchController* pc = row[m_patch_tree_columns.patch_controller_col];
	
	row[m_patch_tree_columns.visible_col] = true;
	pc->window()->show();
	pc->window()->raise();
}


void
OmGtkApp::event_patch_enabled_toggled(const Glib::ustring& path_str)
{
	Gtk::TreeModel::Path path(path_str);
	Gtk::TreeModel::iterator active = m_patch_treestore->get_iter(path);
	Gtk::TreeModel::Row row = *active;
	
	PatchController* pc = row[m_patch_tree_columns.patch_controller_col];
	Glib::ustring patch_path = pc->model()->path();
	
	if ( ! pc->model()->enabled()) {
		if (m_enable_signal)
			controller->enable_patch(patch_path);
		pc->enabled(true);
		row[m_patch_tree_columns.enabled_col] = true;
	} else {
		if (m_enable_signal)
			controller->disable_patch(patch_path);
		pc->enabled(false);
		row[m_patch_tree_columns.enabled_col] = false;
	}
}


void
OmGtkApp::event_patch_visible_toggled(const Glib::ustring& path_str)
{
	Gtk::TreeModel::Path path(path_str);
	Gtk::TreeModel::iterator active = m_patch_treestore->get_iter(path);
	Gtk::TreeModel::Row row = *active;
	
	PatchController* pc = row[m_patch_tree_columns.patch_controller_col];
	Glib::ustring patch_path = pc->model()->path();
	
	if ( ! pc->window()->visible()) {
		pc->window()->show();
		pc->window()->raise();
		row[m_patch_tree_columns.visible_col] = true;
	} else {
		pc->window()->hide();
		row[m_patch_tree_columns.visible_col] = false;
	}
}


void
OmGtkApp::event_show_about()
{
	m_about_window->show();
}


} // namespace OmGtk

