/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "OmGtk.h"
#include "PatchModel.h"
#include "PatchController.h"
#include "OSCController.h"
#include "GtkClientHooks.h"
#include "PatchLibrarian.h"
#include "Controller.h"
#include "Loader.h"

namespace OmGtk {


// Controller //


Controller::Controller()
: m_client_hooks(new GtkClientHooks()),
  m_osc_controller(new OSCController(m_client_hooks)),
  m_patch_librarian(new PatchLibrarian(m_osc_controller, m_client_hooks)),
  m_loader(new Loader(m_patch_librarian)),
  m_is_attached(false)
{
	m_loader->launch();
}


Controller::~Controller()
{
	delete m_loader;
	delete m_patch_librarian;
	delete m_osc_controller;
	delete m_client_hooks;
}


/** "Attach" to the Om engine.
 *
 * See documentation for OSCController::attach.
 */
/*
void
Controller::attach(const string& engine_host, int engine_port, int client_port)
{
	m_osc_controller->attach(engine_host, engine_port, client_port);
	m_is_attached = true;
	activate_engine();
	enable_engine();
}*/


/** "Attach" to the Om engine.
 *
 * See documentation OSCController::attach.
 */
void
Controller::attach(const string& engine_url, int client_port)
{
	m_osc_controller->attach(engine_url, client_port);
	m_is_attached = true;
	activate_engine();
	enable_engine();
}

void
Controller::register_client()
{
	int id = m_osc_controller->get_next_request_id();
	m_osc_controller->set_wait_response_id(id);
	m_osc_controller->register_client(id);
	m_osc_controller->wait_for_response();
	cout << "[Controller] Registered with engine" << endl;

	id = m_osc_controller->get_next_request_id();
	// FIXME: wait?
	//m_osc_controller->set_wait_response_id(id);
	m_osc_controller->load_plugins(id);
	//m_osc_controller->wait_for_response();
}

string Controller::engine_url() { return m_osc_controller->engine_url(); }

void Controller::activate_engine()   { m_osc_controller->activate(); }
void Controller::deactivate_engine() { m_osc_controller->deactivate(); }
void Controller::enable_engine()     { m_osc_controller->enable(); }
void Controller::disable_engine()    { m_osc_controller->disable(); }

void Controller::quit() { m_osc_controller->quit(); }

void
Controller::set_control(const string& port_path, float val)
	{ m_osc_controller->set_control(port_path, val); }

void
Controller::set_control(const string& port_path, int voice, float val)
	{ m_osc_controller->set_control(port_path, voice, val); }
	
void
Controller::request_control(const string& port_path)
	{ m_osc_controller->request_control(port_path); }

void
Controller::add_node(NodeModel* nm)
{
	push_added_node(nm);
	m_osc_controller->add_node(nm);
	char temp_buf[16];
	snprintf(temp_buf, 16, "%f", nm->x());
	m_osc_controller->set_metadata(nm->path(), "module-x", temp_buf);
	snprintf(temp_buf, 16, "%f", nm->y());
	m_osc_controller->set_metadata(nm->path(), "module-y", temp_buf);
}


void
Controller::remove_node(const string& node_path)
	{ m_osc_controller->remove_node(node_path); }


void Controller::connect(const string& port1_path, const string& port2_path)
	{ m_osc_controller->connect(port1_path, port2_path); }
	
void Controller::disconnect(const string& port1_path, const string& port2_path)
	{ m_osc_controller->disconnect(port1_path, port2_path); }

void Controller::disconnect_all(const string& node_path)
	{ m_osc_controller->disconnect_all(node_path); }

void
Controller::create_patch(PatchModel* pm)
{ 
	push_added_patch(pm);

	//int id = m_osc_controller->get_next_request_id();
	//m_osc_controller->set_wait_response_id(id);
	m_osc_controller->create_patch(pm);
	if (pm->parent() != NULL) {
	//	m_osc_controller->wait_for_response();
		char temp_buf[16];
		snprintf(temp_buf, 16, "%f", pm->x());
		m_osc_controller->set_metadata(pm->path(), "module-x", temp_buf);
		snprintf(temp_buf, 16, "%f", pm->y());
		m_osc_controller->set_metadata(pm->path(), "module-y", temp_buf);
	}
	m_osc_controller->enable_patch(pm->path());
}


void
Controller::destroy_patch(const string& path)
	{ m_osc_controller->destroy_patch(path); }

void
Controller::rename(const string& path, const string& name)
	{ m_osc_controller->rename(path, name); }
	
void
Controller::enable_patch(const string& path)
	{ m_osc_controller->enable_patch(path); }

void
Controller::disable_patch(const string& path)
	{ m_osc_controller->disable_patch(path); }
	
void
Controller::midi_learn(const string& node_path)
	{ m_osc_controller->midi_learn(node_path); }

void
Controller::set_metadata(const string& obj_path, const string& key, const string& value)
	{ m_osc_controller->set_metadata(obj_path, key, value); }


void
Controller::request_plugins()
	{ m_osc_controller->request_plugins(); }


void
Controller::request_all_objects()
	{ m_osc_controller->request_all_objects(); }

void
Controller::set_patch_path(const string& path)
{
	m_patch_librarian->path(path);
}


/** Load patch in a seperate thread.
 * This will return immediately and the patch will be loaded in the background. */
void
Controller::load_patch(PatchModel* model)
{
	push_added_patch(model);
	m_loader->load_patch(model);
}

void
Controller::load_patch_blocking(PatchModel* model)
{
	m_patch_librarian->load_patch(model);
}


/** Load patch in a seperate thread.
 * This will return immediately and the patch will be saved in the background. */
void
Controller::save_patch(PatchModel* model, const string& filename, bool recursive)
{
	m_loader->save_patch(model, filename, recursive);
}

void
Controller::save_patch_blocking(PatchModel* model, const string& filename, bool recursive)
{
	m_patch_librarian->save_patch(model, filename, recursive);
}

void
Controller::load_session(const string& filename)
{
	m_loader->load_session(filename);
}


void
Controller::save_session(const string& filename)
{
	m_loader->save_session(filename);
}


/** Returns the added node with the given path and removes it from the cache.
 */
NodeModel*
Controller::yank_added_node(const string& path)
{
	NodeModel* nm = NULL;
	
	for (list<NodeModel*>::iterator i = m_added_nodes.begin(); i != m_added_nodes.end(); ++i) {
		if ((*i)->path() == path) {
			nm = *i;
			m_added_nodes.erase(i);
			break;
		}
	}

	return nm;
}


/** Returns the added patch with the given path and removes it from the cache.
 */
PatchModel*
Controller::yank_added_patch(const string& path)
{
	PatchModel* pm = NULL;
	
	for (list<PatchModel*>::iterator i = m_added_patches.begin(); i != m_added_patches.end(); ++i) {
		if ((*i)->path() == path) {
			pm = *i;
			m_added_patches.erase(i);
			break;
		}
	}

	return pm;
}

} // namespace OmGtk
