/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef OSCCONTROLLER_H
#define OSCCONTROLLER_H

#include <cstdlib>
#include <string>
#include <lo/lo.h>
using std::string;


/** Om C++ client library namespace.
 *
 * \ingroup libomclient
 */
namespace LibOmClient {

class NodeModel;
class PresetModel;
class PatchModel;
class OSCListener;
class ClientHooks;

/** \defgroup libomclient Om client library.
 *
 * This library wraps all the low-level OSC communication in nice (optionally
 * blocking) functions, and provides some other things that should be useful to
 * any Om client.
 */


/** OSC Commands.
 *
 * Handles all OSC communication on the "control band".  For the time being,
 * manages the OSCListener which handles the "notification band", but this
 * will change in the future (for complete separation).  See OSC namespace
 * documentation for more details.
 *
 * \ingroup libomclient
 */
class OSCController
{
public:
	OSCController(ClientHooks* const client_hooks);
	OSCController();
	~OSCController();
	
	void attach(const string& engine_url, int client_port = 0);
	void detach();

	void register_client(int id = 0);
	void unregister_client();

	string engine_url() { return m_engine_url; }

	// FIXME: how to use this needs to be documented
	void set_wait_response_id(int id);
	bool wait_for_response();
	int  get_next_request_id() { return m_request_id++; }
	
	// OSCCommands //
	
	// Engine commands
	void load_plugins() { load_plugins(m_request_id++); }
	void load_plugins(int id);
	void activate();
	void deactivate();
	void enable();
	void disable();
	void quit();
			
	// Patch commands
	void create_patch(const PatchModel* pm) { return create_patch(pm, m_request_id++); }
	void create_patch(const PatchModel* pm, int id);
	void add_node(const NodeModel* nm, int id);
	void request_all_objects(int id);
	void rename(const string& old_path, const string& new_name);
	void destroy_patch(const string& patch_path);
	void enable_patch(const string& patch_path);
	void disable_patch(const string& patch_path);
	void add_node(const NodeModel* nm) { add_node(nm, m_request_id++); }
	void remove_node(const string& node_path);
	void connect(const string& port1_path, const string& port2_path);
	void disconnect(const string& port1_path, const string& port2_path);
	void disconnect_all(const string& node_path);
	void set_control(const string& port_path, float val);
	void set_control(const string& port_path, int voice, float val);
	void set_control_slow(const string& port_path, float val);
	void midi_learn(const string& node_path);

	// Metadata commands
	void set_metadata(const string& obj_path, const string& key, const string& value);

	// Requests //
	
	void ping(int id);
	void request_control(const string& port_path);
	void request_plugins();
	void request_all_objects() { request_all_objects(m_request_id++); }
	

	// Utility commands, etc (ie things that don't map directly to OSC) //
	void set_all_metadata(const NodeModel* nm);
	void set_preset(const string& patch_path, const PresetModel* const pm);

private:
	// Prevent copies
	OSCController(const OSCController& copy)            { exit(EXIT_FAILURE); }
	OSCController& operator=(const OSCController& copy) { exit(EXIT_FAILURE); }
	
	void start_listen_thread(int client_port = 0);
	
	ClientHooks* m_client_hooks;
	OSCListener* m_osc_listener;

	lo_server_thread m_st;
	lo_address       m_engine_addr;
	string           m_engine_url;

	int m_request_id;

	bool m_blocking;
	bool m_waiting_for_response;
	int  m_wait_response_id;
	bool m_response_received;
	bool m_wait_response_was_affirmative;
	
	pthread_mutex_t m_response_mutex;
	pthread_cond_t  m_response_cond;
	
	// liblo callbacks
	static void error_cb(int num, const char* msg, const char* path);	
	static int  generic_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* user_data);
	static int  unknown_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* osc_receiver);
	
	inline static int om_response_ok_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_om_response_ok_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int om_response_error_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_om_response_error_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
};

inline int
OSCController::om_response_ok_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((OSCController*)comm)->m_om_response_ok_cb(path, types, argv, argc, data);
}

inline int
OSCController::om_response_error_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((OSCController*)comm)->m_om_response_error_cb(path, types, argv, argc, data);
}

} // namespace LibOmClient

#endif // OSCCONTROLLER_H
