/*
Sparse matrix functionality for octave, based on the SuperLU package  
Copyright (C) 1998-2000 Andy Adler

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with Octave; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.
   
In addition to the terms of the GPL, you are permitted to link
this program with any Open Source program, as defined by the
Open Source Initiative (www.opensource.org)

$Id: sparse_full.cc,v 1.12 2004/08/02 16:00:07 aadler Exp $

*/

#include "make_sparse.h"
using namespace std;


//
// full
//
DEFUN_DLD (full, args, ,
    "-*- texinfo -*-\n\
@deftypefn {Loadable Function} {@var{FM} =} full (@var{SM})\n\
 returns a full storage matrix from a sparse one\n\
@seealso{sparse}\n\
@end deftypefn")
{
  octave_value_list retval;

  if (args.length() < 1) {
     print_usage ("full");
     return retval;
  }

  if (args(0).type_name () == "sparse") {
     const octave_value& rep = args(0).get_rep ();

     Matrix M = ((const octave_sparse&) rep) . matrix_value ();
     retval(0)= M;
  } else
  if (args(0).type_name () == "complex_sparse") {
     const octave_value& rep = args(0).get_rep ();

     ComplexMatrix M = ((const octave_sparse&) rep) . complex_matrix_value ();
     retval(0)= M;
  } else
  if (args(0).is_real_type()) {
     retval(0)= args(0).matrix_value();
  } else
  if (args(0).is_complex_type()) {
     retval(0)= args(0).complex_matrix_value();
  } else
    gripe_wrong_type_arg ("full", args(0));

  return retval;
}

//
// is_sparse and friends
//
DEFUN_DLD (is_sparse, args, ,
     "-*- texinfo -*-\n\
@deftypefn {Loadable Function} {@var{retval} =} is_sparse (@var{X})\n\
Returns true (ie. 1) if X is a matrix with sparse storage@*\n\
Returns false (ie. 0) otherwise\n\
@seealso{is_real_sparse, is_complex_sparse, sparse}\n\
@end deftypefn")
{
  octave_value_list retval;

  if (args.length() != 1) {
     print_usage ("is_sparse");
     return retval;
  }

  if (args(0).type_name () == "sparse" || 
      args(0).type_name () == "complex_sparse") {
     retval(0)= 1.0;
  } else
     retval(0)= 0.0;

  return retval;
}

DEFUN_DLD (is_real_sparse, args, ,
    "-*- texinfo -*-\n\
@deftypefn {Loadable Function} {@var{retval}=} is_real_sparse (@var{X})\n\
Returns true (ie. 1) if @var{X} is a real matrix with sparse storage@*\n\
Returns false (ie. 0) otherwise\n\
@seealso{is_sparse, is_complex_sparse, sparse}\n\
@end deftypefn")
{
  octave_value_list retval;

  if (args.length() != 1) {
     print_usage ("is_real_sparse");
     return retval;
  }

  if (args(0).type_name () == "sparse" ) {
     retval(0)= 1.0;
  } else
     retval(0)= 0.0;

  return retval;
}

DEFUN_DLD (is_complex_sparse, args, ,
    "-*- texinfo -*-\n\
@deftypefn {Loadable Function} {@var{retval}=} is_complex_sparse (@var{X})\n\
Returns true (ie. 1) if @var{X }is a complex matrix with sparse storage@*\n\
Returns false (ie. 0) otherwise\n\
@seealso{is_sparse, is_real_sparse, sparse}\n\
@end deftypefn")
{
  octave_value_list retval;

  if (args.length() != 1) {
     print_usage ("is_complex_sparse");
     return retval;
  }

  if (args(0).type_name () == "complex_sparse" ) {
     retval(0)= 1.0;
  } else
     retval(0)= 0.0;

  return retval;
}

//
// nnz
//
DEFUN_DLD (nnz, args, ,
   "-*- texinfo -*-\n\
@deftypefn {Loadable Function} {@var{scalar} =} nnz (@var{SM})\n\
returns number of non zero elements in SM\n\
@seealso{sparse}\n\
@end deftypefn")
{
  octave_value_list retval;

  if (args.length() < 1) {
     print_usage ("nnz");
     return retval;
  }

  if (args(0).type_name () == "sparse") {
     const octave_value& rep = args(0).get_rep ();

     retval(0)= (double) ((const octave_sparse&) rep) . nnz ();
  } else
  if (args(0).type_name () == "complex_sparse") {
     const octave_value& rep = args(0).get_rep ();

     retval(0)= (double) ((const octave_complex_sparse&) rep) . nnz ();
  } else
  if (args(0).type_name () == "complex matrix") {
     const ComplexMatrix M = args(0).complex_matrix_value();
     int nnz= 0;
     for( int i=0; i< M.rows(); i++)
        for( int j=0; j< M.cols(); j++)
           if (M(i,j)!=0.) nnz++;
     retval(0)= (double) nnz;
  } else
  if (args(0).type_name () == "matrix") {
     const Matrix M = args(0).matrix_value();
     int nnz= 0;
     for( int i=0; i< M.rows(); i++)
        for( int j=0; j< M.cols(); j++)
           if (M(i,j)!=0.) nnz++;
     retval(0)= (double) nnz;
  } else
  if (args(0).type_name () == "scalar") {
     retval(0)= args(0).scalar_value() != 0.0 ? 1.0 : 0.0;
  } else
  if (args(0).type_name () == "complex scalar") {
     retval(0)= args(0).complex_value() != 0.0 ? 1.0 : 0.0;
  } else
     gripe_wrong_type_arg ("nnz", args(0));

  return retval;
}

//
// spfind - find elements in sparse matrices
//
// PKG_ADD: dispatch ("find", "spfind", "sparse")
// PKG_ADD: dispatch ("find", "spfind", "complex_sparse")
DEFUN_DLD (spfind, args, nargout ,
    "-*- texinfo -*-\n\
@deftypefn {Loadable Function} {[...] =} spfind (...)\n\
SPFIND: a sparse version of the find operator\n\
@enumerate\n\
    @item\n\
@var{x }= spfind( @var{a })\n\
    @itemize @w\n\
is analagous to @var{x}= find(@var{A}(:))@*\n\
where @var{A}= full(@var{a})\n\
    @end itemize\n\
    @item\n\
[@var{i},@var{j},@var{v},@var{nr},@var{nc}] = spfind( @var{a} )\n\
    @itemize @w\n\
returns column vectors @var{i},@var{j},@var{v} such that@*\n\
@var{a}= sparse(@var{i},@var{j},@var{v},@var{nr},@var{nc})\n\
    @end itemize\n\
@end enumerate\n\
@seealso{sparse}\n\
@end deftypefn")
{
   octave_value_list retval;
   int nargin = args.length ();

   if (nargin != 1) {
      print_usage ("spfind");
      return retval;
   }
      
   const octave_value& rep = args(0).get_rep ();

   if( args(0).type_name () == "sparse" ) {

      const octave_sparse& v = ((const octave_sparse&) rep);
      retval= v.find();

   } else
   if( args(0).type_name () == "complex_sparse" ) {

      const octave_complex_sparse& v = ((const octave_complex_sparse&) rep);
      retval= v.find();

   } else {
     gripe_wrong_type_arg ("spfind", args(0));
   }

   if (nargout == 1 || nargout ==0) { // only find location as fortran index
      octave_value_list tmp;
      tmp(0) = retval(0) + (retval(1)-1)*retval(3);
      retval= tmp;
   }

   return retval;
}

/*
 * $Log: sparse_full.cc,v $
 * Revision 1.12  2004/08/02 16:00:07  aadler
 * fixed bug in spfind -> will now return fortran_vector index for nargout ==1
 *
 * Revision 1.11  2004/07/27 16:05:55  aadler
 * simplify find
 *
 * Revision 1.10  2004/03/05 14:30:30  pkienzle
 * Add dispatch commands for spfind
 *
 * Revision 1.9  2003/12/22 15:09:25  pkienzle
 * Use properties tests (real/complex) rather than type tests
 * (matrix/complex matrix) so that full automatically generalizes
 * to other types which can be converted via matrix_value() to
 * full matrices;  use correct function name in usage.
 *
 * Revision 1.8  2003/10/18 01:13:00  aadler
 * texinfo for documentation strings
 *
 * Revision 1.7  2003/07/23 17:21:54  aadler
 * modified help files
 *
 * Revision 1.6  2002/12/11 17:19:32  aadler
 * sparse .^ scalar operations added
 * improved test suite
 * improved documentation
 * new is_sparse
 * new spabs
 *
 * Revision 1.5  2002/11/27 04:46:42  pkienzle
 * Use new exception handling infrastructure.
 *
 * Revision 1.4  2002/01/04 15:53:57  pkienzle
 * Changes required to compile for gcc-3.0 in debian hppa/unstable
 *
 * Revision 1.3  2001/11/16 03:09:42  aadler
 * added spsum.m, is_sparse, is_real_sparse, is_complex_sparse
 *
 * Revision 1.2  2001/10/12 02:24:28  aadler
 * Mods to fix bugs
 * add support for all zero sparse matrices
 * add support fom complex sparse inverse
 *
 * Revision 1.4  2001/09/23 17:46:12  aadler
 * updated README
 * modified licence to GPL plus link to opensource programmes
 *
 * Revision 1.3  2001/04/08 20:18:19  aadler
 * complex sparse support
 *
 * Revision 1.2  2001/02/27 03:01:52  aadler
 * added rudimentary complex matrix support
 *
 * Revision 1.1  2000/12/18 03:31:16  aadler
 * Split code to multiple files
 * added sparse inverse
 *
 */
