(* ocamlgsl - OCaml interface to GSL                        *)
(* Copyright (©) 2002-2005 - Olivier Andrieu                *)
(* distributed under the terms of the GPL version 2         *)


let words_list =
  let lex = Genlex.make_lexer [] in
  let rec parse_s acc = parser
    | [< ' Genlex.Ident i; st >] -> parse_s (i::acc) st
    | [< '_; st >] -> failwith "wrong token"
    | [<>] -> List.rev acc in
  fun s ->
    parse_s [] (lex (Stream.of_string s))

let split ?(collapse=false) c s = 
  let len = String.length s in
  let rec proc accu n = 
    let n' = 
      try String.index_from s n c 
      with Not_found -> len in
    let accu' = 
      if collapse && n' = n
      then accu
      else (String.sub s n (n' - n)) :: accu in
    if n' = len 
    then List.rev accu'
    else proc accu' (succ n')
  in
  proc [] 0


(** Quotation for externals :
   << fun1,arg1,arg2 >> -> 
      external fun1 : arg1 -> arg2 = "fun1"
   << fun1@fun_c,arg1,arg2 >> -> 
      external fun1 : arg1 -> arg2 = "fun_c"
   << fun1@fun_c@fun_f,float,float >> -> 
      external fun1 : float -> float = "fun_c" "fun_f" "float"
*)
let ext_quot = 
  let b = Buffer.create 256 in
  fun expr_flag str ->
    if not expr_flag
    then failwith "ext_quot: expression only quotation" ;
    Buffer.clear b ;
    match split ',' str with
    | [] -> failwith "ext_quot: empty quotation"
    | _ :: [] -> failwith "ext_quot: no arguments"
    | name_r :: (arg1 :: argr as args) ->
	let (name, name_c, name_float) = match split '@' name_r with
	| name :: [] -> name, name, ""
	| name :: name_c :: [] -> name, name_c, ""
	| name :: name_c :: name_f :: _ -> name, name_c, name_f
	| [] -> failwith "ext_quot: too many C function names"
	in
	begin
	  Printf.bprintf b "external %s : " name ;
	  Printf.bprintf b "%s" arg1 ;
	  List.iter (fun a -> Printf.bprintf b " -> %s" a) argr ;
	  Printf.bprintf b "\n    = " ;
	  if List.length args > 6
	  then Printf.bprintf b "\"%s_bc\" " name_c ;
	  if (List.for_all ((=) "float") args) && name_float <> ""
	  then (
	    if List.length args <= 6 
	    then Printf.bprintf b "\"%s\"" name_c ;
	    Printf.bprintf b " \"%s\" \"float\"" name_float )
	  else Printf.bprintf b "\"%s\"" name_c ;
	  Printf.bprintf b "\n\n"
	end ;
	Buffer.contents b
    

let sf_quot =
  let b = Buffer.create 256 in
  fun expr_flag str ->
    if not expr_flag
    then failwith "sf_quot: expression only quotation" ;
    let wl = words_list str in
    match wl with
    | [] -> failwith "sf_quot: empty quotation"
    | _ :: [] -> failwith "sf_quot: no arguments"
    | name :: args ->
	let quot = 
	  Buffer.clear b ;
	  Printf.bprintf b "%s@ml_gsl_sf_%s%s," name name
	  (if List.for_all ((=) "float") args 
	  then "@" ^ "gsl_sf_" ^ name
	  else "") ;
	  List.iter (fun a -> Printf.bprintf b "%s," a) args ;
	  Printf.bprintf b "float" ;
	  Buffer.contents b 
	in
	let quot_res = 
	  Buffer.clear b ;
	  Printf.bprintf b "%s_e@ml_gsl_sf_%s_e," name name ;
	  List.iter (fun a -> Printf.bprintf b "%s," a) args ;
	  Printf.bprintf b "result" ;
	  Buffer.contents b 
	in
	String.concat ""
	  (List.map (ext_quot expr_flag) 
	     [ quot ; 
	       quot_res ;])

let bessel_quot expr_flag str = 
  match words_list str with
  | "cyl" :: letter :: _ ->
      String.concat ""
	[ sf_quot expr_flag 
	    ("bessel_" ^ letter ^ "0 float") ;
	  sf_quot expr_flag 
	    ("bessel_" ^ letter ^ "1 float") ;
	  sf_quot expr_flag 
	    ("bessel_" ^ letter ^ "n int float") ;
	  ext_quot expr_flag
	    (Printf.sprintf 
	       "bessel_%sn_array@ml_gsl_sf_bessel_%sn_array,\
                  int,float,float array,unit" letter letter) ;
	]
  | "cyl_scaled" :: letter :: _ ->
      String.concat ""
	[ sf_quot expr_flag 
	    ("bessel_" ^ letter ^ "0_scaled float") ;
	  sf_quot expr_flag 
	    ("bessel_" ^ letter ^ "1_scaled float") ;
	  sf_quot expr_flag 
	    ("bessel_" ^ letter ^ "n int float") ;
	  ext_quot expr_flag
	    (Printf.sprintf 
	       "bessel_%sn_scaled_array@ml_gsl_sf_bessel_%sn_scaled_array,\
                  int,float,float array,unit" letter letter) ;
	] 
  | "sph" :: letter :: _ ->
      String.concat ""
	[ sf_quot expr_flag 
	    ("bessel_" ^ letter ^ "0 float") ;
	  sf_quot expr_flag 
	    ("bessel_" ^ letter ^ "1 float") ;
	  sf_quot expr_flag 
	    ("bessel_" ^ letter ^ "2 float") ;
	  sf_quot expr_flag 
	    ("bessel_" ^ letter ^ "l int float") ;
	  ext_quot expr_flag
	    (Printf.sprintf 
	       "bessel_%sl_array@ml_gsl_sf_bessel_%sl_array,\
                  int,float,float array,unit" letter letter) ;
	] 
  | "sph_scaled" :: letter :: _ ->
      String.concat ""
	[ sf_quot expr_flag 
	    ("bessel_" ^ letter ^ "0_scaled float") ;
	  sf_quot expr_flag 
	    ("bessel_" ^ letter ^ "1_scaled float") ;
	  sf_quot expr_flag 
	    ("bessel_" ^ letter ^ "l_scaled int float") ;
	  ext_quot expr_flag
	    (Printf.sprintf 
	       "bessel_%sl_scaled_array@ml_gsl_sf_bessel_%sl_scaled_array,\
                  int,float,float array,unit" letter letter) ;
	] 
  | _ -> failwith "bessel_quot: wrong args for quotation"

let _ = 
  Quotation.add "ext"    (Quotation.ExStr ext_quot) ;
  Quotation.add "sf"     (Quotation.ExStr sf_quot) ;
  Quotation.add "bessel" (Quotation.ExStr bessel_quot) ;
  Quotation.default := "sf" 
