/* $Id: nws_nameserver.c,v 1.193 2005/08/04 21:51:23 graziano Exp $ */

#include "config_nws.h"

#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include "diagnostic.h"
#include "dnsutil.h"
#include "host_protocol.h"
#include "messages.h"
#include "osutil.h"
#include "strutil.h"
#include "nws_messages.h"
#include "register.h"
#include "nws_nameserver.h"
#include "timeouts.h"
#include "nws_daemon.h"

/*
 * This program implements the NWS name server host.  See nws_messages.h for
 * supported messages. 
 */

/* those are global because needed by function which won't accept (yet)
 * parameters */
static char registrationFile[FILENAME_MAX]; /* where to backup registrations */

/* we need this to be global because ProcessRequest needs to know where
 * to pick this values */
static registrations *reg;		/* pointer to the registrations */
static nameServerCache *NScache;	/* keep a cache for the NameServers */
static registrations *toSync;		/* keep a list of objects to register */
static int doForward;			/* nameserver replication in place? */

/* since we fork on large asnwers to SEARCH, we keep track of the used
 * socket and we reset is as available when the child is done. We don't
 * do more then CPUCount()*FACTOR children */
#define HOWMANYCHILDS 5
typedef struct {
	pid_t pid;
	Socket sock;
} myChildren;
static myChildren childs[HOWMANYCHILDS];

/* performance data */
static double searchTime;		/* time spent in searching */
static double registerTime;		/* time spent in registering */
static double unregisterTime;		/* time spent in unregistering */
static double forwardTime;		/* time spent in forwarding */
static unsigned long searchQuery;
static unsigned long registerQuery;
static unsigned long unregisterQuery;

/* 
 * this functins adds an #objs# to  #registrations#. If it comes from a
 * nameserver add FROM_NS attribute. 
 */
static void 
AddSetToRegistrations(	ObjectSet objs,
			int fromNS,
			registrations *r) {
	Object obj;
	unsigned long timeOut, now, expTime;
	char *tmp;

	now = CurrentTime();

	for (obj = NextObject(objs, NULL); obj != NULL; obj = NextObject(objs, obj)) {
		/* let's see if we can find our the timeout of this object */
		expTime = 0;
		tmp = NwsAttributeValue_r(FindNwsAttribute(obj, "expiration"));
		if (tmp != NULL) {
			expTime = atol(tmp) - now;
			FREE(tmp);
		}

		/* let's set the timeout */
		timeOut = now + ((expTime  > DEFAULT_HOST_BEAT*2) ? expTime : DEFAULT_HOST_BEAT*2);


		/* we don't need to keep this one ordered: we keep adding
		 * to the end */
		if (InsertRegistration(r, obj, timeOut, r->howMany) && fromNS) {
			AddNwsAttribute(&r->vals[r->howMany - 1], FROM_NS, "yes");
		}
	}
}


/* 
 * read a registration file in.  Return -1 if file was not open, 0
 * otherwise.
 */
static int 
ReadRegistrationFile(	const char *name,
			registrations *r) {
	FILE *regFile;			/* filedes to read a bck file in */
	char *registration;		/* hold the registration */
	char expirationImage[15+1];	/* stringified expiration */
	unsigned long expiration;	/* expiration in number */
	const char *object;
	int len, size;
	unsigned long now;
	
	/* let's see if we have a previous registration file to read in */
	regFile = fopen(name, "r");
	if (regFile == NULL) {
		/* nothing to do here */
		return -1;
	}

	/* starting registration size: we'll increase it if necessary */
	size = 1024;
	registration = (char *)MALLOC(sizeof(char)*size, 0);
	if (registration == NULL) {
		ERROR("ReadRegistrationFile: out of memory\n");
		fclose(regFile);
		return -1;
	}

	now = (unsigned long)CurrentTime();		/* mark the time */
	len = 0;

	/* read the whole string ... */
	while (fgets(registration+len, size-len, regFile) != NULL) {
		len = strlen(registration);
		if (len >= size-1) {
			/* enlarge buffer */
			size += 1024;
			registration = (char *)REALLOC((void *)registration, sizeof(char)*size, 0);
			if (registration == NULL) {
				ERROR("ReadRegistrationFile: out of memory\n");
				fclose(regFile);
				return -1;
			}
			continue;	/* read more */
		}
		if (registration[len-1] != '\n') {
			ERROR("ReadRegistrationFile: line is not newline terminated\n");
			len = 0;
			continue;
		}

		registration[len-1] = '\0';

		(void)GETWORD(expirationImage, registration, &object);

		/* let's check we have at least a digit here */
		if (!isdigit((int)(expirationImage[0]))) {
			ERROR("ReadRegistrationFile: registration contains invalid expiration!\n");
			fclose(regFile);
			return -1;
		}
				
		expiration = atol(expirationImage);
		object++;	/* avoid the space*/
		if (expiration < now) {
			INFO1("ReadRegistrationFile: removing expired entry %s\n", object);
			len = 0;
			continue;
		}

		/* add it to the objects to be sync'd */
		InsertRegistration(r, (const Object)object, expiration, r->howMany);
		len = 0;			/* next round */
	}

	FREE(registration);
	fclose(regFile);

	return 0;			/* done */
}


/*
 * dump the registrations to a file 
 */
static void 
Dump2File(	registrations *r,
		const char *file) {
	FILE *regFile;			/* filedes to read a bck file in */
	char backupFile[FILENAME_MAX];	/* backup name */
	int j;

	/* let's rename the file */
	sprintf(backupFile, "%s.bck", file);
	if (rename(file, backupFile) != 0) {
		DDEBUG("Dump2File: failed to backup the registration file\n");
	}

	/* let's open the file */
	regFile = fopen(file, "w+");
	if (regFile == NULL) {
		ERROR("Dump2File: failed to create registration file\n");
		return;
	}

	/* let's dump the registrations */
	for (j=0; j < r->howMany; j++) {
		/* print expiration ... */
		fprintf(regFile, "%10ld %s\n", r->expirations[j], r->vals[j]);
	}
	fclose(regFile);
}

/* 
 * handle SIGTERM: write the registration and exits 
 */
static int
EmergencyExit() {
	Dump2File(reg, registrationFile);

	return 1;
}

/*
 * This functions parse item to find out if item contains the attribute
 * specified in filter. It returns 1 if it's contained 0 otherwise and -1
 * is a parse error.
 *
 * As you can see it calls itself in case of complex queries.
 * Known operator: < >  ~ = (~ is not equal)
 *
 * Example of accepted filters:
 * 	attribute=ciao
 * 	attribute=*
 *
 * Known parse error:
 * 	attribute=value)) 		is ok
 * 	
 *
 * Note: spaces are allowed at the beginning and at the end of simple
 * attribute<op>value (that is "  attr=1 " is valid but not "attr = 1")
 *
 * MatchSimple works for the simple <attribute><operator><value> query
 * (or part of queries). 
 * MatchComplex works for &( !( |( query. It calls MatchSimple.
 * MatchFilter is just a wrapper for MatchComplex.
 */
static int 
MatchSimple(	const char *item,
		const char **filter) {

	int returnValue;
	char attribute[127 + 1];	/* holds a simple attribute */
	char operator;
	char pattern[127 + 1];		/* pattern to look for in attribute */
	char *nwsAttribute;	/* ptr to the attr in registration */
	char *value;
	size_t valueLen;

	if(!GETTOK(attribute, *filter, "<=>~", filter)) {
		return -1;
	}

	/* get operator: it can be >= <= (that is 2 char) */
	operator = **filter;
	(*filter)++; 
	switch (operator) {
	case '<':
		if(**filter == '=') {
			(*filter)++;
			operator = 'l';
		}
		break;
	case '>':
		if(**filter == '=') {
			(*filter)++;
			operator = 'g';
		}
		break;
	case '~':
		if(**filter == '=')
			(*filter)++;
		break;
	}

	if(!GETTOK(pattern, *filter, ") ", filter)) {
		return -1;
	}

	/* match against the value of each occurence of <attribute> in item. */
	returnValue = 0;
	for (nwsAttribute = FindNwsAttribute((char *)item, attribute); returnValue == 0 && nwsAttribute != NULL; nwsAttribute = FindNwsAttribute(nwsAttribute + 1, attribute)) {

		/* get the value */
		value = NwsAttributeValue_r(nwsAttribute);
		valueLen = strlen(value);

		switch(operator) {
		case '=':
		/* We recognize ~= but treat it as simple = */
		case '~':
			if (strnmatch(value, pattern, valueLen) != 0)
				returnValue = 1;
			break;
		case '<':
			valueLen = (strlen(pattern) < valueLen) ? strlen(pattern) : valueLen;
			if (strncmp(value, pattern, valueLen) < 0)
				returnValue = 1;
			break;
		case 'l':
			valueLen = (strlen(pattern) < valueLen) ? strlen(pattern) : valueLen;
			if (strncmp(value, pattern, valueLen) <= 0)
				returnValue = 1;
			break;
		case '>':
			valueLen = (strlen(pattern) < valueLen) ? strlen(pattern) : valueLen;
			if (strncmp(value, pattern, valueLen) > 0)
				returnValue = 1;
			break;
		case 'g':
			valueLen = (strlen(pattern) < valueLen) ? strlen(pattern) : valueLen;
			if (strncmp(value, pattern, valueLen) >= 0)
				returnValue = 1;
			break;
		}

		free(value);
    	}

	return returnValue;			/* done */
}

static int
MatchComplex(	const char *item,
		const char **filter) {

	char firstChar;				
	int returnValue;

	/* Allow leading and trailing spaces for legibility. */
	while(**filter == ' ')
		(*filter)++;

	firstChar = **filter;

	switch (firstChar) {
	/* Nested filter */
	case '(':
		(*filter)++;
		returnValue = MatchComplex(item, filter);

		/* check if there was a parse error */
		if (returnValue == -1)
			return -1;

		if(**filter == ')') {
			(*filter)++;
		} else {
			return -1;
		}
		break;

	/* Prefix operator: unary not, n-ary or, n-ary and */
	case '!':
		(*filter)++;	/* advance the pointer */
		returnValue = MatchComplex(item, filter);

		/* return right away if there was a parse error */
		if (returnValue == -1)
			return -1;

		/* it's a ! so revert the matching value */
		returnValue = (returnValue + 1) % 2;
		break;

	case '|':
		(*filter)++;
		if ((**filter) != '(') {
			return -1;
		}
		returnValue = 0;
		while((**filter != '\0') && (**filter != ')')) {
			switch (MatchComplex(item, filter)) {
			case -1:
				/* parse error return */
				return -1;
			case 1:
				/* it's an | only one true is enough */
				returnValue = 1;
				break;
			}
		}
		break;

	case '&':
		(*filter)++;
		if ((**filter) != '(') {
			return -1;
		}
		returnValue = 1;
		while((**filter != '\0') && (**filter != ')')) {
			switch (MatchComplex(item, filter)) {
			case -1:
				/* parse error return */
				return -1;
			case 0:
				/* it's an & only one false is enough */
				returnValue = 0;
				break;
			}
		}
		break;

	default:
		/* now only the simple query remains */
		returnValue = MatchSimple(item, filter);
		break;
	}

	/* Allow leading and trailing spaces for legibility. */
	while(**filter == ' ')
		(*filter)++;

	return returnValue;
}

static int 
MatchFilter(	const char *item,
		const char **filter) {
	const char *temp;
	int returnValue;

	/* sanity check */
	if (item == NULL || *filter == NULL) {
		return -1;
	}

	/* since Match{complex,simple} move filter around, we wrap it */
	temp = *filter;
	returnValue = MatchComplex(item, &temp);
	if (strlen(temp) != 0) {
		/* there shouldn't be anything: the parsing is wrong */
		DDEBUG("MatchFilter: syntax error!\n");
		return -1;
	}
	return returnValue;
}


/* 
 * Search for non-expired registration maching filter: returns the index
 * at which the registration lives or -1 if not found. It starts to look
 * for registration from index index.
 *
 * We use this function to look for expired registration too (expired set
 * to 1).
 *
 * NOTE: we don't check the time with the current time since we clean the
 * list quite often.
 */
static int 
DoSearchFrom(	const char *filter,
		int ind,
		int expired) {
	const char *object;
	int ret = -1, j, done;

	if (ind < 0) {
		/* nothing we can do here */
		return -1;
	}
		
	/* loop through the registrations */
	for (done = 0, j = ind; !done && j < reg->howMany; j++) {
		if (reg->vals[j] == NULL) {
			continue;
		}

		object = reg->vals[j];

		if (expired) {
			/* let's return the first expired entry */
			if (reg->expirations[j] == EXPIRED) {
				ret = j;
				break;
			}
		} else if (reg->expirations[j] != EXPIRED) {
			switch (MatchFilter(object, &filter)) {
			case -1:
				/* if parse error return */
				ret = -1;
				done = 1;
				ERROR1("DoSearchFrom: syntax error in %s\n", filter);
				break;
			case 1:
				/* found it */
				ret = j;
				done = 1;
				break;
			}
		}
		
	}

	return ret;
}

/*
 * this are search shortcuts. We recognizes only few patterns. Returns
 * the # of char in #whereTo# or -1 if no shortcut was found.
 */
static int
SearchShortcuts(	const char *filter, 
			char **whereTo) {
	int i, len, len1;
	unsigned long now = 0;
	Object obj;
	char tmp[15], *ptr, *name;

	len = -1;
	len1 = 0;
	
	/* sanity check */
	if (whereTo == NULL || filter == NULL) {
		ERROR("SearchShortcuts: parameter is NULL\n");
		return -1;
	}

	/* current time */
	now = CurrentTime();

	/* this is the pattern used by the nameservers when they
	 * syncrhonize: they want all records no point in searching for
	 * anything */
	if (strcmp(filter, "objectclass=*") == 0) {
		len = 0; 

		/* let's go: get all the registrations */
		for (i = 0; i < reg->howMany; i++) {
			if (now > reg->expirations[i]) {
				continue;
			}
			len1 = strlen(reg->vals[i]);

			/* make room */
			*whereTo = (char *)REALLOC(*whereTo, len + len1+1, 1);

			/* copy the element */
			memcpy(*whereTo + len, reg->vals[i], len1 + 1);
			len += len1;
		}
	} else if (strcmp(filter, "hostType=nameserver") == 0 || strcmp(filter, "&(objectclass=nwsHost)(hostType=nameserver)") == 0) {
		/* this query is used by nameservers and whoever uses
		 * RegisterHost since it asks fot the known nameserver.
		 * This is a big timesaver since we caches the nameserver
		 * for other reasons. */
		len = 0; 
		/* check whereTo is not NULL */
		if (*whereTo == NULL) {
			*whereTo = NewObjectSet();
		}
		/* let's add ourself as the first nameserver */
		obj = RegistrationSet();
		/* and add a timestamp */
		sprintf(tmp, "%10ld", now + DEFAULT_HOST_BEAT*2);
		AddNwsAttribute(&obj, "timestamp", tmp);
		AddObject(whereTo, obj);
		FREE(obj);

		for (i = 0; i < NScache->howMany; i++) {
			if (NScache->reg[i] == NULL) {
				/* empty slot */
				continue;
			}

			/* add the object */
			AddObject(whereTo, NScache->reg[i]);
		}
		len = strlen(*whereTo);		/* returns the # of char */
	} else if (strncmp(filter, "(&(name=", 8) == 0) {
		/* we are looking for a filter that looks like:
		 * 	(&(name=%s)(objectclass=nwsHost))
		 * that is used by Lookup (quite often) by sensors and
		 * memories. If we are we have till the first = so we
		 * need to same the name and check if this is really the
		 * query we are looking for. */
		name = strdup(&filter[8]);
		if (name == NULL) {
			ABORT("SearchShortcuts: out of memory?");
		}
		ptr = strchr(name, ')');
		if (ptr && strcmp(ptr, ")(objectclass=nwsHost))") == 0) {
			/* that's it: we need the name after the = */
			*ptr = '\0';
			if (SearchForName(reg, name, 1, &i)) {
				/* check whereTo is not NULL */
				if (*whereTo == NULL) {
					*whereTo = NewObjectSet();
				}
				/* we got the highest rank object */
				for (; i >= 0; i--) {
					/* let's check we still have a
					 * similar name */
					ptr = NwsAttributeValue_r(FindNwsAttribute(reg->vals[i], "name"));
					if (strcmp(ptr, name) == 0) {
						/* add the object */
						AddObject(whereTo, reg->vals[i]);
					}
					FREE(ptr);
				}
				/* let's see the size of the returned
				 * string */
				len = strlen(*whereTo);
			}
		}
		FREE(name);
	} else if (strncmp(filter, "(name=", 6) == 0) {
		/* we are looking for a filter that looks like:
		 * 	(name=%s)
		 * that is used by nws_extract. */
		name = strdup(&filter[6]);
		if (name == NULL) {
			ABORT("SearchShortcuts: out of memory?");
		}
		ptr = strchr(name, ')');
		if (ptr && strcmp(ptr, ")") == 0) {
			/* that's it: we need the name after the = */
			*ptr = '\0';
			if (SearchForName(reg, name, 1, &i)) {
				/* check whereTo is not NULL */
				if (*whereTo == NULL) {
					*whereTo = NewObjectSet();
				}
				/* we got the highest rank object */
				for (; i >= 0; i--) {
					/* let's check we still have a
					 * similar name */
					ptr = NwsAttributeValue_r(FindNwsAttribute(reg->vals[i], "name"));
					if (strcmp(ptr, name) == 0) {
						/* add the object */
						AddObject(whereTo, reg->vals[i]);
					}
					FREE(ptr);
				}
				/* let's see the size of the returned
				 * string */
				len = strlen(*whereTo);
			}
		}
		FREE(name);
	}

	return len;
}


#if 0
/* 
 * given an IP address (or name?) it check if the machine is reachable
 * (mainly checks machine behind a NAT or firewall). Return 1 if it's
 * reachable, 0 otherwise
 */
static int
Reachable(	const char *ipAddress, 
		short port) {

	struct host_cookie cookie;

	if (ipAddress == NULL)
		return 0;

	if (port <= 0) 
		Host2Cookie(ipAddress, DefaultHostPort(SENSOR_HOST), &cookie);
	else
		Host2Cookie(ipAddress, port, &cookie);

	if(ConnectToHost(&cookie, &cookie.sd)) {
		return 1;
	}

	return 0;
}
#endif

/*
 * check if the #registration# is a nameserver and is
 * already in the cache: if not, add it and sync with it. Is sd isn't
 * NO_SOCKET use that socket to connect. 
 * Returns 1 if it is a nameserver 0 otherwise.
 */
static int
CheckNScache(	Object registration, 
		Socket sd) {
	char *tmp;
	int i, j;
	DataDescriptor objectDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);

	/* sanity check */
	if (registration == NULL) {
		return 0;
	}

	/* we need a resonable object */
	if (FindNwsAttribute(registration, "name") == NULL) {
		ERROR("CheckNScache: object without a name!\n");
		return 0;
	}

	/* we don't add ourself to this cache */
	tmp = RegistrationSet();
	i = AreObjectsEquivalent(registration, tmp);
	FREE(tmp);
	if (i == 1) {
		return 1;		/* we are a nameserver anyway */
	}

	/* has to be a nameserver */
	tmp =  NwsAttributeValue_r(FindNwsAttribute(registration, "hostType"));
	if (tmp == NULL || strcmp(tmp, "nameserver") != 0) {
		FREE(tmp);
		return 0;
	}
	FREE(tmp);

	/* let's check in the cache */
	for (j = -1, i = 0; i < NScache->howMany; i++) {
		/* avoid empty slots */
		if (NScache->reg[i] == NULL && j == -1) {
			/* let's remember an empty slot */
			j = i;
			continue;
		}

		if (AreObjectsEquivalent(registration, NScache->reg[i])) {
			/* found it's an old one: let's check if we have
			 * already a socket */
			if (sd != NO_SOCKET && NScache->cookie[i].sd != sd) {
				DisconnectHost(&NScache->cookie[i]);
				NScache->cookie[i].sd = sd;
				NScache->last[i] = 0;
			}
			return 1;
		}
	}

	/* kinda cheat: I don't count the previous checking in the
	 * timing: just don't want to add too many MicroTime() call */
	forwardTime -= MicroTime();

	/* is it a new (not in cache) nameserver? */
	if (i >= NScache->howMany && j != -1) {
		/* we have an empty slot */
		i = j;
	}

	if (i >= NScache->howMany) {
		/* let's add the entry */
		NScache->howMany++;
		NScache->reg = (char **)REALLOC(NScache->reg, NScache->howMany*(sizeof(char *)), 1);
		NScache->cookie = (struct host_cookie *)REALLOC(NScache->cookie, NScache->howMany*(sizeof(struct host_cookie)), 1);
		NScache->last = (unsigned long *)REALLOC(NScache->last, NScache->howMany*(sizeof(unsigned long)), 1);
		NScache->queue = (registrations *)REALLOC(NScache->queue, NScache->howMany*(sizeof(registrations)), 1);
		NScache->deQueue = (registrations *)REALLOC(NScache->deQueue, NScache->howMany*(sizeof(registrations)), 1);
	}

	/* get the name in the cookie */
	NScache->reg[i] = NULL;
	NScache->cookie[i].name[0] = '\0';
	NScache->cookie[i].port = 0;
	NScache->cookie[i].sd = sd;

	/* reset values */
	NScache->last[i] = 0;
	NScache->queue[i].howMany = 0;
	NScache->queue[i].dirty = 0;
	NScache->queue[i].vals = NULL;
	NScache->queue[i].expirations = NULL;
	NScache->deQueue[i].howMany = 0;
	NScache->deQueue[i].dirty = 0;
	NScache->deQueue[i].vals = NULL;
	NScache->deQueue[i].expirations = NULL;

	/* let's sync with it: we just send a NS_SEARCH message and
	 * ProcessRequest will get the response */
	if (!ConnectToObject(registration, &NScache->cookie[i])) {
		tmp = NwsAttributeValue_r(FindNwsAttribute(registration, "name"));
		ERROR1("CheckNScache: couldn't talk to %s\n", tmp);
		FREE(tmp);
	} else {
		/* we don't use RetrieveObjects becasue we want to receive
		 * asynchronously the data */
		objectDescriptor.repetitions = strlen("objectclass=*") + 1;
		if (!SendMessageAndData(NScache->cookie[i].sd, NS_SEARCH, "objectclass=*", &objectDescriptor, 1, -1)) {
			DisconnectHost(&NScache->cookie[i]);
			tmp = NwsAttributeValue_r(FindNwsAttribute(registration, "name"));
			if (tmp != NULL) {
				ERROR1("CheckNScache: couldn't send message to %s\n", tmp);
			}
			FREE(tmp);
		} else {
			/* let's register the nameserver */
			/* i contains the good slot */ 
			NScache->reg[i] = strdup(registration);
			if (NScache->reg[i] == NULL) {
				ABORT("CheckNScache: out of memory");
			}
		}
	}

	/* done AddToNSqueue do its own forwardTime handling */
	forwardTime += MicroTime();

	/* and let's register with it if we talked to it */
	if (NScache->reg[i] != NULL) {
		tmp = RegistrationSet();
		AddToNSqueue(tmp, DEFAULT_HOST_BEAT*2);
		FREE(tmp);
		ForwardNSqueue(5);
	}

	/* some feedback on how many NS we have */
	INFO1("CheckNScache: we have %d nameservers\n", NScache->howMany);

	return 1;
}



/* 
 * Insert a registration: if we find an old one we override it. If there
 * is more than one registration in object, we register (or try to)
 * register them all and re return success if at least one succeeded.
 *
 * Return # index in reg->vals  on success
 *        -1 if the registration didn't get in
 */
static int 
DoRegister(	char *object,
		unsigned long expiration) {
	int done;
	char expirationImage[15+1]; /* string for expiration */

	/* sanity check */
	if (object == NULL) {
		ERROR("DoRegister: empty object!\n");
		return -1;
	}

#if 0
	/* we do a little tweak on the IP address: we check if the IPs
	 * address in the registration contain the IP the caller is
	 * calling from. If not (NAT? misconfiguration?) we add it */
	name = NwsAttributeValue_r(FindNwsAttribute(newReg, "ipAddress"));

	/* don't consider localhost */
	if (fromIP != NULL && (strncmp("127.0.0.1", fromIP, strlen("127.0.0.1")) == 0 || strncmp("(unknown)", fromIP, strlen("(unknown)")) == 0)) {
		FREE(name);
	}

	if ((fromIP != NULL) && (name != NULL)) {
		for (ptr = name; GETTOK(ip, ptr, ",", &ptr) != 0; ) {
			if (strncmp(ip, fromIP, strlen(ip)) == 0) {
				/* we already have this one */
				break;
			}

			/* IP is not there: we need to add it */
			name = REALLOC(name, strlen(name)+strlen(fromIP)+2);
			if (name == NULL) {
				ABORT("DoRegister: out of memory!\n");
			} else {
				DDEBUG("DoRegister: added IP to IP list\n");
				strcat(name, ",");
				strcat(name, fromIP);
				DeleteNwsAttribute(&newReg, "ipAddress");
				AddNwsAttribute(&newReg, "ipAddress",
				name);

				/* check if the machine is reachable */
				if (Reachable(fromIP, port) == 0) {
/*					AddNwsAttribute(&newReg, "reachable", "no"); */
				}
			}
		}
	}
	FREE(name);
#endif

	/* let's look for an equivalent object */
	if (SearchForObject(reg, object, &done)) {
		/* we found it: let's get the new object in */
		FREE(reg->vals[done]);
		reg->vals[done] = strdup(object);
		if (reg->vals[done] == NULL) {
			ABORT("DoRegister: out of memory\n");
		}
	} else {
		/* didn't find: let's add it */
		if (done == -1 || !InsertRegistration(reg, object, expiration, done)) {
			ERROR("DoRegister: not an NWS object\n");
			return -1;
		}
	}


	/* delete the old timestamp and put the new one in */
	reg->expirations[done] = expiration;
	DeleteNwsAttribute(&reg->vals[done], "timestamp");
	sprintf(expirationImage, "%10.0f", CurrentTime());
	AddNwsAttribute(&reg->vals[done], "timestamp", expirationImage);
	DeleteNwsAttribute(&reg->vals[done], "expiration");
	sprintf(expirationImage, "%10ld", expiration);
	AddNwsAttribute(&reg->vals[done], "expiration", expirationImage);

	/* just some feedback on how many stuff we have */
	INFO1("DoRegister: we have %d registrations\n", reg->howMany);

	return done;
}

/*
 * Mark a registration as expired
 */
static int
DoUnregister(const char *filter) {
	int i, ret;
	char *name, *ptr, *tmp;

	ret = 0;
	i = -1;

	/* let's try to optimize: most of the unregistrations comes with
	 * a filter like (name=%s): this is great since we have the
	 * registrations ordered that way. If so use binary search */
	if (strncmp(filter, "(name=", 6) == 0) {
		name = strdup(&filter[6]);
		if (name == NULL) {
			ABORT("DoUnregister: out of memory\n");
		}
		ptr = strchr(name, ')');
		if (ptr) {
			*ptr = '\0';

			if (SearchForName(reg, name, 1, &i)) {
				int done;

				/* let's find all the entries */
				do {
					tmp = NwsAttributeValue_r(FindNwsAttribute(reg->vals[i], "name"));
					done = strcmp(tmp, name);
					if (done == 0) {
						ret = 1;
						reg->expirations[i] = EXPIRED;
					}
					FREE(tmp);
					i--;
				} while (done == 0);
			}
			/* no normal search after this */
			i = reg->howMany;
		}
		FREE(name);
	}
	do {
		/* do the normal search */
		i = DoSearchFrom(filter, i + 1, 0);
		if (i != -1) {
			/* set EXPIRED to the expiration field */
			ret = 1;
			reg->expirations[i] = EXPIRED;
		}
	} while (i != -1);

	if (ret == 0) {
		INFO("DoUnregister: couldn't find a matching registration\n");
	}

	return ret;
}

/*
 * Clean up the registrations: marked the expired one with the EXPIRED
 * timestamp. We are recompacting the registraions so that howMany will
 * reflects the real numbers of registrations we have.
 */
static void 
CleanRegistrations() {
	int i, j, z;
	unsigned long now;                      /* holds the current time */

	now = (unsigned long)CurrentTime();

	for (i=0; i < reg->howMany; i++) {
		if (reg->expirations[i] < now) {
			/* reset the cache in case is needed */
			for (j = 0;  j < NScache->howMany; j++) {
				if (NScache->reg[j] != NULL && AreObjectsEquivalent(NScache->reg[j], reg->vals[i])) {
					/* got is: free the structure */
					for (z = 0; z < NScache->queue[j].howMany; z++) {
						FREE(NScache->queue[j].vals[z]);
					}
					for (z = 0; z < NScache->deQueue[j].howMany; z++) {
						FREE(NScache->deQueue[j].vals[z]);
					}
					/* reset queues values */
					NScache->queue[j].howMany = 0;
					NScache->deQueue[j].howMany = 0;
					NScache->queue[j].dirty = 0;
					NScache->deQueue[j].dirty = 0;
					FREE(NScache->queue[j].vals);
					FREE(NScache->deQueue[j].vals);
					FREE(NScache->queue[j].expirations);
					FREE(NScache->deQueue[j].expirations);

					/* free the slot */
					FREE(NScache->reg[j]);
					NScache->cookie[j].name[0] = '\0';
					NScache->cookie[j].port = 0;
					DisconnectHost(&NScache->cookie[j]);
				}
			}
			/* delete the registration */
			DeleteRegistration(reg, i);

			/* re-check the current slot */
			i--;
		}
	}
}

/* 
 * add the #registration# to the nameservers queues. #timeOut# indicates how
 * long the registration is good for. If timOut is < 0 it means to
 * unregister the object.
 * Returns 0 if failed, 1 if * succesful..
 */
static int
AddToNSqueue(	Object registration,
		long timeOut) {
	int i, j;
	char *obj;

	/* should we forward? */
	if (!doForward) {
		return 0;
	}

	/* let's check the registration (if it *is* a registration).
	 * Needs to do it only if we register. */
	if (timeOut >= 0) {
		if (strstr(registration, OBJECT_TERMINATOR) == NULL) {
			WARN("AddToNSqueue: object with no termination\n");
			return 0;
		}
	}
	
	/* let's start the counter */
	forwardTime -= MicroTime();

	/* let's loop through all the nameservers */
	for (i = 0; i < NScache->howMany; i++) {
		if (NScache->reg[i] == NULL) {
			/* no current nameserver here */
			continue;
		}

		/* let's copy the registration (or filter) */
		obj = strdup(registration);
		if (obj == NULL) {
			ABORT("AddToNSqueue: out of memory\n");
		}

		/* let's add it to the right queue */
		if (timeOut >= 0) {
			char *tmp;

			/* don't tag it if it's this nameserver's */
			tmp = RegistrationSet();
			if (tmp == NULL) {
				FREE(obj);
				ERROR("AddToNSQueue: out of memory\n");
				return 0;
			}
			if (strcmp(obj, tmp) != 0) {
				AddNwsAttribute(&obj, FROM_NS, "yes");
			}
			FREE(tmp);

			/* add to register queue */
			CheckRegistrations(&NScache->queue[i]);
			j = NScache->queue[i].howMany; /* shortcut */

			/* add the new elelment */
			NScache->queue[i].vals[j] = obj;
			NScache->queue[i].expirations[j] = timeOut;
			NScache->queue[i].howMany++;
			NScache->queue[i].dirty--;
		} else {
			/* add to unregister queue */
			CheckRegistrations(&NScache->deQueue[i]);
			j = NScache->deQueue[i].howMany; /* shortcut */

			/* since this is a pattern we add a
			 * fromNameserver in front of the pattern to
			 * specify it's from a nameserver */
			NScache->deQueue[i].vals[j] = (char *)MALLOC(strlen(obj) + FROM_NS_LEN + 1, 1);
			NScache->deQueue[i].vals[j][0] = '\0';
			strcat(NScache->deQueue[i].vals[j], FROM_NS);
			strcat(NScache->deQueue[i].vals[j], obj);
			FREE(obj);

			NScache->deQueue[i].howMany++;
			NScache->deQueue[i].dirty--;
		}
	}

	/* let's stop the counter */
	forwardTime += MicroTime();

	return 1;
}

/* 
 * check the NS queues and if there is some registrations to forward it
 * does so forwarding at most #howMany#.
 * Returns 1 if there were some messages waiting, 0 otherwise 
 */
static int 
ForwardNSqueue(int howMany) {
	int i, j, z, ret = 0;
	double now;
	char *tmp;

	/* let's start the counter */
	forwardTime -= MicroTime();

	/* let's loop through all the nameservers */
	for (i = 0; i < NScache->howMany; i++) {
		if (NScache->reg[i] == NULL) {
			/* no current nameserver here */
			continue;
		}

		/* is there anything to do here? */
		if (NScache->queue[i].howMany <= 0 && NScache->deQueue[i].howMany <= 0) {
			/* nope */
			continue;
		}

		ret = 1;		/* something is in the queue */
		now = CurrentTime();

		/* is it time to retry to talk to this guy? */
		if (NScache->last[i] != 0 && (now < (NScache->last[i] + NAP_TIME))) {
			tmp = NwsAttributeValue_r(FindNwsAttribute(NScache->reg[i], "name"));
			if (tmp != NULL) {
				DDEBUG1("ForwardNSqueue: timer not expired for %s\n", tmp);
			}
			FREE(tmp);
			continue;		/* nope */
		}


		/* register the time we tried to connect */
		NScache->last[i] = now;

		/* let's send at most howMany objects one at a time */
		for (z = 0; z <= howMany; z++) {
			/* let's send the registrations first */
			j = NScache->queue[i].howMany - 1;	/* shortcut */
			if (j >= 0) {
				if (!Register(&NScache->cookie[i], NScache->queue[i].vals[j], NScache->queue[i].expirations[j])) {
					tmp = NwsAttributeValue_r(FindNwsAttribute(NScache->reg[i], "name"));
					if (tmp != NULL) {
						WARN1("ForwardNSqueue: failed to forward registration to %s\n", tmp);
					}
					FREE(tmp);
					break;		/* out of the loop */
				}
				/* free the slot */
				FREE(NScache->queue[i].vals[j]);
				NScache->queue[i].howMany--;
				NScache->queue[i].dirty++;
			}

			/* let's send the unregister queue */
			j = NScache->deQueue[i].howMany - 1; /* shortcut */
			if (j >= 0) {
				if (!Unregister(&NScache->cookie[i], NScache->deQueue[i].vals[j])) {
					tmp = NwsAttributeValue_r(FindNwsAttribute(NScache->reg[i], "name"));
					if (tmp != NULL) {
						WARN1("ForwardNSqueue: failed to forward unregistration to %s\n", tmp);
					}
					FREE(tmp);
					break;		/* out of the loop */
				}
				/* free the slot */
				FREE(NScache->deQueue[i].vals[j]);
				NScache->deQueue[i].howMany--;
				NScache->deQueue[i].dirty++;
			}
		}
		/* reset the counter for this nameserver if it behaved well */
		if (z >= howMany) {
			NScache->last[i] = 0;
		}
	}

	/* let's stop the counter */
	forwardTime += MicroTime();

	return ret;
}

/*
 *  This is the "worker": dispatch all the request around. It should be
 *  threaded.
 */
static void
ProcessRequest(	Socket *sd,
               	MessageHeader header) {

	unsigned long timeOut, now;
	int matching, last;
	char *object, *pattern;
	DataDescriptor objectDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	DataDescriptor patternDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	DataDescriptor timeOutDescriptor = SIMPLE_DATA(UNSIGNED_LONG_TYPE, 1);
	struct host_cookie cookie;
	HostInfo info;

	switch(header.message) {
	case NS_REGISTER:
		registerQuery++;			/* one more query */
		registerTime -= MicroTime();		/* start the timer */

		objectDescriptor.repetitions = header.dataSize - HomogenousDataSize(UNSIGNED_LONG_TYPE, 1, NETWORK_FORMAT);
		object = (char *)MALLOC(objectDescriptor.repetitions + 1, 0);

		/* sanity check */
		if(object == NULL) {
			(void)SendMessage(*sd, NS_FAILED, -1);
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: out of memory\n");
			break;		/* we can't do anything else */
		}

		/* show me the packets! */
		if(!RecvData(*sd, object, &objectDescriptor, 1, -1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: receive failed\n");
			FREE(object);
			break;
		}
		if (!RecvData(*sd, &timeOut, &timeOutDescriptor, 1, -1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: receive failed\n");
			FREE(object);
			break;
		} 
		object[objectDescriptor.repetitions] = '\0';
		now = (unsigned long)CurrentTime();

		/* if the object comes from a nameserver let's strip the
		 * fromNameserver attribute this attribute: if it's not a
		 * forward we cache the socket */
		if (DeleteNwsAttribute(&object, FROM_NS)) {
			/* it's a forward: no socket caching here */
			matching = NO_SOCKET;
		} else {
			/* it could be a new nameserver; cache socket */
			matching = *sd;
		}

		/* let's register the object (without the extra attribute) */
		last = DoRegister(object, now + timeOut);
		FREE(object);
		if (last < 0) {
			/* we got problem with registering */
			ERROR1("ProcessRequest: bad object from socket %d \n", *sd);
			if (!SendMessage(*sd, NS_FAILED, -1)) {
				WARN("ProcessRequest: failed to send ack\n");
			}
		} else {
			/* let's cache the nameserver socket if we just
			 * got a nameserver registration */
			CheckNScache(reg->vals[last], matching);

			/* only older sensor needs ack */
			if (matching != NO_SOCKET && header.version < 0x02080000) {
				SendMessage(*sd, NS_REGISTERED, -1);
			}
		}

		registerTime += MicroTime();	/* stop the timer */

		/* if it's not a forward, we need to forward the
		 * registration. AddToNSqueue does its own accounting. */
		if (last >= 0 && matching != NO_SOCKET) {
			AddToNSqueue(reg->vals[last], timeOut);
		}

		break;

	case NS_UNREGISTER:
		unregisterQuery++;
		unregisterTime -= MicroTime();
		last = 0;

		/* let's receive the string and be sure to NULL terminate it */
		patternDescriptor.repetitions = header.dataSize;
		pattern = (char *)MALLOC(header.dataSize + 1, 0);
		if (pattern == NULL) {
			SendMessage(*sd, NS_FAILED, -1);
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: out of memory\n");
			break; 	/* nothing more we can do here */
		} 
		if (!RecvData(*sd, pattern, &patternDescriptor, 1, -1)) {
			FREE(pattern);
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: receive failed\n");
			break;
		}
		pattern[header.dataSize] = '\0';

		/* let's check if it comes from a nameserver */
		if (strncmp(pattern, FROM_NS, FROM_NS_LEN) == 0) {
			/* yep: let's remove it */
			memmove(pattern, pattern + FROM_NS_LEN, strlen(pattern+FROM_NS_LEN) + 1);
			DoUnregister(pattern);
			/* it's from a nameserver avoid ack */
		} else {
			last = DoUnregister(pattern);
			/* send confirmation to the client if it's an old
			 * pal */
			if (header.version < 0x02080000) {
				SendMessage(*sd, NS_UNREGISTERED,-1);
			}
		}

		/* do we need to forward?  AddToNSqueue does its own
		 * accounting. */
		if (last) {
			AddToNSqueue(pattern, -1);
		}
		FREE(pattern);

		unregisterTime += MicroTime();	/* stop the timer */

		break;

	case NS_SEARCH:
		searchTime -= MicroTime();	/* start the timer */
		searchQuery++;			/* add the # of query */

		patternDescriptor.repetitions = header.dataSize;
		pattern = (char *)MALLOC(header.dataSize + 1, 0);
		if(pattern == NULL) {
			(void)SendMessage(*sd, NS_FAILED, -1);
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: out of memory\n");
			break; 	/* nothing more we can do here */
		} 

		/* show me the packet! */
		if(!RecvData(*sd,pattern,&patternDescriptor,1,-1)){
			FREE(pattern);
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: receive failed\n");
			break;
		}

		/* null terminate the string */
		pattern[header.dataSize] = '\0';
		object = NULL;
		last = SearchShortcuts(pattern, &object);
		if (last < 0) {
			/* we have to search */
			matching = DoSearchFrom(pattern, 0, 0); 
			last = 0;
		} else {
			/* done */
			matching = -1;
		}
		if (object == NULL) {
			object = strdup("");
		}

		/* let's look for all the matches */
		while (matching != -1) {
			int temp;

			temp = strlen(reg->vals[matching]);
			object = (char *)REALLOC(object, last + temp + 1, 0);
			if (object == NULL) {
				break;
			} 
			memcpy(object+last, reg->vals[matching], temp + 1);
			last += temp;

			/* next match */
			matching = DoSearchFrom(pattern, matching + 1, 0);
		}

		/* done with the search */
		if (object == NULL) {
			ERROR("ProcessRequest: out of memory\n");
			(void)SendMessage(*sd, NS_FAILED, -1);
			DROP_SOCKET(sd);
		} else {
			pid_t pid = -1;

			matching = -1;		/* communication timeout */
			objectDescriptor.repetitions = last+1;

			/* let's find a spot in childs */
			for (last = 0; last < HOWMANYCHILDS; last++) {
				if (childs[last].pid == 0) {
					break;
				}
			}

			/* if we have to send too much data, we fork  (if
			 * we don't have too many children already */
			if (objectDescriptor.repetitions > 1024*128 && last < HOWMANYCHILDS) {
				if (!CreateLocalChild(&pid, NULL, NULL)) {
					ERROR("ProcessRequest: cannot fork!\n");
				}
				if (pid > 0) {
					/* parent: let's remember child
					 * and socket */
					childs[last].pid = pid;
					childs[last].sock = *sd;
					*sd = NO_SOCKET;

					/* hosekeeping */
					FREE(pattern);
					FREE(object);
					searchTime += MicroTime();
					break;
				} else if (pid == 0) {
					/* child: let's answer */
					matching = 0;
				}
			}

			if (!SendMessageAndData(*sd,
       		                           NS_SEARCHED,
       		                           object,
       		                           &objectDescriptor,
       		                           1,
       		                           matching)) {
				ERROR("ProcessRequest: failed to send NS_SEARCHED\n");
			}

			/* if we are the child, we are done */
			if (pid == 0) {
				exit(0);
			}
		}
		/* done */
		FREE(pattern);
		FREE(object);

		searchTime += MicroTime();	/* stop the timer */

		break;

	case NS_SEARCHED:
		/* this comes from a nameserver we want to sync with */
		objectDescriptor.repetitions = header.dataSize;
		object = (char *)MALLOC(header.dataSize + 1, 0);
		if (object == NULL) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: out of memory\n");
			break;
		}
		if(!RecvData(*sd, object, &objectDescriptor, 1, -1)) {
			ERROR("ProcessRequest: failed to receive in NS_SEARCHED\n");
			DROP_SOCKET(sd);
		} else {
			object[header.dataSize] = '\0';

			/* let's add them to our todo list */
			AddSetToRegistrations(object, 1, toSync);
		}
		FREE(object);
		break;

	case HOST_REGISTER:
		object = (char *)MALLOC(header.dataSize + 1, 0);
		if (object == NULL) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: out of memory\n");
			break;
		}

		/* let's get the objects */
		objectDescriptor.repetitions = header.dataSize;
		if(!RecvData(*sd, object, &objectDescriptor, 1, -1)) {
			FREE(object);
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: failed to receive\n");
			break;
		}
		object[header.dataSize] = '\0';

		(void)SendMessage(*sd, HOST_REGISTERED, -1);
		Host2Cookie(object, DefaultHostPort(NAME_SERVER_HOST), &cookie);
		FREE(object);

		/* let's check if it's a name server */
		if (!GetHostInfo(&cookie, &info, -1)) {
			WARN2("ProcessRequest: failed to get info for %s:%d\n", cookie.name, cookie.port);
			break;
		}
		DisconnectHost(&cookie);

		if (info.hostType != NAME_SERVER_HOST) {
			ERROR("ProcessRequest: host is not a nameserver\n");
			break;
		}

		/* let's sync with it */
		if (RetrieveObjects(&cookie, "hostType=nameserver", &pattern, -1)) {
			/* this will force the forwarding */
			doForward = 1;
			AddSetToRegistrations(pattern, 1, toSync);
			FREE(pattern);
		}

		break;

	case NS_FAILED:
		/* we don't care about these messages coming from another
		 * nameserver */
		break;

	default:
		ERROR1("ProcessRequest: unknown message %d\n", header.message);
		DROP_SOCKET(sd);
		break;
	}

	/* done with the socket */
	SocketIsAvailable(*sd);
}

/*
 * check if there are registrations still pending and register them. We
 * pick a fixed number of registration at a time to not take too much
 * time.
 * Return 1 if there are registartions to sync, 0 if we are done.
 */
static int
SyncUP( 	registrations *r,
		int howMany) {
	char *obj;
	unsigned long timeOut;
	int fromNS;

	/* this counts as registration time anyway */
	registerTime -= MicroTime();

	for (; howMany > 0; howMany--) {
		/* do we have more registrations to sync? */
		if (r->howMany <= 0) {
			break;
		} 

		/* let's save the registration and expiration */
		obj = r->vals[r->howMany - 1];
		timeOut = r->expirations[r->howMany - 1];

		/* free the slot */
		r->vals[r->howMany - 1] = NULL;
		r->howMany--;
		r->dirty++;

		/* does it comes from a nameserver? if so, no forwarding */
		fromNS = DeleteNwsAttribute(&obj, FROM_NS);

		if (DoRegister(obj,  timeOut) >= 0) {
			/* let's forward if it's not from a nameserver
			 * and it's not a nameserver: nameserver will get
			 * to all the mirroring in a different manner */
			if (!CheckNScache(obj, NO_SOCKET) && !fromNS) {
				AddToNSqueue(obj, timeOut);
			}
		}
		/* free the memory */
		FREE(obj);
	}

	registerTime += MicroTime();		/* stop the timer */

	return (r->howMany > 0);
}


static void 
usage() {
	DaemonUsage("nws_nameserver", "Nameserver");
	printf("\t-f filename         use filename as registration s file\n");
	printf("\t-m                  enable replica mode\n");
	printf("\t-c time             clean registrations file every time seconds\n");
}

int
main(	int argc,
	char *argv[]) {

	unsigned long compressionFreq,	/* how often to clean registrations */
		nextBeatTime, 
		nextCompress, 
		nextWait,
		now; 
	double wakeup;
	int opt, tmp, ind;
	extern char *optarg;

	/* Set up default values to be overwritten by command line args. */
	compressionFreq = DEFAULT_COMPRESSION_FREQUENCY;
	SAFESTRCPY(registrationFile, DEFAULT_FILE);
	doForward = 0;			/* default: no mirroring */

	/* no error messages from getopt() */
	opterr = 0;
	ind = InitializeDaemonStructure(NAME_SERVER_HOST, "f:mc:", EmergencyExit);

	while((int)(opt = getopt(argc, argv, DaemonAllSwitches(ind))) != EOF) {
		if (DaemonSwitch(opt, optarg, ind)) {
			/* we already dealt with it */
			continue;
		}

		switch(opt) {
		case 'c':
			tmp = atol(optarg);
			if (tmp != 0) {
				compressionFreq = tmp;
			}
			break;

		case 'f':
			SAFESTRCPY(registrationFile, optarg);
			break;

		case 'm':
			doForward = 1;		/* mirroring enabled */
			break;

		default:
			usage();
			exit(1);
			break;
	    	}
	}

	/* initialize the registrations  */
	if (!InitRegistrations(&reg)) {
		ABORT("main: out of memory\n");
	}
	
	/* initialize the NameServers cache */
	NScache = (nameServerCache *) MALLOC(sizeof(nameServerCache), 1);
	NScache->howMany = 0;
	NScache->reg = NULL;
	NScache->last = NULL;
	NScache->cookie = NULL;
	NScache->queue = NULL;
	NScache->deQueue = NULL;

	/* initialize the queue of registration to sync */
	if (!InitRegistrations(&toSync)) {
		ABORT("main: out of memory\n");
	}

	/* initialize the performance counters */
	searchTime = registerTime = forwardTime = unregisterTime = 0;
	searchQuery = registerQuery = unregisterQuery = 0;

	/* initialize the structure to keep track of sockets given to
	 * children */
	for (tmp = 0; tmp < HOWMANYCHILDS; tmp++) {
		childs[tmp].pid = 0;
	}

	/* setting the defaults timeouts: we use tight timeouts to not
	 * block on slow machines  */
	SetDefaultTimeout(CONN, MIN_TIME_OUT, MAX_TIME_OUT/3, TIMEOUT_TIME_OUT);
	SetDefaultTimeout(RECV, MIN_TIME_OUT, MAX_TIME_OUT, TIMEOUT_TIME_OUT);
	SetDefaultTimeout(SEND, MIN_TIME_OUT, MAX_TIME_OUT, TIMEOUT_TIME_OUT);

	/* let's retrieve the list of nameservers */
	if (doForward) {
		ObjectSet objs = NULL;

		if (RetrieveFromMyNameserver("hostType=nameserver", &objs)) {
			AddSetToRegistrations(objs, 1, toSync);
		} else {
			ERROR("main: failed to sync with nameserver\n");
		}
		FREE(objs);
	}

	/* let's read the old registrations if the file is there */
	ReadRegistrationFile(registrationFile, toSync);

	/* ready to do work now */
	if (!InitializeDaemon(0, 0, 0, ind)) {
		exit(-1);
	}
	
	/* let's get ready to work */
	RegisterListener(NS_REGISTER, "NS_REGISTER", &ProcessRequest);
	RegisterListener(NS_SEARCH, "NS_SEARCH", &ProcessRequest);
	RegisterListener(NS_UNREGISTER, "NS_UNREGISTER", &ProcessRequest);
	RegisterListener(NS_FAILED, "NS_FAILED", &ProcessRequest);
	RegisterListener(NS_SEARCHED, "NS_SEARCHED", &ProcessRequest);

	/* we deal ourself with the HOST_REGISTER */
	UnregisterListener(HOST_REGISTER);
	RegisterListener(HOST_REGISTER, "HOST_REGISTER", &ProcessRequest);

	nextWait = nextBeatTime = nextCompress = 0;

	/* main service loop */
	while (1) {
		now = (unsigned long)CurrentTime();

		/* normal BEAT maintenance */
		if (now >= nextBeatTime) {
			/* register with myself... */
			RegisterHost(DEFAULT_HOST_BEAT*2); 

			/* netxt scheduled maintenance: we do it quite
			 * often because we want to be sure we can talk
			 * with all nameservers */
			nextBeatTime = now + DEFAULT_HOST_BEAT/5;
			
			/* print performance timing */
			LOG1("main: spent %.0fms in register\n", registerTime/1000);
			LOG1("main: spent %.0fms in unregistering\n", unregisterTime/1000);
			LOG1("main: spent %.0fms in forward\n", forwardTime/1000);
			LOG1("main: spent %.0fms in search\n", searchTime/1000);
			LOG3("main: got %d search, %d register and %d unregister requests\n", searchQuery, registerQuery, unregisterQuery);
			/* resetting */
			forwardTime = registerTime = searchTime = unregisterTime = 0;
			searchQuery = registerQuery = unregisterQuery = 0;

			/* clean up the registrations */
			CleanRegistrations();
			LOG1("main: I have  %d registrations\n", reg->howMany);
		}

		/* let's wait for the done children */
		if (now >= nextWait) {
			pid_t pid;

			/* let's get rid of the zombies */
			pid = WaitChild();
			for (tmp = 0; pid != 0 && tmp < HOWMANYCHILDS; tmp++) {
				if (childs[tmp].pid == pid) {
					/* found it */
					childs[tmp].pid = 0;
					SocketIsAvailable(childs[tmp].sock);
					break;
				}
			}

			nextWait = now + 5;
		}

		/* let's backup the registrations */
		if (now >= nextCompress) {
			Dump2File(reg, registrationFile);
			nextCompress = now + compressionFreq;
		}

		/* let's calculate next round about */
		if (nextBeatTime > nextCompress) {
			wakeup = nextCompress;
		} else {
			wakeup = nextBeatTime;
		}
		wakeup -= now;

		/* do we have objects to sync with or to forward? */
		if (SyncUP(toSync, 10)) {
			wakeup = -1;
		}
		if (ForwardNSqueue(10)) {
			wakeup = -1;
		}

		/* ready to serve */
		ListenForMessages((wakeup > 0) ? wakeup : -1);
	}

	return 0; /* Never reached */
}
