/* $Id: sensor.c,v 1.25 2005/04/19 22:38:02 graziano Exp $ */

#include "config_nws.h"

#include <unistd.h>
#include <string.h>

#include "nws_sensor.h"
#include "periodic.h"
#include "clique_protocol.h"
#include "skills.h"
#include "osutil.h"
#include "protocol.h"
#include "messages.h"
#include "diagnostic.h"
#include "strutil.h"

/* global variables */
static void *lock = NULL;
static int weDidFork = 1;

/*
 * let's get rid of the zombies. It should be a signal handler (just add
 * int sig as parameter) but there has been some unreliable SIGCHLD
 * notification so we just poll it.
 */
static void
Reap() {
	pid_t pid;

	/* clean up the children */
	while ((pid = WaitChild()) > 0) {
		CliqueChildDeath((int)pid);
		PeriodicChildDeath((int)pid);
	}
}

/*
 * we answer the start and stop of activities remotely. It makes sense
 * only if EstablishedHost has been called.
 */
static void
SensorRequest(	Socket *sd,
		MessageHeader header) {
	DataDescriptor actDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	char *actControl;
	char *actInfo;
	char *actOptions;
	char *actRegistration;
	char *actSkill;
	int ret = SENSOR_FAILED;

	switch(header.message) {
	case ACTIVITY_START:
		actInfo = (char *)MALLOC(header.dataSize + 1, 0);
		if(actInfo == NULL) {
			ERROR("SensorRequest: out of memory\n");
			break;
		} 

		actDescriptor.repetitions = header.dataSize;
		if (!RecvData(*sd, actInfo, &actDescriptor, 1, -1)) {
			DROP_SOCKET(sd);
			ERROR("SensorRequest: data receive failed\n");
			break;
		}
		/* NULL terminate the string */
		actInfo[header.dataSize] = '\0';

		actRegistration = actInfo;
		actControl = actRegistration + strlen(actRegistration) + 1;
		actSkill = actControl + strlen(actControl) + 1;
		actOptions = actSkill + strlen(actSkill) + 1;

		/* let's try to start the activity */
		if (strcmp(actControl, CLIQUE_CONTROL_NAME) == 0) {
			if (StartCliqueActivity(actRegistration, actSkill, actOptions)) {
				ret = ACTIVITY_STARTED;
			}
		} else if (strcmp(actControl, PERIODIC_CONTROL_NAME) == 0) {
			if (StartPeriodicActivity(actRegistration, actSkill, actOptions)) {
				ret = ACTIVITY_STARTED;
			}
		}

		/* and give feedback to the caller */
		(void)SendMessage (*sd, ret, -1);
		
		FREE(actInfo);
		break;

	case ACTIVITY_STOP:
		actInfo = (char *)MALLOC(header.dataSize + 1, 0);
		if(actInfo == NULL) {
			ERROR("SensorRequest: out of memory\n");
			break;
		}

		actDescriptor.repetitions = header.dataSize;
		if(!RecvData(*sd, actInfo, &actDescriptor, 1, -1)) {
			DROP_SOCKET(sd);
			ERROR("SensorRequest: data receive failed\n");
		} else {
			actInfo[header.dataSize] = '\0';
			/* trying to stop the activity */
			if(RecognizedCliqueActivity(actInfo)) {
				if (StopCliqueActivity(actInfo)) {
					ret = ACTIVITY_STOPPED;
				}
			} else if (RecognizedPeriodicActivity(actInfo)) {
				if (StopPeriodicActivity(actInfo)) {
					ret = ACTIVITY_STOPPED;
				}
			} else {
				DDEBUG1("SensorRequest: activity not recognized %s\n", actInfo);
			}
			(void)SendMessage(*sd, ret, -1);
		}

		FREE(actInfo);
		break;

	default:
		DROP_SOCKET(sd);
		ERROR1("SensorRequest: unknown message %d\n", header.message);
	}

	/* done with the socket */
	if (*sd != NO_SOCKET) {
		SocketIsAvailable(*sd);
	}
}



int
NwsSensorInit(const char *options) {
	int ret = 1;
	static int initialized = 0;
	char *tmp;

	/* initialize only once */
	GetNWSLock(&lock);
	if (initialized) {
		ReleaseNWSLock(&lock);
		ERROR("NwsSensorInit: you can initialize only once!\n");
		return 0;
	}
	initialized = 1;
	ReleaseNWSLock(&lock);


	/* if we fork, let's take care of the children */
	tmp = GetOptionValue(options, "fork", NULL);
	if (tmp == NULL || strncmp(tmp, "yes", (strlen(tmp) > 3 ? 3 : strlen(tmp))) == 0) {
		weDidFork = 1;
	} else {
		weDidFork = 0;
	}
	FREE(tmp);

	/* initialize the periodic sensors */
	if (!PeriodicInit(options)) {
		ret = 0;
	}
	
	/* initialize the clique sensors */
	if (!CliqueInit(options)) {
		ret = 0;
	}

	/* initialize the skills */
	if (!SkillsInit(options)) {
		ret = 0;
	}

	/* registering the message we hanlde: effective only if
	 * EstablishHost has been called */
	RegisterListener(ACTIVITY_START, "ACTIVITY_START", &SensorRequest);
	RegisterListener(ACTIVITY_STOP, "ACTIVITY_STOP", &SensorRequest);

	return ret;
}

int
NwsSensorExit() {
	int ret = 1;


	/* stop all activities: only one process should do it */
#ifdef HAVE_GETPGRP
#	ifdef GETPGRP_VOID
	if (getpgrp() == getpid()) {
#	else
	if (getpgrp(getpid()) == getpid()) {
#	endif
#endif       
		ret = (StopCliqueActivity("") && StopPeriodicActivity(""));
#ifdef HAVE_GETPGRP
	}
#endif

	/* Un-register the messages we listen to */
	UnregisterListener(ACTIVITY_START);
	UnregisterListener(ACTIVITY_STOP);

	/* let's get the children */
	if (weDidFork) {
		Reap();
	}

	return ret;
}


double
NwsSensorNextWork() {
	double tmp, tmp1;

	/* let's get the children */
	if (weDidFork) {
		Reap();
	}

	/* compute the next periodic work */
	tmp = NextPeriodicWork();

	/* compute the next clique work */
	tmp1 = NextCliqueWork();

	if (tmp == 0 || (tmp1 != 0 && tmp > tmp1)) {
		tmp = tmp1;
	}

	return tmp;
}

void
NwsSensorWork() {
	PeriodicWork();
	CliqueWork();

	/* let's get the children */
	if (weDidFork) {
		Reap();
	}
}


/* now we got functions that manipulates controls, activies and series */

#define ACTIVITY_CLASS "nwsActivity"
#define CONTROL_CLASS  "nwsControl"
#define SERIES_CLASS   "nwsSeries"
#define SKILL_CLASS    "nwsSkill"


/*
** Returns the names from #list#, a tab-delimited list of attribute name/value
** pairs, as a comma-delimited list.
*/
static const char *
NwsAttributeNames(const char *list) {

  static char returnValue[255 + 1];
  const char *from;
  char *to;

  for(from = list, to = returnValue; *from != '\0'; ) {
    /* Append a comma if we're not on the first name. */
    if(from != list) {
      *to++ = ',';
    }
    /* Copy chars up to the delimiter or name/value separator. */
    while((*from != '\t') && (*from != ':') && (*from != '\0'))
      *to++ = *from++;
    /* Skip the value and delimiter. */
    if(*from == ':') {
      for(from++; (*from != '\t') && (*from != '\0'); from++)
        ; /* Nothing more to do. */
    }
    if(*from == '\t')
      from++;
  }

  *to = '\0';
  return returnValue;

}

char *
NameOfControl_r(	const char *host,
			const char *controlName) {
	char *returnValue;
	int len;

	/* sanity check */
	if (host == NULL || controlName == NULL) {
		WARN("NameOfControl_r: NULL parameter\n");
		return NULL;
	}

	/* create space first */
	len = strlen(host) + strlen(controlName) + 2;
	returnValue = (char *)MALLOC(sizeof(char) * len, 1);

	/* We use host.control as our default name */
	vstrncpy(returnValue, len, 3, host, ".", controlName);

	return returnValue;
}

/* utility functions to get the options ordered: returns the value.value
 * of the options. The memory returned needs to be freed */
static char *
OrderOptions(	const char *opts) {
	typedef struct {
		char name[15 + 1];
		char value[63 + 1];
	} optionInfo;

	optionInfo newOption;
	optionInfo *options;
	int optionsCount, i, len;
	const char *c, *endOfName;
	char text[127 + 1];
	char *ret;

	ret = NULL;

	if (opts == NULL) {
		return strdup("");
	}

	/* We sort the option values, using insertion sort, by option
	 * name so that two series with the same options listed in
	 * different order will generate the same name.  This is none too
	 * efficient, but it would be extremely strange to be sorting
	 * more than a few items here.  */
	options = NULL;
	optionsCount = 0;
	for(c= opts; GETTOK(text, c, "\t", &c);) {
		if(!GETTOK(newOption.name, text, ":", &endOfName)) {
			continue;
		}
		SAFESTRCPY(newOption.value, endOfName + 1);
		optionsCount++;
		options = (optionInfo *)REALLOC(options, optionsCount * sizeof(optionInfo), 1);
		for(i = optionsCount - 2; i >= 0; i--) {
			if(strcmp(newOption.name, options[i].name) < 0) {
				options[i + 1] = options[i];
			} else {
				break;
			}
		}
		options[i + 1] = newOption;
	}

	for(len = 2, i = 0; i < optionsCount; i++) {
		len += strlen(options[i].value) + 1;
	}
	ret = (char *)MALLOC(sizeof(char) * len, 1);
	ret[0] = '\0';
	for(i = 0; i < optionsCount; i++) {
		strcat(ret, ".");
		strcat(ret, options[i].value);
	}
	FREE(options);
		
	return ret;
}


char *
NameOfSeries_r(	const char *host,
		const char *resource,
		const char *options) {
	char *returnValue;
	int len;
	char *tmp;

	/* sanity check */
	if (host == NULL || resource == NULL || options == NULL) {
		WARN("NameOfSeries_r: NULL parameter\n");
		return NULL;
	}

	/* the name of the series is <src>.<resource>.<opts> */

	/* get only the option's value ordered */
	tmp = OrderOptions(options);

	/* Let's create the return value */
	len = strlen(resource) + strlen(host) + strlen(tmp) + 2;
	returnValue = (char *)MALLOC(sizeof(char) * len, 1);

	/* let's fill it now */
	vstrncpy(returnValue, len, 4, host, ".", resource, tmp);
	free(tmp);
	
	return returnValue;
}

char *
NameOfActivity_r(	const char *host,
			const char *skillName,
			const char *options) {
	char *ret, *tmp;
	int len;

	/* sanity check  (options can be NULL) */
	if (host == NULL || skillName == NULL) {
		WARN("NameOfActivity_r: NULL parameter\n");
		return NULL;
	}

	/* the name of an activity is <host>.<skill>.<options>.activity */

	/* get only the option's value ordered */
	tmp = OrderOptions(options);

	/* time to get the name out */
	len = strlen(host) + strlen(skillName) + strlen(tmp) + 12;
	ret = (char *)MALLOC(sizeof(char) * len, 0);
	if (ret == NULL) {
		return NULL;
	}

	/* let's fill it now */
	vstrncpy(ret, len, 6, host, ".", skillName, tmp, ".", "activity");
	free(tmp);

	return (ret);
}

Object
CreateActivityObject(	const char *name,
			const char *controlName,
			const char *host,
			const char *options,
			const char *resources,
			const char *skillName) {
	char *tmp;
	Object returnValue = NewObject();

	/* if name == NULL we generate the default name */
	if (name == NULL) {
		tmp = NameOfActivity_r(host, skillName, options);
	} else {
		tmp = strdup(name);
	}
	if (tmp == NULL) {
		ERROR("CreateActivityObject: out of memory\n");
		return NO_OBJECT;
	}

	AddNwsAttribute(&returnValue, "name", tmp);
	free(tmp);
	AddNwsAttribute(&returnValue, "objectclass", ACTIVITY_CLASS);
	AddNwsAttribute(&returnValue, "controlName", controlName);
	AddNwsAttribute(&returnValue, "host", host);
	AddNwsAttribute(&returnValue, "option", NwsAttributeNames(options));
	AddNwsAttribute(&returnValue, "resource", NwsAttributeNames(resources));
	AddNwsAttribute(&returnValue, "skillName", skillName);
	AddOptionToObject(&returnValue, options);

	return(returnValue);
}


Object
CreateControlObject(	const char *name,
			const char *controlName,
			const char *host,
			const char *options,
			const char *skills) {
	char *tmp;
	Object returnValue = NewObject();

	/* if name == NULL we generate the default name */
	if (name == NULL) {
		tmp = NameOfControl_r(host, controlName);
	} else {
		tmp = strdup(name);
	}
	if (tmp == NULL) {
		ERROR("CreateControlObject: out of memory\n");
		return NO_OBJECT;
	}

	AddNwsAttribute(&returnValue, "name", tmp);
	free(tmp);
	AddNwsAttribute(&returnValue, "objectclass", CONTROL_CLASS);
	AddNwsAttribute(&returnValue, "controlName", controlName);
	AddNwsAttribute(&returnValue, "host", host);
	AddNwsAttribute(&returnValue, "option", NwsAttributeNames(options));
	AddNwsAttribute(&returnValue, "skillName", NwsAttributeNames(skills));
	AddOptionToObject(&returnValue, options);

	return(returnValue);
}


Object
CreateSeriesObject(	const char *name,
			const char *host,
			const char *label,
			const char *memory,
			const char *options,
			const char *resource,
			const char *activity) {
	char *tmp;
	Object returnValue = NewObject();

	/* if name == NULL we generate the default name */
	if (name == NULL) {
		tmp = NameOfSeries_r(host, resource, options);
	} else {
		tmp = strdup(name);
	}
	if (tmp == NULL) {
		ERROR("CreateSeriesObject: out of memory\n");
		return NO_OBJECT;
	}

	AddNwsAttribute(&returnValue, "name", tmp);
	free(tmp);
	AddNwsAttribute(&returnValue, "objectclass", SERIES_CLASS);
	AddNwsAttribute(&returnValue, "host", host);
	if (label != NULL && label[0] != '\0') {
		AddNwsAttribute(&returnValue, "label", label);
	}
	if (memory != NULL && memory[0] != '\0') {
		AddNwsAttribute(&returnValue, "memory", memory);
	}
	AddNwsAttribute(&returnValue, "resource", resource);
	if (activity != NULL && activity[0] != '\0') {
		AddNwsAttribute(&returnValue, "activity", activity);
	}
	if (options != NULL && options[0] != '\0') {
		AddNwsAttribute(&returnValue, "option", NwsAttributeNames(options));
		AddOptionToObject(&returnValue, options);
	}

	return (returnValue);
}

