/* $Id: skills.c,v 1.84 2004/11/23 19:45:13 graziano Exp $ */

#include "config_nws.h"

#include <stdlib.h>

#include "diagnostic.h"
#include "experiments.h"
#include "host_protocol.h"
#include "skills.h"
#include "osutil.h"
#include "strutil.h"

#include "connect_exp.h"
#include "disk.h"
#include "hybridCpu.h"
#include "memory.h"
#include "tcp_bw_exp.h"
#include "fstests.h"
#include "availtest.h"
#include "starttest.h"
#include "memorySpeed.h"

static int wouldYouFork = 1;
static void *lock = NULL;			/* local lock */



/* these are all the options we understand */
static const char* OPTION_FORMATS = "buffer:0_to_1_int\tmessage:0_to_1_int\tnice:0_to_10_int\tpath:1_to_10_string\ttype:0_to_1_string\tsize:0_to_1_int\ttarget:1_to_100_sensors\tfstype:0_to_1_string\tfsblock:0_to_1_int\tfssize:0_to_1_int\tfsbufmode:0_to_1_int\tfstmpdir:0_to_1_string\tfsphysmem:0_to_1_int\tmode:0_to_1_string\tloops:0_to_1_int";


/*
 * this is a single resource: a resource needs to have a label (what we
 * are measuring) a name and a enum type we use internally (defined in
 * the header file) from NWS to refer to this resource
 */
typedef struct {
	const char *label;
	const char *name;
	const MeasuredResources skillResource;
} NwsResource;


static const NwsResource myResources[RESOURCE_COUNT] = {
	{"CPU Fraction", "availableCpu", availableCpu},
	{"CPU Fraction", "currentCpu", currentCpu}, 
	{"Megabytes", "freeDisk", freeDisk,},
	{"Megabytes", "freeMemory", freeMemory,},
	{"Milliseconds", "connectTimeTcp", connectTimeTcp,},
	{"Megabits/second", "bandwidthTcp", bandwidthTcp,},
	{"Milliseconds", "latencyTcp", latencyTcp},
	{"Megabytes/second", "fsreadSpeed", fsreadSpeed},
	{"Megabytes/second", "fswriteSpeed", fswriteSpeed},
	{"Seconds", "upTime", upTime},
	{"Seconds", "startTime", startTime},
	{"Megabytes/second", "memorySpeed", memorySpeed}
};

/* this is needed because the different controls have default options
 * that need to be added to the skill options */
typedef struct {
	int control;
	const char *options;
} Controls;

/*
 * the target option is the target for the network measurement, and the
 * childToParent is the socket to use (usually a pipe) to communicate
 * to the parent the data for the automatic timeout discovery. It's used
 * internally. */
static const Controls controlOptions[ALL_CONTROLS] = {
	{PERIODIC_CONTROL, ""},
	{CLIQUE_CONTROL, "target,childToParent"}
};

/* typedef for the function that monitors the skill availability */
typedef int (*MonitorIsAvailable)(const char *options);
typedef void (*MonitorUseSkill)(const char *options, int *length, SkillResult **results);

/*
 * this is a single skill: a skill has a name, is valid only for a
 * specific control (so far we have only periodic and clique), has a
 * list of options it supports and possibly some default values for them,
 * and a list of the resources it measures. 
 *
 * MonitorIsAvailable is the function to call to check if the particular
 * skill can be exercise with the options: the function should returns 1
 * on success, 0 otherwhise. NULL assumes an always true case.
 *
 * MonitorUseSkill is the workhorse for the experiment: it does all the
 * work and will call AppendResults to add the results. If you are
 * writing a clique-enable skill, you can access the list of targets of
 * the clique automatically: it's been added for you. A stub for
 * MonitorUseSkill is here:
 * 		
 * 	const char c*;
 * 	char option[63+1], opts[255+1];
 * 	double res;
 * 	char *targets;
 *
 * 	get the default option (pippo) with a default value (ciao)
 * 	myCommonOpts = GetOptionValue(options, "pippo", "ciao");
 *
 * 	loop through options that can be specified multpile times
 * 	targets = GetOptionValue(options, "target", "");
 * 	for (c = targets; GETTOK(option, c, ",", &c);) {
 * 		snprintf(opts, sizeof(opts), "pippo:%s,target:%s",
 * 				myCommonOpts, option);
 * 		if (myExperiment(...,&res)) {
 * 			AppendResult(skill, opts, 1, res, lenght,results);
 * 		} else {
 * 			AppendResult(skill, opts, 0, 0, lenght,results);
 * 		}
 * 	}
 *	free(myCommonOpts);
 *	free(targets);
 *
 *
 *
 */
#define MAX_RESOURCES_PER_SKILL 2
typedef struct {
	const char *name;
	const int control;
	const char *options;
	const char *optionsDefault;
	const MeasuredResources resources[MAX_RESOURCES_PER_SKILL];
	const int resourcesCount;
	MonitorIsAvailable skillIsAvailable;
	MonitorUseSkill useSkill;
} NwsSkill;

static const NwsSkill mySkills[SKILL_COUNT] = {
	{"cpuMonitor", PERIODIC_CONTROL, "nice", "0", {availableCpu, currentCpu}, 2, &HybridCpuMonitorAvailable, &HybridCpuUseSkill},
	{"diskMonitor", PERIODIC_CONTROL, "path", NULL, {freeDisk}, 1, &DiskFreeAvailable, &DiskFreeUseSkill}, 
	{"memoryMonitor", PERIODIC_CONTROL, "type", "passive", {freeMemory}, 1, &MemoryMonitorAvailable, &MemoryUseSkill},
	{"tcpConnectMonitor", CLIQUE_CONTROL, "", NULL, {connectTimeTcp}, 1, &TcpConnectAvailable, &TcpConnectUseSkill}, 
	{"tcpMessageMonitor", CLIQUE_CONTROL, "buffer,message,size", "32,16,256", {bandwidthTcp, latencyTcp}, 2, &TcpLtBwAvailable, &TcpLtBwUseSkill},
	{"filesystemMonitor", PERIODIC_CONTROL, "fstype,fsblock,fssize,path,fsbufmode,fstmpdir,fsphysmem", NULL, {fsreadSpeed, fswriteSpeed}, 2, &FileSystemMonitorAvailable, &FileSystemUseSkill},
	{"availabilityMonitor", PERIODIC_CONTROL, "", NULL, {upTime}, 1, &AvailabilityMonitorAvailable, &AvailabilityUseSkill},
	{"startMonitor", PERIODIC_CONTROL, "", NULL, {startTime}, 1, &StartAvailable, &StartUseSkill},
	{"memorySpeedMonitor", PERIODIC_CONTROL, "size,mode,loops", "8192,random,1", {memorySpeed}, 1, &MemorySpeedAvailable, &MemorySpeedUseSkill}
};


/* Adds a new result with the given fields to lastResults. */
void
AppendResult(	MeasuredResources resource,
		const char *options,
		int succeeded,
		double measurement,
		int *howMany,
		SkillResult **results) {
	SkillResult *tmp;

	/* sanity check */
	if (results == NULL|| (void *)howMany == NULL) {
		ERROR("AppendResult: NULL parameters\n");
		return;
	}
	if (resource < 0 || resource > RESOURCE_COUNT) {
		ERROR1("AppendResult: unknown resource %d\n", resource);
		return;
	}

	/* jsut be sure we start right */
	if (*howMany == 0) {
		tmp = NULL;
	} else {
		tmp = *results;
	}
	/* make room for the pointer */
	tmp = REALLOC(tmp, (*howMany + 1) * sizeof(SkillResult));
	if (results == NULL) {
		ABORT("AppendResult: out of memory\n");
	}
	tmp[*howMany].resource = resource;
	tmp[*howMany].options = strdup(options);
	if (tmp[*howMany].options == NULL) {
		ABORT("AppendResult: out of memory\n");
	}
	tmp[*howMany].succeeded = succeeded;
	if (succeeded) {
		tmp[*howMany].measurement = measurement;
	} else {
		tmp[*howMany].measurement = 0.0;
	}

	/* one more result */
	*howMany = *howMany + 1;
	*results = tmp;
}


void
FreeSkillResults(	int howMany,
			SkillResult **results) {
	int i;
	SkillResult *tmp;
	
	tmp = *results;

	for(i = 0; i < howMany; i++) {
		free(tmp[i].options);
	}
	FREE(tmp);
	*results = NULL;
}


/* A message listener (see messages.h) for the skills module. */
static void
HandleSkillMessage(Socket *sd,
                   MessageHeader header) {
	int ret, i, howMany, Ichild;
	KnownSkills skill;
	char *tmp = NULL, *name;
	const char *resName;
	DataDescriptor descriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	DataDescriptor doubleDescriptor = SIMPLE_DATA(DOUBLE_TYPE, 1);
	SkillResult *results = NULL;


	ret = Ichild = 0;

	switch (header.message) {
	case SKILL_EXPERIMENT:
		if (wouldYouFork) {
			pid_t pid;

			if(!CreateLocalChild(&pid, NULL, NULL)) {
				ERROR("HandleSkillMessage: fork failed.\n");
				return;
			}
			if(pid > 0) {
				/* Parent process. */
				PassSocket(sd, pid);
				return;
			}
			/* child process */
			Ichild = 1;
		}

		/* get the options */
		descriptor.repetitions = header.dataSize;
		tmp = (char *)MALLOC(descriptor.repetitions + 1);
		if (tmp == NULL) {
			ERROR("HandleSkillMessage: out of memory\n");
			break;
		}
		if (!RecvData(*sd, tmp, &descriptor, 1, -1)) {
			ERROR("HandleSkillMessage: receive failed\n");
			break;
		}
		tmp[header.dataSize] = '\0';

		/* search for the skill */
		name = NwsAttributeValue_r(FindNwsAttribute(tmp, "skillName"));
		if (name == NULL) {
			ERROR("HandleSkillMessage: no skillName attribute\n");
			break;
		}
		for (i = 0; i < SKILL_COUNT; i++) {
			skill = (KnownSkills) i;
			if (strcmp(name, SkillName(skill)) == 0) {
				break;
			}
		}
		FREE(name);
		if (i >= SKILL_COUNT) {
			ERROR("HandleSkillMessage: no known skill\n");
			break;
		}

		UseSkill(skill, tmp, -1, &results, &howMany);
		for (i = 0; i < howMany; i++) {
			if (!results[i].succeeded) {
				WARN1("HandleSkillMessage: skill %s failed\n", SkillName(skill));
				continue;
			}
			/* we got at least one result */
			ret = 1;

			/* let's send the result back */
			resName = ResourceName(results[i].resource);
			descriptor.repetitions = strlen(resName);
			if (!SendMessageAndDatas(*sd,
					SKILL_RESULT,
					resName,
					&descriptor,
					1,
					&results[i].measurement,
					&doubleDescriptor,
					1,
					-1)) {
				ERROR1("HandleSkillMessage: failed to send SKILL_RESULT on %d\n", *sd);
				break;
			}
		}
		/* free the memory */
		FreeSkillResults(howMany, &results);
		if (ret != 0) {
			SendMessage(*sd, SKILL_FAILED, -1);
		}

		break;

	}
	FREE(tmp);
	DROP_SOCKET(sd);

	if (Ichild) {
		exit(!ret);
	}
}


char *
GetOptionValue(const char *options,
               const char *name,
               const char *defaultValue) {
	const char *c;
	size_t nameLen = strlen(name);
	char *returnValue;
	const char *value;
	const char *valueEnd;
	size_t valuesLen = 0;

	/* sanity check */
	if (options == NULL || name == NULL) {
		valuesLen = 0;
	} else for (c = strstr(options, name); c; c = strstr(c + 1, name)) {
		/* Find the total length of all #name# option values in
		 * #options#. */
		if ((*(c + nameLen) != ':') || ((c != options) && (*(c - 1) != '\t')))
			continue; /* Bogus match. */
		value = c + nameLen + 1;
		valueEnd = strchr(value, '\t');
		valuesLen += (valueEnd == NULL) ? strlen(value) : (valueEnd - value);
		valuesLen++;
	}

	if (valuesLen == 0) {
		if (defaultValue != NULL) {
			returnValue = strdup(defaultValue);
			if (returnValue == NULL) {
				ABORT("GetOptionValue: out of memory\n");
			}
		} else {
			returnValue = NULL;
		}
		return returnValue;
	}

	/* Merge all #name# option values into a single, comma-delimited
	 * list. */
	returnValue = (char *)MALLOC(valuesLen);
	memset(returnValue, 0, valuesLen);
	for(c = strstr(options, name); c != NULL; c = strstr(c + 1, name)) {
		if ((*(c + nameLen) != ':') || ((c != options) && (*(c - 1) != '\t')))
			continue; /* Bogus match. */
		value = c + nameLen + 1;
		valueEnd = strchr(value, '\t');
		if (*returnValue != '\0') {
			strcat(returnValue, ",");
		}
		strncat(returnValue, value, (valueEnd == NULL) ? strlen(value) : (valueEnd - value));
	}

	return returnValue;
}


const char *
ResourceLabel(MeasuredResources resource) {
	if (resource < 0 || resource >= RESOURCE_COUNT) {
		return NULL;
	}

	return myResources[resource].label;
}


const char *
ResourceName(MeasuredResources resource) {
	if (resource < 0 || resource >= RESOURCE_COUNT) {
		return NULL;
	}

	return myResources[resource].name;
}


int
SkillAvailableForControl(	KnownSkills skill,
				const char *options,
				int control) {
	int ret;

	/* sanity check */
	if (skill < 0 || skill >= SKILL_COUNT || control < 0 || control >= SKILL_COUNT) {
		WARN("SkillAvailableForControl: wrong parameters\n");
		return 0;
	}

	/* we check right away if the skill is right for the control */
	if (control != ALL_CONTROLS && control != mySkills[skill].control) {
		return 0;
	}

	ret = 1;
	if (mySkills[skill].skillIsAvailable != NULL) {
		ret = mySkills[skill].skillIsAvailable(options);
	}

	return ret;
}


const char *
SkillName(KnownSkills skill) {
	if (skill < 0 || skill >= SKILL_COUNT) {
		return NULL;
	}

	return mySkills[skill].name;
}

const char *
SkillSupportedOptions(	const KnownSkills skill) {
	/* sanity check */
	if (skill < 0 || skill > SKILL_COUNT) {
		return NULL;
	}
	
	return (mySkills[skill].options);
}


char *
SkillOptionDefault(	const KnownSkills skill,
			const char *option) {
	const char *c, *v;
	char name[63+1], value[63+1], *ret;

	/* sanity check */
	if (option == NULL || skill < 0 || skill > SKILL_COUNT) {
		return NULL;
	}

	/* let's find if we know the option for the skill, and wich
	 * position it has */
	ret = NULL;
	v = mySkills[skill].optionsDefault;
	c = mySkills[skill].options;
	while (GETTOK(name, c, ",", &c) && GETTOK(value, v, ",", &v)) {
		if (strcmp(option, name) == 0) {
			ret = strdup(value);
			if (ret == NULL) {
				ERROR("SkillOptionDefault: out of memory\n");
			}
			break;
		}
	}

	return ret;
}

void
SkillOptions(KnownSkills skill,
             const char *options,
             char *toWhere,
	     int len) {
	const char *c;
	char option[63 + 1];
	char *value;
	int i;

	*toWhere = '\0';
	GetNWSLock(&lock);
	i = 0;
	value = NULL;
	for(c = mySkills[skill].options; GETTOK(option, c, ",", &c); ) {
		value = GetOptionValue(options, option, NULL);
		if (value != NULL) {
			if (len < i + strlen(option) + strlen(value) + 2) {
				ERROR("SkillOptions: option too long\n");
				FREE(value);
				break;
			}
			if(*toWhere != '\0') {
				strcat(toWhere, "\t");
			}
			strcat(toWhere, option);
			strcat(toWhere, ":");
			strcat(toWhere, value);
		}
		FREE(value);
	}
	ReleaseNWSLock(&lock);
	FREE(value);
}

char *
SkillOptionFormat(	const char *option) {
	char *c, *value;
	
	value = NULL;
	c = FindNwsAttribute((Object)OPTION_FORMATS, option);
	if (c != NULL) {
		value = NwsAttributeValue_r(c);
	}

	return value;
}


int
SkillResources(KnownSkills skill,
               const MeasuredResources **resources,
               int *length) {
	/* sanity check */
	if (skill < 0 || skill > SKILL_COUNT) {
		INFO("SkillResources: wrong skill\n");
		return 0;
	}
	if (length == NULL) {
		INFO("SkillResources: NULL parameter\n");
		return 0;
	}

	if ((void *)(resources) != NULL) {
		*resources = mySkills[skill].resources;
	}
	*length = mySkills[skill].resourcesCount;

	return 1;
}


int
SkillsInit(	const char *opts) {
	int i;
	KnownSkills skill;
	Object reg;
	char options[255], *tmp;
	static int initialized = 0;

	/* initialize only once */
	GetNWSLock(&lock);
	if (initialized) {
		ReleaseNWSLock(&lock);
		ERROR("SkillsInit: you can initialize only once!\n");
		return 0;
	}
	initialized = 1;
	ReleaseNWSLock(&lock);

	options[0] = '\0';

	for (i = 0; i < SKILL_COUNT; i++) {
		skill = (KnownSkills) i;
		if (SkillAvailable(skill, opts)) {
			/* let's write a skill object to be registerd */
			reg = NewObject();
			/* name is host.skill format */
			snprintf(options, sizeof(options), "%s.%s", EstablishedRegistration(), mySkills[i].name);
			AddNwsAttribute(&reg, "name", options);
			AddNwsAttribute(&reg, "objectclass", "nwsSkill");
			AddNwsAttribute(&reg, "host", EstablishedRegistration());
			AddNwsAttribute(&reg, "option", mySkills[i].options);
			AddNwsAttribute(&reg, "skillName", mySkills[i].name);
			SkillOptions(skill, OPTION_FORMATS, options, 255);
			AddOptionToObject(&reg, options);

			RegisterObject(reg);
			FreeObject(&reg);
		}
	}
	RegisterListener(SKILL_EXPERIMENT, "SKILL_EXPERIMENT", &HandleSkillMessage);
	
	/* remember if we need to fork */
	tmp = GetOptionValue(opts, "fork", NULL);
	if (tmp == NULL || strncmp(tmp, "yes", (strlen(tmp) > 3 ? 3 : strlen(tmp)))) {
		wouldYouFork = 1;
	} else {
		wouldYouFork = 0;
	}
	FREE(tmp);

	return 1;
}


void
UseSkill(	KnownSkills skill,
		const char *options,
		double timeOut,
		SkillResult **results,
		int *length) {
	const char *c;
	char *opts,
		option[255+1],
		*n;
	int i;
		
	*length = 0;			/* not yet data */

	/* let's check if the skill is valid */
	if (skill < 0 || skill > SKILL_COUNT) {
		ERROR("UseSkill: unknown skill\n");
		return;
	}

	opts = NewObject();
	if (opts == NULL) {
		ERROR("UseSkill: out of memory\n");
		return;
	}

	/* add timeout and defaults options for the skill */
	snprintf(option, sizeof(option), "%.2f", timeOut);
	AddNwsAttribute(&opts, "timeout", option);
	
	/* add the options of the particular skill */
	for (c = mySkills[skill].options; GETTOK(option, c, ",", &c);) {
		/* the passed in parameters gets precedence */
		n = GetOptionValue(options, option, NULL);
		if (n != NULL) {
			AddNwsAttribute(&opts, option, n);
			FREE(n);
			continue;
		}

		/* use the default if exists */
		n = SkillOptionDefault(skill, option);
		if (n != NULL) {
			AddNwsAttribute(&opts, option, n);
			FREE(n);
		}
	}

	/* add the option coming from the control */
	for (c = NULL, i = 0; i < ALL_CONTROLS; i++) {
		if (mySkills[skill].control == controlOptions[i].control) {
			c = controlOptions[i].options;
			break;
		}
	}
	while (GETTOK(option, c, ",", &c)) {
		/* the passed in parameters gets precedence */
		n = GetOptionValue(options, option, NULL);
		if (n != NULL) {
			AddNwsAttribute(&opts, option, n);
			FREE(n);
			continue;
		}

		/* use the default if exists */
		n = SkillOptionDefault(skill, option);
		if (n != NULL) {
			AddNwsAttribute(&opts, option, n);
			FREE(n);
		}
	}

	/* let's check if we can run the skill and run it in case */
	if (mySkills[skill].skillIsAvailable && mySkills[skill].useSkill) {
		if (mySkills[skill].skillIsAvailable(opts)) {
			mySkills[skill].useSkill(opts, length, results);
		} else {
			INFO2("UseSkill: skill %s not available (options %s)\n", mySkills[skill].name, opts);
		}
	}
	FreeObject(&opts);
}


int
RemoteUseSkill(	struct host_cookie *cookie,
		char *opts,
		double tout,
		char ***resources,
		double *d,
		int *howMany) {
	size_t dataSize;
	DataDescriptor objDes = SIMPLE_DATA(CHAR_TYPE, 0);
	DataDescriptor meas = SIMPLE_DATA(DOUBLE_TYPE, 1);
	char *name;
	int len, i;

	/* sanity check */
	if (cookie == NULL || opts == NULL || resources == NULL) {
		ERROR("RemoteUseSkill: NULL parameter(s)\n");
		return 0;
	}

	/* let's see how many resources are monitored by this skill */
	name = NwsAttributeValue_r(FindNwsAttribute(opts, "skillName"));
	if (name == NULL) {
		ERROR("RemoteUseSkill: missing skillName!\n");
		return 0;
	}
	for (dataSize = 0; dataSize < SKILL_COUNT; dataSize++) {
		if (strcmp(name, SkillName((KnownSkills)dataSize)) == 0) {
			break;
		}
	}
	free(name);
	if (dataSize >= SKILL_COUNT) {
		ERROR("RemoteUseSkill: unknown skillName\n");
		return 0;
	}

	/* let's see how many resources we handle */
	SkillResources((KnownSkills)dataSize, NULL, &len);

	/* let's send the skill/options blob */
	objDes.repetitions = strlen(opts);

	/* let's try to talk with the host */
	if (!ConnectToHost(cookie, NULL) || !SendMessageAndData(cookie->sd, SKILL_EXPERIMENT, opts, &objDes, 1, tout)) {
		ERROR2("RemoteUseSkill: failed to send SKILL_EXPERIMENT to %s:%d\n", cookie->name, cookie->port);
		return 0;
	}

	/* makes room for the resources name */
	*resources = (char **)MALLOC(sizeof(char*) * len);
	if (*resources == NULL) {
		ERROR("RemoteUseSkill: out of memory\n");
		return 0;
	}
	for (i = 0; i < len; i++) {
		if (!RecvMessage(cookie->sd, SKILL_RESULT, &dataSize, tout)) {
			ERROR("RemoteUseSkill: failed to receive data\n");
			return 0;
		}

		/* we receive the descriptions first: the length is
		 * dataSize - the size of a double ... */
		objDes.repetitions = dataSize - HomogenousDataSize(DOUBLE_TYPE, 1, NETWORK_FORMAT);
		(*resources)[i] = (char *)MALLOC(objDes.repetitions + 1);
		if ((*resources)[i] == NULL) {
			ERROR("RemoteUseSkill: out of memory\n");
			return 0;
		}
		if (!RecvData(cookie->sd, (*resources)[i], &objDes, 1, tout)) {
			ERROR("RemoteUseSkill: failed to receive data\n");
			return 0;
		}
		(*resources)[i][objDes.repetitions] = '\0';

		/* now receive the measurement */
		if (!RecvData(cookie->sd, &d[i], &meas, 1, tout)) {
			ERROR("RemoteUseSkill: failed to receive data\n");
			return 0;
		}
	}
	*howMany = i;

	return 1;
}




