/* NVTV direct backend -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Backend for direct access of the card.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <pci/pci.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/ioctl.h>

#include "debug.h"
#include "backend.h"
#include "back_direct.h"
#include "nv_type.h"
#include "nv_tv.h"
#include "data.h"

#define INIT_PCI_VENDOR_INFO
#include "xf86PciInfo.h"

/* -------- Extract from xf86Mode -------- */

#define min(a, b) (((a) < (b)) ? (a) : (b))
#define max(a, b) (((a) > (b)) ? (a) : (b))

/* $XFree86: xc/programs/Xserver/hw/xfree86/common/xf86Mode.c,v 1.33 2000/09/26 15:57:08 tsi Exp $ */

/*
 * Copyright (c) 1997,1998 by The XFree86 Project, Inc.
 *
 * Authors: Dirk Hohndel <hohndel@XFree86.Org>
 *          David Dawes <dawes@XFree86.Org>
 *
 * This file includes helper functions for mode related things.
 */

/* ... */

/*
 * xf86SetModeCrtc
 *
 * Initialises the Crtc parameters for a mode.  The initialisation includes
 * adjustments for interlaced and double scan modes.
 */

void xf86SetModeCrtc(DisplayModePtr p, int adjustFlags)
{
    if ((p == NULL) || ((p->type & M_T_CRTC_C) == M_T_BUILTIN))
	return;

    p->CrtcHDisplay             = p->HDisplay;
    p->CrtcHSyncStart           = p->HSyncStart;
    p->CrtcHSyncEnd             = p->HSyncEnd;
    p->CrtcHTotal               = p->HTotal;
    p->CrtcHSkew                = p->HSkew;
    p->CrtcVDisplay             = p->VDisplay;
    p->CrtcVSyncStart           = p->VSyncStart;
    p->CrtcVSyncEnd             = p->VSyncEnd;
    p->CrtcVTotal               = p->VTotal;
    if ((p->Flags & V_INTERLACE) && (adjustFlags & INTERLACE_HALVE_V))
    {
        p->CrtcVDisplay         /= 2;
        p->CrtcVSyncStart       /= 2;
        p->CrtcVSyncEnd         /= 2;
        p->CrtcVTotal           /= 2;
    }
    if (p->Flags & V_DBLSCAN) {
        p->CrtcVDisplay         *= 2;
        p->CrtcVSyncStart       *= 2;
        p->CrtcVSyncEnd         *= 2;
        p->CrtcVTotal           *= 2;
    }
    if (p->VScan > 1) {
        p->CrtcVDisplay         *= p->VScan;
        p->CrtcVSyncStart       *= p->VScan;
        p->CrtcVSyncEnd         *= p->VScan;
        p->CrtcVTotal           *= p->VScan;
    }
    p->CrtcHAdjusted = FALSE;
    p->CrtcVAdjusted = FALSE;

    /*
     * XXX
     *
     * The following is taken from VGA, but applies to other cores as well.
     */
    p->CrtcVBlankStart = min(p->CrtcVSyncStart, p->CrtcVDisplay);
    p->CrtcVBlankEnd = max(p->CrtcVSyncEnd, p->CrtcVTotal);
    if ((p->CrtcVBlankEnd - p->CrtcVBlankStart) >= 127) {
        /* 
         * V Blanking size must be < 127.
         * Moving blank start forward is safer than moving blank end
         * back, since monitors clamp just AFTER the sync pulse (or in
         * the sync pulse), but never before.
         */
        p->CrtcVBlankStart = p->CrtcVBlankEnd - 127;
    }
    p->CrtcHBlankStart = min(p->CrtcHSyncStart, p->CrtcHDisplay);
    p->CrtcHBlankEnd = max(p->CrtcHSyncEnd, p->CrtcHTotal);
    if ((p->CrtcHBlankEnd - p->CrtcHBlankStart) >= 63 * 8) {
        /*
         * H Blanking size must be < 63*8. Same remark as above.
         */
        p->CrtcHBlankStart = p->CrtcHBlankEnd - 63 * 8;
    }
}

/* -------- mmap on devices -------- */

/* from os-support/linux/lnx_video.c, slightly modified */

int openDevMem (CardPtr card)
{
  int fd;

  if ((fd = open(card->dev, O_RDWR)) < 0) {
    fprintf(stderr, "mapDevMem: failed to open %s (%s)\n", card->dev,
		   strerror(errno));
    exit(1);
  }
  return fd;
}

pointer mapDevMem (int fd, unsigned long Base, unsigned long Size)
{
  pointer base;
  int mapflags = MAP_SHARED;
  unsigned long realBase, alignOff;

  realBase = Base & ~(getpagesize() - 1);
  alignOff = Base - realBase;

  base = mmap((caddr_t)0, Size + alignOff, PROT_READ|PROT_WRITE,
	      mapflags, fd, (off_t)realBase);
  return (char *)base + alignOff;
}

void unmapDevMem (unsigned long Base, unsigned long Size)
{
  unsigned long alignOff = Base - (Base & ~(getpagesize() - 1));

  munmap((caddr_t)(Base - alignOff), (Size + alignOff));
}

/* -------- Handle X screen stuff -------- */

/* (This is needed in the X I2C routines) */

static ScrnInfoRec fakeScreen0 =
 {scrnIndex:0, driverPrivate: NULL};

static ScrnInfoPtr fakeScreens[1] = {&fakeScreen0};

ScrnInfoPtr *xf86Screens = fakeScreens;

/* -------- State of common direct backend driver -------- */

/* The state of the backend consists of:
 *
 * - an NVRec (to be compatible to X)
 * - the current card
 * - the current chip (in bdir_nv.TvDev,TvFunc)
 * - the current chip function table (in bdir_data) 
 * - the settings (in bdir_set)
 * - the crt registers before settings (in bdir_crt)
 * - the chip registers before settings (in bdir_tv)
 *
 * To communicate with the (pseudo-X) driver routines, it also keeps a 
 * modeline (bdir_modeline) together with its private part (bdir_priv).
 * Thus, the state also consists of:
 *
 * - the chip type (in bdir_priv.chip)
 * - the chip registers after settings (in bdir_priv.tv)
 * - the crt registers after settings (in bdir_modeline)
 *
 * bdir_modeline.PrivFlags can have following flags:
 *
 * NV_PRIV_TVMODE	tv mode or crt mode
 * NV_PRIV_DUALVIEW     dual view mode
 * NV_PRIV_BYPASS	bypass settings
 */

static CardPtr bdir_card;

static NVRec bdir_nv = 
{
  TvChain: NULL,
  TvBusses: {NULL, NULL, NULL},
};

static DataFunc *bdir_data = NULL;

static NVSettings bdir_set;
static NVCrtRegs bdir_crt;
static NVTvRegs bdir_tv;

static NVPrivate bdir_priv = {
  magic: {'n','v','t','v'}
};

static DisplayModeRec bdir_modeline = {
  prev: NULL,
  next: NULL,
  name: "nvmode",
  PrivSize: sizeof(NVPrivate),
  Private: (void *) &bdir_priv,
  PrivFlags: 0
};

/* -------- Common backend driver routines -------- */

/*
 *  Update crt regs from hardware
 */

void bdir_updateCrt (void)
{
  DisplayModeRec modeline;

  NVGetCrtRegs (&bdir_nv, &modeline);
  bdir_crt.HDisplay    = modeline.HDisplay; 
  bdir_crt.HSyncStart  = modeline.HSyncStart; 
  bdir_crt.HSyncEnd    = modeline.HSyncEnd; 
  bdir_crt.HTotal      = modeline.HTotal; 
  bdir_crt.VDisplay    = modeline.VDisplay; 
  bdir_crt.VSyncStart  = modeline.VSyncStart; 
  bdir_crt.VSyncEnd    = modeline.VSyncEnd; 
  bdir_crt.VTotal      = modeline.VTotal; 
  bdir_crt.Flags       = modeline.Flags;
  bdir_crt.PrivFlags   = modeline.PrivFlags;
}

/*
 *  Init card: read crt regs, probe chips 
 */

void bdir_initCard (void)
{
  bdir_updateCrt ();
  fakeScreen0.driverPrivate = &bdir_nv;
  if (NVTvBusInit (&fakeScreen0)) {
    bdir_probeChips ();
  }
}

/* 
 *  Open card, setup pNv, map mem, call probe
 */

void bdir_openCard (CardPtr card)
{
  NVPtr pNv;
  int fd;
  int i;

  bdir_card = card;
  pNv = &bdir_nv;
  pNv->IOAddress = card->reg_base;
  fd = openDevMem (card);
  pNv->riva.PMC     = mapDevMem(fd, pNv->IOAddress+0x00000000, 0x00001000);
  pNv->riva.PFB     = mapDevMem(fd, pNv->IOAddress+0x00100000, 0x00001000);
  pNv->riva.PEXTDEV = mapDevMem(fd, pNv->IOAddress+0x00101000, 0x00001000);
  pNv->riva.PCRTC   = mapDevMem(fd, pNv->IOAddress+0x00600000, 0x00001000);
  pNv->riva.PCIO    = mapDevMem(fd, pNv->IOAddress+0x00601000, 0x00001000);
  pNv->riva.PRAMDAC = mapDevMem(fd, pNv->IOAddress+0x00680000, 0x00001000);
  close (fd);

  pNv->Chipset = card->pci_id;
  pNv->riva.Architecture = 0x00;
  switch (pNv->Chipset) {
      case PCI_CHIP_RIVA128:
	pNv->riva.Architecture = 0x03; /* NV3 */
	break;
      case PCI_CHIP_TNT:
      case PCI_CHIP_TNT2:
      case PCI_CHIP_TNT2_A:
      case PCI_CHIP_TNT2_B:
      case PCI_CHIP_UTNT2:
      case PCI_CHIP_VTNT2:
      case PCI_CHIP_UVTNT2:
      case PCI_CHIP_ITNT2:
	pNv->riva.Architecture = 0x04; /* NV4 */
	break;
      case PCI_CHIP_GEFORCE256:
      case PCI_CHIP_GEFORCEDDR:
      case PCI_CHIP_QUADRO:
      case PCI_CHIP_GEFORCE2GTS:
      case PCI_CHIP_GEFORCE2GTS_1:
      case PCI_CHIP_GEFORCE2ULTRA:
      case PCI_CHIP_QUADRO2PRO:
      case PCI_CHIP_GEFORCE2MX:
      case PCI_CHIP_GEFORCE2MXDDR:
      case PCI_CHIP_QUADRO2MXR:
      case PCI_CHIP_GEFORCE2GO:
	pNv->riva.Architecture = 0x10; /* NV10 */
	break;
      case PCI_CHIP_GEFORCE3:
      case PCI_CHIP_GEFORCE3_1:
      case PCI_CHIP_GEFORCE3_2:
      case PCI_CHIP_GEFORCE3_3:
	pNv->riva.Architecture = 0x20; /* NV20 */
	break;
      default:
        fprintf (stderr, "Unknown pci card %04x\n", pNv->Chipset);
  }
  for (i = 0; i < NV_NBUS; i++) pNv->TvBusses[i] = NULL;
  bdir_initCard (); /* does probe */
}

/*
 * Close card, unmap memory, free pNv
 */

void bdir_closeCard (void)
{
  NVDestroyBusses (&bdir_nv);
  unmapDevMem(bdir_nv.IOAddress+0x00000000, 0x00001000);
  unmapDevMem(bdir_nv.IOAddress+0x00100000, 0x00001000);
  unmapDevMem(bdir_nv.IOAddress+0x00101000, 0x00001000);
  unmapDevMem(bdir_nv.IOAddress+0x00600000, 0x00001000);
  unmapDevMem(bdir_nv.IOAddress+0x00601000, 0x00001000);
  unmapDevMem(bdir_nv.IOAddress+0x00680000, 0x00001000);
}

/* 
 *  Probe all chips on card.
 */

void bdir_probeChips (void)
{
  ChipPtr chip, del, ins;
  I2CChainPtr chain;

  chip = bdir_card->chips; 
  while (chip) {
    del = chip;
    chip = chip->next;
    /* don't free name */
    free (del);
  }
  bdir_card->chips = NULL;
  NVDestroyDevices (&bdir_nv); 
  NVProbeTvDevices (&bdir_nv);
  ins = NULL;
  for (chain = bdir_nv.TvChain; chain; chain = chain->next)
  {
    NVSetTvDevice (&bdir_nv, chain, FALSE);
    chip = malloc (sizeof(ChipInfo));
    chip->private = chain;
    chip->next = NULL;
    chip->chip = chain->chip;
    chip->name = chain->name;
    if (ins) {
      ins->next = chip;
    } else {
      bdir_card->chips = chip;
    }
    ins = chip;
  }
  bdir_setChip (bdir_card->chips, TRUE);
}

/*
 *  set active chip
 */

void bdir_setChip (ChipPtr chip, Bool init)
{
  if (!chip) {
    bdir_priv.chip = NV_NO_CHIP;
    return;
  }
  NVSetTvDevice (&bdir_nv, (I2CChainPtr) chip->private, init);
  bdir_priv.chip = chip->chip;
  bdir_data = data_func (chip->chip);
  bdir_data->defaults (&bdir_set);
}

/*
 * apply any changes: process settings, call lower level part
 */

void bdir_apply (void)
{
  NVCrtRegs crt;
  NVTvRegs tv;

  bdir_data->clamp (&bdir_set);
  crt = bdir_crt;
  tv = bdir_tv;
  if (! (bdir_modeline.PrivFlags & NV_PRIV_BYPASS)) {
    bdir_data->setup (&bdir_set, &crt, &tv);
  }
  bdir_modeline.HDisplay    = crt.HDisplay;
  bdir_modeline.HSyncStart  = crt.HSyncStart;
  bdir_modeline.HSyncEnd    = crt.HSyncEnd;
  bdir_modeline.HTotal      = crt.HTotal;
  bdir_modeline.HSkew       = 0;
  bdir_modeline.VDisplay    = crt.VDisplay;
  bdir_modeline.VSyncStart  = crt.VSyncStart;
  bdir_modeline.VSyncEnd    = crt.VSyncEnd;
  bdir_modeline.VTotal      = crt.VTotal;
  bdir_modeline.VScan       = 0;
  bdir_modeline.Flags       = crt.Flags;
  xf86SetModeCrtc(&bdir_modeline, 0); /* does V_DBLSCAN */
  bdir_priv.tv = tv;
  NVSetTvMode (&bdir_nv, &bdir_modeline);
}

/*
 * Process NV_PRIV mode flags: TVMODE, DUALVIEW
 * check if mode supports the flags, and set set bdir_modeline.PrivFlags 
 * accordingly 
 */

void bdir_flags (int flags)
{
  if (NV_MODE_TVMODE != NV_PRIV_TVMODE) {
    ERROR ("Assertion failed: NV_MODE_TVMODE and NV_PRIV_TVMODE different.\n");
  }
  if ((bdir_crt.PrivFlags & NV_MODE_TVMODE) ^ (flags & NV_PRIV_TVMODE)) {
    ERROR ("TVMODE flags different for requested and provided mode\n");
    /* Use them anyway */    
  }
  if (! (bdir_crt.PrivFlags & NV_MODE_DUALVIEW)) {
    flags &= ~NV_PRIV_DUALVIEW;
  }
  bdir_modeline.PrivFlags = 
    flags & (NV_PRIV_TVMODE | NV_PRIV_DUALVIEW | NV_PRIV_BYPASS);
}

/*
 * change settings, update tv chip registers 
 */

void bdir_setSettings (NVSettings *set)
{
  if (set) bdir_set = *set;
  if (bdir_priv.chip == NV_NO_CHIP) return;
  bdir_apply ();
}

/*
 *  Get current settings
 */

void bdir_getSettings (NVSettings *set)
{
  if (set) *set = bdir_set;
}

/*
 *  set mode by crt and tv regs
 */

void bdir_setMode (int ModeFlags, NVCrtRegs *crt, NVTvRegs *tv)
{
  if (crt) bdir_crt = *crt;
  if (tv)  bdir_tv  = *tv;
  if (bdir_priv.chip == NV_NO_CHIP) return;
  bdir_flags (ModeFlags);
  bdir_apply ();
}
  
/* 
 *  Get current mode. Update crt from hardware.
 */

void bdir_getMode (NVCrtRegs *crt, NVTvRegs *tv)

{
  bdir_updateCrt ();
  if (crt) *crt = bdir_crt;
  if (tv)  *tv  = bdir_tv;
}

void bdir_setModeSettings (int ModeFlags, NVCrtRegs *crt, 
			   NVTvRegs *tv, NVSettings *set)
{
  if (set) bdir_set = *set;
  if (crt) bdir_crt = *crt;
  if (tv)  bdir_tv  = *tv;
  if (bdir_priv.chip == NV_NO_CHIP) return;
  bdir_flags (ModeFlags);
  bdir_apply ();
}

void bdir_setTestImage (NVTvRegs *tv, NVSettings *set)
{
  if (set) bdir_set = *set;
  if (tv)  bdir_tv  = *tv;
  if (bdir_priv.chip == NV_NO_CHIP) return;
  /* FIXME: Settings?? */
  NVSetTestImage (&bdir_nv);
}

long bdir_getStatus (int index)
{
  if (bdir_priv.chip == NV_NO_CHIP) return 0;
  return NVGetTvStatus (&bdir_nv, index);
}

NVConnect bdir_getConnection (void)
{
  if (bdir_priv.chip == NV_NO_CHIP) return CONNECT_NONE;
  return NVGetTvConnect (&bdir_nv);
}

Bool bdir_findBySize (NVSystem system, int xres, int yres, char *size, 
    NVMode *mode, NVCrtRegs *crt, NVTvRegs *tv)
{
  NVMode *nvm;

  if (bdir_priv.chip == NV_NO_CHIP) return FALSE;
  nvm = data_find (bdir_data->modes, system, xres, yres, size);
  if (!nvm) {
    return FALSE;
  } 
  bdir_data->init (system, nvm->tv);
  if (mode) *mode = *nvm;
  if (crt)  *crt = *nvm->crt;
  if (tv)   *tv  = *nvm->tv;
  return TRUE;
}

Bool bdir_findByOverscan (NVSystem system, int xres, int yres, 
    double hoc, double voc, NVMode *mode, NVCrtRegs *crt, NVTvRegs *tv)
{
  if (bdir_priv.chip == NV_NO_CHIP) return FALSE;
  /* FIXME */
  return FALSE;
}

BackFuncRec bdir_func = {
  openCard:         bdir_openCard,
  closeCard:        bdir_closeCard,
  probeChips:       bdir_probeChips,
  setChip:          bdir_setChip,
  setSettings:      bdir_setSettings,
  getSettings:      bdir_getSettings,
  setMode:          bdir_setMode,
  getMode:          bdir_getMode,
  setModeSettings:  bdir_setModeSettings,
  setTestImage:     bdir_setTestImage, 
  getStatus:        bdir_getStatus,    
  getConnection:    bdir_getConnection,
  findBySize:       bdir_findBySize, 
  findByOverscan:   bdir_findByOverscan
};

/* -------- Root backend -------- */

#define DEV_MEM "/dev/mem"

static CardPtr card_root_list = NULL;

/* 
 *  Check if root access is possible.
 */

Bool back_root_avail (void)
{
  if (getuid() != 0 || geteuid() != 0 ) return FALSE;
  return TRUE;
}

/*
 *  Find all cards in system, and return list.
 */

CardPtr back_root_init (void)
{
  struct pci_access *acc;
  struct pci_dev *dev;
  pciVendorDeviceInfo *vinfo;
  struct pciDevice *dinfo;
  CardPtr card;

  backend = &bdir_func;
  if (card_root_list) return card_root_list; 
    /* only probe once, and never free card_root_list once allocated. */

  acc = pci_alloc ();
  acc->method = PCI_ACCESS_AUTO;
  pci_init(acc);
  pci_scan_bus (acc);

  for (dev = acc->devices; dev; dev = dev->next) 
  {
    pci_fill_info (dev, PCI_FILL_IDENT);
    if (dev->vendor_id == PCI_VENDOR_NVIDIA || 
	dev->vendor_id == PCI_VENDOR_NVIDIA_SGS) 
    {
      for (vinfo = xf86PCIVendorInfoData; vinfo->VendorID; vinfo++) 
	if (vinfo->VendorID == dev->vendor_id) 
	{
	  for (dinfo = vinfo->Device; dinfo->DeviceID; dinfo++) 
	    if (dinfo->DeviceID == dev->device_id)
	    {
	      card = (CardPtr) malloc (sizeof (CardInfo));
	      card->next = card_root_list; card_root_list = card;
	      card->name = (char *) malloc (32 * sizeof (char));
	      snprintf (card->name, 30, "%s (%02x:%02x.%02x)", 
		      dinfo->DeviceName, dev->bus, dev->dev, dev->func); 
	      pci_fill_info (dev, PCI_FILL_BASES);
	      card->dev = DEV_MEM;
	      card->reg_base = dev->base_addr[0] & PCI_ADDR_IO_MASK;
	      card->pci_id = dev->device_id;
	      card->chips = NULL;
	      break;
	    }
	  break;
	}
    }
  }
  /* FIXME: pci_cleanup (acc); segfaults ... but avoid memory leak here. */
  return card_root_list;
}

/* -------- nvdev backend -------- */

#define DEV_NVCTL "/dev/nvidiactl"

#define NV_MAX_DEVICES 4
#define NV_IOCTL_CARD_INFO_FLAG_PRESENT       0x0001

#define NV_MMAP_REG_OFFSET              (0)
#define NV_MMAP_FB_OFFSET                (256 * 1024 * 1024)
#define NV_MMAP_ALLOCATION_OFFSET (1 * 1024 * 1024 * 1024)
#define NV_MMAP_AGP_OFFSET ((unsigned long)(2)*(1024 * 1024 * 1024))

#define NVIDIA_CARD_INFO      _IOWR(NVIDIA_IOCTL_MAGIC, 2, sizeof(void *))
#define NVIDIA_IOCTL_MAGIC      'F'

static CardPtr card_nvdev_list = NULL;

static char *dev_nvcard [NV_MAX_DEVICES] = {
  "/dev/nvidia0", "/dev/nvidia1", "/dev/nvidia2", "/dev/nvidia3"
};

typedef struct nvidia_ioctl_card_info
{
  int flags;               /* see below                   */
  int bus;		   /* bus number (PCI, AGP, etc)  */
  int slot;                /* card slot                   */
  int vendor_id;           /* PCI vendor id               */
  int device_id;
  int interrupt_line;
} nv_ioctl_card_info_t;

/* Check if /dev/nvidiactl is accessible, and if the kernel module
 * is already in use (hopefully by the XFree 'nvdriver' module).
 */

Bool back_nvdev_avail (Bool force)
{
  FILE *f;
  int fd;
  char mod[80];
  int size, count;
  Bool ok;

  /* Better check /proc/modules first, because access /dev/nvidiactrl may
     autoload kernel module */
  if ((f = fopen ("/proc/modules", "r"))) {
    ok = FALSE;
    while (! feof(f)) {
      fscanf (f, "%80s %i %i", mod, &size, &count);
      while (! feof(f) && fgetc(f) != '\n');
      if (strcmp ("NVdriver", mod) == 0 && count > 0) { 
	ok = TRUE;
	break;
      }
    }
    fclose (f);
    if (!ok) return FALSE;
  } else {
    if (!force) {
      fprintf (stderr, "Cannot open /proc/modules, won't allow nvdev access.\n");
      return FALSE;
    }
  }
  if ((fd = open (DEV_NVCTL, O_RDWR, 0)) < 0) return FALSE;
  close (fd);
  return TRUE;
}

CardPtr back_nvdev_init (void)
{
  int fd, error;
  struct nvidia_ioctl_card_info ci[NV_MAX_DEVICES];
  int i;
  pciVendorDeviceInfo *vinfo;
  struct pciDevice *dinfo;
  CardPtr card;

  backend = &bdir_func;
  if (card_nvdev_list) return card_nvdev_list; 
    /* only probe once, and never free card_nvdev_list once allocated. */

  fd = open(DEV_NVCTL, O_RDWR, 0);
  if (fd < 0) return NULL;
  error = ioctl (fd, NVIDIA_CARD_INFO, ci);
  if (error < 0) { close(fd); return NULL; }
  for (i = 0; i < NV_MAX_DEVICES; i++) {
    if ((ci[i].flags & NV_IOCTL_CARD_INFO_FLAG_PRESENT) == 0) {
      continue;
    }
    if (ci[i].vendor_id == PCI_VENDOR_NVIDIA || 
	ci[i].vendor_id == PCI_VENDOR_NVIDIA_SGS) 
    {
      for (vinfo = xf86PCIVendorInfoData; vinfo->VendorID; vinfo++) 
	if (vinfo->VendorID == ci[i].vendor_id) 
	{
	  for (dinfo = vinfo->Device; dinfo->DeviceID; dinfo++) 
	    if (dinfo->DeviceID == ci[i].device_id)
	    {
	      card = (CardPtr) malloc (sizeof (CardInfo));
	      card->next = card_nvdev_list; card_nvdev_list = card;
	      card->name = (char *) malloc (32 * sizeof (char));
	      snprintf (card->name, 30, "%s (%02x:%02x)", 
			dinfo->DeviceName, ci[i].bus, ci[i].slot); 
	      card->dev = dev_nvcard[i];
	      card->reg_base = NV_MMAP_REG_OFFSET;
	      card->pci_id = ci[i].device_id;
	      card->chips = NULL;
	      break;
	    }
	  break;
	}
    }
  }
  return card_nvdev_list;
}

