/* NVTV client backend -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Client backend for accessing the server
 *
 */

#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "backend.h"
#include "back_client.h"
#include "pipe.h"
#include "debug.h"

/* -------- State -------- */

static FILE *pipe_in = NULL;
static FILE *pipe_out = NULL;

static CardPtr bcl_root = NULL;
static CardPtr bcl_card = NULL;

/* -------- Driver routines -------- */

void bcl_openPipes (void)
{
  /* IMPORTANT: Open out pipe first, otherwise deadlock */
  pipe_out = fopen (PIPE_OUT, "w");
  pipe_in  = fopen (PIPE_IN, "r");
}

void bcl_closePipes (void)
{
  fclose (pipe_in );
  fclose (pipe_out);
}

void bcl_openCard (CardPtr card)
{
  CardPtr c;
  int i, index;

  DPRINTF ("bcl_open\n");
  bcl_card = card;
  bcl_openPipes ();
  /* convert card to index */
  i = index = 0;
  for (c = bcl_root; c; c = c->next) {
    i++;
    if (c == card) index = i;
  }
  pipeWriteCmd (pipe_out, PCmd_OpenCard);
  pipeWriteArgs (pipe_out, 1, sizeof(index), &index);
  pipeReadCmd (pipe_in);
  bcl_card->chips = pipeReadList (pipe_in, sizeof (ChipInfo));
}

void bcl_closeCard (void)
{
  DPRINTF ("bcl_close\n");
  pipeWriteCmd (pipe_out, PCmd_CloseCard);
  pipeWriteArgs (pipe_out, 0);
  bcl_closePipes ();
}

void bcl_probeChips (void)
{
  ChipPtr chip, del;

  DPRINTF ("bcl_probe\n");
  chip = bcl_card->chips; 
  while (chip) {
    del = chip;
    chip = chip->next;
    free (del->name);
    free (del);
  }
  pipeWriteCmd (pipe_out, PCmd_ProbeChips);
  pipeWriteArgs (pipe_out, 0);
  pipeReadCmd (pipe_in);
  bcl_card->chips = pipeReadList (pipe_in, sizeof (ChipInfo));
}

void bcl_setChip (ChipPtr chip, Bool init)
{
  ChipPtr c;
  int i, index;

  DPRINTF ("bcl_setChip %s %i\n", chip->name, init);
  /* convert chip to index */
  i = index = 0;
  for (c = bcl_card->chips; c; c = c->next) {
    i++;
    if (c == chip) index = i;
  }
  pipeWriteCmd (pipe_out, PCmd_SetChip);
  pipeWriteArgs (pipe_out, 2, sizeof(index), &index, sizeof(init), &init);
}

void bcl_setSettings (NVSettings *set)
{
  DPRINTF ("bcl_setSettings\n");
  pipeWriteCmd (pipe_out, PCmd_SetSettings);
  pipeWriteArgs (pipe_out, 1, sizeof(NVSettings), set);
}

void bcl_getSettings (NVSettings *set)
{
  DPRINTF ("bcl_getSettings\n");
  pipeWriteCmd (pipe_out, PCmd_GetSettings);
  pipeWriteArgs (pipe_out, 0);
  pipeReadCmd (pipe_in);
  pipeReadArgs (pipe_in, 1, sizeof(NVSettings), set);
}

void bcl_setMode (int ModeFlags, NVCrtRegs *crt, NVTvRegs *tv)
{
  DPRINTF ("bcl_setMode\n");
  pipeWriteCmd (pipe_out, PCmd_SetMode);
  pipeWriteArgs (pipe_out, 3, sizeof(ModeFlags), &ModeFlags,
		 sizeof(NVCrtRegs), crt, sizeof(NVTvRegs), tv);
}

void bcl_getMode (NVCrtRegs *crt, NVTvRegs *tv)
{
  DPRINTF ("bcl_getMode\n");
  pipeWriteCmd (pipe_out, PCmd_GetMode);
  pipeWriteArgs (pipe_out, 0);
  pipeReadCmd (pipe_in);
  pipeReadArgs (pipe_in, 2, sizeof(NVCrtRegs), crt, sizeof(NVTvRegs), tv);
}

void bcl_setModeSettings (int ModeFlags, NVCrtRegs *crt, 
			   NVTvRegs *tv, NVSettings *set)
{
  DPRINTF ("bcl_setModeSettings\n");
  pipeWriteCmd (pipe_out, PCmd_SetModeSettings);
  pipeWriteArgs (pipe_out, 4, sizeof(ModeFlags), &ModeFlags,
		 sizeof(NVCrtRegs), crt, sizeof(NVTvRegs), tv,
		 sizeof(NVSettings), set);
}

void bcl_setTestImage (NVTvRegs *tv, NVSettings *set)
{
  DPRINTF ("bcl_setTestImage\n");
  pipeWriteCmd (pipe_out, PCmd_SetTestImage);
  pipeWriteArgs (pipe_out, 2, sizeof(NVTvRegs), tv,
		 sizeof(NVSettings), set);
}

long bcl_getStatus (int index)
{
  long l;

  DPRINTF ("bcl_getStatus\n");
  pipeWriteCmd (pipe_out, PCmd_GetStatus);
  pipeWriteArgs (pipe_out, 1, sizeof(index), &index);
  pipeReadCmd (pipe_in);
  pipeReadArgs (pipe_in, 1, sizeof(l), &l);
  return l;
}

NVConnect bcl_getConnection (void)
{
  NVConnect c;

  DPRINTF ("bcl_getConnection\n");
  pipeWriteCmd (pipe_out, PCmd_GetConnection);
  pipeWriteArgs (pipe_out, 0);
  pipeReadCmd (pipe_in);
  pipeReadArgs (pipe_in, 1, sizeof(c), &c);
  DPRINTF ("bcl_getConnection got %i\n", c);
  return c;
}

Bool bcl_findBySize (NVSystem system, int xres, int yres, char *size, 
    NVMode *mode, NVCrtRegs *crt, NVTvRegs *tv)
{
  int n;

  DPRINTF ("bcl_findBySize %i %i,%i %s\n", system, xres, yres, size);
  pipeWriteCmd (pipe_out, PCmd_FindBySize);
  pipeWriteArgs (pipe_out, 4, sizeof(system), &system, 
		 sizeof(xres), &xres, sizeof(yres), &yres,
		 strlen(size)+1, size);
  pipeReadCmd (pipe_in);
  n = pipeReadArgs (pipe_in, 3, sizeof(NVMode), mode, 
    sizeof(NVCrtRegs), crt, sizeof(NVTvRegs), tv);
  if (mode) {
    mode->crt = crt;
    mode->tv  = tv;
  }
  return (n >= 1);
}

Bool bcl_findByOverscan (NVSystem system, int xres, int yres, 
    double hoc, double voc, NVMode *mode, NVCrtRegs *crt, NVTvRegs *tv)
{
  int n;
 
  DPRINTF ("bcl_findByOC %i %i,%i\n", system, xres, yres);
  pipeWriteCmd (pipe_out, PCmd_FindByOverscan);
  pipeWriteArgs (pipe_out, 5, sizeof(system), &system, 
		 sizeof(xres), &xres, sizeof(yres), &yres,
		 sizeof(hoc), &hoc, sizeof(voc), &voc);
  pipeReadCmd (pipe_in);
  n = pipeReadArgs (pipe_in, 3, sizeof(NVMode), mode, 
    sizeof(NVCrtRegs), crt, sizeof(NVTvRegs), tv);
  if (mode) {
    mode->crt = crt;
    mode->tv  = tv;
  }
  return (n >= 1);
}

BackFuncRec bcl_func = {
  openCard:         bcl_openCard,
  closeCard:        bcl_closeCard,
  probeChips:       bcl_probeChips,
  setChip:          bcl_setChip,
  setSettings:      bcl_setSettings,
  getSettings:      bcl_getSettings,
  setMode:          bcl_setMode,
  getMode:          bcl_getMode,
  setModeSettings:  bcl_setModeSettings,
  setTestImage:     bcl_setTestImage, 
  getStatus:        bcl_getStatus,    
  getConnection:    bcl_getConnection,
  findBySize:       bcl_findBySize, 
  findByOverscan:   bcl_findByOverscan
};

/* -------- Init -------- */

Bool back_client_avail (void)
{
  int version;
  int fd_out, fd_in;

  /* IMPORTANT: Open out pipe first, otherwise deadlock */
  fd_out = open (PIPE_OUT, O_WRONLY | O_NONBLOCK);
  if (fd_out < 0) return FALSE;
  fd_in  = open (PIPE_IN, O_RDONLY | O_NONBLOCK);
  if (fd_in < 0) { 
    close (fd_out);
    return FALSE;
  }
  close (fd_in);
  close (fd_out);
  bcl_openPipes ();
  if (!pipe_in || !pipe_out) {
    /* FIXME: error message */
    bcl_closePipes ();
    return FALSE;
  }
  pipeWriteCmd (pipe_out, PCmd_Version);
  pipeWriteArgs (pipe_out, 0);
  pipeReadCmd (pipe_in);
  pipeReadArgs (pipe_in, 1, sizeof(version), &version);
  bcl_closePipes ();
  return (version == PIPE_VERSION);
}


CardPtr back_client_init (void)
{
  CardPtr card;

  backend = &bcl_func;
  bcl_openPipes ();
  pipeWriteCmd (pipe_out, PCmd_Init);
  pipeWriteArgs (pipe_out, 0);
  bcl_root = pipeReadList (pipe_in, sizeof (CardInfo));
  bcl_card = bcl_root;
  bcl_closePipes ();
  for (card = bcl_card; card; card = card->next) {
    card->chips = NULL;
    card->dev = NULL;
  }
  return bcl_root;
}

