// file kernel/n/ppc32/mul_n2.S: O(n^2) multiplication of natural integers
/*-----------------------------------------------------------------------+
 |  Copyright 2005-2006, Michel Quercia (michel.quercia@prepas.org)      |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                        Multiplication quadratique                     |
 |                                                                       |
 +-----------------------------------------------------------------------*/

                        ; +---------------------------+
                        ; |  Multiplication droule  |
                        ; +---------------------------+

   ; entre en milieu de boucle :
   ;   r3:r4 = retenue entrante
   ;   ctr  = ceil(l/32)
   ;   r10 = multiplicateur
   ;   r11 = adresse multiplicande cadre sur un multiple de 32
   ;   r12 = adresse rsultat cadre sur un multiple de 32
   ;
   ; sortie:
   ;   (r12) <- r10 * (r11)
   ;   r3 <- retenue sortante
   ;   r4 <- dernier chiffre du rsultat
   ;
   ; registres modifis:
   ;   ctr <- 0
   ;   r5  <- ind
   ;   r11 <- adresse suivante pour le multiplicande
   ;   r12 <- adresse suivante pour le rsultat

        ; corps de boucle  drouler (6 instructions, entrer  la 6me)
#define BODY(x) \
        lwz    r5,   x(r11)     @\
        mullw  r4,   r10,  r5   @\
        mulhwu r5,   r10,  r5   @\
        addc   r4,   r4,   r3   @\
        addze  r3,   r5         @\
        stw    r4,   x(r12)

	; boucle droule pour 32 chiffres
Lsn_mulloop:
        BODY(0)@   BODY(4)@   BODY(8)@   BODY(12)
        BODY(16)@  BODY(20)@  BODY(24)@  BODY(28)
        BODY(32)@  BODY(36)@  BODY(40)@  BODY(44)
        BODY(48)@  BODY(52)@  BODY(56)@  BODY(60)
        BODY(64)@  BODY(68)@  BODY(72)@  BODY(76)
        BODY(80)@  BODY(84)@  BODY(88)@  BODY(92)
        BODY(96)@  BODY(100)@ BODY(104)@ BODY(108)
        BODY(112)@ BODY(116)@ BODY(120)@ BODY(124)
#undef BODY

	addi   r11,  r11,  128
	addi   r12,  r12,  128
	bdnz   Lsn_mulloop   
	blr

              ; +----------------------------------------------+
              ; |  Multiplication droule pour deux chiffres  |
              ; +----------------------------------------------+

   ; entre en milieu de boucle :
   ;   r4:r3 = retenue entrante
   ;   r10:r9= multiplicateur
   ;   ctr   = ceil(l/32)
   ;   r11   = adresse multiplicande cadre sur un multiple de 32
   ;   r12   = adresse rsultat cadre sur un multiple de 32
   ;
   ; sortie:
   ;   (r12) <- r9:r10 * (r11)
   ;   r4:r3 <- retenue sortante
   ;
   ; registres modifis:
   ;   ctr <- 0
   ;   r5,r6,r7,r8 <- ind
   ;   r11 <- adresse suivante pour le multiplicande
   ;   r12 <- adresse suivante pour le rsultat

        ; corps de boucle  drouler (11 instructions, entrer  la 1re)
#define BODY(x) \
        lwz    r8,   x(r11)     @\
        mullw  r5,   r9,   r8   @\
        mulhwu r6,   r9,   r8   @\
        mullw  r7,   r10,  r8   @\
        mulhwu r8,   r10,  r8   @\
        addc   r5,   r5,   r3   @\
        stw    r5,   x(r12)     @\
        adde   r6,   r6,   r7   @\
        addze  r8,   r8         @\
        addc   r3,   r6,   r4   @\
        addze  r4,   r8

	; boucle droule pour 32 chiffres
Lsn_mulloop2:
        BODY(0)@   BODY(4)@   BODY(8)@   BODY(12)
        BODY(16)@  BODY(20)@  BODY(24)@  BODY(28)
        BODY(32)@  BODY(36)@  BODY(40)@  BODY(44)
        BODY(48)@  BODY(52)@  BODY(56)@  BODY(60)
        BODY(64)@  BODY(68)@  BODY(72)@  BODY(76)
        BODY(80)@  BODY(84)@  BODY(88)@  BODY(92)
        BODY(96)@  BODY(100)@ BODY(104)@ BODY(108)
        BODY(112)@ BODY(116)@ BODY(120)@ BODY(124)
#undef BODY

	addi   r11,  r11,  128
	addi   r12,  r12,  128
	bdnz   Lsn_mulloop2
	blr

                  ; +---------------------------------------+
                  ; |  Multiplication et addition droule  |
                  ; +---------------------------------------+

   ; entre en milieu de boucle :
   ;   r4:r3 = retenue entrante
   ;   r9    = multiplicateur
   ;   ctr   = ceil(l/32)
   ;   r11 = adresse multiplicande cadre sur un multiple de 32
   ;   r12 = adresse rsultat cadre sur un multiple de 32
   ;
   ; sortie :
   ;   (r12) <- (r12) + r9 * (r11)
   ;   r3 <- retenue sortante
   ;
   ; registres modifis :
   ;   ctr <- 0
   ;   r4, r5 <- ind
   ;   r11 <- adresse suivante pour le multiplicande
   ;   r12 <- adresse suivante pour le rsultat

        ; corps de boucle  drouler (9 instructions, entrer  la 6me)
#define BODY(x) \
        lwz    r5,   x(r11)     @\
        mullw  r4,   r9,   r5   @\
        mulhwu r5,   r9,   r5   @\
        addc   r4,   r4,   r3   @\
        addze  r3,   r5         @\
        lwz    r5,   x(r12)     @\
        addc   r5,   r5,   r4   @\
        stw    r5,   x(r12)     @\
        addze  r3,   r3

	; boucle droule pour 32 chiffres
Lsn_muladdloop:
        BODY(0)@   BODY(4)@   BODY(8)@   BODY(12)
        BODY(16)@  BODY(20)@  BODY(24)@  BODY(28)
        BODY(32)@  BODY(36)@  BODY(40)@  BODY(44)
        BODY(48)@  BODY(52)@  BODY(56)@  BODY(60)
        BODY(64)@  BODY(68)@  BODY(72)@  BODY(76)
        BODY(80)@  BODY(84)@  BODY(88)@  BODY(92)
        BODY(96)@  BODY(100)@ BODY(104)@ BODY(108)
	BODY(112)@ BODY(116)@ BODY(120)@ BODY(124)
#undef BODY

	addi   r11,  r11,  128
	addi   r12,  r12,  128
	bdnz   Lsn_muladdloop
	blr

                ; +-------------------------------------------+
                ; |  Multiplication et soustraction droule  |
                ; +-------------------------------------------+

   ; entre en dbut de boucle :
   ;   r3  = retenue entrante
   ;   r9  = multiplicateur
   ;   ctr = ceil(l/32)
   ;   r11 = adresse multiplicande cadre sur un multiple de 32
   ;   r12 = adresse rsultat cadre sur un multiple de 32
   ;
   ; sortie :
   ;   (r12) <- (r12) - r9 * (r11)
   ;   r3 <- retenue sortante
   ;
   ; registres modifis :
   ;   ctr <- 0
   ;   r4, r5 <- ind
   ;   r11 <- adresse suivante pour le multiplicande
   ;   r12 <- adresse suivante pour le rsultat

        ; corps de boucle  drouler (10 instructions)
#define BODY(x) \
        lwz    r5,   x(r11)     @\
        mullw  r4,   r9,   r5   @\
        mulhwu r5,   r9,   r5   @\
        addc   r4,   r4,   r3   @\
        addze  r3,   r5         @\
        lwz    r5,   x(r12)     @\
        subfc  r5,   r4,   r5   @\
        stw    r5,   x(r12)     @\
	subfe  r5,   r5,   r5   @\
        subf   r3,   r5,   r3

	; boucle droule pour 32 chiffres
Lsn_mulsubloop:
        BODY(0)@   BODY(4)@   BODY(8)@   BODY(12)
        BODY(16)@  BODY(20)@  BODY(24)@  BODY(28)
        BODY(32)@  BODY(36)@  BODY(40)@  BODY(44)
        BODY(48)@  BODY(52)@  BODY(56)@  BODY(60)
        BODY(64)@  BODY(68)@  BODY(72)@  BODY(76)
        BODY(80)@  BODY(84)@  BODY(88)@  BODY(92)
        BODY(96)@  BODY(100)@ BODY(104)@ BODY(108)
        BODY(112)@ BODY(116)@ BODY(120)@ BODY(124)
#undef BODY

	addi   r11,  r11,  128
	addi   r12,  r12,  128
	bdnz   Lsn_mulsubloop
	blr


        ; +----------------------------------------------------------+
        ; |  Multiplication et addition droule pour deux chiffres  |
        ; +----------------------------------------------------------+

   ; entre en milieu de boucle :
   ;   r4:r3 = retenue entrante
   ;   r10:r9= multiplicateur
   ;   ctr   = ceil(l/32)
   ;   r11 = adresse multiplicande cadre sur un multiple de 32
   ;   r12 = adresse rsultat cadre sur un multiple de 32
   ;
   ; sortie:
   ;   (r12) <- (r12) - r9:r10 * (r11)
   ;   r4:r3 <- retenue sortante
   ;
   ; registres modifis:
   ;   ctr <- 0
   ;   r5,r6,r7,r8 <- ind
   ;   r11 <- adresse suivante pour le multiplicande
   ;   r12 <- adresse suivante pour le rsultat

        ; corps de boucle  drouler (13 instructions, entrer  la 1re
	; ou  la 9me avec la retenue dans r8:(r6+r4):r5)
#define BODY(x) \
        lwz    r8,   x(r11)     @\
        mullw  r5,   r8,   r9   @\
        mulhwu r6,   r8,   r9   @\
        mullw  r7,   r8,   r10  @\
        mulhwu r8,   r8,   r10  @\
        addc   r5,   r3,   r5   @\
        adde   r6,   r7,   r6   @\
        addze  r8,   r8         @\
        lwz    r3,   x(r12)     @\
        addc   r5,   r3,   r5   @\
        stw    r5,   x(r12)     @\
        adde   r3,   r4,   r6   @\
	addze  r4,   r8
	
	; boucle droule pour 32 chiffres
Lsn_muladdloop2:
        BODY(0)@   BODY(4)@   BODY(8)@   BODY(12)
        BODY(16)@  BODY(20)@  BODY(24)@  BODY(28)
        BODY(32)@  BODY(36)@  BODY(40)@  BODY(44)
        BODY(48)@  BODY(52)@  BODY(56)@  BODY(60)
        BODY(64)@  BODY(68)@  BODY(72)@  BODY(76)
        BODY(80)@  BODY(84)@  BODY(88)@  BODY(92)
        BODY(96)@  BODY(100)@ BODY(104)@ BODY(108)
        BODY(112)@ BODY(116)@ BODY(120)@ BODY(124)
#undef BODY

	addi   r11,  r11,  128
	addi   r12,  r12,  128
	bdnz   Lsn_muladdloop2
	blr

                      ; +------------------------------+
                      ; |  Multiplication par un long  |
                      ; +------------------------------+

   ; unsigned long xn(mul_1)(chiffre *a, long la, unsigned long b, chiffre *c)
   ;
   ; entre :
   ; a = naturel de longueur la
   ; b = long >= 0   ; c = naturel de longueur la, peut tre confondu avec a
   ;
   ; sortie :
   ; c <- a*b
   ; retourne la retenue

#ifdef assembly_sn_mul_1
#define L(x) Lsn_mul_1_##x
#define _a_  r3
#define _b_  r5
#define _c_  r6
#define _la_ r4

.globl _sn_mul_1
_sn_mul_1:

        neg.   r8,   _la_       ; r8 <- -la
        bne+   1f
       	li     r3,   0          ; cas la = 0: retourne 0
        blr
1:
        mflr    r0              ; r0  <- adresse de retour
	addi   _la_, _la_, 31   ; ctr <- ceil(la/32)
	srawi  _la_, _la_, 5 
        mtctr  _la_
        clrlslwi r8, r8,   27,2 ; r8 <- 4*((-la) % 32)
	subf   r11,  r8,   _a_  ; cadre a,c sur le multiple de 32 prcdent
	subf   r12,  r8,   _c_
	bcl    20,31, L(here)   ; lr <- adresse d entre dans la boucle
L(here):
	mflr   r9
/*	addis  r9,   r9, ha16(Lsn_mulloop - L(here) + 20) */
	addi   r9,   r9, lo16(Lsn_mulloop - L(here) + 20)
        mulli  r8,   r8,   6
	add    r9,   r9,   r8
        mtlr   r9
        mr     r10,  _b_        ; r10 <- multiplicateur
        lwz    r3,   0(_a_)	; r3:r4 <- a[0]*b
	mullw  r4,   _b_,  r3
	mulhwu r3,   _b_,  r3
        blrl                    ; effectue les multiplications suivantes
        mtlr   r0
        blr

#undef L
#undef _a_
#undef _b_
#undef _c_
#undef _la_
#endif /* assembly_sn_mul_1 */

                      ; +------------------------------+
                      ; |  Multiplication quadratique  |
                      ; +------------------------------+

   ;  void xn(mul_n2)(chiffre *a, long la, chiffre *b, long lb, chiffre *c)
   ;
   ;  entre :
   ;  a = naturel de longueur la
   ;  b = naturel de longueur lb
   ;  c = naturel de longueur la+lb, non confondu avec a ou b
   ;  contraintes : 0 < lb <= la
   ;
   ;  sortie :
   ;  c <- a*b

#ifdef assembly_sn_mul_n2
#define L(x) Lsn_mul_n2_##x
#define _a_  r11
#define _b_  r31
#define _c_  r12
#define _la_ r30
#define _lb_ r29
#define _muladd_ r2

#ifdef debug_mul_n2
.globl _sn_mul_n2_buggy
_sn_mul_n2_buggy:
#else
.globl _sn_mul_n2
_sn_mul_n2:
Lsn_mul_n2:
#endif

	mflr    r0              ; r0  <- adresse de retour
        stmw    r29,  -16(r1)   ; sauvegarde r29 .. r31

	; prpare le droulement de la boucle interne
	addi   _la_, r4,   31   ; la <- ceil(la/32)
        srawi  _la_, _la_,  5 
        neg    r4,   r4         ; r4 <- 4*((-la) % 32)
        clrlslwi r4, r4,   27,2 
	subf   _a_,  r4,   r3   ; cadre a,c sur le multiple de 32 prcdent
	subf   _c_,  r4,   r7 
	bcl    20,31, L(here)   ; r9 <- adresse courante
L(here):
	mflr   r9
        mulli  _muladd_, r4, 13 ; muladd <- adresse d entre dans la boucle interne
/*	addis  _muladd_, _muladd_, ha16(Lsn_muladdloop2 - L(here)) */
	addi   _muladd_, _muladd_, lo16(Lsn_muladdloop2 - L(here))
	add    _muladd_, _muladd_, r9
        mr     _b_,   r5        ; sauve &b
        andi.  r8,    r6,   1	; CR0 <- lb impair ?
        srawi  _lb_,  r6,   1   ; lb <- floor(lb/2)
	beq    L(even)

	; cas lb impair: c <- b[0]*a
        mulli  r8,   r4,   6    ; lr <- adresse entre 1re boucle
/*	addis  r8,   r8,   ha16(Lsn_mulloop - L(here)) */
	addi   r8,   r8,   lo16(Lsn_mulloop - L(here))
	add    r8,   r8,   r9
        mtlr   r8
	lwz    r10,  0(_b_)     ; r10 <- multiplicateur
	li     r3,   0          ; r3 <- 0 (retenue)
        mtctr  _la_             ; init compteur
	blrl                    ; c[0..la-1] <- b[0]*a
	stw    r3,   0(_c_)     ; c[la] <- retenue
        addi   _b_,  _b_,  4    ; b ++
        addi   _c_,  _c_,  4    ; c ++
	b      L(next)

	; cas lb pair: c <- b[0]:b[1]*a
L(even):
        mulli  r8,   r4,   11   ; lr <- adresse entre 1re boucle
/*	addis  r8,   r8,   ha16(Lsn_mulloop2 - L(here)) */
	addi   r8,   r8,   lo16(Lsn_mulloop2 - L(here))
	add    r8,   r8,   r9
        mtlr   r8

	; boucle sur les chiffres suivants de b
L(loop):
	lwz    r9,   0(_b_)     ; r10:r9 <- multiplicateur
	lwz    r10,  4(_b_)
	li     r3,   0          ; r4:r3 <- 0 (retenue)
	li     r4,   0
        mtctr  _la_             ; init compteur
	blrl                    ; c[i..i+la-1] += b[i]:b[i+1]*a
	stw    r3,   0(_c_)     ; c[i+la]:c[i+la+1] <- retenue
	stw    r4,   4(_c_)
        addi   _b_,  _b_,  8    ; b += 2
        addi   _c_,  _c_,  8    ; c += 2
        subi   _lb_, _lb_, 1    ; lb -= 2
L(next):
	and.   _lb_, _lb_, _lb_
        mtlr   _muladd_         ; lr <- adresse de saut
        slwi   r8,   _la_, 7    ; recadre &a et &c
        subf   _a_,  r8,   _a_
        subf   _c_,  r8,   _c_
	bgt+   L(loop)

        ; Termin
        mtlr   r0               ; lr <- adresse de retour
        lmw    r29,  -16(r1)    ; rcupre r29 .. r31
        blr

#undef L
#undef _a_
#undef _b_
#undef _c_
#undef _la_
#undef _lb_
#undef _muladd_
#endif /* assembly_sn_mul_n2 */
#if !defined(assembly_sn_mul_n2) || defined(debug_mul_n2)
REPLACE(sn_mul_n2)
#endif




                                 ; +---------+
                                 ; |  Carr  |
                                 ; +---------+

   ;  void xn(sqr_n2)(chiffre *a, long la, chiffre *b)
   ;
   ;  entre :
   ;  a = naturel de longueur la
   ;  b = naturel de longueur 2*la, non confondu avec a
   ;  contraintes : 0 < la
   ;
   ;  sortie :
   ;  b <- a^2

#ifdef assembly_sn_sqr_n2
#define L(x) Lsn_sqr_n2_##x
#define _a_  r11
#define _b_  r31
#define _c_  r12
#define _la_ r30
#define _lb_ r29
#define _x_  r28
#define _y_  r27
#define _muladd_ r2

#ifdef debug_mul_n2
.globl _sn_sqr_n2_buggy
_sn_sqr_n2_buggy:
#else
.globl _sn_sqr_n2
_sn_sqr_n2:
Lsn_sqr_n2:
#endif

        mflr   r0               ; r0  <- adresse de retour
        stmw    r27,  -24(r1)   ; sauvegarde r27 .. r31
	subi   r1,   r1,  32
        mr     _b_,  r3         ; b <- &a[0]
	la     _a_,  4(r3)      ; a <- &a[1]
	la     _c_,  4(r5)      ; c <- &b[1]

	; initialise b avec les carrs des chiffres de a
        mtctr  r4               ; ctr <- la
        li     r8,   0          ; r8 <- 0 ( = msb(a[-1]) )
L(squares):
        lwz    r6,   0(r3)      ; r6 <- a[i]
        and    r8,   r6,   r8   ; r8 <- a[i]*msb(a[i-1])
        mullw  r7,   r6,   r6   ; r8:r7 <- a[i]*(a[i] + msb(a[i-1])
        addc   r7,   r7,   r8
        mulhwu r8,   r6,   r6
        addze  r8,   r8
        stw    r7,   0(r5)      ; b[2i+1]:b[2i] <- produit
        stw    r8,   4(r5)
        srawi  r8,   r6,   31   ; r8 <- -msb(a[i])
        addi   r3,   r3,   4    ; avance les pointeurs
        addi   r5,   r5,   8
        bdnz   L(squares)
	subi   r4,   r4,   1    ; la--

	; ajoute les doubles produits, deux ranges  la fois
        neg.   _la_, r4         ; la <- 1 - la
	beq-   L(done)
	addi   _lb_, r4,   31   ; lb <- ceil((la-1)/32)
        srwi   _lb_, _lb_,  5
        clrlslwi r8, _la_, 27,2 ; r8 <- 4*((1-la)%32)
	subf   _a_,  r8,   _a_  ; cadre a,c sur le multiple de 32 prcdent
	subf   _c_,  r8,   _c_
	bcl    20,31, L(here)   ; muladd <- adresse d entre dans la boucle interne
L(here):
	mflr   r9
        mulli  _muladd_, r8, 13
/*	addis  _muladd_, _muladd_, ha16(Lsn_muladdloop2 - L(here) + 32) */
	addi   _muladd_, _muladd_, lo16(Lsn_muladdloop2 - L(here) + 32)
	add    _muladd_, _muladd_, r9
        li     _x_,  0          ; x <- 0 (= a[-1])
        li     _y_, -2          ; y[0] <- 0 (retenue en fin de boucle)

L(loop):
        mtlr   _muladd_         ; lr <- adresse de saut
        addc   _x_,  _x_,  _x_  ; CA <- msb(a[i-1])
        lwz    r9,   0(_b_)     ; r10:r9 <- 2*a[i]:a[i+1] + msb(a[i-1])
        lwz    _x_,  4(_b_)
        adde   r9,   r9,   r9
        adde   r10,  _x_,  _x_
        mullw  r5,   _x_,  r9   ; r4:r5 <- (2*a[i] + msb(a[i-1]))*a[i+1]
        mulhwu r4,   _x_,  r9
        li     r6,   0          ; r8,r6 <- 0 (retenues)
        li     r8,   0
        mtctr  _lb_             ; init compteur
	blrl                    ; c[2i+1..i+la-1] += 2*a[i]:a[i+1]*a[i+2..la-1]
        lwz    r5,   0(_c_)     ; ajoute la retenue  c[i+la]:c[i+la+1]
        lwz    r6,   4(_c_)
        srawi  _y_,  _y_,  1    ; CA <- retenue tour prcdent
        adde   r3,   r3,   r5
        adde   r4,   r4,   r6
        addme  _y_,  _y_        ; y <- nouvelle retenue
        stw    r3,   0(_c_)     ; range le rsultat
        stw    r4,   4(_c_)
        addi   _la_, _la_, 2    ; la -= 2
	li     r3,   0          ; r3 <- (la%32 <= 1) ? 0 : -1 
        srawi  r4,   _la_, 1
        srawi  r4,   r4,   4
        addme  r3,   r3
        add    _lb_, _lb_, r3      ; mise  jour compteur
        andi.  r3,   r3,   13*128  ; mise  jour adresse de saut
        and.   _la_, _la_, _la_ ; la = 0 ?
        addi   r3,   r3,  -13*8
        subf   _muladd_, r3, _muladd_ 
        slwi   r3,   _lb_, 7    ; restaure &a et &c
        subf   _a_,  r3,   _a_
        subf   _c_,  r3,   _c_
        la     _b_,  8(_b_)     ; b += 2
        la     _c_,  8(_c_)     ; c += 2
	blt+   L(loop)

	andi.  _y_,  _y_,  1     ; ajoute la dernire retenue
	beq    L(done)
	lwz    r3,   0(_c_)
	addi   r3,   r3,  1
	stw    r3,   0(_c_)

        ; Termin
L(done):
        mtlr   r0               ; lr <- adresse de retour
	addi   r1,  r1,  32
        lmw    r27,  -24(r1)    ; rcupre r27 .. r31
        blr

#undef L
#undef  _a_
#undef  _b_
#undef  _c_
#undef  _la_
#undef  _lb_
#undef  _x_
#undef  _y_
#undef  _muladd_
#endif /* assembly_sn_sqr_n2 */
#if !defined(assembly_sn_sqr_n2) || defined(debug_mul_n2)
REPLACE(sn_sqr_n2)
#endif
