/*  VER 137   TAB P   $Id: lock.c,v 1.23.2.2 2002/10/01 08:54:59 egil Exp $
 *
 *  lock file handling 
 *
 *  copyright 1996, 1997, 1998 Egil Kvaleberg, egil@kvaleberg.no
 *  the GNU General Public License applies
 *
 *  assuming "spool" is the name of the outgoing spool,
 *  the following conventions apply:
 *
 *  for C News:
 *      outgoing lock:  NEWSHOME/LOCKbatch
 *      incoming lock:  NEWSHOME/LOCKinput
 *      spool lock:     BATCH/spoolname/LOCKb
 *      spool file:     BATCH/spoolname/togo
 *
 *      newsx lock:     NEWSHOME/LOCKnewsx.hostname
 *
 *  for INN:
 *      outgoing lock:  LOCKS/LOCK.sendbatch
 *      spool lock:     LOCKS/LOCK.spoolname
 *      spool file:     BATCH/spoolname
 *
 *      newsx lock:     LOCKS/LOCKnewsx.hostname
 *
 *      LOCKS is usually the same as NEWSHOME
 *      BUG: We are ignoring LOCK_STYLE
 *      BUG: What about: shlock -p $$ -f lockfile
 *
 *  we don't need the general lock
 *
 *  $Log: lock.c,v $
 *  Revision 1.23.2.2  2002/10/01 08:54:59  egil
 *  Changed sprintf to snprintf
 *
 *  Revision 1.23.2.1  2002/01/29 06:44:48  egil
 *  Changing from xmalloc, xrealloc, xstrcpy to
 *  malloc_perfect, realloc_perfect and strdup_perfect
 *
 *  Revision 1.23  1999/04/09 04:25:30  src
 *  Scanlogs: no error message if no lockfile; mea culpa
 *
 *  Revision 1.22  1999/04/08 10:19:37  src
 *  Check for extendeddbz in configure
 *
 *  Revision 1.21  1999/04/05 19:01:51  src
 *  Lock for scanlogs
 *
 *  Revision 1.20  1999/03/31 03:38:28  src
 *  Moved errno.h to common.h
 *
 *  Revision 1.19  1999/03/19 07:44:36  src
 *  Fixed bug re. leftover lockfiles introduced in 1.4pre
 *
 *  Revision 1.18  1999/03/16 08:25:44  src
 *  Fixed bug re. multiple locks.
 *
 *  Revision 1.17  1999/03/16 08:03:30  src
 *  Renamed lock() to do_lock()
 *
 *  Revision 1.16  1999/03/14 08:47:54  src
 *  Uprated log for innreport.
 *
 *  Revision 1.15  1999/03/07 14:58:18  src
 *  Read newsconfig supported. Storage API supported.
 *
 *  Revision 1.14  1998/11/22 10:48:22  src
 *  Let --no-queue check for stale lockfiles
 *
 *  Revision 1.13  1998/11/22 08:00:08  src
 *  Option --no-queue
 *
 *  Revision 1.12  1998/10/25 04:08:56  src
 *  A little more debugging on lockfiles.
 *
 *  Revision 1.11  1998/09/11 16:37:43  src
 *  Lockfile for logfile of posted articles and for posted article folder.
 *
 *  Revision 1.10  1998/09/11 09:17:42  src
 *  Check path consistency (--no-path) and length (--max-path)
 *  GNU style option --help, --version, --dry-run, changed --noxx to --no-xx
 *  Check for putenv and setenv, added xstrcpy
 *
 *  Revision 1.9  1998/09/09 07:32:12  src
 *  Version 1.1
 *
 *  Revision 1.8  1998/09/02 06:50:30  src
 *  newsx version 1.0
 *
 *  Revision 1.7  1998/09/02 06:34:41  src
 *  Support @-syntax in newsfeeds, and support AUTHINFO GENERIC
 */

#include "common.h"
#include "proto.h"
#include "options.h"
#include "stat.h"
#include "self.h"
#include "newsconfig.h"

#include <fcntl.h>
#include <signal.h>

/*
 *  local
 */
static int 
force_unlock(char *name,int sig,int giveup);
static int
lockfile_exists(char *name);
static int
add_to_lockfiles(char *name);
static void
drop_lock(int id);
static pid_t
pid_of_lock(char* name,int should_exist);

/*
 *  lockfiles
 */
#define MAX_LOCKS 10
static char *lock_file_list[MAX_LOCKS];

/*
 *  perform a lock
 *  return lock ID
 */
int
do_lock(char *dir_name /* NULL if in main lock */,
	char *lock_name,
	int giveup)
{
    long t = 0L;
    long locktime = timeout ? timeout : LOCKTIME;
    char pid_buf[50];
    int sig = SIGUSR1; /* start as gentle as possible */
    FILE *f;
    static char name_temp[PATH_MAX];
    static char name_lock[PATH_MAX];
    int id = -1;
    int id_temp = -1;

    snprintf(pid_buf,sizeof(pid_buf),"%d",my_pid);

    if (!cfg_locks || strcmp(cfg_locks,"none")==0) {
	log_msg(L_DEBUG,"no lock");
	return -1;
    }

    /* make the filenames */
    if (dir_name) {
	/* path specified */
	if (dir_name[0]) 
	    build_filename(name_lock,dir_name,"/",lock_name,NULL);
	else 
	    build_filename(name_lock,lock_name,NULL,NULL,NULL);
	build_filename(name_temp,name_lock,".",pid_buf,NULL);
    } else {
	/* common lock directory */
	build_filename(name_lock,cfg_locks,"/",lock_name,NULL);
	/* BUG: the C News convention is used for the temp_name */
	build_filename(name_temp,cfg_locks,"/L.",pid_buf,NULL);
    }

    if (!lockfile_exists(name_temp)) {
	/* build our own little file (temporary name) */
	if (!(f = fopen(name_temp,"w"))) {
	    log_msg(L_ERRno,"can't create lock \"%s\"", name_temp);
	    exit_cleanup(5);
	}
	id_temp = add_to_lockfiles(name_temp);

	fprintf(f,"%s\n",pid_buf);
	if (fclose(f) == EOF) {
	    log_msg(L_ERRno,"problems creating lock \"%s\"", name_temp);
	    exit_cleanup(5);
	}
    }

    /* perform the locking action by linking */
    while (link(name_temp,name_lock) < 0) {
	/*
	 *  there was an error
	 */

	/* BUG: what if interrupt within link()? */
	/* BUG: check if name_temp has disappeared? */

	log_msg(L_DEBUGMORE,"waiting: link %s %s failed: %s",
				name_temp,name_lock,str_error(errno));

	/* failed, tried using force? */
	if (sig == SIGKILL) {
	    log_msg(L_ERR,"can't lock \"%s\" after forced unlock", name_lock);
	    exit_cleanup(5);
	}

	if (t==0L) {
	    log_msg(L_INFO,"awaiting lockfile \"%s\" to disappear (max %ld s)",
					       name_lock,       locktime);
	}

	/* don't wait forever for access */
	if (sig == SIGUSR1 || (t += LOCKDELTA) > locktime) {
	    if (sig == SIGUSR1)
		log_msg(L_DEBUG,"send USR1 to see if process exists");
	    else 
		log_msg(L_ERR,"file lock \"%s\" timed out", name_lock);

	    /* there is a lockfile, what shall we do? */
	    if (force_unlock(name_lock,sig,giveup)) {
		/* signal worked */
		if (sig==SIGUSR1) {
		    /* the other process is known to exist */
		    /* wait, then use persuasion if lockfile persists */
		    sig = SIGTERM; 
		} else {
		    /* we've presumably managed to remove the lock */
		    /* use real force if lockfile persists */
		    sig = SIGKILL; 
		}
	    } else {
		/* give up */
		exit_cleanup(13);
	    }
	}
	sleep(LOCKDELTA);
    }

    /* locked successfully - update our tables */
    id = add_to_lockfiles(name_lock);

    log_msg(L_DEBUGMORE,"created lock: %s",name_lock);

    return id;
}

/*
 *  lock failed, try to send a signal to find out if the other
 *  process exists (SIGUSR1) or to force an unlock (SIGTERM/SIGKILL)
 *  return true on success
 *  BUG: we really should have checked that it is the 
 *       same process that has been locking all the time
 */
static int 
force_unlock(char *name,int sig,int giveup)
{
#ifdef NO_FORCE
    log_msg(L_INFO,"unlock by force permanently disabled");
    return 0;
#else
    char name_other[PATH_MAX];
    pid_t other_pid;
    struct stat st;

    if (sig != SIGUSR1 && noforce_opt) {
	log_msg(L_INFO,"no attempt at unlocking by force");
	return 0;
    }

    /* read the file to find the pid */

    if (!(other_pid = pid_of_lock(name,1))) {
	/* 
	 *  whatever shall we do? give up?
	 *  we assume the risk, and continue
	 */ 
	return 0;
    } 

    /* contents is plausible */
    if (other_pid == my_pid) {
	/* 
	 *  ourselves? shouldn't happen 
	 */
	log_msg(L_ERR,"file \"%s\" locked by ourselves?", name);
	other_pid = -1;
    } else {
	/*     
	 *  OK, we know who it is
	 *  we try to execute a kill according to sig
	 */
	log_msg(L_INFO,"sending process %d signal %d", other_pid, sig);

	if (kill(other_pid,sig) == -1) {
	    if (errno == ESRCH) {
		/* pretty likely scenario */
		log_msg(L_INFO,"process %d does not exist", other_pid);
	    } else {
		/* we probably don't have permission for this */
		log_msg(L_ERRno,"kill stale process %ld failed", 
							(long)other_pid);
		return 0;
	    }
	} else {
	    if (sig==SIGUSR1) {
		if (giveup) {
		    log_msg(L_ERR,"process %d has locked \"%s\", don't queue up",
					   other_pid,   name);
		    return 0; /* exit */
		} else {
		    log_msg(L_INFO,"process %d exists, wait...", other_pid);
		    return 1;
		}
	    }
	}
	/* give the process time to clean up */
	sleep(LOCKDELTA);
    }

    /* 
     *  try to remove the stale lockfiles
     *  (if the kill already hasn't done that for us)
     */
    if (unlink(name) == -1) {
	if (stat(name,&st) != -1) {
	    /* file still exists */
	    log_msg(L_ERRno,"can't remove lockfile \"%s\"", name);
	}
    }
	
    /* close the other processes private lockfile too */
    if (other_pid != -1) {
	char pid_buf[50];
	snprintf(pid_buf,sizeof(pid_buf),"%d",other_pid);
	build_filename(name_other,name,".",pid_buf,NULL);
	if (unlink(name_other) == -1) {
	    if (stat(name_other,&st) != -1) {
		/* file still exists */
		log_msg(L_ERRno,"can't remove file \"%s\"", name_other);
	    }
	}
    }
    return 1;
#endif
}

/*
 *  add to lockfile list
 *  should be reasonably interruptable (!)
 */
static int
add_to_lockfiles(char *name)
{
    int id;

    for (id=0; id<MAX_LOCKS; ++id) {
	if (!lock_file_list[id]) {
	    lock_file_list[id] = strdup_perfect(name);
	    return id;
	}
    }
    /* should really not happen */
    log_msg(L_ERR,"lock list overflow");
    return -1;
}

/*
 *  check if lockfile exists
 */
static int
lockfile_exists(char *name)
{
    int id;
    char *p;

    for (id=0; id<MAX_LOCKS; ++id) {
	if ((p=lock_file_list[id]) && strcmp(p,name)==0) {
	    return 1;
	}
    }
    return 0;
}

/*
 *  remove one lock
 */
void 
unlock_one(int id)
{
    char *p;

    if (id < 0 || id >= MAX_LOCKS) return;

    if ((p = lock_file_list[id])) {
	if (unlink(p) == -1)
	    log_msg(L_ERRno,"unlink lock file failed: %s", p);
	else
	    log_msg(L_DEBUGMORE,"unlocked: %s", p);
	drop_lock(id);
    }
}     

/*
 *  drop a lock file from the list
 */
static void
drop_lock(int id)
{
    if (id < 0 || id >= MAX_LOCKS) return;

    if (lock_file_list[id]) {
	free(lock_file_list[id]);
	lock_file_list[id] = NULL;
    }
}     

/*
 *  remove any locks
 */
void 
unlock_all(void)
{
    int id;

    for (id=0; id<MAX_LOCKS; ++id) {
	unlock_one(id);
    }
}     

/*
 *  perform exit(), but unlock_all() first
 */
void 
unlock_exit(int n)
{
    progtitle("unlock/exit");
    history_done();
    close_article();
    unlock_all();
    exit(n);
}

/*
 *  see if the process named in an existing lock still exists by
 *  sending it a null signal (adapted from INN/shlock.c)
 */
int
check_lock(char* name)
{
    pid_t pid;

    if (!(pid = pid_of_lock(name,0))) return 0;

    /* send the signal */
    if (kill(pid, 0) < 0 && errno == ESRCH) return 0;

    /* either the kill worked, or we're optimistic about the error code */
    return 1;
}

/*
 *  get pid belonging to lock file
 *  return 0 if problems
 */
static pid_t
pid_of_lock(char* name,int should_exist)
{
    int fd;
    pid_t pid;
    int i;
    char buff[BUFSIZ];

    /* open the file */
    if ((fd = open(name, O_RDONLY)) < 0) {
	if (should_exist) log_msg(L_ERRno,"can't access lockfile \"%s\"",name);
	return 0;
    }

    /* read the PID that is written there. */
    if ((i = read(fd, buff, sizeof buff - 1)) <= 0) {
	log_msg(L_ERRno,"lockfile \"%s\" is empty", name);
	close(fd);
	return 0;
    }
    buff[i] = '\0';
    pid = (pid_t) atol(buff);

    close(fd);
    if (pid <= 0) {
	log_msg(L_ERR,"lockfile \"%s\" has wrong format", name);
	return 0;
    }

    return pid;
}
