/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// SQL.cpp: implementation of the SQL class.
//
//////////////////////////////////////////////////////////////////////

#include "SQL.h"
#include "FileLog.h"
#include "../svintl.h"
#include <mysql/errmsg.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

#ifndef _WIN32
	#define stricmp strcasecmp
	#define strupr(x) \
	{ \
		size_t dtlen; \
		for(dtlen=0; dtlen<strlen(x); dtlen++) (x)[dtlen]=toupper((x)[dtlen]); \
	}
	#define strlwr(x) \
	{ \
		size_t dtlen; \
		for(dtlen=0; dtlen<strlen(x); dtlen++) (x)[dtlen]=tolower((x)[dtlen]); \
	}
#endif

SQL::SQL(const SQL_Connection * DB_Connection, SQL_ACCESS access)
{
	res=NULL;
	m_access = access;

	if(DB_Connection)
	{
		if(access == SQL_ACCESS_READ)
			((SQL_Connection *)DB_Connection)->Locker.LockRead();
		else if(access == SQL_ACCESS_WRITE)
			((SQL_Connection *)DB_Connection)->Locker.LockWrite();

		sqlHND=DB_Connection->sqlHND;
		m_Connection = DB_Connection;
	}
	else
	{
		sqlHND = NULL;
		m_Connection = NULL;
	}
}

SQL::~SQL()
{
	if(res)
		mysql_free_result(res);
	if(m_Connection)
	{
		if(m_access == SQL_ACCESS_READ)
			((SQL_Connection *)m_Connection)->Locker.UnlockRead();
		else if(m_access == SQL_ACCESS_WRITE)
			((SQL_Connection *)m_Connection)->Locker.UnlockWrite();
	}
}

bool SQL::Execute(const mString & req)
{
	mString beg;
	if(!sqlHND)
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_BAD_PARAM);
		ERR_add_error_data(1, _sv("No connection available"));
		return false;
	}

	if(m_access == SQL_ACCESS_READ)
	{
		beg = req.Left(6);
		strlwr((char*)beg.c_str());
		if(beg != "select")
		{
			NewpkiDebug(LOG_LEVEL_ERROR, "SQL Layer", _sv("Wrong access type for SQL Request \"%s\" !"), req.c_str());
			NEWPKIerr(SQL_ERROR_TXT, ERROR_BAD_PARAM);
			ERR_add_error_data(1, _sv("Bad access type for request !"));
			return false;
		}
	}

	if(res)
	{
		mysql_free_result(res);
		res=NULL;
	}

	m_Connection->ExecuteLock.EnterCS();
	if(mysql_query(sqlHND, req.c_str()))
	{
		if(mysql_errno(sqlHND) == CR_SERVER_LOST ||
			mysql_errno(sqlHND) == CR_SERVER_GONE_ERROR)
		{
			NewpkiDebug(LOG_LEVEL_ERROR, "SQL Layer", _sv("Connection down, reconnecting..."));
			if(!m_Connection->Reconnect())
			{
				NewpkiDebug(LOG_LEVEL_ERROR, "SQL Layer", _sv("Reconnection failed !"));
				NEWPKIerr(SQL_ERROR_TXT, ERROR_ABORT);
				m_Connection->ExecuteLock.LeaveCS();
				return false;
			}
			if(mysql_query(sqlHND, req.c_str()))
			{
				NEWPKIerr(SQL_ERROR_TXT, ERROR_DB_SELECT);
				ERR_add_error_data(1, mysql_error(sqlHND));
				m_Connection->ExecuteLock.LeaveCS();
				return false;
			}
		}
		else
		{	
			NEWPKIerr(SQL_ERROR_TXT, ERROR_DB_SELECT);
			ERR_add_error_data(1, mysql_error(sqlHND));
			m_Connection->ExecuteLock.LeaveCS();
			return false;
		}
	}

	if(mysql_field_count(sqlHND))
	{
		res = mysql_store_result(sqlHND);
		if(!res)
		{
			NEWPKIerr(SQL_ERROR_TXT, ERROR_DB_SELECT);
			ERR_add_error_data(1, mysql_error(sqlHND));
			m_Connection->ExecuteLock.LeaveCS();
			return false;
		}
	}
	m_Connection->ExecuteLock.LeaveCS();
	return true;
}

bool SQL::NumRows(long *num_rows)
{
	if(!sqlHND)
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_BAD_PARAM);
		ERR_add_error_data(1, _sv("No connection available"));
		return false;
	}

	if(!res)
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_BAD_PARAM);
		*num_rows=0;
		return false;
	}

	(*num_rows)=(long)mysql_num_rows(res);

	if(!(*num_rows) && mysql_errno(sqlHND))
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_DB_NUMROWS);
		ERR_add_error_data(1, mysql_error(sqlHND));
		return false;
	}

	return true;
}

bool SQL::Value(int RowNum, const mString & Name, mString & result)
{
	int num_fields;
	int i;
	MYSQL_FIELD *fields;
	long num_rows;
	MYSQL_ROW row;

	if(!NumRows(&num_rows))
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(RowNum<0 || RowNum>=num_rows)
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	mysql_data_seek(res, RowNum);

	row=mysql_fetch_row(res);
	if(!row)
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_DB_FETCHROWS);
		ERR_add_error_data(1, mysql_error(sqlHND));
		return false;
	}

	num_fields = mysql_num_fields(res);
	if(!num_fields && mysql_errno(sqlHND))
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_DB_NUMFIELDS);
		ERR_add_error_data(1, mysql_error(sqlHND));
		return false;
	}


	fields = mysql_fetch_fields(res);
	if(!fields)
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_DB_FETCHFIELD);
		ERR_add_error_data(1, mysql_error(sqlHND));
		return false;
	}


	for(i=0;i<num_fields;i++)
	{
		if(stricmp(fields[i].name, Name.c_str())==0)
		{
			result = row[i];
			return true;
		}
	}

	NEWPKIerr(SQL_ERROR_TXT, ERROR_DB_FIELDNAME);
	ERR_add_error_data(1, Name.c_str());
	return false;
}

mString SQL::FormatString(const mString & String)
{
	mString Buffer;

	if(!sqlHND || !String.size())
		return "";

	Buffer.resize((String.size()*2) + 1);
	mysql_real_escape_string(sqlHND, &Buffer[0], String.c_str(), String.size());

	return Buffer;
}

long SQL::GetLastID()
{
	if(!sqlHND)
		return 0;
	return (long)mysql_insert_id(sqlHND);
}


bool SQL::CreateDatabase(const mString & DataBaseName)
{
	if(!m_Connection)
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_BAD_PARAM);
		ERR_add_error_data(1, _sv("No connection available"));
		return false;
	}
	mString req = "CREATE DATABASE ";
	req += DataBaseName;
	if(!Execute(req))
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!m_Connection->SelectDatabase(DataBaseName))
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_ABORT);
		return false;;
	}
	return true;
}

bool SQL::DropDatabase(const mString & DataBaseName)
{
	if(!m_Connection)
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_BAD_PARAM);
		ERR_add_error_data(1, _sv("No connection available"));
		return false;
	}
	mString req = "DROP DATABASE ";
	req += DataBaseName;

	if(!Execute(req))
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool SQL::OptimizeTable(const mString & TableName)
{
	if(!m_Connection)
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_BAD_PARAM);
		ERR_add_error_data(1, _sv("No connection available"));
		return false;
	}
	mString req = "OPTIMIZE TABLE ";
	req += TableName;
	req += ";";

	if(!Execute(req))
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool SQL::Value(int RowNum, int ColNum, mString & result)
{
	int num_fields;
	long num_rows;
	MYSQL_ROW row;

	if(!NumRows(&num_rows))
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(RowNum<0 || RowNum>=num_rows)
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_BAD_PARAM);
		ERR_add_error_data(1, _sv("No connection available"));
		return false;
	}

	mysql_data_seek(res, RowNum);

	row=mysql_fetch_row(res);
	if(!row)
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_DB_FETCHROWS);
		ERR_add_error_data(1, mysql_error(sqlHND));
		return false;
	}

	num_fields = mysql_num_fields(res);
	if(!num_fields && mysql_errno(sqlHND))
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_DB_NUMFIELDS);
		ERR_add_error_data(1, mysql_error(sqlHND));
		return false;
	}

	if(ColNum >= num_fields)
	{
		NEWPKIerr(SQL_ERROR_TXT, ERROR_DB_NUMFIELDS);
		ERR_add_error_data(1, mysql_error(sqlHND));
		return false;
	}
	
	result = row[ColNum];
	return true;
}
