/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "client.h"
#include "clintl.h"
#include <PkiClient.h>
#include "DlgAcceptPkiCert.h"
#include "DlgLogin.h"
#include "DlgMessage.h"
#include "DlgGUI.h"
#include "DlgServerAdmin.h"
#include "DlgPkiAdmin.h"
#include "DlgCaAdmin.h"
#include "DlgRaAdmin.h"
#include "DlgEeAdmin.h"
#include "DlgSplash.h"
#include "DlgGetPassword.h"


IMPLEMENT_APP(NewPKI_Client)

#ifndef _WIN32
	#ifndef LOCALEDIR
		#define LOCALEDIR "/usr/share/locale"
	#endif
#endif

NewPKI_Client::NewPKI_Client()
{
	// Initialisation d'openssl
	#ifdef _WIN32
		WSADATA StartupData;
		WSAStartup(0x101, &StartupData);

		#if !defined(NO_GETTEXT) && !defined(_DEBUG)
			char path[MAX_PATH];
			char * lastOcc;
			*path=0;


			GetModuleFileName(GetModuleHandle(NULL), path, sizeof(path));
			
			lastOcc = strrchr(path, '\\');
			if(lastOcc)
			{
				*lastOcc=0;
			}
			bindtextdomain("newpki-client", path);
			bindtextdomain("newpki-lib", path);
		#endif
	#else
		#if !defined(NO_GETTEXT) && !defined(_DEBUG)
			bindtextdomain("newpki-client", LOCALEDIR);
			bindtextdomain("newpki-lib", LOCALEDIR);	
			setlocale(LC_ALL,"");
		#endif
	#endif


	INIT_OPENSSL();

	#ifndef _WIN32
		//On cree le rep de newpki dans le home du user
		char FileName[250];
		char * path = getenv("HOME");
		if(path)
		{
			snprintf(FileName, sizeof(FileName), "%s%s", path, ENV_PKI_FOLDER);
			mkdir(FileName, 0700);
		}
	#endif

}

NewPKI_Client::~NewPKI_Client()
{
}

int NewPKI_Client::OnExit()
{
	NewpkiThread::SignalStop();
	if(GetTopWindow()) delete GetTopWindow();
	CLEAN_OPENSSL();
	#ifdef _WIN32
		if(WSAIsBlocking()) WSACancelBlockingCall();		
		WSACleanup();
	#endif
	return 0;
}

// `Main program' equivalent, creating windows and returning main app frame
bool NewPKI_Client::OnInit()
{
	LOGIN_INFO LogInfos;
	PkiClient * pki = NULL;
	DlgLogin * login = NULL;
	bool ret;
	DlgMessage DlgMsg(NULL);
	DlgGUI * Dlg;
	int UserType;
	AdminReqLogin loginReq;

	DlgSplash splash(NULL);

	NewpkiThread::SignalStart();

	do
	{	
		ret = true;
		if(pki) delete pki;

		login = new DlgLogin(NULL);
		if(!login)
		{
			HandleError(NEWPKIerrGetStr(ERROR_MALLOC), NULL);
			return false;
		}

		if(!login->GetLoginInfo(&LogInfos))
		{
			delete login;
			return false;
		}

		DlgMsg.wShow(_("Initializing SSL Layer..."));
		pki=new PkiClient(NetworkWaitFunction);
		if(!pki)
		{
			DlgMsg.wHide();
			HandleError(NEWPKIerrGetStr(ERROR_MALLOC), NULL);
			delete login;
			return false;
		}
		if(!pki->Connect(LogInfos.Server, LogInfos.Port, LogInfos.p12_cert, LogInfos.pCert))
		{
			DlgMsg.wHide();
			HandleError(pki->GetError(), NULL);
			ret = false;
		}
		if(ret)
		{
			loginReq.set_entity(LogInfos.CaName);
			loginReq.set_username(LogInfos.Username);
			loginReq.set_password(LogInfos.Password);

			if(!pki->UserLogin(loginReq, UserType))
			{
				DlgMsg.wHide();
				HandleError(pki->GetError(), NULL);
				ret = false;
			}
		}
		DlgMsg.wHide();
		delete login;
	}
	while(!ret);



	//We retreive the peer certificate and check if we trust it
	PKI_CERT PeerCert;
	pki->GetPeerCertificate(PeerCert);
	if(PeerCert)
	{
		if(!IsPkiCertTrusted(LogInfos.Server, &PeerCert))
		{
			delete pki;
			return false;

		}
	}

	Dlg = NULL;
	switch(UserType)
	{
		case USER_TYPE_SERVER:
			Dlg = new DlgServerAdmin(NULL, LogInfos.CaName, LogInfos.Username, LogInfos.Password, pki);
			break;

		case USER_TYPE_PKI:
			Dlg = new DlgPkiAdmin(NULL, LogInfos.CaName, LogInfos.Username, LogInfos.Password, pki);
			break;

		case USER_TYPE_CA:
			Dlg = new DlgCaAdmin(NULL, LogInfos.CaName, LogInfos.Username, LogInfos.Password, pki);
			break;

		case USER_TYPE_RA:
			Dlg = new DlgRaAdmin(NULL, LogInfos.CaName, LogInfos.Username, LogInfos.Password, pki);
			break;

		case USER_TYPE_EE:
			Dlg = new DlgEeAdmin(NULL, LogInfos.CaName, LogInfos.Username, LogInfos.Password, pki);
			break;

		case USER_TYPE_REPOSITORY:
			break;
		
		default:
			Dlg = NULL;
			break;
	}

	if(!Dlg)
	{
		delete pki;
		return false;
	}

	Dlg->Show(TRUE);
	SetTopWindow(Dlg);

	return true;
}

bool NewPKI_Client::IsPkiCertTrusted(char * Server, PKI_CERT * PeerCert)
{
	PKI_CERT found_cert;

	//We seek the serer's certificate
	if(!FindServerCert(Server, &found_cert))
	{
		//Non on va demander au user s'il veut le truster
		DlgAcceptPkiCert Dlg(NULL, PeerCert);
		if(Dlg.IsAccepted())
		{
			//On l'enregistre
			SaveServerCert(Server, PeerCert);
			return true;
		}
		else
		{
			//Il ne veut pas le truster
			return false;
		}
	}
	else
	{
		//Oui on va verifier qu'il est bien le meme
		if(*PeerCert == found_cert)
		{
			return true;
		}
		else
		{
			HandleError(_("The previously saved server certificate doesn't match the one sent by the PKI server"), NULL);

			//Non on va demander au user s'il veut le truster
			DlgAcceptPkiCert Dlg2(NULL, PeerCert);
			if(Dlg2.IsAccepted())
			{
				//On l'enregistre
				SaveServerCert(Server, PeerCert);
				return true;
			}
			else
			{
				//Il ne veut pas le truster
				return false;
			}
		}				
	}
		
	return true;	

}

bool NewPKI_Client::FindServerCert(char *Server, PKI_CERT * server_cert)
{
	char FileName[250];
	unsigned char * OutDatas;

#ifdef _WIN32
	DWORD OutDatasLen;
	long lRet;
	HKEY KeyRet;		
	DWORD KeyType;


	snprintf(FileName, sizeof(FileName), "server_%s", Server);

	

	lRet = RegOpenKey(HKEY_CURRENT_USER, NEWPKI_PATH, &KeyRet);
	if(lRet!=ERROR_SUCCESS)
	{
		return false;
	}

	OutDatasLen = 0;
	lRet = RegQueryValueEx(KeyRet, FileName, 0, &KeyType, NULL, &OutDatasLen);
	if(lRet!=ERROR_SUCCESS)
	{
		RegCloseKey(KeyRet);
		return false;
	}

	OutDatas = (unsigned char *)malloc(OutDatasLen + 1);
	if(!OutDatas)
	{
		RegCloseKey(KeyRet);
		return false;
	}

	lRet = RegQueryValueEx(KeyRet, FileName, 0, &KeyType, OutDatas, &OutDatasLen);
	if(lRet!=ERROR_SUCCESS)
	{
		RegCloseKey(KeyRet);
		free(OutDatas);
		return false;
	}			
	RegCloseKey(KeyRet);
	OutDatas[OutDatasLen] = 0;
#else
	FILE * fd;
	char * path = getenv("HOME");
	struct stat m_stat;
	if(!path) return false;

	snprintf(FileName, sizeof(FileName), "%s%sserver_%s", path, ENV_PKI_FOLDER, Server);

	if(stat(FileName, &m_stat) == -1) return false;

	fd = fopen(FileName, "rb");
	if(!fd)
	{
		return false;
	}

	OutDatas = (unsigned char *)malloc(m_stat.st_size + 1);
	if(!OutDatas)
	{
		fclose(fd);
		return false;
	}

	if(fread(OutDatas, 1, m_stat.st_size, fd) != m_stat.st_size)
	{
		free(OutDatas);
		fclose(fd);
		return false;
	}			
	fclose(fd);
	OutDatas[m_stat.st_size] = 0;
#endif

	if(!server_cert->SetCert((char*)OutDatas))
	{
		free(OutDatas);
		return false;
	}

	free(OutDatas);

	return true;

}

void NewPKI_Client::SaveServerCert(char *Server, PKI_CERT * server_cert)
{
	char FileName[250];

#ifdef _WIN32
	HKEY hk; 
	snprintf(FileName, sizeof(FileName), "server_%s", Server);


	if (RegCreateKey(HKEY_CURRENT_USER, NEWPKI_PATH, &hk) != ERROR_SUCCESS)
	{
		return;
	}

	
	if (RegSetValueEx(hk, FileName, 0, REG_SZ, (PUCHAR)server_cert->GetCertPEM().c_str(), strlen(server_cert->GetCertPEM().c_str())) != ERROR_SUCCESS)
	{
		RegCloseKey(hk);
		return;
	}
	RegCloseKey(hk);
	return;
#else
	FILE * fd;
	char * path = getenv("HOME");
	if(!path) return;

	snprintf(FileName, sizeof(FileName), "%s%sserver_%s", path, ENV_PKI_FOLDER, Server);

	fd = fopen(FileName, "wb");
	if(!fd)
	{
		return;
	}

	fprintf(fd, "%s", server_cert->GetCertPEM().c_str());

	fclose(fd);

#endif
}

void NewPKI_Client::NetworkWaitFunction()
{
	wxYieldIfNeeded();
}

int newpki_get_password_cb(char *buf, int size, int rwflag, void *userdata)
{
	wxWindow * parent = (wxWindow*)userdata;
	if(!parent) return 0;

	DlgGetPassword Dlg("Enter password", parent);
	if(!Dlg.IsOK()) return 0;

	int len;

	mString tmp = Dlg.GetPassword();
	if(!tmp.size())
		return 0;

	len = tmp.size();

	if (len <= 0) return 0;

	if (len > size) len = size;
	memcpy(buf, tmp.c_str(), len);
	return len;
}
