/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PkiEnroll.cpp: implementation of the PkiEnroll class.
//
//////////////////////////////////////////////////////////////////////

#include "PkiEnroll.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

#ifdef _WIN32

PkiEnroll::PkiEnroll()
{
	PemCsr = NULL;
	LastError = 0;
}

PkiEnroll::~PkiEnroll()
{
	if(PemCsr) free(PemCsr);
}

char * PkiEnroll::CreatePkcs10(HashTable_Dn *Dn, const char * ProviderName, int RsaLen)
{
	LastError = 0;


	CERT_NAME_INFO CertName;
	int i;
	const char * Value;
	const char * Name;


	m_ProviderName = ProviderName;


	CertName.rgRDN =(PCERT_RDN)malloc(sizeof(CERT_RDN)*Dn->EntriesCount());
	if(!CertName.rgRDN)
	{
		LastError = GetLastError();
		return NULL;
	}

	for(i=0, CertName.cRDN=0; i<Dn->EntriesCount(); i++)
	{
		Name = Dn->GetName(i);
		if(!Name) continue;

		Value = Dn->Get(i);
		if(!Value) continue;

		CertName.rgRDN[CertName.cRDN].cRDNAttr=1;
		CertName.rgRDN[CertName.cRDN].rgRDNAttr=(PCERT_RDN_ATTR)malloc(sizeof(CERT_RDN_ATTR));
		if(!CertName.rgRDN[CertName.cRDN].rgRDNAttr) continue;

		CertName.rgRDN[CertName.cRDN].rgRDNAttr[0].dwValueType=CERT_RDN_TELETEX_STRING;
		CertName.rgRDN[CertName.cRDN].rgRDNAttr[0].pszObjId=(char*)Name;
		CertName.rgRDN[CertName.cRDN].rgRDNAttr[0].Value.cbData=strlen(Value);
		CertName.rgRDN[CertName.cRDN].rgRDNAttr[0].Value.pbData=(PBYTE)Value;
		CertName.cRDN++;
	}
	if(!CertName.cRDN)
	{
		LastError = GetLastError();
		return NULL;
	}

	GenerateContainerName();

	if(PemCsr)
	{
		free(PemCsr);
		PemCsr = NULL;
	}

	GenerateCSR(&CertName, RsaLen);

	for(i=0; i<(long)CertName.cRDN; i++)
	{
		free(CertName.rgRDN[i].rgRDNAttr);
	}
	free(CertName.rgRDN);

	return PemCsr;
}

DWORD PkiEnroll::GetEnrollLastError()
{
	return LastError;
}


bool PkiEnroll::GenerateCSR(PCERT_NAME_INFO pCertname, DWORD KeyLen)
{

	LastError = 0;

	
	//-------------------------------------------------------------------
	// Declare and initialize all other variables and structures.

	CERT_REQUEST_INFO  CertReqInfo;
	HCRYPTKEY  pPrivKey;
	CERT_NAME_BLOB  SubjNameBlob;
	DWORD  cbNameEncoded;
	BYTE*  pbNameEncoded;
	HCRYPTPROV  hCryptProv;
	DWORD  cbPublicKeyInfo;
	CERT_PUBLIC_KEY_INFO*  pbPublicKeyInfo;
	DWORD  cbEncodedCertReqSize;
	CRYPT_OBJID_BLOB  Parameters;
	CRYPT_ALGORITHM_IDENTIFIER  SigAlg;
	BYTE*  pbSignedEncodedCertReq;

	ALG_ID TypeKey;

	// Encoding of the DN using OpenSSL

	int nid;
	X509_NAME *subj;
	unsigned int i;
	unsigned char *p;

	TypeKey=AT_SIGNATURE;

	
	subj = X509_NAME_new();
	if(!subj)
	{
		LastError = ERROR_OUTOFMEMORY;
		return false;
	}
	
	for(i=0; i < pCertname->cRDN; i++)
	{
		// We ignore the unknow DN fields
		if ((nid=OBJ_txt2nid(pCertname->rgRDN[i].rgRDNAttr->pszObjId)) == NID_undef) continue;
		if(!X509_NAME_add_entry_by_NID(subj,nid, MBSTRING_ASC,(unsigned char *)pCertname->rgRDN[i].rgRDNAttr->Value.pbData, -1,-1,0))
		{
			LastError = OR_INVALID_OID;
			X509_NAME_free(subj);
			return false;
		}
	}

	if (X509_NAME_entry_count(subj) == 0)
	{
		LastError = OR_INVALID_OID;
		X509_NAME_free(subj);
		return false;
	}

	cbNameEncoded = i2d_X509_NAME(subj,NULL);
	if(cbNameEncoded <= 0)
	{
		LastError = CRYPT_E_ASN1_INTERNAL;
		X509_NAME_free(subj);
		return false;
	}

	pbNameEncoded=(unsigned char *)malloc(cbNameEncoded+20);
	if (!pbNameEncoded)
	{
		LastError = GetLastError();
		X509_NAME_free(subj);
		return false;
	}

	p=pbNameEncoded;

	if((cbNameEncoded=i2d_X509_NAME(subj,&p)) < 0)
	{
		LastError = CRYPT_E_ASN1_INTERNAL;
		free(pbNameEncoded);
		X509_NAME_free(subj);
		return false;
	}
	X509_NAME_free(subj);


	//--------------------------------------------------------------------
	// Set the subject member of CertReqInfo to point to 
	// a CERT_NAME_INFO structure that 
	// has been initialized with the data from cbNameEncoded
	// and pbNameEncoded.

	SubjNameBlob.cbData = cbNameEncoded;
	SubjNameBlob.pbData = pbNameEncoded;
	CertReqInfo.Subject = SubjNameBlob;

	//--------------------------------------------------------------------
	// Generate custom information. This step is not
	// implemented in this code.

	CertReqInfo.cAttribute = 0;
	CertReqInfo.rgAttribute = NULL;
	CertReqInfo.dwVersion = CERT_REQUEST_V1;

			

	//--------------------------------------------------------------------
	//    Call CryptExportPublicKeyInfo to return an initialized
	//    CERT_PUBLIC_KEY_INFO structure.
	//    First, get a cryptographic provider.

	if(!CryptAcquireContext(
		&hCryptProv,							// Address for handle to be returned.
		m_ContainerName.c_str(),			                // Use the current user's logon name.
		m_ProviderName.c_str(),   // Use the default provider.
		PROV_RSA_FULL,							// Need to both encrypt and sign.
		CRYPT_NEWKEYSET ))
	{
		LastError = GetLastError();
		free(pbNameEncoded);
		return false;
	}

	KeyLen = KeyLen * 0x10000;

	if(!CryptGenKey(hCryptProv, TypeKey, KeyLen, &pPrivKey))
	{
		LastError = GetLastError();
		free(pbNameEncoded);
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return false;
	}
	CryptDestroyKey(pPrivKey);







	//--------------------------------------------------------------------
	// Call CryptExportPublicKeyInfo to get the size of the returned
	// information.

	if(!CryptExportPublicKeyInfo(
			  hCryptProv,            // Provider handle
			  TypeKey,          // Key spec
			  PKCS_7_ASN_ENCODING | X509_ASN_ENCODING,      // Encoding type
			  NULL,                  // pbPublicKeyInfo
			  &cbPublicKeyInfo))     // Size of PublicKeyInfo
	{
		LastError = GetLastError();
		free(pbNameEncoded);
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return false;
	}
	//--------------------------------------------------------------------
	// Allocate the necessary memory.

	if(!(pbPublicKeyInfo = (CERT_PUBLIC_KEY_INFO*)malloc(cbPublicKeyInfo)))
	{
		LastError = GetLastError();
		free(pbNameEncoded);
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return false;
	}
	//--------------------------------------------------------------------
	// Call CryptExportPublicKeyInfo to get pbPublicKeyInfo.

	if(!CryptExportPublicKeyInfo(
			  hCryptProv,            // Provider handle
			  TypeKey,          // Key spec
			  PKCS_7_ASN_ENCODING | X509_ASN_ENCODING,      // Encoding type
			  pbPublicKeyInfo,       // pbPublicKeyInfo
			  &cbPublicKeyInfo))     // Size of PublicKeyInfo
	{
		LastError = GetLastError();
		free(pbPublicKeyInfo);
		free(pbNameEncoded);
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return false;
	}






	//--------------------------------------------------------------------
	// Set the SubjectPublicKeyInfo member of the 
	// CERT_REQUEST_INFO structure to point to the CERT_PUBLIC_KEY_INFO 
	// structure created.

	CertReqInfo.SubjectPublicKeyInfo = *pbPublicKeyInfo;

	memset(&Parameters, 0, sizeof(Parameters));
	SigAlg.pszObjId = szOID_OIWSEC_sha1RSASign;
	SigAlg.Parameters = Parameters;

	//--------------------------------------------------------------------
	// Call CryptSignAndEncodeCertificate to get the size of the
	// returned BLOB.

	if(!CryptSignAndEncodeCertificate(
			  hCryptProv,                      // Crypto provider
			  TypeKey,                  // Key spec
			  PKCS_7_ASN_ENCODING | X509_ASN_ENCODING,                // Encoding type
			  X509_CERT_REQUEST_TO_BE_SIGNED,  // Structure type
			  &CertReqInfo,                    // Structure information
			  &SigAlg,                         // Signature algorithm
			  NULL,                            // Not used
			  NULL,                            // pbSignedEncodedCertReq
			  &cbEncodedCertReqSize))          // Size of certificate 
											   // required
	{
		LastError = GetLastError();
		free(pbPublicKeyInfo);
		free(pbNameEncoded);
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return false;
	}

	//--------------------------------------------------------------------
	// Allocate memory for the encoded certificate request.
	if(!(pbSignedEncodedCertReq = (BYTE*)malloc(cbEncodedCertReqSize)))
	{
		LastError = GetLastError();
		free(pbPublicKeyInfo);
		free(pbNameEncoded);
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return false;
	}
	//--------------------------------------------------------------------
	// Call CryptSignAndEncodeCertificate to get the 
	// returned BLOB.

	if(!CryptSignAndEncodeCertificate(
			  hCryptProv,                     // Crypto provider
			  TypeKey,                 // Key spec
			  PKCS_7_ASN_ENCODING | X509_ASN_ENCODING,               // Encoding type
			  X509_CERT_REQUEST_TO_BE_SIGNED, // Struct type
			  &CertReqInfo,                   // Struct info        
			  &SigAlg,                        // Signature algorithm
			  NULL,                           // Not used
			  pbSignedEncodedCertReq,         // Pointer
			  &cbEncodedCertReqSize))         // Length of the message
	{
		LastError = GetLastError();
		free(pbSignedEncodedCertReq);
		free(pbPublicKeyInfo);
		free(pbNameEncoded);
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return false;
	}

	free(pbPublicKeyInfo);
	free(pbNameEncoded);
	CryptReleaseContext(hCryptProv, 0);



	PEM_DER PemConv;
	int PemCertLen; 

	if(!PemConv.Der2Pem((char *)pbSignedEncodedCertReq, cbEncodedCertReqSize, &PemCsr, &PemCertLen))
	{
		LastError = ERROR_OUTOFMEMORY;
		free(pbSignedEncodedCertReq);
		DestroyContainer();
		return false;
	}
	free(pbSignedEncodedCertReq);

	return true;
}

bool PkiEnroll::ImportCert(const char *Cert)
{
	PCCERT_CONTEXT pCert;
	PEM_DER PemDer;
	char * DerCert;
	int DerCertLen;
	HCRYPTPROV	hProv;
	LastError = 0;
	PCERT_PUBLIC_KEY_INFO pPubKey; 
	DWORD pPubKeyLen;

	// Let's decode the PEM
	if(!PemDer.Pem2Der(Cert, strlen(Cert), &DerCert, &DerCertLen))
	{
		LastError = ERROR_OUTOFMEMORY;
		return false;
	}

	// Get a windows representation of the certificate
	pCert=CertCreateCertificateContext(X509_ASN_ENCODING | PKCS_7_ASN_ENCODING, (const unsigned char *)DerCert, DerCertLen);
	if(!pCert)
	{
		LastError = GetLastError();
		free(DerCert);
		return false;
	}
	free(DerCert);



	if(!m_ContainerName.size() ||
		!m_ProviderName.size())
	{
		if(!ImportSearchCert(pCert))
		{
			CertFreeCertificateContext(pCert);
			return false;
		}
		CertFreeCertificateContext(pCert);
		return true;
	}


	// We get a handle for the container
	if(!CryptAcquireContext(&hProv, m_ContainerName.c_str(), m_ProviderName.c_str(), PROV_RSA_FULL, 0))
	{
		if(!ImportSearchCert(pCert))
		{
			CertFreeCertificateContext(pCert);
			return false;
		}
		CertFreeCertificateContext(pCert);
		return true;
	}


	pPubKeyLen = 0;		
	// We key the public key info
	if(!GetContainerPubKeyInfo(hProv, NULL, &pPubKeyLen))
	{
		CryptReleaseContext(hProv, 0);
		if(!ImportSearchCert(pCert))
		{
			CertFreeCertificateContext(pCert);
			return false;
		}
		CertFreeCertificateContext(pCert);
		return true;
	}

	pPubKey = (PCERT_PUBLIC_KEY_INFO)malloc(pPubKeyLen);
	if(!pPubKey)
	{
		CryptReleaseContext(hProv, 0);
		LastError = ERROR_OUTOFMEMORY;
		return false;
	}

	if(!GetContainerPubKeyInfo(hProv, pPubKey, &pPubKeyLen))
	{
		CryptReleaseContext(hProv, 0);
		free(pPubKey);
		if(!ImportSearchCert(pCert))
		{
			CertFreeCertificateContext(pCert);
			return false;
		}
		CertFreeCertificateContext(pCert);
		return true;
	}

	// We compare both keys
	if(!CertComparePublicKeyInfo(PKCS_7_ASN_ENCODING | X509_ASN_ENCODING, pPubKey, &(pCert->pCertInfo->SubjectPublicKeyInfo) ))
	{
		CryptReleaseContext(hProv, 0);
		free(pPubKey);
		if(!ImportSearchCert(pCert))
		{
			CertFreeCertificateContext(pCert);
			return false;
		}
		CertFreeCertificateContext(pCert);
		return true;
	}
	free(pPubKey);

	if(!ImportCertToCSP(pCert, hProv))
	{
		if(!ImportCertToMY(pCert, hProv))
		{
			CryptReleaseContext(hProv, 0);
			if(!ImportSearchCert(pCert))
			{
				CertFreeCertificateContext(pCert);
				return false;
			}
			CertFreeCertificateContext(pCert);
			return false;
		}
	}

	// Importation success
	CertFreeCertificateContext(pCert);
	CryptReleaseContext(hProv, 0);
	return true;
}

bool PkiEnroll::ImportSearchCert(PCCERT_CONTEXT pCert)
{

	LPTSTR      pszName;
	DWORD       dwType;
	DWORD       cbName;
	DWORD       dwIndex=0;
	HCRYPTPROV	hProv;

	dwIndex = 0;

	while(CryptEnumProviders(dwIndex, NULL, 0, &dwType, NULL, &cbName))
	{

		if (!(pszName = (LPTSTR)malloc(cbName)))
		{
			LastError = GetLastError();
			return false;
		}

		if (CryptEnumProviders(
							   dwIndex++,
							   NULL,
							   NULL,
							   &dwType,   
							   pszName,
							   &cbName))     
		{

			hProv = FindKey(pCert, pszName);
			
			// We found the associated key
			if(hProv)
			{
				if(!ImportCertToCSP(pCert, hProv))
				{
					if(!ImportCertToMY(pCert, hProv))
					{
						CryptReleaseContext(hProv, 0);
						free(pszName);
						return false;
					}
				}

				// Importation succeded
				CryptReleaseContext(hProv, 0);
				free(pszName);
				return true;
			}
		}

		free(pszName);
	} // while (EnumProviders)

	LastError = CRYPT_E_NOT_FOUND;
	return false;
}

bool PkiEnroll::GetContainerPubKeyInfo(HCRYPTPROV hProv, PCERT_PUBLIC_KEY_INFO PubKeyInfo, DWORD * PubKeyInfoLen)
{
	HCRYPTKEY phUserKey;
	DWORD KeyType = 0;

	if(CryptGetUserKey(hProv, AT_SIGNATURE, &phUserKey))
	{
		CryptDestroyKey(phUserKey);
		KeyType = KeyType | AT_SIGNATURE;
	}
	else
	{
		if(CryptGetUserKey(hProv, AT_KEYEXCHANGE, &phUserKey))
		{
			CryptDestroyKey(phUserKey);
			KeyType = KeyType | AT_KEYEXCHANGE;
		}
	}

	if(!CryptExportPublicKeyInfo(hProv, KeyType, X509_ASN_ENCODING | PKCS_7_ASN_ENCODING, PubKeyInfo, PubKeyInfoLen))
	{
		return false;
	}

	return true;
}

HCRYPTPROV PkiEnroll::FindKey(PCCERT_CONTEXT pCert, char * ProviderName)
{

	HCRYPTPROV hCrypt_token;

	unsigned char Datas[300];
	DWORD DatasLen;

	PCERT_PUBLIC_KEY_INFO pPubKey; 
	DWORD pPubKeyLen;

	HCRYPTPROV hProv;
	

	if(!CryptAcquireContext(&hCrypt_token,NULL,ProviderName,PROV_RSA_FULL, CRYPT_VERIFYCONTEXT))
	{
		return NULL;
	}


	// We enumerate all the key containers sored in the provider
	DatasLen=sizeof(Datas);
	if(CryptGetProvParam(hCrypt_token, PP_ENUMCONTAINERS, Datas, &DatasLen, CRYPT_FIRST))
	{
		do
		{
			// We get a handle for the container
			if(CryptAcquireContext(&hProv, (char *)Datas, ProviderName, PROV_RSA_FULL, 0))
			{
				pPubKeyLen = 0;
				
				// We key the public key info
				if(GetContainerPubKeyInfo(hProv, NULL, &pPubKeyLen))
				{
					pPubKey = (PCERT_PUBLIC_KEY_INFO)malloc(pPubKeyLen);
					if(pPubKey)
					{
						if(GetContainerPubKeyInfo(hProv, pPubKey, &pPubKeyLen))
						{
							// We compare both keys
							if(CertComparePublicKeyInfo(PKCS_7_ASN_ENCODING | X509_ASN_ENCODING, pPubKey, &(pCert->pCertInfo->SubjectPublicKeyInfo) ))
							{
								free(pPubKey);
								CryptReleaseContext(hCrypt_token, 0);
								return hProv;
							}
						}
						free(pPubKey);
					}
				}
				// We close the current container
				CryptReleaseContext(hProv, 0);
			}


			DatasLen=sizeof(Datas);
		}
		while(CryptGetProvParam(hCrypt_token, PP_ENUMCONTAINERS, Datas, &DatasLen, 0));
	}

	CryptReleaseContext(hCrypt_token, 0);

	return NULL;
}

BSTR PkiEnroll::StringToUnicode(char * buffer)
{
	BSTR ret;
	int buff_len;

	buff_len=(strlen(buffer)+1) * sizeof(unsigned short);
	ret = (BSTR)malloc(buff_len);
	if(!ret)
	{
		LastError = GetLastError();
		return NULL;
	}

	MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED, buffer, -1, ret, buff_len); 

	return ret;
}

bool PkiEnroll::ImportCertToCSP(PCCERT_CONTEXT CertificateContext, HCRYPTPROV hProv)
{
	HCRYPTKEY phUserKey;

	if(!CryptGetUserKey(hProv, AT_SIGNATURE, &phUserKey))
	{
		if(!CryptGetUserKey(hProv, AT_KEYEXCHANGE, &phUserKey))
		{
			return false;
		}
	}

	if(!CryptSetKeyParam(phUserKey, KP_CERTIFICATE, NULL, 0))
	{
		CryptDestroyKey(phUserKey);
		return false;
	}

	if(!CryptSetKeyParam(phUserKey, KP_CERTIFICATE, CertificateContext->pbCertEncoded, CertificateContext->cbCertEncoded))
	{
		CryptDestroyKey(phUserKey);
		return false;
	}

	CryptDestroyKey(phUserKey);

	return true;
}

bool PkiEnroll::ImportCertToMY(PCCERT_CONTEXT CertificateContext, HCRYPTPROV hProv)
{
	DWORD cbData = 0;

	PCCERT_CONTEXT phOutCert;

	CRYPT_KEY_PROV_INFO pCryptKeyProvInfo;
	HCERTSTORE hMyStore;
	HCRYPTKEY phUserKey;


	char f_name[255];

	
	hMyStore = CertOpenStore(CERT_STORE_PROV_SYSTEM, X509_ASN_ENCODING | PKCS_7_ASN_ENCODING, NULL, CERT_SYSTEM_STORE_CURRENT_USER | CERT_STORE_OPEN_EXISTING_FLAG, L"MY");
	if(!hMyStore)
	{
		LastError = GetLastError();
		return false;
	}


	/**********************************************************
					Filling up the structure
	 **********************************************************/


	//Le nom du provider
	cbData=sizeof(f_name);
	CryptGetProvParam(hProv, PP_NAME, (PBYTE)f_name, &cbData, 0);
	pCryptKeyProvInfo.pwszProvName=StringToUnicode(f_name);
	if(!pCryptKeyProvInfo.pwszProvName)
	{
		CertCloseStore(hMyStore, CERT_CLOSE_STORE_FORCE_FLAG);
		return false;
	}


	pCryptKeyProvInfo.dwFlags=0;
	pCryptKeyProvInfo.cProvParam=0;
	pCryptKeyProvInfo.rgProvParam=NULL;

	// The name of the container
	cbData=sizeof(f_name);
	CryptGetProvParam(hProv, PP_CONTAINER, (PBYTE)f_name, &cbData, 0);
	pCryptKeyProvInfo.pwszContainerName=StringToUnicode(f_name);
	if(!pCryptKeyProvInfo.pwszContainerName)
	{
		free(pCryptKeyProvInfo.pwszProvName);
		CertCloseStore(hMyStore, CERT_CLOSE_STORE_FORCE_FLAG);
		return false;
	}

	//The type of provider
	cbData=sizeof(pCryptKeyProvInfo.dwProvType);
	if(!CryptGetProvParam(hProv, PP_PROVTYPE, (PBYTE)&(pCryptKeyProvInfo.dwProvType), &cbData, 0))
	{
		LastError = GetLastError();
		free(pCryptKeyProvInfo.pwszProvName);
		free(pCryptKeyProvInfo.pwszContainerName);
		CertCloseStore(hMyStore, CERT_CLOSE_STORE_FORCE_FLAG);
		return false;
	}

	
	pCryptKeyProvInfo.dwKeySpec=0;

	if(CryptGetUserKey(hProv, AT_SIGNATURE, &phUserKey))
	{
		CryptDestroyKey(phUserKey);
		pCryptKeyProvInfo.dwKeySpec = pCryptKeyProvInfo.dwKeySpec | AT_SIGNATURE;
	}
	else
	{
		if(CryptGetUserKey(hProv, AT_KEYEXCHANGE, &phUserKey))
		{
			CryptDestroyKey(phUserKey);
			pCryptKeyProvInfo.dwKeySpec = pCryptKeyProvInfo.dwKeySpec | AT_KEYEXCHANGE;
		}
		else
		{
			LastError = GetLastError();
			free(pCryptKeyProvInfo.pwszProvName);
			free(pCryptKeyProvInfo.pwszContainerName);
			CertCloseStore(hMyStore, CERT_CLOSE_STORE_FORCE_FLAG);
			return false;
		}
	}



	if(!CertSetCertificateContextProperty(CertificateContext, CERT_KEY_PROV_INFO_PROP_ID, 0, &pCryptKeyProvInfo))
	{
		LastError = GetLastError();
		free(pCryptKeyProvInfo.pwszProvName);
		free(pCryptKeyProvInfo.pwszContainerName);
		CertCloseStore(hMyStore, CERT_CLOSE_STORE_FORCE_FLAG);
		return false;
	}

	free(pCryptKeyProvInfo.pwszProvName);
	free(pCryptKeyProvInfo.pwszContainerName);

	if(!CertAddCertificateContextToStore(hMyStore, CertificateContext, CERT_STORE_ADD_ALWAYS, &phOutCert))
	{
		LastError = GetLastError();
		CertFreeCertificateContext(CertificateContext);
		CertCloseStore(hMyStore, CERT_CLOSE_STORE_FORCE_FLAG);
		return false;
	}
	
	CertCloseStore(hMyStore, CERT_CLOSE_STORE_FORCE_FLAG);

	return true;
}


void PkiEnroll::DestroyContainer()
{
	if(!m_ContainerName.size() ||
		!m_ProviderName.size())
		return;
	
	HCRYPTPROV hCryptProv = NULL;
	CryptAcquireContext(
				&hCryptProv,							// Address for handle to be returned.
				m_ContainerName.c_str(),			                // Use the current user's logon name.
				m_ProviderName.c_str(),   // Use the default provider.
				PROV_RSA_FULL,							// Need to both encrypt and sign.
				CRYPT_DELETEKEYSET);
}

void PkiEnroll::GenerateContainerName()
{
	unsigned char sha1[50];
	EVP_MD_CTX c;
	int i;
	char tt[20];

	//Generate random key container name
	EVP_MD_CTX_init(&c);
	EVP_DigestInit_ex(&c,EVP_sha1(), NULL);
	RAND_pseudo_bytes(sha1, sizeof(sha1));
	EVP_DigestUpdate(&c, sha1, sizeof(sha1));		
	EVP_DigestFinal_ex(&c,sha1,NULL);
	EVP_MD_CTX_cleanup(&c);

	m_ContainerName = "";
	
	for(i=0; i<SHA_DIGEST_LENGTH; i++)
	{
		sprintf(tt,"%02x",sha1[i]);
		m_ContainerName += tt;
	}
}

X509_PUBKEY * PkiEnroll::GeneratePrivateKey(const char * ProviderName, int RsaLen)
{
	HCRYPTPROV  hCryptProv;
	DWORD  cbPublicKeyInfo;
	CERT_PUBLIC_KEY_INFO*  pbPublicKeyInfo;
	ALG_ID TypeKey;
	HCRYPTKEY  pPrivKey;
	DWORD KeyLen;
	BYTE * DerPubKey;
	BYTE * p;
	DWORD DerPubKeyLen;
	X509_PUBKEY * publicKey;

	TypeKey=AT_SIGNATURE;

	
	m_ProviderName = ProviderName;
	GenerateContainerName();
	
	//--------------------------------------------------------------------
	//    Call CryptExportPublicKeyInfo to return an initialized
	//    CERT_PUBLIC_KEY_INFO structure.
	//    First, get a cryptographic provider.

	if(!CryptAcquireContext(
		&hCryptProv,							// Address for handle to be returned.
		m_ContainerName.c_str(),			    // Use the current user's logon name.
		m_ProviderName.c_str(),					// Use the default provider.
		PROV_RSA_FULL,							// Need to both encrypt and sign.
		CRYPT_NEWKEYSET ))
	{
		LastError = GetLastError();
		return NULL;
	}

	KeyLen = RsaLen * 0x10000;

	if(!CryptGenKey(hCryptProv, TypeKey, KeyLen, &pPrivKey))
	{
		LastError = GetLastError();
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return NULL;
	}
	CryptDestroyKey(pPrivKey);







	//--------------------------------------------------------------------
	// Call CryptExportPublicKeyInfo to get the size of the returned
	// information.

	if(!CryptExportPublicKeyInfo(
			  hCryptProv,            // Provider handle
			  TypeKey,          // Key spec
			  PKCS_7_ASN_ENCODING | X509_ASN_ENCODING,      // Encoding type
			  NULL,                  // pbPublicKeyInfo
			  &cbPublicKeyInfo))     // Size of PublicKeyInfo
	{
		LastError = GetLastError();
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return NULL;
	}


	//--------------------------------------------------------------------
	// Allocate the necessary memory.
	if(!(pbPublicKeyInfo = (CERT_PUBLIC_KEY_INFO*)malloc(cbPublicKeyInfo)))
	{
		LastError = GetLastError();
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return NULL;
	}

	//--------------------------------------------------------------------
	// Call CryptExportPublicKeyInfo to get pbPublicKeyInfo.
	if(!CryptExportPublicKeyInfo(
			  hCryptProv,            // Provider handle
			  TypeKey,          // Key spec
			  PKCS_7_ASN_ENCODING | X509_ASN_ENCODING,      // Encoding type
			  pbPublicKeyInfo,       // pbPublicKeyInfo
			  &cbPublicKeyInfo))     // Size of PublicKeyInfo
	{
		LastError = GetLastError();
		free(pbPublicKeyInfo);
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return NULL;
	}

	//--------------------------------------------------------------------
	// We now convert it to DER
	DerPubKeyLen = 0;
	if(!CryptEncodeObject(PKCS_7_ASN_ENCODING | X509_ASN_ENCODING, X509_PUBLIC_KEY_INFO, pbPublicKeyInfo, NULL, &DerPubKeyLen))
	{
		LastError = GetLastError();
		free(pbPublicKeyInfo);
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return NULL;
	}

	//--------------------------------------------------------------------
	// Allocate the necessary memory.
	if(!(DerPubKey = (BYTE*)malloc(DerPubKeyLen)))
	{
		LastError = GetLastError();
		free(pbPublicKeyInfo);
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return NULL;
	}
	if(!CryptEncodeObject(PKCS_7_ASN_ENCODING | X509_ASN_ENCODING, X509_PUBLIC_KEY_INFO, pbPublicKeyInfo, DerPubKey, &DerPubKeyLen))
	{
		LastError = GetLastError();
		free(pbPublicKeyInfo);
		free(DerPubKey);
		CryptReleaseContext(hCryptProv, 0);
		DestroyContainer();
		return NULL;
	}
	free(pbPublicKeyInfo);

	// We now convert it back to OpenSSL's format
	p = DerPubKey;
	publicKey = d2i_X509_PUBKEY(NULL, &p, DerPubKeyLen);
	free(DerPubKey);
	
	if(!publicKey)
	{
		LastError = ERROR_OUTOFMEMORY;
	}

	CryptReleaseContext(hCryptProv, 0);
	return publicKey;
}

PCCERT_CONTEXT PkiEnroll::ConvertCertFromX509(X509 * x509)
{
	unsigned char * datas;
	unsigned char * p;
	long datas_len;
	PCCERT_CONTEXT pCertContext;

	datas_len=i2d_X509(x509, NULL);
	if(datas_len <= 0)
	{
		LastError = ERROR_INVALID_DATA;
		return NULL;
	}
	datas=(unsigned char *)malloc(datas_len);
	if(!datas)
	{
		LastError = ERROR_OUTOFMEMORY;
		return NULL;
	}
	p=datas;
	datas_len=i2d_X509(x509, &p);
	if(datas_len <= 0)
	{
		LastError = ERROR_INVALID_DATA;
		free(datas);
		return NULL;
	}

	pCertContext=CertCreateCertificateContext(X509_ASN_ENCODING | PKCS_7_ASN_ENCODING, datas, datas_len);
	if(!pCertContext)
	{
		LastError = GetLastError();
		free(datas);
		return NULL;
	}
	free(datas);

	return pCertContext;
}

bool PkiEnroll::GetPkcs1KDatas(const RSA * rsakey, BYTE * pbData, DWORD * pbDataLen)
{
	int   nRSAPubKeyBitLen;
	DWORD dwOffset = 0;
	DWORD dwPrivateKeyBOLBSize;

	RSAPUBKEY *pRsaPubKeyHeader;
	PUBLICKEYSTRUC *pPubKeyHeader;
	BYTE* pModulus;
	BYTE* pPrime1;
	BYTE* pPrime2;
	BYTE* pExponent2;
	BYTE* pCoefficient;
	BYTE* pPrivateExponent;

	nRSAPubKeyBitLen=BN_num_bytes(rsakey->n) * 8;



	dwPrivateKeyBOLBSize =	sizeof(BLOBHEADER)
							+sizeof(RSAPUBKEY)
							+nRSAPubKeyBitLen/8  // modulus or n
							+nRSAPubKeyBitLen/16 // prime1  or p
							+nRSAPubKeyBitLen/16 // prime2  or q
							+nRSAPubKeyBitLen/16 // exponent1 or d mod (p-1)
							+nRSAPubKeyBitLen/16 // exponent2 or d mod (q-1)
							+nRSAPubKeyBitLen/16 // coefficient or (inverse of q) mod p
							+nRSAPubKeyBitLen/8; // private exponent or d

	*pbDataLen=dwPrivateKeyBOLBSize;

	if(!pbData) return true;



	pPubKeyHeader = (PUBLICKEYSTRUC*) (&pbData[dwOffset]);
	pPubKeyHeader->bType = PRIVATEKEYBLOB;
	pPubKeyHeader->bVersion = 0x02;
	pPubKeyHeader->reserved = 0;
	pPubKeyHeader->aiKeyAlg = CALG_RSA_SIGN;

	dwOffset += sizeof(PUBLICKEYSTRUC);
	pRsaPubKeyHeader = (RSAPUBKEY*) ( &pbData[dwOffset] );
	pRsaPubKeyHeader->magic = 0x32415352; //RSA2
	pRsaPubKeyHeader->bitlen = nRSAPubKeyBitLen;
	pRsaPubKeyHeader->pubexp = (DWORD) rsakey->e->d[0];

	dwOffset += sizeof(RSAPUBKEY);
	pModulus = (BYTE*) ( &pbData[dwOffset]  );
	memcpy (pModulus, rsakey->n->d, nRSAPubKeyBitLen/8 );  // read in the 1024 bits modulus

	dwOffset += nRSAPubKeyBitLen/8;
	pPrime1 = (BYTE*) ( &pbData[dwOffset]  );
	memcpy ( pPrime1, rsakey->p->d, nRSAPubKeyBitLen/16 );

	dwOffset += nRSAPubKeyBitLen/16;
	pPrime2 = (BYTE*) ( &pbData[dwOffset]  );
	memcpy ( pPrime2, rsakey->q->d, nRSAPubKeyBitLen/16 );

	dwOffset += nRSAPubKeyBitLen/16;
	BYTE* pExponent1 = (BYTE*) ( &pbData[dwOffset]  );
	memcpy ( pExponent1, rsakey->dmp1->d, nRSAPubKeyBitLen/16 );

	dwOffset += nRSAPubKeyBitLen/16;
	pExponent2 = (BYTE*) ( &pbData[dwOffset]  );
	memcpy ( pExponent2, rsakey->dmq1->d, nRSAPubKeyBitLen/16 );

	dwOffset += nRSAPubKeyBitLen/16;
	pCoefficient = (BYTE*) ( &pbData[dwOffset]  );
	memcpy ( pCoefficient, rsakey->iqmp->d, nRSAPubKeyBitLen/16 );

	dwOffset += nRSAPubKeyBitLen/16;
	pPrivateExponent = (BYTE*) ( &pbData[dwOffset]  );
	memcpy ( pPrivateExponent, rsakey->d->d, nRSAPubKeyBitLen/8 );

	return true;
}

HCRYPTPROV PkiEnroll::ConvertKeyFromRSA(const RSA * Key)
{
	BYTE * raw_key;
	DWORD raw_key_len;
	HCRYPTKEY PrivateKey = NULL;
	HCRYPTPROV hProv;
	DWORD Flags;

	GetPkcs1KDatas(Key, NULL, &raw_key_len);
	raw_key=(BYTE *)malloc((WORD)raw_key_len+10);
	if(!raw_key)
	{
		LastError = ERROR_OUTOFMEMORY;
		return NULL;
	}
	GetPkcs1KDatas(Key, raw_key, &raw_key_len);
	
	
	if(!CryptAcquireContext(&hProv, m_ContainerName.c_str(), m_ProviderName.c_str(), PROV_RSA_FULL, 0))
	{                 
		if(!CryptAcquireContext(&hProv, m_ContainerName.c_str(), m_ProviderName.c_str(), PROV_RSA_FULL, CRYPT_NEWKEYSET))
		{
			LastError = GetLastError();
			free(raw_key);
			return NULL;
		}
	}
	
	Flags=CRYPT_OAEP;

	if(!CryptImportKey(hProv, raw_key, raw_key_len, NULL, Flags, &PrivateKey))
	{
		LastError = GetLastError();
		free(raw_key);
		CryptReleaseContext(hProv, 0);
		DestroyContainer();
		return NULL;
	}
	free(raw_key);
	raw_key_len=0;

	CryptDestroyKey(PrivateKey);

	return hProv;
}


bool PkiEnroll::import_PKCS12toCSP(const char * provider, const PKI_PKCS12 & p12)
{
	HCRYPTPROV privateKeyProv;
	PCCERT_CONTEXT pCert;

	m_ProviderName = provider;
	GenerateContainerName();

	pCert = ConvertCertFromX509(p12.GetEndUserCert().GetX509());
	if(!pCert)
	{
		return false;
	}
	privateKeyProv = ConvertKeyFromRSA(p12.GetEndUserKey().GetRSA());
	if(!privateKeyProv)
	{
		CertFreeCertificateContext(pCert);
		return false;
	}

	if(!ImportCertToCSP(pCert, privateKeyProv))
	{
		if(!ImportCertToMY(pCert, privateKeyProv))
		{
			CertFreeCertificateContext(pCert);
			CryptReleaseContext(privateKeyProv, 0);
			DestroyContainer();
			return false;
		}
	}

	CertFreeCertificateContext(pCert);
	CryptReleaseContext(privateKeyProv, 0);
	return true;
}

#endif //_WIN32
