/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// DlgRaAdmin.cpp: implementation of the DlgRaAdmin class.
//
//////////////////////////////////////////////////////////////////////

#ifdef _WIN32
#pragma warning(disable:4786)
#endif

#include "DlgSendMail.h"
#include "DlgCertProperties.h"
#include "DlgP7bProperties.h"
#include "DlgPublishDN.h"
#include "DlgRaAdmin.h"
#include "DlgRequestCert.h"
#include "DlgBashImport.h"
#include "DlgImportP12.h"
#include "dlgs_wdr.h"
#include "clintl.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////



BEGIN_EVENT_TABLE(DlgRaAdminListCtrl, wxListCtrl)
    EVT_LIST_ITEM_RIGHT_CLICK(IDC_LIST_PROFILES, DlgRaAdminListCtrl::OnItemRightClick)
    EVT_LIST_ITEM_SELECTED(IDC_LIST_PROFILES, DlgRaAdminListCtrl::OnItemClick)
    EVT_MOUSE_EVENTS(DlgRaAdminListCtrl::OnMouseEvent)
END_EVENT_TABLE()


DlgRaAdmin * m_DlgRaAdminParent;


#ifdef _WIN32

wxBitmap DlgRaAdminListCtrlhPic = Backgrounds_GetBitmap(0);

long DlgRaAdminListCtrloldProc;
int DlgRaAdminListCtrlcenterX;
int DlgRaAdminListCtrlcenterY;
bool DlgRaAdminListCtrlinProc = false;

LRESULT CALLBACK DlgRaAdminListCtrllstvProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	RECT rRect;
	PAINTSTRUCT ps;
	HDC hDC;
	HDC hDCParent;
	HBITMAP bmp;
	HBITMAP bmpOld;
				
	switch (message) 
	{
		case WM_PAINT:
			GetClientRect(hwnd,&rRect);

			if(DlgRaAdminListCtrlinProc)
			{
				DlgRaAdminListCtrlinProc = false;
				InvalidateRect(hwnd, &rRect, TRUE);
				return TRUE;
			}
			DlgRaAdminListCtrlinProc = true;


			CallWindowProcA((WNDPROC)DlgRaAdminListCtrloldProc, hwnd, message, wParam, lParam);		

			BeginPaint(hwnd, &ps);
			
			hDC=GetDC(hwnd);
			hDCParent=CreateCompatibleDC(hDC);
			bmp = (HBITMAP)DlgRaAdminListCtrlhPic.GetHBITMAP();
			
			
			bmpOld = (HBITMAP)SelectObject(hDCParent, bmp);
			
			
			DlgRaAdminListCtrlcenterX = ((rRect.right - rRect.left) - DlgRaAdminListCtrlhPic.GetWidth()) / 2;
			DlgRaAdminListCtrlcenterY = ((rRect.bottom - rRect.top) - DlgRaAdminListCtrlhPic.GetHeight())  / 2;
			
			
			BitBlt(hDC,DlgRaAdminListCtrlcenterX, DlgRaAdminListCtrlcenterY, DlgRaAdminListCtrlhPic.GetWidth(),DlgRaAdminListCtrlhPic.GetHeight(), hDCParent, 0, 0,SRCAND );
			
			ReleaseDC(hwnd,hDC);
			SelectObject(hDCParent, bmpOld);
			DeleteDC(hDCParent);

			EndPaint(hwnd, &ps);
			return TRUE;
			
		default:
			return CallWindowProcA((WNDPROC)DlgRaAdminListCtrloldProc, hwnd, message, wParam, lParam);
	}
	return FALSE;
}

#endif

DlgRaAdminListCtrl::DlgRaAdminListCtrl(wxWindow *parent, const wxWindowID id, const wxPoint& pos,	const wxSize& size, long style):wxListCtrl(parent, id, pos, size, style)
{
	#ifdef _WIN32
		DlgRaAdminListCtrloldProc = SetWindowLong((HWND)GetHWND(), GWL_WNDPROC, (long)DlgRaAdminListCtrllstvProc);
	#endif
}

DlgRaAdminListCtrl::~DlgRaAdminListCtrl()
{
}

void DlgRaAdminListCtrl::OnItemRightClick(wxListEvent& event)
{
	m_DlgRaAdminParent->OnItemRightClick(event);
}

void DlgRaAdminListCtrl::OnItemClick(wxListEvent& event)
{
	m_DlgRaAdminParent->OnItemClick(event);
}

void DlgRaAdminListCtrl::OnMouseEvent(wxMouseEvent & event)
{
	m_DlgRaAdminParent->OnMouseEvent(event);
}



BEGIN_EVENT_TABLE(DlgRaAdmin, wxFrame)
	EVT_MENU(IDM_QUIT, DlgGUI::OnQuit)
    EVT_MENU(IDM_RELOAD, DlgRaAdmin::OnReload)
    EVT_MENU(IDM_SHOW_LOGS, DlgGUI::OnShowLogs)
    EVT_MENU(IDM_RECONNECT, DlgGUI::OnReconnect)
    EVT_MENU(IDM_SHOW_SSL_CERT, DlgGUI::ShowSSL_Certificate)
    EVT_MENU(IDM_SHOW_ENTITY_CERT, DlgGUI::ShowEntity_Certificate)
    EVT_MENU(IDM_ABOUT, DlgGUI::OnAbout)
    EVT_MENU(IDC_SEND_ADMIN_MAIL, DlgRaAdmin::OnSendAdminMail)
    EVT_MENU(IDC_SHOW_PREV, DlgRaAdmin::OnShowPrevEntries)
    EVT_MENU(IDC_REFRESH, DlgRaAdmin::OnRefreshEntries)
    EVT_TEXT_ENTER(IDC_DN_FILTER, DlgRaAdmin::OnRefreshEntries)
    EVT_COMBOBOX(IDC_FILTER_STATUS, DlgRaAdmin::OnRefreshEntries)
    EVT_MENU(IDC_SHOW_NEXT, DlgRaAdmin::OnShowNextEntries)
    EVT_MENU(IDM_CREATE_PROFILE, DlgRaAdmin::OnCreateProfile)
    EVT_MENU(IDM_REQUEST_CERT, DlgRaAdmin::OnRequestCert)
    EVT_MENU(IDM_REVOKE_CERT, DlgRaAdmin::OnRevokeCert)
    EVT_MENU(IDM_SUSPEND_CERT, DlgRaAdmin::OnSuspendCert)
    EVT_MENU(IDM_UNSUSPEND_CERT, DlgRaAdmin::OnUnsuspendCert)
    EVT_MENU(IDM_CERT_PROPERTIES, DlgRaAdmin::OnCertProperties)
    EVT_MENU(IDM_P7B_PROPERTIES, DlgRaAdmin::OnP7bProperties)
    EVT_MENU(IDM_SAVE_P12, DlgRaAdmin::OnSavePKCS12)
    EVT_MENU(IDM_DELETE_P12, DlgRaAdmin::OnDeletePKCS12)
    EVT_MENU(IDM_IMPORT_P12, DlgRaAdmin::OnImportP12)
    EVT_MENU(IDM_IMPORT_SC, DlgRaAdmin::OnImportCert)
    EVT_MENU(IDM_BATCH_IMPORT, DlgRaAdmin::OnBatchImport)
    EVT_MENU(IDM_CHANGE_UID, DlgRaAdmin::OnChangeProfileUID)
    EVT_MENU(IDM_CHANGE_OWNER, DlgRaAdmin::OnChangeProfileOwner)
    EVT_MENU(IDM_DELETE_PROFILE, DlgRaAdmin::OnDeleteProfile)
    EVT_MENU(IDM_DELETE_REQ, DlgRaAdmin::OnDeleteRequest)
    EVT_MENU(IDM_CHANGE_DN, DlgRaAdmin::OnChangeProfileDN)
    EVT_MENU(IDM_ACCEPT_DN, DlgRaAdmin::OnAcceptProfile)
    EVT_MENU(IDM_REJECT_DN, DlgRaAdmin::OnRejectProfile)
END_EVENT_TABLE()


DlgRaAdmin::DlgRaAdmin(wxWindow * wParent, char * EntityName, char * UserName, char * Password, PkiClient * ClientPki):DlgGUI(wParent, ClientPki, EntityName, UserName, Password)
{
	m_DlgRaAdminParent = this;
	CurrentPosition = 0;
	my_acl = NULL;
	LastSelectedEntry = -1;


    wxToolBar * toolBar = new wxToolBar(this, -1);

	wxString mTitle;
	mTitle.sprintf("RA Admin (%s/%s) [%s:%d]", UserName, EntityName, ClientPki->GetRemoteServer().c_str(), ClientPki->GetRemotePort());
	SetTitle(mTitle);

	DlgRaAdmin_SetWindow(this);
	SetMenuBar(DlgRaAdmin_GetMenu());

	toolBar->SetToolBitmapSize(wxSize(32,32));
	DlgRaAdmin_SetToolbar(toolBar);
    SetToolBar(toolBar);


	CenterOnScreen();

	m_imageListSmall = new wxImageList(16, 16, TRUE);

	
	m_listCtrl = (DlgRaAdminListCtrl *)FindWindow(IDC_LIST_PROFILES);
	if(!m_listCtrl) return;

	wxIcon ico;
	
	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_EMPTY_PROFILE));
	EmptyProfileIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_PLUS_PROFILE));
	PlusProfileIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_MINUS_PROFILE));
	MinusProfileIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_ACT_MID));
	ActMidConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_ACT_END));
	ActEndConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_WAI_MID));
	WaitMidConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_WAI_END));
	WaitEndConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_REV_MID));
	RevMidConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_REV_END));
	RevEndConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_SUSP_MID));
	SuspMidConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_SUSP_END));
	SuspEndConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_WAI_PROFILE));
	WaitingProfileIconId = m_imageListSmall->Add( ico );
	
	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_ERR_MID));
	ErrMidConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_ERR_END));
	ErrEndConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_WAI_REV_MID));
	WaitRevMidConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_WAI_REV_END));
	WaitRevEndConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_WAI_SUSP_MID));
	WaitSuspMidConnectorIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgRaAdmin_GetBitmap(IDB_WAI_SUSP_END));
	WaitSuspEndConnectorIconId = m_imageListSmall->Add( ico );

	m_listCtrl->SetImageList(m_imageListSmall, wxIMAGE_LIST_SMALL);

	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("DN"), wxLIST_ALIGN_DEFAULT, 385);
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("ID"), wxLIST_ALIGN_DEFAULT, 60);

	((wxTextCtrl *)FindWindow(IDC_INFO))->SetBackgroundColour(GetBackgroundColour());

	lastItem = -1;

	((wxTextCtrl *)FindWindow(IDC_INFO))->SetBackgroundColour(GetBackgroundColour());

	if(!ReloadConf())
	{
		Close(TRUE);
		return;
	}

	((wxComboBox *)FindWindow(IDC_FILTER_STATUS))->Append(
					_("All"), (void*)0);
	((wxComboBox *)FindWindow(IDC_FILTER_STATUS))->Append(
					GetProfileStatusString(NEWPKI_PROFILE_STATE_VALIDATED), 
					(void*)NEWPKI_PROFILE_STATE_VALIDATED);
	((wxComboBox *)FindWindow(IDC_FILTER_STATUS))->Append(
					GetProfileStatusString(NEWPKI_PROFILE_STATE_WAITING_FOR_VALIDATION), 
					(void*)NEWPKI_PROFILE_STATE_WAITING_FOR_VALIDATION);
	((wxComboBox *)FindWindow(IDC_FILTER_STATUS))->Append(
					GetProfileStatusString(NEWPKI_PROFILE_STATE_WAITING_FOR_ADMIN_VALIDATION), 
					(void*)NEWPKI_PROFILE_STATE_WAITING_FOR_ADMIN_VALIDATION);
	((wxComboBox *)FindWindow(IDC_FILTER_STATUS))->SetSelection(0);
}

DlgRaAdmin::~DlgRaAdmin()
{
	if(m_listCtrl) m_listCtrl->ClearAll();
	if(m_imageListSmall) delete m_imageListSmall;
	if(my_acl) ASN1_BIT_STRING_free(my_acl);
}


void DlgRaAdmin::OnReload(wxCommandEvent& event)
{
	ReloadConf();
}


void DlgRaAdmin::OnItemClick(wxListEvent &event)
{
}

void DlgRaAdmin::OnItemRightClick(wxListEvent &event)
{
	wxMenu * menu=NULL;
	NewpkiProfileDatas * Profile;
	NewpkiProfileDatasCert * Cert;
	
	
	wxMenuBar * menu_b = DlgRaAdmin_GetPopup();
	if(!menu_b) return;

	if( (Profile = GetSelectedProfile()) )
	{
		// Click on a profile
		menu = menu_b->GetMenu(0);
		if(menu)
		{
			if(Profile->get_state() != NEWPKI_PROFILE_STATE_VALIDATED)
			{
				menu->Enable(IDM_REQUEST_CERT, false);
				menu->Enable(IDM_CHANGE_DN, false);
			}
			if(Profile->get_state() != NEWPKI_PROFILE_STATE_WAITING_FOR_ADMIN_VALIDATION)
			{
				menu->Enable(IDM_ACCEPT_DN, false);
				menu->Enable(IDM_REJECT_DN, false);
			}
			else
			{
				menu->Enable(IDM_REQUEST_CERT, false);
				menu->Enable(IDM_CHANGE_UID, false);
				menu->Enable(IDM_CHANGE_OWNER, false);
				menu->Enable(IDM_DELETE_PROFILE, false);
				menu->Enable(IDM_CHANGE_DN, false);
			}

			if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_RA_DELEGATE_PROFILE_OWNERSHIP))
			{
				menu->Enable(IDM_CHANGE_OWNER, false);
			}
		}
	}
	else if( (Cert = GetSelectedCert()) )
	{
		// Click on a certificate
		menu = menu_b->GetMenu(1);
		if(menu)
		{
			menu->Enable(IDM_IMPORT_SC, false);
			menu->Enable(IDM_IMPORT_P12, false);
			switch(Cert->get_state())
			{
				case NEWPKI_PROFILE_CERT_STATE_ERROR:
				case NEWPKI_PROFILE_CERT_STATE_WAITING:
				case NEWPKI_PROFILE_CERT_STATE_WAITING_REV:
				case NEWPKI_PROFILE_CERT_STATE_REVOKED:
				case NEWPKI_PROFILE_CERT_STATE_WAITING_SUSP:
				case NEWPKI_PROFILE_CERT_STATE_WAITING_UNSUSP:
					menu->Enable(IDM_REVOKE_CERT, false);
					menu->Enable(IDM_SUSPEND_CERT, false);
					menu->Enable(IDM_UNSUSPEND_CERT, false);
					menu->Enable(IDM_CERT_PROPERTIES, false);
					menu->Enable(IDM_P7B_PROPERTIES, false);
					
					menu->Enable(IDM_SAVE_P12, false);
					menu->Enable(IDM_DELETE_P12, false);
					break;
				case NEWPKI_PROFILE_CERT_STATE_SUSPENDED:
					menu->Enable(IDM_REVOKE_CERT, false);
					menu->Enable(IDM_SUSPEND_CERT, false);
					menu->Enable(IDM_CERT_PROPERTIES, false);
					menu->Enable(IDM_P7B_PROPERTIES, false);
					
					menu->Enable(IDM_SAVE_P12, false);
					menu->Enable(IDM_DELETE_P12, false);
					break;
				case NEWPKI_PROFILE_CERT_STATE_ACTIVE:
					menu->Enable(IDM_UNSUSPEND_CERT, false);
					if(!Cert->get_p7b())
						menu->Enable(IDM_P7B_PROPERTIES, false);
					switch(Cert->get_type())
					{
						case NEWPKI_PROFILE_CERT_TYPE_PKCS10:
							menu->Enable(IDM_SAVE_P12, false);
							menu->Enable(IDM_DELETE_P12, false);
							break;
							
						case NEWPKI_PROFILE_CERT_TYPE_PKCS12:
							if(Cert->get_p12())
							{
								menu->Enable(IDM_SAVE_P12, true);
								menu->Enable(IDM_DELETE_P12, true);
#ifdef _WIN32
								menu->Enable(IDM_IMPORT_P12, true);
#endif
							}
							else
							{
								menu->Enable(IDM_SAVE_P12, false);
								menu->Enable(IDM_DELETE_P12, false);
							}
							break;
							
						case NEWPKI_PROFILE_CERT_TYPE_SC:
							menu->Enable(IDM_SAVE_P12, false);
							menu->Enable(IDM_DELETE_P12, false);
#ifdef _WIN32
							menu->Enable(IDM_IMPORT_SC, true);
#endif
							break;
					}
					break;
			}
			if(Cert->get_state() == NEWPKI_PROFILE_CERT_STATE_ERROR)
				menu->Enable(IDM_DELETE_REQ, true);
			else
				menu->Enable(IDM_DELETE_REQ, false);
		}
	}

	if(menu)
		m_listCtrl->PopupMenu(menu, event.GetPoint().x, event.GetPoint().y);
	delete menu_b;
}


bool DlgRaAdmin::ReloadConf()
{
	wxMenuBar * menubar;
	wxString strLng;

	m_DlgMsg->wShow(_("Loading Configuration..."));

	if(my_acl)
	{
		ASN1_BIT_STRING_free(my_acl);
		my_acl = NULL;
	}

	m_EntityConf.Clear();
	if(!m_ClientPki->GetLocalConf(m_EntityConf))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return false;
	}

	if(!m_ClientPki->GetMyACL(&my_acl))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return false;
	}

	menubar = GetMenuBar();

	if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_RA_MANAGE_PROFILES))
	{
		menubar->GetMenu(1)->Enable(IDM_CREATE_PROFILE, false);
	}
	else
	{
		menubar->GetMenu(1)->Enable(IDM_CREATE_PROFILE, true);
		RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
	}
	
	if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_VIEW_LOGS))
	{
		menubar->GetMenu(2)->Enable(IDM_SHOW_LOGS, false);
	}
	else
	{
		menubar->GetMenu(2)->Enable(IDM_SHOW_LOGS, true);
	}
	
	if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_SEND_ADMIN_MAIL))
	{
		GetToolBar()->EnableTool( IDC_SEND_ADMIN_MAIL, FALSE );
	}
	else
	{
		GetToolBar()->EnableTool( IDC_SEND_ADMIN_MAIL, TRUE );
	}
	m_DlgMsg->wHide();
	
	return true;
}

void DlgRaAdmin::OnCreateProfile(wxCommandEvent &event)
{
	DlgPublishDN Dlg(this, m_EntityConf.get_body(), my_acl, m_ClientPki, NULL, 0);
	if(Dlg.IsOk())
	{
		RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
	}
}

void DlgRaAdmin::OnSendAdminMail(wxCommandEvent &event)
{
	DlgSendMail Dlg(this, m_ClientPki);
}


void DlgRaAdmin::ShowNextEntries(int NumEntries)
{
	CurrentPosition = CurrentPosition + NumEntries;
	RefreshDisplay(NumEntries);
	if(!m_listCtrl->GetItemCount())
		ShowPrevEntries(NumEntries);
	else
		listStatus.clear();
}

void DlgRaAdmin::ShowPrevEntries(int NumEntries)
{
	CurrentPosition = CurrentPosition - NumEntries;
	if(CurrentPosition < 0) CurrentPosition = 0;
	RefreshDisplay(NumEntries);
	listStatus.clear();
}

void DlgRaAdmin::RefreshDisplay(int NumEntries)
{
	LoadPROFILES(NumEntries);
}

bool DlgRaAdmin::LoadPROFILES(int NumEntries)
{
	int SelectedItem;
	int state;
	mString filter;

	lastItem = -1;
	
	m_DlgMsg->wShow(_("Loading Profile(s) List..."));
	wxString Page;
	Page.sprintf("%ld", (CurrentPosition / NumEntries) + 1);
	((wxStaticText *)FindWindow(IDC_PAGE))->SetLabel(Page);
	
	((wxTextCtrl *)FindWindow(IDC_INFO))->SetValue("");


	SelectedItem = ((wxComboBox *)FindWindow(IDC_FILTER_STATUS))->GetSelection();
	if(SelectedItem == -1)
		state = 0;
	else
		state = (int)((wxComboBox *)FindWindow(IDC_FILTER_STATUS))->GetClientData(SelectedItem);

	filter = ((wxTextCtrl *)FindWindow(IDC_DN_FILTER))->GetValue().c_str();

	m_listCtrl->DeleteAllItems();
	m_profiles.clear();
	if(!m_ClientPki->EnumProfiles(CurrentPosition, NumEntries, state, filter, m_profiles))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return false;
	}

	PrintPROFILES();
	m_DlgMsg->wHide();
	return true;
}

void DlgRaAdmin::OnShowNextEntries(wxCommandEvent &event)
{
	ShowNextEntries(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgRaAdmin::OnShowPrevEntries(wxCommandEvent &event)
{
	ShowPrevEntries(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgRaAdmin::OnRefreshEntries(wxCommandEvent &event)
{
	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgRaAdmin::PrintPROFILES()
{
	size_t i;
	mString tmpDatas;
	int IconId;
	HashTable_Dn Dn;
	int item;
	
	for(i=0; i<m_profiles.size(); i++)
	{
		if(m_profiles[i].get_certs().size())		
			IconId = PlusProfileIconId;
		else
			IconId = EmptyProfileIconId;


		if(!Dn.From_X509_NAME(m_profiles[i].get_profile().get_dn()))
		{
			tmpDatas = _("Unknown");
		}
		else
		{
			tmpDatas = FormatDN(Dn, true);
			if(!tmpDatas.size())
				tmpDatas = _("Unknown");
		}

		item = m_listCtrl->GetItemCount();
		m_listCtrl->InsertItem(item , tmpDatas.c_str(), IconId);
		m_listCtrl->SetItemData(item, (long)&m_profiles[i]);

		tmpDatas = m_profiles[i].get_id();
		m_listCtrl->SetItem(item, 1, tmpDatas.c_str());

		if(m_profiles[i].get_eeId().size())
			m_listCtrl->SetItemTextColour(item, wxColour(255,0,255));

		switch(listStatus[tmpDatas.c_ulng()]) {
			case PROFILE_STATUS_UNKNOWN:
				listStatus[tmpDatas.c_ulng()] = PROFILE_STATUS_CLOSED;
				break;
			case PROFILE_STATUS_CLOSED:
				break;
			case PROFILE_STATUS_OPENED:
				DrawOpenProfile(item, &m_profiles[i]);
				break;
		}
	}
}

void DlgRaAdmin::MouseDown(long item)
{
	wxListItem info;
	NewpkiProfileDatas * Profile;
	
	info.m_mask = wxLIST_MASK_IMAGE ;
	info.m_itemId = item;
	if (!m_listCtrl->GetItem(info))
	{
		return;
	}
	
	if( (Profile = GetSelectedProfile(item)) )
	{
		if(Profile->get_certs().size())
		{
			if(info.m_image == PlusProfileIconId)
			{
				DrawOpenProfile(item, Profile);
			}
			else if(info.m_image == MinusProfileIconId)
			{
				DrawCloseProfile(item, Profile);
			}
		}
	}
}

void DlgRaAdmin::MouseMove(long item)
{
	wxListItem info;
	NewpkiProfileDatas * Profile;
	NewpkiProfileDatasCert * currCert;
	size_t i;
	mString strInfo;
	const char * strState;
	
	if(lastItem == item)
		return;

	lastItem = item;

	if( (Profile = GetSelectedProfile(item)) )
	{
		strState = GetProfileStatusString(Profile->get_state());
		strInfo.sprintf(_("Id: %ld\nState: %s\n"), Profile->get_id(), strState);

		if(Profile->get_profile().get_ldapUid().size())
		{
			strInfo += _("LDAP UID: ");
			strInfo += Profile->get_profile().get_ldapUid();
			strInfo += "\n";
		}

		if(Profile->get_eeId().size())
		{
			strInfo += _("EE Id: ");
			strInfo += Profile->get_eeId();
			strInfo += "\n";
		}

		if(Profile->get_profile().get_ownerGroupSerial())
		{
			strInfo += _("Owner group: ");
			// Search the owner group
			for(i=0; i<m_EntityConf.get_groups().size(); i++)
			{
				if(m_EntityConf.get_groups()[i].get_serial() == Profile->get_profile().get_ownerGroupSerial())
				{
					strInfo += m_EntityConf.get_groups()[i].get_name();
					break;
				}
			}
			if(i == m_EntityConf.get_groups().size())
			{
				strInfo += _("Unknown");
			}
			strInfo += "\n";
		}
	}
	else if( (currCert = GetSelectedCert(item)) )
	{
		strInfo.sprintf(_("Id: %ld"), currCert->get_id());

		strInfo += _("\nState: ");
		strInfo += GetCertificateStatusString(currCert->get_state());

		strInfo += _("\nType: ");
		strInfo += GetCertificateTypeString(currCert->get_type());

		strInfo += _("\nCA: ");
		strInfo += currCert->get_caName();
		strInfo += "\n";

		strInfo += "\n";
		strInfo += currCert->get_error();
	}
	else
	{
		strInfo = "";
	}
	
	((wxTextCtrl *)FindWindow(IDC_INFO))->SetValue(strInfo.c_str());
}

void DlgRaAdmin::OnMouseEvent(wxMouseEvent &event)
{
	int flags;
	long item = m_listCtrl->HitTest(event.GetPosition(), flags);
	if (item > -1)
	{
		if(event.LeftDown())
		{
			MouseDown(item);
		}
		else if(event.Moving())
		{
			MouseMove(item);
		}
	}
	event.Skip();
}

void DlgRaAdmin::OnRequestCert(wxCommandEvent &event)
{
	NewpkiProfileDatas * Profile;
	unsigned long reqid;
	NewpkiRequestCert cert_request;
	
	Profile = GetSelectedProfile();
	if(!Profile)
		return;

	
	DlgRequestCert Dlg(this, Profile->get_profile().get_dn(), m_EntityConf, NULL);
	if(!Dlg.IsOk())
	{
		return;
	}

	cert_request = Dlg.get_Request();
	if(!cert_request)
	{
		Dlg.Cancel();
		return;
	}
	cert_request.set_profileId(Profile->get_id());


	m_DlgMsg->wShow(_("Publishing Certificate Request..."));
	if(!m_ClientPki->RequestCertificate(cert_request, reqid))
	{
		Dlg.Cancel();
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgRaAdmin::OnCertProperties(wxCommandEvent &event)
{
	NewpkiProfileDatasCert * Cert;
	
	Cert = GetSelectedCert();
	if(Cert && Cert->get_cert())
	{
		DlgCertProperties Dlg(this, Cert->get_cert());
	}
}

void DlgRaAdmin::OnP7bProperties(wxCommandEvent &event)
{
	NewpkiProfileDatasCert * Cert;
	
	Cert = GetSelectedCert();
	if(Cert && Cert->get_p7b())
	{
		DlgP7bProperties Dlg(this, Cert->get_p7b());
	}
}

void DlgRaAdmin::OnRevokeCert(wxCommandEvent &event)
{
	NewpkiProfileDatasCert * Cert;

	Cert = GetSelectedCert();
	if(!Cert)
		return;

	if(DisplayMessage(this, _("Are you sure, you want to revoke this certificate ?"), wxYES_NO) == wxNO) return;
	
	m_DlgMsg->wShow(_("Revoking Certificate..."));
	if(!m_ClientPki->RevokeCert(Cert->get_id()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgRaAdmin::OnSuspendCert(wxCommandEvent &event)
{
	NewpkiProfileDatasCert * Cert;

	Cert = GetSelectedCert();
	if(!Cert)
		return;

	if(DisplayMessage(this, _("Are you sure, you want to suspend this certificate ?"), wxYES_NO) == wxNO) return;
	
	m_DlgMsg->wShow(_("Suspending Certificate..."));
	if(!m_ClientPki->SuspendCert(Cert->get_id()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgRaAdmin::OnUnsuspendCert(wxCommandEvent &event)
{
	NewpkiProfileDatasCert * Cert;

	Cert = GetSelectedCert();
	if(!Cert)
		return;

	if(DisplayMessage(this, _("Are you sure, you want to unsuspend this certificate ?"), wxYES_NO) == wxNO) return;
	
	m_DlgMsg->wShow(_("Unsuspending Certificate..."));
	if(!m_ClientPki->UnsuspendCert(Cert->get_id()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgRaAdmin::DrawOpenProfile(long item, NewpkiProfileDatas * Profile)
{
	bool LastOne;
	size_t i;
	int IconId;
	mString Id;
	char IsError = false;

	for(i=0; i<Profile->get_certs().size(); i++)
	{
		IsError = false;

		if((i+1) == Profile->get_certs().size())
			LastOne = true;
		else
			LastOne = false;

		switch(Profile->get_certs()[i].get_state())
		{
			case NEWPKI_PROFILE_CERT_STATE_REVOKED:
				if(LastOne)
					IconId = RevEndConnectorIconId;
				else
					IconId = RevMidConnectorIconId;
				break;

			case NEWPKI_PROFILE_CERT_STATE_SUSPENDED:
				if(LastOne)
					IconId = SuspEndConnectorIconId;
				else
					IconId = SuspMidConnectorIconId;
				break;

			case NEWPKI_PROFILE_CERT_STATE_ACTIVE:
				if(LastOne)
					IconId = ActEndConnectorIconId;
				else
					IconId = ActMidConnectorIconId;
				break;

			case NEWPKI_PROFILE_CERT_STATE_WAITING:
				if(LastOne)
					IconId = WaitEndConnectorIconId;
				else
					IconId = WaitMidConnectorIconId;
				break;

			case NEWPKI_PROFILE_CERT_STATE_ERROR:
				if(LastOne)
					IconId = ErrEndConnectorIconId;
				else
					IconId = ErrMidConnectorIconId;
				break;
			case NEWPKI_PROFILE_CERT_STATE_WAITING_REV:
				if(LastOne)
					IconId = WaitRevEndConnectorIconId;
				else
					IconId = WaitRevMidConnectorIconId;
				break;
			case NEWPKI_PROFILE_CERT_STATE_WAITING_SUSP:
			case NEWPKI_PROFILE_CERT_STATE_WAITING_UNSUSP:
				if(LastOne)
					IconId = WaitSuspEndConnectorIconId;
				else
					IconId = WaitSuspMidConnectorIconId;
				break;
			default:
				if(LastOne)
					IconId = ActEndConnectorIconId;
				else
					IconId = ActMidConnectorIconId;
				break;
		}

		if(!IsError)
		{
			if(Profile->get_certs()[i].get_error().size())
				IsError = true;
		}

		m_listCtrl->InsertItem(item + i + 1, Profile->get_certs()[i].get_caName().c_str(), IconId);
		m_listCtrl->SetItemData(item + i + 1, (long)&Profile->get_certs()[i]);
		Id.sprintf("%ld", Profile->get_certs()[i].get_id());
		m_listCtrl->SetItem(item + i + 1, 1, Id.c_str());
		if(IsError)
			m_listCtrl->SetItemTextColour(item + i + 1, *wxRED);

	}

	m_listCtrl->SetItemImage(item, MinusProfileIconId, MinusProfileIconId);
	listStatus[Profile->get_id()] = PROFILE_STATUS_OPENED;
}

void DlgRaAdmin::DrawCloseProfile(long item, NewpkiProfileDatas * Profile)
{
	int i;
	i = Profile->get_certs().size();
	while(i)
	{
		m_listCtrl->DeleteItem(item + 1);
		i--;
	}
	m_listCtrl->SetItemImage(item, PlusProfileIconId, PlusProfileIconId);
	listStatus[Profile->get_id()] = PROFILE_STATUS_CLOSED;
}



NewpkiProfileDatasCert * DlgRaAdmin::GetSelectedCert(long Item)
{
	wxListItem info;

	if(Item == -1)
	{
		Item = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
		if(Item == -1)
			return NULL;
	}
	
	info.m_mask = wxLIST_MASK_IMAGE ;
	info.m_itemId = Item;
	if(!m_listCtrl->GetItem(info))
		return NULL;

	if(!IS_CERT_ICON(info.m_image))
		return NULL;
	
	return (NewpkiProfileDatasCert *)m_listCtrl->GetItemData(Item);
}

NewpkiProfileDatas * DlgRaAdmin::GetSelectedProfile(long Item)
{
	wxListItem info;

	if(Item == -1)
	{
		Item = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
		if(Item == -1)
			return NULL;
	}
	
	info.m_mask = wxLIST_MASK_IMAGE ;
	info.m_itemId = Item;
	if(!m_listCtrl->GetItem(info))
		return NULL;

	if(!IS_PROFILE_ICON(info.m_image))
		return NULL;
	
	return (NewpkiProfileDatas *)m_listCtrl->GetItemData(Item);
}

void DlgRaAdmin::OnSavePKCS12(wxCommandEvent& event)
{
	NewpkiProfileDatasCert * Cert;

	Cert = GetSelectedCert();
	if(!Cert || !Cert->get_p12())
		return;

	SavePKCS12(Cert->get_p12());
}


void DlgRaAdmin::OnDeletePKCS12(wxCommandEvent &event)
{
	NewpkiProfileDatasCert * Cert;

	Cert = GetSelectedCert();
	if(!Cert)
		return;

	m_DlgMsg->wShow(_("Deleting PKCS#12..."));

	if(!m_ClientPki->DeletePkcs12(Cert->get_id()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgRaAdmin::OnImportCert(wxCommandEvent &event)
{
#ifdef _WIN32
	NewpkiProfileDatasCert * Cert;
	PkiEnroll Enroll;

	Cert = GetSelectedCert();
	if(!Cert || !Cert->get_cert())
		return;

	m_DlgMsg->wShow(_("Importing Certificate..."));
	if(!Enroll.ImportCert(Cert->get_cert().GetCertPEM().c_str()))
	{
		HandleErrorResult(NULL, this, Enroll.GetEnrollLastError());
		m_DlgMsg->wHide();
		return;
	}
	m_DlgMsg->wHide();

	DisplayMessage(this, _("Importation was successful"), wxOK);
#endif
}

void DlgRaAdmin::OnImportP12(wxCommandEvent &event)
{
#ifdef _WIN32
	NewpkiProfileDatasCert * Cert;

	Cert = GetSelectedCert();
	if(!Cert || !Cert->get_cert())
		return;

	DlgImportP12 dlg(this, Cert->get_p12());
#endif
}

void DlgRaAdmin::OnChangeProfileUID(wxCommandEvent &event)
{
	wxString uid;
	NewpkiProfileDatas * Profile;
	
	Profile = GetSelectedProfile();
	if(!Profile)
		return;

	uid = ::wxGetTextFromUser(_("Set Profile's UID"), _("Profile's UID"), Profile->get_profile().get_ldapUid().c_str(), this);
	if(!uid.Len())
		return;

	m_DlgMsg->wShow(_("Changing Profile's UID..."));

	if(!m_ClientPki->ChangeProfileUID(Profile->get_id(), uid.GetData()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgRaAdmin::OnRejectProfile(wxCommandEvent &event)
{
	wxString reason;
	NewpkiProfileDatas * Profile;
	
	Profile = GetSelectedProfile();
	if(!Profile)
		return;

	reason = ::wxGetTextFromUser(_("Reason:"), _("Get Rejection Reason"), "", this);
	if(!reason.Len())
		return;

	m_DlgMsg->wShow(_("Rejecting Profile..."));

	if(!m_ClientPki->RejectProfile(Profile->get_id(), reason.GetData()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgRaAdmin::OnAcceptProfile(wxCommandEvent &event)
{
	wxString uid;
	NewpkiProfileDatas * Profile;
	
	Profile = GetSelectedProfile();
	if(!Profile)
		return;

	m_DlgMsg->wShow(_("Accepting Profile..."));

	if(!m_ClientPki->AcceptProfile(Profile->get_id()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgRaAdmin::OnChangeProfileOwner(wxCommandEvent &event)
{
	size_t i;
	int index;
	NewpkiProfileDatas * Profile;
	wxArrayString choices;
	mVector<UsersGroup> groups;
	
	Profile = GetSelectedProfile();
	if(!Profile)
		return;

	m_DlgMsg->wShow(_("Loading Groups..."));
	if(!m_ClientPki->GetGroups(groups))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	for(i=0; i<groups.size(); i++)
	{
		choices.Add(groups[i].get_name().c_str());
	}

	index = ::wxGetSingleChoiceIndex(_("Set Profile's Owner"), _("Profile's Owner"), choices, this);
	if(index == -1)
		return;

	m_DlgMsg->wShow(_("Changing Profile's Owner..."));

	if(!m_ClientPki->ChangeProfileOwner(Profile->get_id(), groups[index].get_serial()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}


void DlgRaAdmin::OnDeleteRequest(wxCommandEvent &event)
{
	NewpkiProfileDatasCert * Cert;
	Cert = GetSelectedCert();
	if(!Cert || Cert->get_state() != NEWPKI_PROFILE_CERT_STATE_ERROR)
		return;

	if(DisplayMessage(this, _("Are you sure, you want to delete this request ?"), wxYES_NO) == wxNO) return;

	m_DlgMsg->wShow(_("Deleting request..."));
	if(!m_ClientPki->DeleteRequest(Cert->get_id()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();
	
	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}


void DlgRaAdmin::OnDeleteProfile(wxCommandEvent &event)
{
	NewpkiProfileDatas * Profile;
	Profile = GetSelectedProfile();
	if(!Profile)
		return;

	if(DisplayMessage(this, _("Are you sure, you want to delete this profile and all its associated certificates ?"), wxYES_NO) == wxNO) return;

	m_DlgMsg->wShow(_("Deleting profile..."));
	if(!m_ClientPki->DeleteProfile(Profile->get_id()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();
	
	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}


void DlgRaAdmin::OnChangeProfileDN(wxCommandEvent &event)
{
	NewpkiProfileDatas * Profile;
	Profile = GetSelectedProfile();
	if(!Profile)
		return;

	DlgPublishDN Dlg(this, m_EntityConf.get_body(), my_acl, m_ClientPki, Profile->get_profile().get_dn(), Profile->get_id());
	if(Dlg.IsOk())
	{
		RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
	}
}

void DlgRaAdmin::OnBatchImport(wxCommandEvent &event)
{
	DlgBashImport Dlg(this, m_EntityConf, m_ClientPki);
	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

