/*  Nomad:  nomad-id3.c
 *
 *  Copyright (C) 2002 David A Knight <david@ritter.demon.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <errno.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#endif

#include <glib/gconvert.h>
#include <glib/gunicode.h>

#include <glib/gtypes.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>
#include <sys/types.h>

#include "nomad-id3.h"
#include "nomad-genre.h"

#include "mp3file.h"

#include "nomad-id3-util.h"

struct NomadID3Details {
	gchar *uri;
	
	guint64 size;
	
	ID3V1 *v1tag;
	ID3V2 *v2tag;
};

static gchar *secs_to_mmss(guint seconds);
static guint mmss_to_secs( const gchar *mmss );


static gboolean nomad_string_is_empty( const gchar *field, guint field_size );
static gboolean nomad_id3_scan_id3v1( NomadID3 *id3 );
static gboolean nomad_id3_scan_id3v2( NomadID3 *id3 );

static void *nomad_id3_frame_get_data( const ID3V2Frame *frame, guint32 *length );
static ID3V2Frame *nomad_id3_get_frame( const NomadID3 *id3, 
					   const gchar *frame_name );

static gchar *nomad_id3_text_frame_get_string( const ID3V2Frame *frame,
						  gchar *encoding );
static guint32 nomad_id3_fill_text_frame( ID3V2Frame *frame,
					    const gchar *text,
					    guint32 length,
					    const gchar *fname );

static guint nomad_id3_to_v2_length( guint32 length );
static guint nomad_id3_from_v2_length( guint32 length );

static void nomad_id3_class_init( NomadID3Class *klass );
static void nomad_id3_init( NomadID3 *id3 );
static void nomad_id3_finalize( GObject *object );
static void nomad_id3_set_prop( GObject *object, guint prop_id, 
				   const GValue *value, GParamSpec *spec );
static void nomad_id3_get_prop( GObject *object, guint prop_id, 
				   GValue *value, GParamSpec *spec );

NomadID3 *nomad_id3_new( const gchar *uri )
{
	NomadID3 *id3;
	
	id3 = NOMAD_ID3( g_object_new( nomad_id3_get_type(), NULL ) );
	
	id3->details->uri = g_strdup( uri );
	
	if( ! nomad_id3_scan_id3v2( id3 ) ) {
		if( ! nomad_id3_scan_id3v1( id3 ) ) {
			/* no id3 tag */
			g_object_unref( G_OBJECT( id3 ) );
			id3 = NULL;
		}
	}

	return id3;
}

gchar *nomad_id3_get_filesize( NomadID3 *id3 )
{
	gchar *size;

	g_return_val_if_fail( NOMAD_IS_ID3( id3 ), NULL );

	size = g_strdup_printf( "%llu", (long long unsigned int)id3->details->size );

	return size;
}

gchar *nomad_id3_get_length( NomadID3 *id3 )
{
	ID3V2Frame *frame;
	gchar *length;
	
	g_return_val_if_fail( NOMAD_IS_ID3( id3 ), NULL );
	
	length = NULL;

	if( id3->details->v2tag ) {
		frame = nomad_id3_get_frame( id3, "TLEN" );
		
		if( frame ) {
			gchar *text;
			guint32 millisec;
			
			text = nomad_id3_text_frame_get_string( frame,
								   NULL );
			
			millisec = strtoul( text, NULL, 10 );
			millisec /= 1000;
			g_free( text );
			length = secs_to_mmss( millisec );
		}
	} else {
		frame = NULL;
	}

	if( ! frame ) {
		gchar *size;

		size = nomad_id3_get_filesize( id3 );

		length = length_from_file( id3->details->uri, size );

		g_free( size );
	}

	return length;
}


gchar *nomad_id3_get_title( NomadID3 *id3 )
{
	gchar *title;
	gchar *temp;
	g_return_val_if_fail( NOMAD_IS_ID3( id3 ), NULL );

	title = NULL;

	if( id3->details->v1tag && 
	    ! nomad_string_is_empty( id3->details->v1tag->title, ID3V1_TITLE_SIZE ) ) {

		title = g_strndup( id3->details->v1tag->title, 
				   ID3V1_TITLE_SIZE );
		temp = nomad_id3_util_charset_convert( title );
		if( temp ) {
			g_free( title );
			title = temp;
		}

		title = g_strstrip( title );
	} else if( id3->details->v2tag ) {
		ID3V2Frame *frame;

		frame = nomad_id3_get_frame( id3, "TIT2" );
		if( frame ) {
			title = nomad_id3_text_frame_get_string( frame,
								    NULL );
		}
	}

	return title;
}

gchar *nomad_id3_get_artist( NomadID3 *id3 )
{
	gchar *artist;
	gchar *temp;
	
	g_return_val_if_fail( NOMAD_IS_ID3( id3 ), NULL );

	artist = NULL;

	if( id3->details->v1tag && 
	    ! nomad_string_is_empty( id3->details->v1tag->artist,
					ID3V1_ARTIST_SIZE )  ) {
		artist = g_strndup( id3->details->v1tag->artist, 
				    ID3V1_ARTIST_SIZE);
		temp = nomad_id3_util_charset_convert( artist );
		if( temp ) {
			g_free( artist );
			artist = temp;
		}

		artist = g_strstrip( artist );
	} else if( id3->details->v2tag ) {
		ID3V2Frame *frame;

		frame = nomad_id3_get_frame( id3, "TPE1" );
		if( ! frame ) {
			frame = nomad_id3_get_frame( id3, "TPE2" );
		}
		if( ! frame ) {
			frame = nomad_id3_get_frame( id3, "TPE3" );
		}
		if( ! frame ) {
			frame = nomad_id3_get_frame( id3, "TPE4" );
		}
		if( frame ) {
			artist = nomad_id3_text_frame_get_string( frame,
								     NULL );
		}
	}

	return artist;
}

gchar *nomad_id3_get_album( NomadID3 *id3 )
{
	gchar *album;
	gchar *temp;
	
	g_return_val_if_fail( NOMAD_IS_ID3( id3 ), NULL );

	album = NULL;

	if( id3->details->v1tag && 
	    ! nomad_string_is_empty( id3->details->v1tag->album, ID3V1_ALBUM_SIZE ) ) {
		album = g_strndup( id3->details->v1tag->album, 
				   ID3V1_ALBUM_SIZE );
		temp = nomad_id3_util_charset_convert( album );
		if( temp ) {
			g_free( album );
			album = temp;
		}

		album = g_strstrip( album );
	} else if( id3->details->v2tag ) {
		ID3V2Frame *frame;

		frame = nomad_id3_get_frame( id3, "TALB" );
		if( frame ) {
			album = nomad_id3_text_frame_get_string( frame,
								    NULL );
		}
	}

	return album;
}

gchar *nomad_id3_get_year( NomadID3 *id3 )
{
	gchar *year;
	gchar *temp;

	g_return_val_if_fail( NOMAD_IS_ID3( id3 ), NULL );

	year = NULL;

	if( id3->details->v1tag && 
	    ! nomad_string_is_empty( id3->details->v1tag->year, ID3V1_YEAR_SIZE ) ) {
		year = g_strndup( id3->details->v1tag->year, ID3V1_YEAR_SIZE );
		temp = nomad_id3_util_charset_convert( year );
		if( temp ) {
			g_free( year );
			year = temp;
		}

		year = g_strstrip( year );
	} else if( id3->details->v2tag ) {
		ID3V2Frame *frame;

		frame = nomad_id3_get_frame( id3, "TYER" );

		if( ! frame ) {
			frame = nomad_id3_get_frame( id3, "TPRO" );
		}
		if( ! frame ) {
			frame = nomad_id3_get_frame( id3, "TCOP" );
		}
		if( frame ) {
			year = nomad_id3_text_frame_get_string( frame,
								   NULL ); 
		}
	}


	return year;
}

gchar *nomad_id3_get_comment( NomadID3 *id3 )
{
	gchar *comment;
	gchar *temp;
	
	g_return_val_if_fail( NOMAD_IS_ID3( id3 ), NULL );

	comment = NULL;

	if( id3->details->v1tag && 
	    ! nomad_string_is_empty( id3->details->v1tag->comment.comment,
					ID3V1_COMMENT_SIZE )  ) {
		comment = g_strndup( id3->details->v1tag->comment.comment,
				     ID3V1_COMMENT_SIZE );
		temp = nomad_id3_util_charset_convert( comment );
		if( temp ) {
			g_free( comment );
			comment = temp;
		}

		comment = g_strstrip( comment );
	}

	return comment;
}

gchar *nomad_id3_get_track( NomadID3 *id3 )
{
	gchar *track;

	g_return_val_if_fail( NOMAD_IS_ID3( id3 ), NULL );

	track = NULL;

	if( id3->details->v1tag ) {
	
		if( id3->details->v1tag->comment.V1dot1.empty == '\0' &&
		    id3->details->v1tag->comment.V1dot1.track != '\0' ) {
			track = g_strdup_printf("%i",
						id3->details->v1tag->comment.V1dot1.track);
		}
	} else if( id3->details->v2tag ) {
		ID3V2Frame *frame;

		frame = nomad_id3_get_frame( id3, "TRCK" );
		if( frame ) {
			track = nomad_id3_text_frame_get_string( frame,
								    NULL );
		}
	}

	return track;
}

gchar *nomad_id3_get_genre( NomadID3 *id3 )
{
	gchar *genre;

	g_return_val_if_fail( NOMAD_IS_ID3( id3 ), NULL );

	genre = NULL;

	if( id3->details->v1tag ) {
		const gchar *genre_string;

		guint i;

		i = id3->details->v1tag->genre;
		if( i <= genre_count ) {
			genre_string = nomad_id3_genre_table[ i ];
			
			if( *genre_string ) {
				genre = g_strdup( genre_string );
			}
		}
	} else if( id3->details->v2tag ) {
		ID3V2Frame *frame;
		
		frame = nomad_id3_get_frame( id3, "TCON" );
		if( frame ) {
			genre = nomad_id3_text_frame_get_string( frame,
								    NULL );
		}
	}
	
	
	return genre;
}


gboolean nomad_id3_tag_v1dot1( const gchar *uri,
			      const gchar *title,
			      const gchar *artist,
			      const gchar *album,
			      const gchar *year,
			      const gchar *track,
			      const gchar *genre )
{
	gboolean ok;
	
	/* tag built, now write it */
#ifndef HAVE_GNOME_VFS
	FILE *handle;
	
	handle = fopen( uri, "a+" );
	ok = ( handle != NULL );
	if( ok ) {
		ok = nomad_id3_tag_v1dot1_with_handle( handle,
				title, artist, album, year, track,
				genre );
		fclose( handle );
	}
#else
	GnomeVFSHandle *handle;
	GnomeVFSResult result;
	GnomeVFSFileInfo *info;
	
	info = NULL;
	handle = NULL;
	ok = FALSE;
	result = gnome_vfs_open( &handle, uri,
				 GNOME_VFS_OPEN_READ | 
				 GNOME_VFS_OPEN_RANDOM |
				 GNOME_VFS_OPEN_WRITE );
	if( result == GNOME_VFS_OK ) {
		info = gnome_vfs_file_info_new();

		result = gnome_vfs_get_file_info_from_handle( handle,
							      info,
							      GNOME_VFS_FILE_FLAGS_NONE );
	}
	if( result == GNOME_VFS_OK ) {
		result = gnome_vfs_seek( handle, GNOME_VFS_SEEK_START,
					 info->size );
	}
	if( result == GNOME_VFS_OK ) {
		/* write the tag */
		ok = nomad_id3_tag_v1dot1_with_handle( handle,
				title, artist, album, year, track,
				genre );
	}

	if( info ) {
		gnome_vfs_file_info_unref( info );
	}
	if( handle ) {
		gnome_vfs_close( handle );
	}
#endif

	return ok;
}

gboolean nomad_id3_tag_v1dot1_with_handle( gpointer handle,
		const gchar *title, const gchar *artist,
		const gchar *album, const gchar *year,
		const gchar *track, const gchar *genre )
{
	ID3V1 *tag;
	guint32 tracknum;
	gboolean ok;
	const gchar *comment = "libnomadid3 tagged";

#ifdef HAVE_GNOME_VFS
	GnomeVFSFileSize wrote;
	GnomeVFSResult result;
#endif
	
	tag = g_new0( ID3V1, 1 );
       
	if( title ) {
		strcpy( tag->title, title );
	}
	if( artist ) {
		strcpy( tag->artist, artist );
	}
	if( album ) {
		strcpy( tag->album, album );
	}
	if( year ) {
		strcpy( tag->year, year );
	}
	strcpy( tag->comment.comment, comment );

	if( track ) {
		tracknum = strtoul( track, NULL, 10 );
		if( tracknum > 255 ) {
			tracknum = 0;
		}
	} else {
		tracknum = 0;
	}
	tag->comment.V1dot1.track = tracknum;

	if( genre ) {
		guint index;

		for( index = 0; index < genre_count; ++ index ) {
			if( ! strcmp( genre, nomad_id3_genre_table[ index ] ) ) {
				tag->genre = index + 1;
			}
		}
		if( tag->genre == 0 ) {
			tag->genre = (gchar)255;
		}
	} else {
		tag->genre = (gchar)255;
	}

	/* tag built, now write it */
#ifndef HAVE_GNOME_VFS
	fwrite( "TAG", 1, 3, handle );
	fwrite( tag, 1, sizeof( ID3V1 ), handle );
	ok = TRUE;
#else
	ok = FALSE;
	/* write the tag */
	result = gnome_vfs_write( handle, "TAG", 3, &wrote );
	if( result == GNOME_VFS_OK ) {
		result = gnome_vfs_write( handle, tag,
					  sizeof( ID3V1 ), &wrote );
		ok = TRUE;
	}

#endif
	g_free( tag );

	return ok;
}

gboolean nomad_id3_tag_v2( const gchar *uri,
			  const gchar *title,
			  const gchar *artist,
			  const gchar *album,
			  const gchar *year,
			  const gchar *track,
			  const gchar *genre,
			  const gchar *length )
{
	gchar *tempuri;
	gboolean ok;	
#ifndef HAVE_GNOME_VFS
	FILE *handle;
	guint written;

	tempuri = nomad_id3_util_create_backup_filename( uri );
	handle = fopen( tempuri, "w+" );
	ok = ( handle != NULL );
	if( ok ) {
		ok = nomad_id3_tag_v2_with_handle( handle,
				title, artist, album, year, track,
				genre, length );
	}
#else
	GnomeVFSHandle *handle;
	GnomeVFSResult result;
	GnomeVFSFileSize written;

	ok = FALSE;
	tempuri = nomad_id3_util_create_backup_filename( uri );
	result = gnome_vfs_create( &handle, tempuri,
				   GNOME_VFS_OPEN_WRITE,
				   TRUE,
				   GNOME_VFS_PERM_USER_READ |
				   GNOME_VFS_PERM_USER_WRITE |
				   GNOME_VFS_PERM_GROUP_READ |
				   GNOME_VFS_PERM_OTHER_READ );
	if( result == GNOME_VFS_OK ) {
		ok = nomad_id3_tag_v2_with_handle( handle,
				title, artist, album, year, track,
				genre, length );
	}
#endif

	if( ok ) {
		if( nomad_id3_util_concat_file_to_handle( handle, uri,
							      0, &written ) ) {
#ifndef HAVE_GNOME_VFS
			fclose( handle );
			handle = fopen( uri, "w" );
			if( handle ) {
				nomad_id3_util_concat_file_to_handle( handle, tempuri, 0, &written );
				fclose( handle );
			}
#else
			gnome_vfs_close( handle );
			gnome_vfs_move( tempuri, uri, TRUE );
#endif
		}
	} else if( handle ) {
#ifndef HAVE_GNOME_VFS
		fclose( handle );
#else
		gnome_vfs_close( handle );
#endif
	}

	g_free( tempuri );
	return ok;
}

gboolean nomad_id3_tag_v2_with_handle( gpointer handle,
		const gchar *title, const gchar *artist,
		const gchar *album, const gchar *year,
		const gchar *track, const gchar *genre,
		const gchar *length )
{

	ID3V2Header *header;
	gint32 header_size;
	void *frames;
	ID3V2Frame *frame;
	guint size;
#ifndef HAVE_GNOME_VFS
#else
	GnomeVFSResult result;
	GnomeVFSFileSize wrote;
#endif
	gboolean ok;

	gint index;
	gchar *fullgenre;

	guint seconds;
	gchar *len;
	
	const gchar *lang = "en";
	const gchar *desc = "tagger";
	const gchar *comment = "libnomadid3 tagged";
	gchar *fullcomment;
	
	header = g_new0( ID3V2Header, 1 );

	strncpy( header->id, "ID3", 3 );

	header->version = 0x0003;
	header->flags = 0;

	/* calc header size */
	header_size = sizeof( ID3V2Frame ) * 7;
	/* +1 for encoding, +1 for \0 */

	fullcomment = g_new0( gchar, 3 +
				strlen( desc ) + 1 + 
				strlen( comment ) + 1 );
	strcpy( fullcomment, lang );
	strcpy( fullcomment + 3, desc );
	strcpy( fullcomment + 3 + strlen( desc ) + 1, comment );
	
	header_size += strlen( title ) + 2;
	header_size += strlen( artist ) + 2;
	header_size += strlen( album ) + 2;
	header_size += strlen( year ) + 2;
	header_size += strlen( track ) + 2;
	/* +3 for language, +2 for \0, +1 for encoding */
	header_size += strlen( desc ) + strlen( comment ) + 6;

	/* we need to convert genre to "genre number" NIL "genre name" NIL */
	for( index = 0; index < genre_count; ++ index ) {
		if( ! strcmp( genre, nomad_id3_genre_table[ index ] ) ) {
			break;
		}
	}
	if( index == genre_count ) {
		index = -1;
	}
	fullgenre = g_strdup_printf( "%i%c%s", index, '\0', genre );
	/* fullgenre strlen will stop at the embedded \0, so we +1 for that
	   and add on the length of genre, + it's \0 and of course the byte
	   for the encoding type */
	header_size += strlen( fullgenre ) + 1 + strlen( genre ) + 1;

	/* we need to convert length to milliseconds */
	seconds = mmss_to_secs( length );
	if( seconds != 0 ) {
		len = g_strdup_printf( "%u", seconds * 1000 );
		/* 1 for encoding, 1 for \0 */
		header_size += strlen( len ) + 2;
		/* add on the size of another frame, as by default we
		   only have 6 frames */
		header_size += sizeof( ID3V2Frame );
	} else {
		len = NULL;
	}

	frames = g_malloc0( header_size );

	header->size = nomad_id3_to_v2_length( header_size );

	/* fill in frames data */
	frame = (ID3V2Frame*)frames;
	size = 0;

	size += nomad_id3_fill_text_frame( frame, 
					      title, strlen( title ),
					      "TIT2" );
	frame = (ID3V2Frame*)( frames + size );

	size += nomad_id3_fill_text_frame( frame, 
					      artist, strlen( artist ),
					      "TPE1" );
	frame = (ID3V2Frame*)( frames + size );

	size += nomad_id3_fill_text_frame( frame, 
					      album, strlen( album ),
					      "TALB" );
	frame = (ID3V2Frame*)( frames + size );

	size += nomad_id3_fill_text_frame( frame, 
					      year, strlen( year ),
					      "TYER" );
	frame = (ID3V2Frame*)( frames + size );

	size += nomad_id3_fill_text_frame( frame, 
					      track, strlen( track ),
					      "TRCK" );

	frame = (ID3V2Frame*)( frames + size );

	size += nomad_id3_fill_text_frame( frame, 
					      fullgenre, 
					      strlen( fullgenre ) +
					      1 +
					      strlen( genre ),
					      "TCON" );
	g_free( fullgenre );

	frame = (ID3V2Frame*)( frames + size );

	size += nomad_id3_fill_text_frame( frame,
			fullcomment, 
			3 + strlen( desc ) + 1 + strlen( comment ), "COMM" );
	
	if( len ) {
		frame = (ID3V2Frame*)( frames + size );

		size += nomad_id3_fill_text_frame( frame, len, 
						      strlen( len ),
						      "TLEN" );

		g_free( len );
	}
	frame = (ID3V2Frame*)( frames + size );

	/* write header, followed by frames, followed by data from uri */
	ok = FALSE;

#ifndef HAVE_GNOME_VFS
	fwrite( header, 1, sizeof( ID3V2Header ), handle );
	fwrite( frames, 1, header_size, handle );
	ok = TRUE;
#else
	/* write the tag */
	result = gnome_vfs_write( handle, header,
				  sizeof( ID3V2Header ), &wrote );
	if( result == GNOME_VFS_OK ) {
		result = gnome_vfs_write( handle, frames,
					  header_size, &wrote );
		ok = TRUE;
	}
#endif
	g_free( frames );
	g_free( header );

	return ok;
}


void nomad_id3_strip_v1dot1( const gchar *uri )
{
#ifndef HAVE_GNOME_VFS
	FILE *handle;
	gboolean ok;
	gsize size;
	gsize offset;
	gchar buffer[ 3 ];

	offset = 0;
	handle = fopen( uri, "r" );
	ok = ( handle != NULL );

	if( ok ) {
		ok = ( fseek( handle, -128, SEEK_END ) == 0 );
	}
	if( ok ) {
		offset = ftell( handle );
		strcpy( buffer, "\0\0\0" );
		size = fread( buffer, 1, 3, handle );
		ok = ( size == 3 );
	}
	if( handle != NULL ) {
		fclose( handle );
	}
	if( ok && ! strncmp( buffer, "TAG", strlen( "TAG" ) ) ) {
		truncate( uri, offset );	
	}
#else
	GnomeVFSHandle *handle;
	GnomeVFSResult result;
	GnomeVFSFileSize size;

	result = gnome_vfs_open( &handle, uri, 
				 GNOME_VFS_OPEN_READ | 
				 GNOME_VFS_OPEN_RANDOM |
				 GNOME_VFS_OPEN_WRITE );

	size = 0;
	if( result == GNOME_VFS_OK ) {
		GnomeVFSFileInfo *info;

		info = gnome_vfs_file_info_new();
		
		result = gnome_vfs_get_file_info_from_handle( handle,
							      info,
							      GNOME_VFS_FILE_FLAGS_NONE );
		if( result == GNOME_VFS_OK ) {
			size = info->size;
		}
		gnome_vfs_file_info_unref( info );
	}
	if( result == GNOME_VFS_OK ) {
		result = gnome_vfs_seek( handle,
					 GNOME_VFS_SEEK_START,
					 size - 128 );
	}
	if( result == GNOME_VFS_OK ) {
		/* read 3 bytes, see if == 'TAG' */
		gchar buffer[ 3 ];
		GnomeVFSFileSize got;

		strcpy( buffer, "\0\0\0" );
		result = gnome_vfs_read( handle, buffer, 3, &got );
		if( result == GNOME_VFS_OK &&
		    ! strncmp( buffer, "TAG", strlen( "TAG" ) ) ) {
			/* we got and ID3V1 tag */
			gnome_vfs_truncate_handle( handle,
						   size - 128 );
		}
	}
	
	if( handle ) {
		gnome_vfs_close( handle );
	}
#endif		
}

void nomad_id3_strip_v2( const gchar *uri )
{
	ID3V2 *tag;

#ifndef HAVE_GNOME_VFS
	FILE *handle;
	gboolean ok;
	gchar buffer[ ID3V2HeaderSize ];
	gsize got;
	gchar *tempuri;
	
	tag = NULL;
	handle = fopen( uri, "r+" );
	ok = ( handle != NULL );

	if( ok ) {
		/* read possible header */
		got = fread( buffer, 1, ID3V2HeaderSize, handle );
		ok = ( got == ID3V2HeaderSize );
	}
	if( ok && ! strncmp( buffer, "ID3", strlen( "ID3" ) ) ) {
		/* we got and ID3V2 tag */
		tag = g_new0( ID3V2, 1 );

		memcpy( tag, buffer, ID3V2HeaderSize );
	
		tag->header.size = 
			nomad_id3_from_v2_length( tag->header.size);

		tag->hasExtended = ( tag->header.flags & 64 );

		tag->frames = g_new0( gchar, tag->header.size );
		/* read frames */

		got = fread( tag->frames, 1, tag->header.size, handle );
		ok = ( got == tag->header.size );
	}
	if( ok ) {
		/* got an id3 v2 tag */
		tempuri = nomad_id3_util_create_backup_filename( uri );
		ok = nomad_id3_util_concat_handle_to_file( handle,
						tempuri, 0, &got );
		fclose( handle );
		handle = NULL;
		if( ok ) {
			gchar *temp;
			gchar *temp2;
			
			temp = g_strconcat( "\"", uri, "\"", NULL );
			remove( temp );
			temp2 = g_strconcat( "\"", tempuri, "\"", NULL );
			rename( temp2, temp );
			
			g_free( temp );
			g_free( temp2 );
		}
		g_free( tempuri );
	}
	
	if( handle ) {
		fclose( handle );
	}
#else
	GnomeVFSHandle *handle;
	GnomeVFSResult result;
	
	tag = NULL;

	result = gnome_vfs_open( &handle, uri, 
				 GNOME_VFS_OPEN_READ |
				 GNOME_VFS_OPEN_RANDOM |
				 GNOME_VFS_OPEN_WRITE );

	if( result == GNOME_VFS_OK ) {
		/* read possible header */
		gchar buffer[ ID3V2HeaderSize ];
		GnomeVFSFileSize got;

		result = gnome_vfs_read( handle, buffer, ID3V2HeaderSize, &got );
		if( result == GNOME_VFS_OK &&
		    ! strncmp( buffer, "ID3", strlen( "ID3" ) ) ) {
			/* we got and ID3V2 tag */
			tag = g_new0( ID3V2, 1 );

			memcpy( tag, buffer, ID3V2HeaderSize );
	
			tag->header.size = 
				nomad_id3_from_v2_length( tag->header.size);

			tag->hasExtended = ( tag->header.flags & 64 );

			tag->frames = g_new0( gchar, tag->header.size );
			/* read frames */
			result = gnome_vfs_read( handle,
						 tag->frames,
						 tag->header.size,
						 &got );
			if( result == GNOME_VFS_OK ) {
				/* got an id3 v2 tag */
				gchar *tempuri;
				GnomeVFSFileSize written;

				tempuri = nomad_id3_util_create_backup_filename( uri );
				nomad_id3_util_concat_handle_to_file( handle,
						tempuri, 0, &written );
				gnome_vfs_close( handle );
				handle = NULL;
				gnome_vfs_move( tempuri, uri, TRUE );
				g_free( tempuri );
			}
		}
	}

	if( handle ) {
		gnome_vfs_close( handle );
	}
#endif	
	if( tag ) {
		if( tag->frames ) {
			g_free( tag->frames );
		}
		g_free( tag );
	}

}

gint nomad_id3_get_length_from_header( ID3V2Header *header )
{
	return nomad_id3_from_v2_length( header->size );
}

/* static stuff */

/* Converts a figure representing a number of seconds to
 * a string in mm:ss notation */
static gchar *secs_to_mmss(guint seconds)
{ 
	gchar tmp2[4];
	gchar tmp[10];
	
	if (!seconds)
		return g_strdup("0:00");
	sprintf(tmp2, "0%u", seconds%60);
	while (strlen(tmp2)>2) {
		tmp2[0]=tmp2[1];
		tmp2[1]=tmp2[2];
		tmp2[2]='\0';
	}
	sprintf(tmp, "%lu:%s", (unsigned long)seconds/60, tmp2);
	return g_strdup(tmp);
}

static guint mmss_to_secs( const gchar *mmss )
{
	gchar **tmp;
	guint seconds = 0;
	
	if( mmss ) {
		tmp = g_strsplit(mmss, ":", 0);
		if( tmp[ 0 ] && tmp[ 1 ] && ! tmp[ 2 ] ) {
			gchar *dummy;
			
			seconds = 60 * strtoul( tmp[0], &dummy, 10 );
			seconds += strtoul( tmp[1], &dummy, 10 );
		}
		if (tmp != NULL) {
			g_strfreev(tmp);
		}
	}

	return seconds;
}


/* mp3info fills fields with spaces (at least in interactive mode),
   this sucks and is against the spec which states fields should be
   NIL padded */
static gboolean nomad_string_is_empty( const gchar *field, 
					  guint field_size )
{
	gboolean empty = FALSE;
	int i = 0;

	while( i < field_size && field[ i ] != '\0' && 
	       g_ascii_isspace( field[ i ] ) ) {
		i ++;
	}

	if( i == field_size || ( i = 0 && field[ i ] != '\0' ) ) {
		empty = TRUE;
	}

	return empty;
}

static gboolean nomad_id3_scan_id3v1( NomadID3 *id3 )
{
#ifndef HAVE_GNOME_VFS
	const gchar *uri;
	FILE *handle;
	gboolean ok;
	gsize size;
	gchar buffer[ 4 ];

	g_return_val_if_fail( NOMAD_IS_ID3( id3 ), FALSE );
	g_return_val_if_fail( ! id3->details->v2tag, TRUE );

	uri = id3->details->uri;
	id3->details->v1tag = NULL;
	
	handle = fopen( uri, "r+" );
	ok = ( handle != NULL );

	if( ok ) {
		ok = ( fseek( handle, -128, SEEK_END ) == 0 );
	}
	if( ok ) {
		id3->details->size = ftell( handle ) + 128;
		memset( buffer, 0, 4 ); 
		size = fread( buffer, 1, 3, handle );
		ok = ( size == 3 );
	}
	if( ok && ! strncmp( buffer, "TAG", strlen( "TAG" ) ) ) {
		/* found */
		id3->details->v1tag = g_new0( ID3V1, 1 );

		size = fread( id3->details->v1tag, 1, 125, handle );
		ok = ( size == 125 );
		if( ! ok ) {
			g_free( id3->details->v1tag );
			id3->details->v1tag = NULL;
		}
	}
	if( handle != NULL ) {
		fclose( handle );
	}

	return ok;
#else
	const gchar *uri;
	gboolean found;
	GnomeVFSHandle *handle;
	GnomeVFSResult result;
	GnomeVFSFileInfo *info;

	g_return_val_if_fail( NOMAD_IS_ID3( id3 ), FALSE );
	g_return_val_if_fail( ! id3->details->v2tag, TRUE );

	uri = id3->details->uri;
	id3->details->v1tag = NULL;

	found = FALSE;
	info = NULL;
	handle = NULL;

	result = gnome_vfs_open( &handle, uri, 
				 GNOME_VFS_OPEN_READ | GNOME_VFS_OPEN_RANDOM );
	if( result == GNOME_VFS_OK ) {
		info = gnome_vfs_file_info_new();

		result = gnome_vfs_get_file_info_from_handle( handle,
							      info,
							      GNOME_VFS_FILE_FLAGS_NONE );
	} 
	if( result == GNOME_VFS_OK ) {
		id3->details->size = info->size;
		result = gnome_vfs_seek( handle,
					 GNOME_VFS_SEEK_START,
					 info->size - 128 );
	}
	if( result == GNOME_VFS_OK ) {
		/* read 3 bytes, see if == 'TAG' */
		gchar buffer[ 3 ];
		GnomeVFSFileSize got;

		strcpy( buffer, "\0\0\0" );
		result = gnome_vfs_read( handle, buffer, 3, &got );
		if( result == GNOME_VFS_OK &&
		    ! strncmp( buffer, "TAG", strlen( "TAG" ) ) ) {
			/* we got and ID3V1 tag */
			id3->details->v1tag = g_new0( ID3V1, 1 );

			result = gnome_vfs_read( handle, id3->details->v1tag,
						 125, &got );

		}
		if( result == GNOME_VFS_OK ) {
			found = TRUE;
		} else if( id3->details->v1tag ) {
			g_free( id3->details->v1tag );
			id3->details->v1tag = NULL;
		}
	}
	
	if( info ) {
		gnome_vfs_file_info_unref( info );
	}
	
	if( handle ) {
		gnome_vfs_close( handle );
	}
		
	return found;
#endif
}

static gboolean nomad_id3_scan_id3v2( NomadID3 *id3 )
{
#ifndef HAVE_GNOME_VFS
	const gchar *uri;
	FILE *handle;
	gboolean ok;
	gchar buffer[ ID3V2HeaderSize ];
	gsize got;
	gchar *tempuri;
	ID3V2 *tag;

	uri = id3->details->uri;
	
	tag = NULL;
	handle = fopen( uri, "r+" );
	ok = ( handle != NULL );

	if( ok ) {
		/* read possible header */
		got = fread( buffer, 1, ID3V2HeaderSize, handle );
		ok = ( got == ID3V2HeaderSize );
	}
	if( ok ) {
		ok = ( ! strncmp( buffer, "ID3", strlen( "ID3" ) ) );
	}
	if( ok ) {
		/* we got and ID3V2 tag */
		tag = id3->details->v2tag = g_new0( ID3V2, 1 );

		memcpy( tag, buffer, ID3V2HeaderSize );
	
		tag->header.size = 
			nomad_id3_from_v2_length( tag->header.size);

		tag->hasExtended = ( tag->header.flags & 64 );

		tag->frames = g_new0( gchar, tag->header.size );
		/* read frames */

		got = fread( tag->frames, 1, tag->header.size, handle );
		ok = ( got == tag->header.size );
	}
	if( ok ) {
		ok = ( fseek( handle, 0, SEEK_END ) == 0 );
		if( ok ) {
			id3->details->size = ftell( handle );
		}
	}
	if( tag && ! ok ) {
		if( id3->details->v2tag->frames ) {
			g_free( id3->details->v2tag->frames );
		}
		g_free( id3->details->v2tag );
		id3->details->v2tag = NULL;
		tag = NULL;
	}
	return ( tag != NULL );
#else
	const gchar *uri;
	gboolean found;
	GnomeVFSHandle *handle;
	GnomeVFSResult result;

	g_return_val_if_fail( NOMAD_IS_ID3( id3 ), FALSE );

	uri = id3->details->uri;

	found = FALSE;

	result = gnome_vfs_open( &handle, uri, 
				 GNOME_VFS_OPEN_READ | GNOME_VFS_OPEN_RANDOM );

	if( result == GNOME_VFS_OK ) {
		GnomeVFSFileInfo *info;

		info = gnome_vfs_file_info_new();

		result = gnome_vfs_get_file_info_from_handle( handle,
							      info,
							      GNOME_VFS_FILE_FLAGS_NONE );
		if( result == GNOME_VFS_OK ) {
			id3->details->size = info->size;
		}
		gnome_vfs_file_info_unref( info );
	}

	if( result == GNOME_VFS_OK ) {
		/* read possible header */
		gchar buffer[ ID3V2HeaderSize ];
		GnomeVFSFileSize got;

		result = gnome_vfs_read( handle, buffer, ID3V2HeaderSize, &got );
		if( result == GNOME_VFS_OK &&
		    ! strncmp( buffer, "ID3", strlen( "ID3" ) ) ) {
			/* we got and ID3V2 tag */
			ID3V2 *tag;

			tag = id3->details->v2tag = g_new0( ID3V2, 1 );

			memcpy( tag, buffer, ID3V2HeaderSize );

			/* FIXME: is right? */
			tag->header.flags = tag->header.flags;		
			tag->header.size = 
				nomad_id3_from_v2_length(tag->header.size);

			tag->hasExtended = ( tag->header.flags & 64 );

			tag->frames = g_new0( gchar, tag->header.size );
			/* read frames */
			result = gnome_vfs_read( handle,
						 tag->frames,
						 tag->header.size,
						 &got );
			if( result == GNOME_VFS_OK ) {
				found = TRUE;
			}

		}
		if( ! found && id3->details->v2tag ) {
			if( id3->details->v2tag->frames ) {
				g_free( id3->details->v2tag->frames );
			}
			g_free( id3->details->v2tag );
			id3->details->v2tag = NULL;
		}
	} 

	if( handle ) {
		gnome_vfs_close( handle );
	}

	return found;
#endif
}

static void *nomad_id3_frame_get_data( const ID3V2Frame *frame, guint32 *length )
{
	gint16 flags;		
	void *ret;
	guint32 dsize;

	flags = g_ntohs( frame->flags );

	ret = (void*)frame;
	ret += sizeof( ID3V2Frame );

	dsize = nomad_id3_from_v2_length( frame->size );

	if( flags & 16384 ) {
		/* a - tag alter preservation */
	}
	if( flags & 8192 ) {
		/* b - file alter preservation */
	}
	if( flags & 4096 ) {
		/* c - read only */
	}
	if( flags & 128 ) {
		/* h - grouping id */
		ret = ret + 1;
	}
	if( flags & 8 ) {
		/* k - compression */
	}
	if( flags & 4 ) {
		/* m - encryption */
		ret = ret + 1;
	}
	if( flags & 2 ) {
		/* n - unsynchronisation */
	}
	if( flags & 1 ) {
		/* p - frame data length indicator -
		   the actual length of the data in the frame,
		   before decompression etc */
		dsize = nomad_id3_from_v2_length( (*(long*)ret) );
		ret += 4;
	}

	if( length ) {
		*length = dsize;
	}

	return ret;
}

static ID3V2Frame *nomad_id3_get_frame( const NomadID3 *id3, 
					   const gchar *frame_name )
{
	ID3V2Frame *frame;
	void *start;
	void *end;
	gint32 size;
	
	start = id3->details->v2tag->frames;
	size = id3->details->v2tag->header.size;
	if( id3->details->v2tag->hasExtended ) {
		size -= *(long*)start;
		start += *(long*)start;
	}
	
	/* %0abc0000 %0h00kmnp */

	frame = NULL;
	end = start + size;
	/* start points to the start of our frames */

	while( start < end ) {
		gint32 real_size;

		frame = (ID3V2Frame*)start;

		if( ! strncmp( frame->id, frame_name, 4 ) ) {
			/* this is the frame we want */
			break;
		} else if( frame->id[ 0 ] == '\0' ) {
			frame = NULL;
			break;
		} else {
			void *data;

			data = nomad_id3_frame_get_data( frame, 
							    &real_size );
			start = data + real_size;
			frame = NULL;
		}
	}

	return frame;
}

static gchar *nomad_id3_text_frame_get_string( const ID3V2Frame *frame,
						  gchar *encoding )
{
	ID3V2TextFrame *text;
	guint32 length;
	gchar *ret;
	const gchar *enc;
	gboolean utf8;

	glong tenc;
	
	length = 0;
	text = (ID3V2TextFrame*) nomad_id3_frame_get_data( frame, 
			&length );
	
	if( encoding ) {
		*encoding = text->encoding;
	}
	
	utf8 = FALSE;
	tenc = text->encoding;
	switch( tenc ) {
		case 1:
			enc = "UTF-16";
			break;
		case 2:
			enc = "UTF-16BE";
			break;
		case 3:
			enc = "UTF-8";
			utf8 = TRUE;
			break;
		case 0:
			enc = "ISO-8859-1";
			break;
		default:
			/* handle wrong byte order, neutrino
			   got this wrong */
			tenc = g_ntohl( tenc );
			switch( tenc ) {
				case 1:
					enc = "UTF-16";
					break;
				case 2:
					enc = "UTF-16BE";
					break;
				case 3:
					enc = "UTF-8";
					utf8 = TRUE;
					break;
				default:
					enc = "ISO-8859-1";
					break;
			}
			break;
	}
	
	if( length > 1 ) {
		ret = g_new0( gchar, length + 1 );
		memcpy( ret, text->string, length - 1 );
		
		if( ! utf8 ) {
			gchar *temp;

			temp = g_convert( ret, length - 1, "UTF-8",
					  enc, NULL, NULL, NULL );
			if( temp ) {
				g_free( ret );
				ret = temp;
			}
		}
	} else {
		ret = NULL;
	}

	return ret;
}

static guint32 nomad_id3_fill_text_frame( ID3V2Frame *frame,
					    const gchar *text,
					    guint32 length,
					    const gchar *fname )
{
	ID3V2TextFrame *text_frame;

	text_frame = (ID3V2TextFrame*)((void*)frame) + sizeof( ID3V2Frame );

	/* no byte swapping needed, this is a char value */
	text_frame->encoding = 3;

	/* memcpy so we can deal with embeded \0 */
	memcpy( text_frame->string, text, length );

	strncpy( frame->id, fname, 4 );
	/* +1 for encoding, do not +1 for \0 as we shouldn't nil term */
	frame->size = nomad_id3_to_v2_length( length + 1 );

	return ( length + 1 + sizeof( ID3V2Frame ) );
}

static guint nomad_id3_to_v2_length( guint32 length )
{
	if( length & 0x80 ) {
		length += 0x100;
		length ^= 0x80;
	}
	if( length & 0x8000 ) {
		length += 0x1000;
		length ^= 0x8000;
	}
	if( length & 0x800000 ) {
		length += 0x1000000;
		length ^= 0x800000;
	}
	if( length & 0x80000000 ) {
		length ^= 0x80000000;
	}

	return g_htonl( length );
}

static guint nomad_id3_from_v2_length( guint32 length )
{
	length = g_ntohl( length );
	
	length = ( (length & 0xFF000000 ) >> 1 ) +
		( (length & 0xFF0000 ) >> 1 ) +
		( (length & 0xFF00) >> 1 ) + (length & 0xFF);

	return length;
}


/* G Object stuff */

#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void
nomad_id3_class_init( NomadID3Class *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS( klass );

	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = nomad_id3_finalize;
	object_class->get_property = nomad_id3_get_prop;
	object_class->set_property = nomad_id3_set_prop;
}

static void
nomad_id3_init( NomadID3 *id3 )
{
	id3->details = g_new0( NomadID3Details, 1 );
}

static void
nomad_id3_set_prop( GObject *object, guint prop_id, 
			 const GValue *value, GParamSpec *spec )
{
	NomadID3 *id3;

	id3 = NOMAD_ID3( object );

	switch( prop_id ) {
	default:
		break;
	}
}

static void
nomad_id3_get_prop( GObject *object, guint prop_id, 
			 GValue *value, GParamSpec *spec )
{
	NomadID3 *id3;

	id3 = NOMAD_ID3( object );

	switch( prop_id ) {
	default:
		break;
	}
}

static void
nomad_id3_finalize( GObject *object )
{
	NomadID3 *id3;

	id3 = NOMAD_ID3( object );

	g_free( id3->details->uri );

	if( id3->details->v1tag ) {
		g_free( id3->details->v1tag );
	}

	if( id3->details->v2tag ) {
		g_free( id3->details->v2tag );
	}

	g_free( id3->details );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}

GType nomad_id3_get_type()
{
	static GType type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( NomadID3Class ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)nomad_id3_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( NomadID3 ),
			0, /* n_preallocs */
			(GInstanceInitFunc)nomad_id3_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "NomadID3",
					       &info, 0 );
	}

	return type;
}
