/*  Neutrino:  neutrino-mp3.c
 *
 *  Copyright (C) 2002 David A Knight <david@ritter.demon.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#ifndef HAVE_GNOME_VFS
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#else
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#endif

#include <libnjb.h>

#include "neutrino-mp3.h"

#include "neutrino-id3.h"

#include "neutrino-filename.h"

#include "neutrino-util.h"
#include "nomad-util.h"

NomadTrack *neutrino_mp3_track( const gchar *uri )
{
	NeutrinoID3 *id3;
	NomadTrack *track;

	gchar *artist;
	gchar *title;
	gchar *album;
	gchar *genre;
	gchar *length;
	gchar *year;
	gchar *size;
	gchar *tracknum;
	gchar *name;
	
	id3 = neutrino_id3_new( uri );

	artist = title = album = genre = length = 
		year = size = tracknum = NULL;

	if( id3 ) {
		artist = neutrino_id3_get_artist( id3 );
		title = neutrino_id3_get_title( id3 );
		album = neutrino_id3_get_album( id3 );
		genre = neutrino_id3_get_genre( id3 );
		length = neutrino_id3_get_length( id3 );
		year = neutrino_id3_get_year( id3 );
		size = neutrino_id3_get_filesize( id3 );
		tracknum = neutrino_id3_get_track( id3 );
	} else {
#ifndef HAVE_GNOME_VFS
		struct stat s;

		if( stat( uri, &s ) == 0 ) {
			size = g_strdup_printf( "%lu", s.st_size );
		}
#else
		GnomeVFSFileInfo *info;

		info = gnome_vfs_file_info_new();
		if( gnome_vfs_get_file_info( uri, info, 0 ) == GNOME_VFS_OK ) {
			size = g_strdup_printf( "%lu", info->size );
		}
		gnome_vfs_file_info_unref( info );
#endif
		neutrino_get_info_from_uri( uri,
					    &artist, &album, &title );
	}

	name = NULL;
#ifdef HAVE_GNOME_VFS
	name = gnome_vfs_get_local_path_from_uri( uri );
#endif
	if( ! name ) {
		name = g_strdup( uri );
	}

	track = nomad_track_new( artist, title, album, genre,
				    length, year, size, NJB_CODEC_MP3,
				    tracknum, name );

	if( artist ) {
		g_free( artist );
	}
	if( title ) {
		g_free( title );
	}
	if( album ) {
		g_free( album );
	}
	if( genre ) {
		g_free( genre );
	}

	if( length ) {
		g_free( length );
	}
	if( year ) {
		g_free( year );
	} 
	if( size ) {
		g_free( size );
	}
	if( tracknum ) {
		g_free( tracknum );
	} 
	if( name ) {
		g_free( name );
	}

	if( id3 ) {
		g_object_unref( id3 );
	}
	
	return track;
}

void neutrino_mp3_tag_file( const gchar *uri, NomadTrack *track,
			    gboolean v1dot1 )
{
	gchar *title;
	gchar *artist;
	gchar *album;
	gchar *trck;
	gchar *genre;
	gchar *year;
	gchar *length;

	neutrino_id3_strip_v1dot1( uri );
	neutrino_id3_strip_v2( uri );

	nomad_track_get( track, 
			    &artist, &title, &album,
			    &genre, &length, &year, NULL, NULL, &trck,
			    NULL, NULL );

	if( v1dot1 ) {
		neutrino_id3_tag_v1dot1( uri,
					 title, artist, album, year,
					 trck, genre );
	} else {
		neutrino_id3_tag_v2( uri,
				     title, artist, album, year,
				     trck, genre, length );
	}

	g_free( title );
	g_free( artist );
	g_free( album );
	g_free( trck );
	g_free( genre );
	g_free( length );
	g_free( year );
}

GList *neutrino_mp3_read_mpu( const gchar *uri, GList *list )
{
	GString *string;
	gchar **uris;
	gint i;

	string = neutrino_util_load_file( uri );

	uris = g_strsplit( string->str, "\n", -1 );
	for( i = 0; uris[ i ]; ++ i ) {
		GList *newitem;

		newitem = g_list_prepend( NULL, g_strdup( uris[ i ] ) );
		if( list ) {
			newitem->next = list->next;
			if( list->next ) {
				list->next->prev = newitem;
			}
			newitem->prev = list;
		}
		list = newitem;
	}

	return list;
}
