#!/usr/bin/perl
#
# See LICENSE for copyright information
#
# check_all_disks.pl <host> <ignmntpts|none> [warn] [crit] [port]
#
# Checks all disks on a host running netsaint_statd (version 2.01+) and warns
# if any are above [warn] percentage full and goes critical if higher than
# [crit] percentage full.  Some of the code came from Charlie Cook.
# ignmntpts is a comma delimited list of mount points not to look at. (or use
# none if there are none)
#

require 5.003;
BEGIN { $ENV{PATH} = '/bin' }
use Socket;
use POSIX;
use strict;

my $TIMEOUT = 15;

my %ERRORS = ('UNKNOWN', '-1',
		'OK', '0',
		'WARNING', '1',
		'CRITICAL', '2');

my $remote = shift || &usage(%ERRORS);
my $ignoreddisks = shift || &usage(%ERRORS);
my $warn = shift || 85;
my $critical = shift || 95;
my $port = shift || 1040;

my $remoteaddr = inet_aton("$remote");
my $paddr = sockaddr_in($port, $remoteaddr) || die "Can't create info for connection: #!\n";;
my $proto = getprotobyname('tcp');
socket(Server, PF_INET, SOCK_STREAM, $proto) || die "Can't create socket: $!";
setsockopt(Server, SOL_SOCKET, SO_REUSEADDR, 1);
connect(Server, $paddr) || die "Can't connect to server: $!";

my $state = "OK";

# Just in case of problems, let's not hang NetSaint
$SIG{'ALRM'} = sub { 
     close(Server);
     select(STDOUT);
     print "No Answer from Client\n";
     exit $ERRORS{"UNKNOWN"};
};
alarm($TIMEOUT);

select(Server);
$| = 1;

print Server "alldisks\n";

my $servanswer = <Server>;
alarm(0);
close(Server);
select(STDOUT);
chomp($servanswer);

#
# Setting up the ignored list and the server returned list
#

my @ignoreddisks = split(/,/,$ignoreddisks);
$servanswer =~ s/\(//g;
my @servanswer = split(/\)/,$servanswer);

#
# Removing the mount points that you said to ignore...
#

prune_disk_list() if $ignoreddisks ne "none";

#
# Lets get the ball rolling... the disk_warning sub and lets
# see how full our disks are... 
#

my $state = disk_warnings();
exit $ERRORS{$state};


sub prune_disk_list
  {
#
# prune_disk_list removes any entries that the server sends back that are in
# the ignored disk list
#
OUT:for (my $i=0;$i<=$#ignoreddisks;$i++)
    {
    for (my $j=0;$j<=$#servanswer;$j++)
      {
      if ($servanswer[$j] =~ /$ignoreddisks[$i],\d+/)
        {
        splice(@servanswer,$j,1);
        $j--;
        next OUT unless ($ignoreddisks[$i] =~ /.*[*].*/);
        }
      }
    }
  }

sub disk_warnings
  {
  my $state = "OK";
  my $critical_answer;
  my $warning_answer;
  for (my $i=0;$i<=$#servanswer;$i++) # Seeing which are warning
    {                                 # and which are critical
    split(/\,/,$servanswer[$i]);
    if ($_[1] >= $critical)
      {
      $critical_answer .= " $_[0]\($_[1]\%\)";
      $state = "CRITICAL" if ($state eq "OK" or $state eq "WARNING");
      next;
      }
    elsif ($_[1] >= $warn)
      {
      $warning_answer .= " $_[0]\($_[1]\%\)";
      $state = "WARNING" if ($state eq "OK");
      }
    }
  if ($critical_answer ne "")
    {
    print "Critical:$critical_answer\n" if $critical_answer ne "";
    }
  if ($warning_answer ne "")
    {
    print "<BR>" if $critical_answer ne "";
    print "Warning:$warning_answer\n" if $warning_answer ne "";
    }
  if (($warning_answer eq "") and ($critical_answer eq ""))
    {
    print "All disks below warning/critical thresholds\n";
    }
  return($state);
  }

sub usage
  {
#
# usage is called if you pass the check_all_disks.pl any parameters
# that aren't recognized.  (Don't do this!  :^D )
#
  print "Minimum arguments not supplied!\n\n";
  print "Perl Disk Checking Plugin for NetSaint (statd)\n";
  print "Copyright (c) 2000 Nick Reinking\n\n";
  print "Usage: $0 <host> <ignmntpts|none> [warn] [crit] [port]\n\n";
  print "[ignmntpts] = Comma delimited list of mount points to ignore (like /cdrom)\n";
  print "[warn] = Percentage to warn at (default 85%)\n";
  print "[crit] = Percentage to go critical at (default 95%)\n";
  exit $ERRORS{"UNKNOWN"};
  }
