/*
 * Copyright (c) 2004, 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.IO;
using Nemerle.Utility;

using Nemerle.Compiler;
using Nemerle.Compiler.Typedtree;
using Nemerle.Macros;

using PT = Nemerle.Compiler.Parsetree;

namespace Nemerle.Core {

module Operators
{
  internal cache_indices (expr : PT.PExpr, idxes : list [PT.PExpr]) 
                       : PT.PExpr * list [PT.PExpr]
  {
    def idx_exprs = List.Map (idxes, fun (_) { 
      <[ $(Macros.NewSymbol () : name) ]> 
    });
    (<[ def (.. $(<[ tmp ]> :: idx_exprs)) = (.. $(expr :: idxes)) ]>,
     idx_exprs)
  }

  [Hygienic]
  internal cache_assign_expr (e : PT.PExpr)
  : PT.PExpr * PT.PExpr
  {
    match (e) {
      | <[ $obj.$mem ]> when !Macros.IsTypeName (obj) =>
        (<[ _N_ref_cache (tmp, $obj) ]>, <[ tmp.$mem ]>)
        
      | <[ $obj . $mem [.. $idxes] ]> when !Macros.IsTypeName (obj) =>
        def (cache, idx_exprs) = cache_indices (obj, idxes);
        (cache, <[ tmp . $mem [ .. $idx_exprs ] ]>)
        
      | <[ $tab [.. $idxes] ]> =>
        def (cache, idx_exprs) = cache_indices (tab, idxes);
        (cache, <[ tmp [ .. $idx_exprs ] ]>)
        
      | _ => (<[ () ]>, e)
    }
  }
}


  macro @&& (e1, e2) 
  {
    <[ 
      match ($e1) {
        | false => false
        | _ => $e2
      }
    ]>
  }

  macro @|| (e1, e2) 
  {
    <[ 
      match ($e1) {
        | true => true
        | _ => $e2
      }
    ]>
  }

  macro @%|| (e1, e2)
  {
    <[ (($e1 | $e2) :> int) != 0 ]>                                    
  }

  macro @%&& (e1, e2)
  {
    <[ (($e1 & $e2) :> int) != 0 ]>                                    
  }

  macro @%^^ (e1, e2)
  {
    <[ (($e1 ^ $e2) :> int) != 0 ]>                                    
  }

  macro @++ (e) {
    def (cached, safe) = Operators.cache_assign_expr (e);
    <[ $cached; $safe = _N_op_Increment ($safe) ]>
  }

  macro @-- (e) {
    def (cached, safe) = Operators.cache_assign_expr (e);
    <[ $cached; $safe = _N_op_Decrement ($safe) ]>
  }

  macro @+= (e, val) {
    def (cached, safe) = Operators.cache_assign_expr (e);
    <[ $cached; $safe = $safe + $val ]>
  }

  macro @-= (e, val) {
    def (cached, safe) = Operators.cache_assign_expr (e);
    <[ $cached; $safe = $safe - $val ]>
  }

  macro @*= (e, val) {
    def (cached, safe) = Operators.cache_assign_expr (e);
    <[ $cached; $safe = $safe * $val ]>
  }

  macro @/= (e, val) {
    def (cached, safe) = Operators.cache_assign_expr (e);
    <[ $cached; $safe = $safe / $val ]>
  }

  macro @<<= (e, val) {
    def (cached, safe) = Operators.cache_assign_expr (e);
    <[ $cached; $safe = $safe << $val ]>
  }

  macro @>>= (e, val) {
    def (cached, safe) = Operators.cache_assign_expr (e);
    <[ $cached; $safe = $safe >> $val ]>
  }

  macro @%= (e, val) {
    def (cached, safe) = Operators.cache_assign_expr (e);
    <[ $cached; $safe = $safe % $val ]>
  }

  macro @|= (e, val) {
    def (cached, safe) = Operators.cache_assign_expr (e);
    <[ $cached; $safe = $safe %| $val ]>
  }

  macro @&= (e, val) {
    def (cached, safe) = Operators.cache_assign_expr (e);
    <[ $cached; $safe = $safe %& $val ]>
  }

  macro @^= (e, val) {
    def (cached, safe) = Operators.cache_assign_expr (e);
    <[ $cached; $safe = $safe %^ $val ]>
  }
  
  macro @<-> (e1, e2)
  {
    def (cached1, safe1) = Operators.cache_assign_expr (e1);
    def (cached2, safe2) = Operators.cache_assign_expr (e2);    
    <[
      $cached1;
      $cached2;
      def tmp = $safe1;
      $safe1 = $safe2;
      $safe2 = tmp;
    ]>
  }

  macro @::= (e1, e2) {
    def (cached, safe) = Operators.cache_assign_expr (e1);
    <[ $cached; $safe = $e2 :: $safe ]>
  }
  
  macro @=> (parms, body) {
    def convert_to_parm (x) {
      | <[ $(nm : name) ]> => <[ parameter: $(nm : name) ]>
      | <[ $(nm : name) : $ty ]> => <[ parameter: $(nm : name) : $ty ]>
      | _ => Message.FatalError ($"unsupported syntax for parameter of 'parms => body' lambda expression: $x");
    }
   
    match (parms) {
      | <[ () ]> => <[ fun () { $body } ]>
      | <[ (..$parms) ]> =>
        def parms = parms.Map (convert_to_parm);
        <[ fun (..$parms) { $body } ]>
        
      | _ => 
        def single = convert_to_parm (parms);
        <[ fun (..$([single])) { $body } ]>
    }
  }
}
