--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Strings.Unbounded;
with Ada.Text_IO;
with Ada.Strings.Fixed;
with GNAT.OS_Lib;

with Exception_Message;

with Narval.Actors;
with Narval.Configurator.Abstract_Actors_Coordination;
with Narval.Narval_Naming_Registry;
with Narval.Environment;

with McKae.XML.EZ_Out.String_Stream;

with Ok_With_Message;
with Options;
with Load_Command;

package body Set_Command is

   use Ada.Strings.Unbounded;

   ---------
   -- Set --
   ---------

   function Set (Shell_Line : String) return String is
   begin
      declare
         Arguments : constant Shell_Commands.Arguments_Array :=
           Shell_Commands.Arguments (Shell_Line);
         use Narval.Configurator.Abstract_Actors_Coordination;
         Config : Configuration_Access;
         Actor : Narval.Actors.Actor_Class_Access;
      begin
         if Arguments'Length = 3 then
            Config := Narval.Narval_Naming_Registry.Get_Configuration
              (To_String (Arguments (3)));
            if Arguments (1) = "action" and Arguments (2) = "finish" then
               End_Actor_Handler (Config);
            else
               Set (Config, To_String (Arguments (1)),
                    To_String (Arguments (2)));
               --  load a shell script if we are in loaded state coming
               --  from configured state
               if Arguments (2) = "load" and
                 Options.Present_Option ("automatic-script-loading") then
                  declare
                     Scripts_Directory : constant String :=
                       Narval.Environment.Scripts_Directory.all;
                     Configuration_File : constant String :=
                       Get_Image (Config, "configuration_file");
                     Configuration_Name : constant String :=
                       Get_Image (Config, "name");
                     --  removing the .xml extension
                     Script_File : constant String :=
                       Ada.Strings.Fixed.Head (Configuration_File,
                                               Configuration_File'Length - 4);
                  begin
                     if GNAT.OS_Lib.Is_Regular_File (Scripts_Directory &
                                                     "/" & Script_File) then
                        Ada.Text_IO.Put_Line
                          ("SHELL : Automatic Script Loading returned :" &
                           Load_Command.Load ("load " & Script_File & " " &
                                              Configuration_Name));
                     else
                        Ada.Text_IO.Put_Line
                          ("SHELL : Automatic Script Loading enabled" &
                           " but no shell for " & Script_File);
                     end if;
                  end;
               end if;
            end if;
            return Ok_With_Message
              ("set",
               Sub_System_Name => To_String (Arguments (3)),
               Message => To_String (Arguments (1)));
         elsif Arguments'Length = 4 then
            Config := Narval.Narval_Naming_Registry.Get_Configuration
              (To_String (Arguments (3)));
            Actor := Get_Actor (Config, To_String (Arguments (4)));
            Narval.Actors.Set
              (Actor, To_String (Arguments (1)), To_String (Arguments (2)));
            return Ok_With_Message
              ("set",
               Sub_System_Name => To_String (Arguments (3)),
               Message => To_String (Arguments (1)),
               Actor_Name => To_String (Arguments (4)));
         else
            return Syntax_String;
         end if;
      exception
         when E : others =>
            return Exception_Message ("inner set", E);
      end;
   exception
      when Shell_Commands.No_Argument =>
         return Syntax_String;
      when E : others =>
         return Exception_Message ("set", E);
   end Set;

   function Syntax_String
     (Status : Shell_Commands.Status_Type := Shell_Commands.Syntax_Error)
     return String is
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Xml_Buffer : String_Buffer;
   begin
      Clear (Xml_Buffer);
      Start_Element (Xml_Buffer, "result",
                     ("cmd" = "set", "status" = Status'Img));
      Start_Element (Xml_Buffer, "message", "type" = "help");
      Output_Element (Xml_Buffer, "syntax",
                      "set argument value sub_system_name [actor_name]");
      End_Element (Xml_Buffer, "message");
      End_Element (Xml_Buffer, "result");
      declare
         String_To_Return : constant String := Get_String (Xml_Buffer);
      begin
         Full_Clear (Xml_Buffer);
         return String_To_Return;
      end;
   end Syntax_String;

end Set_Command;
